;------------------------------------------------------------------------------ 
; File: MICS_SUN.PRO     CAMMICE MICS Sun Sector and Other Parameters 
; Revision: 03/14/96    J L Roeder             
;------------------------------------------------------------------------------ 
; Routines:             
;      def_msunstat_strc        Create structure for MICS Status from Mux
;      load_msunstat            Loads structure with MICS status data
;------------------------------------------------------------------------------ 
pro def_msunstat_strc, strc, nbytes

;       Define the structure to hold the MICS Multiplexed Status bits

strc = { MICS_SunStat,          $
	ral_flag: 0B,		$ ; Rutherford Armed/Safe Flag
	cal_flag: 0B,		$ ; MDPU Calibration Flag
	mtr_flag: 0B,		$ ; Maximum TOF Reject Flag
	dep_scheme: 0B,		$ ; Direct Event Priority Scheme
	rss_page: 0B, 		$ ; Rate Scaler Select Page Number
	result_mode: 0B,	$ ; Cause of DPU Reset and Ram Reload
	HVPS_On: 0B,		$ ; Postacceleration On Flag
	HVPS_Step: 0B,		$ ; Postacceleration Step, 0-7
	Sun_Detected: 0B, 	$ ; Sun Pulse Received Flag
	Sun_Sector: 0B	 	$ ; Sector in which Sun Detected
	}

nbytes = 10

end
;------------------------------------------------------------------------------ 
pro load_msunstat, status, msun_stat

;       Loads structure with MICS status info from mux data

; Inputs:
;	status		Array of bytes from mux data, intarr( nrec, 2)

; Outputs:
;       msun_stat       Structure filled with sun sector and status flags

;       Number of major frames to process
;       assume status array is nmajor x 3
sz = size( status)
nmajor = sz( 1)

;       Define output structure
def_msunstat_strc, struc, nbytes

;       Create output array
msun_stat = replicate( struc, nmajor)

;	Load data into structure
;	RAL Armed/Safe Flag
temp = status( *, 0)
ral = ishft( temp and '20'X, -5) * (temp ne -1) + 255B * (temp eq -1) 
msun_stat.ral_flag = ral

;	MDPU Cal Flag
cal = ishft( temp and '10'X, -4) * (temp ne -1) + 255B * (temp eq -1) 
msun_stat.cal_flag = cal

;	Maximum TOF Reject Flag
mtr = ishft( temp and '08'X, -3) * (temp ne -1) + 255B * (temp eq -1) 
msun_stat.mtr_flag = cal

;	Direct Event Priority Scheme
dep = ishft( temp and '04'X, -2) * (temp ne -1) + 255B * (temp eq -1) 
msun_stat.dep_scheme = dep

;	Rate Scaler Select Page Number
rss = ( temp and '03'X) * (temp ne -1) + 255B * (temp eq -1) 
msun_stat.rss_page = rss

;	MDPU Result Mode
temp = status( *, 1)
rm = ishft( temp and '30'X, -4) * (temp ne -1) + 255B * (temp eq -1) 
msun_stat.result_mode = rm

;	HVPS On Flag
hvpson = ishft( temp and '08'X, -3) * (temp ne -1) + 255B * (temp eq -1) 
msun_stat.HVPS_On = hvpson

;	HVPS Step Number
hvpsstep = ( temp and '07'X) * (temp ne -1) + 255B * (temp eq -1) 
msun_stat.HVPS_Step = hvpsstep

;	MDPU Sun Detected
temp = status( *, 2)
sun = ishft( temp and '20'X, -5) * (temp ne -1) + 255B * (temp eq -1) 
msun_stat.Sun_Detected = sun

;	MDPU Sun Sector
sunsect = ( temp and '1F'X) * (temp ne -1) + 255B * (temp eq -1) 
msun_stat.Sun_Sector = sunsect

end
