;******************************************************************************
;* file papcoadd_polar_cammice
;*      papcoadd_polar_cammice
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for a polar_cammice. 
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_polar_cammice,plottypes_base,buttonGeometry,pb_plottype
;* 
;* DESCRIPTION:  
;*      This procedure adds a button for this polar_cammice to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*      package, so that they can be automatically called when needed.
;*      The button itself can contain access to a sub-menu for further
;*      polar_cammices.
;*
;* INPUTS:       
;*      plottypes_base  the widget ID of the parent widget to contain the
;*                      plottype buttons.
;*
;* OUTPUTS: 
;*      pb_plottype     the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for sub-types.
;*
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      addplottype_polar_cammice,plottypes_base,buttonGeometry,pb_plottype
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro addplottype_polar_cammice,plottypes_base,buttonGeometry,pdm_polar_cammice

; This example is for a simple button with a submenu. 
; **NEW** from PAPCO 09 is the inclusion of the configure button as
; the first entry of the module submenu. Further entries into the
; submenu are easily added using
;
;        { CW_PDMENU_S, 0, ' polar_cammice submenu ' }, $
;  
; The second entry in this structure defines the following:
;                       3   -   label for the button
;                       0   -   submenu entry
;                       2   -   last submenu entry  
;
; The third entry in this structure is the name of the submenu entry.

  MenuDesc = [ $
        { CW_PDMENU_S, 3, strcenter('POLAR-CAM',10) }, $ 
        { CW_PDMENU_S, 0, ' configure polar_cammice ' }, $
        { CW_PDMENU_S, 2, ' plot polar_cammice      ' } ]  
  
  pdm_polar_cammice = CW_PDMENU(plottypes_base, MenuDesc, $
                            UVALUE='pdm_polar_cammice')
  
; The name used for the submenu entries is user by PAPCO to construct
; the procedure/function calls needed for that part of the module.
  
; Given here is one submenu entry. IF YOU ADD FURTHER SUBMENU
; ENTRIES, THEN YOU WILL NEED TO WRITE ONE EACH THE
; plottable_polar_cammice, descriptor_polar_cammice,  addpanel_polar_cammice,
; editor_polar_cammice AND draw_polar_cammice 
; ROUTINES THAT FOLLOW FOR EACH SUBMENU ENTRY, MAKING SURE YOU ADHERE
; TO THE CORRECT NAMING CONVENTION.
  
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_polar_cammice
;*
;* DESCRIPTION:  
;*      Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      Using an instance of the PAPCO_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled. The
;*      desired default values for the plotinfo tags need to be edited by
;*      the user. This plotinfo structure will be passed to the panel editor
;*      When it is first called for the polar_cammice.
;*      "panelEditor" must be set to the name of the panel editor routine.
;*
;* INPUTS:       
;*      event           event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_crres_plots
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      addpanel_polar_cammice
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;***************************************************************************
pro addpanel_polar_cammice,event
  
  common get_error, get_err_no, get_err_msg
  
; The case statement here needs an entry for every submenu type you
; added in  the first procedure in this file (addplottype_polar_cammice).
; For each submenu entry, you will need to specify the panelEditor to use.  
  
; The FIRST time you use this module you will have to configure it,
; i.e. set the data paths etc. Once you have done this PAPCO will write
; a polar_cammice.config file into your papco_user_XX/papco_modules_defaults
; directory. 
; So here we check if this polar_cammice.config file exists, and if not,
; we diallow the execution of any other option apart from "configure
; polar_cammice". In this way the user is forced to configure the module
; before using it the first time!
  
  r_module_config, strlowcase('polar_cammice'), config_data, msg
  if (get_err_no ne 0) and (event.value ne 1) then begin
    print,'%addpanel_polar_cammice: You need to configure the module first'
    papco_module_configure,'polar_cammice'
    return
  endif  
 
  CASE event.value of
    1: BEGIN
        papco_module_configure,'polar_cammice'
    end
    2: BEGIN
        panelKind='polar_cammice'
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
        defaults.typeVector=[0,1,0,0]      ;set some tags to desired defaults
        defaults.yLinLog=1                
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults     
    end
    else: begin
      print,'%addpanel_polar_cammice: Nothing defined for this submenu entry'
      return
    end  
  endcase  
 
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO plottable_polar_cammice, plottable
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate plottable without orbit.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      plottable        a boolean 0 if plottable only by CRRES orbit
;*                                1 if plottable by time also.
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      plottable_polar_cammice, plottable
;*
;* MODIFICATION HISTORY:       
;*      written November 1995, Reiner Friedel
;******************************************************************************
pro plottable_polar_cammice, plottable

   plottable=1     ; 1 -> plotable without orbit info, by time 
                   ; 0 -> plotable only by orbit number not by time 
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO canget_polar_cammice, canget
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate that polar_cammice is capable of
;*      returning data from it's plot routine into a standard PAPCO
;*      array structure. Used for the scratch pad utility module.
;*      
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "canget_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_polar_cammice.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      canget         a boolean 0 if this feature is not available
;*                                1 if this feature is available
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      canget_polar_cammice, canget
;*
;* MODIFICATION HISTORY:       
;*      written November 1997, Reiner Friedel
;******************************************************************************
pro canget_polar_cammice, canget

   canget=  1      ; 1 -> plot routine can return just data to file
                   ; 0 -> feature disabled
 END
 
;******************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_polar_cammice, plotInfo, descriptStr
;*
;* DESCRIPTION:  
;*      Constructs the string used to label plots in the panel window.
;*      The plotinfo structure passed in here is the panel's plotinfo
;*      as returned by the panel editor for this polar_cammice. You can use
;*      the information in that structure to construct some informative
;*      label for the panel which then appears in the Panel Compose list.
;*
;* INPUTS:       
;*      plotInfo        structure containing all data for this plot.
;*                      (structure type: PAPCO_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*      descriptStr     a string containing the label
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      descriptor_polar_cammice, plotInfo, descriptStr
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro descriptor_polar_cammice, plotInfo, descriptStr
  
  common polar_cammice, typeNames

  descriptStr=descriptStr+typeNames(PlotInfo.ioptions(0))
   
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO editor_polar_cammice, plotNumber, newPlotData
;* 
;* DESCRIPTION:  
;*      This procedure calls the panel editor when called from 
;*      PAPCO_EditPlot (papco_paneledit.pro). Provided here to allow
;*      several polar_cammices to share a common panel editor. Normally this
;*      procedure does not need to be edited beyond the required name
;*      change for polar_cammice.
;*
;* INPUTS:       
;*      plotNumber      anInteger. 0<=plotNumber<=widgetData.numberOfPlots
;*                      the index of the plot in widgetData.plotInfos.
;*            
;* OUTPUTS: 
;*      newPlotData     the new structure returned by the panel editor
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      editor_polar_cammice, plotNumber, newPlotData
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
PRO editor_polar_cammice, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=polar_cammice_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
      GROUP=widgetData.main_base)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO draw_polar_cammice, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;* DESCRIPTION:  
;*      This procedure performs the actual data read call and the plot call
;*      for this plottype. The user need to pass to his plot routine all the
;*      parameters needed, typically these parameters are stored in the 
;*      currplotinfo structure or constructed from them. The currplotinfo
;*      structure tags normally used for passing information from the panel
;*      editor to the plot routine are:
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          swittch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      
;*      These tags have grown out of traditional usage and can be used for
;*      anything the user wants to use them for. There are plenty variables
;*      for most applications: Adding a new tag to the plotinfo structure
;*      is not user-definable but need to be done at the PAPCo core level.
;* 
;*      The panelVector MUST be passed to your plotroutine as it is used
;*      to set the panel position.
;*      The _EXTRA=extra_PlotInfo should be passed if you intend to make use
;*      of PAPCO functionality such as overplotting, linestyle choice and
;*      color bar choice.
;*
;* INPUTS:       
;*      panelVector    position of the plot (see papco_conventions.text, 6)
;*      currPlotInfo   structure containing all data for this plot.
;*                     (structure PAPCO_PLOTINFO, see papco_variables.text) 
;*
;* OUTPUTS: 
;*      get_data_call  a string containing the data call. Used by other
;*                     routines that need to know how to get the data
;*                     in a panel, e.g. to draw spectra, distribution functions
;*      IsRead         a boolean which is set when the the data is read the
;*                     first time. Useful if a given data-read puts data
;*                     into commons which can be plotted in many ways.
;*                     When set, data does not have to be read again.
;   slice_type:   Selects which slices are supported and for which slice
;                 routines have been written.
;                 This is an integer logical with 16 swittches. Currently
;                 there are 5 PAPCO slices. You may extend this by
;                 further USER slices up to a total of 16 slices.
;                     0 = disable all slices
;                     1 = PAPCO Zoom                 (2^0)
;                     2 = PAPCO View Slice           (2^1)
;                     4 = PAPCO Access level Zero    (2^2)
;                     8 = PAPCO Write Paneldata      (2^3)
;                    16 = PAPCO Hair Cursor          (2^4)
;                    32 = USER slice A or not used   (2^5)
;                    64 = USER slice B or not used   (2^6)
;                    "     "     "   "  "  "    "      "    
;                 32768 = USER slice Z or not used  (2^15)
;                        
;   You can support more than one type of slice by arithmetically adding the
;   options, eg slice_type=1+4  support Zoom and Level Zero
;*
;* KEYWORDS:
;*      OUTPUT        if set then plot routine should output data to file,
;*                    as well as to screen.
;* 
;* CALLING SEQUENCE:
;*      draw_polar_cammice, panelVector, currPlotInfo, orbitNo, $
;*                          get_data_call, slice_type, IsRead, NODATA=NODATA
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro draw_polar_cammice, panelVector, currPlotInfo, orbitNo, $
                        get_data_call, slice_type, IsRead, $
                        OUTPUT=OUTPUT

COMMON extra, extra_PlotInfo  ; passes in PAPCO-set keywords structure

get_data_call=''
   
; as PAPCO dynamically allocates slice values for the user defined
; slices, one needs to be able to find out this assignment! Use the
; following function to find out the slice values for any slices you
; might have added for your module, so that you know which values to enable!
  
slice_assigned_value=PAPCO_slice_value('USER View Ephemeris')
   
; set up which mouse functions to allow
slice_type=1+2+8+16+slice_assigned_value

IF NOT currPlotInfo.ioptions(9) THEN IsRead = 0
IF NOT IsRead THEN BEGIN          ; check if data has already been read
    r_polar_cammice, currPlotInfo   ; read data
    ; check the read for success.
    IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel)        THEN IsRead=1
ENDIF
   
IF IsRead THEN BEGIN
    p_polar_cammice_spec, panelVector, currPlotInfo, OUTPUT=OUTPUT,$
       _EXTRA=extra_PlotInfo
ENDIF   

; Note: The structure extra_PlotInfo passed to the plot routine contains the
;       following tags: 
;                  overPlot               - overplot swittch
;                  plots_AtThisPosition   - no. of overplots here
;                  subtable               - index of PAPCO colortable 
;                  PSYM                   - linestyle keyword for plot
;                  COLOR                  - linecolor keyword for plot
;               
;       They are set by PAPCO and your plot routine can use them if you
;       want to support that functionality.

END

