;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  polar_cammice_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the PAPCO_PLOTINFO-type
;*                      this structure is edited in the panel  editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type. 
;*      All the functions in $PAPCO_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at last some fields. Those fields might make up a structure, but as no
;*      program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*      poor structure is not alive but only virtual.
;*      The return-values of plot-editors have to expand this structure - I 
;*      like to think, that they inherit from that structure - and contain at
;*      least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*                      swittch:[0,0,0], $
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, swittch and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          swittch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      NB! If you need more variables to commincate between your
;*     panel editor and your plot routine then use your own common block
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which PAPCO supplied widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      PAPCO internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $                       
;*                      xut2:0.0}                           
;*
;*                  
;*      For an enhanced description read 
;*                          $PAPCO_XX/PAPCO_doc/papco_variables.text
;*
;* CALLING SEQUENCE:
;*      newPlotData=PAPCO_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  polar_cammice_panel, aPlotInfo, GROUP=group, ACTION=ACTION

COMMON polar_cammice_panel_data, widgetdata
COMMON polar_cammice, typeNames

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
panel_base=papco_widget_base(title=aPlotInfo.panelKind, column=1, /modal, $
                                GROUP_LEADER=group)

; All PAPCO_panel_Add* routines are PAPCO panel widgets that can be included
; in the panel editor or not. They provide functionality that PAPCO supports.
; They dynamically extend the panel editors widget structure, and all the 
; events they generate are supported by PAPCO_panel_Event, transparent to the
; user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
widgets=PAPCO_panel_AddHeightWidgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)

;
; <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;
   
choices_base= WIDGET_BASE(panel_base, ROW=1)

;--- Counts/ Flux buttons -----------------------------------------------------
flux_names = ['counts', 'flux']
flux_base = CW_BGROUP(choices_base, flux_names, /FRAME, COL=1, /EXCLUSIVE, $
                      UVALUE='spb_flux', SET_VALUE=aplotinfo.ioptions(1))
  
;--- SAMPLES/DATA  buttons ----------------------------------------------------
sample_names = ['data', 'samples']
sample_base=CW_BGROUP(choices_base, sample_names, /FRAME, COL=1, /EXCLUSIVE, $
                      UVALUE='spb_sample', SET_VALUE=aplotinfo.ioptions(2))
widget_control, sample_base, sensitive = 0

;--- Full / avergeage buttons -------------------------------------------------
detail_names = ['average', 'detail']
detail_base=CW_BGROUP(choices_base, detail_names, /FRAME, COL=1, /EXCLUSIVE, $
                      UVALUE='spb_detail', SET_VALUE=aplotinfo.ioptions(4))


;--- read orig files or NSSDCA files
orig_names = ['Originals', 'NSSDCA']
orig_base=CW_BGROUP(choices_base, orig_names, /FRAME, COL=1, /EXCLUSIVE, $
                    UVALUE='spb_orig', SET_VALUE=aplotinfo.ioptions(10))

;--- data reload buttons ------------------------------------------------------
reload_names=['reload', 'adopt']
reload_base=CW_BGROUP(choices_base, reload_names, /FRAME, COL=1, /EXCLUSIVE, $
                      UVALUE='spb_reload', SET_VALUE=aPlotInfo.ioptions(9))
  
;--- SMOOTH choice fields -----------------------------------------------------
smooth_base=WIDGET_BASE(choices_base, ROW=1)
smooth_entry=CW_FIELD(smooth_base, /column, $
                        title='Slice +/-', $
                        value=aPlotInfo.ioptions(3), $
                        uvalue='smooth_entry', /integer, xsize=2)
   
;--- Channel choice  ----------------------------------------------------------
type_base = CW_BGROUP(panel_base,typeNames, /FRAME, ROW=1, /EXCLUSIVE, $
                      UVALUE='spb_type', SET_VALUE=aplotinfo.ioptions(0))

;--- Data type choice --------------------------------------------------------
data_type_base = WIDGET_BASE(panel_base, /FRAME,COLUMN = 1)
data_names = ['Spin Averaged Spectra', 'Pitch angle/Energy']
data_base = CW_BGROUP(data_type_base, data_names, ROW = 1, /EXCLUSIVE, $
                      LABEL_LEFT='Type of Data:', $
                      UVALUE='spb_data', $
                      SET_VALUE=aplotinfo.ioptions(5))

;--- add choices for plot types -----------------------------------------------
plot_names = ['Spectrum',  'Line', 'Ek @ PA',  'PA @ Ek']
plot_base = papco_CW_BGROUP(data_type_base, plot_names, $ 
                        ROW=1, /EXCLUSIVE, $
                        LABEL_LEFT='Plot type:', BUTTON_IDS = plot_ids, $
                        UVALUE='spb_plot', $
                        SET_VALUE=aplotinfo.ioptions(8)) 
widget_control, plot_ids(1), sensitive = 0

;--- select for pitch - energy, pitch angle ----------------------------------
pitch_base = WIDGET_BASE(panel_base, ROW = 1, /base_align_center, /FRAME)
sl_pa =WIDGET_SLIDER(pitch_base, XSIZE=245, $
		       /DRAG, TITLE='pitch angle bin #', $
		       VALUE=aPlotInfo.ioptions(6) < 19 > 1, $
                       UVALUE='sl_pa', MINIMUM=1, MAXIMUM=19)

sl_ek =WIDGET_SLIDER(pitch_base, XSIZE=245, $
		       /DRAG, TITLE='energy channel #', $
		       VALUE=aPlotInfo.ioptions(7) < 32 > 1, $
                       UVALUE='sl_ek', MINIMUM=1, MAXIMUM=32)

;
; <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   

;--- add the PAPCO panel widget for choosing the color bar for color plots ----
widgets=papco_panel_Addcolorbar_Widgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
;--- add the  PAPCO panel widget for choosing the plot-style for line plots ---
;   widgets=PAPCO_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
;   additionalWData=create_struct(widgets, additionalWData)
;--- add the PAPCO panel widget for choosing the y-axis-ranges ----------------
widgets=PAPCO_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
;--- add the PAPCO panel widgets for choosing the z-axis-ranges ---------------
   widgets=PAPCO_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors.
;    if you want to make use of the configure feature, then add the
;    keyword CONFIG
widgets=PAPCO_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. You should also actually create 
;    that helpfile in the papco_interface directory of your module (a template
;    for this help file exist in the polar_cammice example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
thisWidgetsData={helpfile:'polar_cammice_panel.help', $
                 plot_base:plot_base, $
                 detail_base:detail_base, $
                 smooth_entry:smooth_entry, $
                 ioptions:aPlotInfo.ioptions}
widgetData=PAPCO_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

polar_cammice_panel_refresh, /ALL

;--- realize the window
PAPCO_Panel_Realize, panel_base, widgetData

; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reported but the 
; data contained in aPlotInfo would be returned.
papco_xmanager, 'polar_cammice_panel', panel_base

; the returned data consists of the standard-data inherited from
; PAPCO_PANEL.PRO and your new data - the ioptions-tag and swittch-tag must
; always be included, but you can add more. However, if possible stick to the 
; tags provided in aPlotInfo, it makes life easier. 
returnData={ioptions:widgetData.ioptions}
additionalRData=PAPCO_panel_ReturnData(widgetData)

return, create_struct(ReturnData, additionalRData)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO polar_cammice_panel_refresh
;******************************************************************************
PRO polar_cammice_panel_refresh, DATA_TYPE = DATA_TYPE,  PITCH = PITCH, $
                                 ALL = ALL

COMMON polar_cammice_panel_data, widgetData

IF keyword_set(ALL) THEN BEGIN
    DATA_TYPE = 1 & PITCH = 1
ENDIF

IF keyword_set(DATA_TYPE) THEN BEGIN
    CASE widgetData.ioptions(5) OF ;Type of Data
        0: BEGIN
            widget_control, widgetData.plot_base, set_value = 0
            widgetData.ioptions(8) = 0
            widget_control, widgetData.detail_base, sensitive = 1           
        END
        1: BEGIN
            widget_control, widgetData.plot_base, set_value = 3
            widgetData.ioptions(8) = 3
            widget_control, widgetData.detail_base, set_value = 0
            widget_control, widgetData.detail_base, sensitive = 0
        END
    ENDCASE     
ENDIF 

IF keyword_set(PITCH) THEN BEGIN
    IF widgetData.ioptions(8) EQ 3 THEN BEGIN
        widgetData.ylinlog = 0
    ENDIF ELSE BEGIN
        widgetData.ylinlog = 1
    ENDELSE 
    papco_panel_refresh, widgetData, /YSCALE
ENDIF

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO polar_cammice_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO polar_cammice_panel_event, event

  COMMON polar_cammice_panel_DATA, widgetData
  WIDGET_CONTROL, GET_UVALUE=uval, event.id
   
  CASE uval OF

    'spb_type': widgetData.ioptions(0) = event.value
    'spb_flux': WidgetData.ioptions(1) = event.value
    'spb_sample': WidgetData.ioptions(2) = event.value
    'spb_detail' : WidgetData.ioptions(4) = event.value
    'spb_orig': WidgetData.ioptions(10) = event.value
    'spb_reload': WidgetData.ioptions(9) = event.value
    'spb_data': BEGIN
        WidgetData.ioptions(5) = event.value
        polar_cammice_panel_refresh, /DATA_TYPE, /PITCH
    END     
    'spb_plot': BEGIN
        WidgetData.ioptions(8) = event.value
        polar_cammice_panel_refresh, /PITCH
    END     
    'sl_pa': widgetData.ioptions(6)=event.value
    'sl_ek': widgetData.ioptions(7)=event.value
    
    ELSE : begin
      WIDGET_CONTROL, widgetData.smooth_entry, GET_VALUE=x
      widgetData.ioptions(3)=x   
      WidgetsData=PAPCO_Panel_event(event,WidgetData)
    END
    
  ENDCASE    

   
END
