;******************************************************************************
;* file polar_cammice_slice.pro
;*
;* this file contains the routines needed to enable the extra mouse features
;* provided for in PAPCO. The feature "mouse zooms" is part of main PAPCO, and
;* is enabled by slice_type = +1. "mouse writes paneldata" has a default 
;* functionality provided for in main PAPCO (writes time and y-value of mouse 
;* click to file), but can be extended by a user written routine to include 
;* auxiliary data.
;*
;* Currently possible:
;*
;*    mouse zooms                        no routine here, in PAPCO main
;*    mouse views slice                  routine here
;*    mouse views slice (polar)          specialty slice - no routine here
;*    mouse views ephemeris              under development (polar only)
;*    mouse accesses level zero          routine here
;*    mouse writes paneldata             routine here
;* 
;* Here the procedure names must again adhere to a strict naming convention so 
;* that they can be automatically called when needed.
;*
;* The slice types supported by these routines must be ENABLED by the user, so 
;* that PAPCO can call them. This is done by setting the value of 
;* slice_type in draw_polar_cammice.pro (contained in papcoadd_polar_cammice.pro).
;* slice_type is defined in the following way:
;*
;*      slice_type      Selects which slices are supported and for which
;*                      slice routines have been written.
;*                      This is a byte logical with 8 swittches. Currently
;*                      Three are supported (this may be extended later)
;*                        1 = mouse zooms (utility slice, always possible)
;*                        2 = mouse slices
;*                        4 = mouse calls level zero
;*                        8 = mouse views slice (polar)  
;*                       16 = mouse views ephemeris
;*                       32 = mouse writes paneldata (utility slice)
;*                       64, 128 = not supported yet
;*                      you can support more than one type of slice
;*                      by arithmetically adding the options, eg
;*                      slice_type=5  support zoom and level zero
;*
;* The "mouse zooms" function uses the mouse to define an area of a panel. 
;* Dragging the mouse with the left button pressed marks out a shaded rectangle
;* in the panel. The time range and y-range of the rectangle are read out in a
;* window and the user has the option of adopting these new ranges. THE PAPCO 
;* main common block for time (common mjdt, mjdt_start, mjdt_end) and y-range
;* (common yscale, yscl) are updated accordingly.
;*
;* The "mouse views slice" function uses the time returned by PAPCO and the
;* get_data_call specified for the polar_cammice to read the data and to draw a 
;* slice at the clicked position. This is commonly used in spectra color plots 
;* to draw a spectrum at that time, or a pitch angle distribution etc, or to
;* call up any other plot at that time as specified by the user. 
;* -> user defined interface routine needed, template included here
;*
;* The "mouse accesses levelzero" uses the time passed and calls up another
;* widget application to read and work with level zero data. This application
;* must be entirely independent of PAPCO and is only called from PAPCO via
;* a mouse click in the panel required for the time required.
;* -> user defined interface routine needed, template included here
;*
;* The "mouse views ephemeris" function calls up a window with a 3-D display of
;* the satellite orbit. For this PAPCO-compatible routines that read the
;* ephemeris data for a given satellite are needed. Currently only supported
;* for polar. The idea is to extend this to draw a physical model of the 
;* magnetosphere too, to be able to view this model + orbit + satellite 
;* position from any angle etc - definitely still under development! A simple 
;* orbit visualization is in place for polar (CEPPAD and HYDRA modules).
;* -> user defined interface routine needed, no template included here
;*
;* The "mouse writes paneldata" function allows the user to read off data from 
;* panels using the mouse and to write that data to a file. This is a PAPCO
;* supplied utility slice which by default only writes the time and y-value of
;* the mouse-click to file. The user can however interface his own routine to
;* add auxiliary data to the string written to file.
;* The output file is written in ASCII and has the same format as the .event
;* files used by the event times utility module. The files can thus be used
;* in the events time module to mark events on panels.
;* -> user defined interface routine possible, template included here
;*
;* If your new data contains several sub-data types, one slice and/or levelzero
;* routine has to be provided for each type, since in general they would do
;* different things. See 
;* $PAPCO_MODULES\CRRES_meb\papco_interface\crres_slices.pro  
;* for such an example.
;*
;*      PRO Vert_Slice_polar_cammice, panelNr, seconds, yValue
;*        -> Draw a slice for polar_cammice
;*
;*      PRO levelzero_polar_cammice, seconds, yValue
;*        -> This procedure calls the widget for levelzero data access 
;*           for this plottype.
;*
;*      PRO WriteSlice_polar_cammice, panelNr, seconds, yValue
;*        -> This procedure calls the widget for writing panelinfo data to file
;*    
;* MODIFICATION HISTORY:       
;*      December 1995, written by R.Friedel at Goddard
;*      modified may 1997, Reiner Friedel
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO Vert_Slice_polar_cammice, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw a slice for polar_cammice
;*      calling a separate window for the plot.
;*      slice_type in draw_polar_cammice must be set to +2 to enable this feature.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by PAPCO_DrawSLice (in $PAPCO_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      December 1995, written by R.Friedel at Goddard
;******************************************************************************
PRO Vert_Slice_polar_cammice, panelNr, seconds, yValue, READDATA=READDATA

COMMON papco_vert_slice, slice_widget
COMMON PLOT_COMPOSER, widgetData

; add in here any other common blocks you would need to make your plot data
; available. The common block included here is a dummy for illustration.
; PAPCO uses the the getdatacall you set in papco_add_polar_cammice to read the
; data you need for the slice. As long as you stay in one panel with the 
; mouse, the data is not read again. When you swittch panels, the new data is 
; read. 

COMMON polar_cammice_slice, time,zmat,yarrng,factor,utitle
COMMON zscale, zscl           ;man/auto zscaling
COMMON mjdt, mjdt_start, mjdt_end        ;common time limit in mjdt

WIDGET_CONTROL,/HOURGLASS

; use the same typevector, swittch etc as is used in the call to your plot
; routine. thisPlot contains the full PAPCO structure of plot information for
; the panel in which the mouse was clicked in. 
thisPlotdrawn=widgetData.plotsDrawn(panelNr)
thispanel=widgetData.PLOTINFOS(panelNr)
type=thisPanel.ioptions

; set the panel window for the slice to full window
!p.position=[0.1,0.1,0.9,0.9]

; select the index for the time chosen out of the time array
DiffTimes=ABS(time(*, 0)- seconds)   ; this sets timeIndex to the index with
result=MIN(DiffTimes, timeIndex)     ; time(index) is closest to 'seconds'
   
; these two info strings are user-definable output strings to label the slice 
; window. To calculate the time use the start time in mjdt and the seconds 
; passed in. If your plot uses the PAPCo philosophy then the time axis is in 
; running seconds since the start of the start day.

; However, you of course can use your own time format, then the information
; in seconds is whatever your time format is. papco provides a function
; to return a verbose time for ANY time format.    
; papco always knows the start end time of a plot from the time common
; block, and the start/end value of the x_axis shouldalways be stored in 
; papco by the user's plot routine:;
;
;   common coordinateSystems, plotted_x, plotted_y  
;
;   plotted_x = !x
;   plotted_y = !y     -   after the user has set up his axis.
;
; If this is done properly then the call
;   
;   time_str=papco_cursor_time(seconds,panelNr)
;   
; should return the correct time.
   
time_str=papco_cursor_time(seconds,panelNr)
time_mjdt=papco_cursor_time(seconds,panelNr,/mjdt)
   
av=type(3)

info1='POLAR CAMMICE Slice [Av: +/-'+string(av,format="(2i2)")+']'
info2=string(time_str, yValue, $
                       format="('time: ',A20,'   Y-value: ',f9.2)")
   
;--- now, we need to get the data --------------------------------------------
;    normally papco_cursor takes care of this. For cammice, there is a
;    lengthy procedure in making the plot file once the data has been
;    read. So, here we only need to remake the data arrays if the user
;    clicks in a different cammice window.    

IF slice_widget.lastPanelDrawn NE panelNr THEN BEGIN
    message, 'Need to re-make data arrays', /cont
    p_polar_cammice_spec, thisPlotdrawn.panelvector, thispanel, OUTPUT = 2
ENDIF

; Do the plot.   
  st_idx=(timeIndex-av) > 0
  en_idx=(timeIndex+av) < (n_elements(time)-1)
  yarr_sum=fltarr(n_elements(yarrng[*,1]))
  yarr_cnt=fltarr(n_elements(yarrng[*,1]))
   
  t_st_st=papco_cursor_time(time(st_idx),panelNr)
  t_st_en=papco_cursor_time(time((en_idx+1)<(n_elements(time)-1)),panelNr)
   
  for i=st_idx, en_idx do begin
    yarr=zmat(i,*)
    index=where(yarr ne -999,c)
    if c ne 0 then begin
      yarr_sum(index)=yarr_sum(index)+yarr(index)
      yarr_cnt(index)=yarr_cnt(index)+1
    endif  
  endfor   
   
  index=where(yarr_cnt ne 0,c)
  if c ne 0 then yarr_sum(index)=yarr_sum(index)/yarr_cnt(index)
  yarr=yarr_sum
   
  xarr=(yarrng[*,0]+yarrng[*,1])/2

  ymin=thisplotdrawn.ZSCL_MIN
  ymax=thisplotdrawn.ZSCL_MAX 
   
  xmin=thisplotdrawn.YSCL_MIN
  xmax=thisplotdrawn.YSCL_MAX
      
  !x.ticks=0
  if type(1) eq 0 then ytit='counts/s' $
    else ytit='flux [cm!e-2!ns!e-1!nsr!e-1!nkeV!e-1!n]'
      
  erase
    
  plot, xarr,yarr,xrange=[xmin,xmax], $
    yrange=[ymin,ymax], color=1, /xlog, /ylog, /nodata, $
    xtitle='Energy [keV]', ytitle=ytit, title=info1+'!C',psym=-6
  
  ndat=n_elements(yarr)
  plots,xarr(0),yarr(0), color=1, psym=-6
  for i=1,ndat-1 do begin
    if (yarr(i-1) ne 1e-20) and (yarr(i) ne 1e-20) then $
      plots,xarr(i),yarr(i), color=1, psym=-6, /continue else $
      plots,xarr(i),yarr(i), color=1, psym=-6
  endfor  
   
  xyouts,0.6,0.875,utitle+'!CFrom: '+t_st_st+'!CTo:    '+t_st_en, $
    /normal,color=1
  
; overplot zero count level
  yarr=factor(0:ndat-1)/ (5.804 * (en_idx-st_idx+1))
  oplot,xarr,yarr,linestyle=1, color=1
  
; user defined routine for plotting the slice included here. An example is in 
; $PAPCO_MODULES/crres_MEB/papco_interface/crres_slices.pro
; You typically need to include the commons from the read routine, a way
; of finding the index of the data you want to plot using the time "seconds"
; plus a simple plot statement to plot the data.
; Two strings, info1 and info2, can be used to annotate the slice window
; by providing additional information, ie ephemeris data, etc.

; these calls put your info strings in the provided labels of the slice widget

WIDGET_CONTROL, slice_widget.lbl_slice1, SET_VALUE=info1
WIDGET_CONTROL, slice_widget.lbl_slice2, SET_VALUE=info2

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO levelzero_polar_cammice, seconds, yvalue
;*                                     
;* DESCRIPTION:  
;*      This procedure calls the widget for level zero data access 
;*      for this plottype. 
;*      slice_type in draw_polar_cammice must be set to +4 to enable this feature.
;*
;* INPUTS:       
;*      seconds           T90 time at which mouse is clicked
;*      yvalue            y-value at which mouse is clicked
;*
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      levelzero_polar_cammice, seconds, yvalue
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel at Goddard
;******************************************************************************
pro levelzero_polar_cammice, seconds, yvalue

; The actual procedure is very dependent on what the level zero widget 
; application needs. An example is in $PAPCO_MODULES/addpapco_swe_plots.pro.
; Provided as parameters to be passed to the new application, are the values
; of the mouse pointer when clicked. 
;
; The main widget remains active and can be swittched to (e.g. to get a new
; time from the graph) by using a "Parent" button in the new application
; which then executes the line
;            current_appl=xregistered('PAPCO_main') 
; to swittch back to the PAPCO main calling widget. 

; call the levelzero widget.  
;   polar_cammice_levelzero, seconds, yvalue

; the following line is printed on your xterm as a check to see if PAPCO
; correctly called this routine. If there is a problem make sure your naming 
; convention is O.K. : This routine MUST be called levelzero_polar_cammice, 
; where polar_cammice is the name of the panel you have set in papcoadd_polar_cammice.

    print, 'got to the levelzero interface  ',seconds, yvalue

END


;******************************************************************************
;* PROCEDURE:     
;*      PRO WriteSlice_polar_cammice, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw a Write Paneldata slice for polar_cammice
;*      calling a seperate window for the widget
;*      slice_type in draw_ceppad must be set to +32 to enable this feature.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      READDATA    if set the data needs to be re-read from file
;* 
;* CALLING SEQUENCE:
;*      called by PAPCO_DrawSLice (in $PAPCO_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      may 1997, written by R.Friedel
;******************************************************************************
PRO WriteSlice_polar_cammice, panelNr, seconds, yValue,READDATA=READDATA

   COMMON PAPCO_WRITE_PANELDATA, slice_widget
   COMMON PLOT_COMPOSER, widgetData

; THIS TEMPLATE ACTUALLY DOES NOTHING, ONLY THE DEFAULT WRITE SLICE 
; FUNCTIONALITY WILL WORK (WRITE TIME AND Y-VALUE OF MOUSECLICK TO FILE). THE 
; EXAMPLE HERE IS TO SHOW HOW ONE COULD ADD ADDITIONAL INFORMATION.

; add in here any other common blocks you would need to make your additional
; data available (ie ephemeris).

; PAPCO uses the the getdatacall you set in papco_add_polar_cammice to read the
; plotdata For your panel. As long as you stay in one panel with the 
; mouse, the data does not need to be read again (READDATA=0). When you swittch
; panels, new data needs to be read (READDATA=0). It's up to the user to read 
; in the data he needs here. 

;   common polar_cammice_data, input_header, input_data
;   common polar_cammice_ephem, polar_cammice_ephem
;   common mjdt, mjdt_start, mjdt_end    ;common time limits in mjdt

   WIDGET_CONTROL,/HOURGLASS
   thisPlot=widgetData.plotsDrawn(panelNr)

; use the same typevector, swittch etc as is used in the call to your plot
; routine. thisPlot contains the full PAPCO structure of plot information for
; the panel in which the mouse was clicked in. 

   thisPlot=widgetData.plotsDrawn(panelNr)
   type=thisPlot.typeVector
   getdatacall=thisPlot.get_data_call

;read the data if needed

;   if READDATA then begin 
;      callgood=execute(getdatacall)
;      r_polar_cammice_ephem
;   endif

; select the index for the time chosen out of the time array

;   DiffTimes=ABSpolar_cammice(.time - seconds)
                                        ; this sets timeIndex to the index 
;   result=MIN(DiffTimes, tidx)         ; closest to 'seconds'

; these strings are user-definable output strings to label the slice window.
; The "write Panelinfo" window PAPCo supplies has 6 lines of user-definable
; text. 

   info1='User label 1'
   info2='User data  1'
   info3='User label 2'
   info4='User data 2'
   info5='User label 3'
   info6='User data 3'

; Time and y-value of mouse click are automatically put into labels.
; The calls below put your info strings in the provided user labels (max 6).
;
; NB NB NB:
; However, only info2, info4, info6 are written to file in the form
; info2+info4+info6.
;
; The idea is that info1 will have the labels for the data in info2, info3 the
; labels for the data in info4, and info5 the labels for the data in info6.
; Time and y-value of mouse click are written first, then info2+info4+info6
; followed by a user defined comment from the Write Paneldata window.

   WIDGET_CONTROL, slice_widget.lbl_slice1, SET_VALUE=info1
   WIDGET_CONTROL, slice_widget.lbl_slice2, SET_VALUE=info2
   WIDGET_CONTROL, slice_widget.lbl_slice3, SET_VALUE=info3
   WIDGET_CONTROL, slice_widget.lbl_slice4, SET_VALUE=info4
   WIDGET_CONTROL, slice_widget.lbl_slice5, SET_VALUE=info5
   WIDGET_CONTROL, slice_widget.lbl_slice6, SET_VALUE=info6

END


;******************************************************************************
PRO DRAW_EPHEMERIS_polar_CAMMICE, panelNr, seconds, yValue, READDATA=READDATA
   if READDATA eq 1 then r_polar_ephem
   Draw_ephemeris_ephemeris, panelNr, seconds, yValue
end

;******************************************************************************
;* PROCEDURE:     
;*      PRO USER_VIEW_EPHEMERIS_polar_CAMMICE_Event, Event
;* 
;* DESCRIPTION:  
;*	The slice-window displays a slice through the data, 
;*	when the user clicks in the drawing-widget.
;*	This procedure handles events for that window.
;* INPUTS:       
;*	Event	 	an XMANAGER-event
;*	
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     written august and september 1995, Andreas Keese
;******************************************************************************
PRO USER_VIEW_EPHEMERIS_polar_CAMMICE_Event, Event

  COMMON DRAW_polar_EPHEMRIS_SLICE, slice_widget

  WIDGET_CONTROL,Event.Id,GET_UVALUE=Ev

  CASE Ev OF
    'pb_done' : BEGIN
        slice_widget.lastPanelDrawn=-1   ;set to "no panel chosen"
        WIDGET_CONTROL, slice_widget.base, /DESTROY
     END
     
     'pb_help': $
       xdisplayfile, papco_helpfile('papco_ephemeris_slice.help'), $
       title='PAPCO Online-Help', group=event.top

     ELSE : BEGIN
     END
  ENDCASE

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO USER_VIEW_EPHEMERIS_polar_CAMMICE, panelNr, seconds,
;*                                           yValue, CLOSE=CLOSE
;* 
;* DESCRIPTION:  
;*	The slice-window displays ephemeris info, when the user clicks
;*	in the drawing-widget.
;*	This procedure opens that window and calls other routines, that draw
;*	the ephemris info.
;*
;* INPUTS:       
;*	panelNr	 	an integer
;*			the number of the panel for which the slice is to be
;*			drawn
;*	seconds		a float
;*	yvalue		a float
;*			the time and the y-value, for that the slice shall
;*			be drawn
;*
;* KEYWORDS:
;* 	CLOSE		when specified, the window is destroyed
;* 
;* CALLING SEQUENCE:
;*	PAPCO_DrawSlice, aPanel, convert_tzero_to_t90(3600, 1, 1990), 42.42
;*	PAPCO_DrawSlice, /CLOSE
;*
;* MODIFICATION HISTORY:       
;*     written october 1996, Reiner Friedel
;******************************************************************************
PRO USER_VIEW_EPHEMERIS_polar_CAMMICE, panelNr, seconds, yValue, CLOSE=CLOSE

   COMMON DRAW_polar_EPHEMRIS_SLICE, slice_widget
   COMMON PLOT_COMPOSER, widgetData
   COMMON ADDINS, plot_types
   COMMON ISREAD, IsReadArr

;-- close the window ? --------------------------------------------------------
   IF KEYWORD_SET(CLOSE) THEN BEGIN
      IF XREGISTERED('USER_VIEW_EPHEMERIS_polar_CAMMICE') THEN $
         IF N_ELEMENTS(slice_widget) GT 0 THEN BEGIN
	    IF slice_widget.base NE 0 THEN $
  	       WIDGET_CONTROL, slice_widget.base, /DESTROY
            slice_widget.base=0
         ENDIF
      RETURN
   ENDIF

;-- if the window exists, bring it in front. Else, create it ------------------
   IF NOT XREGISTERED('USER_VIEW_EPHEMERIS_polar_CAMMICE') THEN BEGIN
      Slice_Base=WIDGET_BASE(COLUMN=1, $
		GROUP=widgetData.main_base, $
		title='slice-window')
      dw_Slice  =WIDGET_DRAW(Slice_Base, XSIZE=500, YSIZE=500)
      lbl_slice1=WIDGET_LABEL(Slice_Base, VALUE=strcenter('',80))
      lbl_slice2=WIDGET_LABEL(Slice_Base, VALUE=strcenter('',80))
  
      Butt_Base=WIDGET_BASE(Slice_Base, ROW=1, /FRAME)
      pb_help=WIDGET_BUTTON(Butt_Base, $
                              Value=strcenter('Help', 12), $
			      UVALUE='pb_help')
      pb_done=WIDGET_BUTTON(Butt_Base, $
                              Value=strcenter('Close', 12), $
			      UVALUE='pb_done')
      slice_widget={USER_VIEW_EPHEMERIS_polar_CAMMICE, $
		base:slice_base, $
	        dw_slice:dw_slice, $
		lbl_slice1:lbl_slice1, $
		lbl_slice2:lbl_slice2, $
	        panelNr:panelNr, $
		lastPanelDrawn:-1}
      WIDGET_CONTROL, Slice_Base, /REALIZE
      XMANAGER, 'USER_VIEW_EPHEMERIS_polar_CAMMICE', Slice_Base
   ENDIF

   thisPlot=widgetData.plotsDrawn(panelNr)

;--- now, we need to set the get data flag-------------------------------------
;    but we will set the flag, if the panel the user clicked is not the
;    panel that he last clicked into (in this case, the data is already read).
   
   IF slice_widget.lastPanelDrawn NE panelNr THEN READDATA=1 $
                                                ELSE READDATA=0
   slice_widget.lastPanelDrawn = panelNr
 
;--- prepare the drawing widget for plotting ----------------------------------
   SET_PLOT, 'X'  
   WIDGET_CONTROL, slice_widget.dw_slice,  GET_VALUE=drawIndex
   WSET, drawIndex

;--- store the current coordinate system --------------------------------------
   old_x = !x
   old_y = !y
   old_p = !p

; ---  Call add-in routine for slice  -----------------------------------------
   result=strpos(plot_types,strchminus(thisPlot.panelKind),0)
   typeindex=where(result eq 0,count)
   IF count ne 0 then BEGIN
      drawslice_call='draw_ephemeris_'+strchminus(thisPlot.panelKind)
      drawslice_prmt='panelNr, seconds, yValue, READDATA=READDATA'
;      print,drawslice_call,', ',drawslice_prmt
      callgood=execute(drawslice_call+', '+drawslice_prmt)
;      print,callgood
      IF callgood then goto, jumpslice
   ENDIF 

   jumpslice:   

; --- restore the coordinate-system -------------------------------------------
   !x = old_x
   !y = old_y
   !p = old_p
END
