forward_function mesa_keV

;------------------------------------------------------------------------------
; File: CMICSCAL.PRO    Calibration Routines for CAMMICE MICS Rates Scalers   
; Revision: 9/10/97     J. L. Roeder   
;------------------------------------------------------------------------------
; Routines:   
;       def_mrtseff     Define the mics rates efficiency structure   
;       rd_cmics_cals   Reads efficiencies into common block from file
;       cmics_conv_fact Extract the conversion factors for a particular channel
;       conv_cmics_flux Converts counting rate energy spectra to fluxes   
;       conv_cmics_pad  Converts counting rate pitch angle dists to fluxes   
;------------------------------------------------------------------------------
function def_cmicseff,  dummy   
   
;       Defines mics efficiency structure   
   
cmicseff = {eff_cmics,          $   
        fsr : fltarr(32),       $   
        dcr_h : fltarr(32),     $   
        dcr_he1 : fltarr(32),   $   
        dcr_he2 : fltarr(32),   $   
        dcr_o1 : fltarr(32),    $   
        tcr : fltarr(32),       $   
        mss : fltarr(32),       $   
        proton:fltarr(32),      $   
        alpha : fltarr(32),     $   
        r0 : fltarr(32),        $   
        r1 : fltarr(32),        $   
        r2 : fltarr(32),        $   
        r3 : fltarr(32),        $   
        r4 : fltarr(32),        $   
        r5 : fltarr(32),        $   
        r6 : fltarr(32),        $   
        r7 : fltarr(32),        $   
        r8 : fltarr(32),        $   
        r9 : fltarr(32),        $   
        r10 : fltarr(32),       $   
        r11 : fltarr(32),       $   
        r12 : fltarr(32),       $   
        r13 : fltarr(32),       $   
        r14 : fltarr(32),       $   
        r15 : fltarr(32)        $   
        }   
   
return,  cmicseff   
end   
;------------------------------------------------------------------------------
pro rd_cmics_cals   
   
;       Read CAMMICE MICS Rate Scaler calibrations from disk file   
   
; Inputs: None   
; Outputs: In common block cmics_cals   

; BLOCK cmics_cals holds conversion factors for CAMMICE MICS rates to fluxes

common cmics_cals,  $   ; Holds conv. factors CAMMICE MICS rates to fluxes
   eoq,		   $   ; Energy per charge in keV/e for 32 channels
   geom,           $   ; Geometric factor in cm2 ster
   area,           $   ; Detector area in cm2
   deltax,         $   ; Full width angle in x direction
   deltay,         $   ; Full width angle in y direction
   de_over_e,      $   ; Relative energy width of ESA steps
   effic               ; Eff. struc for 32 steps for 4 singles + 15 chans

if n_elements(effic) eq 0 then begin   
         
;	Get energy per charge of esa steps
	eoq = mesa_keV( indgen(32))

;       Get path to calibration files 
        cals_path=papco_getenv('PAPCO_MODULES')+'polar_cammice/plot_read/'
        prog_path='' & data_path=''
           
        fn = cals_path + 'cmicscal.dat'   
        openr, unit, fn, /GET_LUN   
        print, 'Reading ', fn   
   
;       Read header label   
        label = ' '   
        readf, unit, label   
        print, label  
   
;       Efficiency structure   
        effic = def_cmicseff()   
   
;       Read area, angles and dE/E   
        area = 0. 
        deltax = 0. 
        deltay = 0. 
        de_over_e = 0. 
        for i = 0, 2 do readf, unit, label   
        readf, unit, area, deltax, deltay, de_over_e   
        geom = area * (deltax * !Pi/180.) * (deltay * !Pi/180.)   
           
;       Read efficiencies for FSR    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.fsr = a   
   
;       Read efficiencies for DCR as protons
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.dcr_h = a   
   
;       Read efficiencies for DCR as He+
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.dcr_he1 = a   
   
;       Read efficiencies for DCR as He++
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.dcr_he2 = a   
   
;       Read efficiencies for DCR as O+
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.dcr_o1 = a   
   
;       Read efficiencies for TCR   
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.tcr = a   
   
;       Read efficiencies for MSS   
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.mss = a   
   
;       Read efficiencies for PROTON Channel   
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.proton = a   
   
;       Read efficiencies for ALPHA Channel   
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.alpha = a   
   
;       Read efficiencies for R0 (H+)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r0 = a   
   
;       Read efficiencies for R1 (He+)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r1 = a   
   
;       Read efficiencies for R2 (He++)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r2 = a   
   
;       Read efficiencies for R3 (O<+2)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r3 = a   
   
;       Read efficiencies for R4 (?)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r4 = a   
   
;       Read efficiencies for R5 (Low Energy H)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r5 = a   
   
;       Read efficiencies for R6 (Low Energy He)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r6 = a   
   
;       Read efficiencies for R7 (Low Energy O)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r7 = a   
   
;       Read efficiencies for R8 (?)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r8 = a   
   
;       Read efficiencies for R9 (Low Energy H)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r9 = a   
   
;       Read efficiencies for R10 (Low Energy He)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r10 = a   
   
;       Read efficiencies for R11 (Low Energy O)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r11 = a   
   
;       Read efficiencies for R12 (?)   
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r12 = a   
   
;       Read efficiencies for R13 (?)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r13 = a   
   
;       Read efficiencies for R14 (?)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r14 = a   
   
;       Read efficiencies for R15 (?)    
        for i = 0, 2 do readf, unit, label   
        a = fltarr( 32)   
        readf, unit, a   
        effic.r15 = a   
   
        free_lun, unit   
   
endif   
   
end   
;------------------------------------------------------------------------------
pro mrts_chan_ident, number, ident

;	Looks up identifier string from sectored rates channel number

; Inputs:
;	number		Channel number, 0-10

; Outputs:
;	ident		Identifier string for sectored rates channel

ids = ['R0', 'R1', 'R2', 'R3', 'R4', 'R5', 'R6', 'R7', 'R8', 'DCR_H', 'FSR']
if (number ge 0) and (number le 10) then ident = ids( number) else ident = ''

end
;------------------------------------------------------------------------------
pro mtots_chan_ident, number, ident

;	Looks up identifier string from spin totals rates channel number

; Inputs:
;	number		Channel number, 0-10

; Outputs:
;	ident		Identifier string for spin totals rates channel

ids = ['R0', 'R1', 'R2', 'R3', 'R4', 'R5', 'R6', 'R7', 'R8', $
	'R9', 'R10', 'R11', 'R12', 'R13', 'R14', 'R15']
if (number ge 0) and (number le 15) then ident = ids( number) else ident = ''

end
;------------------------------------------------------------------------------
pro msing_chan_ident, number, ident

;	Looks up identifier string from parallel/perp singles rates 
;	channel number

; Inputs:
;	number		Channel number, 0-10

; Outputs:
;	ident		Identifier string for singles rates channel

ids = ['TCR', 'MSS', 'PROTON', 'ALPHA']
if (number ge 0) and (number le 3) then ident = ids( number) else ident = ''

end
;------------------------------------------------------------------------------
pro cmics_conv_fact, ident, factor   
   
;       Extract the conversion factors from count rate to flux    
;       for a particular channel   
   
; Inputs:   
;       ident   Channel identifier string, ie 'r1' or 'dcr'   
   
;       factor  Multiplicative factor to convert counts/s to diff flux
;		for 32 steps
   
common cmics_cals,  $
   eoq,		   $   ; Energy per charge in keV/e for 32 channels
   geom,           $   ; Geometric factor in cm2 ster
   area,	   $   ; Detector area in cm2
   deltax,         $   ; Full width angle in x direction
   deltay,         $   ; Full width angle in y direction
   de_over_e,      $   ; Relative energy width of ESA steps
   effic               ; Efficiency structure for 32 steps for 15 + 4 channels
   
        ;Read calibration file for first execution   
if n_elements( eoq) ne 32L then rd_cmics_cals   
   
;	String array with tag names from efficiency structure
tags = tag_names( effic)

;	Index of required tag in structure
ichan = where( tags eq ident)
if ichan(0) ne -1L then ichan = ichan(0) else $
	stop,'Error cmics_conv_fact: Chan ID not found in structure = ',ident

;       Extract efficiency from structure using channel number as a tag index   
eta = effic.(ichan)   

;	Charge state for each species
charges = [1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 2, 1, 3, 1, 1, 1, 1, 1, 1, 1, $
	1, 1, 1, 1]
charge = charges( ichan)

;	Conversion factor from counts/s to particle/(s cm2 sr keV)   
factor  =  (eta gt 0.) * (1./(geom * de_over_e * (eta>1.e-20) * eoq * charge))

end   
;------------------------------------------------------------------------------
pro conv_cmics_spect, ident, cnt_rate, flux
   
;       Convert MICS counting rate spectra to differential fluxes   
   
; Inputs:   
;       ident 		Channel ID string   
;       cnt_rate        Array of spectra in counts per second   
   
; Outputs:   
;       flux            Array of spectra in differential number fluxes   

;	Get conversion factors from common block structures   
cmics_conv_fact, ident, factor   
   
;       Number of spectra to process   
s = size(cnt_rate)   
if s(0) eq 1 then nspect = 1 else nspect = s(2)   
   
flux = fltarr(32, nspect)   
for ispect = 0,  nspect - 1 do begin   
        flux(0, ispect) = cnt_rate(*, ispect)*factor
endfor   
   
end   
;------------------------------------------------------------------------------
pro conv_cmics_pad, ident, iesa, cnt_rate, flux   
   
;       Convert CAMMICE MICS counting rate pitch angle distribution 
;	to differential fluxes   
   
; Inputs:   
;       ident           Channel ID string   
;       iesa            ESA step number   
;       cnt_rate        Array of pitch angle distributions in counts/sec   
   
; Outputs:   
;       flux            Array of pitch angle distributions in differential    
;                       fluxes   
   
;	Get conversion factors from common block structures   
cmics_conv_fact, ident, factor   

;	Only one ESA step used
if (iesa ge 0) and (iesa le 31) then fact = factor( iesa) else $
	stop,'Error conv_cmics_pad: Out of range esa step number = ',iesa
   
flux = cnt_rate * fact    
   
end   

;------------------------------------------------------------------------------
;pro to get paths for inclusion in PAPCO

pro path_lz, prog_path, data_path, cals_path   

  cals_path=papco_getenv('PAPCO_MODULES')+'polar_cammice/plot_read/'
  prog_path='' & data_path=''
  
end

;------------------------------------------------------------------------------
;pro to getenergy per charge od esa steps

function mesa_keV, inarr
  
  esa=[  1.0,  1.3,  1.8,  2.4,  3.2,  4.2,  5.6,  7.4,  9.9, 13.2, 17.5, $
        23.3, 30.9, 41.1, 54.7, 72.8, 80.3, 89.7,100.1,111.7,124.7,139.1, $
       155.3,173.3,193.4,215.9,241.0,268.9,300.2,335.0,373.9,417.3 ]
     
  return, esa
  
end
