;******************************************************************************
;* PROCEDURE:     
;*      p_polar_cammice_spec, panel, plotinfo, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in PAPCO.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*   plotinfo   the full; plotinfostruct from panel editor
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_polar_cammice, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;******************************************************************************
pro p_polar_cammice_spec, panel, plotinfo, OUTPUT=OUTPUT, $
                 OVERPLOT=OVERPLOT, $
                 PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                 SUBTABLE=SUBTABLE, $
                 _EXTRA=extra_par

; The following common blocks are needed:
COMMON mjdt, mjdt_start, mjdt_end    ;time limit in mjdt
COMMON yscale, yscl		     ;man/auto yscaling
COMMON zscale, zscl		     ;man/auto zscaling
COMMON shift_label, down             ;for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_polar_cammice. 
COMMON  polar_cammice_data, input_header, input_data
COMMON  cmics_cals, $   ; Holds conv. factors CAMMICE MICS rates to fluxes
   eoq,	            $   ; Energy per charge in keV/e for 32 channels
   geom,            $   ; Geometric factor in cm2 ster
   area,            $   ; Detector area in cm2
   deltax,          $   ; Full width angle in x direction
   deltay,          $   ; Full width angle in y direction
   de_over_e,       $   ; Relative energy width of ESA steps
   effic                ; Eff. struc for 32 steps for 4 singles + 15 chans
; read in callibration structures.
rd_cmics_cals      

;commons for data "made" by plot routine, and error block
COMMON polar_cammice_slice, time, zmat, yarr, fc, utitle, uztit, uytitle
COMMON get_error, get_err_no, get_err_msg
      
IF keyword_set(OUTPUT) THEN output = OUTPUT ELSE output = 0
   
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

species = plotinfo.ioptions(0)
data_type = plotinfo.ioptions(5)
plot_type = plotinfo.ioptions(8)
type = plotinfo.ioptions

; set labeling for type selected   
CASE input_header.DATATYPE OF
    'POLAR CAMMICE SpinAv': BEGIN
        utitle = input_header.DATATYPE + '!C'+ $
          strmid(input_header.SUBTYPE(species), $
                 strlen(input_header.DATATYPE)+1, 10)
        uztit = input_header.IDL_DATA_UNITS
        uytitle = input_header.EK_UNITS
        zmat = transpose(input_data.data(*,species))
        yarr = input_header.CH_POSITIONS 
        yst=yarr(0,0)*0.95  &  yen = yarr(input_header.NBANDS-1, 1)*1.05
        data_type = 3
        ; set indices for some data values, needed later   
        idx_zero=where(zmat eq 0,   c_zero)
        idx_bad =where(zmat eq -1,  c_bad)
        idx_none=where(zmat eq -999,c_none)
    END     
    ELSE: BEGIN
        CASE  species OF 
            0:begin         
                utitle='DCR (H!U+!N)'  &  ident='DCR_H' & ch_idx = 7
            end
            1:begin
                utitle='H!U+!N'        &  ident='R0' & ch_idx = 0
            end
            2:begin
                utitle='He!U+!N'       &  ident="R1" & ch_idx = 1
            end
            3:begin
                utitle='He!U+2!N'      &  ident="R2" & ch_idx = 2
            end
            4:begin
                utitle='O!U<+3!N'      &  ident="R3" & ch_idx = 3
            end
            5:begin
                utitle='O!U>+2!N'      &  ident="R4" & ch_idx = 4
            end
            6:begin
                utitle='CN!U<+3!N'     &  ident="R5" & ch_idx = 5
            end
            7:begin
                utitle='O!U+all!N'    &  ident="R8" & ch_idx = 6
            END 
        ENDCASE
     END 
ENDCASE 

index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)
if c ne 0 then data=input_data(index) ELSE GOTO, noplot   
xut1=0  &  xut2=tai_2-tai_1
 
;make the plot arrays according to data type - spin av. spectra, pitch
CASE data_type OF 
    
0: BEGIN ;spin averaged spectra
    utitle='POLAR CAMMICE SpinAv!C'+utitle  
    
    ;if yscale is set linear, use ESA steps, for log use ESA energies.  
    r= where(data(*).esa lt 255)
    esa_steps= max(data(r).esa )+1
    if yscl(panel(0),3) eq 0 then begin
        yminarr=findgen(esa_steps) & ymaxarr=yminarr+1
        uytitle='ESA step'
    endif else begin
        esa=eoq(0: esa_steps-1)
        yminarr = esa-esa*de_over_e & ymaxarr = esa+esa*de_over_e
        uytitle='E/Q (keV)'
    endelse  

    ; make yarray for channels   
    yarr=fltarr(n_elements(yminarr),2)
    yarr(*,0)=yminarr  &  yarr(*,1)=ymaxarr
    yst=yarr(0,0)*0.95  &  yen=yarr(n_elements(yminarr)-1,1)*1.05
   
    ; make plot array. 
    ; -999 indicates no data   - will be plotted grey
    ; -1 bad data              - will be plotted white
    ; 0 is zero counts in bin  - will be plotted black

    if type(2) then SAMPLES=1 else SAMPLES=0
    if type(4) then DETAIL=1 else DETAIL=0
    polar_cammice_m_plotarray, data, ch_idx, SAMPLES=SAMPLES, DETAIL=DETAIL
    time = time-tai_1

    ; set to "nodata flag" those ranges not covered by given channels
    case type(0) of
         0: a=0
         1: a=0
         2: zmat(*,0:10)=-999
         3: a=0
         4: zmat(*,0:12)=-999  
         5: zmat(*,19:n_elements(yminarr)-1)=-999  
         6: a=0
         7: a=0
    endcase  
   
    ; set indices for some data values, needed later   
    idx_zero=where(zmat eq 0,   c_zero)
    idx_bad =where(zmat eq -1,  c_bad)
    idx_none=where(zmat eq -999,c_none)
   
    ; select counts or calibrated flux   
    if type(2) then begin
        uztit='# of samples'
        fc=fltarr(32)  &  fc(*)=1
    endif else begin  
        if type(1) eq 0 then begin
            uztit='counts/s' 
            fc=fltarr(32)  &  fc(*)=1
        endif else begin
            uztit='[cm!e-2!ns!e-1!nsr!e-1!nkeV!e-1!n]'
            cmics_conv_fact, ident, fc     ;gets array(32) of conv. factors
            for i = 0,n_elements(time)/2-1 DO $
              zmat(i,*)=zmat(i,*)*fc(0:esa_steps-1)
        endelse  
    endelse  

    ; keep original array for possible output to file.
    if c_none ne 0 then zmat(idx_none) = -999
    orig_zmat=zmat 

END 

1: BEGIN                        ;pitch angle data
    energy = plotinfo.ioptions(7) > 1 < 32
    pangle = plotinfo.ioptions(6)
    n_ch = input_header.NBANDS
    n_pa = input_header.NSECTORS
    n_det = input_header.NDETECTORS
    utitle='POLAR CAMMICE Pitch!C'+utitle  
    ndat=n_elements(data)
    
    time = dblarr(ndat, 2)
    time(*, 0)= data.tai
    time(*, 1)= data.endtai
    time = time-tai_1

    ; select counts or calibrated flux   
    if type(1) eq 0 then begin
        uztit='counts/s' 
        fc=fltarr(32)  &  fc(*)=1
    endif else begin
        uztit='[cm!e-2!ns!e-1!nsr!e-1!nkeV!e-1!n]'
        cmics_conv_fact, ident, fc    ;gets array(32) of conv. factors
    endelse  

    CASE plot_type OF 
        2 : BEGIN               ;spectrum @ pitch
            ;if zscale is linear, use ESA steps, for log use ESA energies.  
            max_esa = input_header.MAX_ESA
            if yscl(panel(0),3) eq 0 then begin
                yminarr=findgen(max_esa) & ymaxarr=yminarr+1
                uytitle='ESA step'
            endif else begin
                esa=eoq
                yminarr=esa(0:max_esa) & ymaxarr=esa(1:max_esa+1)
                uytitle='E/Q (keV)'
            endelse  
            yarr=fltarr(n_elements(yminarr),2)
            yarr(*,0)=yminarr  &  yarr(*,1)=ymaxarr
            yst=yarr(0,0)  &  yen=yarr(n_elements(yminarr)-1,1)               
            pitch = transpose(data.data(0:max_esa,pangle,ch_idx)) 
            count = transpose(data.count(0:max_esa,pangle,ch_idx)) 
            idx1 = where(count GT 0, c)
            IF c NE 0 THEN BEGIN
                zmat = fltarr(ndat, max_esa+1)
                zmat(idx1) = pitch(idx1)/count(idx1)
                for i = 0,ndat-1 DO zmat(i,*)=zmat(i,*)*fc(0:max_esa)
            ENDIF 

            utitle = utitle+'!CPA '+ strcompress( $
              string(input_header.PA_POSITIONS(pangle,0), $
                     input_header.PA_POSITIONS(pangle,1), $
                     format = "(f6.1,'-',f6.1)"), /remove_all)           
        END
        3 : BEGIN               ;pitch @ energy
            yarr = fltarr(n_pa, 2)
            yarr(*, 0) = input_header.PA_POSITIONS(*,0)
            yarr(*, 1) = input_header.PA_POSITIONS(*,1)
            yst = yarr(0,0) & yen=yarr(n_pa-1,1)

            ;if zscale is linear, use ESA steps, for log use ESA energies.  
            max_esa = input_header.MAX_ESA
            if zscl(panel(0),3) eq 0 then begin
                yminarr=findgen(max_esa) & ymaxarr=yminarr+1
            endif else begin
                esa=eoq
                yminarr=esa(0:max_esa) & ymaxarr=esa(1:max_esa+1)
            endelse  

            pitch = transpose(data.data(energy-1,*,ch_idx)) 
            count = transpose(data.count(energy-1,*,ch_idx)) 
            idx1 = where(count GT 0, c)
            IF c NE 0 THEN BEGIN
                zmat = fltarr(ndat, n_pa)
                zmat(idx1) = pitch(idx1)/count(idx1)
                zmat = zmat*fc(energy-1)
            ENDIF 

            utitle = utitle+'!CEK '+ strcompress( $
              string(yminarr(energy-1), ymaxarr(energy-1), $
              format = "(f6.1,'-',f6.1)"), /remove_all)
            uytitle='Pitch Angle (deg)' 
        END
    ENDCASE
    idx_zero = where(pitch eq 0, c_zero)
    idx_bad  = where(pitch eq -1, c_bad)
    idx_none = where(pitch EQ -999, c_none)
END 

ELSE: BEGIN  ;do nothing, data read is processed
    
END  
ENDCASE 

IF output EQ 2 THEN return

panelset,panel	          ;sets the panel position viewport
PAPCO_Set_SubTable, SUBTABLE  ;sets the papco color table index

; set zero, bad and no data to no data flag for autoscaling. 
; Zero data is later set to zmin/2 and bad data to zmin/4 for plotting.
if c_zero ne 0 then zmat(idx_zero) = -999
if c_bad  ne 0 then zmat(idx_bad)  = -999
if c_none ne 0 then zmat(idx_none) = -999 

if (zscl(panel(0),0) eq 1) then begin
    zmin=zscl(panel(0),1) &  zmax=zscl(panel(0),2)
endif else begin
    papco_autorange, zmat, zmin, zmax, $
      log=zscl(panel(0),3), exclude=0, nodata=-999     
    zscl(panel(0),1)=zmin &  zscl(panel(0),2)=zmax
endelse
   
if c_zero ne 0 then zmat(idx_zero) = zmin/2.0
if c_bad  ne 0 then zmat(idx_bad)  = zmin/4.0
;the remainder of -999 in data now correspond to no data.   
      
if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
endif else begin
    yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
endelse
   
if zmax eq zmin then GOTO, noplot
  
papco_y_label, yst, yen, log=yscl(panel(0),3)

; set up extra plot keywords, common ones first. 
; yrange, xrange  are always needed. 
; the rest you can set if you want to override PAPCO defaults.
; See $PAPCO_BASE/PAPCO/papco_draw for the definition of the PAPCO default
; graphic sytem variable settings.
extra_plotPar={xrange:[xut1, xut2], yrange:[yst, yen], zrange:[zmin,zmax], $
               ylog:yscl(panel(0),3), zlog:zscl(panel(0),3),  $
               xtickformat:'noticks', ztitle:uztit, $
               ztickformat:'papco_color_bar_log_ticks'}

; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this rotuine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar   
   
;do color plot and color bar plot. Use the new color plot routine from Iowa.
;set time resolution according to average time seperation
if type(4) then resx=0.5 else resx=20.0
resy=0.2
; special_colors = fltarr(2,n) can be used to insert special colors,
;                              e.g. saturation.  It is an array of pairs,
;                              each pair is (value,color).  Where
;                              value is found in zmat, use display
;                              color.
  
; set all zero values to black, all bad data values to white.
COMMON papco_color_names
special_colors=fltarr(2,2)
special_colors(0,0)=zmin/2  &  special_colors(1,0)=black
special_colors(0,1)=zmin/4  &  special_colors(1,1)=white

papco_plot_colorspec, zmat, time, yarr, nodata=-999, resx = resx, resy =resy, $
     special_colors=special_colors, xstyle=5, ystyle=5, _extra=extra_plotPar
papco_color_bar, _extra=extra_plotPar
  
; plot y-axis label at left of plot. Use scalable routine!  
left_side_label,panel,uytitle,/rot90  
  
; store the coordinate information into common block
plotted_x = !x  &  plotted_y = !y  

; plot info right of the panel
right_side_label,panel,utitle,/rot90
   
; check if data needs to be output to file. As a minimum, the actual
; data array zmat and time array time need to be specified. 
; Use the optional keywords to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.  
;      ONECOUNT      and (ch) array giving the one count level for the data  
if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data out to file', /cont
    description='POLAR CAMMICE - Energy spectra'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=-999
    onecount=fc(0:esa_steps-1)
    papco_write_data, time, orig_zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata, ONECOUNT=onecount
endif  
   
return

;jump to here if no data to plot 
noplot: 
  get_err_no = 10
  get_err_msg = 'No data to plot'
  papco_Draw_ErrorPanel, panel, plotinfo.panelkind+': ' + get_err_msg
  message, get_err_msg, /cont
  plotted_x = !x  &  plotted_y = !y
 
END  


;******************************************************************************
; If keyword SAMPLES is set return array of # of samples per spectrum
; If keyword DETAIL is set return a full high res array with spectra
; NOT averaged into a single spectrum at a single time!
pro polar_cammice_m_plotarray, in_data, ch_idx, SAMPLES=SAMPLES, DETAIL=DETAIL
  
COMMON polar_cammice_slice, time, zmat, yarr, fc, utitle, uztit, uytitle
  
if keyword_set(SAMPLES) then message,'making plot array - samples...',/cont $
                        else message,'making plot array - data...', /cont

ndat = n_elements(data)
r= where( in_data(*).esa lt 255, c) ;255 bad data flag for esa
IF c EQ 0 THEN return

data = in_data(r)
ndat = n_elements(data)
esa_steps= max( data(r).esa ) + 1  

if keyword_set(DETAIL) then BEGIN
    message,'no assembly into spectra',/cont 
    time = data.tai
    zmat=fltarr(ndat,esa_steps) & zmat(*) = -1
    FOR i = 0, ndat-1 DO BEGIN
        esa_step = data(i).esa
        IF esa_step EQ 255 THEN continue
        zmat(i, esa_step) = data(i).data(ch_idx)/data(i).acct(ch_idx)
    ENDFOR
    return 
ENDIF 

;data is in time ordered esa steps. Need to find times of start of esa
;step cycle - which is typically esa step 0. Then assemble all esa
;steps at different times into one spectrum.

spec = fltarr(esa_steps) & acct = fltarr(esa_steps)
idx = where(data.esa EQ 0, c)

IF c NE 0 THEN BEGIN
    time = dblarr(c, 2) & zmat=fltarr(c,esa_steps) & zmat(*) = -999
    steps = idx(1)-idx(0)
    FOR i = 0l, c-1 DO BEGIN
        end_idx = (idx(i)+steps-1) <  (ndat-1)
        IF idx(i) EQ end_idx THEN continue
        time(i, 0) = data(idx(i)).tai
        time(i, 1) = data(end_idx).tai
        IF data(end_idx).tai-data(idx(i)).tai GT 240 THEN continue

        ;print, tai2utc(time(i, 0), /ecs),  ' - ',  tai2utc(time(i, 1), /ecs)

        cycle_data = data(idx(i):end_idx).data(ch_idx)
        cycle_acct = data(idx(i):end_idx).acct
        cycle_esa  = data(idx(i):end_idx).esa
        acct(*) = 1.0
        spec(*) = -999  ; set all as no data initially
        FOR j = 0, esa_steps-1 DO BEGIN
            idx1 = where(cycle_esa EQ j, c1)
            IF c1 NE 0 THEN BEGIN 
                dat = cycle_data(idx1)
                acc = cycle_acct(idx1)
                idx2 = where(dat NE 0, c2)
                IF c2 NE 0 THEN BEGIN
                    spec(j) = total(dat(idx2))
                    acct(j) = total(acc(idx2))
                ENDIF 
            ENDIF
        ENDFOR 
        idx3 = where(spec NE -999, c3)
        IF c3 NE 0 THEN zmat(i, *) = spec/acct
    ENDFOR
ENDIF    

END   
