;******************************************************************************
;* file p_polar_cammice_spec.pro
;*
;* this file is a template for a plot routine which can be used by papco.
;* it includes all the periphery needed, the user simply has to add in
;* the actual plot statement for his/her kind of data.
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_polar_cammice_spec, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in PAPCO.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_polar_cammice, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;******************************************************************************
pro p_polar_cammice_spec, panel, type, OUTPUT=OUTPUT, $
                 OVERPLOT=OVERPLOT, $
                 PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                 SUBTABLE=SUBTABLE, $
                 _EXTRA=extra_par
    
; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Line styles are also not useful. So the keywords 
; OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,_EXTRA=extra_par
; are not used at all here, but are in the call for consistency...

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_polar_cammice_line, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags whichare meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type, swittch. Replace type in the call with whatever variables
; you need, or add more

; Comments here should include an explanation of the plot options used
; in type, swittch. Replace type in the call with whatever variables
; you need, or add more

; The following common blocks are needed:
  common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
  common yscale, yscl			;man/auto yscaling
  common zscale, zscl			;man/auto zscaling
  common shift_label, down             ;common for x-axis label shifting
  common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_polar_cammice. 
  common polar_cammice_data, input_header, input_data
  common cmics_cals,  $   ; Holds conv. factors CAMMICE MICS rates to fluxes
     eoq,	      $   ; Energy per charge in keV/e for 32 channels
     geom,            $   ; Geometric factor in cm2 ster
     area,            $   ; Detector area in cm2
     deltax,          $   ; Full width angle in x direction
     deltay,          $   ; Full width angle in y direction
     de_over_e,       $   ; Relative energy width of ESA steps
     effic                ; Eff. struc for 32 steps for 4 singles + 15 chans
   
  common polar_cam_slice, time,zmat,yminarr,ymaxarr,factor,utitle 
                       ;for slice routine
      
;if panel(2) is set negative, only make the plot array, do not output no
;plot! This is used by the slice routine to make new plot array when
;swittching panels between cammice plots.   
  if panel(2) lt 0 then begin
    noplot=1
  endif else begin
    panelset,panel	          ;sets the panel position viewport
    PAPCO_Set_SubTable, SUBTABLE  ;sets the papco color table index
    noplot=0
  endelse
   
; read in callibration strcutures.
  rd_cmics_cals   
  
; set labeling for type selected   
   case type(0) of
     0:begin         
        utitle='DCR (H!U+!N)'  &  ident='DCR_H'
     end
     1:begin
        utitle='H!U+!N'        &  ident='R0'
     end
     2:begin
        utitle='He!U+!N'       &  ident="R1"
     end
     3:begin
        utitle='He!U+2!N'      &  ident="R2"
     end
     4:begin
        utitle='O!U<+3!N'      &  ident="R3"
     end
     5:begin
        utitle='O!U>+2!N'      &  ident="R4"
     end
   endcase   
   utitle='POLAR!CCAMMICE!C'+utitle   
   
; set the begin end end times. PAPCo convention is to use running secconds
; starting with the no of seconds since the begiing of the start day.
   xut1=mjdt_start.t                       
   xut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t
   
; restrict input data to range +/- 3 min to catch full spectra - this speeds
; up constructiuon of data array.
   last_time=input_data(n_elements(input_data)-1).time
   index=where((input_data.time ge (xut1-180.0)>0.0) and $
               (input_data.time le (xut2+180.0)<last_time),count)
   if count ne 0 then this_input_data=input_data(index)

; if yscale is set linear, use ESA steps, for log use ESA energies.  
  r= where( this_input_data(*).esa lt 255 )
  esa_steps= max( this_input_data(r).esa ) + 1  
  if yscl(panel(0),3) eq 0 then begin
    yminarr=findgen(esa_steps)
    ymaxarr=yminarr+1
    uytitle='ESA step'
  endif else begin
    esa=eoq
    yminarr=esa(0:esa_steps-1)
    ymaxarr=esa(1:esa_steps)
    uytitle='E/Q (keV)'
  endelse  
   
; make yarray for channels   
  yarr=fltarr(n_elements(yminarr),2)
  yarr(*,0)=yminarr  &  yarr(*,1)=ymaxarr
  yst=yarr(0,0)  &  yen=yarr(n_elements(yminarr)-1,1)   
   
; make plot array. 
; -999 indicates no data   - will be plotted grey
; -1 bad data              - will be plotted white
; 0 is zero counts in bin  - will be plotted black
  if type(2) then SAMPLES=1 else SAMPLES=0
  if type(4) then DETAIL=1 else DETAIL=0
  p_polar_cammice_spec_m_plotarray, this_input_data, type, $
    SAMPLES=SAMPLES,DETAIL=DETAIL
      
; restrict plot data to actual time range requested -    
; this makes for faster zooming
  index=where((time ge xut1) and (time le xut2),count)
  if count gt 2 then begin
    time=time(index)  &  zmat=zmat(index,*)
  endif  
   
; set to "nodata flag" those ranges not covered by given channels
  case type(0) of
     0: a=0
     1: a=0
     2: zmat(*,0:10)=-999
     3: a=0
     4: zmat(*,0:12)=-999  
     5: zmat(*,19:n_elements(yminarr)-1)=-999  
  endcase  
   
; set indices for some data values, needed later   
  idx_zero=where(zmat eq 0,   c_zero)
  idx_bad =where(zmat eq -1,  c_bad)
  idx_none=where(zmat eq -999,c_none)
   
; select counts or calibrated flux   
  if type(2) then begin
    uztit='# of samples'
    factor=fltarr(32)  &  factor(*)=1
  endif else begin  
    if type(1) eq 0 then begin
      uztit='counts/s' 
      factor=fltarr(32)  &  factor(*)=1
    endif else begin
      uztit='[cm!e-2!ns!e-1!nsr!e-1!nkeV!e-1!n]'
      cmics_conv_fact, ident, factor      ;reads the 32 array of conv. factors
      for i=0,n_elements(time)-1 do zmat(i,*)=zmat(i,*)*factor(0:esa_steps-1)
    endelse  
  endelse   
 
; check for noplot and return if no plot is required
  if noplot then return
   
; keep original array for possible output to file.
  if c_none ne 0 then zmat(idx_none) = -999
  orig_zmat=zmat 
   
; set zero data, bad data and no data flags all to no data flag for
; autoscaling.  
; Zero data is later set to zmin/2 and bad data to zmin/4 for plotting.
  if c_zero ne 0 then zmat(idx_zero) = -999
  if c_bad  ne 0 then zmat(idx_bad)  = -999
  if c_none ne 0 then zmat(idx_none) = -999
      
; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling
  if (zscl(panel(0),0) eq 1) then begin
    zmin=zscl(panel(0),1) &  zmax=zscl(panel(0),2)
  endif else begin
    papco_autorange, zmat, zmin, zmax, $
      log=zscl(panel(0),3), exclude=0, nodata=-999     
    zscl(panel(0),1)=zmin &  zscl(panel(0),2)=zmax
  endelse
   
  if c_zero ne 0 then zmat(idx_zero) = zmin/2.0
  if c_bad  ne 0 then zmat(idx_bad)  = zmin/4.0
  ;the remainder of -999 in data now corresponf to no data.   
      
  if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
  endif else begin
    yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
  endelse
   
  if zmax eq zmin then begin
    plotted_x = !x  &  plotted_y = !y 
    message,'No valid data points', /cont
    return
  endif    
  
; option to automatically calculate ytick numbers in order to avoid to
; many ticks if panels get too narrow. Maximum is set to 10. You may
; leave this out and allow plot to use the built-in defaults.
  ytickno=50*(!p.position(3)-!p.position(1))
  if ytickno gt 10 then ytickno=10
      
; set up extra plot keywords, common ones first. 
; yrange, xrange  are always needed. 
; the rest you can set if you want to override PAPCO defaults.
; See $PAPCO_BASE/PAPCO/papco_draw for the definition of the PAPCO default
; graphic sytem variable settings.
  extra_plotPar={xrange:[xut1, xut2], yrange:[yst, yen], zrange:[zmin,zmax], $
                 ylog:yscl(panel(0),3), zlog:zscl(panel(0),3),  $
                 xtickformat:'noticks', ztitle:uztit, $
                 ztickformat:'papco_color_bar_log_ticks'}

; add keyword structure set here with the one passed in
  extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this rotuine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
  down=0
  papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar   
   
;do color plot and color bar plot. Use the new color plot routine from Iowa.
;set time resolution according to average time seperation
  if type(4) then resx=0.5 else resx=20
  resy=0.05
; special_colors = fltarr(2,n) can be used to insert special colors,
;                              e.g. saturation.  It is an array of pairs,
;                              each pair is (value,color).  Where
;                              value is found in zmat, use display
;                              color.
  
; set all zero values to black, all bad data values to white.
  COMMON papco_color_names
  special_colors=fltarr(2,2)
  special_colors(0,0)=zmin/2  &  special_colors(1,0)=black
  special_colors(0,1)=zmin/4  &  special_colors(1,1)=white
  
  papco_plot_colorspec,zmat,time,yarr, nodata=-999, resx=resx, resy=resy, $
     special_colors=special_colors, xstyle=5,_extra=extra_plotPar
  papco_color_bar, _extra=extra_plotPar
  
; plot y-axis label at left of plot. Use scalable routine!  
  left_side_label,panel,uytitle,/rot90  
  
; store the coordinate information into common block
  plotted_x = !x  &  plotted_y = !y  

; plot info right of the panel. If you want overplots to have their own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION. 
; Since here papco_colorbars are plotted, labels are rotated
; by 90 degrees and then plotted. This leave lttile room for overplot labels:
; consider putting those inside the panel.
  if not keyword_set(overplot) then $
    right_side_label,panel,utitle,/rot90
   
; check if data needs to be output to file. As a minimum, the actual
; data array zmat and time array time need to be specified. 
; Use the optional keywords to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.  
;      ONECOUNT      and (ch) array giving the one count level for the data  
  if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data out to file', /cont
    description='POLAR CAMMICE - Energy spectra'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=-999
    onecount=factor(0:esa_steps-1)
    papco_write_data, time, orig_zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata, ONECOUNT=onecount
  endif  
      
end 


;******************************************************************************
; If keyword SAMPLES is set return array of # of samples per spectrum!
; If keyword DETAIL is set return a full high res array with spectra
; NOT averaged into a single spectrum at a singel time!
;
pro p_polar_cammice_spec_m_plotarray, input_data, type, $
                                      SAMPLES=SAMPLES, DETAIL=DETAIL
  
  common polar_cam_slice, time, zmat, yminarr, ymaxarr, factor, utitle 
  
  if keyword_set(SAMPLES) then $
    message,'making plot array - samples...',/cont else $
    message,'making plot array - data...', /cont
  if keyword_set(DETAIL) then $
    message,'using full time res, no averaging',/cont 
  
  r= where( input_data(*).esa lt 255 )
  esa_steps= max( input_data(r).esa ) + 1  
    
  n=n_elements(input_data) 
  time=fltarr(n)  &  zmat=fltarr(n,esa_steps)
      
  time=input_data.time
  zmat(*,*)= -999          ; set all as no data initially
  for i=0l,n-1 do begin
    esa = input_data(i).esa
    if (esa gt esa_steps-1) then goto,next1
    ;spin averaged counts - need to divide by spin period to get counts/sec
    ;values of -1 indicate bad data, skip that spin!
    one_spin=input_data(i).cnts(type(0),*)      
    index=where(one_spin eq -1,c)
    if c ne 0 then begin
      message,'spin '+varprt(i)+', '+varprt(c)+' points bad out of ' + $
        varprt(n_elements(one_spin)), /cont
      zmat(i,esa)=-1
    endif else begin
      if keyword_set(SAMPLES) then $
        zmat(i,esa)=1 else $
        zmat(i,esa)=total(one_spin) / 5.804
    endelse  
    next1:
  endfor
  
  if keyword_set(DETAIL) then return ;return full detailed data array
  
;work out "full" spectra and their center time. Center time is taken
;as the time between the minima esa steps.
  
  min_esa=min(input_data.esa)
  min_esa_data=zmat(*,min_esa+10)
  index=where(min_esa_data ge -1,c) ;select all data points 
  
  time=fltarr(c-1)  &  zmat_av=fltarr(c-1,esa_steps)
  
  onesp=fltarr(esa_steps)  &  nospe=intarr(esa_steps)
  
  for i=0,c-2 do begin          ;assemble data between min esa steps
    a=index(i)  &  b=index(i+1)-1
    time(i) = (input_data(a).time+input_data(b).time) / 2.0
    for k=0,esa_steps-1 do begin
      
      points=zmat(a:b,k)
      idx=where(points ge -1, c_dat) ; get all data that isn't no data
      if c_dat eq 0 then begin  ; none found
        onesp(k) = -999
        nospe(k)=c_dat
        goto,next_k
      endif   
      points=points(idx)
      nospe(k)=c_dat
      
      idx=where(points ne -1,c_good)           ; select good data only
      if c_good eq 0 then onesp(k) = -1  $     ; all bad
      else onesp(k)=total(points(idx))/c_good  ;average good ones
      
      next_k:
    endfor
    if keyword_set(SAMPLES) then zmat_av(i,*)=nospe $
      else zmat_av(i,*)=onesp
  endfor    
  
  zmat=zmat_av

end  



;******************************************************************************
; If keyword SAMPLES is set return array of # of samples per spectrum!
pro old_p_polar_cammice_spec_m_plotarray, input_data, type, SAMPLES=SAMPLES
  
  common polar_cam_slice, time,zmat,yminarr,ymaxarr,factor,utitle 
  
  if keyword_set(SAMPLES) then $
    message,'making plot array - samples...',/cont else $
    message,'making plot array - data...', /cont
  
  r= where( input_data(*).esa lt 255 )
  esa_steps= max( input_data(r).esa ) + 1  
  
  time=fltarr(4000)
  zmat=fltarr(4000,esa_steps)
  onesp=fltarr(esa_steps)
  spect=0.0
  nospe=intarr(esa_steps)
  sum=fltarr(esa_steps)
  cnt=intarr(esa_steps)
   
  n=-1l                         ;counter for number of spectra
    
  for i=0l,long(n_elements(input_data)-2) do begin
    esa = input_data(i).esa
    if (esa gt esa_steps-1) then goto,next
    ;spin averaged counts - need to divide by spin period to get counts/sec
    ;values of -1 indicate bad data, skip that spin!
    one_spin=input_data(i).cnts(type(0),*)
    index=where(one_spin eq -1,c)
    if c ne 0 then begin
      message,'spin '+varprt(i)+', '+varprt(c)+' points bad out of ' + $
        n_elements(one_spin), /cont
      onesp(esa)=-1
    endif else $
      onesp(esa)=onesp(esa)+total(one_spin) / 5.804
    spect=spect+input_data(i).time
    nospe(esa)=nospe(esa)+1
    print,esa, nospe(esa)
    ;work out time to next point. Steffano's data is orgnized such that the 
    ;bottom energy scan and the top energy scan are assigned common times.
    t_diff=input_data(i+1).time-input_data(i).time
    ;print,i, esa, input_data(i).time, t_diff
    if t_diff gt 8 then begin
    ;if esa eq 0 then begin
      n=n+1                                    ;increment spec counter
      idx=where(nospe ne 0)
      onesp(idx)=onesp(idx)/nospe(idx)         ;divide by samples per esa
      time(n)=spect/total(nospe)               ;find average time for spectrum
      
      if keyword_set(SAMPLES) then begin
        onesp=nospe
        index=where(onesp eq 0,c)
        print,c
      endif  
      zmat(n,*)=onesp(*) 
      
      goto, nnnn
      ;now fill in n-1 spec by averaging bewteen this one and
      ;n-2 spec as n-1 is empty for this energy range
      if n ge 2 then begin
        prev_spec=zmat(n-2,*)
        sum=onesp+prev_spec
        index=where(onesp ne 0,count)
        if count ne 0 then cnt(index)=cnt(index)+1
        index=where(prev_spec ne 0,count)
        if count ne 0 then cnt(index)=cnt(index)+1
        index=where(cnt eq 2,count)
        if count ne 0 then zmat(n-1,index)=sum(index)/float(cnt(index))
      endif
      
      nnnn:
      
      ;reset accumulators for next spectrum
      onesp(*)=0.0             
      spect=0.0
      nospe(*)=0
      sum(*)=0.0
      cnt(*)=0
    endif  
    next:
  endfor  
  
  time=time(0:n)
  zmat=zmat(0:n,*)
  
end  
