;******************************************************************************
;*  NAME:        r_polar_cammice
;*
;*  DESCRIPTION: Routine to read polar_cammice files. Adheres to
;*               PAPCO conventions: Error common block
;*                                  Uses MJDT time
;*                                  PAPCO read philosophy
;*
;*  INPUT:       none (uses common block time from PAPCO)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*                              running example module on it's own
;******************************************************************************
pro r_polar_cammice, plotinfo, ORIG = ORIG

COMMON polar_cammice_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg
COMMON r_polar_cammice, data_dir, archive_dir

get_err_no=0  &   get_err_msg='polar_cammice read succesfully'
data_dir='' &  archive_dir='' & l0_dir = ''

IF plotinfo.ioptions(10) THEN orig = 0 ELSE orig = 1

; check env. variable pointing to data directories
if not papco_check_data_env('POLAR_CAMMICE_L0', PATH=l0_dir) then return
if not papco_check_data_env('POLAR_CAMMICE_DATA', PATH=data_dir) then return
if not papco_check_data_env('POLAR_CAMMICE_NSSDCA', PATH=nssdca_dir) then return

; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0

; clear data arrays for reuse
input_header=0  &  input_data=0

; loop through from start to end time concatonating data files into one array.
time=mjdt_start  &  no_files_read=0  &  day_count=0.0

WHILE mjdt_end.mjd ge time.mjd DO BEGIN
    ;assemble the data file from the time range requested.
    tst = INT2UTC({mjd:time.mjd, time:time.t*1000})
    datestr=string(tst.year,tst.month,tst.day,format='(i4.4,i2.2,i2.2)')

    r_polar_cammice_day_l0, plotinfo,  datestr, ORIG = ORIG

    if get_err_no eq 0 then begin ;a file was read succesfully
      if no_files_read eq 0 then begin ;start the array buffer
        input_data_buf=temporary(input_data)
      endif else begin         ;otherwise concatonate arrays
        input_data_buf=[temporary(input_data_buf),temporary(input_data)]
      endelse
      no_files_read=no_files_read+1
    endif
    day_count=day_count+1
    time.mjd=time.mjd+1                   ; next day
    ; check for any time left on last day - skip if none
    if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) then goto, done
endwhile
done:

if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    input_data=0 & input_data=input_data_buf & input_data_buf=0
endif else begin
    get_err_no=1
    return
endelse

return

END


;------------------------------------------------------------------------------
; reads original cammice level 0 files using the CAMLZ routines from
; Jim Roeder, Aerospace Corp. Needs to have these added to papco path on
; startup!
;
; Also uses Jim's routines to get pitch angle data. Might change this
; to use actual mag field data at some stage!
PRO r_polar_cammice_day_l0, plotinfo, datestr, ORIG = ORIG

COMMON polar_cammice_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
COMMON r_polar_cammice, data_dir, archive_dir

; check env. variable pointing to data directories
if not papco_check_data_env('POLAR_CAMMICE_L0', PATH=l0_dir) then return
if not papco_check_data_env('POLAR_CAMMICE_DATA', PATH=data_dir) then return
if not papco_check_data_env('POLAR_CAMMICE_NSSDCA', PATH=nssdca_dir) then return

;if data files are zipped, unzip to PAPCO_DOWNLOAD directory
if not papco_check_data_env('PAPCO_DOWNLOAD', PATH=temp_dir) then return

full_date = strmid(datestr, 0, 8)
date = strmid(datestr, 2, 6)
year = fix(strmid(datestr, 0, 4))
mon = fix(strmid(datestr, 4, 2))
day = fix(strmid(datestr, 6, 2))
sep=papco_get_pathsep() ;get path seperator for operating system

CASE plotinfo.ioptions(5) OF
    0 : plot = 'spin_av'
    1 : plot = 'pitch'
ENDCASE

;set mode for pitch angle data
pa_mode = 0

;try to read data center fully processed files first
IF NOT keyword_set(ORIG) THEN BEGIN
    fln_sav = nssdca_dir + plot+sep+'po_cam_'+full_date+'.idl'
    file = findfile(fln_sav, count = c)
    IF c NE 0 THEN BEGIN
        message, 'restoring ', /cont & print, '  ', fln_sav
        restore, file = fln_sav, verbose = verbose
        return
    ENDIF ELSE BEGIN ;try to get remotely if file not there
        url=papco_getenv('POLAR_CAMMICE_NSSDCA_FTP')
        download_dir=nssdca_dir
        remote_file=plot+sep+'po_cam_'+full_date+'.idl'
        papco_get_remote_data, url, remote_file, download_dir
        IF get_err_no eq 0 THEN BEGIN
            file = findfile(fln_sav, count = c)
            IF c NE 0 THEN restore, file = fln_sav, verbose = verbose
            return
        ENDIF
    ENDELSE
ENDIF

;try to read saveset of L0 data next
fln_sav = data_dir+'savesets'+sep+plot+sep+full_date+'_pol_cam_'+plot+'.idl'
file = findfile(fln_sav, count = c)
IF c NE 0 THEN BEGIN
    message, 'restoring ', /cont & print, '  ', fln_sav
    restore, file = fln_sav, verbose = verbose
    return
ENDIF

;now look for level zero file and read, make saveset
file = findfile(l0_dir+varprt(year)+'/'+date+'*', count = c)
IF c EQ 0 THEN file = findfile(l0_dir+varprt(year)+'/'+'*'+full_date+'*', count = c)
IF c EQ 0 THEN BEGIN
   ;try to get remotely if file not there
    url=papco_getenv('POLAR_CAMMICE_L0_FTP')
    download_dir=l0_dir
    year = strmid(full_date, 0, 4)
    remote_file=year+'/po_lz_cam_'+full_date+'_v01.dat.gz'
    papco_get_remote_data, url, remote_file, download_dir
    IF get_err_no NE 0 THEN BEGIN
        get_err_no = 1 & get_err_msg = 'No CAMMICE L0 file for '+date
        message, get_err_msg, /cont & return
    ENDIF
ENDIF

file = findfile(l0_dir+date+'*', count = c)
IF c EQ 0 THEN file = findfile(l0_dir+varprt(year)+'/'+'*'+full_date+'*', count = c)
unzipfile = ''
IF strmid(file(0), strlen(file(0))-3, 3) EQ 'zip' THEN BEGIN
    zipfile = file(0)
    papco_sepdir, zipfile, p, f, e
    unzipfile = temp_dir+f+'.dat'
    papco_unzip, zipfile, unzipfile, /VERB
    file = [unzipfile]
ENDIF
IF strmid(file(0), strlen(file(0))-2, 2) EQ 'gz' THEN BEGIN
    zipfile = file(0)
    papco_sepdir, zipfile, p, f, e
    unzipfile = temp_dir+f+'.dat'
    papco_gunzip, zipfile, unzipfile, /VERB
    file = [unzipfile]
ENDIF

;call CAMLZ routines from Jim Roeder to read data.
opn_lz, file(0), err              ;open level zero file
IF keyword_set(err) THEN BEGIN
    IF err EQ 'opn_LZ: A Level Zero file is already open.' THEN BEGIN
        cls_lz
        opn_lz, file(0), err
    ENDIF
ENDIF

set_times, [0.0, 86400.0]         ;Times are in UT seconds from start of day
set_mrates, [0, 1, 2, 3, 4, 5, 6, 9] ;select rate scalars
message, 'reading level0 file - slow...', /cont
CASE plot OF    ;read the data
    'spin_av' : get_msrates,times,delta,sectors,esa,chans,cnts,err
    'pitch'   : get_msrts,pa_mode,times,delta,sectors,esa,chans,cnts,pang,err
ENDCASE
cls_lz                          ;close level zero file

IF unzipfile NE '' THEN  BEGIN
    print, 'Deleting '+unzipfile
    file_delete, unzipfile
ENDIF

IF err NE '' THEN BEGIN
    get_err_no = 2 & get_err_msg = err
    message, get_err_msg, /cont & return
ENDIF

n_ch = n_elements(chans)
;get indices of start of spin
spin_st = where(sectors EQ 0, ndat)

IF ndat EQ 0 THEN BEGIN
    get_err_msg = 'No spins starts found'
    message, get_err_msg, /cont
    get_err_no = 8
    return
ENDIF

;decompress all counts
decomprs_cnts,cnts

CASE plot OF
    'spin_av':BEGIN
        message, 'creating spin averaged spectra - slow...', /cont
        ;create input_data, structure for this data
        data = fltarr(n_ch) & data(*) = -999
        dummy = {polar_cammice_data_5, $
                 tai:-1.0d, time:-1.0, esa:255, acct:-999.0, data:data}
        input_data = replicate(dummy, ndat)

        esa_sp = intarr(ndat)
        cnts_sp = fltarr(n_ch, ndat) &  cnts_sp(*, *) = -999

        FOR i = 0, ndat-1 DO BEGIN
            this_spin_idx = indgen(32)+spin_st(i)
            this_spin_times = times(this_spin_idx)

            ;check for any bad times in spin, skip if there are any
            idx = where(this_spin_times EQ -1, c)
            IF c NE 0 THEN BEGIN
                input_data(i).data(*) = -1
                continue
            ENDIF
            ;now find spin averages, spin esa step (is the same for each spin)
            this_spin_t = total(this_spin_times)/32000.0
            msec = fix((this_spin_t-fix(this_spin_t))*1000)
            input_data(i).time = this_spin_t
            this_spin_t = long(this_spin_t)
            hr = this_spin_t / 3600
            min = (this_spin_t-hr*3600) / 60
            sec =  this_spin_t -hr*3600-min*60

            ;convert time to tai
            utc = {YEAR:year, MONTH:mon, DAY:day, $
                   HOUR:hr, MINUTE:min, SECOND:sec, MILLISECOND:msec}
            input_data(i).tai = utc2tai(utc)

            this_spin_esa = esa(this_spin_idx)
            ;check for bad esa values, skip if any
            idx = where(this_spin_esa EQ 255, c)
            IF c NE 0 THEN BEGIN
                input_data(i).data(*) = -1
                continue
            ENDIF ELSE BEGIN
                input_data(i).esa = this_spin_esa(0)
                ;now average counts, acc times
                ;use acc. times (delta) to work out counts/sec
                ;delta is in msec
                FOR ch = 0, n_ch-1 DO BEGIN
                    this_spin_cts = float(cnts(ch,this_spin_idx))
                    this_spin_acc = delta(this_spin_idx)
                    ;lost counts are marked -1, take out
                    idx = where(this_spin_cts NE -1, c)
                    IF c NE 0 THEN BEGIN
                        input_data(i).acct = $
                          total(this_spin_acc(idx))/1000.0 ;seconds
                        input_data(i).data(ch) = $
                          total(this_spin_cts(idx))
                    ENDIF ELSE input_data(i).acct(ch) = -999
                ENDFOR
            ENDELSE
        ENDFOR

        ;create input_header structure for this data
        input_header = {polar_cammice_header_1, $
                DATATYPE     : 'POLAR CAMMICE Spin Averaged', $
                NPOINTS      : ndat, $
                NDETECTORS   : n_ch, $
                NBANDS       : 32, $
                NSECTORS     : 1, $
                DETECTORS    : 0l, $
                TSTART       : input_data(0).time, $
                TSTOP        : input_data(ndat-1).time}

    END

    'pitch': BEGIN ;fold data into pitch angle / energy arrays far all species
        pa_bin = 10.0
        ;set up pitch angle ranges for binning. Always make sure that center
        ;of "middle" bin has 90 degree value in it's center!
        pa_range = fltarr(1, 2) & top = 90.0+pa_bin/2.0 & bot = 90.0-pa_bin/2.0
        pa_range(0, *) = [bot, top] & next_top = 90.0
        WHILE next_top LT 180 DO BEGIN
            next_top = (top+pa_bin) < 180.0 & next_bot = (bot-pa_bin) > 0.0
            pa_range = [transpose([next_bot, bot]), pa_range, $
                        transpose([top, next_top])]
            top = next_top & bot = next_bot
        ENDWHILE

        ;find max esa channel
        r= where(esa lt 255) & max_esa =max(esa(r))

        n_bin = n_elements(pa_range)/2
        pitch = fltarr(32, n_bin, n_ch)
        count = fltarr(32, n_bin, n_ch)
        dummy = create_struct(name = 'POLAR_CAMMICE_PITCH_'+varprt(n_bin), $
                             'tai', 0.0d, 'endtai', 0.0d, $
                             'data', pitch, 'count', pitch )

        message,'binning data into pitch angle ranges (slow)',/cont

        ;do binning in time for each esa cycle!
        esa_idx = where(esa(spin_st) EQ 0, ndat)
        input_data = replicate(dummy, ndat)

        FOR i = 0, ndat-2 DO BEGIN
            ;get start / end indices for data from one esa cycle
            s_idx = spin_st(esa_idx(i)) & e_idx = spin_st(esa_idx(i+1))-1

            ;calculate tai time
            t1 = times(s_idx)/1000.0 & t2 = times(e_idx)/1000.0
            msec = fix((t1-fix(t1))*1000)
            t1 = long(t1)
            hr = t1 / 3600 & min = (t1-hr*3600) / 60 & sec = t1-hr*3600-min*60
            utc = {YEAR:year, MONTH:mon, DAY:day, $
                   HOUR:hr, MINUTE:min, SECOND:sec, MILLISECOND:msec}
            input_data(i).tai = utc2tai(utc)
            msec = fix((t2-fix(t2))*1000)
            t2 = long(t2)
            hr = t2 / 3600 & min = (t2-hr*3600) / 60 & sec = t2-hr*3600-min*60
            utc = {YEAR:year, MONTH:mon, DAY:day, $
                   HOUR:hr, MINUTE:min, SECOND:sec, MILLISECOND:msec}
            input_data(i).endtai = utc2tai(utc)

            ;now do binning. Return cts/sec
            FOR j = s_idx, e_idx DO BEGIN ;loop through esa cycle
                e_ch = esa(j) & IF e_ch EQ 255 THEN continue
                IF sectors(j) EQ 255 THEN continue
                pa = pang(j)
                pa_idx = where(pa LT pa_range(*,1)) & pa_idx = pa_idx(0)
                FOR ch = 0, n_ch-1 DO BEGIN ;loop through species channels
                    input_data(i).data(e_ch,pa_idx, ch) = $
                      input_data(i).data(e_ch,pa_idx, ch) + $
                      cnts(ch, j)/(delta(j)/1000.0)
                    input_data(i).count(e_ch,pa_idx, ch) = $
                      input_data(i).count(e_ch,pa_idx, ch) + 1
                ENDFOR
            ENDFOR
        ENDFOR

        input_header={ $
                DATATYPE     : 'POLAR/CAMMICE/PITCH ANGLES', $
                NPOINTS      : ndat, $
                NDETECTORS   : n_ch, $
                NBANDS       : 32, $
                NSECTORS     : long(n_bin), $
                NLUTS        : 'not used' , $
                DETECTORS    : 0l, $
                TSTART       : input_data(0).tai, $
                TSTOP        : input_data(ndat-1).endtai, $
                CH_POSITIONS : 'get from cal routine', $
                PA_POSITIONS : pa_range, $
                MAX_ESA      : max_esa, $
                TITLE        : 'CAMMICE_PITCH', $
                XTITLE       : 'UT', $
                YTITLE       : 'Pitch Angle', $
                ZTITLE       : 'diff flux', $
                PTITLE       : 'not used'          }

        input_header = create_struct(name='POLAR_CAMMICE_PITCH_'+ $
                                     varprt(n_bin)+'_header', input_header)
    END
ENDCASE

;write data to saveset
message, 'writing '+ fln_sav, /cont
papco_sepdir, fln_sav,p,f,e
if not papco_finddir(p) then papco_create_dir, p

save, file = fln_sav, input_header, input_data, /compress, /verb

END


;------------------------------------------------------------------------------
;routine to outpur data files for NSSDCA data center / Bill Mish, Bob McGuire

PRO w_polar_cammice, DATE_1 =DATE_1, DATE_2 = DATE_2,  $
                     CHECK = CHECK, FORCE = FORCE, IGNORE_PROB = IGNORE_PROB

COMMON mjdt, mjdt_start, mjdt_end
COMMON polar_cammice_slice, time, zmat, yarr, fc, utitle, uztit, uytitle
COMMON get_error, get_err_no, get_err_msg
COMMON yscale, yscl		     ;man/auto yscaling

papco, /set_only

if not papco_check_data_env('POLAR_CAMMICE_L0', PATH=l0_dir) then return

out_dir  = '/n/radbelt/dlnk/polar/cammice/nssdca/spin_av/'
prob_dir = '/n/radbelt/dlnk/polar/cammice/problems/'

if keyword_set(DATE_1) then date_1=DATE_1 else date_1='19960318'
day=fix(strmid(date_1,6,2)) & month=fix(strmid(date_1,4,2))
year=fix(strmid(date_1,0,4))
mjdt_1=convert_date_to_t90(day=day, month=month, year=year, /MJDT)
mjdt_start=mjdt_1
if keyword_set(DATE_2) then date_2=DATE_2 else date_2='20020430'
day=fix(strmid(date_2,6,2)) & month=fix(strmid(date_2,4,2))
year=fix(strmid(date_2,0,4))
mjdt_2=convert_date_to_t90(day=day, month=month, year=year, /MJDT)
mjdt_end=mjdt_2

;select data to get (CAMMICE spin Averaged)
plotinfo = papco_getplotinfostruct()
plotinfo.ioptions(1) = 1 ;flux
plotinfo.ioptions(2) = 0 ;data
plotinfo.ioptions(4) = 0 ;average
plotinfo.ioptions(9) = 0 ;reload
plotinfo.ioptions(5) = 0 ;spin av spec
plotinfo.ioptions(8) = 0 ;plot type spectrum

keep_mjdt_start = mjdt_start & keep_mjdt_end = mjdt_end
;loop through period in dayly intervals

FOR i=keep_mjdt_start.mjd, keep_mjdt_end.mjd, 1 DO BEGIN

    mjdt_start.mjd=i & mjdt_start.t = 0
    mjdt_end.mjd=i+1 & mjdt_end.t = 0
    tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})

    ;convert and print date
    convert_t90_to_Date,mjdt_start,yr,day_oy,h,m,s,mon,cmonth,day_om, /MJDT
    date = string(yr, mon, day_om, format = "(i4.4,i2.2,i2.2)")
    print,''

    fln = 'po_cam_'+date+'.idl'

    ;check for problem
    IF NOT keyword_set(IGNORE_PROB) THEN begin
        r = findfile(prob_dir+'po_cam_'+date,  count = c)
        IF c NE 0 THEN BEGIN
            message, 'problem flag set '+ date, /cont
            continue
        ENDIF
    ENDIF

    ;see if file already exists
    IF NOT keyword_set(FORCE) THEN BEGIN
        result = findfile(out_dir+fln, count = c)
        IF c EQ 1 THEN BEGIN
            message,'got '+ date, /cont
            continue
        ENDIF
    ENDIF
    message,'doing '+ date, /cont

    ;catch errors
    Error_status = 0
    CATCH, Error_status
    IF Error_status NE 0 THEN BEGIN
        spawn, 'echo "'+get_err_msg+'" > '+prob_dir+'po_cam_'+date
        catch, /cancel & continue
    ENDIF

    r_polar_cammice, plotinfo, /orig

    catch, /cancel

    IF get_err_no NE 0 THEN BEGIN
        spawn, 'echo "'+get_err_msg+'" > '+prob_dir+'po_cam_'+date
        continue
    ENDIF

    ;loop through 6 species products to make data files
    FOR s = 0, 5 DO BEGIN

        plotinfo.ioptions(0) = s ;choose species data
        panel = [0, 0, 1]
        yscl(panel(0),3) = 1
        p_polar_cammice_spec, panel, plotinfo, output = 2


        IF s EQ 0 then BEGIN
            ndat = n_elements(time(*, 0))
            nbands = n_elements(yarr)/2
            ;make data structures for output.
            dummy = {tai : dblarr(2), $
                     data : fltarr(nbands, 6)}
            input_data = replicate(dummy, ndat)
            input_data.tai = transpose(time)+tai_1
            input_header = {DATATYPE : 'POLAR CAMMICE SpinAv', $
                            SUBTYPE : strarr(6), $
                            NPOINTS      : ndat, $
                            NDETECTORS : 6, $
                            NBANDS: nbands, $
                            CH_POSITIONS: yarr, $
                            TSTART : time(0, 0), $
                            TSTOP : time(ndat-1, 1), $
                            TIME_FRMT: 'International Atomic Time', $
                            DATA_UNITS: 'cm-2 s-1 sr-1 keV-1', $
                            IDL_DATA_UNITS: uztit, $
                            EK_UNITS: uytitle}
            input_header = create_struct(name='POLAR_CAMMICE_SPIN_AV'+ $
                                         varprt(nbands), input_header)
        ENDIF
        input_header.SUBTYPE(s) =  papco_unformat_idlstring(utitle)
        input_data.data(*,s) = transpose(zmat)
    ENDFOR
    ;save data to file
    save, file = out_dir+fln, input_header, input_data, /compress, /verb

ENDFOR

END
