;******************************************************************************
;******************************************************************************
;* FILE:
;*    $papco_PATH/ceppad_panel.pro
;* 
;*    This file belongs to the papco-project. See $papco_PATH/papco.pro for
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains the panel-editor for CEPPAD-plots - that is, 
;*    CEPPAD-spec
;*
;*    typeVector:   a four-dim. array of integers
;*		    For 'cep_spec', 
;*		    its contents have the following meaning:
;*
;*     	  	     type(0) MODE
;*     	  	     type(1) detector number
;*     	 	     type(2) sector number
;*     	    	     type(3) lin/log yscale switch
;*
;* FILES YOU MAY WANT TO READ
;*       $papco_PATH/papco_conventions.text     which conventions were used
;*						when developing ?
;*	 $papco_PATH/papco_variables.text	what structures and global
;*						variables are used by 
;*						which file ?
;*
;* USED MODULES:
;*    For a list of files, that are needed for papco to run, read the 
;*    file addpapco_init.PRO in $papco_MODULES
;*
;* COMMON-BLOCKS:
;*    ceppad_panel_DATA	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    papco_PANEL_EDITOR-structure (see $papco_PATH/papco_panel.pro for a
;*    description). 
;*       {ceppad_panel_DATA, 
;*	     ; imagine the papco_PANEL_EDITOR-structure was inserted...
;*	      
;*    For a more complete description see $papco_PATH/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  ceppad_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;*    You might try this:
;*	 print, ceppad_panel(papco_getPlotInfoStruct())
;*       which calls up the window.
;*
;* MODIFICATION HISTORY:       
;*     june 1996, written by R.Friedel
;*     jan  1997, to include CLEAN_SURVEY R. Friedel
;*     feb  1997, to include highres spin averaged mode R. Friedel
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  ceppad_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the papco_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $papco_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{       canceled:1, $
;*	    		panelHeight:1, $
;*	    		channel:0, $
;*	    		typeVector:[0,0,0,0], $ 
;*	    		manualYScaling:0 $
;*	    		yScaleMin:LONG(1), $
;*	    		yScaleMax:LONG(1), $
;*	    		panelKind:'some string' }
;*
;*      Additionally, some Panel-editors may return these fields in
;*      structure:
;*               {  color:1, $
;*		    plotStyle:1 }
;*			
;*      For an enhanced description read $papco_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	aPlotInfoStruct=papco_getPlotInfoStruct()
;*
;* MODIFICATION HISTORY:       
;*     june 1996, written by R.Friedel
;*     jan  1997, to include CLEAN_SURVEY R. Friedel
;*     feb  1997, to include highres spin averaged mode R. Friedel
;***************************************************************************
FUNCTION  ceppad_panel, aPlotInfo, GROUP=group, ACTION=ACTION

   common ceppad_panel_DATA, widgetData
 
; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
   panel_base=WIDGET_BASE(title=aPlotInfo.panelKind,$
	 column=1, $
	 GROUP_LEADER=group)

;--- the first part of the window will consist in the widgets -----------------
;    that control the height of that plot
   widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;--- Mode selection buttons ---------------------------------------------------
   type_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=3, /EXCLUSIVE)

   typeNames=[ $
              'IPS LOW_RESOLUTION  ', $
              'IES LOW_RESOLUTION  ', $
              'HISTp LOW_RESOLUTION', $
              'HISTe LOW_RESOLUTION', $
              'IPS SURVEY          ', $
              'IES CLEAN_SURVEY    ', $
              'HISTp SURVEY        ', $
              'HISTe SURVEY        ', $
              'IPS Spin Averaged   ', $
              'IES CLEAN Spin Av.  ', $
              'HISTp Spin Average  ', $
              'HISTe Spin Average  ' ]

   spb_type=lonarr(n_elements(typeNames))
   for i=0,n_elements(typeNames)-1 do $
     spb_type(i)=WIDGET_BUTTON(type_base, $
			VALUE=typeNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))

   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(0)), SET_BUTTON=1
   WIDGET_CONTROL, spb_type(10), SENSITIVE=0
   WIDGET_CONTROL, spb_type(11), SENSITIVE=0

; --- Outer Slider Base -------------------------------------------------------

   slider_Base=WIDGET_BASE(panel_base, /FRAME, column=2)

; --- Detector ----------------------------------------------------------------
   detector_outerBase=WIDGET_BASE(slider_Base, /FRAME, ROW=1) 
   detector_leftBase=WIDGET_BASE(detector_OuterBase, COLUMN=1) 
   detector_rightBase=WIDGET_BASE(detector_OuterBase, COLUMN=1) 

   sl_detector=WIDGET_SLIDER(detector_leftBase, $
			/DRAG, TITLE=strcenter('detector number',45), $
		        VALUE=aPlotInfo.typeVector(1) < 11 > 1, $
                        UVALUE='sl_detector',$
                        MINIMUM=1, MAXIMUM=10)

; --- Sector ------------------------------------------------------------------
   sector_outerBase=WIDGET_BASE(slider_Base, /FRAME, ROW=1) 
   sector_leftBase=WIDGET_BASE(sector_OuterBase, COLUMN=1) 
   sector_rightBase=WIDGET_BASE(sector_OuterBase, COLUMN=1) 

   sl_sector=WIDGET_SLIDER(sector_leftBase, $
			/DRAG , TITLE=strcenter('sector number',45), $
		        VALUE=aPlotInfo.typeVector(2) < 17 > 1, $
                        UVALUE='sl_sector',$
                        MINIMUM=1, MAXIMUM=16)

;--- Plotsyle Base ------------------------------------------------------------

   IF aPlotInfo.panelKind EQ 'ceppad_flux' THEN BEGIN
;--- add the widgets for choosing the plot-style ------------------------------
   widgets=papco_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
   additionalWData=create_struct(widgets, additionalWData)
   ENDIF ELSE BEGIN
;--- add the widgets for choosing the color bar for color plots ----------------
   widgets=papco_panel_Addcolorbar_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   ENDELSE

;--- now, add the widgets for choosing the y-axis-ranges ----------------------
   widgets=papco_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, add the widgets for choosing the z-axis-ranges ----------------------
   widgets=papco_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
   widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
   thisWidgetsData={helpfile:'ceppad_panel.help', $
                    typeVector:aPlotInfo.typeVector, $
                    spb_type:spb_type, $
                    sl_sector:sl_sector, $
                    sl_detector:sl_detector }

   widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)
   ceppad_panel_refresh, /SL_DETECTOR, /SL_SECTOR

;--- realize the window
   papco_Panel_Realize, panel_base, widgetData
   ceppad_panel_refresh, /SL_DETECTOR, /SL_SECTOR
 
; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reportet but the 
; data contained in aPlotInfo would be returned.
   xmanager, 'ceppad_panel', panel_base, /MODAL

; the returned data consists of the standard-data inherited from
; papco_PANEL.PRO and your new data - e.g. the typeVector-tag.
   returnData={typeVector:widgetData.typeVector, $
               switch:aPlotInfo.switch}
   additionalRData=papco_panel_ReturnData(widgetData)
   returnData.typeVector(0)=widgetData.typeVector(0)

   return, create_struct(ReturnData, additionalRData)
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO ceppad_panel_refresh, $
;*                SL_DETECTOR=SL_DETECTOR,SL_SECTOR,SL_SECTOR,HIST=HIST
;* 
;* DESCRIPTION:  
;*      update the display of the editor-widgets
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*      have a look at PRO ceppad_panel_event
;*
;* MODIFICATION HISTORY:       
;*     june 1996, written by R.Friedel
;*     jan  1997, to include CLEAN_SURVEY R. Friedel
;******************************************************************************
PRO ceppad_panel_refresh, SL_DETECTOR=SL_DETECTOR,SL_SECTOR=SL_SECTOR,HIST=HIST

   COMMON ceppad_panel_DATA, widgetData

; ---- Detector - Slider ------------------------------------------------------
   IF KEYWORD_SET(SL_DETECTOR) THEN BEGIN
      x=widgetData.typeVector(0)
      case 1 of
         (x eq 0) or (x eq 1) or (x eq 4) or (x eq 5) or $
         (x eq 8) or (x eq 9): begin
            WIDGET_CONTROL, widgetData.sl_detector, SENSITIVE=1
         end
         (x eq 2) or (x eq 3) or (x eq 6) or (x eq 7) : begin
            WIDGET_CONTROL, widgetData.sl_detector, SENSITIVE=0
         end
      endcase
   ENDIF 

; ---- Sector - Slider --------------------------------------------------------
   IF KEYWORD_SET(SL_SECTOR) THEN BEGIN
      x=widgetData.typeVector(0)
      case 1 of
         (x gt 3) and (x lt 8): begin
            WIDGET_CONTROL, widgetData.sl_sector, SENSITIVE=1
         end
         (x lt 4) or (x gt 7): begin
            WIDGET_CONTROL, widgetData.sl_sector, SENSITIVE=0
         end
      endcase
   ENDIF 

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO ceppad_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     june 1996, written by R.Friedel
;*     jan  1997, to include CLEAN_SURVEY R. Friedel
;******************************************************************************
PRO ceppad_panel_event, event

   COMMON ceppad_panel_DATA, widgetData

   WIDGET_CONTROL,GET_UVALUE=uval, event.id
  
   IF strleft(uval, 8) EQ 'spb_type' THEN BEGIN
      widgetData.typeVector(0)=fix(strfrom(uval, 8))
      ceppad_panel_refresh, /SL_DETECTOR, /SL_SECTOR
      return
   ENDIF 

   CASE uval OF
	'sl_detector': BEGIN
		      widgetData.typeVector(1)=event.value
		  END
	'sl_sector': BEGIN
		      widgetData.typeVector(2)=event.value
		  END

	ELSE: widgetData=papco_panel_Event(event, widgetData)

   ENDCASE

END
















