;******************************************************************************
;******************************************************************************
;* FILE:
;*    $papco_PATH/ceppad_pitch_panel.pro
;* 
;*    This file belongs to the papco-project. See $papco_PATH/papco.pro for
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains the panel-editor for CEPPAD-plots - that is, 
;*    CEPPAD-spec
;*
;*    typeVector:   a four-dim. array of integers
;*		    For 'cep_spec', 
;*		    its contents have the following meaning:
;*
;*     	  	     type(0) MODE
;*     	  	     type(1) detector number
;*     	 	     type(2) sector number
;*     	    	     type(3) lin/log yscale switch
;*
;* FILES YOU MAY WANT TO READ
;*       $papco_PATH/papco_conventions.text     which conventions were used
;*						when developing ?
;*	 $papco_PATH/papco_variables.text	what structures and global
;*						variables are used by 
;*						which file ?
;*
;* USED MODULES:
;*    For a list of files, that are needed for papco to run, read the 
;*    file addpapco_init.PRO in $papco_MODULES
;*
;* COMMON-BLOCKS:
;*    ceppad_pitch_panel_DATA	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    papco_PANEL_EDITOR-structure (see $papco_PATH/papco_panel.pro for a
;*    description). 
;*       {ceppad_pitch_panel_DATA, 
;*	     ; imagine the papco_PANEL_EDITOR-structure was inserted...
;*	      
;*    For a more complete description see $papco_PATH/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  ceppad_pitch_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;*    You might try this:
;*	 print, ceppad_pitch_panel(papco_getPlotInfoStruct())
;*       which calls up the window.
;*
;* MODIFICATION HISTORY:       
;*     june 1996, written by R.Friedel
;*     jan  1997, to include CLEAN_SURVEY R. Friedel
;*     feb  1997, to include highres spin averaged mode R. Friedel
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  ceppad_pitch_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the papco_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $papco_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{       canceled:1, $
;*	    		panelHeight:1, $
;*	    		channel:0, $
;*	    		typeVector:[0,0,0,0], $ 
;*	    		manualYScaling:0 $
;*	    		yScaleMin:LONG(1), $
;*	    		yScaleMax:LONG(1), $
;*	    		panelKind:'some string' }
;*
;*      Additionally, some Panel-editors may return these fields in
;*      structure:
;*               {  color:1, $
;*		    plotStyle:1 }
;*			
;*      For an enhanced description read $papco_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	aPlotInfoStruct=papco_getPlotInfoStruct()
;*
;* MODIFICATION HISTORY:       
;*     june 1996, written by R.Friedel
;*     jan  1997, to include CLEAN_SURVEY R. Friedel
;*     feb  1997, to include highres spin averaged mode R. Friedel
;***************************************************************************
FUNCTION  ceppad_pitch_panel, aPlotInfo, GROUP=group, ACTION=ACTION

   common ceppad_pitch_panel_DATA, widgetData
 
; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
   panel_base=WIDGET_BASE(title=aPlotInfo.panelKind,$
	 column=1, $
	 GROUP_LEADER=group)

;--- the first part of the window will consist in the widgets -----------------
;    that control the height of that plot
   widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- data switch base   
   switch_base=WIDGET_BASE(panel_base,ROW=1)
   
;--- Mode selection buttons ---------------------------------------------------
   type_base=WIDGET_BASE(switch_base, /FRAME, COLUMN=3, /EXCLUSIVE)

   typeNames=[' IPS   ', $
              ' IES   ', $
              ' HISTp ', $
              ' HISTe '  ]

   spb_type=lonarr(n_elements(typeNames))
   for i=0,n_elements(typeNames)-1 do $
     spb_type(i)=WIDGET_BUTTON(type_base, $
			VALUE=typeNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))

   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(0)), SET_BUTTON=1
   ;for i=1,3 do WIDGET_CONTROL, spb_type(i), SENSITIVE=0
   
; --- Integral / differential -------------------------------------------------
   int_diff_base=WIDGET_BASE(switch_base, /FRAME, ROW=1, /EXCLUSIVE)
   spb_integral=WIDGET_BUTTON(int_diff_base, $
			VALUE=strcenter('Integral',12), $
			UVALUE='spb_integral')
   spb_differential=WIDGET_BUTTON(int_diff_base, $
			VALUE=strcenter('Differential',12), $
			UVALUE='spb_differential')

   IF aPlotInfo.typeVector(1) EQ 0 THEN BEGIN
      WIDGET_CONTROL, spb_integral, SET_BUTTON=1
      WIDGET_CONTROL, spb_differential, SET_BUTTON=0
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_integral, SET_BUTTON=0
      WIDGET_CONTROL, spb_differential, SET_BUTTON=1
   ENDELSE
    
; --- Energy / pitch ----------------------------------------------------------
   ener_pitch_outerBase=WIDGET_BASE(panel_base, /FRAME, ROW=1) 
   ener_pitch_leftBase=WIDGET_BASE(ener_pitch_OuterBase, COLUMN=1) 
   ener_pitch_rightBase=WIDGET_BASE(ener_pitch_OuterBase, COLUMN=1) 

   ener_pitch_Base=WIDGET_BASE(ener_pitch_leftBase, ROW=1, /EXCLUSIVE)
   spb_energy=WIDGET_button(ener_pitch_base, $
               VALUE=strcenter('energy spectrum at pitch angle:',35), $
               UVALUE='spb_energy')
   spb_pitch=widget_button(ener_pitch_base, $
	       VALUE=strcenter('pitch angle distribution at energy:',35), $
	       UVALUE='spb_pitch')

   IF aPlotInfo.typeVector(2) EQ 0 THEN BEGIN
      WIDGET_CONTROL, spb_energy, SET_BUTTON=1
      WIDGET_CONTROL, spb_pitch, SET_BUTTON=0
      sl_energy_Pitch=WIDGET_SLIDER(ener_pitch_leftBase, $
			/DRAG, $
		        VALUE=aPlotInfo.typeVector(3) < 14 > 1, $
                        UVALUE='sl_energy_pitch',$
                        MINIMUM=1, MAXIMUM=14)
      lbl_energy_Pitch=WIDGET_LABEL(ener_pitch_leftBase, $
			VALUE=strcenter('pitch angle range',50), $
			/ALIGN_CENTER)			      
      WIDGET_CONTROL, spb_differential, SET_BUTTON=1

   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_energy, SET_BUTTON=0
      WIDGET_CONTROL, spb_pitch, SET_BUTTON=1
      sl_energy_Pitch=WIDGET_SLIDER(ener_pitch_leftBase, $
			/DRAG, $
		        VALUE=aPlotInfo.typeVector(3) < 36 > 1, $
                        UVALUE='sl_energy_pitch',$
                        MINIMUM=1, MAXIMUM=36)
      lbl_energy_Pitch=WIDGET_LABEL(ener_pitch_leftBase, $
			VALUE=strcenter('energy range',50), $
			/ALIGN_CENTER)			      
   ENDELSE
   
;
; <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   

;--- add the widgets for choosing the color bar for color plots ---------------
   widgets=papco_panel_Addcolorbar_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, add the widgets for choosing the y-axis-ranges ----------------------
   widgets=papco_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, add the widgets for choosing the z-axis-ranges ----------------------
   widgets=papco_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
   widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
   thisWidgetsData={helpfile:'ceppad_pitch_panel.help', $
                    slider_displaysEnergy:0, $
                    lbl_energy_Pitch:lbl_energy_Pitch, $
                    sl_energy_pitch:sl_energy_Pitch, $
                    spb_integral:spb_integral, $
                    spb_differential:spb_differential, $
                    spb_energy:spb_energy, $                     
                    typeVector:aPlotInfo.typeVector, $
                    switch:aPlotInfo.switch, $
                    spb_type:spb_type }

   widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   papco_Panel_Realize, panel_base, widgetData
   ceppad_pitch_panel_refresh
 
; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reportet but the 
; data contained in aPlotInfo would be returned.
   xmanager, 'ceppad_pitch_panel', panel_base, /MODAL

; the returned data consists of the standard-data inherited from
; papco_PANEL.PRO and your new data - e.g. the typeVector-tag.
   returnData={typeVector:widgetData.typeVector, $
               switch:widgetData.switch}
   additionalRData=papco_panel_ReturnData(widgetData)
   returnData.typeVector(0)=widgetData.typeVector(0)

   return, create_struct(ReturnData, additionalRData)
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO ceppad_pitch_panel_refresh, $
;*                SL_ENERGY=SL_ENERGY
;* 
;* DESCRIPTION:  
;*      update the display of the editor-widgets
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*      have a look at PRO ceppad_pitch_panel_event
;*
;* MODIFICATION HISTORY:       
;*     june 1996, written by R.Friedel
;*     jan  1997, to include CLEAN_SURVEY R. Friedel
;******************************************************************************
PRO ceppad_pitch_panel_refresh, SL_ENERGY_PITCH=SL_ENERGY_PITCH

  COMMON ceppad_pitch_panel_DATA, widgetData
  COMMON polar_cep, ips_yarr, ies_yarr, histp_yarr, histe_yarr
  
  IF widgetData.typeVector(1) THEN BEGIN     ; differential
    WIDGET_CONTROL, widgetData.spb_energy, SENSITIVE=1
    WIDGET_CONTROL, widgetData.sl_energy_pitch, SENSITIVE=1
  ENDIF ELSE BEGIN			     ; integral
    WIDGET_CONTROL, widgetData.spb_energy, SENSITIVE=0
    WIDGET_CONTROL, widgetData.sl_energy_pitch, SENSITIVE=0
  ENDELSE

  IF widgetData.typeVector(2) THEN BEGIN     ; pitch angle
    WIDGET_CONTROL, widgetData.spb_integral, SENSITIVE=1
    widgetData.slider_displaysEnergy=1
    widgetData.YLINLOG=0
    papco_panel_Refresh,widgetData,/YSCALE
  ENDIF ELSE BEGIN			     ; energy range
    WIDGET_CONTROL, widgetData.spb_integral, SENSITIVE=0
    widgetData.slider_displaysEnergy=0
    widgetData.YLINLOG=1
    papco_panel_Refresh,widgetData,/YSCALE
  ENDELSE

; ---- Energy / Pitch - Slider ------------------------------------------------
  species=['(ips ion) ','(ies elec) ','(hist ion) ','(hist elec) ']
  IF widgetData.slider_displaysEnergy THEN BEGIN ;slider shows energy range
    
    CASE widgetData.typeVector(0) of 
      0: yarr=ips_yarr 
      1: yarr=ies_yarr 
      2: yarr=histp_yarr 
      3: yarr=histe_yarr
    ENDCASE
    n_en=n_elements(yarr(*,0))
      
    WIDGET_CONTROL, widgetData.sl_energy_pitch, $
      	            SET_SLIDER_MIN=1, $
                    SET_SLIDER_MAX=n_en, $
                    SET_VALUE=(widgetData.typeVector(3)<n_en)>1
      
    range=((widgetData.typeVector(3)<n_en)>1)-1
    rangeLabel=species(widgetData.typeVector(0)) + $
      strtrim(string(yarr(range,0), format='(F8.1)'),2) + ' - ' + $
      strtrim(string(yarr(range,1), format='(F8.1)'),2) + ' keV'
    
    WIDGET_CONTROL, widgetData.lbl_energy_Pitch, $
        			 SET_VALUE=strcenter('energy range '+$
                                                     rangeLabel, 50)

  ENDIF ELSE BEGIN              ; slider shows pitch angle range
    ; 5 deg PA bins - 36 PA rangeshere are 19 pitch angle ranges:
    range=(widgetData.typeVector(3)<36)>1
    rLower=5*(range-1)
    rangeLabel=strtrim(string(rLower),2)+' - '+ $
        		     strtrim(string(rLower+5),2)
    
    rangeLabel=species(widgetData.typeVector(0)) + $
      'pitch angle range '+rangelabel+' degree'
    WIDGET_CONTROL, widgetData.lbl_energy_Pitch, $
                    SET_VALUE=strcenter(rangeLabel, 50)
    WIDGET_CONTROL, widgetData.sl_energy_pitch, $
        	    SET_SLIDER_MIN=1, $
        	    SET_SLIDER_MAX=36
  ENDELSE ; END energy/pitch slider  

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO ceppad_pitch_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     june 1996, written by R.Friedel
;*     jan  1997, to include CLEAN_SURVEY R. Friedel
;******************************************************************************
PRO ceppad_pitch_panel_event, event

   COMMON ceppad_pitch_panel_DATA, widgetData

   WIDGET_CONTROL,GET_UVALUE=uval, event.id
  
   IF strleft(uval, 8) EQ 'spb_type' THEN BEGIN
      widgetData.typeVector(0)=fix(strfrom(uval, 8))
      ceppad_pitch_panel_refresh
      return
   ENDIF 

   CASE uval OF
	'spb_electrons' : BEGIN
		      widgetData.typeVector(0)=0
                      ceppad_pitch_panel_refresh
		  END

	'spb_ions' : BEGIN
	              widgetData.typeVector(0)=1
                      ceppad_pitch_panel_refresh
		  END

	'spb_differential' : BEGIN
	              widgetData.typeVector(1)=1
                      ceppad_pitch_panel_refresh
		  END

	'spb_integral'	 : BEGIN
		      widgetData.typeVector(1)=0
                      ceppad_pitch_panel_refresh
		  END

	'spb_energy' : BEGIN
	              widgetData.typeVector(2)=0
                      ceppad_pitch_panel_refresh
		  END

	'spb_pitch' : BEGIN
		      widgetData.typeVector(2)=1
                      ceppad_pitch_panel_refresh
		  END

	'sl_energy_pitch': BEGIN
		      widgetData.typeVector(3)=event.value
                      ceppad_pitch_panel_refresh, /SL_ENERGY_PITCH
		  END

	ELSE: widgetData=papco_panel_Event(event, widgetData)

   ENDCASE

END
















