;******************************************************************************
;* file papcoadd_ceppad
;*      papcoadd_ceppad_spec
;*      papcoadd_ceppad_pitch
;*      papcoadd_ceppad_overview
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for ceppad. 
;*
;* CEPPAD is a a sub-menu for several CEPPAD plot types. To inform PACPCO
;* which plot-types are supported in this file, the plot types are listed at
;* the top of this file: The location of this information in critical as it is
;* read by papco.
;*
;*    Those procedures are contained:
;*
;*    PRO addplottype_ceppad,plottypes_base,pb_plottype
;*        ->  adds the button for the new plot type to the main window.
;*
;*    PRO plottable_ceppad, plottable
;*        -> Sets a variable to indicate plottable without orbit.
;*
;*    PRO descriptor_ceppad, plotInfo, descriptStr
;*        -> Constructs the string used to label plots in the panels window
;*
;*    PRO addpanel_ceppad
;*        -> Calls the panel editor for this plottype and adds the plot to the
;*            panel window.
;*
;*    PRO editor_ceppad, plotNumber, newPlotData
;*        -> This procedure calls the panel editor when called from 
;*	     papco_EditPlot (papco_paneledit.pro).
;*
;*    PRO  draw_ceppad, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual read call and the plot call
;*           for this plottype
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_ceppad,plottypes_base,buttonGeometry,pb_plottype
;* 
;* DESCRIPTION:  
;*	This procedure adds a button for this plottype to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*	papckage, so that they can be automatically called when needed.
;*
;* INPUTS:       
;*	plottypes_base	the widget ID of the parent widget to contain the
;*			plottype buttons.
;*
;* OUTPUTS: 
;*   	pb_plottype	the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for subtypes.
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	addplottype_ceppad,plottypes_base,buttonGeometry,pb_plottype
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;*     modified February 1995, Reine Friedel, added SURVEY modes
;******************************************************************************
pro addplottype_ceppad,plottypes_base,buttonGeometry,pdm_plottype

; VALUE assigns a label for the button
; The ceppad portion of UVALUE is used to construct the calls to all
; other procedures in this add-in needed by papco.

  ;MenuDesc544 = [ $
  ;    { CW_PDMENU_S,   3, strcenter('POLAR-CEP',10) }, $ 
  ;      { CW_PDMENU_S, 0, ' plot Spectra        ' }, $
  ;      { CW_PDMENU_S, 0, ' plot Pitch Angle    ' }, $
  ;      { CW_PDMENU_S, 2, ' plot Overview       ' } $
  ;]

  MenuDesc544 = [ $
      { CW_PDMENU_S,   3, strcenter('POLAR-CEP',10) }, $ 
        { CW_PDMENU_S, 0, ' Pitch Angle/Spectra ' }, $
        { CW_PDMENU_S, 2, ' Detector/Channel    ' } $
  ]

  pdm_plottype = CW_PDMENU(plottypes_base, MenuDesc544, $
                           UVALUE='pdm_ceppad')
  
; initialize the ceppad commons here. This only ever gets called once!
  m_polar_cep_commons  

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO plottable_ceppad, plottable
;*
;* DESCRIPTION:  
;*	Sets a variable to indicate plottable without orbit.
;*
;* INPUTS:       
;*	none
;*
;* OUTPUTS: 
;*   	plottable	a boolean 0 if plottable only by CRRES orbit
;*                                1 if plottable by time also.
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	plottable_ceppad, plottable
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro plottable_ceppad_spec, plottable &  plottable=1 & END

pro plottable_ceppad_pitch, plottable &  plottable=1 & END

pro plottable_ceppad_overview, plottable &  plottable=1 & END

;******************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_ceppad, plotInfo, descriptStr
;*
;* DESCRIPTION:  
;*	Constructs the string used to label plots in the panel window
;*
;* INPUTS:       
;*	plotInfo        structure containing all data for this plot.
;*                      (structure type: papco_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*   	descriptStr	a string containing the label
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	descriptor_ceppad, plotInfo, descriptStr
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro descriptor_ceppad_spec, plotInfo, descriptStr

   typeNames=['IPS LOW_RESOLUTION  ', $
              'IES LOW_RESOLUTION  ', $
              'HISTp LOW_RESOLUTION', $
              'HISTe LOW_RESOLUTION', $
              'IPS SURVEY          ', $
              'IES CLEAN_SURVEY    ', $
              'HISTp SURVEY        ', $
              'HISTe SURVEY        ', $
              'IPS SPIN_1          ', $
              'IES CLEAN_1         ', $
              'HISTp Spin Average  ', $
              'HISTe Spin Average  ' ]

   index=plotInfo.typeVector(0) 
   descriptStr=descriptStr+typeNames(index)

   det= '; Det. ' + string(plotInfo.typeVector(1),format='(i2.2)')
   sec= '; Sec. ' + string(plotInfo.typeVector(2),format='(i2.2)')
   descriptStr=descriptStr+det+sec
   
END

pro descriptor_ceppad_pitch, plotInfo, descriptStr

   descriptStr=descriptStr

END

pro descriptor_ceppad_overview, plotInfo, descriptStr

   descriptStr=descriptStr

END
;******************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_ceppad
;*
;* DESCRIPTION:  
;*	Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      using an instance of the papco_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled.
;*      panelEditor must be set to the name of the panel editor routine
;*
;* INPUTS:       
;*	event		event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_crres_plots
;*
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	addpanel_ceppad
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro addpanel_ceppad,event

   CASE Event.Value OF 
      2:BEGIN
           panelKind='ceppad_spec'
           panelEditor='ceppad_spec_panel'
           defaults=papco_getPlotInfoStruct()
           defaults.typeVector=[0,1,0,1]
           papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults
        END
      1:BEGIN
           panelKind='ceppad_pitch'
           panelEditor='ceppad_pitch_panel'
           defaults=papco_getPlotInfoStruct()
           defaults.typeVector=[0,1,0,1]
           papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults
        END
      3:BEGIN
           panelKind='ceppad_overview'
           panelEditor='ceppad_overview_panel'
           defaults=papco_getPlotInfoStruct()
           defaults.typeVector=[0,0,1,0]
           papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults
        END
   ENDCASE

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO editor_ceppad, plotNumber, newPlotData
;* 
;* DESCRIPTION:  
;*	This procedure calls the panel editor when called from 
;*	papco_EditPlot (papco_paneledit.pro).
;*
;* INPUTS:       
;*	plotNumber	anInteger. 0<=plotNumber<=widgetData.numberOfPlots
;*			the index of the plot in widgetData.plotInfos.
;*		
;* OUTPUTS: 
;*   	newPlotData     the new structure returned by the panel editor
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	editor_ceppad, plotNumber, newPlotData
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
PRO editor_ceppad_spec, plotNumber, newPlotData
  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=ceppad_spec_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
	GROUP=widgetData.main_base)

END

PRO editor_ceppad_pitch, plotNumber, newPlotData
  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=ceppad_pitch_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
	GROUP=widgetData.main_base)

END

PRO editor_ceppad_overview, plotNumber, newPlotData
  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=ceppad_overview_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
	GROUP=widgetData.main_base)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO draw_ceppad, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;* DESCRIPTION:  
;*	This procedure performs the actual data read call and the plot call
;*      for this plottype.
;*      
;*
;* INPUTS:       
;*	panelVector	position of the plot (see papco_conventions.text, 6)
;*	currPlotInfo	structure containing all data for this plot.
;*                      (structure type: papco_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*   	get_data_call	a string containing the data call. Used by other
;*                      routines that need to know how to get the data
;*                      in a panel, eg. to draw spectra, distribution functions
;*      IsRead		a boolean which is set when the the data is read the
;*                      first time. Useful if a given data read puts data
;*                      into commons which can be plotted in many ways.
;*                      When set, data does not have to be read again.
;*
;   here the type of mouse action is checked according to the slice_type flag
;   set by the user. Depending on which bits he set, the action is enabled.
      
;   slice_type:   Selects which slices are supported and for which slice
;                 routines have been written.
;                 This is an integer logical with 16 switches. Currently
;                 there are 5 papco slices. You may extend this by
;                 further USER slices up to a total of 16 slices.
;                     0 = disable all slices
;                     1 = papco Zoom                 (2^0)
;                     2 = papco View Slice           (2^1)
;                     4 = papco Access level Zero    (2^2)
;                     8 = papco Write Paneldata      (2^3)
;                    16 = papco Hair Cursor          (2^4)
;                    32 = USER slice A or not used   (2^5)
;                    64 = USER slice B or not used   (2^6)
;                    "     "     "   "  "  "    "      "    
;                 32768 = USER slice Z or not used  (2^15)
;                        
;   You can support more than one type of slice by arithmetically adding the
;   options, eg slice_type=1+4  support Zoom and Level Zero
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	draw_ceppad, panelVector, currPlotInfo, orbitNo, $
;*                                 get_data_call, slice_type, IsRead
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro draw_ceppad_spec, panelVector, currPlotInfo, orbitNo, $
                                 get_data_call, slice_type, IsRead

  common extra, extra_PlotInfo
  
   modeNames=['LOW_RESOLUTION', $
              'SURVEY', $
              'CLEAN_SURVEY', $
              'Spin Averaged', $
              'CLEAN Spin Av.', $
              'CLEAN_1Z16' ]

   typeNames=['IPS', $
              'IES', $
              'HISTp', $
              'HISTe' ]  

   currmode=modeNames(currPlotInfo.typeVector(0))
   currinst=typenames(currPlotInfo.switch(1))
   currlocation='LANL'

   get_data_call= "r_cep_if, INSTRUMENT='"+currinst+ $
                  "',MODE='"+currmode+"',LOCATION='"+currlocation+"'"
   slice_Type=1+16+64
   
   stop

   IF NOT IsRead THEN BEGIN ; check if data has already been read
      r_cep_if,INSTRUMENT=currinst,MODE=currmode,LOCATION=currlocation
      ; check the read for success.      
      IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                                panelVector, orbitNr, $
                                currPlotInfo.typeVector, $
                                currPlotInfo.channel)       THEN IsRead=0
   ENDIF
    
   IF IsRead  THEN BEGIN 
      p_cep_spec, panelVector, currPlotInfo.typeVector, $
                     currPlotInfo.switch,  _EXTRA=extra_PlotInfo
   ENDIF

END

;******************************************************************************
pro draw_ceppad_pitch, panelVector, currPlotInfo, orbitNo, $
                                 get_data_call, slice_type, IsRead

common extra, extra_PlotInfo

   typeNames=['ips pitch ', $
              'ies pitch ', $
              'histp pitch', $
              'histe pitch'  ]

   currtypeName=strtrim(typeNames(currPlotInfo.typeVector(0)))
   dummy=str_sep(currtypeName,' ')
   currentinst=dummy(0)
   currmode=dummy(1)
   currlocation='LANL'

   get_data_call= "r_cep_pitch, INSTRUMENT='"+currentinst+ $
     "',MODE='"+currmode+"',LOCATION='"+currlocation+"'"

   slice_Type=1+16

   IF NOT IsRead THEN BEGIN  ; check if data has already been read
      r_cep_pitch,INSTRUMENT=currentinst,MODE=currmode,LOCATION=currlocation
      ; check the read for success.
      IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                                panelVector, orbitNr, $
                                currPlotInfo.typeVector, $
                                currPlotInfo.channel)      THEN IsRead=1
   ENDIF
   
   IF IsRead THEN BEGIN
      p_cep_pitch, panelVector, currPlotInfo.typeVector, $
                     currPlotInfo.switch,  _EXTRA=extra_PlotInfo
   ENDIF

END

;******************************************************************************
pro draw_ceppad_overview, panelVector, currPlotInfo, orbitNo, $
                                 get_data_call, slice_type, IsRead

common extra, extra_PlotInfo

   get_data_call= "r_cep_lav, type"
   slice_Type=1+16

   IF NOT IsRead THEN BEGIN
      r_cep_lav, currPlotInfo.typeVector
      IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                                panelVector, orbitNr, $
                                currPlotInfo.typeVector, $
                                currPlotInfo.channel) $
      THEN  IsRead=1 ELSE return

   ENDIF

   p_cep_lav, panelVector, currPlotInfo.typeVector, $
              currPlotInfo.switch,  _EXTRA=extra_PlotInfo

END

