;******************************************************************************
;* FILE:
;*    $papco_modules/POLAR_ceppad/ceppad_overview_panel.pro
;* 
;*    This file belongs to the papco-project. See $papco_BASE/papco_doc
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains the panel-editor for CEPPAD-overview plots
;*
;* USAGE OF THE FIELDS IN THE papco_PLOTINFO-STRUCTURE ('aPlotInfo'):
;*    copied from papco_variables.text
;*
;*    typeVector:   a five-dim. array of integers
;*		    For 'ceppad-overview-plot', 
;*		    its contents have the following meaning:
;*
;*                    typeVector(0)  an integer
;*				   0 for plotting IPS
;*				   1 for plotting IES
;*				   2 for plotting HISTp
;*				   3 for plotting HISTe
;*
;*                    typeVector(1)  an integer
;*                                 contains the energy channel
;*
;*                    typeVector(2)  a positive Integer
;*                                 contains the leg number
;*                                 0 for IB HA
;*                                 1 for IB LA
;*                                 2 for OB LA
;*                                 3 for OB HA
;*
;*                    typeVector(3)  a positive Integer 
;*                                 not used
;*
;* COMMON-BLOCKS:
;*    CRRES_PANEL_DATA	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    papco_PANEL_EDITOR-structure (see $papco_PATH/papco_panel.pro for a
;*    description). 
;*       {CRRES_PANEL_DATA,...
;*	     ; imagine the papco_PANEL_EDITOR-structure was inserted...
;*      
;*    the additional fields are :
;*                    typeVector:typeVector, $
;*                    spb_integral:spb_integral, $
;*                    spb_differential:spb_differential, $
;*                    spb_allenergies:spb_allenergies, $
;*                    spb_Selectenergies:spb_Selectenergies, $
;*                    sl_energy:sl_energy, $
;*                    lbl_energy:lbl_energy, $
;*                    spb_allpitch:spb_allpitch, $
;*                    spb_Selectpitch:spb_Selectpitch, $
;*                    sl_pitch:sl_pitch,$
;*                    lbl_pitch:lbl_pitch, $
;*                    spb_allLvalues:spb_allLvalues, $
;*                    spb_selectLvalues:spb_selectLvalues, $
;*                    sl_Lvalue:sl_Lvalue,$
;*                    lbl_Lvalue:lbl_Lvalue}
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  ceppad_overview_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;*
;* MODIFICATION HISTORY:       
;*    June 1997, written by R.Friedel
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  ceppad_overview_panel GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a ceppad_overview plot
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the papco_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*
;* CALLING SEQUENCE:
;*	aPlotInfoStruct=papco_getPlotInfoStruct()
;*      print, ceppad_overview_panel(aPlotInfoStruct)
;*
;* MODIFICATION HISTORY:       
;*    June 1997, written by R.Friedel
;******************************************************************************
FUNCTION  ceppad_overview_panel, aPlotInfo, GROUP=group, ACTION=ACTION
   COMMON CRRES_PANEL_DATA, widgetData
 
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; --- create Widget -----------------------------------------------------------
   additionalWData={canceled:1}

   panel_base=WIDGET_BASE(TITLE=aPlotInfo.panelKind, $
			COLUMN=1, $
			GROUP_LEADER=GROUP) 

; --- height-Widgets ----------------------------------------------------------
   widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

  comb_base=WIDGET_BASE(panel_base, Row=1)

; --- Choose CEPPAD instrument type -------------------------------------------

   type_base=WIDGET_BASE(comb_base, /FRAME, COLUMN=3, /EXCLUSIVE)

   typeNames=['IPS   ', $
              'IES   ', $
              'HISTp ', $
              'HISTe '  ]

   spb_type=lonarr(n_elements(typeNames))
   for i=0,n_elements(typeNames)-1 do $
     spb_type(i)=WIDGET_BUTTON(type_base, $
			VALUE=typeNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))

   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(0)), SET_BUTTON=1


; --- Integral / differential -------------------------------------------------
   int_diff_base=WIDGET_BASE(comb_base, /FRAME, ROW=1, /EXCLUSIVE)
   spb_integral=WIDGET_BUTTON(int_diff_base, $
			VALUE='Integral', $
			UVALUE='spb_integral')
   spb_differential=WIDGET_BUTTON(int_diff_base, $
			VALUE='Differential', $
			UVALUE='spb_differential')

   IF aPlotInfo.typeVector(1) EQ 0 THEN BEGIN
      WIDGET_CONTROL, spb_integral, SET_BUTTON=1
      WIDGET_CONTROL, spb_differential, SET_BUTTON=0
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_integral, SET_BUTTON=0
      WIDGET_CONTROL, spb_differential, SET_BUTTON=1
   ENDELSE

; --- Plot swittches -----------------------------------------------------------
   swittch_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=3)

   smooth_base=WIDGET_BASE(swittch_base, /FRAME, ROW=1, /EXCLUSIVE)
   spb_nosmooth=WIDGET_BUTTON(smooth_base, $
			VALUE='no smooth', $
			UVALUE='spb_nosmooth')
   spb_smooth=WIDGET_BUTTON(smooth_base, $
			VALUE='smooth', $
			UVALUE='spb_smooth')
   IF aPlotInfo.ioptions(17) EQ 0 THEN BEGIN
      WIDGET_CONTROL, spb_nosmooth, SET_BUTTON=1
      WIDGET_CONTROL, spb_smooth, SET_BUTTON=0
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_nosmooth, SET_BUTTON=0
      WIDGET_CONTROL, spb_smooth, SET_BUTTON=1
   ENDELSE

   fill_base=WIDGET_BASE(swittch_base, /FRAME, ROW=1, /EXCLUSIVE)
   spb_nofill=WIDGET_BUTTON(fill_base, $
			VALUE='no fill', $
			UVALUE='spb_nofill')
   spb_fill=WIDGET_BUTTON(fill_base, $
			VALUE='fill', $
			UVALUE='spb_fill')
   IF aPlotInfo.ioptions(18) EQ 0 THEN BEGIN
      WIDGET_CONTROL, spb_nofill, SET_BUTTON=1
      WIDGET_CONTROL, spb_fill, SET_BUTTON=0
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_nofill, SET_BUTTON=0
      WIDGET_CONTROL, spb_fill, SET_BUTTON=1
   ENDELSE

   orbitlabel_base=WIDGET_BASE(swittch_base, /FRAME, ROW=1, /EXCLUSIVE)
   spb_noorbit=WIDGET_BUTTON(orbitlabel_base, $
			VALUE='no orbit labels', $
			UVALUE='spb_noorbit')
   spb_orbit=WIDGET_BUTTON(orbitlabel_base, $
			VALUE='label orbits', $
			UVALUE='spb_orbit')
   IF aPlotInfo.ioptions(19) EQ 0 THEN BEGIN
      WIDGET_CONTROL, spb_noorbit, SET_BUTTON=1
      WIDGET_CONTROL, spb_orbit, SET_BUTTON=0
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_noorbit, SET_BUTTON=0
      WIDGET_CONTROL, spb_orbit, SET_BUTTON=1
   ENDELSE

;--- buttons for choosing orbit legs ----------------------------------------- 
;    Any combination of buttons may be pressed. Default is first chosen.
 
   leg_names=['Inbound High Alt.','Inbound Low Alt.', $
              'Outbound Low Alt.','Outbound High Alt.']

;--- use typevector(2), an 16 bit integer, as a logical flag to set legs.
   leg_set=bytarr(n_elements(leg_names)) 
   for i=0,n_elements(leg_names)-1 do begin
      if (aPlotInfo.typeVector(2) and 2^i) eq 2^i then leg_set(i)=1 $
      else leg_set(i)=0
   endfor

   leg_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=1) 
   mpb_leg=CW_BGROUP(leg_base, leg_names, $
                      ROW=1,/FRAME,/NONEXCLUSIVE, $
                      UVALUE='mpb_leg', $
                      SET_VALUE=leg_set)

; --- SLIDERS: ----------------------------------------------------------------

; --- Energy ------------------------------------------------------------------
   energyBase=WIDGET_BASE(panel_base, COLUMN=1, /FRAME)
   energyLeftBase=WIDGET_BASE(energyBase, /EXCLUSIVE, ROW=1)
   energyRightBase=WIDGET_BASE(energyBase, COLUMN=1)
   spb_allenergies=WIDGET_BUTTON(energyLeftBase, $
				VALUE='All energies        ',$
				UVALUE='spb_allenergies')
   spb_Selectenergies=WIDGET_BUTTON(energyLeftBase, $
				VALUE='Select energy   ',$
				UVALUE='spb_selectenergies')

   IF aPlotInfo.typeVector(1) ne 0 then BEGIN
       WIDGET_CONTROL, spb_allenergies, SET_BUTTON=0 
       WIDGET_CONTROL, spb_Selectenergies, SET_BUTTON=1
   ENDIF ELSE BEGIN
       WIDGET_CONTROL, spb_allenergies, SET_BUTTON=1
       WIDGET_CONTROL, spb_Selectenergies, SET_BUTTON=0
   ENDELSE

   sl_energy=WIDGET_SLIDER(energyrightBase, $
			/DRAG, $
		        VALUE=aPlotInfo.typeVector(1) < 18 > 1, $
                        UVALUE='sl_energy',$
                        MINIMUM=1, MAXIMUM=18)
   lbl_energy=WIDGET_LABEL(energyrightBase, $
			VALUE=strcenter('energy range',50), $
			/ALIGN_CENTER)			      
              
; --- Pitch angle--------------------------------------------------------------
   pitchBase=WIDGET_BASE(panel_base, COLUMN=1, /FRAME)
   pitchLeftBase=WIDGET_BASE(pitchBase, /EXCLUSIVE, ROW=1)
   pitchRightBase=WIDGET_BASE(pitchBase, COLUMN=1)
   spb_allpitch=WIDGET_BUTTON(pitchLeftBase, $
				VALUE='All pitch angles    ',$
				UVALUE='spb_allpitch')
   spb_Selectpitch=WIDGET_BUTTON(pitchLeftBase, $
				VALUE='Select pitch angles  ',$
				UVALUE='spb_selectpitch')

   IF aPlotInfo.typeVector(3) ne 0 then BEGIN
       WIDGET_CONTROL, spb_allpitch, SET_BUTTON=0 
       WIDGET_CONTROL, spb_Selectpitch, SET_BUTTON=1
   ENDIF ELSE BEGIN
       WIDGET_CONTROL, spb_allpitch, SET_BUTTON=1
       WIDGET_CONTROL, spb_Selectpitch, SET_BUTTON=0
   ENDELSE

   sl_pitch=WIDGET_SLIDER(pitchrightBase, $
			/DRAG, $
		        VALUE=aPlotInfo.typeVector(3) < 4 > 1, $
                        UVALUE='sl_pitch',$
                        MINIMUM=1, MAXIMUM=4)
   lbl_pitch=WIDGET_LABEL(pitchrightBase, $
			VALUE=strcenter('pitch angle range',50), $
			/ALIGN_CENTER)			

; --- L-value -----------------------------------------------------------------
   LvalueBase=WIDGET_BASE(panel_base, COLUMN=1, /FRAME)
   LvalueLeftBase=WIDGET_BASE(LvalueBase, /EXCLUSIVE, ROW=1)
   LvalueRightBase=WIDGET_BASE(LvalueBase, COLUMN=1)
   spb_allLvalues=WIDGET_BUTTON(LvalueLeftBase, $
				VALUE='All L-values        ',$
				UVALUE='spb_allLvalues')
   spb_SelectLvalues=WIDGET_BUTTON(LvalueLeftBase, $
				VALUE='Select L-value   ',$
				UVALUE='spb_selectLvalues')

   tmp=WIDGET_INFO(panel_base, /GEOMETRY)
   sl_Lvalue=widget_slider(LvalueRightBase, $
       MINIMUM=1, MAXIMUM=35, /DRAG, $
 	 VALUE=(aPlotinfo.ioptions(19)>1), $
	 UVALUE='sl_Lvalue')
   lbl_Lvalue =WIDGET_LABEL(LvalueRightBase, $
			VALUE=strcenter('L-value', 50), $
			/ALIGN_CENTER)
   IF aPlotinfo.ioptions(19) ne 0 THEN BEGIN
      WIDGET_CONTROL, spb_allLvalues, SET_BUTTON=0
      WIDGET_CONTROL, spb_selectLvalues, SET_BUTTON=1
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_allLvalues, SET_BUTTON=1
      WIDGET_CONTROL, spb_selectLvalues, SET_BUTTON=0
      WIDGET_CONTROL, sl_Lvalue, SENSITIVE=0      
   ENDELSE

;--- add the papco panel widget for choosing the color bar for color plots ----
   widgets=papco_panel_Addcolorbar_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)   
;--- now, add the widgets for choosing the y-axis-ranges ----------------------
   widgets=papco_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
   widgets=papco_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
   widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
   additionalWData=create_struct(widgets, additionalWData)
 
   thisWidgetsData={helpFile:'ceppad_overview_panel.help', $
                    typeVector:aPlotInfo.typeVector, $
                     $
                    spb_integral:spb_integral, $
                    spb_differential:spb_differential, $
                    spb_allenergies:spb_allenergies, $
                    spb_Selectenergies:spb_Selectenergies, $
                    spb_nosmooth:spb_nosmooth, $
                    spb_smooth:spb_smooth, $
                    spb_nofill:spb_nofill, $
                    spb_fill:spb_fill, $
                    spb_noorbit:spb_noorbit, $
                    spb_orbit:spb_orbit, $
                    sl_energy:sl_energy, $
                    lbl_energy:lbl_energy, $
                    sl_pitch:sl_pitch,$
                    lbl_pitch:lbl_pitch, $
                    spb_allpitch:spb_allpitch, $
                    spb_Selectpitch:spb_Selectpitch, $
                    spb_allLvalues:spb_allLvalues, $
                    spb_selectLvalues:spb_selectLvalues, $
                    sl_Lvalue:sl_Lvalue,$
                    lbl_Lvalue:lbl_Lvalue,$
                    mpb_leg:mpb_leg}

   widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

   IF widgetData.panelKind EQ 'CRRES-pitch-Plot' THEN BEGIN
      widgetData.typeVector(2)=1
      WIDGET_CONTROL, spb_allenergies, SENSITIVE=0
      WIDGET_CONTROL, spb_pitch, /SET_BUTTON
   ENDIF

   papco_Panel_Realize, panel_base, widgetData
   ceppad_overview_panel_refresh,/SL_PITCH,/SL_LVALUE,/SL_ENERGY, $
                                /PB_PITCH,/PB_ENERGY,/PB_LVALUE
   ceppad_overview_panel_refresh,/SL_PITCH,/SL_LVALUE,/SL_ENERGY

   xmanager, 'ceppad_overview_panel', panel_base, /MODAL
   
;--- return the data  ---------------------------------------------------------
   additionalRData=papco_panel_ReturnData(widgetData)

   thisReturnData={typeVector:widgetData.typeVector}	 

   return, create_struct(thisReturnData, additionalRData)
END


;******************************************************************************
;* PROCEDURE:     
;*      PRO ceppad_overview_panel_refresh, $
;*                SL_PITCH=SL_PITCH, SL_ENERGY=SL_ENERGY, SL_LVALUE=SL_LVALUE $
;*                PB_PITCH=PB_PITCH, PB_ENERGY=PB_ENERGY, PB_LVALUE=PB_LVALUE
;* DESCRIPTION:  
;*      update the display of the editor-widgets
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*      have a look at PRO ceppad_overview_panel_event
;*
;* MODIFICATION HISTORY:       
;*    June 1997, written by R.Friedel
;***************************************************************************
PRO ceppad_overview_panel_refresh,  $
                 SL_PITCH=SL_PITCH, SL_ENERGY=SL_ENERGY, SL_LVALUE=SL_LVALUE, $
                 PB_PITCH=PB_PITCH, PB_ENERGY=PB_ENERGY, PB_LVALUE=PB_LVALUE
   COMMON CRRES_PANEL_DATA, widgetData

   WIDGET_CONTROL, widgetData.spb_integral, SENSITIVE=0

   IF KEYWORD_SET(PB_LVALUE) then BEGIN
     IF widgetData.ioptions(19) eq 0 then BEGIN  ; all L-value
        WIDGET_CONTROL, widgetData.spb_allLvalues, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.spb_selectLvalues, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_allenergies, SENSITIVE=0
        WIDGET_CONTROL, widgetData.spb_allpitch, SENSITIVE=0
        WIDGET_CONTROL, widgetData.spb_selectenergies, SENSITIVE=1
        WIDGET_CONTROL, widgetData.spb_selectpitch, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.spb_selectenergies, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.spb_selectpitch, SENSITIVE=1
        WIDGET_CONTROL, widgetData.sl_Lvalue, SENSITIVE=0
        WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=1
        WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=1
        return 
     ENDIF ELSE BEGIN                            ; select L-value
        WIDGET_CONTROL, widgetData.spb_allLvalues, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_selectLvalues, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.sl_Lvalue, SENSITIVE=1
        WIDGET_CONTROL, widgetData.spb_allenergies, SENSITIVE=1
        WIDGET_CONTROL, widgetData.spb_allpitch, SENSITIVE=1
        WIDGET_CONTROL, widgetData.spb_selectenergies, SENSITIVE=1
        WIDGET_CONTROL, widgetData.spb_selectpitch, SENSITIVE=1
     ENDELSE
   ENDIF
  
   IF KEYWORD_SET(PB_ENERGY)  then BEGIN
     IF widgetData.typeVector(2) eq 0 then BEGIN  ; all energy
        WIDGET_CONTROL, widgetData.spb_allenergies, SET_BUTTON=1 
        WIDGET_CONTROL, widgetData.spb_Selectenergies, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_allpitch, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_Selectpitch, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=0
        WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=1
     ENDIF ELSE BEGIN                             ; SELECT ENERGY
        WIDGET_CONTROL, widgetData.spb_allenergies, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_Selectenergies, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.spb_allpitch, SET_BUTTON=1 
        WIDGET_CONTROL, widgetData.spb_Selectpitch, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=1
        WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=0
        widgetData.typeVector(3)=0
     ENDELSE
   ENDIF

   IF KEYWORD_SET(PB_PITCH) then BEGIN
      IF widgetData.typeVector(3) eq 0 then BEGIN  ; all pitch angle
        WIDGET_CONTROL, widgetData.spb_allpitch, SET_BUTTON=1 
        WIDGET_CONTROL, widgetData.spb_Selectpitch, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_allenergies, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_Selectenergies, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=0
        WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=1
      ENDIF ELSE BEGIN                             ; select pitch angle
        WIDGET_CONTROL, widgetData.spb_allpitch, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_Selectpitch, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.spb_allenergies, SET_BUTTON=1 
        WIDGET_CONTROL, widgetData.spb_Selectenergies, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=1
        WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=0
        widgetData.typeVector(2)=0
      ENDELSE
   ENDIF

   IF KEYWORD_SET(SL_ENERGY) then BEGIN
      Erange=fltarr(21)
      Prange=fltarr(30)


      case widgetData.typeVector(0) of
        0:BEGIN   ; for IPS
          WIDGET_CONTROL, widgetData.sl_energy, $
                          SET_SLIDER_MIN=1, SET_SLIDER_MAX=18
          widgetData.typeVector(1)= widgetData.typeVector(1) < 18
          range=(widgetData.typeVector(1)-1) > 0
          yminarr = [16.80,  21.20,  27.90,  37.50,  49.60,  65.90, $
                     87.70,  118.0,  161.0,  221.0,  303.0,  417.0, $
                     574.0,  791.0,  1091.,  1505.,  16.80 ]

          ymaxarr = [21.20,  27.90,  37.50,  49.60,  65.90,  87.70, $
                     118.0,  161.0,  221.0,  303.0,  417.0,  574.0, $
                     791.0,  1091.,  1505.,  2000.,  2000. ]

          if range le 16 then $
          rangeLabel='IPS energy range '+ $
	       strtrim(string(yminarr(range),format='(F8.3)'),2) + ' - ' + $
	       strtrim(string(ymaxarr(range),format='(F8.3)'),2) + ' keV' $
          else rangelabel='background channel'
          END
        1:BEGIN  ; for IES
          WIDGET_CONTROL, widgetData.sl_energy, $
                          SET_SLIDER_MIN=1, SET_SLIDER_MAX=10
          widgetData.typeVector(1)= widgetData.typeVector(1) < 10
          range=(widgetData.typeVector(1)-1) > 0
          yminarr = [0.00,  3.00,  9.00,  15.0,  23.0,  34.00, $
                     49.0,  70.0,  100.,  141.000]
          ymaxarr = [3.00,  9.00,  15.0,  23.0,  34.0,  49.00, $
                     70.0,  100.,  141.,  197.000]
          rangeLabel='IES energy range '+ $
	       strtrim(string(yminarr(range),format='(F8.3)'),2) + ' - ' + $
	       strtrim(string(ymaxarr(range),format='(F8.3)'),2) + ' keV' 
         END
       else:  rangeLabel='not implemented yet'
     endcase

   WIDGET_CONTROL, widgetData.lbl_energy, $
        	SET_VALUE=strcenter(rangeLabel, 50)
   ENDIF

   IF KEYWORD_SET(SL_PITCH) then BEGIN
      WIDGET_CONTROL, widgetData.sl_pitch, $
			       SET_SLIDER_MIN=1, $
			       SET_SLIDER_MAX=4
      WIDGET_CONTROL, widgetData.sl_pitch, $
			       SET_VALUE=widgetData.typeVector(3) < 4 > 1
               pitch=(widgetData.typeVector(3)-1) < 3 > 0
      Parr=[90,90,90,90] 
      pitchlabel=string(Parr(pitch),format="(i3.3)")
      
      pitchLabel=pitchLabel + ' degree'
      WIDGET_CONTROL, widgetData.lbl_pitch, $
			 SET_VALUE=strcenter('pitch angle '+$
			         pitchLabel,50)   
   ENDIF

   IF KEYWORD_SET(SL_LVALUE) then BEGIN
      WIDGET_CONTROL, widgetData.sl_Lvalue, $
			       SET_SLIDER_MIN=1, $
			       SET_SLIDER_MAX=35
      WIDGET_CONTROL, widgetData.sl_Lvalue, $
			       SET_VALUE=(widgetData.ioptions(19)>1)
      Lvalue=float(widgetData.ioptions(19)-1)*0.2 +1.0
      LvalueLabel=string(Lvalue,Lvalue+0.2,format="(f4.1,' - ',f4.1)")

      WIDGET_CONTROL, widgetData.lbl_Lvalue, $
			 SET_VALUE=strcenter('L-value '+$
			         LvalueLabel,50)   
   ENDIF

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO ceppad_overview_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*    June 1997, written by R.Friedel
;******************************************************************************
PRO ceppad_overview_panel_event, EVENT
   COMMON CRRES_PANEL_DATA, widgetData
   WIDGET_CONTROL,GET_UVALUE=uval, event.id

   IF strleft(uval, 8) EQ 'spb_type' THEN BEGIN
      widgetData.typeVector(0)=fix(strfrom(uval, 8))
      ceppad_overview_panel_refresh,/SL_ENERGY
      return
   ENDIF 

   CASE uval OF
	'spb_differential' : BEGIN
               widgetData.typeVector(1)=1
		  END
	'spb_integral'	 : BEGIN
               widgetData.typeVector(1)=0
		  END

        'spb_nosmooth'   : widgetData.ioptions(17)=0
        'spb_smooth'     : widgetData.ioptions(17)=1
        'spb_nofill'     : widgetData.ioptions(18)=0
        'spb_fill'       : widgetData.ioptions(18)=1
        'spb_noorbit'    : widgetData.ioptions(19)=0
        'spb_orbit'      : widgetData.ioptions(19)=1

	'spb_allenergies' : BEGIN
               widgetData.typeVector(2)=0
               ceppad_overview_panel_refresh,/PB_LVALUE,/PB_ENERGY
		  END
        'spb_selectenergies' :BEGIN
               widgetData.typeVector(2)=3
               ceppad_overview_panel_refresh,/PB_LVALUE,/PB_ENERGY
		  END
	'sl_energy': BEGIN
               widgetData.typeVector(1)=event.value
               ceppad_overview_panel_refresh, /SL_ENERGY
		  END
	'spb_allpitch' : BEGIN
               widgetData.typeVector(3)=0
               ceppad_overview_panel_refresh,/PB_LVALUE,/PB_PITCH
		  END
	'spb_selectpitch' : BEGIN
               widgetData.typeVector(3)=10
               ceppad_overview_panel_refresh,/PB_LVALUE,/PB_PITCH
		  END
	'sl_pitch': BEGIN
               widgetData.typeVector(3)=event.value
               ceppad_overview_panel_refresh,/PB_LVALUE,/SL_PITCH
               ceppad_overview_panel_refresh,/SL_PITCH
		  END
	'spb_allLvalues': BEGIN
               widgetData.ioptions(19)=0
               ceppad_overview_panel_refresh,/PB_LVALUE
                END
	'spb_selectLvalues': BEGIN
               widgetData.ioptions(19)=25
               ceppad_overview_panel_refresh,/PB_LVALUE,/PB_ENERGY,/PB_PITCH
                END
        'sl_Lvalue' : BEGIN
               widgetData.ioptions(19)=event.value
               ceppad_overview_panel_refresh, /SL_LVALUE
		END
        'mpb_leg' : BEGIN
               WIDGET_CONTROL,widgetData.mpb_leg,GET_VALUE=leg_event
                   widgetData.typevector(2)=0
                   for i=0,n_elements(leg_event)-1 do $
                      if leg_event(i) then $
                         widgetData.typevector(2)=widgetData.typevector(2)+2^i
                END
 
	ELSE: $
   	   widgetData=papco_panel_Event(event, widgetData)

   ENDCASE

END


