;******************************************************************************
;* FUNCTION:     
;*      FUNCTION polar_ceppad_panel, PlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	PlotInfo	a structure of the papco_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*      SIZE_ONLY       returns size info of widget only. Used to determine
;*                      scrolling
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $papco_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{       canceled:1, $
;*	    		panelHeight:1, $
;*	    		channel:0, $
;*	    		typeVector:[0,0,0,0], $ 
;*	    		manualYScaling:0 $
;*	    		yScaleMin:LONG(1), $
;*	    		yScaleMax:LONG(1), $
;*	    		panelKind:'some string' }
;*
;*      Additionally, some Panel-editors may return these fields in
;*      structure:
;*               {  color:1, $
;*		    plotStyle:1 }
;*			
;*      For an enhanced description read $papco_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	plotinfoStruct = papco_getplotinfoStruct()
;*
;* MODIFICATION HISTORY:       
;*     june 1996, written by R.Friedel
;*     jan  1997, to include CLEAN_SURVEY R. Friedel
;*     feb  1997, to include highres spin averaged mode R. Friedel
;***************************************************************************
FUNCTION polar_ceppad_panel, plotinfo, GROUP=group, ACTION=ACTION, $
                             SIZE_ONLY = SIZE_ONLY, SCROLL = scroll, $
                             X_SCROLL_SIZE = xsize, Y_SCROLL_SIZE = ysize 

COMMON polar_ceppad_panel_data, widgetData
COMMON polar_ceppad ;see polar_ceppad_commons.pro for contents of this block

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
additionalWData={canceled:1}

;***NEW*** All the user's widget info is now stored in a structure,
;and the only thing stored in the plotinfostruct is the pointer to
;that structure. This allows unlimited freedom in variables for the
;module.

;test if the pointer already points to the modules control
;structure. If so, load it in, if not, load in the default.
r = size(*plotinfo.USR_PTR1, /st)
IF r.TYPE_NAME EQ 'INT' THEN plotinfo.USR_PTR1 = ptr_new(polar_ceppad_control)
control = *plotinfo.USR_PTR1

;--- create the base-window ---------------------------------------------------
title = plotinfo.panelKind + ' - ' + papco_module_version('polar_ceppad')
panel_base = papco_WIDGET_BASE(TITLE=title, COLUMN = 1, GROUP_LEADER=group)
panel_top =    widget_base(panel_base, COLUMN = 1)
panel_middle = widget_base(panel_base, ROW = 1, SCROLL = scroll, $
                           X_SCROLL_SIZE = xsize, Y_SCROLL_SIZE = ysize)
panel_left=widget_base(panel_middle,COLUMN=1, /FRAME)
panel_right=widget_base(panel_middle,COLUMN=1,/FRAME)
panel_bottom = widget_base(panel_base, ROW = 1)

;--- the first part of the window will consist in the widgets -----------------
;    that control the height of that plot
widgets=papco_panel_AddHeightWidgets(panel_top, plotinfo)
additionalWData=create_struct(widgets, additionalWData)

;--- Mode selection buttons ---------------------------------------------------
prod_base = widget_base(panel_left, ROW = 1, /FRAME)
if_base=papco_CW_BGROUP(prod_base, typeNames, ROW=1, $
                   /EXCLUSIVE,  LABEL_LEFT='Products:', $
                   UVALUE='spb_if', BUTTON_IDS = if_ids, $
                   BUTTON_UVALUE = if_uvalues, SET_VALUE=control.product) 

;--- Avereaging rnages - sectorn/spins ----------------------------------------
fld_sect = CW_FIELD(prod_base, /ROW,  XSIZE = 2, TITLE = 'Sectors:', $
                    UVALUE= 'fld_sect', VALUE = control.sectors)
fld_spin = CW_FIELD(prod_base, /ROW,  XSIZE = 2, TITLE = 'Spins:', $
                    UVALUE= 'fld_spin', VALUE = control.spins)

;--- Plot type selection buttons ----------------------------------------------
plot_base=WIDGET_BASE(panel_left, COLUMN=1, /ALIGN_CENTER)

names = papco_make_choice_names(control.pl_type_info)
pl_type_base = papco_CW_BGROUP(plot_base, names, ROW = 1, /FRAME, /EXCLUSIVE, $
                   UVALUE='spb_pl_type', SET_VALUE=control.pl_type)

names = papco_make_choice_names(control.cal_info)
cal_base = papco_CW_BGROUP(plot_base, names, ROW = 1, /FRAME, /EXCLUSIVE, $
                   UVALUE='spb_cal', SET_VALUE=control.cal)

;--- Toggle buttons for bckgnd, satur, crosscal -----------------------------
intercal_base = widget_base(plot_base, ROW = 1, /NONEXCLUSIVE, /FRAME)
spb_bckgnd = WIDGET_BUTTON(intercal_base, UVALUE = 'spb_bckgnd', $
                            VALUE = 'Background Subtraction')
WIDGET_CONTROL, spb_bckgnd, SET_BUTTON = control.bckgnd
spb_satur = WIDGET_BUTTON(intercal_base, UVALUE = 'spb_satur', $
                            VALUE = 'Saturation Correction')
WIDGET_CONTROL, spb_satur, SET_BUTTON = control.satur
spb_crosscal = WIDGET_BUTTON(intercal_base, UVALUE = 'spb_crosscal', $
                            VALUE = 'Cross Adj. Applied')
WIDGET_CONTROL, spb_crosscal, SET_BUTTON = control.crosscal


dmy1_base = widget_base(plot_base, ROW = 1)
names = papco_make_choice_names(control.reload_info)
reload_base = papco_CW_BGROUP(dmy1_base, names, ROW = 1, /EXCLUSIVE, /FRAME, $
                   UVALUE='spb_reload', SET_VALUE=control.reload)

names = papco_make_choice_names(control.spec_info)
spec_base = papco_CW_BGROUP(dmy1_base, names, ROW = 1, /EXCLUSIVE, /FRAME, $
                   UVALUE='spb_spec', SET_VALUE=control.spec)


dmy2_base = widget_base(plot_base, ROW = 1)
fld_pa_bin = CW_FIELD(dmy2_base, /ROW,  XSIZE=3, $
                   TITLE = 'Pitch Angle bin width:', $
                   UVALUE= 'fld_pa_bin', VALUE = control.pa_bin)
fld_spec_int = CW_FIELD(dmy2_base, /ROW,  XSIZE=3, $
                   TITLE = '# of ch to interpol., 0=off:', $
                   UVALUE= 'fld_pa_bin', VALUE = control.spec_intpol)

fld_mu = CW_FIELD(panel_left, /ROW,  XSIZE=50, $
                   TITLE = 'Enter mu1, mu2, ... (MeV/G):', $
                   UVALUE= 'fld_mu', VALUE = control.mu)

; --- Choices for HIST modes --------------------------------------------------
names = papco_make_choice_names(control.hist_info)
hist_base=WIDGET_BASE(panel_left,ROW=1,/FRAME)
spb_hist=CW_BGROUP(hist_base,names, ROW=1, /EXCLUSIVE, $
                   LABEL_LEFT=strFormat('HIST modes:', 12,/LEFT), $
                   UVALUE='spb_hist', IDS=hist_ids, SET_VALUE= control.hist)

; --- IPS use MASK choice -----------------------------------------------------
names = papco_make_choice_names(control.mask_info)
mask_base = WIDGET_BASE(panel_left, ROW = 1, /FRAME)
spb_mask = CW_BGROUP(mask_base, names, ROW = 1, /EXCLUSIVE, $
                     LABEL_LEFT='IPS Sun/Earth Mask:', UVALUE='spb_mask',$
                     SET_VALUE=control.mask) 
   
; --- Outer Slider Base -------------------------------------------------------
average_Base=WIDGET_BASE(panel_left, COLUMN=1, /ALIGN_CENTER)

;--- make select widgets for all variables in "average_names" -----------------
;    use papco_panel compound widget for this!
FOR i = 0, n_elements(average_names)-1 DO BEGIN

   r = execute('FROM_VAL = control.'+average_names(i)+'_AVERAGE_FROM_VAL')
   r = execute('TO_VAL = control.'+average_names(i)+'_AVERAGE_TO_VAL')

   FROM_VAL = FROM_VAL > average_range(0, i) < average_range(1, i)
   TO_VAL = TO_VAL > average_range(0, i) < average_range(1, i)

   widgets = papco_panel_AddVar_Ave_Widgets(average_base, average_names(i), $
               FROM_VAL, TO_VAL, $
               average_range(0, i), average_range(1, i), average_range(2, i), $
               average_frmts(i))
   additionalWData=create_struct(widgets, additionalWData)

END

;--- Add some choices that affect rapid slices only ---------------------------
slice_choice_base = WIDGET_BASE(panel_left, ROW=1, /base_align_center)

;choices for USER_3DD_Slice
Slice_base_3dd = WIDGET_BASE(slice_choice_base, ROW=1, $
                             /FRAME, /BASE_ALIGN_CENTER)

labl_base = WIDGET_LABEL(Slice_base_3dd, VALUE = 'USER_3DD_Slice:')

ef_av_3dd = CW_FIELD(Slice_base_3dd, /INTEGER, /RETURN_EVENTS, XSIZE = 4, $
                     TITLE = 'Frame av. (+/-)', $
                     VALUE = fix(control.av_3dd), UVALUE='ef_av_3dd')

;-- set up some toggle switch options for this slice --------------------------
u3dd_toggle_base=WIDGET_BASE(Slice_Base_3dd, COLUMN=1, /NONEXCLUSIVE)
spb_pitch_3dd = WIDGET_BUTTON(u3dd_toggle_base, UVALUE='spb_pitch_3dd', $
                              VALUE='Overlay PA Contour')
WIDGET_CONTROL, spb_pitch_3dd, SET_BUTTON = fix(control.pa_ctr_3dd)

;-- set up some toggle switch options for this slice --------------------------
spb_label_3dd = WIDGET_BUTTON(u3dd_toggle_base, UVALUE='spb_label_3dd', $
                              VALUE='Label cells with PA')
WIDGET_CONTROL, spb_label_3dd, SET_BUTTON = fix(control.pa_lbl_3dd)
   
;--- button to choose "select" variable ---------------------------------------
filter_outer_base=WIDGET_BASE(panel_left, /FRAME, COLUMN=1) 
filter_control_base = WIDGET_BASE(filter_outer_base, ROW = 1)
filter_names = 'select '+select_names
filter_base=CW_BGROUP(filter_control_base, filter_names, ROW=2, /EXCLUSIVE, $
                      UVALUE='spb_filter', SET_VALUE=control.filter)

names = papco_make_choice_names(control.seg_av_info)
seg_av_base = CW_BGROUP(filter_control_base, names, COLUMN=1, /EXCLUSIVE, $
                        UVALUE='spb_seg_av', SET_VALUE=control.seg_av)

;--- Add sliders to filter data - Multiple map bases --------------------------
select_outer_base=WIDGET_BASE(filter_outer_base, COLUMN=1, /align_center)
select_base=WIDGET_BASE(select_outer_base, /align_center)

;--- make select widgets for all variables in "filter_names" ------------------
;    use papco_panel compound widget for this!
FOR i = 0, n_elements(select_names)-1 DO BEGIN

   r = execute('RANGE = control.'+select_names(i)+'_SELECT_RANGE')
   r = execute('FROM_VAL = control.'+select_names(i)+'_SELECT_FROM_VAL')
   r = execute('TO_VAL = control.'+select_names(i)+'_SELECT_TO_VAL')

   FROM_VAL = FROM_VAL > select_range(0, i) < select_range(1, i)
   TO_VAL = TO_VAL > select_range(0, i) < select_range(1, i)

   widgets = papco_panel_AddVar_Sel_Widgets(select_base, select_names(i), $
                 RANGE, FROM_VAL, TO_VAL, $
                 select_range(0, i), select_range(1, i), select_range(2, i), $
                 select_frmts(i))             
   additionalWData=create_struct(widgets, additionalWData)

END

;--- add the papco panel widget for specifying own plot description label -----
widgets=papco_panel_AddDescr_Widgets(panel_right, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

;--- make base for color bar OR plotstyle widget. One only is mapped ----------
line_col_outer_base=WIDGET_BASE(panel_right, COLUMN=1)
line_col_base=WIDGET_BASE(line_col_outer_base)
   
plot_style_base=WIDGET_BASE(line_col_base, COLUMN = 1, MAP=0)
widgets=papco_panel_add_PlotStyleButtons(plot_style_base, plotinfo, 0, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
   
colorbar_base=WIDGET_BASE(line_col_base, COLUMN = 1, MAP=0)
widgets=papco_panel_Addcolorbar_Widgets(colorbar_base, plotinfo, /VERTICAL)  
additionalWData=create_struct(widgets, additionalWData)
   
Scale_Widget_base=widget_base(panel_right, ROW=1,/align_center)

;--- now, add the widgets for choosing the y-axis-ranges ----------------------
widgets=papco_PANEL_AddYScale_Widgets(Scale_Widget_base, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

;--- now, add the widgets for choosing the z-axis-ranges ----------------------
widgets=papco_PANEL_AddZScale_Widgets(Scale_Widget_base, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
widgets=papco_panel_AddActionButtons(panel_bottom, Action, plotinfo, $
                                     ROW = 1, /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
thisWidgetsData={helpfile:'polar_ceppad_panel.help', $
                 fld_mu:fld_mu, fld_pa_bin:fld_pa_bin, $
                 fld_spec_int:fld_spec_int, $
                 fld_sect:fld_sect, fld_spin:fld_spin, $
                 plot_style_base:plot_style_base, $
                 colorbar_base:colorbar_base }

widgetData=papco_PANEL_CreateWidgetData(control, thisWidgetsData)
widgetData=papco_PANEL_CreateWidgetData(widgetData, additionalWData)

;--- return size of the window only here --------------------------------------
IF keyword_set(SIZE_ONLY) THEN BEGIN
    size=WIDGET_INFO(panel_base, /GEOMETRY)
    size_only = size & return, 1
ENDIF 

;--- realize the window -------------------------------------------------------
papco_Panel_Realize, panel_base, widgetData
WIDGET_CONTROL, panel_base, HOURGLASS=1
polar_ceppad_refresh, /ALL

papco_xmanager, 'polar_ceppad_panel', panel_base

;now return data. Since all the widget's info in in a control structure,
;only load it's pointer into USR_PTR1 and return that.

;control is part of the widgetData structure. Extract it
struct_assign, widgetData, *plotinfo.USR_PTR1

;work out if there has been a change in the control structure
plotinfo.modified = plotinfo.modified + $
                    papco_compare_struct(control, *plotinfo.USR_PTR1)

ReturnData={USR_PTR1:plotinfo.USR_PTR1, $
            MODIFIED:plotinfo.modified}

additionalRData=papco_panel_ReturnData(widgetData)

RETURN, create_struct(ReturnData, additionalRData)
   
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO polar_ceppad_refresh, $
;*                SL_DET=SL_DET,SL_SEC,SL_SEC,HIST=HIST
;* 
;* DESCRIPTION:  
;*      update the display of the editor-widgets
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*      have a look at PRO polar_ceppad_panel_event
;*
;* MODIFICATION HISTORY:       
;*     june 1996, written by R.Friedel
;*     jan  1997, to include CLEAN_SURVEY R. Friedel
;******************************************************************************
PRO polar_ceppad_refresh, AVERAGE = AVERAGE, FILTER = FILTER, $
                          LINLOG = LINLOG, COL_LINE = COL_LINE, ALL=ALL

COMMON polar_ceppad_panel_data, widgetData
COMMON polar_ceppad      ;see polar_cep_commons.pro for contents of this block
  
; ---- Handle ALL   -----------------------------------------------------------
IF keyword_set(ALL) THEN BEGIN 
    AVERAGE = 1 & FILTER = 1 
    LINLOG = 1 & COL_LINE = 1
ENDIF 
    
; ---- Lin/Log --------- -----------------------------------------------------
IF KEYWORD_SET(LINLOG) THEN BEGIN 
    IF widgetdata.cal EQ 4 THEN GOTO, out_linlog
    CASE widgetData.pl_type OF
        0: BEGIN
            widgetData.YLINLOG=1 & widgetData.ZLINLOG=1
        END 
        1: BEGIN
            widgetData.YLINLOG=0 & widgetData.ZLINLOG=1
        END 
        2: BEGIN
            widgetData.YLINLOG=0 & widgetData.ZLINLOG=0
            widgetData.ZSCALEMIN=0 & widgetData.ZSCALEMAX=2
            widgetData.MANUALZSCALING=1
        END 
        3: BEGIN
            widgetData.YLINLOG=1 & widgetData.ZLINLOG=1
        END  
        4: BEGIN
            widgetData.YLINLOG=0 & widgetData.ZLINLOG=1
            widgetData.YSCALEMIN = widgetData.YSCALEMIN > 1e-31
        END  
        5: BEGIN
        END  
        ELSE:  BEGIN
            message, 'not done yet', /info
        END  
    ENDCASE 
    print,   widgetData.YLINLOG
    papco_panel_refresh, widgetData, /YSCALE
    papco_panel_refresh, widgetData, /ZSCALE
    out_linlog:
ENDIF 

; ---- COLORBAR or PLT SYYLE widget -------------------------------------------
IF keyword_set(COL_LINE) THEN BEGIN    
    IF widgetData.spec EQ 0 THEN BEGIN 
        widget_control, widgetData.plot_style_base, MAP=1
        widget_control, widgetData.colorbar_base, MAP=0
    endif else BEGIN 
        widget_control, widgetData.plot_style_base, MAP=0
        widget_control, widgetData.colorbar_base, MAP=1
    ENDELSE
ENDIF 

; --- Filter Variable multiple bases ------------------------------------------
IF keyword_set(FILTER) THEN BEGIN 
    ;make a list of widget ID's for the filter variable bases
    FOR i = 0, n_elements(select_names)-1 DO BEGIN
        idx = where(tag_names(widgetData) EQ select_names(i)+'_SELECT_BASE') 
        IF i EQ 0 THEN wid_ids = [widgetData.(idx(i))] ELSE $
           wid_ids = [wid_ids, widgetData.(idx(0))]
    ENDFOR 
    n_wid = n_elements(wid_ids) & map = bytarr(n_wid)
    map(widgetData.filter) = 1
    FOR i = 0, n_wid-1 DO WIDGET_CONTROL, wid_ids(i), MAP = map(i)
ENDIF

; --- Averaging bases ---------------------------------------------------------
IF keyword_set(AVERAGE) THEN BEGIN 
    ;make a list of widget ID's for the average variable bases
    ;wid_ids = [widgetData.DETECTOR_AVERAGE_BASE, $
    ;           widgetData.SECTOR_AVERAGE_BASE, $
    ;           widgetData.ENERGY_AVERAGE_BASE, $
    ;           widgetData.PITCH_AVERAGE_BASE]
    ; 
    ;n_wid = n_elements(wid_ids) & sen = bytarr(n_wid)
 
    ;CASE widgetData.product OF
    ;    0: mask = [1, 1, 1, 1]
    ;    1: mask = [1, 1, 1, 1]
    ;    2: mask = [0, 1, 1, 1]
    ;    3: mask = [0, 1, 1, 1]
    ;ENDCASE 
    ; 
    ;CASE widgetData.pl_type OF
    ;    0: IF widgetData.spec THEN sen = [1, 1, 0, 0] ELSE sen = [1, 1, 1, 0]
    ;    1: IF widgetData.spec THEN sen = [1, 0, 1, 0] ELSE sen = [1, 1, 1, 0]
    ;    2: IF widgetData.spec THEN sen = [1, 0, 0, 0] ELSE sen = [1, 1, 0, 0]
    ;    3: IF widgetData.spec THEN sen = [0, 0, 0, 1] ELSE sen = [0, 0, 1, 1]
    ;    4: IF widgetData.spec THEN sen = [0, 0, 1, 0] ELSE sen = [0, 0, 1, 1]
    ;    5: IF widgetData.spec THEN sen = [0, 0, 1, 0] ELSE sen = [0, 0, 1, 0]
    ;ENDCASE
    ;
    ;sen = sen*mask
    ;IF widgetData.cal EQ 4 THEN sen = [1, 0, 1, 1]
    ;
    ;FOR i = 0, n_wid-1 DO WIDGET_CONTROL, wid_ids(i), SENSITIVE = sen(i)
ENDIF 

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO polar_ceppad_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     june 1996, written by R.Friedel
;*     jan  1997, to include CLEAN_SURVEY R. Friedel
;******************************************************************************
PRO polar_ceppad_panel_event, event

COMMON polar_ceppad_panel_data, widgetData

WIDGET_CONTROL, GET_UVALUE=uval, event.id
    
;test uvalue for structure - select variable sliders return structure uvals.
r = size(uval, /st)
IF r.type_name EQ 'STRUCT' THEN BEGIN
    uval_str = uval & uval = uval.uvalue
ENDIF

CASE uval OF
    'spb_bckgnd':widgetData.bckgnd = event.select
    'spb_satur':widgetData.satur = event.select
    'spb_crosscal':widgetData.crosscal = event.select

    'spb_if': BEGIN
        widgetData.product = event.value
        polar_ceppad_refresh, /AVERAGE
    END 
    
    'spb_filter' : begin
        widgetData.filter = event.value
        polar_ceppad_refresh, /FILTER
    END 

    'spb_spec'   : BEGIN
        widgetData.spec = event.value
        polar_ceppad_refresh, /COL_LINE, /AVERAGE
    END 

    'spb_seg_av'   : widgetData.seg_av = event.value

    'spb_pl_type'  : BEGIN
        widgetData.pl_type = event.value
        polar_ceppad_refresh, /AVERAGE, /LINLOG
    END 

    'spb_av'       : widgetData.av = event.value
    'spb_reload'   : widgetData.reload = event.value

    'spb_cal'      : BEGIN
        widgetData.cal = event.value
        polar_ceppad_refresh, /AVERAGE
    END 

    'ef_av_3dd' :    widgetData.mus = string(event.value)
    'spb_pitch_3dd': widgetData.pa_ctr_3dd = string(event.select)
    'spb_label_3dd': widgetData.pa_lbl_3dd = string(event.select)



    'spb_mask'  : widgetData.mask = event.value
    'sl_det1'   : widgetData.det1 = event.value
    'sl_sec1'   : widgetData.sec1 = event.value
    'sl_eky1'   : widgetData.eky1 = event.value
    'sl_pia1'   : widgetData.pia1 = event.value
    'sl_det2'   : widgetData.det2 = event.value
    'sl_sec2'   : widgetData.sec2 = event.value
    'sl_eky2'   : widgetData.eky2 = event.value
    'sl_pia2'   : widgetData.pia2 = event.value
    'spb_hist'  : widgetData.hist = event.value

    ELSE: BEGIN
        WIDGET_CONTROL, widgetData.fld_mu, GET_VALUE = X
        widgetData.mu = x
        WIDGET_CONTROL, widgetData.fld_spec_int, GET_VALUE = X
        widgetData.spec_intpol = x
        WIDGET_CONTROL, widgetData.fld_pa_bin, GET_VALUE = X
        widgetData.pa_bin = x
        WIDGET_CONTROL, widgetData.fld_sect, GET_VALUE = X
        widgetData.sectors = x
        WIDGET_CONTROL, widgetData.fld_spin, GET_VALUE = X
        widgetData.spins = x

        WidgetData=PAPCO_Panel_event(event, WidgetData)    

    END        
ENDCASE

END
