;******************************************************************************
;* FUNCTION:     
;*      FUNCTION polar_ceppad_pitch_panel,aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the papco_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $papco_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{       canceled:1, $
;*	    		panelHeight:1, $
;*	    		channel:0, $
;*	    		typeVector:[0,0,0,0], $ 
;*	    		manualYScaling:0 $
;*	    		yScaleMin:LONG(1), $
;*	    		yScaleMax:LONG(1), $
;*	    		panelKind:'some string' }
;*
;*      Additionally, some Panel-editors may return these fields in
;*      structure:
;*               {  color:1, $
;*		    plotStyle:1 }
;*			
;*      For an enhanced description read $papco_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	aPlotInfoStruct=papco_getPlotInfoStruct()
;*
;* MODIFICATION HISTORY:       
;*     june 1996, written by R.Friedel
;*     jan  1997, to include CLEAN_SURVEY R. Friedel
;*     feb  1997, to include highres spin averaged mode R. Friedel
;***************************************************************************
FUNCTION  polar_ceppad_pitch_panel, aPlotInfo, GROUP=group, ACTION=ACTION

common polar_ceppad_pitch_panel_data, widgetData
common polar_ceppad ;see m_polar_cep_commons.pro for contents of this block
   
forward_function polar_ceppad_pitch_comb_there
   
; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
   panel_base=papco_widget_base(title=aPlotInfo.panelKind,$
	                        column=1, GROUP_LEADER=group)

;--- the first part of the window will consist in the widgets -----------------
;    that control the height of that plot
   widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- data swittch base   
   swittch_base=WIDGET_BASE(panel_base,ROW=1, /ALIGN_CENTER)
   
;--- Inst selection buttons ---------------------------------------------------
   type_base=CW_BGROUP(swittch_base, pitchNames, ROW = 2, /EXCLUSIVE, /FRAME, $
                       UVALUE='spb_type', $
                       SET_VALUE=aPlotInfo.typeVector(0))
   
; --- Integral / differential -------------------------------------------------
   intdiffNames = ['Integral', 'Differential']
   intdiff_base = CW_BGROUP(swittch_base, intdiffNames, $
                            ROW = 2, /EXCLUSIVE, /FRAME, $
                            UVALUE='spb_intdiff', $
                            SET_VALUE=aplotinfo.typeVector(1))

   reloadNames=['Relaod', 'Adopt' ]  
   reload_base=CW_BGROUP(swittch_base, reloadNames, ROW =2, /EXCLUSIVE, /FRAME, $
                         UVALUE='spb_reload', $
                         SET_VALUE=aPlotInfo.ioptions(1))
   
; --- HIST mode choice --------------------------------------------------------
modeNames = ['HBC (0)', 'ABC (1)']
mode_base = CW_BGROUP(panel_base, modeNames, ROW = 1, /EXCLUSIVE, /FRAME, $
                      LABEL_LEFT='HIST Mode:', UVALUE='spb_mode_base', $
                      SET_VALUE=aplotinfo.ioptions(2))

; --- Spin/Sector combinations  -----------------------------------------------
   rows=n_elements(combNames) / 6
   comb_base=WIDGET_BASE(panel_base, $
                         /FRAME, ROW=rows, /EXCLUSIVE, /ALIGN_CENTER) 
   spb_comb=lonarr(n_elements(combNames))
   for i=0,n_elements(combNames)-1 do $
     spb_comb(i)=WIDGET_BUTTON(comb_base, $
			VALUE=' '+combNames(i)+' ', $
                        UVALUE='spb_comb'+strtrim(string(i), 2))
   
   WIDGET_CONTROL, spb_comb(aPlotInfo.ioptions(0)), SET_BUTTON=1
   
; --- Energy / pitch ----------------------------------------------------------
   ener_pitch_outerBase=WIDGET_BASE(panel_base,/FRAME,COLUMN=1,/ALIGN_CENTER) 
   
   ener_pitch_buttonBase=WIDGET_BASE(ener_pitch_OuterBase, ROW=1, /EXCLUSIVE) 
   spb_energy=WIDGET_button(ener_pitch_buttonbase, $
               VALUE='energy spectrum @ pitch angle:', $
               UVALUE='spb_energy')
   spb_pitch=widget_button(ener_pitch_buttonbase, $
	       VALUE='pitch angle distribution @ energy:', $
	       UVALUE='spb_pitch')
   
    WIDGET_CONTROL, spb_energy, SET_BUTTON=1
    WIDGET_CONTROL, spb_pitch, SET_BUTTON=0
    sl_energy_Pitch=WIDGET_SLIDER(ener_pitch_outerBase, /DRAG, $
		      VALUE=aPlotInfo.typeVector(3) < 14 > 1, $
                      UVALUE='sl_energy_pitch',$
                      MINIMUM=1, MAXIMUM=14,XSIZE=400,/ALIGN_CENTER)
    lbl_energy_Pitch=WIDGET_LABEL(ener_pitch_outerBase, $
                      VALUE=strcenter('pitch angle energy range',50), $
                      /ALIGN_CENTER)			      

;
; <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   

;--- add the widgets for choosing the color bar for color plots ---------------
   widgets=papco_panel_Addcolorbar_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, add the widgets for choosing the y-axis-ranges ----------------------
   widgets=papco_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, add the widgets for choosing the z-axis-ranges ----------------------
   widgets=papco_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
   widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
   thisWidgetsData={helpfile:'polar_ceppad_pitch_panel.help', $
                    mode_base:mode_base, $
                    slider_displaysEnergy:0, $
                    lbl_energy_Pitch:lbl_energy_Pitch, $
                    sl_energy_pitch:sl_energy_Pitch, $
                    typeVector:aPlotInfo.typeVector, $
                    ioptions:aPlotInfo.ioptions, $
                    spb_comb:spb_comb, $
                    spb_pitch:spb_pitch, $
                    spb_energy:spb_energy}

   widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   papco_Panel_Realize, panel_base, widgetData
   polar_ceppad_pitch_panel_refresh, /all
 
; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reportet but the 
; data contained in aPlotInfo would be returned.
   papco_xmanager, 'polar_ceppad_pitch_panel', panel_base

; the returned data consists of the standard-data inherited from
; papco_PANEL.PRO and your new data - e.g. the typeVector-tag.
   returnData={typeVector:widgetData.typeVector, $
               ioptions:widgetData.ioptions}
   additionalRData=papco_panel_ReturnData(widgetData)
   returnData.typeVector(0)=widgetData.typeVector(0)

   return, create_struct(ReturnData, additionalRData)
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO polar_ceppad_pitch_panel_refresh, $
;*                SL_ENERGY=SL_ENERGY
;* 
;* DESCRIPTION:  
;*      update the display of the editor-widgets
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*      have a look at PRO polar_ceppad_pitch_panel_event
;*
;* MODIFICATION HISTORY:       
;*     june 1996, written by R.Friedel
;*     jan  1997, to include CLEAN_SURVEY R. Friedel
;******************************************************************************
PRO polar_ceppad_pitch_panel_refresh, ALL=ALL, $
                                      SL_ENERGY_PITCH=SL_ENERGY_PITCH, $
                                      PB_COMB=PB_COMB,  MODE = MODE

  common polar_ceppad_pitch_panel_data, widgetData
  common polar_ceppad ;see m_polar_cep_commons.pro for contents of this block
  
  if keyword_set(ALL) then begin
    SL_ENERGY_PITCH=1
    PB_COMB=1 & MODE = 1
  endif  
  
  if keyword_set(MODE) THEN BEGIN
      IF widgetData.typeVector(0) LE 1  THEN sensitive = 0 ELSE sensitive = 1
      WIDGET_CONTROL, widgetData.mode_base, SENSITIVE=sensitive       
  endif

  if keyword_set(PB_COMB) then begin
    comb=polar_ceppad_pitch_comb_there()
    for i=0,n_elements(widgetData.spb_comb)-1 do begin
      index=where(strtrim(combNames(i),2) eq comb ,c)
      if c eq 1 then $
        WIDGET_CONTROL, widgetData.spb_comb(i), SET_VALUE=' '+combNames(i) $
      else $
        WIDGET_CONTROL, widgetData.spb_comb(i), SET_VALUE='('+combNames(i)+')'
    endfor 
  endif  

  if keyword_set(SL_ENERGY_PITCH) then begin
    IF widgetData.typeVector(1) THEN BEGIN     ; differential
      WIDGET_CONTROL, widgetData.spb_energy, SENSITIVE=1
      WIDGET_CONTROL, widgetData.sl_energy_pitch, SENSITIVE=1
    ENDIF ELSE BEGIN			     ; integral
      WIDGET_CONTROL, widgetData.spb_energy, SENSITIVE=0
      WIDGET_CONTROL, widgetData.sl_energy_pitch, SENSITIVE=0
    ENDELSE

    IF widgetData.typeVector(2) THEN BEGIN     ; pitch angle
      WIDGET_CONTROL, widgetData.spb_energy, SET_BUTTON=0
      WIDGET_CONTROL, widgetData.spb_pitch, SET_BUTTON=1
      widgetData.slider_displaysEnergy=1
      widgetData.YLINLOG=0
      papco_panel_Refresh,widgetData,/YSCALE
    ENDIF ELSE BEGIN			     ; energy range
      WIDGET_CONTROL, widgetData.spb_energy, SET_BUTTON=1
      WIDGET_CONTROL, widgetData.spb_pitch, SET_BUTTON=0    
      widgetData.slider_displaysEnergy=0
      widgetData.YLINLOG=1
      papco_panel_Refresh,widgetData,/YSCALE
    ENDELSE

; ---- Energy / Pitch - Slider ------------------------------------------------
    species=['(ips ion) ','(ies elec) ','(hist ion) ','(hist elec) ']
    IF widgetData.slider_displaysEnergy THEN BEGIN ;slider shows energy range
 
     CASE widgetData.typeVector(0) of 
        0: yarr=ips_yarr 
        1: yarr=ies_yarr 
        2: yarr=histp_yarr 
        3: yarr=histe_yarr
      ENDCASE
      n_en=n_elements(yarr(*,0))  
      WIDGET_CONTROL, widgetData.sl_energy_pitch, $
        	            SET_SLIDER_MIN=1, SET_SLIDER_MAX=n_en, $
                            SET_VALUE=(widgetData.typeVector(3)<n_en)>1
      range=((widgetData.typeVector(3)<n_en)>1)-1
      rangeLabel=species(widgetData.typeVector(0)) + $
        strtrim(string(yarr(range,0), format='(F8.1)'),2) + ' - ' + $
        strtrim(string(yarr(range,1), format='(F8.1)'),2) + ' keV'
          WIDGET_CONTROL, widgetData.lbl_energy_Pitch, $
          			 SET_VALUE=strcenter('energy range '+$
                                                       rangeLabel, 50)
    ENDIF ELSE BEGIN              ; slider shows pitch angle range
    ; check on which sector resolution was chosen, this sets the PA resolution
      comb=combNames(widgetData.ioptions(0))
      sectors=strmid(comb,0,2)
      case sectors of
        '32': pa_bin=18
        '16': pa_bin=12
        '08': pa_bin=6
        '04': pa_bin=3
      endcase  
      pa_bin_size=180/pa_bin
      widgetData.typeVector(3)=(widgetData.typeVector(3)<pa_bin)>1
      range=(widgetData.typeVector(3)<pa_bin)>1
      rLower=pa_bin_size*(range-1)
      rangeLabel=strtrim(string(rLower),2)+' - '+ $
          		     strtrim(string(rLower+pa_bin_size),2)
    
      rangeLabel=species(widgetData.typeVector(0)) + $
        'pitch angle range '+rangelabel+' degree'
      WIDGET_CONTROL, widgetData.lbl_energy_Pitch, $
                      SET_VALUE=strcenter(rangeLabel, 50)
      WIDGET_CONTROL, widgetData.sl_energy_pitch, $
        	    SET_SLIDER_MIN=1, $
        	    SET_SLIDER_MAX=pa_bin
    ENDELSE                     ; END energy/pitch slider  
  endif ;SL_ENERGY_PITCH  
    
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO polar_ceppad_pitch_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     june 1996, written by R.Friedel
;*     jan  1997, to include CLEAN_SURVEY R. Friedel
;******************************************************************************
PRO polar_ceppad_pitch_panel_event, event

   COMMON polar_ceppad_pitch_panel_DATA, widgetData

   WIDGET_CONTROL,GET_UVALUE=uval, event.id

   IF strleft(uval, 8) EQ 'spb_comb' THEN BEGIN
      widgetData.ioptions(0)=fix(strfrom(uval, 8))
      polar_ceppad_pitch_panel_refresh, /SL_ENERGY_PITCH
      return
   ENDIF 
   
   CASE uval OF

       'spb_mode_base' : BEGIN
            widgetData.ioptions(2)=event.value
            polar_ceppad_pitch_panel_refresh, /PB_COMB
            polar_ceppad_pitch_panel_refresh, /SL_ENERGY_PITCH 
        END     
       'spb_intdiff' :  BEGIN
           widgetData.typeVector(1)=event.value
           polar_ceppad_pitch_panel_refresh, /SL_ENERGY_PITCH
        END   
        'spb_reload' : widgetData.ioptions(1)=event.value
        'spb_type': BEGIN
            widgetData.typeVector(0)=event.value
            polar_ceppad_pitch_panel_refresh, /MODE, /SL_ENERGY_PITCH, /PB_COMB
        END 

	'spb_electrons' : BEGIN
		      widgetData.typeVector(0)=0
                      polar_ceppad_pitch_panel_refresh, /SL_ENERGY_PITCH
		  END

	'spb_ions' : BEGIN
	              widgetData.typeVector(0)=1
                      polar_ceppad_pitch_panel_refresh, /SL_ENERGY_PITCH
		  END

	'spb_energy' : BEGIN
	              widgetData.typeVector(2)=0
                      polar_ceppad_pitch_panel_refresh, /SL_ENERGY_PITCH
		  END

	'spb_pitch' : BEGIN
		      widgetData.typeVector(2)=1
                      polar_ceppad_pitch_panel_refresh, /SL_ENERGY_PITCH
		  END

	'sl_energy_pitch': BEGIN
		      widgetData.typeVector(3)=event.value
                      polar_ceppad_pitch_panel_refresh, /SL_ENERGY_PITCH
		  END

	ELSE: widgetData=papco_panel_Event(event, widgetData)

   ENDCASE

END

;******************************************************************************
function polar_ceppad_pitch_comb_there

  common mjdt, mjdt_start, mjdt_end               ;common time limits in mjdt
  common polar_ceppad ;see m_polar_cep_commons.pro for contents of this block
  common polar_ceppad_pitch_panel_data, widgetData
  COMMON polar_ceppad_pitch_comb_there, old_file
   
  IF NOT keyword_set(old_file) THEN old_file = ''
  mode =  widgetData.ioptions(2)

; check env. variable pointing to data directory
  if not (papco_check_data_env('PO_CEP_PITCH_DIR', PATH=path)) then return,''
 
;check if times are set   
  if not keyword_set(mjdt_start) then begin
    message,'No timerange entered'  &  stop
  endif   
  
; assemble the data file from the time range requested.
  convert_t90_to_date, mjdt_start, year, day_oy, hour, minute, second,$
                       month, cmonth, day_om, /mjdt
  date = string(year,month,day_om,format='(i4.4,i2.2,i2.2)')      
  
  inst=pitchNames(widgetData.typevector(0))
  fln=path + date + '_*_'+inst+'_pitch.idl'
  files=findfile(fln,count=c)
  IF c NE 0 THEN comb=strmid(files,strlen(path)+9,5) ELSE comb = ''

  chosen_comb = combNames(widgetData.ioptions(0))
  pitch_file = date + '_' + chosen_comb + '_' + inst
  fln = path + pitch_file+ '_pitch.idl'
  files=findfile(fln,count=c) 

; if the current file is not old_file, channels need to be read
      
  CASE inst OF
      'HISTe': BEGIN
          channels = 1
          result=polar_ceppad_flux_conv(MODE=mode, /HISTe ,CHANNELS=channels)  
          yarr=transpose(channels)
      END    
      ELSE: BEGIN
          IF old_file NE fln THEN BEGIN
              IF c NE 0 THEN BEGIN
                  message, 'Getting Energy channels for: '+ pitch_file, /cont
                  restore,fln
                  yarr=fltarr(input_header.NBANDS,2)
                  for i=0,input_header.NBANDS-1 do BEGIN
                      yarr(i,0) = input_header.CH_POSITIONS(0,i,0,0)
                      yarr(i,1) = input_header.CH_POSITIONS(1,i,0,0)
                  ENDFOR
                  old_file = fln
              ENDIF ELSE BEGIN     
                  message,'No energy channels for: ' + pitch_file, /cont
                  yarr=fltarr(10,2) & yarr(*,0)=fltarr(10) & yarr(*,1)=fltarr(10)+1
                  old_file = '' 
              ENDELSE 
          ENDIF ELSE BEGIN
              yarr=fltarr(10,2) & yarr(*,0)=fltarr(10) & yarr(*,1)=fltarr(10)+1
          ENDELSE    
      END  
  ENDCASE

  CASE inst OF
     'IPS':  ips_yarr=yarr  
     'HISTe': histe_yarr=yarr
  ENDCASE 


  return,comb
  
end
