;***************************************************************************
;***************************************************************************
;* PROCEDURE AND FILE:     
;*      cep_l1_to_if.pro
;* 
;*      This file belongs to CEPPAD data processing
;*
;* DESCRIPTION:  
;*      a wrapper program to allow batch processing of CEPPAD Level 1 to 
;*      RAL intermediat format files (IF files). 
;*
;*      Expects the l1 files in directory pointed to by environmental variable
;*      PO_CEP_L1_DIR
;*
;*      Puts output IF files in directory pointed to by environmental variable
;*      PO_CEP_IF_DIR
;*
;*      the routine checks in PO_CEP_IF_DIR if the files have already been 
;*      processed, unless the keyword NOCHECK hes been set. This is used for   
;*      updates,if the same processing has to be re-run with updated software.
;*
;*      Read Martin carter's Manual for detailed description on MODES.
;*      and suitable values for N, M
;* 
;* INPUTS:       
;*      type          a string indicating IPS, IES or HIST
;* 	if_format     a string indicating the RAL IF format required
;* 
;* OUTPUTS:
;*      none, but a journal file cep_l1_to_if+date+time.txt is written
;*
;* KEYWORDS:
;*      LOOK_ONLY     only checks to see which files need to be processed
;*      FORCE         if set forces processing, overwriting old files
;*      N             no of of sectors averaged per spin
;*      M             spin averaging number  
;*      START         the number of days to jump at beginning. Used to
;*                    skip bad data files after a crash. 
;* 
;* CALLING SEQUENCE:
;*	cep_l1_to_if,type,if_format,LOOK_ONLY=LOOK_ONLY,FORCE=FORCE,N=N,M=M
;*
;* MODIFICATION HISTORY:       
;*     January 1997, written by R Friedel
;***************************************************************************
;***************************************************************************
PRO cep_l1_to_if, type, if_format, LOOK_ONLY=LOOK_ONLY, FORCE=FORCE, $
                                   N=N, M=M, START=START

; establish error handling

; open journal file
name='cep_if-'
datetime=str_sep(strcompress(systime()),' ')
time_tag=datetime(1)+'_'+datetime(2)+'_'+datetime(3)
name=name+time_tag+'.txt'
openw,unit,name,/get_lun

pjl,unit,''
pjl,unit,'Batch job to create RAL IF files from CEPPAD level one files'
pjl,unit,''

if_names=['SPIN_AVERAGED', $
          'MAJOR_SPIN_AVERAGED', $
          'INTEGRAL_ENERGY', $
          'LOW_RESOLUTION', $
          'FULL_RESOLUTION', $
          'FULL_ANGLE', $
          'SURVEY', $
          'CLEAN_SURVEY', $
          'SPIN_n', $
          'CLEAN_n', $
          'SPIN_nZm', $
          'CLEAN_nZm' ]

pos = strpos(if_names, if_format)
index = where(pos NE -1, c)
IF c NE 0 THEN begin
      if keyword_set(N) then begin	;substitute "n" by value passed in N
         pos=strpos(if_format,'n')
         if_format = strmid(if_format, 0, pos)+varprt(N)+ $
           strmid(if_format, pos+1, strlen(if_format))
      ENDIF
      if keyword_set(M) then begin	;substitute "m" by value passed in N
         pos=strpos(if_format,'m')
         if_format = strmid(if_format, 0, pos)+varprt(M)
      endif
ENDIF ELSE BEGIN
      print, 'IF format '+if_format+' not found'
      return
ENDELSE

pjl,unit,'MODE=',if_format
pjl,unit,''

;check if paths are set...
path=getenv('PO_CEP_L1_DIR')
if strlen(path) eq 0 then begin
    pjl,unit,'Environmental variable (PO_CEP_L1_DIR) for L1 data path not set'
    return
endif
newfiles=findfile(path+'/PO_CEP_L1_'+type+'*01*',count=nof)
if nof eq 0 then begin
    pjl,unit, 'no data files found '
    return
endif

if type eq 'IPS' or type eq 'IES' then len=14 else len=16

stop

start_if:

newdates=strarr(n_elements(newfiles))
for i=0,n_elements(newfiles) -1 do begin
    check=strpos(newfiles(i),'.DAT')
    newdates(i)=strmid(newfiles(i),check-len,len-3)
endfor

path=getenv('PO_CEP_IF_DIR')
oldfiles=findfile(path+'/PO_CEP_RAL_'+if_format+'_'+type+'*',count=nof)

if keyword_set(START) then skip=START else skip=0

for i=0,n_elements(newfiles) -1 do begin
      check=strpos(oldfiles,newdates(i))
      if (max(check) eq -1) then begin
         if (skip le 0) then begin
            pjl,unit,'day not  processed yet: ',newdates(i)
            t=systime(1)
            if not keyword_set(look_only) then begin
               cep_l1toif, newdates(i), MODE=if_format
               pjl,unit,'time for ',newdates(i),' : ', $
                              (systime(1)-t)/60,' Minutes'
            endif 
         endif else skip=skip-1
      endif else begin
         pjl,unit,'day already processed: ',newdates(i)
         if keyword_set(FORCE) then begin
            pjl,unit,'reprocessing day: ',newdates(i)
            cep_l1toif, newdates(i), MODE=if_format
            pjl,unit,'time for ',newdates(i),' : ',(systime(1)-t)/60,' Minutes'
         endif
      endelse
endfor

catch, Error_status
if Error_status ne 0 then begin
      pjl,unit,''
      pjl,unit,'Error message', !ERR_STRING
      pjl,unit,''
      START=1   ; skip this file and continue!
      goto, start_if
endif

close,unit
free_lun,unit

END 

;simple wrapper for the print routine to both write to screen and to file
pro pjl,unit,string1,string2,string3,string4,string5,string6,string7,string8

nostr=n_params()
case nostr of
   2: begin
         print,string1
         printf,unit,string1
   end
   3: begin
         print,string1,string2
         printf,unit,string1,string2
   end
   4: begin
         print,string1,string2,string3
         printf,unit,string1,string2,string3
   end
   5: begin
         print,string1,string2,string3,string4
         printf,unit,string1,string2,string3,string4
   end
   6: begin
         print,string1,string2,string3,string4,string5
         printf,unit,string1,string2,string3,string4,string5
   end
   else: begin
         print,string1,string2,string3,string4,string5
         printf,unit,string1,string2,string3,string4,string5
   endelse
endcase

END 
