;******************************************************************************
;* PROCEDURE AND FILE:     
;*      cep_l1_to_if.pro
;* 
;*      This file belongs to CEPPAD data processing
;*
;* DESCRIPTION:  
;*      a wrapper program to allow batch processing of CEPPAD Level 0 to 
;*      CEPPAD L1 files. 
;*
;*      Expects the lz files in directory pointed to by environmental variable
;*      PO_CEP_LZ_DIR    
;*
;*      Puts output l1 files in directory pointed to by environmental variable
;*      PO_CEP_L1_DIR
;*
;*      the routine checks in PO_CEP_LZ_DIR if the files have already been 
;*      processed, unless the keyword NOCHECK hes been set. This is used for   
;*      updates,if the same processing has to be re-run with updated software.
;*
;*      Read Martin carter's Manual for detailed description of cep_lztol1
;* 
;* INPUTS:       
;*      dates         date(s) to process - can be an array
;* 
;* OUTPUTS:
;*      none
;*
;* KEYWORDS:
;*      NOCHECK       if set forces processing, overwriting old files
;*      START         the number of days to jump at beginning. Used to
;*                    skip bad data files after a crash. 
;*      BATCH         batch mode, check for missing and do
;*      PROB_IF       read list of dates of if processing problem and do those
;*      FORCE         do even if prob is set
;* 
;* CALLING SEQUENCE:
;*	cep_cdlz_to_l1
;*
;* MODIFICATION HISTORY:       
;*     January 1997, written by R Friedel
;******************************************************************************
pro cep_lz_to_l1, dates, PROB_IF = PROB_IF, FORCE = FORCE

papco, /set_only
close,/all 

;check if paths are set...
if not (papco_check_data_env('PO_CEP_LZ_DIR', path=l0_path)) then return
if not (papco_check_data_env('PO_CEP_L1_DIR', path=l1_path)) then return
if not (papco_check_data_env('PO_CEP_PROB', path=p_path)) then return

IF keyword_set(FORCE) THEN force = FORCE ELSE force = 0

;set temp dir
tmp_dir = papco_getenv('PAPCO_DOWNLOAD')
;set problem path
prob_dir = p_path+'/l1/'

;level zero files are in directories per year, and gzipped.
;file naming is of form po_lz_cep_19961226_v01.dat.gz

IF keyword_set(PROB_IF) THEN BEGIN
    if_prob_path = p_path+'/if/'
    files = findfile(if_prob_path+'*', count = c)
    IF c NE 0 THEN dates = strmid(files, strlen(if_prob_path), 8) ELSE BEGIN
        message, 'No problem dates found', /cont
        return
    ENDELSE 
ENDIF

;now loop through dates and make l1 files
i = 0
WHILE i LE n_elements(dates)-1 DO BEGIN
    print, ''
    message, 'Doing for '+dates(i), /cont
    date = dates(i)
    close, /all
    yyyy = strmid(date, 0, 4)
    yymmdd = strmid(date, 2, 6)

    ;check if problem flag is set
    prog_check = file_search(prob_dir+date+'.prob',count=p_c)
    IF ((p_c NE 0) AND (force GE 3)) THEN BEGIN
        message, 'Problem flag set, skip', /info
        i = i+1 & CONTINUE
    ENDIF  

    message, 'Looking on '+l0_path+yyyy, /cont
    file = findfile(l0_path+yyyy+'/po_lz_cep_'+date+'*.gz', count = c)

    IF c EQ 0 THEN BEGIN
        prob_txt = 'No ceppad L0 file found'
        message, prob_txt, /cont
        spawn, 'echo '+prob_txt+'cep_lz_tp_l1'+' >! '+prob_dir+date+'.prob'
        i = i+1 & continue
    ENDIF 

    file = file(c-1);take the higher version number!
    IF c GT 1 THEN i = i+(c-1)

    ;now that l0 file is found, gunzip onto l0_path
    papco_sepdir, file(0), pp, ff, ee
    unzipped_l0 = pp+ff+strmid(ee, 0, strlen(ee)-3)
    cmd = 'gunzip -c '+file(0)+' > '+unzipped_l0
    print, cmd & spawn, cmd

    ;make RAL date
    result=convert_timeToSeconds(date,/T90,/MJDT)
    convert_t90_to_Date,result.value, year, day_oy, hour, minute, second, $
                                      month, cmonth, day_om, /MJDT
    l1date=string(year,day_oy,format="(i4.4,i3.3)")

    t=systime(1)    
    CD, tmp_dir, current = cur_dir

    ;trap errors in call to  make_level_1
    CATCH, Error_Status
    if Error_Status NE 0 then BEGIN 
        msg=!ERR_STRING 
        err_jump:
        msg = 'Error trapped in make_level_1: '+msg
        message, msg, /cont 
        spawn, 'echo "' + msg + '" > '+prob_dir+date+'.prob' 
        CATCH,/cancel                ;turn off error checking
        CD, cur_dir
        file_delete, unzipped_l0, /allow_nonexistent, /verb 
        file_delete, tmp_dir+'PO_CEP_L1_*', /allow_nonexistent, /verb
        close, /all
        i = i+1 & continue
    ENDIF 
    
    ;call original l1 routine - pass in full level 0 filename 
    make_level_1, unzipped_l0
        
    CATCH,/cancel                ;turn off error checking

    CD, cur_dir

    ;rm unzippel l0 file
    file_delete, unzipped_l0;, /verb

    ;see if files were made!
    files = findfile(l0_path+'*_'+l1date+'_*', count = c)
    IF c GE 5 THEN BEGIN
        msg = 'No output files produced' & GOTO, err_jump
    ENDIF

    print,'time taken to make l1 for ',date,' : ',(systime(1)-t),' Sec'

    message, 'zipping and moving to L1 yyyy directory', /cont
    ;rename L1 files to bu file names to keep consistency

    ;skip this step for now...

    GOTO,  no_BU

    ;set up array of BU L1 extensions
    BU_ext=['HK','HSe','HSp','IES','IPS','SH']
    ;set up array with Aerospace filename leaders
    AS_lead=['PO_CEP_L1_HK_',  'PO_CEP_L1_HISTe_', 'PO_CEP_L1_HISTp_', $
             'PO_CEP_L1_IES_', 'PO_CEP_L1_IPS_',   'PO_CEP_L1_SH_' ]
    for k=0,5 do begin
        file = file_search(l0_path + AS_lead(k) + l1date+'*.DAT', count = c)
        FOR j = 0, c-1 DO BEGIN
            RAL_name=file(j)
            BU_name =l0_path+date+string(j, format="(i2.2,'.')")+ BU_ext(k)
            file_move, RAL_name, BU_name, /overwrite, /verb
        ENDFOR     
    ENDFOR 

    CD, l0_path, current = cur_dir
    print, 'zip -m '+date+'.zip '+date+'*'
    spawn, 'zip -m '+date+'.zip '+date+'*'
    CD, cur_dir

    no_BU:

    CD, tmp_dir, current = cur_dir
    cmd = 'zip -m '+date+'.zip *'+l1date+'*.DAT'
    print, cmd & spawn, cmd

    ;see if l1_path subyear exists, make if needed
    IF NOT(file_test(l1_path+yyyy)) THEN papco_create_dir, l1_path+yyyy
    file_move, date+'.zip', l1_path+yyyy+'/', /overwrite, /verb

    print,'time taken to make l1 zip for ',date,' : ',(systime(1)-t),' Sec'
    
    close, /all
    i = i+1
ENDWHILE

END 


;******************************************************************************
;* PROCEDURE AND FILE:     
;*      cep_l0_to_l1_batch.pro
;******************************************************************************
PRO cep_l0_to_l1_batch, DATE1 = DATE1, DATE2 = DATE2, FORCE = FORCE

if not (papco_check_data_env('PO_CEP_L1_DIR', path=l1_path)) then return
if not (papco_check_data_env('PO_CEP_LZ_DIR', path=l0_path)) then return

;get start, end date, list of dates
IF keyword_set(DATE1) THEN date1  = DATE1 ELSE date1 = '19960225'
Year1  = fix(strmid(date1, 0, 4))
Month1 = fix(strmid(date1, 4, 2))
Day1   = fix(strmid(date1, 6, 2))
start = JULDAY(Month1, Day1, Year1)

IF keyword_set(DATE2) THEN BEGIN
    date2  = DATE2 
    Year2  = fix(strmid(date2, 0, 4))
    Month2 = fix(strmid(date2, 4, 2))
    Day2   = fix(strmid(date2, 6, 2))
ENDIF ELSE BEGIN
    final = systime(/julian)
    CALDAT, final, Month2, Day2, Year2 
    date2 = string(Year2, Month2, Day2, format = "(i4.4,i2.2,i2.2)")
ENDELSE 
final = JULDAY(Month2, day2, year2)

dates = TIMEGEN(START = start, FINAL = final, UNITS = 'Days')
c1 = n_elements(dates)
dates_str = strarr(c1)
FOR i = 0, c1-1 DO BEGIN
    CALDAT, dates(i), Month, Day, Year
    dates_str(i) = string(Year, Month, Day, format = "(i4.4,i2.2,i2.2)")
ENDFOR 
check = bytarr(c1)

;work out how many Level 1 dates are not present

;get list of dates to do for batch
l1_files = file_search(l1_path+'/????/*.zip',count=c_l1)
l1_dates = strmid(l1_files, strlen(l1_path)+5, 8)
    message,  'found '+varprt(c_l1)+ ' l1 files', /info
    l0_files = file_search(l0_path+'/????/*.gz',count=c_l0)
    l0_dates = strmid(l0_files, strlen(l0_path)+15, 8)
    message,  'found '+varprt(c_l0)+ ' l0 files', /info

idx = bytarr(c_l0)
FOR i = 0, c_l0-1 DO BEGIN
    index = where(l0_dates(i) EQ l1_dates, c)
    IF c EQ 0 THEN idx(i) = 1
ENDFOR


IF keyword_set(force) THEN BEGIN
    dates = l0_dates
    c = n_elements(dates)
ENDIF ELSE BEGIN 
    idx2 = where(idx EQ 1, c)
    dates = l0_dates(idx2)
ENDELSE 
message,  'Doing '+varprt(c)+' dates', /info

FOR i = 0, c1-1 DO BEGIN 
    idx = where(dates EQ dates_str(i), c2)
    IF c2 EQ 1 THEN check(i) = 1
ENDFOR 

idx = where(check EQ 1, c2)
dates_to_do = dates_str(idx)
message,  'Doing '+varprt(c2)+' dates', /info

cep_lz_to_l1, dates_to_do

END
