;***************************************************************************** 
;* PROCEDURE AND FILE:     
;*      cep_make_surv_froml1.pro
;* 
;*      This file belongs to CEPPAD data processing
;*
;* DESCRIPTION:  
;*      a wrapper program making RAL CEPPAD 
;*      Survey plots from L1 files for a specific day.
;*
;*      Works on existing L1 format files. Expects the L1 files in directory 
;*      pointed to by environmental variable PO_CEP_L1_DIR.
;*      Makes IES CLEAN_SURVEY and IPS SURVEY RAL if format files.
;*
;*      Makes an eph file from the CDHF orbit files which is required by
;*      RAL survey plot generation.
;*
;*      Produces one output plot (postscript file) per day into
;*      directory pointed to by environmental variable PO_CEP_PS_DIR 
;*
;* INPUTS: 
;*      year, a year string. Processing will only be performed on that year.
;*      date, a day string 
;* 
;* OUTPUTS:
;*      postcript files are written to directory pointed to by environmental
;*      variable PO_CEP_PS_DIR
;*      RAL if files are written to directory pointed to PO_CEP_IF_DIR.
;*
;* KEYWORDS
;*      GET_L1_ZIP if set forces L1 files to be searched in zipped archive
;*      GETIF      if set it uses the zipped archive if files
;*      LINKS      if set, then links already exisit
;*      NOZIP      if set, it inhipits zipping of if files
;*      DO_SURVEY  if set, make RAL survey plot 
;*      HIGHRES    if set, make 1-spin highres if files too.  
;*      MAKE_IF    if set, make if files from l1 files
;*      NO_REMOVE  if set, don't delete L1, if files
;*      
;*
;* CALLING SEQUENCE:
;*	cep_make_surv_froml1,date
;*
;* MODIFICATION HISTORY:       
;*     January 1997, written by R Friedel at LANL
;******************************************************************************
PRO cep_make_surv_froml1, year, date,GET_L1_ZIP=GET_L1_ZIP,GETIF=GETIF, $
                          LINKS=LINKS, $
                          NOZIP=NOZIP, DO_SURVEY=DO_SURVEY, $
                          HIGHRES=HIGHRES, MAKE_IF=MAKE_IF, NO_REMOVE=NO_REMOVE
  
  COMMON cep_path, l1_path, link_path, if_zip_path, l1_mass_storage, $
                   l1_archive, eph_path, if_path
  
  COMMON firstplot, first_flag
  
  print,''
  print,'Batch job to create RAL IES survey plots and "if" files from L1 files'
  print,'for day ',date
  print,''

;check if paths are set
  l1_path=getenv('PO_CEP_L1_DIR')
  if strlen(l1_path) eq 0 then begin
    print,'Env. variable (PO_CEP_L1_DIR) for L1 data path not set'
    return
  endif

  IF strmid(l1_path, strlen(l1_path)-1,1) NE '/' THEN l1_path=l1_path+'/'

;create required paths
  base_data = getenv('PO_CEP_BASE')
  if_zip_path=base_data+'if_archive/'
  l1_mass_storage=base_data+'l1_archive/'


  ;if_zip_path='/n/papco/data/polar/ceppad/if_mass_storage/'
  if_path=getenv('PO_CEP_IF_DIR')
  ;l1_mass_storage='/n/papco/data/polar/ceppad/bu_l1/mass_storage/'
  ;l1_archive='/n/papco/data/polar/ceppad/bu_l1/archive/'+year+'/'
  eph_path='/n/radbelt/dlnk/polar/eph/'
  
;make a "file flag" to indicate this date is being worked on. Thgis
;flag is read by the batch processing rotuine to avoid running the
;same date more than once, if more than one batch process is running
;at the same time.
  spawn, 'touch /n/radbelt/dlnk/polar/ceppad/if/'+date+'.busy'
  
;get exisiting level 1 archive file from mass storage to l1_path and unzip
;check if level 1 archive file has already been retreived
  if keyword_set(GET_L1_ZIP) then begin
    print,''
    print,'getting l1 files from archive or mass storage'
    l1_zipfile=date+'.zip '
    ;check if files are already retreived    
    file_there=findfile(l1_archive+l1_zipfile,count=c)
    if c eq 0 then begin   ;get from mass storage, put in archive
      print,'cp '+l1_mass_storage+l1_zipfile+' '+l1_archive
      spawn,'cp '+l1_mass_storage+l1_zipfile+' '+l1_archive  
    endif
    ;unzip from archive into l1 directory
    print,'unzip -u '+l1_archive+l1_zipfile+' -d '+l1_path
    spawn,'unzip -u '+l1_archive+l1_zipfile+' -d '+l1_path
  endif  
  
  l1date=cep_get_l1day(date+'_v01.dat')   ;make RAL date

  if not keyword_set(LINKS) then begin   ;creates links to RAL L1 filenames
    print,''
    ;set up array of BU L1 extensions
    BU_ext=['HK','HSe','HSp','IES','IPS','SH']
    ;set up array with Aerospace filename leaders
    AS_lead=['PO_CEP_L1_HK_',  'PO_CEP_L1_HISTe_', 'PO_CEP_L1_HISTp_', $
             'PO_CEP_L1_IES_', 'PO_CEP_L1_IPS_',   'PO_CEP_L1_SH_'     ]
    ;find out version number of unzipped L1 files
    l1_files=findfile(l1_path+'*'+date+'*',count=c)
    if c ne 0 then begin
      version=strmid(l1_files(0),strlen(l1_path)+8,2)
      print,'L1 version number: ',version 
    endif else begin
      print,'no l1 files'
      return
    endelse  
    for i=0,5 do begin
      BU_name =date + version + '.' + BU_ext(i)
      RAL_name=AS_lead(i) + l1date + '.DAT'
      print,'ln -s '+l1_path+BU_name+' '+l1_path+RAL_name
      spawn,'ln -s '+l1_path+BU_name+' '+l1_path+RAL_name
    endfor
  endif
  
  if keyword_set(GETIF) then begin ;get if files from zipped archive and unzip
    print,''
    zip_file='PO_CEP_RAL_IF_'+date
    check_zip=findfile(if_path+zip_file+'.zip',count=nf)
    if nf eq 1 then begin
       print,'zip exist in if directory'
    endif else begin
      check_zip=findfile(if_zip_path+zip_file+'.zip',count=nf)
      if nf ne 0 then begin   
        print,'zip exist on mass storage'
        print,'cp '+if_zip_path+zip_file+'.zip'+' '+if_path
        spawn,'cp '+if_zip_path+zip_file+'.zip'+' '+if_path      
      endif else begin
        print,'zip file not found'
        stop
      endelse  
    endelse  
    print,'cd ' + if_path + '; unzip -u '+ zip_file+'.zip'
    spawn,'cd ' + if_path + '; unzip -u '+ zip_file+'.zip'
    print,'ls ' + if_path +'*'+date+'.DAT'
    spawn,'ls ' + if_path +'*'+date+'.DAT',result
    survey_if_files=strarr(5); if file list for RAL processing
    survey_if_files(0)=result(1)
    survey_if_files(1)=result(5)
    survey_if_files(2)=result(3)
    survey_if_files(3)=result(4)
    survey_if_files(4)=result(2)
  endif
  
  if keyword_set(MAKE_IF) then begin ;make RAL if format
    print,''
    ;call RAL make survey IF file routine
    survey_if_files = cep_survey_files ( /L1,  DATE=l1date)
    ;make additional IF files
    ;cep_l1toif, 'IES_'  +l1date, MODE='CLEAN_1Z16', FILE=ies_file
    ;high resolution "if" files if required
    if keyword_set(HIGHRES) then begin
      print,''
      cep_l1toif, 'IES_'+l1date, MODE='CLEAN_1', FILE=ies_file_2
      cep_l1toif, 'IPS_'+l1date, MODE='SPIN_1', FILE=ips_file_2
      cep_l1toif, 'HISTe_'+l1date, MODE='SPIN_1', FILE=histe_file_2
      cep_l1toif, 'HISTp_'+l1date, MODE='SPIN_1', FILE=histp_file_2
    endif    
  endif
  
  if keyword_set(DO_SURVEY) then begin  ;make RAL Survey plot
    path=getenv('PO_CEP_PS_DIR' )
    if not keyword_set(first_flag) then begin
      out=cep_survey(RF=survey_if_files)
      wset, !D.WINDOW  &  wdelete, !D.WINDOW
      first_flag=1
    endif        
    out=cep_survey(RF=survey_if_files, PS=path+date+'_ies_ral.ps') 
    print,'Written output file '+path+date+'_ies_ral.ps'
  endif
  
  if not keyword_set(GETIF) and not keyword_set(NOZIP)then begin
    ; zip the if files made and move to tempest
    print,''
    print,'Zipping if files and moving to tempest'
    zip_file=if_path+'PO_CEP_RAL_IF_'+date
    ; check if a zipped file already exisits
    check_zip=findfile(if_zip_path+zip_file+'.zip',count=nf)
    if nf ne 0 then begin ;add /update to exisiting zip file
      print,''
      print,'zip exist - updading'
      print,'cp '+if_zip_path+zip_file+'.zip'+' '+if_path
      spawn,'cp '+if_zip_path+zip_file+'.zip'+' '+if_path
      print,'cd ' + if_path + '; zip -u -m -T '+ zip_file + ' *'+l1date+'*.DAT'
      spawn,'cd ' + if_path + '; zip -u -m -T '+ zip_file + ' *'+l1date+'*.DAT'
      print,'mv '+zip_file+'.zip '+if_zip_path+'PO_CEP_RAL_IF_'+date+'.zip '
      spawn,'mv '+zip_file+'.zip '+if_zip_path+'PO_CEP_RAL_IF_'+date+'.zip '
    endif else begin   ;create new zip file
      ; first cd to directory with if's, then zip
      print,''
      print,'making new zip file'
      print,'cd ' + if_path + '; zip -m -T '+ zip_file + ' *'+date+'*.DAT'
      spawn,'cd ' + if_path + '; zip -m -T '+ zip_file + ' *'+date+'*.DAT'
      print,'mv '+zip_file+'.zip '+if_zip_path+'PO_CEP_RAL_IF_'+date+'.zip '
      spawn,'mv '+zip_file+'.zip '+if_zip_path+'PO_CEP_RAL_IF_'+date+'.zip &'
    endelse
  endif
  
;remove the temporary link files and the unzipped L1 files
  if not keyword_set(LINKS) then begin
    print,''
    print,'rm '+l1_path+'PO_CEP_L1*'+l1date+'*'
    spawn,'rm '+l1_path+'PO_CEP_L1*'+l1date+'*'
  endif  
  if not keyword_set(NO_REMOVE) then begin
    print,''
    print,'rm '+l1_path+'*'+l1date+'*'
    spawn,'rm '+l1_path+'*'+l1date+'*'
    print,'rm '+l1_path+'*'+date+'*'
    spawn,'rm '+l1_path+'*'+date+'*'
    print,'rm '+if_path+'*'+date+'*'
    spawn,'rm '+if_path+'*'+date+'*'
  endif
  print,''
  
;remove "flag file" 
  spawn, 'rm /n/papco/data/polar/ceppad/if/'+date+'.busy'

end


;******************************************************************************
;program to batch process a list of dates through cep_make_surv_froml1
; year - string!
pro cep_make_surv_froml1_batch, year, CHECK_ONLY=CHECK_ONLY, SKIP=SKIP, ALL=ALL
  
  COMMON cep_path, l1_path, link_path, if_zip_path, l1_mass_storage, $
    l1_archive, eph_path, if_path
      
  l1_archive='/n/papco/data/polar/ceppad/bu_l1/archive/'+year+'/'
  l1_work='/n/papco/data/polar/ceppad/bu_l1/archive/'
  if_archive='/n/papco/data/polar/ceppad/if_archive/'
  if_archive='/n/papco/data/polar/ceppad/if_mass_storage/'
  if_path=getenv('PO_CEP_IF_DIR')
  
  l1_list=findfile(l1_archive+year+'*.zip',count=c1)
  if_list=findfile(if_archive+'*'+year+'*.zip',count=c2)
  
  if keyword_set(ALL) then begin
  ;make a list all l1 files on archive (not mass storage!)
    count=n_elements(l1_list)
    date_list=strarr(count)
    for i=0,count-1 do date_list(i)=strmid(l1_list(i),strlen(l1_archive),8)
  endif else begin
  ;make a list of date where there are no if_archive files for the year
  ;but there ARE l1_files
    for i=0, n_elements(l1_list)-1 do begin
      l1_file=l1_list(i)
      p=strpos(l1_file,year)
      l1_date=strmid(l1_file,p+5,8) 
      result=strpos(if_list,l1_date)
      index=where(result ne -1,c)
      if c eq 0 then if n_elements(date_list) eq 0 then date_list=l1_date $
        else date_list=[date_list,l1_date]
    endfor  
  endelse  
  c=n_elements(date_list)
  if year eq '1996' then first=5 else first=0
  print,date_list(first:c-1)
  
  print,'found',c-first,'  l1 files to process'
  
  if keyword_set(CHECK_ONLY) then return
  if keyword_set(SKIP) then first=first+skip else first=first
  
  if c ne 0 then begin
    for i=first,c-1 do begin
      print,''
      print,'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~'
      date=date_list(i)
      l1date=cep_get_l1day(date+'_v01.dat') ;make RAL date
      print,'doing',i+1,'  of',c,'   for date: ',date
      ;re-check if file hasn't been done in the meantime - or is in process
      result=findfile(if_archive+'*'+date+'*.zip',count=c3)
      if c3 ne 0 then begin
        message, 'if zip exisits',/cont
        continue
      endif
      result=findfile('/n/papco/data/polar/ceppad/if/'+date+'.busy',count=c4)
      if c4 ne 0 then begin
        message, 'date being processed already',/cont
        continue
      endif  
      ;catch any errors during batch process, jump to next date!
      catch, Error_status
      ;Error_status=0
      if Error_status ne 0 then begin
        print, 'ERROR FOR DATE ',date
        print, 'Error index: ',Error_status
        print, 'Error message: ',!ERR_STRING
        ;clean up "corpses" from run
        print,''
        print,'rm '+l1_path+'PO_CEP_L1*'+l1date+'*'
        spawn,'rm '+l1_path+'PO_CEP_L1*'+l1date+'*'
        print,''
        print,'rm '+l1_path+'*'+l1date+'*'
        spawn,'rm '+l1_path+'*'+l1date+'*'
        print,'rm '+l1_path+'*'+date+'*'
        spawn,'rm '+l1_path+'*'+date+'*'
        print,'rm '+if_path+'*'+date+'*'
        spawn,'rm '+if_path+'*'+date+'*' 
        close,/all
        if Error_status eq -110 then begin ;make new level one file!
          catch, /CANCEL          ;disable error catching
          result=cep_lztol1_lanl(date)
          if result(0) eq '' then continue
          cep_make_surv_froml1,year,date,/make_if,/links,/no_remove
          change_ral_cep_l1_date, date
          print,'cp '+l1_path+date+'.zip '+ l1_archive
          spawn,'cp '+l1_path+date+'.zip '+ l1_archive
        endif  
        continue
      endif  
      cep_make_surv_froml1,year,date,/make_if,/get_l1_zip, /DO_SURVEY
    endfor  
  endif
  
end

;******************************************************************************
;program to fix 2 letter year to 4 letter year on ceppad l1 files

pro fix_cep_l1_date
  
  l1_archive='/data1/POLAR/CEPPAD/BU_L1/archive/'
  spawn,'pwd',current_dir
  
  CD, l1_archive
  
  file_list=findfile('96*.zip*',count=c)
  if c ne 0 then begin
    print,'found',c,'  l1 files to process'
    for i=0,c-1 do begin
      print,''
      print,'~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~'
      fln=file_list(i)
      date=strmid(fln,0,6)
      new_date='19'+date
      print,fln,'   ',date,'   ',new_date
      print,'unzip '+fln
      spawn,'unzip '+fln
      print,'rm '+fln
      spawn,'rm '+fln
      ;rename all l1 files to 4 letter year
      l1_files=findfile('*'+date+'*',count=c)
      if c ne 0 then begin
        for j=0,c-1 do begin
          print,'mv '+l1_files(j)+' 19'+l1_files(j)
          spawn,'mv '+l1_files(j)+' 19'+l1_files(j)
        endfor  
      endif  
      ;zip renamed l1 files to 4 letter year filename
      print,'zip -m '+new_date+'.zip *'+new_date+'*'
      spawn,'zip -m '+new_date+'.zip *'+new_date+'*'
    endfor
  endif
  
  CD, current_dir(0)
  
end  


;******************************************************************************
;program to change RAL doy date format to mmdd in level1 file produced
;with RAL code. Change file names too to BU file names used!

pro change_ral_cep_l1_date, date, UNZIP=UNZIP
  
  l1_path=getenv('PO_CEP_L1_DIR')
  spawn,'pwd',current_dir
  
;set up array of BU L1 extensions
  BU_ext= ['HK','HSe','HSp','IES','IPS','SH','IES_RATES']
;set up array with Aerospace filename leaders
  AS_lead=['PO_CEP_L1_HK_',  'PO_CEP_L1_HISTe_', 'PO_CEP_L1_HISTp_', $
           'PO_CEP_L1_IES_1', 'PO_CEP_L1_IPS_',   'PO_CEP_L1_SH_'   , $
           'PO_CEP_L1_IES_RATES_1' ]  
  
  l1date=cep_get_l1day(date+'_v01.dat')   ;make RAL date

  CD, l1_path
  
  if keyword_set(UNZIP) then begin
    file_list=findfile('*'+date+'.zip*',count=c)
    print,file_list
    spawn,'unzip -l '+file_list(0),result
    spawn,'unzip '+file_list(0)
    spawn,'rm  '+file_list(0)
  endif else begin
    spawn,'ll *'+l1date+'*.DAT', result
  endelse  
  
  for i=0,n_elements(result)-1 do print, result(i)

  for i=0,n_elements(AS_lead)-1 do begin
    res=strpos(result,AS_lead(i))
    index=where(res ne -1,c)
    if c ne 0 then begin
      oldname=strmid(result(index(0)),res(index(0)),strlen(result(index(0))))
      newname=date+'01.'+BU_ext(i)
      print,'mv '+oldname+' '+newname
      spawn,'mv '+oldname+' '+newname
    endif  
  endfor  
  
  print,'zip -m '+date+'.zip '+date+'*'
  spawn,'zip -m '+date+'.zip '+date+'*'
  
  CD, current_dir(0)
  
end


;******************************************************************************
;program to delet old gif files from cep l1 archives

pro del_cep_l1_zip_gifs
  
  l1_archive='/data/polar/ceppad/bu_l1/archive/1996'
  l1_archive='/data1/POLAR/CEPPAD/BU_L1/archive/1996'
  spawn,'pwd',current_dir
  
  CD, l1_archive
  
  file_list=findfile('1996*.zip*',count=c)
  if c ne 0 then begin
    print,'found',c,'  l1 files to process'
    for i=0,c-1 do begin
      file=file_list(i)
      date=strmid(file,0,8)
      print,'unzip -l '+file
      spawn,'unzip -l '+file, result
      index=strpos(result,'.old')
      index=where(index ne -1, c)
      if c ne 0 then begin
        for j=0,n_elements(index)-1 do begin
          line=result(index(j))
          p=strpos(line,'199')
          name=strmid(line,p,strlen(line))
          print,'zip -d '+file+' '+name
          spawn,'zip -d '+file+' '+name
        endfor  
      endif else print,'    -> no .old files found'
    endfor
    
  endif  
end  

;******************************************************************************
;program to make missing level 1 files

pro make_missing_l1, year, MOVE=MOVE, LOOK=LOOK, HAVE_DATES = HAVE_DATES, $
                     DATE = DATE
  
COMMON make_missing_l1, do_dates

;check if paths are set
  l1_path=getenv('PO_CEP_L1_DIR')
  if strlen(l1_path) eq 0 then begin
    print,'Env. variable (PO_CEP_L1_DIR) for L1 data path not set'
    return
  endif
  
  base_data = getenv('PO_CEP_BASE')
  l0_path=getenv('PO_CEP_LZ_DIR') 
  l1_path_ms = base_data+'l1_archive/'
  l1_archive = base_data+'l1/'+ string(year,format="(i4.4)")+'/'

  ;l0_path='/n/papco/data/polar/ceppad/l0/'
  ;l1_path_ms='/n/papco/data/polar/ceppad/bu_l1/mass_storage/'
  ;l1_archive='/n/papco/data/polar/ceppad/bu_l1/archive/'+ $
  ;           string(year,format="(i4.4)")+'/'
  l1_mass_storage='/n/papco/data/polar/ceppad/bu_l1/mass_storage/'
  l1_mass_storage =  l1_path_ms
  
  IF keyword_set(DATE) THEN BEGIN
      do_dates = [DATE]
      GOTO, have_do_dates 
  ENDIF

  IF keyword_set(HAVE_DATES) THEN goto, have_do_dates 

  startday=julday(01,01,year)
  endday=julday(12,31,year)
  
  all_dates=strarr(endday-startday+1)
  yr_st=string(year,format="(i4.4)")
  ;make array of day numbers
  for i=startday,endday do begin
    caldat,i,month,day,year
    all_dates(i-startday)=string(year,month,day,format="(i4.4,i2.2,i2.2)")
  endfor  
    
  ;get list of exisiting level 0 and 1 files for year  
  l0_list=findfile(l0_path+'po_lz_cep_'+yr_st+'*',count=c)
  l1_list=findfile(l1_path_ms+yr_st+'*.zip',count=c)
  
  for i=0,n_elements(all_dates)-1 do begin
    date=all_dates(i)
    result=strpos(l0_list,date)
    index=where(result ne -1,count)
    if count ne 0 then begin
      result=strpos(l1_list,date)
      index=where(result ne -1,count)
      if count ne 0 then begin
        print,date,' -> found level 0 file, found level 1 file'
      endif else begin
        print,date,' -> found level 0 file, no level 1 file'
        if n_elements(do_dates) eq 0 then do_dates=[date] $
          else do_dates=[do_dates,date]
      endelse  
    endif else begin
      print,date,' -> no level zero file on mass storage'
    endelse  
  endfor  
  
  have_do_dates:

  count=n_elements(do_dates)
  if count ne 0 then begin
    print,''
    print,'Need to make l1 files from l0 files for dates:'
    print,do_dates
    if keyword_set(LOOK) then return
    if year eq 1996 then st=1 else st=0
    for i=st,count-1 do begin
      date=do_dates(i)
      if keyword_set(MOVE) then begin
        print,'cp '+l1_archive+date+'.zip '+ l1_mass_storage
        spawn,'cp '+l1_archive+date+'.zip '+ l1_mass_storage
      endif else begin
        l0_file=findfile(l0_path+'po_lz_cep_'+date+'*.dat')
        result=cep_lztol1(LZ_FILE=l0_file)
        if result(0) eq '' then continue
        change_ral_cep_l1_date, date
        print,'mv '+l1_path+date+'.zip '+ l1_archive
        spawn,'mv '+l1_path+date+'.zip '+ l1_archive
      endelse 
stop 
    endfor
  endif else begin  
    print,'No l1 files missing for',year
  endelse  
    
end
