;******************************************************************************
;* PROCEDURE AND FILE:     
;*      cep_make_ral_if.pro
;* 
;*      This file belongs to CEPPAD data processing
;*
;* DESCRIPTION:  
;*      a wrapper program making RAL CEPPAD if data files and / or
;*      survey plots from L1 files for a specific day.
;*
;*      Works on exisiting L1 format files. Expects the L1 files in directory 
;*      pointed to by environmental variable PO_CEP_L1_DIR.
;*      Makes IES CLEAN_SURVEY and IPS SURVEY RAL if format files.
;*
;*      Produces one output plot (postscript file) per day into
;*      directory pointed to by environmental variable PO_CEP_PS_DIR 
;*
;* INPUTS: 
;*      date, a day string
;* 
;* OUTPUTS:
;*      postcript files are written to directory pointed to by environmental
;*      variable PO_CEP_PS_DIR
;*      RAL if files are written to directory pointed to PO_CEP_IF_DIR.
;*
;* KEYWORDS
;*      IF_CNTR     0 do nothing
;*                  1 make if files - left in tmp_dir
;*                  2 make if files - move output to if directory
;*                  3 get if files from if directory -> tmp_dir
;*      NOZIP     if set, it inhibits zipping of if files
;*      SURVEY    if set, plot is made
;*      CCR_COPY  set, make ccr png and copy to mrfrench
;*      TO, T1    if set, contain the start and end time string "11:00:00"
;*      ROUTINE   if set, do the routine survey IF file production
;*      SINGLE    set to single IF type  to be made
;*      MODE      mode for SINGLE processing, eg SPIN_1Z16
;*      NODELETE
;*      FORCE       0 only process files if they do not already exist
;*                  1 only redo survey plots, implies IF_CNTR=3
;*                  2 redo survey plots and if files
;*                  3 redo survey plots, if files and l1 files
;*                  4 redo survey plots, if files, l1 files and l0 files
;*                    (delete existing l0 fiels and re-get from gsfc)  
;*
;* CALLING SEQUENCE:
;*	cep_make_ral_if, date
;*
;* MODIFICATION HISTORY:       
;*     January 1997, written by R Friedel at LANL
;******************************************************************************
PRO cep_make_ral_if, date, IF_CNTR = IF_CNTR, FORCE = FORCE, $
                     SURVEY=SURVEY, MODE=MODE, T0=T0, T1=T1, $
                     NODELETE=NODELETE, CCR_COPY = CCR_COPY, $
                     ROUTINE=ROUTINE, SINGLE=SINGLE, FILE = FILE
papco, /set_only

print,'% ---------------------------------------------------------------------'
message,'Wrapper to create RAL "if" files from L1 files', /cont
print, ' for day '+date
print,''

;check if start/end time keywords are set
if not keyword_set(T0) then xut1='00:00:00' else xut1=T0
if not keyword_set(T1) then xut2='23:59:59' else xut2=T1
if not keyword_set(IF_CNTR) THEN if_cntr = 0 ELSE if_cntr = IF_CNTR
IF keyword_set(FORCE) THEN force = FORCE ELSE force = 0
   
; check env. variable pointing to data directory
if not (papco_check_data_env('PO_CEP_L1_DIR', path=l1_path)) then return
if not (papco_check_data_env('PO_CEP_IF_DIR', path=if_path)) then return
if not (papco_check_data_env('PO_CEP_HISTe_DIR', path=if_1)) then return
if not (papco_check_data_env('PO_CEP_HISTp_DIR', path=if_2)) then return
if not (papco_check_data_env('PO_CEP_IES_DIR',   path=if_3)) then return
if not (papco_check_data_env('PO_CEP_IPS_DIR',   path=if_4)) then return
if not (papco_check_data_env('PO_CEP_MF_DIR',    path=if_5)) then return
if not (papco_check_data_env('PO_CEP_HK_DIR',    path=if_6)) then return
if not (papco_check_data_env('PO_CEP_PROB', path=p_path)) then return

;set path to problem directory - reports on processing problems
prob_dir = p_path+'/if/'
prob_txt = 'cep_make_ral_if: '

;set temp dir
tmp_dir = papco_getenv('PAPCO_DOWNLOAD')

; make RAL date
result=convert_timeToSeconds(date,/T90,/MJDT)
convert_t90_to_Date,result.value, year, day_oy, hour, minute, second, $
                    month, cmonth, day_om, /MJDT
l1date=string(year,day_oy,format="(i4.4,i3.3)")
yyyy = string(year,format="(i4.4)")
  
;see if we just need to get if files from if dir
IF if_cntr EQ 3 THEN BEGIN
    if_files = file_search(IF_path+'PO_CEP_RAL*'+date+'.DAT', count = n)
    IF n NE 0 THEN BEGIN
        message, 'Found exisiting if files', /info
        file_copy, if_files, tmp_dir 
        GOTO, have_if
    ENDIF ELSE BEGIN
        message, 'Need to make if files', /info
    ENDELSE 
ENDIF 

;get / make ceppad l1 files. They exisit all in zipped archive files
;locally, in yearly directroies, or need to be made from the level0 files. 
l1_zip_file=l1_path+yyyy+papco_get_pathsep()+date+'.zip'
l1_zip = file_search(l1_zip_file, count=nf)

IF (nf EQ 0) OR (force GE 3) THEN BEGIN           ;make l1
    message, 'calling cep_lz_to_l1 for '+date, /info
    cep_lz_to_l1, date, FORCE = FORCE
ENDIF

l1_zip = file_search(l1_zip_file, count=nf)

IF nf EQ 1 THEN BEGIN           ; get the L1 file from the zipped archive
    message,'l1 zip exists',/cont
    print,'unzip -u '+ l1_zip_file+ ' -d '+ tmp_dir
    spawn,'unzip -u '+ l1_zip_file+ ' -d '+ tmp_dir
    print,''
ENDIF ELSE BEGIN
    msg = 'l1 zip '+l1_path+l1_zip_file+' does not exist on achive'
    message, msg, /cont
    spawn, 'echo '+msg+'cep_make_ral_if'+' >! '+prob_dir+date+'.prob'
    message, 'Cannot make if files', /info
    goto, out
ENDELSE

;see if there were any zero size l1 files delete those before processing
l1_files = file_search(tmp_dir+'PO_CEP_L1*.DAT', count = c)
FOR i = 0, c-1 DO IF file_test(l1_files(i), /ZERO_LENGTH) THEN $
    file_delete, l1_files(i), /verb

;leave out - new L1 files are in aerospace naming convention.
;make links, create new if files

;GOTO, no_BU

;create links to RAL L1 filenames - set up array of BU L1 extensions

BU_ext=['HK','HSe','HSp','IES','IPS','SH']
;set up array with Aerospace filename leaders
AS_lead=['PO_CEP_L1_HK_',  'PO_CEP_L1_HISTe_', 'PO_CEP_L1_HISTp_', $
         'PO_CEP_L1_IES_', 'PO_CEP_L1_IPS_',   'PO_CEP_L1_SH_' ]
;check for version number in BU L1 files
BU_L1 = findfile(tmp_dir+date+'*')
AS_L1 = file_search(tmp_dir+'PO_CEP_L1*', count = c)
IF c NE 0 THEN GOTO, no_BU
ver = strmid(BU_L1(0), strlen(tmp_dir)+8, 2)
cd, tmp_dir, current = curr_dir
FOR i=0,5 DO BEGIN
    BU_name  = date + ver + '.' + BU_ext(i)
    RAL_name = AS_lead(i) + l1date + '.DAT'
    cmd = 'ln -s '+BU_name+' '+RAL_name
    print, cmd & spawn, cmd
ENDFOR 
cd, curr_dir

no_BU:

; call required routine for RAL if format production
IF if_cntr GT 1 THEN BEGIN   
      
    ;set l1,if dir to tmp dir for processing, then set back!
    SETENV, 'PO_CEP_L1_DIR='+tmp_dir
    SETENV, 'PO_CEP_IF_DIR='+tmp_dir
    SETENV, 'PO_CEP_HISTe_DIR='+tmp_dir     
    SETENV, 'PO_CEP_HISTp_DIR='+tmp_dir
    SETENV, 'PO_CEP_IES_DIR='+tmp_dir
    SETENV, 'PO_CEP_IPS_DIR='+tmp_dir               
    SETENV, 'PO_CEP_MF_DIR='+tmp_dir              
    SETENV, 'PO_CEP_HK_DIR='+tmp_dir

    ;enable error checking on RAL routine called below.
    ;CATCH, Error_Status  
    Error_Status = 0
    if Error_Status NE 0 then begin
        msg=!ERR_STRING 
        message,'Error trapped (cep_l1toif)',/cont
        message, msg, /cont
        CATCH,/cancel 
        SETENV, 'PO_CEP_L1_DIR='+l1_path
        SETENV, 'PO_CEP_IF_DIR='+if_path
        SETENV, 'PO_CEP_HISTe_DIR='+if_1
        SETENV, 'PO_CEP_HISTp_DIR='+if_2
        SETENV, 'PO_CEP_IES_DIR='+if_3
        SETENV, 'PO_CEP_IPS_DIR='+if_4
        SETENV, 'PO_CEP_MF_DIR='+if_5
        SETENV, 'PO_CEP_HK_DIR='+if_6
        spawn, 'echo '+prob_txt+'cep_l1toif'+' >! '+prob_dir+date+'.prob'
        goto,out
    ENDIF
      
    if keyword_set(ROUTINE) then BEGIN
        cep_l1toif, 'HK_'+l1date, MODE='', FILE=hk_file
        message, 'Made if file: ', /cont
        print, '  ', hk_file
        cep_l1toif, 'HK_'+l1date, MODE='', /BF, FILE=mf_file
        message, 'Made if file: ', /cont
        print, '  ', mf_file
        cep_l1toif, 'IES_'+l1date, MODE='CLEAN_SURVEY', FILE=ies_file
        message, 'Made if file: ', /cont
        print, '  ', ies_file
        cep_l1toif, 'IPS_'+l1date, MODE='SURVEY', FILE=ips_file
        message, 'Made if file: ', /cont
        print, '  ', ips_file
        cep_l1toif, 'HISTe_'+l1date, MODE='SURVEY', FILE=histe_file
        message, 'Made if file: ', /cont
        print, '  ', histe_file
        cep_l1toif, 'HISTp_'+l1date, MODE='SURVEY', FILE=histp_file
        message, 'Made if file: ', /cont
        print, '  ', histp_file
    endif   
      
    IF keyword_set(SINGLE) THEN BEGIN
        filespec = single+'_'+l1date
        message, l1date, /info
        cep_l1toif, filespec, T0=T0, T1=T1, MODE=mode, FILE=file
    ENDIF 
      
    CATCH,/cancel     ;turn off error checking

    ;set l1,if dir back!
    SETENV, 'PO_CEP_L1_DIR='+l1_path
    SETENV, 'PO_CEP_IF_DIR='+if_path
    SETENV, 'PO_CEP_HISTe_DIR='+if_1
    SETENV, 'PO_CEP_HISTp_DIR='+if_2
    SETENV, 'PO_CEP_IES_DIR='+if_3
    SETENV, 'PO_CEP_IPS_DIR='+if_4
    SETENV, 'PO_CEP_MF_DIR='+if_5
    SETENV, 'PO_CEP_HK_DIR='+if_6      
ENDIF

have_if:

IF keyword_set(SURVEY) then BEGIN
    print,''
    IF NOT keyword_set(if_cntr) THEN BEGIN ;look for exisiting if files
        IF_files = file_search(if_path+'*'+date+'.DAT', count = n)
        IF n LT 6 THEN BEGIN
            message, "Didn't find all/any if files",/info
            GOTO, out
        ENDIF 
    ENDIF 
    CATCH, Error_Status  
    if Error_Status NE 0 then begin
        msg=!ERR_STRING 
        message,'Error trapped (cep_survey)',/cont
        message, msg, /cont
        CATCH,/cancel 
        SETENV, 'PO_CEP_IF_DIR='+if_path
        spawn, 'echo '+prob_txt+'cep_survey'+' >! '+prob_dir+date+'.prob'
        goto, out
    endif       
    
    if_files=[tmp_dir+'PO_CEP_RAL_CLEAN_SURVEY_IES_'+date+'.DAT', $
              tmp_dir+'PO_CEP_RAL_SURVEY_IPS_'+date+'.DAT', $
              tmp_dir+'PO_CEP_RAL_SURVEY_HISTe_'+date+'.DAT', $
              tmp_dir+'PO_CEP_RAL_SURVEY_HISTp_'+date+'.DAT', $
              tmp_dir+'PO_CEP_RAL_HK_'+date+'.DAT', $
              tmp_dir+'PO_CEP_RAL_MF_'+date+'.DAT' ]
    path=getenv('PO_CEP_PS_DIR')

    message,'Doing RAL IES survey plot',/cont
    print, ''
    out=cep_survey_plot(RF=if_files, PS=path+date+'_ies_ral.ps', TYPE=0)

    message,'Doing RAL HIST survey plot',/cont
    print, ''
    out=cep_survey_plot(RF=if_files, PS=path+date+'_hist_ral.ps', TYPE=1)

    CATCH,/cancel     ;turn off error checking
    
    IF keyword_set(CCR_COPY) THEN BEGIN
        CD,  getenv('PO_CEP_PS_DIR'), current = old_path
        cmd = '/u/friedel/bin/install_ccr_plots '+date+'_ies_ral.ps'
        print, cmd & spawn, cmd
        cmd = '/u/friedel/bin/install_ccr_plots '+date+'_hist_ral.ps'
        print, cmd & spawn, cmd
        CD,  old_path
    ENDIF

ENDIF

IF if_cntr EQ 2 THEN BEGIN      ;move output to if directory
    if_files = file_search(tmp_dir+'PO_CEP_RAL*'+date+'.DAT', count = n)
    IF n NE 0 THEN file_move, if_files, if_path+yyyy+'/', /overwrite, /verb ELSE $
        message, 'No if files made', /info
ENDIF 

out:
   
;clean up
if not keyword_set(NODELETE) then BEGIN
    message, 'doing cleanup...', /cont
    result = file_search(tmp_dir+'PO_CEP_L1*', count = n)
    IF n NE 0 THEN file_delete, result, /verb
    result = file_search(tmp_dir+date+'*', count = n)
    IF n NE 0 THEN file_delete, result, /verb
    result = file_search(tmp_dir+'PO_CEP_RAL*'+date+'.DAT', count = n)
    IF n NE 0 THEN file_delete, result, /verb
ENDIF 

print,'% ---------------------------------------------------------------------'
   
END 

;******************************************************************************
PRO cep_make_ral_if_batch, DATE1 = DATE1, DATE2 = DATE2, CHECK_IF = CHECK_IF, $
                           CHECK_RALPLOT = CHECK_RALPLOT,  FORCE = FORCE, $
                           LIST_ONLY = LIST_ONLY,  CHECK_L1 = CHECK_L1, $
                           CHECK_L0 = CHECK_L0, IF_CNTR = IF_CNTR, $
                           MODE = MODE, SINGLE = SINGLE

papco, /set_only
if not (papco_check_data_env('PO_CEP_IF_DIR', path=if_path)) then return
if not (papco_check_data_env('PO_CEP_PS_DIR', path=ral_path)) then return
if not (papco_check_data_env('PO_CEP_L1_DIR', path=l1_path)) then return
if not (papco_check_data_env('PO_CEP_LZ_DIR', path=l0_path)) then return
if not (papco_check_data_env('PO_CEP_PROB', path=p_path)) then return

;get start, end date, list of dates
IF keyword_set(DATE1) THEN date1  = DATE1 ELSE date1 = '19960225'
Year1  = fix(strmid(date1, 0, 4))
Month1 = fix(strmid(date1, 4, 2))
Day1   = fix(strmid(date1, 6, 2))
start = JULDAY(Month1, Day1, Year1)

IF keyword_set(DATE2) THEN BEGIN
    date2  = DATE2 
    Year2  = fix(strmid(date2, 0, 4))
    Month2 = fix(strmid(date2, 4, 2))
    Day2   = fix(strmid(date2, 6, 2))
ENDIF ELSE BEGIN
    final = systime(/julian)
    CALDAT, final, Month2, Day2, Year2 
    date2 = string(Year2, Month2, Day2, format = "(i4.4,i2.2,i2.2)")
ENDELSE 
final = JULDAY(Month2, day2, year2)
dates = TIMEGEN(START = start, FINAL = final, UNITS = 'Days')
c1 = n_elements(dates) & dates_str = strarr(c1)
FOR i = 0, c1-1 DO BEGIN
    CALDAT, dates(i), Month, Day, Year
    dates_str(i) = string(Year, Month, Day, format = "(i4.4,i2.2,i2.2)")
ENDFOR 

message, date1+' to '+date2+', '+varprt(c1)+' days', /cont

;make test array - all set to "1" intially, -> do all
check = bytarr(c1) &  check(*) = 1

IF keyword_set(CHECK_RALPLOT) THEN BEGIN
    IF CHECK_RALPLOT EQ 1 THEN ralplot_name = '*ies_ral.ps*' ELSE $
                               ralplot_name = '*hist_ral.ps*' 
    ;check dates of ralplots made, and get list of missing ones
    have_ral = file_search(ral_path+ralplot_name, count = c0)
    have_ral_dates = strmid(have_ral, strlen(ral_path), 8)
    idx = where( (long(have_ral_dates) GE long(date1)) AND $
                 (long(have_ral_dates) LE long(date2)), c0)
    IF c0 NE 0 THEN have_ral_dates = have_ral_dates(idx)
    message, 'In range '+varprt(c0)+' ies_ral.ps.gz files', /cont
    FOR i = 0, c1-1 DO BEGIN 
        idx = where(have_ral_dates EQ dates_str(i), c2)
        IF c2 EQ 1 THEN check(i) = 0 ;don't do those we have
    ENDFOR 
    idx = where(check EQ 1, c3)
    IF c3 NE 0 THEN BEGIN
        dates_to_do = dates_str(idx)
        message, 'RP: '+varprt(c3)+' days to do', /cont
    ENDIF ELSE BEGIN
        message, 'RP: No dates to do, returning', /cont
        return 
    ENDELSE
ENDIF
   
IF keyword_set(CHECK_L0) THEN BEGIN
    ;check dates of level 0 files, only do for available dates
    have_l0 = file_search(l0_path+'/*/*.gz', count = c0)
    have_l0_dates = strmid(have_l0, strlen(l0_path)+15, 8)
    idx = where( (long(have_l0_dates) GE long(date1)) AND $
                 (long(have_l0_dates) LE long(date2)), c0)
    IF c0 NE 0 THEN have_l0_dates = have_l0_dates(idx)
    message, 'In range '+varprt(c0)+' level 0 files', /cont
    FOR i = 0, c1-1 DO BEGIN 
        idx = where(have_l0_dates EQ dates_str(i), c2)
        IF c2 EQ 0 THEN check(i) = 0
    ENDFOR         
    idx = where(check EQ 1, c3)
    IF c3 NE 0 THEN BEGIN
        dates_to_do = dates_str(idx)
        message, 'L0: '+varprt(c3)+' days to do', /cont
    ENDIF ELSE BEGIN
        message, 'L0: No dates to do, returning', /cont
        return 
    ENDELSE
ENDIF 

IF keyword_set(CHECK_L1) THEN BEGIN
    ;check dates of level 1 files, only do for available dates
    have_l1 = file_search(l1_path+'/*/*.zip', count = c0)
    have_l1_dates = strmid(have_l1, strlen(l1_path)+5, 8)
    idx = where( (long(have_l1_dates) GE long(date1)) AND $
                 (long(have_l1_dates) LE long(date2)), c0)
    IF c0 NE 0 THEN have_l1_dates = have_l1_dates(idx)
    message, 'In range '+varprt(c0)+' level 1 files', /cont
    FOR i = 0, c1-1 DO BEGIN 
        idx = where(have_l1_dates EQ dates_str(i), c2)
        IF c2 EQ 0 THEN check(i) = 0
    ENDFOR         
    idx = where(check EQ 1, c3)
    IF c3 NE 0 THEN BEGIN
        dates_to_do = dates_str(idx)
        message, varprt(c3)+' days to do', /cont
    ENDIF ELSE BEGIN
        message, 'No dates to do, returning', /cont
        return 
    ENDELSE
ENDIF 

IF keyword_set(CHECK_IF) THEN BEGIN
    ;check dates of if dates, only do those missing
    fm = '????/PO_CEP_RAL_'+mode+'_'+single+'_????????.DAT'
    have_if = file_search(if_path+fm, count = c0)
    pos = strpos(fm, '.DAT')
    have_if_dates = strmid(have_if, strlen(if_path)+pos-8, 8)
    idx = where(long(have_if_dates) GT long(date1), c0)
    IF c0 NE 0 THEN have_if_dates = have_if_dates(idx)
    message, 'In range '+varprt(c0)+' IF files', /cont
    FOR i = 0, c1-1 DO BEGIN 
        idx = where(have_if_dates EQ dates_str(i), c2)
        IF c2 EQ 1 THEN check(i) = 0 ;don't do those we have!
    ENDFOR
    idx = where(check EQ 1, c3)
    IF c3 NE 0 THEN BEGIN
        dates_to_do = dates_str(idx)
        message, 'IF: '+varprt(c3)+' days to do', /cont
    ENDIF ELSE BEGIN
        message, 'IF: No dates to do, returning', /cont
        return 
    ENDELSE
ENDIF 

print, ''
dates_to_do = dates_str(where(check EQ 1, c3))
message, varprt(c3)+' days to do', /cont

IF NOT keyword_set(FORCE) THEN BEGIN 
    ;get list of problem dates, don't re try doing those
    prob_path = p_path+'if/'
    prob_if = file_search(prob_path+'*', count=c4)
    prob_if_dates = strmid(prob_if, strlen(prob_path), 8)
    check = bytarr(c3) & check(*) = 1
    FOR i = 0, c3-1 DO BEGIN 
        idx = where(prob_if_dates  EQ dates_to_do(i), c5)
        IF c5 EQ 0 THEN check(i) = 0 
    ENDFOR 
    idx = where(check NE  1, c6)
    IF c6 NE 0 THEN dates_to_do = dates_to_do(idx) ELSE dates_to_DO = ['']
    message, 'Found '+ varprt(c3-c6)+' problem dates', /cont 
    message, 'PR: '+varprt(c6)+' days to do', /cont
ENDIF ELSE c6 = c3

IF keyword_set(LIST_ONLY) THEN BEGIN
    print, dates_to_do & return
ENDIF 
    
for i = 0, c6-1 do begin

    products = ['HISTe','SPIN_1Z16']

    cep_make_ral_if, dates_to_do(i), IF_CNTR = IF_CNTR, FORCE = FORCE, $
      MODE = MODE, SINGLE = SINGLE, ROUTINE = ROUTINE, SURVEY = SURVEY

    ;for survey and routine processing
    ;cep_make_ral_if, dates_to_do(i), IF_CNTR = 2, FORCE = FORCE, $
    ;                 /ROUTINE, /SURVEY, /CCR_COPY
        
ENDFOR
  
END  


;******************************************************************************
; To be run from cron. Looks at last week, makes sure all l1 files, if
; files and RAL plots are made.
;******************************************************************************
PRO cep_make_ral_if_update

;last day to do is today
final = systime(/julian)
CALDAT, final, Month2, Day2, Year2 
date2 = string(Year2, Month2, Day2, format = "(i4.4,i2.2,i2.2)")

;first day to do is a month before "today"
start = final-30.0
CALDAT, start, Month1, Day1, Year1 
date1 = string(Year1, Month1, Day1, format = "(i4.4,i2.2,i2.2)")

cep_make_ral_if_batch, DATE1 = DATE1, DATE2 = DATE2, $
                       /CHECK_RALPLOT, /CHECK_L0

END
