;***************************************************************************
;***************************************************************************
;* PROCEDURE AND FILE:     
;*      cep_make_survey.pro
;* 
;*      This file belongs to CEPPAD data processing
;*
;* DESCRIPTION:  
;*      a wrapper program to allow batch processing of RAL CEPPAD 
;*      Survey plots.
;*
;*      Works on exisiting if format files: requires IES CLEAN_SURVEY
;*                                               and IPS SURVEY
;*      and produces one output plot (postscript file) per day into
;*      directory pointed to by environmental variable PO_CEP_PS_DIR
;*      
;*
;*      Expects the IF files in directory pointed to by environmental variable
;*      PO_CEP_IF_DIR
;*
;*      the routine checks in for exisiting poscript files and only processes
;*      new days, unless the keyword NOCHECK hes been set. This is used for   
;*      updates, if the same processing has to be re-run with updated software.
;* 
;* INPUTS: none  
;*
;* 
;* OUTPUTS:
;*      postcript files are written to directory pointed to by environmental
;*      variable PO_CEP_PS_DIR
;*
;* KEYWORDS:
;*      LOOK_ONLY     only checks to see which files need to be processed
;*      NOCHECK       if set forces processing, overwriting old files
;*      START          the number of days to jump at beginning. Used to
;*                    skip bad data files after a crash. 
;*      LANL          if set do LANL preprocessing required to get files
;* 
;* CALLING SEQUENCE:
;*	cep_make_survey.pro
;*
;* MODIFICATION HISTORY:       
;*     January 1997, written by R Friedel
;*     January 1997, modified by R Friedel at LANL to work with LANL setup
;***************************************************************************
;***************************************************************************
PRO cep_make_survey,LOOK_ONLY=LOOK_ONLY,NOCHECK=NOCHECK,START=START

; open journal file
   name='cep_ral_survey-'
   datetime=str_sep(strcompress(systime()),' ')
   time_tag=datetime(1)+'_'+datetime(2)+'_'+datetime(3)
   name=name+time_tag+'.txt'
   openw,unit,name,/get_lun

   pjl,unit,''
   pjl,unit,'Batch job to create RAL survey plots from CEPPAD RAL if files'
   pjl,unit,''

;check if paths are set...

   path=getenv('PO_CEP_IF_DIR')
   if strlen(path) eq 0 then begin
      pjl,unit,'Environmental variable (PO_CEP_IF_DIR) for L1 data path not set'
      return
   endif
   newfiles=findfile(path+'/PO_CEP_RAL_CLEAN_SURVEY_IES*',count=nof)
   if nof eq 0 then begin
      pjl,unit, 'no if data files found '
      return
   endif

   dates=strarr(n_elements(newfiles))
   for i=0,n_elements(newfiles) -1 do begin
      check=strpos(newfiles(i),'.DAT')
      dates(i)=strmid(newfiles(i),check-7,7)
   endfor

   pjl,unit,'No files to be processed: ',n_elements(newfiles)

   path=getenv('PO_CEP_PS_DIR' )
   oldfiles=findfile(path+'/PO_CEP_RAL_SURVEY*.ps',count=nof)

   if keyword_set(START) then skip=START else skip=0

   for i=0,n_elements(newfiles) -1 do begin
      check=strpos(oldfiles,dates(i))
      if (max(check) eq -1) then begin
         if (skip le 0) then begin
            pjl,unit,'day not  processed yet: ',dates(i)
            t=systime(1)
            if not keyword_set(look_only) then begin
               out=cep_survey(RF=newfiles(i),PS=path+'/PO_CEP_RAL_SURVEY_' + dates(i) + '.ps')
               pjl,unit,'time for ',dates(i),' : ', $
                              (systime(1)-t)/60,' Minutes'
            endif 
         endif else skip=skip-1
      endif else begin
         pjl,unit,'day already processed: ',dates(i)
         if keyword_set(NOCHECK) then begin
            pjl,unit,'reprocessing day: ',dates(i)
            out=cep_survey(RF=newfiles(i),PS='/PO_CEP_RAL_SURVEY_' + dates(i) + '.ps')
            pjl,unit,'time for ',dates(i),' : ',(systime(1)-t)/60,' Minutes'
         endif
      endelse
   endfor

   close,unit
   free_lun,unit

end


