pro cep_mask, date, mask, error, out=out, PNG=PNG
;+
; NAME:
;       MASK
;
; PURPOSE:
;       Create a realistic Sun and Earth mask for the IPS instrument
;       on the POLAR spacecraft.  The instrument is contaminated by
;       light, so this code is written to find out which sectors and
;       look directions are affected by this light contamination.
;       Note this program works for SINGLE days only.  For multiple
;       days,  repeat the call to this program.
;
; CATEGORY:
;       Data reduction/data processing.
;
; CALLING SEQUENCE:
;       MASK, date[, data_loc, out, PNG]
; 
; INPUTS:
;       date = a date string type yyymmdd 
;
; OPTIONAL INPUT PARAMETERS:
;        none
;
; KEYWORDS:
;        out:      the name (including location) for the output file.
;                  If missing, create one based on the start time.
;        PNG:      Specifies that the binary mask array will be stored
;                  in a PNG file.  If this is not set, the whole day 
;                  will be output into a sparse array and saved that way.
;   
; OUTPUTS:
;         mask:    the mask and time in a structure, mask is binary as
;                  a function of sector and look direction.  This is
;                  over one minute, assuming 32 sectors.
;
; OPTIONAL OUTPUT PARAMETERS:
;         none
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRITCTIONS:
;         none
;
; PROCEDURE:
;         Read in the spacecraft ephemeris.  Call the routine for
;         locating the the Sun.  Calculate from that the location of
;         the sun and earth in sector, detector space
;
; CREATED:
;         Karen Lewis Hirsch
;         on leave at Los Alamos National Laboratories
;         August 30 - September 11, 1998
;         Fixed and Modified, Reiner Friedel, Aug 2002
;-

print,''
message,'making mask file',/cont

; check for env. variable pointing to data directories
if not (papco_check_data_env('PO_CEP_MASK_DIR', PATH = m_path)) then return

error = 1
date = strcompress(string(date(0)),/remove_all)
yyyy = strmid(date, 0, 4)+'/'
date_yymmdd = strmid(date, 2, 6)

;create file names for finding the appropriate ephemeris and attitude files. 

;------find file names here ---------------------------------------------------
if not (papco_check_data_env('PO_CEP_OR_DEF_DIR', PATH = path)) then return
dum = file_search(path+yyyy+'po_or_def_'+date+'*.cdf',count=c)
IF c NE 0 THEN GOTO,  ephem_found

if not (papco_check_data_env('PO_CEP_OR_PRE_DIR', PATH = path)) then return
dum = file_search(path+yyyy+'po_or_pre_'+date+'*.cdf',count=c)
IF c NE 0 THEN GOTO,  ephem_found

message, 'ephemeris file is not found',/cont 
RETURN

ephem_found:

ephem_name = dum(0)
message, ephem_name, /cont

if not (papco_check_data_env('PO_CEP_AT_DEF_DIR', PATH = path)) then return
dum = file_search(path+yyyy+'po_at_def_'+date+'*.cdf',count=c)
IF c NE 0 THEN GOTO,  at_found

if not (papco_check_data_env('PO_CEP_AT_PRE_DIR', PATH = path)) then return
dum = file_search(path+yyyy+'po_at_pre_'+date+'*.cdf',count=c)
IF c NE 0 THEN GOTO,  at_found

message, 'attitude file is not found',/cont 
RETURN

at_found:

att_name = dum(0)
message, att_name, /cont

;------------------------------------------------------------------------------
;open the attitude and read the data:
cdfid=cdf_open(att_name)
IF cdfid LT 0 THEN RETURN
cdf_control, cdfid, VARIABLE='Epoch', GET_VAR_INFO = rec
maxrec = rec.maxrec
cdf_varget,cdfid,'Epoch',epoch_a,rec_count=maxrec
epoch_a = reform(epoch_a(0,*))
cdf_varget,cdfid,'BODY_SPIN_RATE',spin,rec_count=maxrec
spin = reform(spin(0,*))        ;in seconds
cdf_varget,cdfid,'GCI_R_ASCENSION',RA,rec_count=maxrec
RA   = reform(RA(0,*))          ;in radians (?!)
cdf_varget,cdfid,'GCI_DECLINATION',Dec,rec_count=maxrec
Dec  = reform(Dec(0,*))         ;in radians
cdf_close, cdfid

;now the same for the ephemeris:
cdf_id=cdf_open(ephem_name)
IF cdf_id LT 0 THEN RETURN
cdf_control, cdfid, VARIABLE='Epoch', GET_VAR_INFO = rec
maxrec = rec.maxrec
cdf_varget,cdfid,'Epoch',epoch,rec_count=maxrec
epoch = reform(epoch(0,*))
cdf_varget,cdfid,'GCI_POS',gci,rec_count=maxrec
gci = gci/6371. ;from km to R_e
cdf_close,cdfid

Radius = SQRT(GCI(0,*)^2+GCI(1,*)^2+GCI(2,*)^2)
;get attitude and ephemeris on same epoch
RA = interpol(RA, epoch_a, epoch)
Dec= interpol(Dec, epoch_a, epoch)

;------------------------------------------------------------------------------
ndat = n_elements(epoch)

;make data structure
MASK    = {TAI:dblarr(ndat),UT:fltarr(ndat),blank:bytarr(9,32,ndat)}

yr=fix(strmid(date,0,4)) 
mon=fix(strmid(date,4,2))
day=fix(strmid(date,6,2))

;convert time to TAI
message, 'converting time to TAI, UTe', /cont
UTe = fltarr(ndat)
ET = fltarr(ndat)

FOR i = 0, ndat-1 DO BEGIN
    cdf_epoch,epoch(i),yr,mon,dy,hr,mn,sc,milli,/break
    UTe(i) = hr+mn/60.+sc/3600.+milli/3.600E6
    MASK.TAI(i) = UTC2TAI({YEAR:yr, MONTH:mon, DAY:day, HOUR:hr, MINUTE:mn, $
                           SECOND:sc, MILLISECOND:milli})
ENDFOR

;T =(julday(yr, mon, day, fix(UTe), fix((UTe-fix(UTe))*60), $
;     fix((UTe-fix((UTe-fix(UTe))*60))*60)) -2451545) / 36525.
;ET = UTe + 0.017261111 +  0.0037055556*T - 0.10769444*T^2 - $
;     0.21761667*T^3 - 0.12486111*T^4        ;ephemeris time is used in sun.pro

;ignore ephemeris time correction for now!
et = ute & ndat = n_elements(ET)
MASK.UT = UTe

;use the program sun.pro from the JHU astronomical programs to find
;the right ascension, declination, and distance from Earth of the sun.
DIST = 1 & TRUE_RA = 1 & TRUE_DEC = 1
dis = fltarr(ndat) & tru_r = fltarr(ndat) & tru_de = fltarr(ndat)
FOR i = 0, N_ELEMENTS(ET)-1 DO BEGIN
    jhu_sun, yr, mon, day, ET(i), $
             DIST=dist, TRUE_RA=true_ra, TRUE_DEC=true_dec
    dis(i) = dist & tru_r(i) = true_ra & tru_de(i)= true_dec
ENDFOR
dist = dis & true_ra = tru_r & true_dec = tru_de
DIST = DIST*23988.3 ;convert to Re from AU

;we now have both the sun and the Polar spacecraft vectors (in RA,
;Dec, and distance) from Earth.  We need to convert them to something 
;useful for vector manipulation - 
;say GEI, and then get the Earth and the Sun vectors from POLAR.

SUN   = fltarr(2,n_elements(et)) ;look direction, sector
EARTH = fltarr(2,n_elements(et))

FOR i = 0, n_elements(et)-1 DO BEGIN

    GEI_sun = cv_coord(from_sphere = $
              [true_ra(i)*360./24.,true_dec(i),dist(i)], to_rect=1, degrees=1)
    GEI_pol = [GCI(0,i),GCI(1,i),GCI(2,i)]
    
    ;now sun vector - polar vector = polar-sun vector
    polar_to_sun  = GEI_sun - GEI_pol
    polar_to_sun  = polar_to_sun/NORM(polar_to_sun)
    polar_to_earth= -1.*GEI_pol/NORM(GEI_POL)
    
    ;now convert to a Polar Spacecraft frame: 
    ;Use the one that Mike Henderson does with Z parallel to the 
    ;spin axis and X such that the Sun Vector lies in the
    ;X-Z plane and Y completing the triad. 
    ;Take the RA and Dec from the attitude file and convert them to cartesian
    ;coordinates -> yielding the spin axis vector in GEI

    POLAR_spin_axis=cv_coord(from_sphere=[ra(i),dec(i),1],/to_rect)

    ;we need only a unit vector for POLAR_spin_axis
    POLAR_spin_axis = POLAR_spin_axis/norm(POLAR_spin_axis)

    ;Polar_spin_axis x Sun Vector -> Polar_y_axis
    POLAR_y_axis = crossp(POLAR_spin_axis, polar_to_sun)
    POLAR_y_axis = POLAR_y_axis / norm(POLAR_y_axis) 

    ;Polar_y_axis x Polar_spin_axis -> Polar_x_axis
    POLAR_x_axis = crossp(POLAR_y_axis, POLAR_spin_axis)
    POLAR_x_axis = POLAR_x_axis / norm(POLAR_x_axis)

    ips_offst = 5.0098*!dtor; displ. angle for the IPS on the spacecraft (deg)

    GEI_to_POLAR = fltarr(3,3)
    GEI_to_POLAR(*, 0) = POLAR_x_axis
    GEI_to_POLAR(*, 1) = POLAR_y_axis
    GEI_to_POLAR(*, 2) = POLAR_spin_axis
    
    sun_in_pol  = GEI_to_POLAR##polar_to_sun       ;sun vector in s/c coords.
    earth_in_pol= GEI_to_POLAR##polar_to_earth     ;earth vector in s/c coords.

    ;convert to sun, earth coords in fractional IPS detector, sector space
    ;need to add in IPS sector displacement
    SUN(0,i)   = acos(sun_in_pol(2))*9./!pi        ;9 detectors over 180 deg 
    SUN(1,i)   = atan(sun_in_pol(1),sun_in_pol(0))+ips_offst      
    while SUN(1,i) lt 0 do SUN(1,i)=SUN(1,i)+2*!pi
    SUN(1,i)   = SUN(1,i)*32./(2*!pi)              ;32 sectors over 360 deg

    EARTH(0,i)   = acos(EARTH_in_pol(2))*9./!pi    ;9 detectors over 180 deg
    EARTH(1,i)   = atan(EARTH_in_pol(1),EARTH_in_pol(0))+ips_offst
    while EARTH(1,i) lt 0 do EARTH(1,i)=EARTH(1,i)+2*!pi
    EARTH(1,i)   = EARTH(1,i)*32./(2*!pi)          ;32 sectors over 360 deg

ENDFOR

sun_sect_width = 8.0         ;from Reiner Friedel, modified
earth_width_fac =  3      ;from Reiner Friedel, modified

FOR i = 0, n_elements(et)-1 DO BEGIN
    ;a whole head is affected by sunlight
    det  = fix(sun(0,i)/3)
    det  = [det*3, det*3+2]
    sect = [fix(sun(1,i)-.5*sun_sect_width+32)-32, $
            fix(sun(1,i)+.5*sun_sect_width+32)-31]
   
    WHILE sect(1) GT 31 DO sect(1)=sect(1) mod 32
    WHILE sect(0) LT 0 DO sect(0) = sect(0) + 32
    IF sect(1) LT sect(0) THEN BEGIN 
        mask.blank(det(0):det(1),sect(0):31,i)=1
        mask.blank(det(0):det(1),0:sect(1),i)=1
    ENDIF ELSE $
        mask.blank(det(0):det(1),sect(0):sect(1),i) = 1 ;sun blanking

    alph = asin(1/radius(i))  ;Earth is subtended over this half-angle
    WHILE alph LT 0 DO alph = alph+2*!pi
    alph = alph*32/(2*!pi)    ;converting to the sectors subtended from 0-alph
    alph=alph*earth_width_fac ;R. Friedel, modified
    det  = fix(earth(0,i)/3)
    ;find where the grid lies if it is a radius
    det  = [det*3, det*3+2] ;again remove the whole head
    sect = [fix(earth(1,i)-alph), fix(earth(1,i)+alph)+1]
    WHILE sect(1) GT 31 DO sect(1)=sect(1) mod 32
    WHILE sect(0) LT 0 DO sect(0) = sect(0) + 32
    IF sect(1) LT sect(0) THEN BEGIN 
        mask.blank(det(0):det(1),sect(0):31,i)=1
        mask.blank(det(0):det(1),0:sect(1),i)=1
    ENDIF ELSE $
        mask.blank(det(0):det(1),sect(0):sect(1),i) = 1 ;earth blanking

ENDFOR

;reverse detector order in mask.blank
idx = 8-bindgen(9) & mask.blank = mask.blank(idx, *, *)
  
IF KEYWORD_SET(OUT) THEN filename=out $
ELSE filename=m_path+'ips_mask_'+date+'.idl'
  
IF KEYWORD_SET(PNG) then begin
   dum = strpos(filename,'.')
   filename=strmid(filename,0,dum)+'.png'
   IF KEYWORD_SET(PNG) THEN BEGIN
       FOR d = 0, 8 DO BEGIN
           r = [255,0] & g = [255,0] & b = [255,0]
           TVLCT,r,g,b
           im=transpose(mask.blank(d,*,*)) 
           dum = strpos(filename,'.')
           fname = strmid(filename,0,dum)+'_'+strtrim(string(20*d+10),2)+$
             'deg'+strmid(filename,dum,4)
           print,'writing png : ',fname
           write_png,fname,im,r,g,b
       ENDFOR
   ENDIF
ENDIF ELSE SAVE, mask, filename=filename, /compress, /verb

END
