;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_survey
;
; Purpose     :	Converts CEPPAD level 0 data to RAL intermediate format for survey plots.
;               
; Explanation : IES data with IPS plot for comparison.
;               Prompts for CEPPAD level 0 file names.
;               Converts each level 0 file to level 1 files.
;               Converts the level 1 files to the RAL intermediate format needed for the 
;               survey plots.
;               Sets default file name for plots.
;
; Use         : < if_files = cep_survey ( /L1, /RF, /WINDOWS, /PS, /GIF, /BWPS, $
;                                         T0=start_time, T1=end_time ) >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : if_files : STRARR containing names of IES intermediate format files used.
;               CEPPAD level 1 and RAL intermediate format files.
;
; Opt. Outputs:	None.
;
; Keywords    : /L1         = Specifies that the l1 files already exist. 
;               /RF         = Specifies that the if files already exist. 
;                                if STRING or STRARR then assumes file names   
;               /WINDOWS    = plots output in x window.
;               /PS         = plots output as postscript file    
;                                if string then assumes it is a file name   
;               /GIF        = plots output as GIF file    
;                                if string then assumes it is a file name   
;               /BWPS       = plots output as black and white postscript file                            
;                                if string then assumes it is a file name   
;               /NOCT       = does not load default color table
;               T0          = See cep_l1toif.
;               T1          = See cep_l1toif.
;
; Calls       :	cep_lztol1, cep_l1toif, cep_get_l1day, strip_dirname, concat_dir, cep_get_lz_files, datatype, cep_exttime.
;                
; Common      :	ies_draw_widget_block.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 18/9/96
;
; Modified    :	Version 0.1, MKC, 7/11/96
;                 Corrected bug in processing rf keyword.
;                 Added start and end time keywords.
;               Version 0.2, 26/11/96
;                 Changed IPS colour scale.
;
; Version     :	Version 0.2, 26/11/96
;-
;**********************************************************

FUNCTION cep_survey, L1=l1, RF=rf, WINDOWS=windows, PS=postscript, GIF=gif, BWPS=bwps, NOCT=noct, T0=start_time, T1=end_time

   ; define common blocks used internally for draw widgets
   ; full_header   = structure containing original data header information
   ; full_data     = structure array containing original data
   ; header        = structure containing extracted data header information  ** used
   ; data          = structure array containing extracted data               ** used
   ; output_info   = structure containing plotting and processing info       ** used 
   ; output_chans  = array containing processed channel positions           
   ; output_array  = array containing processed data for plotting       
   ; ids           = structure giving widget IDs 
   ; dnfull        = co-ordinate conversion array for full window
   ; dnscroll      = co-ordinate conversion array for magnified windows
   ; files         = STRARR giving data file names
   ; ephem         = structure containing ephemeris data

  COMMON ies_draw_widget_block, full_header, full_data, input_header, input_data, output_info, output_chans, output_array, $
                                ids, dnfull, dnscroll, files, ephem

  ; load RAINBOW colour table

  IF NOT KEYWORD_SET(noct) THEN LOADCT, 13

  ; get lz files

  IF NOT KEYWORD_SET(l1) AND NOT KEYWORD_SET(rf) THEN BEGIN

    ; do level 0 to level 1 conversion

    lzfiles = cep_lztol1()  

    IF lzfiles(0) EQ '' THEN RETURN, ''

  ENDIF ELSE IF KEYWORD_SET(rf) THEN BEGIN

    IF datatype(rf) EQ 'STR' THEN lzfiles = STRARR(N_ELEMENTS(rf)) ELSE lzfiles = ''

  ENDIF ELSE BEGIN

    lzfiles = ''

  ENDELSE

  ; set flag indicating whether standard survey plot

  standard = 1

  ; set up variables for data

  input_header = 0
  input_data   = 0

  ; set up output STRARR, ensure at least one file if no lz file 

  if_files = STRARR(N_ELEMENTS(lzfiles))

  ; loop through level 0 files

  FOR f = 0, N_ELEMENTS(lzfiles)-1 DO BEGIN

    ; set up file name variables

    ies_file = 0
    ips_file = 0

    IF KEYWORD_SET(rf) THEN BEGIN

      ; claim that RAL intermediate format files already exist, get file names

      IF datatype(rf) EQ 'STR' THEN ies_file = rf(f) ELSE $
        ies_file = PICKFILE ( PATH=GETENV('IES_DATA_DIR'), FILTER='*CLEAN_SURVEY_IES*.DAT', $
                                          /READ, /MUST_EXIST, TITLE='Select IES intermediate format data file' ) 

      IF STRMID(strip_dirname(ies_file),0,28) EQ 'PO_CEP_RAL_CLEAN_SURVEY_IES_' THEN BEGIN

        ips_file = concat_dir ( GETENV('IES_DATA_DIR'),'PO_CEP_RAL_SURVEY_IPS_' + STRMID(strip_dirname(ies_file),28,20) )

      ENDIF ELSE BEGIN

        standard = 0

        ips_file = PICKFILE ( PATH=GETENV('IES_DATA_DIR'), FILTER='*_IPS_*.DAT', $
                              /READ, /MUST_EXIST, TITLE='Select IPS intermediate format data file' )
      ENDELSE

    ENDIF ELSE IF KEYWORD_SET(l1) THEN BEGIN

      ; claim that l1 format files already exist, get file names

      ies_file = PICKFILE ( PATH=GETENV('PO_CEP_L1_DIR'), FILTER='PO_CEP_L1_IES_*.DAT', $
                            /READ, /MUST_EXIST, TITLE='Select IES level 1 format data file' )

      ips_file = concat_dir ( GETENV('PO_CEP_L1_DIR') ,'PO_CEP_L1_IPS_' + STRMID(strip_dirname(ies_file),14,20) )

      IF ies_file EQ '' OR ips_file EQ '' THEN MESSAGE, 'INVALID LEVEL 1 FORMAT FILE NAME', /TRACEBACK

      ; do level 1 to RAL intermediate format conversion on existing l1 files
      ; prompt for file names

      cep_l1toif, ies_file, MODE='CLEAN_SURVEY', FILE=ies_file

      cep_l1toif, ips_file, MODE='SURVEY', FILE=ips_file       

    ENDIF ELSE BEGIN

      ; do level 1 to RAL intermediate format conversion 

      cep_l1toif, 'IES_' + cep_get_l1day ( strip_dirname(lzfiles(f)) ), MODE='CLEAN_SURVEY', FILE=ies_file

      cep_l1toif, 'IPS_' + cep_get_l1day ( strip_dirname(lzfiles(f)) ), MODE='SURVEY', FILE=ips_file

    ENDELSE

    IF ies_file EQ '' OR ips_file EQ '' THEN MESSAGE, 'INVALID INTERMEDIATE FORMAT FILE NAME', /TRACEBACK

    ; add to output

    if_files(f) = ies_file

    ; get IES data for survey plot

    RESTORE, ies_file

    ; extract requested time range

    cep_exttime, input_header, input_data, T0=start_time, T1=end_time

    ; get ephemeris data for midpoint of range

    ephem = get_ephem ( (input_header.tstart+input_header.tstop)/2.0 )

    ; set file name for output

    IF standard THEN BEGIN

      temp = STRMID(strip_dirname(ies_file),28,20) 

      date = STRMID(temp,0,STRPOS(temp,'.DAT')) 

      IF KEYWORD_SET(postscript) THEN BEGIN   
        IF datatype(postscript) NE 'STR' THEN new_postscript ='SURVEY_' + date + '.ps' ELSE new_postscript = postscript
      ENDIF ELSE IF KEYWORD_SET(bwps) THEN BEGIN   
        IF datatype(bwps) NE 'STR' THEN new_bwps ='SURVEY_' + date + '.ps' ELSE new_bwps = bwps
      ENDIF ELSE IF KEYWORD_SET(gif) THEN BEGIN   
        IF datatype(gif) NE 'STR' THEN new_gif ='SURVEY_' + date + '.gif' ELSE new_gif = gif
      ENDIF
            
    ENDIF 
        
    ; do spectral plots

    ; set up processing info

    output_info = { minimum:1., maximum:1.e6, log:1, cal:0, div:0, rat:0, clean:0, subtract:0, track:1, $
                    button_labels:['BINS','SHIFT','INTERP',''], button_values:[1,0,0,0], procname:'SPECTRAL', $
                    full_detector:0, full_sector:0, band_labels:STRARR(20), noutputs:0, $
                    format:'ENERGY vs DETECTOR', ptitle:STRARR(20), time:0.0D0, ysamples:0 }

    ; process data

    IF ies_calibrate() THEN RETURN, ''

    ; fiddle things for plot

    output_info.noutputs = 0

    ; kill integral channel if present

    IF input_header.nbands EQ 16 THEN output_info.ysamples = 15

    ; change detector labels to fit

    output_info.ptitle = 'IES ' + input_header.ptitle

    ; set default plot type

    IF NOT KEYWORD_SET(windows) AND NOT KEYWORD_SET(postscript) AND $
       NOT KEYWORD_SET(gif) AND NOT KEYWORD_SET(bwps) THEN windows=1 

    ; save current device

    devname = !D.NAME

    ; plot data

    IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, GIF=new_gif, BWPS=new_bwps, $
                        /NO_XAXIS, /NO_END, SHRINK=standard, $
                        HEADING = 'POLAR/CEPPAD SURVEY PLOT for ') THEN MESSAGE, 'ies_window_plot error', /TRACEBACK

    ; do detector 4

    output_info.full_detector = 3

    IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, GIF=new_gif, BWPS=new_bwps, $
                        PG_POS=1, /NO_XAXIS, /NO_END, $
                        SHRINK=standard, HEADING = 'POLAR/CEPPAD SURVEY PLOT for ') THEN $
                            MESSAGE, 'ies_window_plot error', /TRACEBACK

    ; do detector 7

    output_info.full_detector = 6

    IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, GIF=new_gif, BWPS=new_bwps, $
                        PG_POS=2, /NO_XAXIS, /NO_END, $
                        SHRINK=standard, HEADING = 'POLAR/CEPPAD SURVEY PLOT for ') THEN $
                            MESSAGE, 'ies_window_plot error', /TRACEBACK

    ; do sector plot

    ; set up processing info

    output_info = { minimum:1., maximum:1.e6, log:1, cal:0, div:0, rat:0, clean:0, subtract:0, track:1, $
                    button_labels:['BINS','SHIFT','INTERP',''], button_values:[1,0,1,0], procname:'SPECTRAL', $
                    full_detector:3, full_sector:0, band_labels:STRARR(20), noutputs:0, $
                    format:'SECTOR vs DETECTOR', ptitle:STRARR(20), time:0.0D0, ysamples:0 }

    ; sum energy channels and put results in top channel
    ; remove integral channel if present

    IF input_header.nbands EQ 16 THEN $
      input_data.data(*,*,input_header.nbands-1) = TOTAL ( input_data.data(*,*,5:14), 3 ) $
    ELSE $
      input_data.data(*,*,input_header.nbands-1) = TOTAL ( input_data.data(*,*,2:*), 3 ) 

    ; process data

    IF ies_calibrate() THEN RETURN, ''

    ; fiddle things for plot

    output_info.noutputs = 0

    ; change detector labels to fit

    output_info.ptitle = 'IES ' + input_header.ptitle

    ; plot data

    IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, GIF=new_gif, BWPS=new_bwps, $
                        PG_POS=3, /NO_XAXIS, /NO_END, $
                        SHRINK=standard, HEADING = 'POLAR/CEPPAD SURVEY PLOT for ') THEN $
                            MESSAGE, 'ies_window_plot error', /TRACEBACK

    ; get IPS data for survey plot

    RESTORE, ips_file

    ; extract requested time range

    cep_exttime, input_header, input_data, T0=start_time, T1=end_time

    ; set up processing info

    output_info = { minimum:1.E-2, maximum:1.E4, log:1, cal:0, div:0, rat:1, clean:0, subtract:0, track:1, $
                    button_labels:['BINS','SHIFT','INTERP',''], button_values:[1,0,0,0], procname:'SPECTRAL', $
                    full_detector:1, full_sector:0, band_labels:STRARR(20), noutputs:0, $
                    format:'ENERGY vs DETECTOR', ptitle:STRARR(20), time:0.0D0, ysamples:0 }

    ; process data

    IF ies_calibrate() THEN RETURN, ''

    ; fiddle things

    output_info.noutputs = 0
    output_info.ysamples = 16
    output_info.ptitle   = 'IPS ' + input_header.ptitle
    input_header.xtitle  = 'UT:'

    ; plot data

    IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, GIF=new_gif, BWPS=new_bwps, $
                        PG_POS=4, SHRINK=standard, $
                        HEADING = 'POLAR/CEPPAD SURVEY PLOT for ') THEN $
                            MESSAGE, 'ies_window_plot error', /TRACEBACK

  ENDFOR

  ; deallocate data variables

  input_header = 0
  input_data   = 0
  output_array = 0

  ; return IEF file names

  RETURN, if_files

END
