;       PROCEDURES CALLED:                  PURPOSE

;       EST_REC_NUMBER               Search for the CEPPAD LEVEL-1 normal 
;                                    mode logical record (a logical record
;                                    equals 2 physical records) that has 
;                                    the TIME for which you want to
;                                    extract data

;       ESTIMATE_L1_TIMES            Fill in missing times of a normal
;                                    mode LEVEL-1 logical record using 
;                                    interpolation/extrapolation (a record 
;                                    has a max of 16 times )

;	EXTRACT_NOSUB_HEALTH	     This procedure fills the NOSUB_CEP_HEALTH 
;				     structure with the parameters that 
;				     describe the health of the CEPPAD 
;				     instrument. These parameters are not 
;				     based on a subcom value and are available
;				     every frame. 

;	EXTRACT_CEP_HEALTH	     This procedure fills the CEP_HEALTH 
;				     structure with the parameters that 
;				     describe the health of the CEPPAD 
;				     instrument. These parameters are extracted
;				     based on a subcom parameter.



;       FILL_CEP_COMNS               Control the filling of defined
;                                    common variables for the
;                                    analysis of CEPPAD data

;       FILL_NORM_COMN               Fill the normal mode common variable 
;                                    data arrays

;       FILL_PLCP_ORG                Fill an array that describes the 
;                                    contents/organization of the data
;                                    stored in POLAR (PL) CEPPAD (CP)
;                                    normal mode common variable data arrays


;       FUNCTION FLOATING_PT        Converts CEPPAD's key parameter 16-bit 
;				    integers to floating point values.


;       GET_CEP_HK_DATA		    Extracts data from the LEVEL-1 
;				    HOUSEKEEPING file and fill defined
;				    common blocks


;       GET_HR_DEFS                  Get the Header Reader (HR) definition 
;                                    parameters from the HR and puts those 
;                                    definitions in common blocks 
;                                    FILDEF and TMADEF


;       GET_NEW_TIM_RNG              Set 'BEG_TIME' to the first time in the
;                                    logical record in which 'TIME1' was found. 
;                                    Set 'END_TIME' to the last time of the  
;                                    logical record in which 'TIME2' was found.
;                                     ( 'TIME1' and 'TIME2' are user inputs)

;       GET_SH_INFO                  Extract Spin Header (SH) information 
;                                    from the SH file

;       GET_TIMES                    Extract the times in a CEPPAD normal 
;                                    mode LEVEL-1 logical record

;       GET_TM_ACQ_TBL               Read the TeleMetry/accumulation table
;                                    values from CEPPAD LEVEL-1 HR

;       L1_COMN_MAPPING              Calculate the locations for sector 0
;                                    for the first pixel of each 
;                                    spin/energy/channel in the normal mode 
;                                    common data array

;       L1_DATA_MAPPING              Calculate locations that map to sector 0 
;                                    of a spin/energy/channel in a LEVEL-1 
;                                    logical record 

;       L1_TIME_MAPPING              Calculate the location of the 
;                                    16 longword time values in a logical 
;                                    record of a CEPPAD LEVEL-1 normal mode 

;       OPEN_CEP_LEVEL1              Open a LEVEL-1 file 

;       RATE_TNEW_RNG                Resets the user inputs for beginning and 
;                                    ending time using the first time in the 
;                                    record containing the user specified
;                                    beginning time and the last time in the
;                                    record containing the user specified
;                                    ending time

;       RATES_COMN_FILL              Transfers data from the rates level-1 file
;                                    to the Rates common block array(s)

;       READ_CEP_HK_DATA	     Reads the Housekeeping ( HK )   
;				     parameters from the LEVEL-1 HK file and 
;				     fills the HK_data STRUCTURE with all the 
;				     data for the time span requested.


;       REPLACE_NETWORK_LONGS_WITH_HOST Replaces 4-byte (big-endian/network) 
;                                    elements of specified byte array with 
;                                    'HOST' formatted 4-byte elements using
;                                    'BYTEORDER'

;       REPLACE_NETWORK_SHORTS_WITH_HOST Replaces 2-byte (big-endian/network) 
;                                    elements of specified byte array with 
;                                    'HOST' formatted 2-byte elements using
;                                    'BYTEORDER'

;       RD_CEP_LEV1_HR               Read the HR in LEVEL-1 file 

;       SETUP_COMMONS                Set up the defined common variables 
;                                    and constants ( arrays are dimensioned )

;	UNPACK_SUBCOM_DATA           Unpacks the data that are subcomed 
;				     fills the following structures:
;				     HIST_S, HIST_E, Table_CRC, IPSCAL,
;				     Table_address, DPU_MEMORY_DMP

;__________________________________________________________________________
;__________________________________________________________________________

;    The defined COMMON variables which are used to process CEPPAD LEVEL-1 
;    data on the Remote Data Analysis Facility (RDAF) computers.
;
;  COMMON BLOCK NAME                   DESCRIPTION OF VARIABLES
;
; COMMON POL_EPHP,       eph
; COMMON POL_MAG,        mag
; COMMON POL_ATT,        att
; COMMON POL_SC,         plscmagaz, plscmagel


; COMMON POL_CEP       PLCPNum_tables -- Longword defining the number of 
;                                        T/M tables encountered in reading 
;                                        the LEVEL-1 file corresponding to 
;			                 data stored in the PLCPIPS_bytes, 
;                                        PLCPIES_bytes and PLCPHST_bytes arrays
;                      PLCPMax_tables -- The maximum number of tables allowed 
;                                        to be accounted for in the requested
;                                        time range ( This version PLCPM

;  POL_CEP_SH          PLCPSpn_hed	Byte array containing all CEPPAD 
;                                       Spin Header ('SH') values stored in
;			                the Spin header Level-1 file for the
;			                requested time range. ( Dimensioned
;                                       (13,plcpspn_hed_num) 
;			Bytes	        Description
;		       0-3	Longword - Start of spin time in msec
;		       4	Byte - T/M table Number | T/M Acquisition Mode
;		       5	Byte - IPS Lookup Table Number
;		       6	Byte - IES Lookup Table Number
;		       7	Byte - IES Command byte
;		       8	Byte - HIST Lookup Table Number
;		       9-10	Integer - HIST Operating State
;		       11	Byte - IPS Mux setting
;                      12       Byte - SCOM - CEPPAD Spin #
;	               PLCPSpn_hed_num  Longword number of data records in 
;                                       the Spin header Level-1 file for the
;                                       requested time range


;  POL_CEP_IPS         PLCPIPS_Time -- Longword array containing the time of 
;                                      day in milliseconds at the start of 
;                                      acquisition of each spin containing
;                                      LEVEL-1 IPS data for the user requested
;                                      time range, 'TIME1 - TIME2'
;	               PLCPIPS_bytes -- Byte array (single dimensioned) 
;                                      containing the de-compressed count 
;                                      values accumulated through each 'sector'
;                                      or 'pixel'/ channel/energy-range. Data 
;                                      are extracted from the LEVEL-1
;			               normal mode file for the users specified
;                                      time range,'TIME1 - TIME2'.
;	               PLCPIPS_Org --  Longword array items defining the
;			               contents/organization of the
;                                      PLCPIPS_bytes array.  It is dimensioned
;                                      to allow for up to Max_tables (T/M
;			               tables encountered over the data range 
;                                      to be plotted.) It contains values for 4
;                                      parameters for each channel and energy.
;     		       PLCPIPS_Org(*, *, *, *)	
;		                   |  |  |  |
;                                  |  |  |  |__T/M table (Max of 1 this version)
;                                  |  |  |_ Energy ( Depends on table)
;		                   |  |__ Detector channel (9 channels)
;       		           |__ Organization item  (4 parameters )
;                                  (The four parameters are described below)
;                      PlCPIPS_Org(0, *, *, * ) -- Number of sectors per spin
;                                      as defined in the T/M accumulation table
;                      PLCPIPS_Org(1, *, *, * ) -- Number of spins per sample 
;                                      as defined in the T/M accumulation table
;                      PLCPIPS_Org(2, *, *, * ) -- Number of values stored
;                                      ( will be a function of the number of 
;                                      spins of data to which the current table
;                                      is applied  AND the data accumulation 
;                                      rate ( the number of spins over which
;                                      the data accumulated )
;                      PLCPIPS_Org(3, *, *, * ) -- Index location of first value
;                                      in 'PLCPIPS_Bytes' array ( the value from
;                                      the 1st sector and first spin of data to
;                                      which the T/M accumulation table applies
;                      PLCPIPS_Start_Time -- Longword scalar containing the 
;                                      last requested start time for IPS data 
;                                      - the time range of data actually stored 
;                                      in common resulting from an execution of 
;                                      'GET_DATA' may be a subset of the 
;                                      requested time-range
;                      PLCPIPS_Stop_Time -- Longword scalar containing the last
;                                      requested stop time for IPS data 



;  POL_CEP_IES         PLCPIES_Time -- Longword array containing the time of 
;                                      day in milliseconds at the start of 
;                                      acquisition of each spin containing
;                                      LEVEL-1 IPS data for the user requested
;                                      time range, 'TIME1 - TIME2'
;	              PLCPIES_bytes -- Byte array containing the 
;                                      de-compressed count values 
;                                      accumulated through each 'sector' or
;                                      'pixel' channel/energy-range. Data 
;                                      are extracted from the LEVEL-1
;			               normal mode file for the users specified
;                                      time range,'TIME1 - TIME2'.
;     		       PLCPIES_Org(*, *, *, *)	
;		                   |  |  |  |
;                                  |  |  |  |__T/M table (Max of 1 this version)
;                                  |  |  |_ Energy  ( Depends on table )
;		                   |  |__ Detector channel  ( 9 channels )
;       		           |__ Organization item  ( 4 parameters )
;                      PLCPIES_Start_Time -- Longword scalar containing the 
;                                      last requested start time for IES data 
;                                      - the time range of data actually stored 
;                                      in common resulting from an execution of 
;                                      'GET_DATA' may be a subset of the 
;                                      requested time-range
;                      PLCPIES_Stop_Time -- Longword scalar containing the last
;                                      requested stop time for IES data 


;  POL_CEP_HSTE       PLCPHSTE_Time -- Longword array containing the time of 
;                                      day in milliseconds at the start of 
;                                      acquisition of each spin containing
;                                      LEVEL-1 IPS data for the user requested
;                                      time range, 'TIME1 - TIME2'
;	              PLCPHSTE_Bytes -- Longword array containing the 
;                                      de-compressed count values 
;                                      accumulated through each 'sector' or
;                                      'pixel' channel/energy-range. Data 
;                                      are extracted from the LEVEL-1
;			               normal mode file for the users specified
;                                      time range,'TIME1 - TIME2'.
;     		       PLCPHSTE_Org(*, *, *, *)	
;		                     |  |  |  |   ( Max of 1 table this version)
;                                    |  |  |  |__T/M table 
;                                    |  |  |_ Energy  ( depends on table)
;		                     |  |__ Detector channel  ( 1 channel)
;       		             |__ Organization item ( 4 parameters)
;                      PLCPHSTE_Start_Time -- Longword scalar containing the 
;                                      last requested start time for HISTE data 
;                                      - the time range of data actually stored 
;                                      in common resulting from an execution of 
;                                      'GET_DATA' may be a subset of the 
;                                      requested time-range
;                      PLCPHSTE_Stop_Time -- Longword scalar containing the 
;                                      last requested stop time for HISTE data 


;  POL_CEP_HSTP      PLCPHSTP_Time -- Longword array containing the time of 
;                                      day in milliseconds at the start of 
;                                      acquisition of each spin containing
;                                      LEVEL-1 IPS data for the user requested
;                                      time range, 'TIME1 - TIME2'
;	              PLCPHSTP_Bytes -- Longword array containing the 
;                                      de-compressed count values 
;                                      accumulated through each 'sector' or
;                                      'pixel' channel/energy-range. Data 
;                                      are extracted from the LEVEL-1
;			               normal mode file for the users specified
;                                      time range,'TIME1 - TIME2'.
;     		       PLCPHSTP_Org(*, *, *, *)	
;		                     |  |  |  |   ( Max of 1 table this version)
;                                    |  |  |  |__T/M table 
;                                    |  |  |_ Energy  ( depends on table)
;		                     |  |__ Detector channel  ( 1 channel)
;       		             |__ Organization item ( 4 parameters)
;                      PLCPHSTP_Start_Time -- Longword scalar containing the 
;                                      last requested start time for HISTP data 
;                                      - the time range of data actually stored 
;                                      in common resulting from an execution of 
;                                      'GET_DATA' may be a subset of the 
;                                      requested time-range
;                      PLCPHSTP_Stop_Time -- Longword scalar containing the 
;                                      last requested stop time for HISTP data 


;  POL_CEP_IPS_RTS     PLCPIPS_Rts --  Longword array containing IPS 
;                                      Rates-histogram values accumulated 
;                                      through all IPS channels (0-9), 16
;			               energy ranges, and an integral channel
;                                      for each spin time in the user 
;                                      specified time range. It is dimensioned
;                                      (PLCPIPS_Rts_num,10,17)
;	               PLCPIPS_Rts_num -- Longword containing the number of 
;                                      times (spins) of IPS Rates-histogram 
;                                      data within the user specified time 
;			               range.
;	               PLCPIPS_Rts_tim -- Longword array containing the time 
;                                       values in msec for each set of IPS 
;                                       Rates-histogram spin data within the 
;			                user specified time range.
;	               PLCPIPS_Rts_mx -- BYTE array containing bit flags 
;                                      indicating the IPS data source 
;			               used for channels 1, 3, 5, & 8 
;                                      corresponding to the sets of data 
;                                      stored in the PLCPIPS_Rts array.
;                      PLCPIPS_Rts_Start_Time -- Longword scalar containing the 
;                                      last requested start time for IPS Rates
;                                      mode data - the time range of data 
;                                      actually stored in common resulting from 
;                                      an execution of 'GET_DATA' may be a 
;                                      subset of the requested time-range
;                      PLCPHIPS_Rts_Stop_Time -- Longword scalar containing the 
;                                      last requested stop time for IPS Rates 
;                                      mode data 

;  POL_CEP_IES_RTS     PLCPIES_Rts --  Longword array containing IES 
;                                      Rates-histogram values accumulated 
;                                      through 256 energies ranges 
;                                      for each spin time in the user 
;                                      specified time range. It is dimensioned
;                                      (PLCPIES_Rts_num, 256)
;	               PLCPIES_Rts_num -- Longword containing the number of 
;                                      times (spins) of IES Rates-histogram 
;                                      data within the user specified time 
;			               range.
;	               PLCPIES_Rts_tim -- Longword array containing the time 
;                                       values in msec for each set of IES 
;                                       Rates-histogram spin data within the 
;			                user specified time range.
;	               PLCPIES_Rts_ch -- BYTE array containing a channel id
;                                      indicating the source IES detectors 
;			               used for corresponding sets of data 
;                                      stored in the PLCPIES_Rts array.
;                      PLCPIES_Rts_Start_Time -- Longword scalar containing the 
;                                      last requested start time for IES Rates
;                                      mode data - the time range of data 
;                                      actually stored in common resulting from 
;                                      an execution of 'GET_DATA' may be a 
;                                      subset of the requested time-range
;                      PLCPHIES_Rts_Stop_Time -- Longword scalar containing the last
;                                      requested stop time for IES Rates mode
;                                      data 


;  POL_CEP_HIST_RTS     PLCPHIST_Rts --  Longword array containing HIST 
;                                      Rates-histogram values accumulated 
;                                      through 256 energy ranges 
;                                      for each spin time in the user 
;                                      specified time range. It is dimensioned
;                                      (PLCPHIST_Rts_num, 256)
;	               PLCPHIST_Rts_num -- Longword containing the number of 
;                                      times (spins) of HIST Rates-histogram 
;                                      data within the user specified time 
;			               range.
;	               PLCPHIST_Rts_tim -- Longword array containing the time 
;                                       values in msec for each set of HIST 
;                                       Rates-histogram spin data within the 
;			                user specified time range.
;	               PLCPHIST_Rts_id -- Longword array containing HIST 
;                                       identification values for 
;                                       corresponding sets of data in the 
;                                       PLCPHIST_Rts array.
;                      PLCPHIST_Rts_Start_Time -- Longword scalar containing 
;                                      the last requested start time for HIST 
;                                      Rates mode data - the time range of data 
;                                      actually stored in common resulting from 
;                                      an execution of 'GET_DATA' may be a 
;                                      subset of the requested time-range
;                      PLCPHIST_Rts_Stop_Time -- Longword scalar containing 
;                                      the last requested stop time for HIST 
;                                      Rates mode data 
;
;
;    The defined COMMON variables which are used to process CEPPAD LEVEL-1 
;    HOUSEKEEPING DATA ( data from telemetry words 11, 12, and 15 ) 
;	 on the Remote Data Analysis Facility (RDAF) computers.
;______________________________________________________________________________
;		DESCRIPTION OF HOUSKEEPING COMMON BLOCK PARAMETERS
 
;  COMMON CEP_KEY_PARAMETERs,     CEP_KP, NUM_KP
;       CEP_KP			 STRUCTURE

;   CEP_KP TAG_NAMES	TYPE				DESCRIPTION

;       TIME		LONG		Time in milliseconds for key parameters
;	AZ		FLOAT		Magnetic azimuth in degrees 
;	MAGEL		FLOAT		Magnetic elevation in degrees
;       MAGBX		FLOAT		Magnetic bx---cos(az) * sin(magel)
;	MAGBY		FLOAT		Magnetic by---sin(az) * sin(magel)
;	MAGBZ		FLOAT		Magnetic bz---cos(magel)
;	IPS_ERR		FLOAT		IPS cubic fit err( sum of sqares )
;	IES_ERR		FLOAT		IES cubic fit err( sum of sqares )
;	IPS_COEF        FLTARR(4 )	IPS cubic fit coefficients:(c0,c1,c2,c3)
;	IES_COEF        FLTARR(4 )	IES cubic fit coefficients:(c0,c1,c2,c3)
;	CUBIC_CNTRLS:   INTEGER		Control bits for cubic fit
;	IPS_SPER	FLTARR( 5 )	IPS spherical harmonic fit coefficients:
;					( c0, c1, c2, c3, c4)
;	IPS_SPER_DRIFT  FLOAT		IPS spherical harmonic fit - DRIFT
;	IPS_SPER_ANGLE	FLOAT		IPS spherical harmonic fit - ANGLE
;	IPS_SPER_RMS	FLOAT		IPS spherical harmonic fit-residual(RMS)
;	IES_SPER	FLTARR( 5 )	IES spherical harmonic fit coefficients:
;					( c0, c1, c2, c3, c4)
;	IES_SPER_DRIFT  FLOAT		IES spherical harmonic fit - DRIFT
;	IES_SPER_ANGLE	FLOAT		IES spherical harmonic fit - ANGLE
;	IES_SPER_RMS	FLOAT		IES spherical harmonic fit-residual(RMS)
;	IPS_SUN_OFFSET: FLOAT		OFFSET(deg) between sun pulse and start
;					of IPS sector zero
;	IES_SUN_OFFSET: FLOAT		OFFSET(deg) between sun pulse and start
;					of IES/HIST sector zero
;
;		NUM_KP	LONG		Number of records in CEP_KP structure
;						( not included in structure )	


;   COMMON CEP_HIST_Singles,         HIST_S, HIST_S_sets_extracted
;   	HIST_S                   STRUCTURE 

;   HIST_S TAG_NAMES	TYPE				DESCRIPTION
;       TIME		LONG		Time in milliseconds for HIST singles
;    	SECTOR:         BYTE		The beginning physical sector number
;       SPIN:           BYTE		The CEPPAD spin counter ( 0 - 255 )
;       DATA:           BYTARR( 16 )	Data for HIST singles ( 16 channels )
;
; HIST_S_SETS_EXTRACTED LONG     Number of sets of HIST single data extracted
;					( not part of the structure )



;   COMMON CEP_HIST_Events,          HIST_E, HIST_E_sets_extracted
;       HIST_E			 STRUCTURE 
;   HIST_E TAG_NAMES	TYPE				DESCRIPTION
;       TIME		LONG		Time in milliseconds for HIST singles
;    	SECTOR:         BYTE		The beginning physical sector number
;       SPIN:           BYTE		The CEPPAD spin counter ( 0 - 255 )
;       DATA:           BYTARR( 16 )	Data for HIST events ( 16 channels )
;
; HIST_E_SETS_EXTRACTED LONG     	Number of sets of HIST event data 
;					extracted( not part of the structure )


;   COMMON CEP_Table_CRCs,        Table_CRC, num_table_CRC

;       Table_CRC,              STRUCTURE
;   Table_CRC TAG_NAMES	TYPE				DESCRIPTION
;	Flt_SW   	INTEGER		 Flight software
;	MST             INTARR( 16)	 Mode Switching Table
;	SECT_A		INTARR( 16)	 Sector & Accumulation
;	IPS_CNF         INTARR( 16)	 IPS Configuration 
;	IESLUT   	INTARR( 12)      IES Lookup Table
;       HISTLUT  	INTARR(  4)	 HIST Lookup Table
;       TIME            LONG		 Time in milliseconds
;
;
;       NUM_TABLE_CRC   LONG		 Number of records in Table_CRC 
;					structure ( not part of the structure )
;

;   COMMON CEP_IPS_calib,         IPSCal, num_IPS_cal

;       IPSCAL	              STRUCTURE
;   IPSCAL TAG_NAMES	TYPE				DESCRIPTION
;
;	TIME		LONG		Time in milliseconds
;	MASKREQ		INTEGER		IPS cal mask requested by command
;	MASKREM		INTEGER		IPS cal mask remaining to execute
;	CURRENTCAL	BYTE            IPS cal state currently executing
;       C1		BYTARR( 9 )     Contents of the Channel C1 positions:
;					0 =  low  D/A for cal D/A = low res
;					1 =  high D/A for cal D/A = low res	
;					2 =  cal  D/A for cal D/A = low res
;					3 =  low  D/A for cal D/A = high res
;					4 =  high D/A for cal D/A = high res
;					5 =  cal  D/A for cal D/A = high res
;					6 =  high D/A for cal = bin 14 top
;					7 =  cal  D/A for cal = bin 14 top
;					8 =  cal  D/A for cal = bin 4 top
;	B1		BYTARR(9 )	Same format as C1 but for B1
;	T1		BYTARR(9 )	Same format as C1 but for T1
;	C2		BYTARR(9 )	Same format as C1 but for C2
; 	C5		BYTARR(9 )	Same format as C1 but for C5
;       C3	 	BYTARR(9 )	Same format as C1 but for C3
;       C0		BYTARR(9 )	Same format as C1 but for C0
;       B5		BYTARR(9 )	Same format as C1 but for B5
;       B3		BYTARR(9 )	Same format as C1 but for B3
;       B2		BYTARR(9 )	Same format as C1 but for B2
;       B0		BYTARR(9 )	Same format as C1 but for B0
;       T5		BYTARR(9 )	Same format as C1 but for T5
;       T3		BYTARR(9 )	Same format as C1 but for T3
;       T2		BYTARR(9 )	Same format as C1 but for T2
;       T0		BYTARR(9 )	Same format as C1 but for T0
;
;       NUM_IPS_CAL     LONG		Number of  records in IPSCAL structure 
;				 	( not part of the structure )
;


;   COMMON CEP_Table_addresses,   Table_address, num_tab_add_recs 
;       Table_address            STRUCTURE
;   Table_address TAG_NAMES		TYPE		DESCRIPTION

;	Time				LONG	 Time in milliseconds
;	SpinBuf0_page			INTEGER	 PAGE No.- SPINbuf0 Base Address
;	SpinBuf0_offset			INTEGER  OFFSET  -      "	"
;	SpinBuf0_len			INTEGER  LENGTH  - SPINBuf             
;	ModeSwitching_page		INTEGER  PAGE N0.- MST0 Base Address
;	ModeSwitching_offset		INTEGER  OFFSET  -	"	"     
;	ModeSwitching_len		INTEGER  LENGTH  - MST Table          
;	ModeSwitch_GenBase_page		INTEGER  PAGE N0.- MST GENBase Address
;       ModeSwitch_GenBase_offset	INTEGER  OFFSET  -	"	"
;       SectAccumt0_page		INTEGER	 PAGE N0.- SectAccumTab Base ADD
;       SectAccumt0_offset		INTEGER  OFFSET  -	"	"
;       SectAccumt_len			INTEGER  LENGTH	 - Sector Accum Table
;       IPS_configuration_tab0_page	INTEGER  PAGE N0.- ICT0 Base Address
;       IPS_configuration_tab0_offset	INTEGER	 OFFSET	 -	"	"  
;       IPS_configuration_tab_len	INTEGER	 LENGTH	 - ICT table     
;       IES_lookup_tab0_page		INTEGER  PAGE N0.- IESLUT0 Base Address
;       IES_lookup_tab0_offset		INTEGER	 OFFSET	 - 	"	"
;       IES_lookup_Tab_len		INTEGER	 LENGTH	 - IESLUT Table
;       IES_lookup_GenBase_page		INTEGER  PAGE N0.- IESLUT GenBASE Addres
;       IES_lookup_GenBase_offset	INTEGER  OFFSET	 -	"	" 
;       HIST_lookup_tab0_page		INTEGER  PAGE N0.- HISTLUT0 Base  Addres
;       HIST_lookup_tab0_offset		INTEGER	 OFFSET	 -	"	"       
;       HIST_lookup_Tab_len		INTEGER	 LENGTH	 - HIST Table          
;       HIST_lookup_GenBase_page	INTEGER	 PAGE N0.- HISTLUT GenBASE Add
;       HIST_lookup_GenBase_offset	INTEGER	 OFFSET	 -	"`	"    
;       Analogs_page			INTEGER	 PAGE N0.- Analog Base Address
;       Analogs_offset			INTEGER	 OFFSET	 -	"	"
;       InputCmdStruct_page		INTEGER	 PAGE N0._ Input Command Struct
;       InputCmdStruct_offset		INTEGER	 OFFSET	 - 	"	"
;       CmdEchoStruct_page		INTEGER	 PAGE N0.- CMD Echo Structure
;       CmdEchoStruct_offset		INTEGER	 OFFSET  _	"	"	
;
;       Num_tab_add_recs	        LONG	 Number of records in 
;						 TABLE_ADDRESS structure
;				 		 ( not part of the structure )

;
;   COMMON CEP_DPU_Mem_Dump,      DPU_MEMORY_DMP, num_mem_dmp_recs

;       DPU_MEMORY_DMP          STRUCTURE
;   DPU_MEMORY_DMP TAG_NAMES	TYPE				DESCRIPTION

;	Time			LONG		Time in milliseconds	
;       DATA_SEGMENT_ADDRESS    INTEGER		Segment address for data
;	DATA_OFFSET_ADDRESS	INTEGER		Offset address for data
;	DATA_DUMP		BYTARR( 128 )   Dump of data 
;	
;       NUM_MEM_DMP_RECS	LONG 		Number of records in 
;						DPU_MEMORY_DMP structure
;				 		( not part of the structure )



;   COMMON CEP_HEALTH_SUBCOM, 	CEP_HEALTH, nrecs_for_each_sub
;

;       CEP_HEALTH	        STRUCTURE
;   CEP_HEALTH TAG NAMES	TYPE			DESCRIPTION
;
;       SUBCOM0_TIME		LONG		Time in milliseconds subcom = 0
;	SUBCOM0_HISTCMD         INTEGER		Sensor status subcom = 0
;       SUBCOM0_PCmdCntr        INTEGER         General status subcom = 0
;       SUBCOM0_HVLVv 		BYTE      	ANALOG 1 subcom = 0 HV/LV(V)
;       SUBCOM0_HVLVi 		BYTE      	ANALOG 2 subcom = 0 HV/LV(I)
;       SUBCOM0_DPUv 		BYTE      	ANALOG 3 subcom = 0 DPU(V)
;       SUBCOM0_DPUi 		BYTE      	ANALOG 4 subcom = 0 DPU(I)
;       SUBCOM1_TIME		LONG		Time in milliseconds subcom = 1
;	SUBCOM1_HISTCmdErrCnt   INTEGER		Sensor status subcom = 1
;       SUBCOM1_SCmdCntr        INTEGER         General status subcom = 1
;       SUBCOM1_IPSP30V		BYTE      	ANALOG 1 subcom = 1 IPS+30(V)
;       SUBCOM1_IPSP7i 		BYTE      	ANALOG 2 subcom = 1 IPS+7(I)
;       SUBCOM1_IPSP7v 		BYTE      	ANALOG 3 subcom = 1 IPS+7(V)
;       SUBCOM1_IPSM4v 		BYTE      	ANALOG 4 subcom = 1 IPS-4(V)
;       SUBCOM2_TIME		LONG		Time in milliseconds subcom = 2
;	SUBCOM2_IESCMD		INTEGER		Sensor status subcom = 2
;       SUBCOM2_SCmdErrCnt      INTEGER         General status subcom = 2
;       SUBCOM2_IPSM4i		BYTE      	ANALOG 1 subcom = 2 IPS-4(I)
;       SUBCOM3_TIME		LONG		Time in milliseconds subcom = 3
;	SUBCOM3_IESCmdErrCnt	INTEGER		Sensor status subcom = 3
;       SUBCOM3_MemLoadCnt      INTEGER         General status subcom = 3
;       SUBCOM4_TIME		LONG		Time in milliseconds subcom = 4
;	SUBCOM4_IESZero		INTEGER		Sensor status subcom = 4
;       SUBCOM4_MemLoadErrCnt   INTEGER         General status subcom = 4
;       SUBCOM4_HeadTemp1	BYTE      	ANALOG 1 subcom = 4 HeadTemp1
;       SUBCOM4_HeadTemp2	BYTE      	ANALOG 2 subcom = 4 HeadTemp2
;       SUBCOM4_HeadTemp3	BYTE      	ANALOG 3 subcom = 4 HeadTemp3
;       SUBCOM4_HeadTemp4	BYTE      	ANALOG 4 subcom = 4 HeadTemp4
;       SUBCOM5_TIME		LONG		Time in milliseconds subcom = 5
;	SUBCOM5_IPSZero		INTEGER		Sensor status subcom = 5
;       SUBCOM5_CodeVersion     INTEGER         General status subcom = 5
;       SUBCOM5_HIST1		BYTE      	ANALOG 1 subcom = 5 HIST1
;       SUBCOM5_HIST2		BYTE      	ANALOG 2 subcom = 5 HIST2
;       SUBCOM5_HIST3		BYTE      	ANALOG 3 subcom = 5 HIST3
;       SUBCOM5_HIST4		BYTE      	ANALOG 4 subcom = 5 HIST4
;       SUBCOM6_TIME		LONG		Time in milliseconds subcom = 6
;	SUBCOM6_ICTCRCs		INTEGER		Sensor status subcom = 6
;       SUBCOM6_MSTCRCs		INTEGER         General status subcom = 6
;       SUBCOM6_IES1		BYTE      	ANALOG 1 subcom = 6 IES1
;       SUBCOM6_IES2		BYTE      	ANALOG 2 subcom = 6 IES2
;       SUBCOM6_IES3		BYTE      	ANALOG 3 subcom = 6 IES3
;       SUBCOM6_IES4		BYTE      	ANALOG 4 subcom = 6 IES4
;       SUBCOM6_TIME		LONG		Time in milliseconds subcom = 7
;	SUBCOM7_LUTCRCs		INTEGER		Sensor status subcom = 7
;       SUBCOM7_SATCRCs		INTEGER         General status subcom = 7
;
;       NRECS_FOR_EACH_SUB	LONARR( 8 )	Number of records for each 
;						subcom value in CEP_HEALTH 
;						structure 
;				 		( not part of the structure )


;   COMMON CEP_HEALTH_NOSUB,        NOSUB_CEP_HEALTH, recs_extracted
;
;       NOSUB_CEP_HEALTH	        STRUCTURE
;   NOSUB_CEP_HEALTH TAG NAMES	TYPE			DESCRIPTION
;       TIME			LONG	Time in milliseconds		
;	SC_AN0			BYTE	See 'Polar Command & Telemetry Handbook'
;	SC_AN1			BYTE    See 'Polar Command & Telemetry Handbook'
;	SC_AN2			BYTE	See 'Polar Command & Telemetry Handbook'
;	SC_AN3			BYTE	See 'Polar Command & Telemetry Handbook'See 'Polar Command & Telemetry Handbook'	
;	SC_bstat		BYTE	See 'Polar Command & Telemetry Handbook'
;	SC_th0			BYTE	See 'Polar Command & Telemetry Handbook'
;	SC_th1			BYTE	See 'Polar Command & Telemetry Handbook'
;	SC_th2			BYTE	See 'Polar Command & Telemetry Handbook'	
;	SC_th3			BYTE	See 'Polar Command & Telemetry Handbook'
;	CMDEcho_1		BYTE	See 'Polar Command & Telemetry Handbook'
;	CMDEcho_2		BYTE	See 'Polar Command & Telemetry Handbook'		
;	CMDEcho_3		BYTE	See 'Polar Command & Telemetry Handbook'
;	CMDEcho_4		BYTE	See 'Polar Command & Telemetry Handbook'
;       CMDEcho_5		BYTE	See 'Polar Command & Telemetry Handbook'
;	hw_status		BYTE    See 'CEPPAD Telemetry Format Handbook'
;       hw_status		BYTE	See 'CEPPAD Telemetry Format Handbook'
;       sw_status		BYTE	See 'CEPPAD Telemetry Format Handbook'
;       CheckSum		BYTE	See 'Polar Command & Telemetry Handbook'
;
;       RECS_EXTRACTED	        LONG		Number of records in
;						NOSUB_CEP_HEALTH structure
;
;               End of RDAF defined COMMON variables
;__________________________________________________________________________
;__________________________________________________________________________

;  COMMON BLOCK NAME                   DESCRIPTION OF VARIABLES
   
;  COMMON ACC_TABL               Num_energies -- Integer word for the number of 
;                                                energies
;                                Num_channels -- Integer word for the number of
;                                                channels
;                                Spins_per_samp -- Integer array for number of
;                                                  spins per sample for each 
;                                                  energy
;                                Sects_per_spin -- Integer array for the sectors
;                                                  ( pixel/sample) per channel
;                                                   for each energy
;                                Total_sects_ene -- Long array for the total 
;                                                   number of sectors in each
;                                                   energy


;  COMMON ACQ_FLAGS              One_spin_acq -- An integer array of flags  
;                                                indicating which of the 
;                                                16 time sub-records contain
;                                                data acquired in each energy
;                                                range
;                                                ( if the spin and energy you 
;                                                are on matches with the flag,
;                                                all channels in that energy 
;                                                start data accumulation )
;                                                
;                                Two_spin_acq   -- The same as above except 
;                                                  flags are based on 2 spin
;                                                  accumulation 
;                                Four_spin_acq  -- The same as above except 
;                                                  flags are based on 4 spin
;                                                  accumulation 
;                                Eight_spin_acq -- The same as above except 
;                                                  flags are based on 8 spin
;                                                  accumulation 
;                                Sixtn_spin_acq -- The same as above except 
;                                                  flags are based on 16 spin
;                                                  accumulation 


;  COMMON COMN_IND               Comn_indx   --  Longword array with the 
;                                                locations of sector 0
;                                                for each energy
;                                Npts_ene    --  Longword array with the 
;                                                number of points for each 
;                                                energy for the requested time 
;                                                span
;                                Comn_ch_indx -- Longword array with locations
;                                                sector 0 for each channel and
;                                                energy. Used as index into
;                                                common data array such as 
;                                                PLPCIPS_BYTES


;  COMMON DATA_MAP               Data_rec_ind -- Location into normal mode 
;                                                LEVEL-1 logical record for 
;                                                sector 0 at each energy/spin


;  COMMON FILDEF                 Num_recs_file -- Longword number of logical 
;                                                 records in a CEPPAD normal 
;                                                 mode LEVEL-1 file
;                                Max_rec_size  -- Long word giving the size of
;                                                 LEVEL-1 logical in bytes
;                                File_type -- Longword denoting type of data
;                               	     1L = Housekeeping file
;                                            2L = IPS Normal mode telemetry
;                                            3L = IES Normal mode telemetry
;                                            4L = HISTE Normal mode telemetry
;                                            5L = HISTP Normal mode telemetry
;                                            6L	= IPS Rates mode telemetry
;                                            7L = IES Rates mode telemetry
;     	                                     8L = HIST Rates mode telemetry
;	                                     9L = Spin header file data
;                                Iyear -- Longword year of data in LEVEL-1 file
;                                Iday  -- Longword beginning day of data 
;                                         in LEVEL-1 file
;                                Begt  -- Longword beginning time of data 
;                                         in LEVEL-1 file
;                                Endt  -- Longword ending time of data 
;                                         in LEVEL-1 file


;  COMMON MUXS                   Num_mux -- Longword number of mux values 
;                                           found in the requested time range
;                                           (IPS normal mode only)  
;                                Time_mux -- Longword array of times at 
;                                            which the mux values were found
;                                           (IPS normal mode only)  


;  COMMON PHYS_REC_SIZ           Num_byts_rec -- Longword number of bytes in 
;                                                in LEVEL-1 normal mode 
;                                                physical rec


;  COMMON RATES_ACC,             N_energies -- Integer word for the number of
;                                              rates energies
;                                Nchans -- Integer for the number of IES rates
;                                          channels
;                                Nrecs_tim_span -- Longword equal to the number
;                                                  of RATES records read for 
;                                                  user requested time span


;  COMMON REC_NUM                First_rec_read -- Longword first record 
;                                                  which has the 'TIME1' value
;                                Last_rec_read  -- Longword last record which
;                                                  has the 'TIME2' value


;  COMMON SENSOR                 Inst -- Character word with the name of a
;                                        sensor ( Depends on the value of
;                                        file_type )


;  COMMON SPIN_ACQ               Spin_acq_flg -- Integer acquisition 
;                                                flags (one_spin, two_spin, 
;                                                four spin ,etc. ) for each
;                                                energy based on spins per 
;                                                sample defined by the
;                                                Telemetry/Accumulation table
;                                                used for the requested 
;                                                time range


;  COMMON SPN_HEAD               Num_head --     Longword number of spin header
;                                                records read from Spin Header
;                                                file ( same as variable, 
;                                                PLCPSPN_HED_NUM in 
;                                                common POL_CEP_SH )
;                                Head     --     Byte array of Spin Header (SH)
;                                                data ( same as variable
;                                                PLCPSPN_HED in common
;                                                POL_CEP_SH )


;  COMMON TIME_MAP               Time_ind     -- Longword array mapping of 
;                                                the spin times in a LEVEL-1
;                                                normal mode logical record


;  COMMON TMADEF                 Tm_tabl_num  -- Longword DPU table number
;                                Tm_tabl_crc  -- Unique calculated longword
;                                                table number which represents
;                                                TM table contents
;     
;__________________________________________________________________________

FORWARD_FUNCTION FLOATING_PT

PRO GET_CEP_DATA, FILENAME_LEVEL1, TIME1, TIME2

; The purpose of this procedure is to extract CEPPAD data from the LEVEL-1 
; data file for the time range given by the value of parameters 'TIME1' and 
; 'TIME2' and to fill the defined common variables.  

;		This is version 2.0		07 June 1996
;               This is version 2.01            31 July 1996 
;               This is version 2.02            15 August 1996
;               This is version 2.03            22 November 1996
 
;      PARAMETER                DESCRIPTION                       TYPE

;  FILENAME_LEVEL1          name of file to open                  character 
;  TIME1                    beginning time of time range          longword
;  TIME2                    ending time of time range             longword

;	revised	
;	 6 May 1994	G. M. Boyd
;	13 Sep 1995	M.T.Redding	Added new and prviously missing 
;					descriptions of procedures
;					'Rates_TNEW_RNG', 'Rates_COMN_FILL',
;					'REPLACE_NETWORK_LONGS_WITH_HOST',
;					'REPLACE_NETWORK_SHORTS_WITH_HOST',
;	13 Jan 1996	J L Roeder	Modified Rates_TNEW_RNG to accept
;					times outside of file limits
;        3 Apr 1996			Added changes to incorporate the
;					extraction of data from the
;					Housekeeping file.  All changes will
;					be dated 3 APR 1993.
;	04 Apr 1996	M.T.Redding	Extended variable 'PLCPSpn_hed' in 
;					COMMON block POL_CEP_SH to 13 bytes -
;					additional byte contains SCOM - CEPPAD
;					spin number (0-255).  Corrected code
;					to include IPS MUX as 12th byte of
;					each record.  Likewise altered
;					variable 'head' in COMMON block 
;					SPN_HEAD.  Spin header file is now
;					expected to contain records for ALL
;					spins processed instead of just spins
;					at which any of the first 7 parameters
;					are noticed to have changed.
;        3 Apr 1996			Added changes to incorporate the
;					extraction of data from the
;					Housekeeping file.  All changes will
;					be dated 3 APR 1993.
;	24 April 1996                   Implemented Rob Sheldon's index mapping
;                                       ( vectorization ) of the data such the 
;					extraction of the data is much faster.
;       07 June 1996			The reading of CEPPAD Level-1 
;					HouseKeeping file was added.  Data are
;					extracted and put into COMMON blocks.
;       31 July 1996			Renamed the houskeeping common blocks
;					because they conflicted with some 
;					common blocks in the 'MAKE_LEVEL_1'
;					procedure
;					  Old name	     New Name
;					HIST_Singles	 CEP_HIST_Singles
;					HIST_Events	 CEP_HIST_Events
;                                       Table_CRCs	 CEP_Table_CRCs
;					IPS_calib	 CEP_IPS_calib
;					Table_addresses  CEP_Table_addresses
;                                       DPU_Mem_Dump     CEP_DPU_Mem_Dump
;					Also,  corrected some errors
;					in the extraction of hist events data.
;       22 November 1996                The structure 'DPU_MEMORY_DUMP' was 
;					changed to 'DPU_MEMORY_DMP'. The user
;					input 'TIME1' and 'TIME2' were made to
;					be a LONG type, no matter what type the
;					user input is.
;
;_____________________________________________________________________________

 COMMON FILDEF,       num_recs_file, max_rec_size, file_type, iyear, iday, $
                      begt, endt
 COMMON PHYS_REC_SIZ, num_byts_rec
 COMMON REC_NUM,      first_rec_read, last_rec_read
 COMMON TMADEF,       tm_tabl_num, tm_tabl_crc


 version_number = 2.03
 version_date = '22 November 1996'
 print, ' '
 print, ' This is VERSION ', version_number, '   ', version_date

; 4 March 1996
;  The "ESTIMATE_TIMES" procedure and the corresponding references were changed 
;  to Estimate_L1_Times in order not to conflict with the "ESTIMATE_TIMES"
;  procedure used by the "Make_level_1" procedure.  The conflict would have 
;  come about if the "Make_level_1" procedure and the "Get_Cep_Data" procedure
;  were compiled in the same session.

 time1 = LONG( time1 ) * 1000L                    ; Time is in millsecs in file
 time2 = LONG( time2 ) * 1000L  

;   Procedure OPEN_CEP_LEVEL-1 will get a unit and open filename_level1.
;   If there is a problem opening the file, parameter ierror_1 is set to 1
;   and the unit is set free.

OPEN_CEP_LEVEL1, filename_level1, iunit, ierror_0

if( ierror_0 ne 0) then begin
     print, ' '
     print, ' ERROR OPENING FILE ', filename_level1
     print, ' STOPPED IN PROCEDURE GET_CEP_DATA '
        ;create crash so that calling
        ;routine's error checker gets it and processing continues!
        xxx = yyy1
     stop
endif else begin
     print, ' '
     print, ' ////// OPENED ',  filename_level1
endelse        


RD_CEP_LEV1_HR, iunit, head_rec, ierror      ; Read 1st physical record of 
                                             ; LEVEL-1 file 
                                             ; If there is a problem reading,
                                             ; ierror is set to non-zero

if( ierror ne 0) then begin                  ; Stop, if error reading LEVEL-1
      print,' '
      print,' ERROR READING LEVEL1 HEADER RECORD'
      print, ' STOPPED IN PROCEDURE GET_CEP_DATA '
      FREE_LUN, iunit
        ;create crash so that calling
        ;routine's error checker gets it and processing continues!
        xxx = yyy1
      stop          
endif 

GET_HR_DEFS, iunit, filename_level1, head_rec    ; Get LEVEL-1 File and
                                                 ; Telemetry Accumulation table
                                                 ; definitions. Common
                                                 ; blocks FILDEF and TMADEF 
                                                 ; are filled.
; 4 MARCH 1994
print, ' Beginning Time of Data in milliseconds', begt
print, '    Ending Time of Data in milliseconds', endt


; 3 APR 1996
;    Check to see if the Housekeeping file is to be read and set 
;	 the extraction flag, "extract_hk', appropriately.  0 = no reading
nphys_recs = 1
num_byts_rec = max_rec_size
if( file_type eq 1 )THEN extract_hk = 1 ELSE extract_hk = 0


;	3 APR 1996
;if(file_type lt 2) or ( file_type gt 8 ) then begin 
if( (file_type lt 2) or ( file_type gt 8 ) ) and ( extract_hk eq 0 ) then begin 
     print, ' '                                     ; This is not a normal
     print, ' PROBLEMS WITH TYPE OF FILE'           ; mode acquisition type 
     print, ' THIS IS NOT A NORMAL MODE FILE '      ; file or a rates type
     print, ' THIS IS NOT A RATES FILE'             ; file
     print, ' NORMAL MODE FILE TYPE CAN BE 2, 3, 4, or 5'
     print, '  RATES MODE FILE TYPE CAN BE 6, 7, OR 8'
     print, ' THIS IS FILE TYPE', file_type
     print, ' STOPPED IN PROCEDURE GET_CEP_DATA '
        ;create crash so that calling
        ;routine's error checker gets it and processing continues!
        xxx = yyy1
     stop                              
 endif else begin
     if( file_type ge 2 ) and ( file_type le 5) then nphys_recs = 2L
     if( file_type ge 6 ) and ( file_type le 8) then nphys_recs = 1L
     num_byts_rec = max_rec_size/nphys_recs     ; Size of a physical record
 endelse
    
     ;     A  LEVEL-1 normal mode logical record is 2 physical records
     ;     (nphys_recs = 2). This logical record has 16 spin times
     ;     (sub-records) of data.

     ;     For all other LEVEL-1 files, a logical record equals a 
     ;     physical record (nphys_recs = 1). This logical record
     ;     has one time.
    
     print, ' Max_rec_size,nphys_recs', max_rec_size, nphys_recs
     LEV1_CEP = assoc( iunit, bytarr(max_rec_size/nphys_recs) ) ; Structure file

     beg_time = -1L				; 4 MARCH 1996
     end_time = -1L

; 3 APR 1996
if( file_type eq 1 ) then $
    READ_CEP_HK_DATA, iunit, time1, time2, beg_time, end_time


if( file_type ge 2) and ( file_type le 5 ) then begin ; Extract normal mode data

     GET_TM_ACQ_TBL, iunit                     ; Get the telemetry acquisition
                                               ; table for retrieval of data
      
     L1_TIME_MAPPING                           ; Calculate the locations of the
                                               ; time in the LEVEl-1 logical
                                               ; record

     ;   Find the logical record that has 'TIME1' and set 'BEG_TIME' 
     ;   to the first time in that record. Find the logical record that has 
     ;   'TIME2' and set 'END_TIME' to the last time in that record.
     ;   'BEG_TIME' and 'END_TIME' will be the time span for the data 
     ;   extraction.  The common block, 'REC_NUM', will have the number of 
     ;   the first logical record (FIRST_REC_READ) to read and the number 
     ;   of the last record (LAST_REC_READ), to read.

      GET_NEW_TIM_RNG, lev1_cep, time1, time2, beg_time, end_time, error

; 4 March 1996
      if( error ne 0 ) then begin
          print, 'COULD not find requested time range', time1, time2
          print, 'The start and end time of data in milliseconds', $
                                         begt, endt
          print, ' STOPPED in GET_CEP_DATA after call to GET_NEW_TIM_RNG'
        ;create crash so that calling
        ;routine's error checker gets it and processing continues!
        xxx = yyy1
          STOP
      endif   
;xxx
      L1_DATA_MAPPING  ; Calculate locations 
                       ; that map to sector 0  of a spin/energy/channel
                       ; in LEVEL-1 normal mode logical data record   
                                                        
      L1_COMN_MAPPING, nspins_tim_span, nbyts_tim_span  ; Calculate locations
                       ; in the telemetry array filled for the requested time 
                       ; range that map to sector 0 of a spin/energy/channel
endif               


if ( file_type ge 6 ) and ( file_type le 8) then begin     ; Rates 
    
     ;   Find the record where 'TIME1" is greater or equal to the 
     ;   time in that record. Set 'BEG_TIME' equal to the time in the 
     ;   record before and set 'FIRST_REC_READ' to that record number.
     ;   Find the record where 'TIME2' is less than or equal to the time 
     ;   in that record.  Set 'END_TIME' equal to the time in the next
     ;   record and set 'LAST_REC_READ' to that record number.
     ;   The common block, 'REC_NUM', will have the number of 
     ;   the first record (FIRST_REC_READ) to read and the number 
     ;   of the last record (LAST_REC_READ), to read.

    RATES_TNEW_RNG, lev1_cep, time1, time2, beg_time, end_time, ierror_r
    if( ierror_r ne 0 ) then begin
       print, ' '
       print, ' ERROR GETTING NEW TIME RANGE FOR EXTRACTING RATES DATA '
       print, ' THE ERROR RETURNED FROM PROCEDURE RATES_TNEW_RNG = ', ierror_r
       print, ' STOPPED IN PROCEDURE GET_CEP_DATA '
        ;create crash so that calling
        ;routine's error checker gets it and processing continues!
        xxx = yyy1
       stop
    endif

endif
    

; 4 MARCH 1996
print, 'REQUESTED time range ', time1, time2
print, 'Extraction time range', beg_time, end_time

;GET_SH_INFO, beg_time, end_time   ; Extract all Spin Header (SH) information
;                       ; between 'beg_time' and 'end_time'
; 3 APRIL 1996
   ; Extract all Spin Header (SH) information between 'beg_time' and 'end_time'
GET_SH_INFO, beg_time, end_time, filename_level1

SETUP_COMMONS, nspins_tim_span, nbyts_tim_span    ; Set contants 
                       ; initialize variables, and dimension arrays for
                       ; the defined commons 

FILL_CEP_COMNS, nspins_tim_span, nbyts_tim_span, lev1_cep, time1, time2
                       ; Fill the defined common ( depends on the value of
                       ; file_type ) with data for the revised time range.
                       ; Include the requested time range values


;	Close the Level-1 data file and free up the unit number

CLOSE, iunit
FREE_LUN, iunit

return
end


;##############################################################################
PRO EST_REC_NUMBER, LEV1_CEP, TIME, RECORD_INDEX, IERROR

;   This procedure searches for the CEPPAD logical ( 16 spin time ) record 
;   that has the TIME for which you want to extract data.
;      NOTE:  This procedure works only on times that are ascending 


;      PARAMETER                DESCRIPTION                       TYPE

;  LEV1_CEP            Data access file ( associated file )      File structure
;  TIME                This is what you are searching for        Longword
;  RECORD_INDEX        Record number ( long) containing time     Longword
;             (TIME is located between position LOW_REC_NUM and HIGH_REC_NUM )
;  IERROR               0 --  Everything is okay                 Integer
;                      -1 --  Number of records is lt 2
;                      -2 --  Time for which you are searching outside
;				times in the data file
;                      -3 --  All times in logical record  = -1. This
;			      should not happen.	


;        beginning time of data (msec)             begt
;        ending time of data    (msec)             endt


;	REVISION
;	13 Apr 1994	G. M. Boyd
;	14 Aug 1995	M.T.Redding	Commented out the initial setting of
;					variables 'low' and 'high' - these are
;					supposed to be passed by the calling
;					procedure;  also removed the range
;					comparison with the File beginning and
;					ending times 'begt' and 'endt' - instead
;					moved this code to procedure 
;					'GET_NEW_TIM_RNG' where special
;					assignments for 'first_rec_read',
;					'last_rec_read', 'beg_time' and
;					'end_time' are made when the desired
;					time range extends partly outside the
;					data range available in the file.
					
;	14 Sep 1995	M. T. Redding	Added common block 'TIME_MAP' and code
;					to convert LONGWORD time values from
;					NETWORK format to HOST format

;	14 Aug 1996	G. M. Boyd     Corrected the binary search such that 
;					the 'RECORD_INDEX' parameter would
;					return the correct index if the 
;					time for which you are searching
;					was found on the first try. Also, 
;					removed the stops by adding another
;					error condition ( - 3 ).
;____________________________________________________________________________

 COMMON FILDEF,       num_recs_file, max_rec_size, file_type, iyear, iday, $
                      begt, endt
 COMMON PHYS_REC_SIZ, num_byts_rec
 COMMON TIME_MAP,     time_ind

; 4 March 1996
;  The "ESTIMATE_TIMES" procedure and the corresponding references were changed 
;  to Estimate_L1_Times in order not to conflict with the "ESTIMATE_TIMES"
;  procedure used by the "Make_level_1" procedure.  The conflict would have 
;  come about if the "Make_level_1" procedure and the "Get_Cep_Data" procedure
;  were compiled in the same session.


ierror = 0

low_rec_num = 2L			; Data starts in 2nd logical record
hi_rec_num = num_recs_file		; Num_recs_file = number of logical
					; records in file.  A logical record
					; is a 16 spin time LEVEL-1 record and
					; is equal to 2 physical records.

if( num_recs_file lt 2L) then begin
   print, ' THE NUMBER OF RECORDS IN DATA FILE IS LESS THAN 2 '
   ierror = -1       
   return
endif

xb = begt                         ; First time in data file
xl = endt                         ; Last time in data file

if( time lt xb ) or ( time gt xl) then begin
   print, ' TIME SEARCHING FOR IS OUTSIDE THE TIMES IN DATA FILE '
   ierror = -2 
   return
endif 

new_index = ( hi_rec_num + low_rec_num) / 2L    ; Initial logical record number 

WHILE ( ( hi_rec_num - low_rec_num) GE 1L) DO BEGIN

    dat = bytarr(max_rec_size)
    phys_rec1 = (new_index * 2L) - 1L         ; Calculate physical record
    phys_rec2 =  phys_rec1 + 1L               ; location 
    d = lev1_cep( phys_rec1 - 1L)             ; Get 1st half of logical record
    dat(0 : num_byts_rec - 1) = d                  
    d = lev1_cep( phys_rec2 - 1L)             ; Get 2nd half of logical record
    dat(num_byts_rec : max_rec_size - 1L) = d                 

    Replace_NETWORK_LONGs_with_HOST, 16, $    ; Convert the LONGWORD time 
       time_ind, dat                          ; values from network 
                                              ; (big-endian) to HOST format
    GET_TIMES, dat, times_in_rec              ; Find the times in a logical rec
 
    times_in_rec = ABS( times_in_rec )		; 29 may 1996
                                              ; record ( max of 16 )

    ESTIMATE_L1_TIMES, times_in_rec, $        ; Interpolate/extrapolate
       est_times_in_rec, ierror               ; when any of the 16 times
                                              ; is negative

   if ( ierror eq 1 ) then begin             ; All times were less than zero
        print, ' '
        print, ' THERE WERE NO POSITIVE TIMES IN THIS LOGICAL RECORD', new_index
        print, ' STOPPED IN PROCEDURE EST_REC_NUMBER'
        ierror = -3
        RETURN
    endif

;	Check for time preceding the current record---Reduce the high record
;    if ( time LT est_times_in_rec( 0 ) ) then  hi_rec_num = new_index 15 Aug	
    if ( time LE est_times_in_rec( 15 ) ) then  hi_rec_num = new_index	

;	Check for time within the current record
    IF ( ( time GE est_times_in_rec(0) ) AND $
         ( time LE est_times_in_rec(15) ) ) THEN BEGIN		; Found the time
        record_index =  hi_rec_num 				; 15 Aug 
        ierror = 0
        RETURN
    endif 

;	Check for time after the last time in the current record. Such a 
;	time may still be before the first time in the next record - in which
;	case say the current record contains the desired time  and return.
;	Otherwise bump up the low_rec_num to the next record and continue the
;	search

    IF ( time GT est_times_in_rec(0) ) THEN BEGIN
        
;	Read the next record, if it exists, check to see if the desired 
;	time is in the current record but after its 16th time

       IF ( new_index + 1 LE num_recs_file ) THEN BEGIN

;	Next record exists - read it, get its times and compare with the
;	desired time

          dat = bytarr(max_rec_size)
          phys_rec1 = ((new_index+1) * 2L) - 1L	; figure next logical record's
          phys_rec2 =  phys_rec1 + 1L		; physical physical locations
          d = lev1_cep( phys_rec1 - 1L)         ; Get 1st half of logical record
          dat(0 : num_byts_rec - 1) = d                  
          d = lev1_cep( phys_rec2 - 1L)         ; Get 2nd half of logical record
          dat(num_byts_rec : max_rec_size - 1L) = d                 

          Replace_NETWORK_LONGs_with_HOST, 16, $    ; Convert the LONGWORD time 
             time_ind, dat                          ; values from network 
                                                    ; (big-endian) to HOST format
          GET_TIMES, dat, times_in_rec              ; Find the times in a logical 

          times_in_rec = ABS( times_in_rec )		; 29 may 1996
                                                    ; record ( max of 16 )
          ESTIMATE_L1_TIMES, times_in_rec, $           ; Interpolate/extrapolate
          est_times_in_rec, ierror                  ; when any of the 16 times
                                                    ; is negative

          if ( ierror eq 1 ) then begin             ; All times were less than zero
               print, ' '
               print, ' THERE WERE NO POSITIVE TIMES IN THIS LOGICAL RECORD', $
                        new_index + 1
               print, ' STOPPED IN PROCEDURE EST_REC_NUMBER'
               ierror = -3
               RETURN
          endif

          ;  Times between 16th time of previous record and first time of next 
          IF ( time LT est_times_in_rec(0) ) THEN BEGIN
             record_index = new_index
             print, ' The first time of next record ', est_times_in_rec( 0 )
             ierror = 0
             RETURN
          ENDIF	ELSE BEGIN	; ( time LT est_times_in_rec(0) )
             low_rec_num = new_index + 1	; Use the upper half of range for next search
          ENDELSE	; ( time LT est_times_in_rec(0) )

       ENDIF ELSE BEGIN	; Time desired is not in the file
           print, ' DESIRED TIME IS OUTSIDE THE TIMES IN DATA FILE '
           ierror = -2
           RETURN
       ENDELSE	;( new_index + 1 LE num_recs_file )

    endif	; ( ( time GT est_times_in_rec(0) ) 
 
    new_index = (hi_rec_num + low_rec_num)/2L   ; The new record number

ENDWHILE

record_index =  low_rec_num 

d = 0                                   ; Release space
dat = 0                                 ; Release space

return
end
;##############################################################################
PRO ESTIMATE_L1_TIMES, TRGT_TIMES, EST_16_TIMES, ERROR_RETURN
;PRO ESTIMATE_TIMES, TRGT_TIMES, EST_16_TIMES, ERROR_RETURN

;	Make an array of 16 times based on the existent times within the
;	the first 16 times of the Trgt_times array 
;	and filling in the missing times (those having values
;	of '-1') using interpolation/extrapolation.


;   AUTHOR:                                 
;   M. REDDING                              11 April 1994
;   G. BOYD (rev.)                          22 April 1994
;          Removed common blocks, CEP_lv10_1_map and CEP_constants.
;          Put ' TRG_times'  as the first parameter of the call     
;          The spin rate was set to literal, 6000L (milliseconds)

; 4 March 1996
;  This  procedure and the corresponding references were changed 
;  to Estimate_L1_Times in order not to conflict with the "ESTIMATE_TIMES"
;  procedure used by the "Make_level_1" procedure.  The conflict would have 
;  come about if the "Make_level_1" procedure and the "Get_Cep_Data" procedure
;  were compiled in the same session.


;--------------------------------------------------------------------


; error_return		; 
; Est_16_times		; array of 16 'estimated' times

;--------------------------------------------------------------------

;	Initialize the time array to be filled.

   est_16_times    = lonarr(16)
   est_16_times(*) = -1L

   non_missing = where( Trgt_times(0:15) NE -1L )

;	Fill in any missing times

   size_non_missing = SIZE( non_missing )
   dim_non_missing  = size_non_missing(0)

   IF ( dim_non_missing EQ 0 ) THEN BEGIN	; No defined times in this block
       print, ' 16-spin block encountered with NO defined times '
       error_return = 1
       RETURN
   ENDIF

;	Transfer what times do exist from the Trgt_times array 

   est_16_times(non_missing) = Trgt_times(non_missing)
   len_non_missing  = size_non_missing(1)
   IF ( len_non_missing EQ 16 ) THEN BEGIN	; All 16 times already defined 
      error_return = 0
      RETURN
   ENDIF

;	Extrapolate where necessary to get 1st and last time in array

   first_non_missing = non_missing(0)
   last_non_missing  = non_missing(len_non_missing-1)

;	Determine an appropriate nominal time increment - base it on existing
;	time values if possible, otherwise use the defined nominal spin 
;	duration

   IF ( len_non_missing GE 2 ) THEN BEGIN
      ave_inc = ( Trgt_times(non_missing(len_non_missing-1)) - $
                    Trgt_times(non_missing(0)) ) / $
                ( non_missing(len_non_missing-1) - non_missing(0) )
   ENDIF ELSE BEGIN
      ave_inc = 6000L                       ; Nominal spin duration
   ENDELSE


   If ( first_non_missing NE 0 ) THEN BEGIN
      est_16_times(0) = Trgt_times(first_non_missing) - $
                           first_non_missing * ave_inc
   ENDIF
   If ( last_non_missing NE 0 ) THEN BEGIN
      est_16_times(15) = Trgt_times(last_non_missing) + $
                           ( 15 - last_non_missing ) * ave_inc
   ENDIF

;	Now get the status of the Estimated time array

   non_missing = where( Est_16_times NE -1L )
   size_non_missing = SIZE( non_missing )
   dim_non_missing  = size_non_missing(0)

   IF ( dim_non_missing EQ 0 ) THEN BEGIN	; No defined times in this block
       print, ' 16-spin block encountered with NO defined times '
       error_return = 1
       RETURN
   ENDIF
   len_non_missing  = size_non_missing(1)

   IF ( len_non_missing EQ 16 ) THEN BEGIN	; All 16 times already defined 
      error_return = 0
      RETURN
   ENDIF

;	Calculate each missing time by adding to its predecessor an average
;	increment based on the non-missing time values.

   FOR index = 0, len_non_missing-2 DO BEGIN
      ave_inc = ( Est_16_times(non_missing(index+1)) - $
                  Est_16_times(non_missing(index)) ) / $
            LONG( non_missing(index+1) - non_missing(index) )  

      jmissing = non_missing(index) + 1
      WHILE jmissing LT non_missing(index+1) DO BEGIN
         Est_16_times(jmissing) = Est_16_times(jmissing-1) + ave_inc
         jmissing = jmissing + 1
      ENDWHILE
   ENDFOR

   error_return = 0

   RETURN
END

;###############################################################################
PRO EXTRACT_NOSUB_HEALTH


 COMMON FILDEF,      		num_recs_HK, nbyts_rec_HK, file_type, $
                                iyear_HK, day_HK, begt_HK, endt_HK
COMMON CEP_HEALTH_NOSUB,        NOSUB_CEP_HEALTH, recs_extracted
COMMON CEP_HK_EXTRACTION,       ten_mjrfrm_milsecs, num_recs_extracted, hk_data

;     This procedure fills the NOSUB_CEP_HEALTH structure with the parameters
;     that describe the health of the CEPPAD instrument. These parameters
;     are not based on a subcom value and are available every frame. 
;     See below for structure tag names.
;
;
;	G. M. Boyd					1 June 1996
;_____________________________________________________________________________


nosub = { hk_nosub,   time: 0L,						$
		sc_an0: 0b,           	    				$
		sc_an1: 0b,           	    				$
		sc_an2: 0b,           	    				$
		sc_an3: 0b,           	    				$
		sc_bstat: 0b,          	    				$
		sc_th0: 0b,          	    				$
		sc_th1: 0b,          	    				$
		sc_th2: 0b,          	    				$
		sc_th3: 0b,          	    				$
		CMDEcho_1: 0b,						$
		CMDEcho_2: 0b,						$
		CMDEcho_3: 0b,						$
		CMDEcho_4: 0b,						$
 		CMDEcho_5: 0b,          				$
		hw_status1: 0b,             hw_status2: 0b,        	$
		sw_status: 0,               CheckSum: 0			}


  NOSUB_CEP_HEALTH = replicate( {hk_nosub}, num_recs_extracted )
 
  NOSUB_CEP_HEALTH.time =   hk_data( 0 : num_recs_extracted - 1 ).tmsec    
  NOSUB_CEP_HEALTH.sc_an0 = hk_data( 0 : num_recs_extracted - 1 ).sc_an0 
  NOSUB_CEP_HEALTH.sc_an1 = hk_data( 0 : num_recs_extracted - 1 ).sc_an1 
  NOSUB_CEP_HEALTH.sc_an2 = hk_data( 0 : num_recs_extracted - 1 ).sc_an2 
  NOSUB_CEP_HEALTH.sc_an3 = hk_data( 0 : num_recs_extracted - 1 ).sc_an3 

  NOSUB_CEP_HEALTH.sc_bstat = hk_data( 0 : num_recs_extracted - 1 ).sc_bstat
  NOSUB_CEP_HEALTH.sc_th0 = hk_data( 0 : num_recs_extracted - 1 ).sc_th0
  NOSUB_CEP_HEALTH.sc_th1 = hk_data( 0 : num_recs_extracted - 1 ).sc_th1 
  NOSUB_CEP_HEALTH.sc_th2 = hk_data( 0 : num_recs_extracted - 1 ).sc_th2 
  NOSUB_CEP_HEALTH.sc_th3 = hk_data( 0 : num_recs_extracted - 1 ).sc_th3 


  NOSUB_CEP_HEALTH.CMDEcho_1 = hk_data( 0 : num_recs_extracted - 1 ).CMDEcho_1
  NOSUB_CEP_HEALTH.CMDEcho_2 = hk_data( 0 : num_recs_extracted - 1 ).CMDEcho_2
  NOSUB_CEP_HEALTH.CMDEcho_3 = hk_data( 0 : num_recs_extracted - 1 ).CMDEcho_3
  NOSUB_CEP_HEALTH.CMDEcho_4 = hk_data( 0 : num_recs_extracted - 1 ).CMDEcho_4
  NOSUB_CEP_HEALTH.CMDEcho_5 = hk_data( 0 : num_recs_extracted - 1 ).CMDEcho_5

  NOSUB_CEP_HEALTH.hw_status1 = hk_data( 0 : num_recs_extracted - 1 ).hw_status1
  NOSUB_CEP_HEALTH.hw_status2 = hk_data( 0 : num_recs_extracted - 1 ).hw_status2


  NOSUB_CEP_HEALTH.sw_status = $
       FIX( hk_data( 0 : num_recs_extracted - 1 ).sw_status( 0 ) ) * 256   +  $
       FIX( hk_data( 0 : num_recs_extracted - 1 ).sw_status( 1 ) )

  NOSUB_CEP_HEALTH.CheckSum = $
       FIX( hk_data( 0 : num_recs_extracted - 1 ).CheckSum( 0 ) ) * 256   +   $ 
       FIX( hk_data( 0 : num_recs_extracted - 1 ).CheckSum( 1 ) )

recs_extracted = num_recs_extracted
 
    return
    end

;##############################################################################
PRO EXTRACT_SUBCOM_HEALTH

 COMMON FILDEF,      		num_recs_HK, nbyts_rec_HK, file_type, $
                                iyear_HK, day_HK, begt_HK, endt_HK
COMMON CEP_HEALTH_SUBCOM,       CEP_HEALTH, nrecs_for_each_sub
COMMON CEP_HK_EXTRACTION,       ten_mjrfrm_milsecs, num_recs_extracted, hk_data

;     This procedure fills the CEP_HEALTH structure with the parameters
;     that describe the health of the CEPPAD instrument. These parameters
;     are extracted based on a subcom parameter as opposed to health
;     parameters that are not on a subcom.  See below for structure tag names.
;
;
;	G. M. Boyd					27 May 1996
;_____________________________________________________________________________


;   Initialize health parameter structures - ( based on a subcom value )
    hk0 =  { hkeep, subcom0_time: 0L,         subcom0_HistCmd: 0,	$    
		    subcom0_PCmdCntr: 0,      subcom0_HVLVV: 0b,	$
                    subcom0_HVLVi: 0b,	      subcom0_DPUv: 0b,		$
		    subcom0_DPUi: 0b,       				$
                    subcom1_time: 0L,	      subcom1_HistCmdErrCnt: 0,	$
		    subcom1_SCmdCntr: 0,      subcom1_IPSP30V: 0b,	$
                    subcom1_IPSP7i: 0b,       subcom1_IPSP7v: 0b,	$
		    subcom1_IPSM4v: 0b, 				$
                    subcom2_time: 0L,         subcom2_IESCmd: 0,        $
		    subcom2_SCmdErrCnt: 0,    subcom2_IPSM4i: 0b,	$
                    subcom3_time: 0L,	      subcom3_IESCmdErrCnt: 0,	$
		    subcom3_MemLoadCnt: 0,                             	$
                    subcom4_time: 0L,         subcom4_IESZero: 0,      	$
                    subcom4_MemLoadErrCnt: 0, subcom4_HeadTemp1: 0b,	$
                    subcom4_HeadTemp2: 0b,    subcom4_HeadTemp3: 0b,	$
      		    subcom4_IPSTemp: 0b,     				$
                    subcom5_time: 0L,         subcom5_IPSZero: 0,      	$
  		    subcom5_CodeVersion: 0,   subcom5_HIST1: 0b,	$
                    subcom5_HIST2: 0b,	      subcom5_HIST3: 0b,	$
                    subcom5_HIST4: 0b,					$
                    subcom6_time: 0L,         subcom6_ICTCRCs: 0,	$ 
		    subcom6_MSTCRCs: 0,       subcom6_IES1: 0b,		$
                    subcom6_IES2: 0b,	      subcom6_IES3: 0b,		$
                    subcom6_IES4: 0b,					$
                    subcom7_time: 0L,	      subcom7_LUTCRCs: 0,	$
                    subcom7_SATCRCs: 0                                  }	

    ; The 100 is added for cushion;  data skips could cause a problem
 ;   possible_nrecs = ( num_recs_extracted / 8L ) + 100L
;    possible_nrecs = ( num_recs_extracted / 8L ) + 200L ; found problem processing HK_1998163 mkc
    possible_nrecs = ( num_recs_extracted / 8L ) + 500L ; found problem processing HK_1998141 mkc
    CEP_HEALTH = replicate ({ HKeep },  possible_nrecs )

    nrecs_for_each_sub = LONARR( 8 )
    ; Fill the above structure

    subcoms = hk_data( 0 : num_recs_extracted - 1 ).kp_hk_subcom AND 15
    zero_indx = WHERE( ( subcoms eq 0 ), count  )    

    ; Program will not bomb if somehow the subcom stays on same number
    if( count gt possible_nrecs ) then zero_indx = zero_indx( 0 : count - 1 )
    if( count gt 0 ) then begin
        nrecs_for_each_sub( 0 ) = count
        CEP_HEALTH( 0 : count - 1 ).subcom0_time = hk_data( zero_indx ).tmsec 
    
        CEP_HEALTH( 0 : count - 1 ).subcom0_HistCmd = $
               FIX( hk_data( zero_indx ).sub_sens_stat( 0 ) ) * 256 +   $
               FIX( hk_data( zero_indx ).sub_sens_stat( 1 ) )

        CEP_HEALTH( 0 : count - 1 ).subcom0_PCmdCntr = $
               FIX( hk_data( zero_indx ).sub_gen_stat( 0 ) ) * 256 +   $
               FIX( hk_data( zero_indx ).sub_gen_stat( 1 ) )

        CEP_HEALTH( 0 : count - 1 ).subcom0_HVLVV = $
                     hk_data( zero_indx ).analog_1

        CEP_HEALTH( 0 : count - 1 ).subcom0_HVLVi = $
                     hk_data( zero_indx ).analog_2

        CEP_HEALTH( 0 : count - 1 ).subcom0_DPUv = $
                     hk_data( zero_indx ).analog_3

        CEP_HEALTH( 0 : count - 1 ).subcom0_DPUi = $
                     hk_data( zero_indx ).analog_4

    ENDIF 

    ;Housekeeping subcom value = 1
    one_indx = WHERE( ( subcoms eq 1 ), count  )    
    if( count gt possible_nrecs ) then one_indx = one_indx( 0 : count - 1 )

    if( count gt 0 ) then begin
        nrecs_for_each_sub( 1 ) = count
        CEP_HEALTH( 0 : count - 1 ).subcom1_time = hk_data( one_indx ).tmsec
    
        CEP_HEALTH( 0 : count - 1 ).subcom1_HISTCmdErrCnt = $
               FIX( hk_data( one_indx ).sub_sens_stat( 0 ) ) * 256 +   $
               FIX( hk_data( one_indx ).sub_sens_stat( 1 ) )

        CEP_HEALTH( 0 : count - 1 ).subcom1_SCmdCntr = $
               FIX( hk_data( one_indx ).sub_gen_stat( 0 ) ) * 256 +   $
               FIX( hk_data( one_indx ).sub_gen_stat( 1 ) )

        CEP_HEALTH( 0 : count - 1 ).subcom1_IPSP30v = $
                     hk_data( one_indx ).analog_1

        CEP_HEALTH( 0 : count - 1 ).subcom1_IPSP7i = $
                     hk_data( one_indx ).analog_2

        CEP_HEALTH( 0 : count - 1 ).subcom1_IPSP7v = $
                     hk_data( one_indx ).analog_3

        CEP_HEALTH( 0 : count - 1 ).subcom1_IPSM4V = $
                     hk_data( one_indx ).analog_4

    ENDIF 
        
    ;Housekeeping subcom value = 2
    two_indx = WHERE( ( subcoms eq 2 ), count  )    
    if( count gt possible_nrecs ) then two_indx = two_indx( 0 : count - 1 )

    if( count gt 0 ) then begin
        nrecs_for_each_sub( 2 ) = count
        CEP_HEALTH( 0 : count - 1 ).subcom1_time = hk_data( two_indx ).tmsec
    
        CEP_HEALTH( 0 : count - 1 ).subcom1_HISTCmdErrCnt = $
               FIX( hk_data( two_indx ).sub_sens_stat( 0 ) ) * 256 +   $
               FIX( hk_data( two_indx ).sub_sens_stat( 1 ) )

        CEP_HEALTH( 0 : count - 1 ).subcom1_SCmdCntr = $
               FIX( hk_data( two_indx ).sub_gen_stat( 0 ) ) * 256 +   $
               FIX( hk_data( two_indx ).sub_gen_stat( 1 ) )

        CEP_HEALTH( 0 : count - 1 ).subcom1_IPSP30v = $
                     hk_data( two_indx ).analog_1

        CEP_HEALTH( 0 : count - 1 ).subcom1_IPSP7i = $
                     hk_data( two_indx ).analog_2

        CEP_HEALTH( 0 : count - 1 ).subcom1_IPSP7v = $
                     hk_data( two_indx ).analog_3

        CEP_HEALTH( 0 : count - 1 ).subcom1_IPSM4V = $
                     hk_data( two_indx ).analog_4

    ENDIF 

    ;Housekeeping subcom value = 3
    three_indx = WHERE( ( subcoms eq 3 ), count  )    
    if( count gt possible_nrecs ) then three_indx = three_indx( 0 : count - 1 )

    if( count gt 0 ) then begin
        nrecs_for_each_sub( 3 ) = count
        CEP_HEALTH( 0 : count - 1 ).subcom3_time = hk_data( three_indx ).tmsec
    
        CEP_HEALTH( 0 : count - 1 ).subcom3_IESCmdErrCnt = $
               FIX( hk_data( three_indx ).sub_sens_stat( 0 ) ) * 256 +   $
               FIX( hk_data( three_indx ).sub_sens_stat( 1 ) )

        CEP_HEALTH( 0 : count - 1 ).subcom3_MemLoadCnt = $
               FIX( hk_data( three_indx ).sub_gen_stat( 0 ) ) * 256 +   $
               FIX( hk_data( three_indx ).sub_gen_stat( 1 ) )
    ENDIF


    ;Housekeeping subcom value = 4
    four_indx = WHERE( ( subcoms eq 4 ), count  )    
    if( count gt possible_nrecs ) then four_indx = four_indx( 0 : count - 1 )

    if( count gt 0 ) then begin
        nrecs_for_each_sub( 4 ) = count
        CEP_HEALTH( 0 : count - 1 ).subcom4_time = hk_data( four_indx ).tmsec 
    
        CEP_HEALTH( 0 : count - 1 ).subcom4_IESZero = $
               FIX( hk_data( four_indx ).sub_sens_stat( 0 ) ) * 256 +   $
               FIX( hk_data( four_indx ).sub_sens_stat( 1 ) )

        CEP_HEALTH( 0 : count - 1 ).subcom4_MemLoadErrCnt = $
               FIX( hk_data( four_indx ).sub_gen_stat( 0 ) ) * 256 +   $
               FIX( hk_data( four_indx ).sub_gen_stat( 1 ) )

        CEP_HEALTH( 0 : count - 1 ).subcom4_HeadTemp1 = $
                     hk_data( four_indx ).analog_1

        CEP_HEALTH( 0 : count - 1 ).subcom4_HeadTemp2 = $
                     hk_data( four_indx ).analog_2

        CEP_HEALTH( 0 : count - 1 ).subcom4_HeadTemp3 = $
                     hk_data( four_indx ).analog_3

        CEP_HEALTH( 0 : count - 1 ).subcom4_IPSTemp = $
                     hk_data( four_indx ).analog_4

    ENDIF 

    ;Housekeeping subcom value = 5
    five_indx = WHERE( ( subcoms eq 5 ), count  )    
    if( count gt possible_nrecs ) then five_indx = five_indx( 0 : count - 1 )

    if( count gt 0 ) then begin
        nrecs_for_each_sub( 5 ) = count
        CEP_HEALTH( 0 : count - 1 ).subcom5_time = hk_data( five_indx ).tmsec
    
        CEP_HEALTH( 0 : count - 1 ).subcom5_IPSZero = $
               FIX( hk_data( five_indx ).sub_sens_stat( 0 ) ) * 256 +   $
               FIX( hk_data( five_indx ).sub_sens_stat( 1 ) )

        CEP_HEALTH( 0 : count - 1 ).subcom5_CodeVersion = $
               FIX( hk_data( five_indx ).sub_gen_stat( 0 ) ) * 256 +   $
               FIX( hk_data( five_indx ).sub_gen_stat( 1 ) )

        CEP_HEALTH( 0 : count - 1 ).subcom5_HIST1 = $
                     hk_data( five_indx ).analog_1

        CEP_HEALTH( 0 : count - 1 ).subcom5_HIST2 = $
                     hk_data( five_indx ).analog_2

        CEP_HEALTH( 0 : count - 1 ).subcom5_HIST3 = $
                     hk_data( five_indx ).analog_3

        CEP_HEALTH( 0 : count - 1 ).subcom5_HIST4 = $
                     hk_data( five_indx ).analog_4

    ENDIF 

    ;Housekeeping subcom value = 6
    six_indx = WHERE( ( subcoms eq 6 ), count  )    
    if( count gt possible_nrecs ) then six_indx = six_indx( 0 : count - 1 )

    if( count gt 0 ) then begin
        nrecs_for_each_sub( 6 ) = count
        CEP_HEALTH( 0 : count - 1 ).subcom6_time = hk_data( six_indx ).tmsec
    
        CEP_HEALTH( 0 : count - 1 ).subcom6_ICTCRCs = $
               FIX( hk_data( six_indx ).sub_sens_stat( 0 ) ) * 256 +   $
               FIX( hk_data( six_indx ).sub_sens_stat( 1 ) )

        CEP_HEALTH( 0 : count - 1 ).subcom6_MSTCRCs = $
               FIX( hk_data( six_indx ).sub_gen_stat( 0 ) ) * 256 +   $
               FIX( hk_data( six_indx ).sub_gen_stat( 1 ) )

        CEP_HEALTH( 0 : count - 1 ).subcom6_IES1 = $
                     hk_data( six_indx ).analog_1

        CEP_HEALTH( 0 : count - 1 ).subcom6_IES2 = $
                     hk_data( six_indx ).analog_2

        CEP_HEALTH( 0 : count - 1 ).subcom6_IES3 = $
                     hk_data( six_indx ).analog_3

        CEP_HEALTH( 0 : count - 1 ).subcom6_IES4 = $
                     hk_data( six_indx ).analog_4

    ENDIF 

    ;Housekeeping subcom value = 7
    seven_indx = WHERE( ( subcoms eq 7 ), count  )    
    if( count gt possible_nrecs ) then seven_indx = seven_indx( 0 : count - 1 )

    if( count gt 0 ) then begin
        nrecs_for_each_sub( 7 ) = count
        CEP_HEALTH( 0 : count - 1 ).subcom7_time = hk_data( seven_indx ).tmsec
    
        CEP_HEALTH( 0 : count - 1 ).subcom7_LUTCRCs = $
               FIX( hk_data( seven_indx ).sub_sens_stat( 0 ) ) * 256 +   $
               FIX( hk_data( seven_indx ).sub_sens_stat( 1 ) )

        CEP_HEALTH( 0 : count - 1 ).subcom7_SATCRCs = $
               FIX( hk_data( seven_indx ).sub_gen_stat( 0 ) ) * 256 +   $
               FIX( hk_data( seven_indx ).sub_gen_stat( 1 ) )

     ENDIF
  

    maxval = max( nrecs_for_each_sub )
    CEP_HEALTH = CEP_HEALTH( 0 : maxval - 1 )
    zero_indx = 0
    one_indx = 0
    two_indx = 0
    three_indx = 0
    four_indx = 0
    five_indx = 0
    six_indx = 0
    seven_indx = 0
    
  RETURN
  END  

;##############################################################################
PRO FILL_CEP_COMNS, NSPINS_TIM_SPAN, NBYTS_TIM_SPAN, LEV1_CEP, TIME1, TIME2

;    This procedure fills CEPPAD common blocks depending on the LEVEL-1
;    file type for the time span requested.  


;      PARAMETER                DESCRIPTION                       TYPE

;  NSPINS_TIM_SPAN        Number of spins in time span           Longword
;  NBYTS_TIM_SPAN      Total number of bytes in time span        Longword
;  LEV1_CEP            Data access file ( associated file )      File structure
;  TIME1               User requested data start time (msec)     Longword
;  TIME2               User requested data stop time (msec)      Longword
;

;	29 April 1994	G. M. Boyd
;	04 Apr 1996	M.T.Redding	Extended variable 'PLCPSpn_hed' in 
;					COMMON block POL_CEP_SH to 13 bytes -
;					additional byte contains SCOM - CEPPAD
;					spin number (0-255).  Corrected code
;					to include IPS MUX as 12th byte of
;	`				each record.  Likewise altered
;					variable 'head' in COMMON block 
;					SPN_HEAD.  Spin header file is now
;					expected to contain records for ALL
;					spins processed instead of just spins
;					at which any of the first 7 parameters
;					are noticed to have changed.
;	3 April 1996	G. M. Boyd	Added procedure to fill the
;					Housekeeping common block(s)

;-----------------------------------------------------------------------


;           COMMON variables for the RDAF POLAR CEPPAD data analysis

; COMMON POL_EPHP,       eph
; COMMON POL_MAG,        mag
; COMMON POL_ATT,        att
; COMMON POL_SC,         plscmagaz, plscmagel


  COMMON POL_CEP_SH,      PLCPSpn_hed, PLCPSpn_hed_num
  COMMON POL_CEP_IPS,     PLCPIPS_Time, PLCPIPS_bytes, PLCPIPS_Org, $
                          PLCPIPS_Start_Time, PLCPIPS_Stop_Time
  COMMON POL_CEP_IES,     PLCPIES_time, PLCPIES_bytes, PLCPIES_Org, $
                          PLCPIES_Start_Time, PLCPIES_Stop_Time
  COMMON POL_CEP_HSTE,    PLCPHSTE_time, PLCPHSTE_bytes, PLCPHSTE_Org, $
                          PLCPHSTE_Start_Time, PLCPHSTE_Stop_Time
  COMMON POL_CEP_HSTP,    PLCPHSTP_time, PLCPHSTP_bytes, PLCPHSTP_Org, $
                          PLCPHSTP_Start_Time, PLCPHSTP_Stop_Time
  COMMON POL_CEP_IPS_RTS, PLCPIPS_Rts, PLCPIPS_Rts_num, PLCPIPS_Rts_tim, $
                          PLCPIPS_Rts_mx, $
                          PLCPIPS_Rts_Start_Time, PLCPIPS_Rts_Stop_Time
  COMMON POL_CEP_IES_RTS, PLCPIES_Rts, PLCPIES_Rts_num, PLCPIES_Rts_tim, $
                          PLCPIES_Rts_ch, $
                          PLCPIES_Rts_Start_Time, PLCPIES_Rts_Stop_Time
  COMMON POL_CEP_HIST_RTS, PLCPHIST_Rts, PLCPHIST_Rts_num, PLCPHIST_Rts_tim, $
                          PLCPHIST_Rts_id, $
                          PLCPHIST_Rts_Start_Time, PLCPHIST_Rts_Stop_Time

  

  COMMON RATES_ACC,       n_energies, nchans,  nrecs_tim_span
  COMMON FILDEF,          num_recs_file, max_rec_size, file_type, iyear, iday, $
                          begt, endt
  COMMON MUXS,            num_mux, mux_val, time_mux
  COMMON SPN_HEAD,        num_head, head

 plcpspn_hed = head
 plcpspn_hed_num = num_head

; 3 APR 1996
 if( file_type eq 1L ) then GET_CEP_HK_DATA

if( file_type eq 2L ) then begin
     FILL_NORM_COMN, nspins_tim_span, nbyts_tim_span, lev1_cep, plcpips_bytes, $
                    plcpips_time
     FILL_PLCP_ORG, plcpips_org, nspins_tim_span
     PLCPIPS_Start_Time = Time1
     PLCPIPS_Stop_Time  = Time2
endif

if( file_type eq 3L ) then begin
     FILL_NORM_COMN, nspins_tim_span, nbyts_tim_span, lev1_cep, plcpies_bytes, $
                    plcpies_time
     FILL_PLCP_ORG, plcpies_org, nspins_tim_span
     PLCPIES_Start_Time = Time1
     PLCPIES_Stop_Time  = Time2
endif

if( file_type eq 4L ) then begin
     FILL_NORM_COMN, nspins_tim_span, nbyts_tim_span, lev1_cep, plcphste_bytes,$
                    plcphste_time
     FILL_PLCP_ORG, plcphste_org, nspins_tim_span
     PLCPHSTE_Start_Time = Time1
     PLCPHSTE_Stop_Time  = Time2
endif

if( file_type eq 5L ) then begin
     FILL_NORM_COMN, nspins_tim_span, nbyts_tim_span, lev1_cep, plcphstp_bytes,$
                    plcphstp_time
     FILL_PLCP_ORG, plcphstp_org, nspins_tim_span
     PLCPHSTP_Start_Time = Time1
     PLCPHSTP_Stop_Time  = Time2
endif

if( file_type eq 6L ) then begin

   RATES_COMN_FILL, lev1_cep, file_type, plcpips_rts_tim, plcpips_rts, $
                    plcpips_rts_mx
     PLCPIPS_Rts_Start_Time = Time1
     PLCPIPS_Rts_Stop_Time  = Time2
endif

if( file_type eq 7L ) then begin

     RATES_COMN_FILL, lev1_cep, file_type, plcpies_rts_tim, plcpies_rts, $
                    plcpies_rts_ch
     PLCPIES_Rts_Start_Time = Time1
     PLCPIES_Rts_Stop_Time  = Time2
endif

if( file_type eq 8L ) then begin

     RATES_COMN_FILL, lev1_cep, file_type, plcphist_rts_tim, plcphist_rts, $
                    plcphist_rts_id
     PLCPHIST_Rts_Start_Time = Time1
     PLCPHIST_Rts_Stop_Time  = Time2
endif

return
end

;##############################################################################
PRO FILL_NORM_COMN, NSPINS_TIM_SPAN, NBYTS_TIM_SPAN, LEV1_CEP, INST_BYTS, $
                                                             TIMES

;       This procedure fills the normal mode common variable data arrays.
;         It is called by its controlling procedure FILL_CEP_COMNS.

;      PARAMETER                DESCRIPTION                       TYPE

;  NSPINS_TIM_SPAN        Number of spins in time span           Longword
;  NBYTS_TIM_SPAN      Total number of bytes in time span        Longword
;  LEV1_CEP            Data access file ( associated file )      File structure
;  INST_BYTS           The number of bytes based on number       Byte array
;                      of spins, spins per sample and total 
;                      number of sectors in an instrument 
;                      accumulation table for time span
; TIMES                Spin times in time span                   Long array


;	revision
;	29 Apr 1994	G. M. Boyd
;	14 Sep 1995	M. T. Redding	Added call to 
;					'Replace_NETWORK_LONGs_with_HOST' to 
;					convert the LONGWORD time values 
;					embedded in the byte array read from the
;					Level-1 file
;_________________________________________________________________________

 COMMON ACC_TABL,     num_energies, num_channels, spins_per_samp, $
                      sects_per_spin, total_sects_ene
 COMMON PHYS_REC_SIZ, num_byts_rec
 COMMON COMN_IND,     comn_indx, npts_ene, comn_ch_indx
 COMMON DATA_MAP,     data_rec_ind
 COMMON FILDEF,       num_recs_file, max_rec_size, file_type, iyear, iday, $
                      begt, endt
 COMMON REC_NUM,      first_rec_read, last_rec_read
 COMMON SENSOR,       inst
 COMMON SPIN_ACQ,     spin_acq_flg
 COMMON ACQ_FLAGS,    one_spin_acq, two_spin_acq, four_spin_acq, $
                      eight_spin_acq, sixtn_spin_acq 
 COMMON TIME_MAP,     time_ind


times = lonarr(nspins_tim_span)                 ; Initialize time array to be
                                                ; filled
data = bytarr(max_rec_size)                     ; Initialize LEVEL-1 logical
                                                ; record
phys_rec_on =  (first_rec_read * 2L) - 1L       ; Calculate the physical 
last_phys_rec = (last_rec_read * 2L) - 1L       ; record number for the
                                                ; time range of data extraction
indt = 0L                                       ; Time index for time span

ns = lonarr(num_energies)                       ; Number of spin counter for
                                                ; each energy 
ns(*) = -1L                                   


; Setup the index mapping of the data from level zero to common block
; Mapping code from Rob Sheldon.  Change made 24 April 1996.

 ;openw, 33, 'faster_indx_rob.dat'

 spin_acq = intarr(6,16)
 zero_spin_acq = intarr(16)*0
 spin_acq(0,0:15) = zero_spin_acq
 spin_acq(1,0:15) = one_spin_acq
 spin_acq(2,0:15) = two_spin_acq
 spin_acq(3,0:15) = four_spin_acq
 spin_acq(4,0:15) = eight_spin_acq
 spin_acq(5,0:15) = sixtn_spin_acq

 sp_aq_fl = bytarr(num_energies)
 sp_aq_fl = (	FIX(spins_per_samp ge 1) +$
		FIX(spins_per_samp ge 2) +$
		FIX(spins_per_samp ge 4) +$
		FIX(spins_per_samp ge 8) +$
		FIX(spins_per_samp ge 16))

nrecs = ( last_rec_read - first_rec_read ) + 1L

num_sectors = MAX(sects_per_spin)
nsects_ch_indx = lonarr(num_sectors,num_channels,16,num_energies)
nsects_com_indx =lonarr(num_sectors,num_channels,16,num_energies)

for ene = 0, num_energies - 1 do begin
    for nspin = 0, 15 do begin
        if( spin_acq(sp_aq_fl(ene), nspin) eq (nspin+1)) then begin
            wch_chan = where(sects_per_spin(0:num_channels -1, ene) gt 0) 
            if( wch_chan(0) ge 0 ) then begin
            nchans = n_elements( wch_chan) 
            nsects_ch_total = 0L 
            nsects_com_total = 0L 
            ns(ene) = ns(ene) + 1L 
            for ich = 0, nchans - 1 do begin 
                ichan = wch_chan( ich )
                nsects = long ( indgen(sects_per_spin(ichan, ene)))
                nsects_ch_indx(0:sects_per_spin(ichan,ene)-1,ichan,nspin,ene) =$
		nsects + (data_rec_ind(ene,nspin) + nsects_ch_total)
                nsects_com_indx(0:sects_per_spin(ichan,ene)-1,ichan,nspin,ene) =$
		ns(ene) + (nsects*npts_ene(ene)) + $
		comn_indx(ene) + nsects_com_total
;printf,33,'E1:',ns(ene),npts_ene(ene),comn_indx(ene),nsects_com_total
;printf,33,'E2:',data_rec_ind(ene,nspin),nsects_ch_total
;for ii = 0,sects_per_spin(ichan,ene)-1 do $
;printf,33,BYTE(ene),BYTE(nspin),BYTE(ichan),BYTE(nsects(ii)),':',$
;nsects_com_indx(ii,ichan,nspin,ene),$
;nsects_ch_indx(ii,ichan,nspin,ene)

                nsects_ch_total = nsects_ch_total + sects_per_spin(ichan, ene)
                nsects_com_total = comn_ch_indx(ichan, ene) 
            endfor
            endif 
        endif  
    endfor
endfor 

;close, 33

dn = intarr(num_energies)
dn(*) = ns(*) + 1
ibig = LONG(16L*num_channels*num_sectors*num_energies)
di = REFORM(nsects_ch_indx, ibig)			; Data indices
ci = REFORM(nsects_com_indx,ibig)			; Common indices
bi = LONG(where(di gt 0))
ni = REFORM((REPLICATE(1L,(16L*num_channels*num_sectors)) # dn),ibig) 
inst_byts = bytarr(LONG(n_elements(bi) * nrecs))
iid = di(bi)
iic = ci(bi)
iin = ni(bi)
;	End Mapping

while phys_rec_on le last_phys_rec do begin     ; Create the logical records 
   d = lev1_cep(phys_rec_on - 1L)               ; from physical records ending
   data(0:num_byts_rec - 1L) = d                ; with last_phys_rec
   d = lev1_cep(phys_rec_on)
   data(num_byts_rec:max_rec_size - 1L) = d
   phys_rec_on = phys_rec_on + 2L

   Replace_NETWORK_LONGs_with_HOST, 16, $       ; Convert the LONGWORD time 
      time_ind, data                             ; values from network 
                                                ; (big-endian) to HOST format
   GET_TIMES, data, times_in_rec                ; Get the times in a logical 
                                                ; record
   times(indt: indt+15) = times_in_rec          ; Fill time array for time
                                                ; range
   indt = indt + 16L                            ; Sixteen times have been 
                                                ; extracted. Increment 

   inst_byts(iic) = data(iid)
 
   iic = iic + iin

endwhile

return
end

;##############################################################################
PRO FILL_PLCP_ORG, PLCP_ORG, NSPINS_TIM_SPAN


;      PARAMETER                DESCRIPTION                       TYPE

;  PLCP_ORG            Organization/contents of the common       Long array
;                      data variables 
;                      (see more description below)
;  NSPINS_TIM_SPAN     Number of spins in time span              Longword
;  FILE_TYPE           The type of normal mode data acquisition  Longword
;                       (ex., if IPS, file_type = 2)

;	revision
;	29 Apr 1994	G. M. Boyd
;	22 Sep 1995	M. T. Redding	Corrected code to assign values in
;					PLCP_org(3,...) 'locations'
;______________________________________________________________________

 COMMON ACC_TABL,  num_energies, num_channels, spins_per_samp, sects_per_spin, $
                   total_sects_ene
 COMMON COMN_IND,  comn_indx, npts_ene, comn_ch_indx
 COMMON POL_CEP,   plcpnum_tables, plcpmax_tables
 COMMON SENSOR,    inst

; The plcp_org is a long word array whose dimension allocation follows:
; (4, num_channels, num_energies, plcpmax_tables ). This array defines the
; contents/organization of the 'plcp_bytes' array.  It is dimensioned to
; allow up to 'plcpmax_tables' ( telemetry accumulation tables over the data
; range to be plotted ).

;  If 'plcp_org' = is an IPS telemetry table (lonarr(4, 10, 17, 2)), then:
;    The first element ( 4 parameters)  are values that describe each of 
;    possible IPS pixels ( 10 channels/ 17 energy-channels) which together
;    describe the type of data and its location in the 'plcpips_bytes' array.
; The four parameters are :
;    Plcp_org(0, *, *, * ) -- Number of sectors per spin as defined in 
;                                the telemetry accumulation table
;    Plcp_org(1, *, *, * ) -- Number of spins per sample as defined in 
;                                the telemetry accumulation table
;    Plcp_org(2, *, *, * ) -- Number of values stored  ( will be a function
;                                of the number of spins of data to which the 
;                                the current table is applied  AND the data
;                                accumulation rate ( the number of spins over
;                                which the data accumulated )
;    Plcp_org(3, *, *, * ) -- Index location of first value in 
;                                'plcpips_bytes' array ( the value from the
;                                first sector and first spin of data to which
;                                the telemetry accumulation table applies


   plcp_org = lonarr(4, num_channels, num_energies, plcpnum_tables+1)
   for ene = 0, num_energies - 1 do begin
      
      loc_spn0_sect0 = comn_indx(ene)        ; Location of the first sector 
                                             ; and first spin for the 
                                             ; first pixel of each energy
      for ich = 0, num_channels - 1 do begin
          plcp_org(0, ich, ene, plcpnum_tables) = sects_per_spin(ich, ene) 
          plcp_org(1, ich, ene, plcpnum_tables) = spins_per_samp(ene)
          plcp_org(2, ich, ene, plcpnum_tables) = npts_ene(ene) ; The number of
                                             ; values stored ( the number
                                             ; of spins of data and the number
                                             ; of spins  over which  the data 
                                             ; were accumulated )
          IF ( ich EQ 0) THEN BEGIN
             plcp_org(3, ich, ene, plcpnum_tables) = $ ; starting location of
                 comn_indx(ene) 		; sector zero data for this 
						; channel and energy
          ENDIF ELSE BEGIN
             plcp_org(3, ich, ene, plcpnum_tables) = $ ; starting location of
                 comn_indx(ene) + $		; sector zero data for this 
                 comn_ch_indx(ich-1,ene)	; channel and energy
          ENDELSE ;
              
;          loc_spn0_sect0 =  loc_spn0_sect0  +         $
;                         ( sects_per_spin(ich, ene) * ( nspins_tim_span/ $
;                           spins_per_samp(ene) ) ) 
      endfor

   endfor
        
return
end

;#############################################################################
FUNCTION FLOATING_PT, INTEGER_VALUES

;       INTEGER_VALUES =   An array of 16bit words ( integers )
;			( NOTE:  'INTEGER_VALUE' can be a scalar'  

;  This function converts 16bit ( or 2 bytes ) to a floating point value 
;	according to the following :
;     
;		 BITS	
;		15  -  11  	Signed exponent
;				Bit 15 sign bit ( 1 negative , 0 = positive )
;		10  -   0	10: sign bit    ( 1 negative , 0 = positive )
;                                9:bit weight = .5
;                                8:bit weight = .25
;				 etc.
;	G. M. Boyd					8 May 1996
;___________________________________________________________________________
;FUNCTION FLOATING_PT, INTEGER_VALUES


possible_exp_values = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, $
                       -16, -15, -14, -13, -12, -11, -10, -9, -8, -7, -6, -5,$
                        -4,  -3, -2, -1]

iexp_value = ISHFT( integer_values, -11 ) AND 31 
exp_value = DOUBLE( possible_exp_values( iexp_value ) )
mantissa_sign = -2.d0 * ( ISHFT( integer_values, -10 ) AND 1 )  + 1.d0
mantissa_val  = DOUBLE( integer_values AND 1023 ) / 1024.d0
                
                
RETURN, FLOAT( mantissa_sign * mantissa_val ) * 2.d0^ exp_value
                
END

;#############################################################################
PRO GET_CEP_HK_DATA

;       GET_CEP_HK_DATA			Extracts data from the LEVEL-1 
;					HOUSEKEEPING file and fill defined
;					common blocks
;
;	G. M. Boyd			1 June 1996

;____________________________________________________________________________


COMMON CEP_HK_EXTRACTION,       ten_mjrfrm_milsecs, num_recs_extracted, hk_data

;            Tag names for the original raw data set, HK_DATA structure
; HK_DATA = {	tmsec: 0L,                  tmicro: 0L,			$
;		sw_status: BYTARR( 2 ),     kp_hk_subcom: 0b,		$
;		hw_status1: 0b,             mjf_seq_num: 0b,        	$
;		hw_status2: 0b,             keyprms: BYTARR( 8 ),     	$ 
;		sub_sens_stat: BYTARR( 2 ), sub_gen_stat: BYTARR( 2 ),	$ 
;		sc_an0: 0b,           	    analog_1: 0b,		$
;		sc_an1: 0b,           	    analog_2: 0b,		$
;		sc_an2: 0b,           	    analog_3: 0b,		$
;		sc_an3: 0b,           	    analog_4: 0b,		$
;		sc_bstat: 0b,          	    CMDEcho_1: 0b,		$
;		sc_th0: 0b,          	    CMDEcho_2: 0b,		$
;		sc_th1: 0b,          	    CMDEcho_3: 0b,		$
;		sc_th2: 0b,          	    CMDEcho_4: 0b,		$
;		sc_th3: 0b,          	    CMDEcho_5: 0b,		$
;		CheckSum: BYTARR( 2 ),      CEP_sub: BYTARR( 160 ),	$
;		KeyParam_time: 0L,          HIST_s_e_time: LONARR( 8 )  }
;


;   Tag names for key parameter structure----CEP_KP

;    CEP_KP  = {   	time: 0L,				$
;				az:   0.0,			$
;                                magel: 0.0,			$
;                                magbx: 0.0,			$
;                                magby: 0.0,			$
;                                magbz: 0.0,			$
;                                ips_err: 0.0,			$
;                                ies_err: 0.0,			$
;                                ips_coef: FLTARR( 4 ),		$
;                                ies_coef: FLTARR( 4 ),		$
;                                cubic_cntrls: 0,		$
;                                ips_sper: FLTARR( 5 ),		$
;                                ips_sper_drift: 0.0,		$  
;                                ips_sper_angle: 0.0, 		$
;                                ips_sper_rms:   0.0,		$
;                                ies_sper: FLTARR( 5 ),		$
;                                ies_sper_drift: 0.0,		$  
;                                ies_sper_angle: 0.0, 		$
;                                ies_sper_rms:   0.0,		$
;                                ips_sun_offset: 0.0,		$
;                                ies_sun_offset: 0.0                }
;
 

;   Tag names for HIST singles structure----HIST_S
; Hist single tag names 
;      HIST_S = { $
;      time:    0L,     $
;      sector:  0b,     $
;      spin:    0b,     $
;      data:    BYTARR( 16 )  }


;   Tag names for HIST events structure----HIST_E
;      HIST_E = { $
;      time:    0L,     $
;      sector:  0b,     $
;      spin:    0b,     $
;      data:    BYTARR( 16 )  }


;      Tag names CRC table structure------TABLE_CRC
;     Table_CRC = { $
;                Flt_SW:   0,	        $	; Flight software
;                MST:      INTARR( 16), $	; Mode Switching Table
;                SECT_A:   INTARR( 16), $	; Sector & Accumulation
;                IPS_CNF:  INTARR( 16), $	; IPS Configuration
;                IESLUT:   INTARR( 12), $	; IES Lookup Table
;                HISTLUT:  INTARR(  4), $	; HIST Lookup Table
;                TIME:     0L     }		; Main frame time
 

;   Tag names for IPS calibration parameter structure----IPSCAL
;          IPSCal = { time:       0L  	$    ; Mainfrm time in millisecs
;                     Maskreq:    0,   	$    ; IPS cal mask requested by command
;                     Maskrem:    0,    $    ; IPS cal mask remaining to execute
;                     CurrentCal: 0b,   $    ; IPS cal state currently executing
;                     C1:         BYTARR(9),  B1:         BYTARR(9), $
;                     T1:         BYTARR(9),  C2:         BYTARR(9), $
;                     C5:         BYTARR(9),  C3:         BYTARR(9), $
;                     C0:         BYTARR(9),  B5:         BYTARR(9), $
;		      B3:         BYTARR(9),  B2:         BYTARR(9), $
;                     B0:         BYTARR(9),  T5:         BYTARR(9), $
;                     T3:         BYTARR(9),  T2:         BYTARR(9), $
;                     T0:         BYTARR(9)  }
;   NOTE:  Using Channel C1 as an example, the row value of 9 represents the
;   following for each channel:
;           0 =  low  D/A for cal D/A = low res
;           1 =  high D/A for cal D/A = low res
;           2 =  cal  D/A for cal D/A = low res
;           3 =  low  D/A for cal D/A = high res
;           4 =  high D/A for cal D/A = high res
;           5 =  cal  D/A for cal D/A = high res
;           6 =  high D/A for cal = bin 14 top
;           7 =  cal  D/A for cal = bin 14 top
;           8 =  cal  D/A for cal = bin 4 top


;   Tag names for table address parameter structure----TABLE_ADDRESS
;       TABLE_ADDRESS = { Time: 0L
;	SpinBuf0_page: 0, 	         SpinBuf0_offset: 0,		$
;	SpinBuf0_len: 0,   	         ModeSwitching_page: 0,		$
;	ModeSwitching_offset: 0,         ModeSwitching_len: 0,		$
;	ModeSwitch_GenBase_page: 0,      ModeSwitch_GenBase_offset: 0,	$
;	SectAccumt0_page: 0,   	         SectAccumt0_offset: 0,		$
;	SectAccumt_len: 0,               IPS_configuration_tab0_page: 0,$
;       IPS_configuration_tab0_offset: 0,IPS_configuration_tab_len: 0,	$
;       IES_lookup_tab0_page: 0,         IES_lookup_tab0_offset: 0,	$
;       IES_lookup_Tab_len: 0,		 IES_lookup_GenBase_page: 0,    $
;	IES_lookup_GenBase_page: 0,      IES_lookup_GenBase_offset: 0,	$
;	HIST_lookup_tab0_page: 0,        HIST_lookup_tab0_offset: 0,	$
;	HIST_lookup_Tab_len: 0,          HIST_lookup_GenBase_offset: 0, $
;	Analogs_page:  0,                Analogs_offset:  0, 		$
;	InputCmdStruct_page:  0,         InputCmdStruct_offset:  0,	$
;       CmdEchoStruct_page:  0,          CmdEchoStruct_offset: 0          }
  

;   Tag names for DPU memory dump structure----DPU_MEMORY_DMP 
;         DPU_MEMORY_DMP = { Time: 0L,                      $
;                            data_segment_address: 0,       $
;                            data_offset_address: 0,        $
;                            data_dump: BYTARR( 128 )       }
  

;   Tag names for health subcommed parameter structure----CEP_HEALTH 
;    CEP_HEALTH =  {subcom0_time: 0L,         subcom0_HistCmd: 0,	$    
;		    subcom0_PCmdCntr: 0,      subcom0_HVLVV: 0b,	$
;                   subcom0_HVLVi: 0b,	      subcom0_DPUv: 0b,		$
;		    subcom0_DPUi: 0b,       				$
;                   subcom1_time: 0L,	      subcom1_HistCmdErrCnt: 0,	$
;		    subcom1_SCmdCntr: 0,      subcom1_IPSP30V: 0b,	$
;                   subcom1_IPSP7i: 0b,       subcom1_IPSP7v: 0b,	$
;		    subcom1_IPSM4v: 0b, 				$
;                   subcom2_time: 0L,         subcom2_IESCmd: 0,        $
;		    subcom2_SCmdErrCnt: 0,    subcom2_IPSM4i: 0b,	$
;                   subcom3_time: 0L,	      subcom3_IESCmdErrCnt: 0,	$
;		    subcom3_MemLoadCnt: 0,                             	$
;                   subcom4_time: 0L,         subcom4_IESZero: 0,      	$
;                   subcom4_MemLoadErrCnt: 0, subcom4_HeadTemp1: 0b,	$
;                   subcom4_HeadTemp2: 0b,    subcom4_HeadTemp3: 0b,	$
;      		    subcom4_IPSTemp: 0b,     				$
;                   subcom5_time: 0L,         subcom5_IPSZero: 0,      	$
;  		    subcom5_CodeVersion: 0,   subcom5_HIST1: 0b,	$
;                   subcom5_HIST2: 0b,	      subcom5_HIST3: 0b,	$
;                   subcom5_HIST4: 0b,					$
;                   subcom6_time: 0L,         subcom6_ICTCRCs: 0,	$ 
;		    subcom6_MSTCRCs: 0,       subcom6_IES1: 0b,		$
;                   subcom6_IES2: 0b,	      subcom6_IES3: 0b,		$
;                   subcom6_IES4: 0b,					$
;                   subcom7_time: 0L,	      subcom7_LUTCRCs: 0,	$
;                   subcom7_SATCRCs: 0                                  }	
;


;   Tag names for health non-subcommed parameter structure----NOSUB_CEP_HEALTH
; NOSUB_CEP_HEALTH =  {    time, 0L,					$
;		sc_an0: 0b,           	    				$
;		sc_an1: 0b,           	    				$
;		sc_an2: 0b,           	    				$
;		sc_an3: 0b,           	    				$
;		sc_bstat: 0b,          	    				$
;		sc_th0: 0b,          	    				$
;		sc_th1: 0b,          	    				$
;		sc_th2: 0b,          	    				$
;		sc_th3: 0b,          	    				$
;		CMDEcho_1: 0b,						$
;		CMDEcho_2: 0b,						$
;		CMDEcho_3: 0b,						$
;		CMDEcho_4: 0b,						$
; 		CMDEcho_5: 0b,          				$
;		hw_status1: 0b,             hw_status2: 0b,        	$
;		sw_status: 0,               CheckSum: 0			}
;

 
 ; Extract CEPPAD  housekeeping data
   
   EXTRACT_KEY_PARAMS
   UNPACK_SUBCOM_DATA
   EXTRACT_SUBCOM_HEALTH
   EXTRACT_NOSUB_HEALTH
   hk_data = 0				; release space

    RETURN
 END

;##############################################################################
PRO GET_HR_DEFS, IUNIT, FILENAME_LEVEL1, HEAD_REC

;     This procedure retrieves the LEVEL-1 File Definition parameters from the
;     LEVEL-1 header record (HR ) which is the first logical record in the 
;     LEVEL-1 file. If the LEVEL-1 file type has normal mode data, the
;     Telemetry Accumulation Definitions are also retrieved. Common variables
;     in 'FILDEF' and  'TMADEF' are filled. 



;      PARAMETER                DESCRIPTION                       TYPE

;  IUNIT               LEVEL-1 file access unit                  Integer
;  HEAD_REC            Array of longwords which contain          Long array
;                      File and Telemetry definitions
;                      ( Read from 1st physical record)

;_________________________________________________G. M. Boyd 29 April 1994

;    The LEVEL-1 File Definition parameters are the first seven words in the HR.
;    The LEVEL-1 Telemetry Accumulation parameters are words 8 and 9.
;    ** NOTE:
;            WORDs 8 and 9 are acqusition table numbers.  If the file_type
;            is ge 2 and le 5  ( Normal mode telemetry ),  WORDs 8 and 9 will 
;            have table number values.  WORD 8 is a 'Normal' mode telemetry type
;            representing the DPU table number which is a number between 
;            0 and 15. Because WORD 8 may refers to a table whose contents
;            could change over a period of time, WORD 9 is a unique calculated
;            value based on the table contents. 

;      ****  All seven or nine WORDS are of type LONG ****

;       WORD            DEFINITION                    PARAMETER NAME
;        1 = Num of recs file (includes this word)     num_recs_file 
;        2 = Data record length in bytes               max_rec_size 
;        3 = File type (see description below)         file_type
;        4 = Year   (4 digit year)                     iyear
;        5 = Day of year                               iday
;        6 = Beginning time of data (msec)             begt
;        7 = Ending time of data    (msec)             endt
;      **8 = telemetry table number                    tm_tabl_num
;      **9 = telemetry table CRC value                 tm_tabl_crc

;   The first seven words are in the COMMON block named  FILDEF.
;             WORDs 8 and 9 are in COMMON block TMADEF.


;	File type	File contents
;  ( Parameter value )      
;	1L		Housekeeping file
;	2L 		IPS Normal mode telemetry
;	3L 		IES Normal mode telemetry
;	4L 		HISTE Normal mode telemetry
;	5L 		HISTP Normal mode telemetry
;	6L		IPS Rates mode telemetry
;	7L		IES Rates mode telemetry
;	8L		HIST Rates mode telemetry
;	9L		Spin header file information

 COMMON FILDEF, num_recs_file, max_rec_size, file_type, iyear, iday, begt, endt
 COMMON TMADEF, tm_tabl_num, tm_tabl_crc

      num_recs_file = head_rec( 0 )
      max_rec_size  = head_rec( 1 )
      file_type     = head_rec( 2 )
      iyear         = head_rec( 3 )
      iday          = head_rec( 4 )
      begt          = head_rec( 5 )
      endt          = head_rec( 6 )
                                                           
      if( file_type ge 2) and (file_type le 5 )  then begin
         tm_tabl_num = head_rec(7)      ; File_type equal to  ( normal mode )
         tm_tabl_crc = head_rec(8)      

      endif else begin
         tm_tabl_num = -1L
         tm_tabl_crc = -1L
      endelse

return
end

;##############################################################################
PRO GET_NEW_TIM_RNG, LEV1_CEP, TIME1, TIME2, BEG_TIME, END_TIME, ERROR

;   This procedure finds the logical record ( 16 spin time ) that has 'TIME1' 
;   and sets 'BEG_TIME' to the first time in that record. It finds the logical 
;    record that has 'TIME2' and sets 'END_TIME' to the last time in that 
;    record. 'BEG_TIME' and 'END_TIME' will be the time range for the data 
;    extraction. The common block, 'REC_NUM', will have the number of the 
;    first logical record (FIRST_REC_READ) to read and the number of the
;    last record (LAST_REC_READ) to read.

;      PARAMETER                DESCRIPTION                       TYPE

;  LEV1_CEP                Data file access structure           File structure
;  TIME1                   Start time inputted by user          Longword
;  TIME2                   End time inputted by user            Longword
;  BEG_TIME                Start time for data extraction       Longword
;  END_TIME                End time for data extraction         Longword
;  ERROR                   Status of call 0  -- OK		Longword
;                                         -1 -- no range found

;	revision
;	29 Apr 1994	G. M. Boyd
;	14 Sep 1995	M. T. Redding	Added call to
;					'Replace_NETWORK_LONGs_with_HOST'
;					following read of L1 record into dat.
;					Added common block TIME_MAP
;__________________________________________________________________________

 COMMON FILDEF,       num_recs_file, max_rec_size, file_type, iyear, iday, $
                      begt, endt
 COMMON PHYS_REC_SIZ, num_byts_rec
 COMMON REC_NUM,      first_rec_read, last_rec_read
 COMMON TIME_MAP,     time_ind
 

; 4 March 1996
;  This  procedure and the corresponding references were changed 
;  to Estimate_L1_Times in order not to conflict with the "ESTIMATE_TIMES"
;  procedure used by the "Make_level_1" procedure.  The conflict would have 
;  come about if the "Make_level_1" procedure and the "Get_Cep_Data" procedure
;  were compiled in the same session.

; A logical rec is a 16 spin time data set.
; It is equal to two physical recs.
; 4 March 1996
 dat = bytarr(max_rec_size)

 need_new_first_rec = 1		; initialize to requiring new time and record
				; for start of time range
 need_new_last_rec  = 1		; initialize to requiring new time and record
				; for end of time range
 min_data_rec_in_file = 2L		; First logical data record in file
 max_data_rec_in_file = num_recs_file	; Last logical data record in file
 error = 0

;	Check for desired range completely preceding data range

 IF ( time2 LT begt ) THEN BEGIN
    need_new_first_rec = 0
    need_new_last_rec  = 0
    error = -1			; Set status to indicate there is no data available
 ENDIF

;	Check for desired range completely following data range

 IF ( endt LT time1 ) THEN BEGIN
    need_new_first_rec = 0
    need_new_last_rec  = 0
    error = -1			; Set status to indicate there is no data available
 ENDIF

;	Check for desired range completely within data range

 IF ( begt LE time1 ) AND ( time2 LE endt ) THEN BEGIN
    need_new_first_rec = 1
    need_new_last_rec  = 1
 ENDIF

;	Check for starting time within but ending time following data range

 IF ( ( begt LE time1 ) AND ( time1 LE endt ) ) AND ( endt LT time2 ) THEN BEGIN
    need_new_first_rec = 1	; Set up to search for the record containing the first time
    need_new_last_rec  = 0	; Set up to skip the search for the record containing the last time
    end_time = endt		; Return the last time in the file as the end range-time
    last_rec_read = max_data_rec_in_file
 ENDIF


;	Check for starting time preceding but ending time within data range

 IF ( time1 LT begt ) AND ( ( begt LE time2 ) AND ( time2 LE endt ) ) THEN BEGIN
    need_new_first_rec = 0	; Set up to skip the search for the record containing the first time
    need_new_last_rec  = 1	; Set up to search for the record containing the last time
    beg_time = begt		; Return the first time in the file as the start range-time
    first_rec_read = min_data_rec_in_file
 ENDIF


;	Check for starting time preceding and ending time following data range

 IF ( time1 LT begt ) AND ( endt LE time2 ) THEN BEGIN
    need_new_first_rec = 0	; Set up to skip the search for the record containing the first time
    need_new_last_rec  = 0	; Set up to skip the search for the record containing the last time
    beg_time = begt		; Return the first time in the file as the start range-time
    end_time = endt		; Return the last time in the file as the end range-time
    first_rec_read = min_data_rec_in_file
    last_rec_read  = max_data_rec_in_file
 ENDIF


;	Do search for record containing the first time in the time range

 IF ( need_new_first_rec ) THEN BEGIN

;	Do a binary search for 'time1' and put the logical record number
;	in which 'time1' was found in variable 'record_index'

    EST_REC_NUMBER, lev1_cep, time1, record_index, ierror
 
    IF ( ierror GE 0 ) THEN BEGIN

;	Read the indicated record and use the first time as the beginning
;	time-range value

       ;dat = bytarr(max_rec_size)		4 March 1996
       phys_rec1 = (record_index * 2L) - 1L	; Time1 was found in logical rec 'record_index' 
       phys_rec2 =  phys_rec1 + 1L             
       d = lev1_cep( phys_rec1 - 1L)		; Get 1st half of logical record
       dat(0:num_byts_rec - 1L) = d                 
       d = lev1_cep( phys_rec2 - 1L )		; Get 2nd half of logical record
       dat(num_byts_rec:max_rec_size - 1L) = d                  

       Replace_NETWORK_LONGs_with_HOST, 16, $   ; Convert the LONGWORD time 
          time_ind, dat                         ; values from network 
                                                ; (big-endian) to HOST format

;	Find the times in a logical record ( max of 16 times )

       GET_TIMES, dat, times_in_rec
    times_in_rec = ABS( times_in_rec )		; 29 may 1996

;	Fill in any 'fill' (negative) or 'missing' (zero) time values

       ;ESTIMATE_TIMES, times_in_rec, est_times_in_rec, ierror
       ESTIMATE_L1_TIMES, times_in_rec, est_times_in_rec, ierror

       IF ( ierror EQ 1 ) THEN BEGIN	; All times were less than zero
          print, ' '
          print, ' THERE WERE NO POSITIVE TIMES IN THIS LOGICAL RECORD '
          print, ' TRYING TO FIND THE FIRST TIME, "END_TIME" '
          print, ' STOPPED IN PROCEDURE GET_NEW_TIM_RNG'
        ;create crash so that calling
        ;routine's error checker gets it and processing continues!
        xxx = yyy1
          STOP
       ENDIF

       beg_time = est_times_in_rec(0)	; Set 'beg_time' to 1st of the 16 times
       first_rec_read = record_index                   

    ENDIF ELSE BEGIN	; ( ierror GE 0 )	; Time not found - return error status
       error = -1
       RETURN
    ENDELSE	; ( ierror GE 0 )

 ENDIF	; ( need_new_first_rec )


;	Do search for record containing the first time in the time range

 IF ( need_new_last_rec ) THEN BEGIN

;	Do a binary search for 'time2' and put the logical record number
;	in which 'time1' was found in variable 'record_index'

    EST_REC_NUMBER, lev1_cep, time2, record_index, ierror
 
    IF ( ierror GE 0 ) THEN BEGIN

;	Read the indicated record and use the last time as the ending
;	time-range value

       phys_rec1 = ( record_index * 2L ) - 1L	; Time2 was found in logical rec 'record_index' 
       phys_rec2 =  phys_rec1 + 1L            
       d = lev1_cep( phys_rec1 - 1L)		; Get 1st half of logical record
       dat(0:num_byts_rec - 1L) = d
       d = lev1_cep( phys_rec2 - 1L)		; Get 2nd half of logical record
       dat(num_byts_rec:max_rec_size - 1L) = d

       Replace_NETWORK_LONGs_with_HOST, 16, $   ; Convert the LONGWORD time 
          time_ind, dat                         ; values from network 
                                                ; (big-endian) to HOST format

;	Find the times in a logical record ( max of 16 times )

       GET_TIMES, dat, times_in_rec           ; Find the times in a logical record
                                           ;       ( max of 16 times )
    times_in_rec = ABS( times_in_rec )		; 29 may 1996
;	Fill in any 'fill' (negative) or 'missing' (zero) time values

       ;ESTIMATE_TIMES, times_in_rec, $       ; Interpolate/extrapolate when any of
       ESTIMATE_L1_TIMES, times_in_rec, $     ; Interpolate/extrapolate when any of
          est_times_in_rec, ierror            ; the 16 times are negative
       if ( ierror eq 1 ) then begin          ; All times were less than zero
          print, ' '
          print, ' THERE WERE NO POSITIVE TIMES IN THIS LOGICAL RECORD '
          print, ' TRYING TO FIND THE LAST TIME, "END_TIME" '
          print, ' STOPPED IN PROCEDURE GET_NEW_TIM_RNG '
        ;create crash so that calling
        ;routine's error checker gets it and processing continues!
        xxx = yyy1
          stop
       endif

; 4 MARCH  1996
       ;end_time = est_times_in_rec( 15 ) 
       ; Set 'end_time' to last positive time of estimated 16 times
       last_positive_tindx = WHERE( ( est_times_in_rec gt 0 ), count ) 
       end_time = est_times_in_rec( last_positive_tindx( count - 1 ) ) 
       last_rec_read = record_index                    

    ENDIF ELSE BEGIN	; ( ierror GE 0 )	; Time not found - return error status
       error = -1
       RETURN
    ENDELSE	; ( ierror GE 0 )

 ENDIF	; ( need_new_last_rec ) THEN BEGIN

; RELEASE space
 tmparray = BYTARR( 2 )
 tmparray( * ) = 0b
 dat = TEMPORARY( tmparray ) 	; Tmparray becomes undefined--dat a bytarr of 2
 d = TEMPORARY( dat )          	; Dat becomes undefined--d a byte array of 2
 tmparray = 0
 dat = 0

 return
 end

;##############################################################################
PRO GET_SH_INFO, BEG_TIME, END_TIME, FILENAME_LEVEL1		

;       This routine opens and reads CEPPAD Spin_Header (SH) file
;       and extracts the SH information that fills common blocks
;       MUX and SPN_HEAD for the time range defined by BEG_TIME
;       and END_TIME

;      PARAMETER                DESCRIPTION                       TYPE

;  BEG_TIME                Start time for data extraction       Longword
;  END_TIME                End time for data extraction         Longword

;	revision
;	29 Apr 1994	G. M. Boyd
;	14 Sep 1995	M. T. Redding	Added calls to 'BYTEORDER' to convert
;					the Spin Header LONGWORD heading record
;					values from NETWORK to HOST format.
;					Added call to
;					'Replace_NETWORK_LONGs_with_HOST' and
;					'Replace_NETWORK_SHORTs_with_HOST' to
;					convert the time (LONGWORDS) and
;					HIST command state (SHORTWORD) elements
;					of each data record
;	21 Sep 1995	M. T. Redding	Correction - Repositioned comparison
;					of time_hed to precede the transfer of
;					record data into the array head_tmp - 
;					the code was accepting all records
;					regardless of whether the time on the
;					record was in range or not
;	22 Sep 1995	M. T. Redding	Corrected assignment of time_MUX_tmp
;					and MUX_val_tmp for case of num_MUX lt 0
;					- values are taken from 'head_tmp'
;					rather than from 'head'.
;					Added Spin header file-close.
;	04 Apr 1996	M. T. Redding	Rewrote extraction logic.
;					Expanded variable 'head' to 13 bytes
;					it now contains 'SCOM' - CEPPAD spin #.
;					MUX values returned include all stored
;					values within the specified time range
;					plus values just outside (if the
;					specifed time range do not exactly 
;					match the stored times 
;	04 Apr 1996	M.T.Redding	Extended variable 'PLCPSpn_hed' in 
;					COMMON block POL_CEP_SH to 13 bytes -
;					additional byte contains SCOM - CEPPAD
;					spin number (0-255).  Corrected code
;					to include IPS MUX as 12th byte of
;					each record.  Likewise altered
;					variable 'head' in COMMON block 
;					SPN_HEAD.  Spin header file is now
;					expected to contain records for ALL
;					spins processed instead of just spins
;					at which any of the first 7 parameters
;					are noticed to have changed.
;	03 Apr 1996      G. M. Boyd	The way that the name for the   
;					the Spin Header (SH) Level 1 file is
;					created was changed.
;
;       05 Jun 1996	G. M. Boyd      Changed the reading of spin
;					header such that it is done faster   
;	06 Jun 1996	M. T. Redding	Added code to construct Spin header
;					file name using Rob Sheldon's naming
;					convention if the input L1 file uses it.
;__________________________________________________________________________

 COMMON FILDEF,    num_recs_file, max_rec_size, file_type, iyear, iday, $
                   begt, endt
 COMMON MUXS,      num_MUX, MUX_val, time_MUX
 COMMON SPN_HEAD,  num_head, head

;__________________________________________________________________________


    ; 3 APRIL 1996
    ; Create the SH level 1 file name from the level1 data file
    ; This change allows for the filename to have the same directory path
    ; as the user specified level1 data file.  
    find_string_aer = ' '
    find_string_rbs = ' '
    if ( file_type eq 1L ) then begin
       find_string_aer = '_HK_'
       find_string_rbs = '.HK'
    endif
    if ( file_type eq 2L ) then begin
       find_string_aer = '_IPS_'
       find_string_rbs = '.IPS'
    endif
    if ( file_type eq 3L ) then begin
       find_string_aer = '_IES_'
       find_string_rbs = '.IES'
    endif
    if ( file_type eq 4L ) then begin
       find_string_aer = '_HISTE_'
       find_string_rbs = '.HSe'
    endif
    if ( file_type eq 5L ) then begin
       find_string_aer = '_HISTP_'
       find_string_rbs = '.HSp'
    endif
    if ( file_type eq 6L ) then begin
       find_string_aer = '_IPS_RATES_'
       find_string_rbs = '.PRA'
    endif
    if ( file_type eq 7L ) then begin
       find_string_aer = '_IES_RATES_'
       find_string_rbs = '.ERA'
    endif
    if ( file_type eq 8L ) then begin
       find_string_aer = '_HIST_RATES_'
       find_string_rbs = '.HRA'
    endif
    if( ( find_string_aer ne ' ' ) or ( find_string_rbs ne ' ' ) ) then begin
        pos = STRPOS( STRUPCASE( filename_level1 ), find_string_aer )  
        if( pos ne -1 ) then begin
            len_level1 = STRLEN( filename_level1 )
            len_string = STRLEN( find_string_aer )
            filename_head = STRMID( filename_level1, 0 , pos ) + '_SH_' +   $
                            STRMID( filename_level1, pos + len_string, 7 ) + $
                            '.DAT'
        endif else begin
           pos = STRPOS( STRUPCASE( filename_level1 ), $
                         STRUPCASE(find_string_rbs ) ) 
           if( pos ne -1 ) then begin
               len_level1 = STRLEN( filename_level1 )
               len_string = STRLEN( find_string_rbs )
               filename_head = STRMID( filename_level1, 0 , pos ) + '.SH'
           endif
        endelse
    endif

;	OPEN spin header file 

   OPEN_CEP_LEVEL1, filename_head, iunit_h, ierror_0
   IF ( ierror_0 NE 0) THEN BEGIN
        PRINT, ' '
        PRINT, ' ERROR OPENING FILE ', filename_head
        PRINT, ' STOPPED IN PROCEDURE GET_SH_INFO '
        ;create crash so that calling
        ;routine's error checker gets it and processing continues!
        xxx = yyy1
        STOP
   ENDIF ELSE BEGIN
;        PRINT, ' '
;        PRINT, ' ////// OPENED ',  filename_head
   ENDELSE        



;	READ beginning records of spin header file for file type, number of
;	records stored, and number of bytes in a record.

   head = BYTARR( 12 )
   readu, iunit_h, head

   num_recs_hed	 = LONG( head, 0 )	; Get long word from 4 bytes
					; starting with byte, 0
   nbyts_rec_hed = LONG( head, 4 )	; Get long word from 4 bytes
					; starting with byte, 4
   fil_typ_hed   = LONG( head, 8 )      ; Get long word from 4 bytes 
					; starting with byte, 8

   BYTEORDER, fil_typ_hed, $            ; Convert fil_typ_hed to host format
              nbyts_rec_hed, $          ; Convert nbyts_rec_hed to host format
              num_recs_hed,  $		; Convert num_recs_hed to host format
              /NTOHL	 

;	Check for valid file type

   IF ( fil_typ_hed(0) NE 9L ) THEN BEGIN
      PRINT, ' '
      PRINT, ' THIS IS NOT A SPIN_HEADER FILE '
      PRINT, ' STOPPED IN PROCEDURE GET_SH_INFO'
      ;create crash so that calling
      ;routine's error checker gets it and processing continues!
      xxx = yyy1
      STOP
   ENDIF 
  
   CLOSE, iunit_h			; Close after a sequential read
   FREE_LUN, iunit_h


   OPEN_CEP_LEVEL1, filename_head, iunit_h, ierror_0
   IF ( ierror_0 NE 0) THEN BEGIN
        PRINT, ' '
        PRINT, ' ERROR OPENING FILE ', filename_head
        PRINT, ' STOPPED IN PROCEDURE GET_SH_INFO '
        ;create crash so that calling
        ;routine's error checker gets it and processing continues!
        xxx = yyy1
        STOP
   ENDIF 


;	Define array for direct access of SH file

   header = ASSOC(iunit_h, BYTARR( nbyts_rec_hed, num_recs_hed ) )

   head = header( 0 )
   ;print, ' 1 size ', size( head )
   dummy = head( *, 1 )
   year_in_hed  = LONG( dummy, 0 )	; Get long word from 4 bytes
						; starting with byte, 0
   day_in_hed   = LONG( dummy, 4 )	; Get long word from 4 bytes
						; starting with byte, 4
   spn_btim_hed = LONG( dummy, 8 )	; Beginning time in file
   dummy = head( * , 2 )
   spn_etim_hed = LONG( dummy, 0 )	; Ending time in file


;	Convert remaining variables from NETWORK to HOST format

   BYTEORDER, year_in_hed,	$
              day_in_hed,	$
              spn_btim_hed,	$
              spn_etim_hed, /NTOHL
 

;	Check for year and day match with those in L1 file

   IF ( (year_in_hed NE iyear) OR (day_in_hed NE iday) ) THEN BEGIN
       PRINT, ' '
       PRINT, 'THE SPIN HEADER FILE HAS A DIFFERENT DATE FROM THE LEVEL-1 FILE'
       PRINT, ' THE LEVEL-1 FILE DAY AND YEAR  = ', iday, iyear
       PRINT, ' THE SPIN HEADER FILE DAY AND YEAR  = ', day_in_hed, year_in_hed
       PRINT, ' STOPPED IN PROCEDURE GET_SH_INFO '
       ;create crash so that calling
       ;routine's error checker gets it and processing continues!
       xxx = yyy1
       STOP
   ENDIF

   header = $
   ASSOC(iunit_h, BYTARR( nbyts_rec_hed, num_recs_hed - 3 ), 3 * nbyts_rec_hed )

;	Initialize

   mux_loc      = 11    ; Location of mux byte in spin header record
   num_sh_data_recs  = num_recs_hed - 3

   head = header ( 0 )


;	Define constants and array locations for converting LONG and SHORT
;	word spin header values from NETWORK to HOST format

   SHORT_loc        = $		; starting location of bytes containing SHORT
                      ( LINDGEN( num_sh_data_recs  ) * nbyts_rec_hed ) + 9L
   number_of_SHORTs = num_sh_data_recs 
   LONG_loc         = $		; starting location of bytes containing LONG
                        LINDGEN( num_sh_data_recs  ) * nbyts_rec_hed 
   number_of_LONGs  =  num_sh_data_recs  

;	Convert LONGWORD time and SHORTWORD HIST command state variables
;	from NETWORK to HOST format
      head = REFORM( head, nbyts_rec_hed * num_sh_data_recs   )


      Replace_NETWORK_LONGs_with_HOST, number_of_LONGs, LONG_loc, head
      Replace_NETWORK_SHORTs_with_HOST, number_of_SHORTs, SHORT_loc, head

      head = REFORM( head, nbyts_rec_hed , num_sh_data_recs  )

      MUX_val  = BYTARR( num_sh_data_recs  )
      time_MUX = LONARR( num_sh_data_recs  )
      time_MUX = LONG( head( 0 : 3, * ), 0, num_sh_data_recs  )
      MUX_val  = head( mux_loc, * )

;	Extract those values within and just outside the specified time range

   Before_Time_range = WHERE( ( time_MUX LT beg_time ), before_count )
   After_Time_range  = WHERE( ( time_MUX GT end_time ), after_count )

   IF ( before_count GT 0 ) THEN $
      start_index = Before_time_range( before_count - 1 ) ELSE $
      start_index = 0

   IF ( after_count GT 0 ) THEN $
      stop_index = After_time_range( 0 ) ELSE $
      stop_index = num_sh_data_recs - 1

   num_head = stop_index - start_index + 1	; Total number of SH records
   num_MUX  = num_head				; Total number of MUX records
   head     = head( 0 : nbyts_rec_hed - 1, start_index: stop_index )
   MUX_val  = MUX_val( start_index : stop_index )
   time_MUX = time_MUX( start_index: stop_index )

;	Close the spin Header file

   CLOSE, iunit_h
   FREE_LUN, iunit_h


; 	Release space

   head_tmp     = 0
   time_MUX_tmp = 0
   MUX_val_tmp  = 0
   time_hed_tmp = 0
   header = 0

   RETURN
END     

;##############################################################################
PRO GET_TIMES, DAT, TIMES_IN_REC

;     This procedure finds the 16 times in the 'DAT' logical record.

;      PARAMETER                DESCRIPTION                       TYPE

;  DAT                 One 16-time telemetry logical record      Byte array
;  TIMES_IN_REC        The 16 times in logical record, 'DAT'     Long array

;_________________________________________________G. M. Boyd 29 April 1994

 COMMON FILDEF, num_recs_file, max_rec_size, file_type, iyear, iday, begt, endt
 COMMON TIME_MAP,  time_ind


times_in_rec = lonarr(16)                      ; Time is in milliseconds
for i = 0, 15 do begin
 
   byt_strt = time_ind(i)                      ; Time_ind has the location of
                                               ; the first (lsb) byte 
    times_in_rec(i) =  $                       ; Get longword, 'times_in_rec'
    long( dat, byt_strt, 1)                    ; from 4 bytes starting with 
                                               ; byte, byt_strt
endfor                                

return
end

;##############################################################################
PRO GET_TM_ACQ_TBL, IUNIT 

;    The purpose of this routine is to read the values for the Tememetry 
;    Accumulation table from the LEVEL-1 file and calculate the Spin
;    Acquisition Flags (spin_acq_flags)


;       PARAMETER                DESCRIPTION                       TYPE

;  IUNIT                        File access unit                  Integer

;	revision
;	29 Apr 1994	G. M. Boyd
;	14 Sep 1995	M. T. Redding	Added calls to 'BYTEORDER' to convert
;					extracted SHORTWORD values for
;					'num_energies', 'num_channels',
;					'spins_per_samp' and 'sects_per_spin'
;________________________________________________________________________

;    All LEVEL-1 files have a header record ( HR ) as the first record.
;    This record has all the values ( Accumulation/Telemetry table ) used 
;    for calculating the mapping parameters. 

;  The overall structure of the LEVEL-1 files for IES, IPS, HIST-E, and
;  HIST-P are similar.  Their differences reflect the numbers of energy ranges
;  and detector channels available on the respective sensors.  For example:

;       SENSOR     NUMBER DETECTOR CHANNELS          NUMBER OF ENERGY RANGES 
;        IPS                10                               17
;        IES                 9                               16
;       HIST-E               1                               17
;       HIST-P               1                               17


;  Each record contains data accumulated over a unique set of 16 spins.
;  Within each record the data is ordered first by spin time, that is, 
;  all data whose acqusition started on a particular spin are assigned 
;  the time for that spin.  The second level of ordering is energy.  
;  The third  level of  ordering is channel_number. Finally, the last 
;  level of ordering  is by sector.  Each pixel ( channel/energy_range 
;  combination) has a defined  number of sectors_per_spin.  Definitions 
;  of sectors_per_spin for each pixel will be found in Accumulation/Telemetry
;  format tables.  The Accumulation/telemetry table defined for the telemetry
;  in the LEVEL-1 file has been put in the HR. An example of such a table
;  follows:

;                           IPS Telemetry table
;      P0  P1  P2  P3  P4  P5  P6  P7  P8  P9 P10 P11 P12 P13 P14 P15 P16 ENERGY
;    ____________________________________________________________________
;    | 1 | 1 | 4 | 1 | 4 | 1 | 8 | 1 | 8 | 4 | 8 | 8 | 8 | 8 | 16| 16| 1 |NSPINS
;    ---------------------------------------------------------------------
;CH1 | 4 | 8 | 8 | 8 | 8 | 4 | 8 | 4 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 4 | EACH
;    ---------------------------------------------------------------------
;CH2 | 16| 32| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 32| ROW 
;    ---------------------------------------------------------------------
;CH3 | 16| 32| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 32| HAS
;    --------------------------------------------------------------------- 
;CH4 | 16| 32| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 32| THE
;    ---------------------------------------------------------------------
;CH5 | 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 32| SECT-
;    ---------------------------------------------------------------------
;CH6 | 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 8 | 8 | 8 | 8 | 8 | 8 | 32| ORS
;    ---------------------------------------------------------------------
;CH7 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 16| PER
;    ---------------------------------------------------------------------
;CH8 | 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 32| SPIN
;    ---------------------------------------------------------------------
;CH9 | 16| 16| 16| 16| 16| 16| 16| 16| 16| 16| 8 | 8 | 8 | 8 | 8 | 8 | 32| 
;    ---------------------------------------------------------------------
;CH10| 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 8 | 16| 
;    ---------------------------------------------------------------------


;  Reorganizing the data such that all data whose acqusition starts on a 
;  particular spin time , and such that each block contains 16 spin times
;  followed by its associated set of data (compressed accumulated counts) 
;  produces a LEVEL-1 file whose block is approximately 96 seconds 
;  (a spin is ~ 6 seconds) of data.  Part of the mapping for the above 
;  acqusition telemetry table is described below.

;  Note: Time is in milliseconds and is a longword. All comprressed accumulated
;  counts are bytes. Therefore,  the time would be equivalent to 4 bytes.

;			Spin time 0	(starting at byte 0)
;Time    ch01   ch02   ch03   ch04   ch05   ch06   ch07   ch08   ch09   ch10
;4 + P0(   4  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P1(   8  +  32  +  32  +  32  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P2(   8  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P3(   8  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P4(   8  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P5(   4  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P6(   8  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P7(   4  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P8(   8  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P9(   8  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;   P10(   8  +  16  +  16  +  16  +  16  +   8  +   8  +  16  +   8  +   8  + )
;   P11(   8  +  16  +  16  +  16  +  16  +   8  +   8  +  16  +   8  +   8  + )
;   P12(   8  +  16  +  16  +  16  +  16  +   8  +   8  +  16  +   8  +   8  + )
;   P13(   8  +  16  +  16  +  16  +  16  +   8  +   8  +  16  +   8  +   8  + )
;   P14(   8  +  16  +  16  +  16  +  16  +   8  +   8  +  16  +   8  +   8  + )
;   P15(   8  +  16  +  16  +  16  +  16  +   8  +   8  +  16  +   8  +   8  + )
;   P16(   4  +  32  +  32  +  32  +  32  +  32  +  16  +  32  +  32  +  16  + )

;			Spin time 1	(starting at byte 2380)
;4 + P0(   4  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P1(   8  +  32  +  32  +  32  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P3(   8  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P5(   4  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P7(   4  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;   P16(   4  +  32  +  32  +  32  +  32  +  32  +  16  +  32  +  32  +  16  + )

;			Spin time 2	(starting at byte 3360)
;4 + P0(   4  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P1(   8  +  32  +  32  +  32  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P3(   8  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P5(   4  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;    P7(   4  +  16  +  16  +  16  +  16  +  16  +   8  +  16  +  16  +   8  + )
;   P16(   4  +  32  +  32  +  32  +  32  +  32  +  16  +  32  +  32  +  16  + )

;   ETC.

 COMMON ACQ_FLAGS, one_spin_acq, two_spin_acq, four_spin_acq, $
                   eight_spin_acq, sixtn_spin_acq
 COMMON ACC_TABL,  num_energies, num_channels, spins_per_samp, sects_per_spin, $
                   total_sects_ene
 COMMON FILDEF,    num_recs_file, max_rec_size, file_type, iyear, iday, begt,endt
 COMMON SPIN_ACQ,  spin_acq_flg
 COMMON TMADEF,    tm_tabl_num, tm_tabl_crc
 

;         Set up the LEVEL-1 array struture 

;   A logical record is equal to 2 physical records.  The first 2 physical 
;   records have file header information.  'Max_rec_size' is number of bytes
;  in a logical record.  

num_ints_rec = max_rec_size/2L           ; Get number of integers in logical rec
n_integer = num_ints_rec/ 2L             ; Get number of integers in a phys rec
hed = assoc(iunit, intarr(n_integer))    ; Structure file to get integers
header_rec = intarr(num_ints_rec)
;print, ' max_rec_size, num_ints_rec, n_integer', $
;          max_rec_size, num_ints_rec, n_integer

h = hed( 0 )
header_rec(0: n_integer - 1L) = h
h = hed( 1 )
header_rec(n_integer:num_ints_rec - 1L)  = h       
                                               
num_energies = header_rec( 18 )                    ; Number of energies 
num_channels = header_rec( 19 )                    ; Number of detectors

BYTEORDER, num_energies, num_channels, /NTOHS      ; Convert SHORTWORDs from
                                                   ; NETWORK to HOST format

spins_per_samp = intarr( num_energies )            ; Spins per sample
sects_per_spin = intarr( num_channels, num_energies )      ; Sectors per spin
total_sects_ene = lonarr( num_energies )           ; Total sectors per energy

ind1 = 20                                 ; Beg location for spins_per_samp data
ind2 = ind1 + num_energies  - 1           ; End location for spins_per_samp data
spins_per_samp = header_rec( ind1:ind2 )  ; Get spins_per_samp data

BYTEORDER, spins_per_samp, /NTOHS         ; Convert SHORTWORDs from
                                          ; NETWORK to HOST format

ind = ind2
for i = 0, num_channels - 1 do begin

    ind3 = ind + 1                       ; Beg location for sects_per_spin data
    ind4 = ind + num_energies            ; End location for sects_per_spin data
    sects_per_spin( i, 0:num_energies - 1 ) = header_rec( ind3:ind4 )   
    ind = ind4 

endfor

BYTEORDER, sects_per_spin, /NTOHS       ; Convert SHORTWORDs from
                                        ; NETWORK to HOST format

;      Calculate the total number of sectors per energy

print, ' '
for i = 0, num_energies - 1 do begin

    total_sects_ene(i) = total( sects_per_spin( 0:num_channels - 1, i) )

endfor         

;       Spin acquisition flags for all powers of two up to max spin rate of 16

; Set up the spin aquisition flags for each spin per sample. Spin acquisition   
; flags indicate which of the 16-time subrecords contain data acquired in each
; energy range. When the spin you are on matches the spin_acq_flag value of a 
; energy, all channels in that energy will have initiated data accumulation 
; at that spin. 
 
 one_spin_acq   = indgen(16) + 1
 two_spin_acq   = [1, 1, 3, 3, 5, 5, 7, 7, 9, 9, 11, 11, 13, 13, 15, 15 ]
 four_spin_acq  = [1, 1, 1, 1, 5, 5, 5, 5, 9, 9, 9, 9, 13, 13, 13, 13 ]
 eight_spin_acq = [1, 1, 1, 1, 1, 1, 1, 1, 9, 9, 9, 9, 9, 9, 9, 9]
 sixtn_spin_acq = intarr(16)
 sixtn_spin_acq( * ) = 1

;
; Set up the aquisition flags for each spin per sample.  If the spin you 
; are on matches in one of the below sets, the channels in that energy will
; start accumulating data at that spin.

 spin_acq_flg = intarr( num_energies, 16)

 for i = 0, num_energies - 1 do begin
    if( spins_per_samp(i) eq 1 ) then spin_acq_flg( i, 0:15) = one_spin_acq
    if( spins_per_samp(i) eq 2 ) then spin_acq_flg( i, 0:15) = two_spin_acq
    if( spins_per_samp(i) eq 4 ) then spin_acq_flg( i, 0:15) = four_spin_acq
    if( spins_per_samp(i) eq 8 ) then spin_acq_flg( i, 0:15) = eight_spin_acq
    if( spins_per_samp(i) eq 16) then spin_acq_flg( i, 0:15) = sixtn_spin_acq
 endfor

 h = 0                               ; Release space
 hed = 0                             ; Release space

return
end 

;##############################################################################
PRO L1_COMN_MAPPING, NSPINS_TIM_SPAN, NBYTS_TIM_SPAN

;    This procedure calculates the number of spins in time span, the total
;    number of bytes in time span, and the position indices in the common block,
;    The indices map the first sector and the first spin of data for each 
;    energy.


;      PARAMETER                DESCRIPTION                       TYPE

;  NSPINS_TIM_SPAN        Number of spins in time span           Longword
;  NBYTS_TIM_SPAN      Total number of bytes in time span        Longword

;_________________________________________________G. M. Boyd 29 April 1994

 COMMON ACC_TABL,  num_energies, num_channels, spins_per_samp, sects_per_spin, $
                   total_sects_ene
 COMMON REC_NUM,   first_rec_read, last_rec_read
 COMMON COMN_IND,  comn_indx, npts_ene, comn_ch_indx

nspins_tim_span = ( ( last_rec_read -  $      ; Number of spins in time span
                  first_rec_read ) + 1L ) * 16L 

comn_ch_indx = lonarr(num_channels, num_energies )
comn_indx = lonarr( num_energies )
npts_ene = lonarr( num_energies )                

npts_ene0 = nspins_tim_span/long( spins_per_samp(0) ) ; The number of values
                                                 ; in a time span depends on 
                                                 ; the number of spins of data
                                                 ; and the number of spins over
                                                 ; which the data were
                                                 ; accumulated 
comn_indx0 = 0L
sects_ene0 = 0L

for i = 0, num_energies - 1 do begin

    npts_ene(i) = nspins_tim_span/    $              ; Number of pts per energy 
    long( spins_per_samp( i ) )                      ; based on spins per sample

    comn_indx(i) = comn_indx0  +      $              ; Location of sector 0 for 
   ( sects_ene0 * npts_ene0 )                        ; each energy in common 
                                                     ; block
    num_spns_ene = 0L
    for ich = 0, num_channels - 1 do begin   
       comn_ch_indx(ich, i) = (sects_per_spin(ich, i) * npts_ene(i)) + $  
       num_spns_ene                          ; Calculate a running total of 
       num_spns_ene = comn_ch_indx(ich, i)   ; points for each channel per
    endfor                                   ; energy for defined time span

    sects_ene0 = total_sects_ene(i)
    comn_indx0 = comn_indx(i)
    npts_ene0 = npts_ene(i)

endfor

nbyts_tim_span = comn_indx(num_energies - 1) + $    ; Total bytes in data common
       total_sects_ene(num_energies - 1) * npts_ene(num_energies -1) 

return 
end


;##############################################################################
PRO L1_DATA_MAPPING

;    This procedure calculates the position indices which map energies
;    that start accumulating data at each spin.

;_________________________________________________G. M. Boyd 29 April 1994

 COMMON ACC_TABL,  num_energies, num_channels, spins_per_samp, sects_per_spin, $
                   total_sects_ene
 COMMON REC_NUM,   first_rec_read, last_rec_read
 COMMON SPIN_ACQ,  spin_acq_flg
 COMMON TIME_MAP,  time_ind
 COMMON DATA_MAP,  data_rec_ind
 COMMON ACQ_FLAGS, one_spin_acq, two_spin_acq, four_spin_acq, eight_spin_acq, $
                   sixtn_spin_acq 


;       Spin flags for all powers of two up to max spin rate of 16
; Set up the aquisition flags for each spin per sample.  If the spin you 
; are on matches the spin_acq_flg, the channels in that energy will start 
; accumulating data at that spin

 
; Set up the indices for each energy at sector 0

 data_rec_ind = lonarr(num_energies, 16)

 for nspin = 0, 15 do begin
     nbyts = time_ind( nspin ) + 4L           ; Starting index for first energy
                                              ; at spin, 'nspin'
     dr_indx = 0                              ; Initialize data rec indx for 
                                              ; 'nspin'
     for ene = 0, num_energies - 1 do begin

        if(spin_acq_flg(ene, nspin) eq (nspin + 1) ) then begin   
             dr_indx = dr_indx + nbyts              ; Starting index energy, 
                                                    ; 'ene' at spin 'nspin'
                                                    ; (if the spin,
                                                    ; 'nspin'  you are on
             data_rec_ind(ene, nspin ) = dr_indx    ; and  flag,'spin_acq_flg'
                                                    ; are equal, the energy, 
                                                    ; 'ene', starts accumulat-
                                                    ; ing data )
            nbyts = total_sects_ene( ene)           ; Total number of bytes in
                                                    ; the accumulating energy
        endif
     endfor
    
endfor             

return
end

;##############################################################################
PRO L1_TIME_MAPPING

;     Calculate the mapping indices for the 16 spin times.
;     Maximum spin sample accumulation rate is 16 spins per sample


;_________________________________________________G. M. Boyd 29 April 1994

 COMMON ACC_TABL, num_energies, num_channels, spins_per_samp, sects_per_spin, $
                  total_sects_ene
 COMMON FILDEF,   num_recs_file, max_rec_size, file_type, iyear, iday, begt,endt
 COMMON SPIN_ACQ, spin_acq_flg
 COMMON TIME_MAP, time_ind


time_ind = lonarr( 16 )
time_ind(0) = 0L                 ; Index for the first spin time (tm data that 
                                 ; starts accumulating at spin 1 )
num_dat_follows = 0L
for i = 0, num_channels - 1 do begin         ; Total number of tm data bytes 
    num_dat_follows = num_dat_follows + $    ; that follow.  For the first
   long ( total ( sects_per_spin( i, 0:num_energies - 1))) ; spin time all data
endfor                                                ; start accumulating


num_dat_follows =  num_dat_follows  + 4L             ; Add 4 bytes for time
time_ind(1) = num_dat_follows                        ; Index for 2nd time

  for nspin = 1, 14 do begin                         ; Fourteen time indices 
                                                     ; left to calculate

     for ene = 0, num_energies - 1 do begin          ; Find what energies have
        if(spin_acq_flg(ene, nspin) eq (nspin + 1))$ ; Channels accumulating
        then begin                                   ; data at spin, nspin + 1
            num_dat_follows = num_dat_follows +    $
            total_sects_ene(ene)
         endif
     endfor
       
     num_dat_follows =  num_dat_follows  + 4L         ; Add 4 bytes for time
     
     time_ind( nspin + 1) = num_dat_follows

endfor

return
end

;##############################################################################
PRO OPEN_CEP_LEVEL1, FILENAME_LEVEL1, IUNIT, IERROR

;  This  procedure opens a LEVEL-1 file and returns the file access unit

;      PARAMETER                DESCRIPTION                       TYPE

;  IUNIT               LEVEL-1 file access unit                  Integer
;  FILENAME_LEVEL1     Name of LEVEL-1 file                      Character
;  IERROR               0 --  Successful open                    Integer
;                       1 --  Problem opening LEVEL-1

;	revision
;	Aug 31 1995	M.T.Redding	Changed OPENR to stream to allow access
;					of Level-1 files which have been
;					reblocked to 512 bytes per record
;					resulting from FTP transfer

;_________________________________________________G. M. Boyd 12 April 1994


;                Open CEPPAD LEVEL-1 file

ierror = 0                     ; If no error occurs in opening LEVEL-1 file,
                               ;    ierror is equal to zero.

GET_LUN, iunit                 ; Get available unit number (iunit) which
                               ;    specifies the file unit from which the 
                               ;    input is taken

ON_IOERROR,  OPEN_ERR          ;  Error opening file  go to OPEN_ERR
;bugbug		Changed to stream file opening to ignore physical record blocking
;;;OPENR, iunit, filename_level1    ; Open file, FILENAME_LEVEL1, to be read
OPENR, iunit, filename_level1, $
       /stream    ; Open file, FILENAME_LEVEL1, to be read
;bugbug


return

OPEN_ERR:
print, ' AN ERROR HAS OCCURED OPENING ', filename_level1
print, !ERR_STRING
FREE_LUN, iunit
ierror = 1

return
end

;##############################################################################
PRO RATES_COMN_FILL, LEV1_CEP, FILE_TYPE, TIME_ARR, COUNTS_ARR, CH_ID_ARR   


;           This procedure extract RATES mode data 

;      PARAMETER                DESCRIPTION                       TYPE

;  LEV1_CEP            Data access file ( associated file )      File structure
;  FILE_TYPE           Denotes the type of file                  Longword
;                       ( IPS rates =  6 )
;                       ( IES rates =  7 )
;                       ( HIST rates = 8 )
;  TIME_ARR            Times for the requested time span         Long array
;  COUNTS_ARR          Data for the requested time span          Long array 
;  CH_ID_ARR           Channel identification for each time      Byte array

;	revision
;	14 Sep 1995	M. T. Redding	Added calls to 'BYTEORDER',
;					Replace_NETWORK_LONGs_with_HOST' and
;					Replace_NETWORK_SHORTs_with_HOST' to
;					convert LONGWORD time and SHORTWORD
;					HIST command state variables from
;					NETWORK format (Big-endian) to HOST 
;					format and to convert extracted 3-byte
;					elements from Little-endian to host
;	22 Sep 1995	M. T. Redding	Corrected 'temp_counts' to
;					 'temp_counts_arr'
;___________________________________________________________________________

 COMMON RATES_ACC, n_energies, nchans,  nrecs_tim_span
 COMMON REC_NUM,   first_rec_read, last_rec_read
 COMMON SENSOR,    inst

 
; The telemetry data is in counts . The count value word length is 3 bytes.
; For a series of three bytes the first byte is the least significant.
;  EXAMPLE.  If the data to extract which is in bytes start with byte
;            number 20 and you want to extract two count values, then
;            byte 20 would be the least significant byte for the first 
;            count value and byte 23 would be the least significant byte 
;            for the second count value.

;	Define constants for the location and number of times and HIST
;	command state values to be converted to HOST format

 time_loc           = [ 0 ]
 number_of_times    = 1
 HIST_com_loc       = [ 10 ]
 number_of_hist_com = 1

 if( file_type eq 6 ) then begin
    num_wrds_extract = long (n_energies*nchans ) ; Number of 3 byte words
                                                 ; in a record to extract
    num_bytes = num_wrds_extract * 4             ; Total number of bytes 
 endif else begin                                ; (VAX wrd length is 4 bytes)
    num_wrds_extract = long( n_energies)         ; Number of 3 byte words
                                                 ; in a record to extract
    num_bytes = num_wrds_extract * 4L            ; Total number of bytes 
 endelse                                         ; (VAX wrd length is 4 bytes)

 nbyts4_indx = long (indgen(num_bytes) )
 nbyts3_indx = where( ( (nbyts4_indx + 1L) mod 4L) ne 0) ; Picks out the 3 byte
                                                         ; series
 nbyts3_series = num_wrds_extract * 3L              ; Number of bytes if word
                                                    ; length is 3 bytes
 ib = 14L                                           ; Series of 3 bytes start 
                                                    ; location in data record
 ie = ib + nbyts3_series - 1                        ; Series of 3 bytes end 
                                                    ; location in data record
 data = bytarr(num_bytes)                           ; Dimension target array

 nrec = -1                                         ; Initialize location counter
 for irec = first_rec_read, last_rec_read do begin

      dat = lev1_cep( irec )                       ; Get record irec

      Replace_NETWORK_LONGs_with_HOST, $           ; Convert LONGWORD time
         number_of_times, time_loc, dat            ; value from NETWORK format
                                                   ; to HOST
      Replace_NETWORK_SHORTs_with_HOST, $          ; Convert SHORTWORD HIST
         number_of_hist_com, HIST_com_loc, dat     ; command state value from 
                                                   ; format NETWORK to HOST
      nrec = nrec + 1                              ; Increment location 
                                                   ; counter            
      time_arr(nrec) = long(dat, 0)                ; Time is the first longword
                                                   ; in a record. Form longword
                                                   ; starting with byte 0
      if( file_type eq 6 ) then begin
         ch_id_arr(nrec) = dat(13)                 ; IPS mux state
         data(nbyts3_indx) = dat(ib:ie)            ; Put the 3 byte series into
                                                   ; the target array. Every
                                                   ; fourth byte of target array
                                                   ; is zero
         for i = 0, nchans - 1 do begin
             byt_strt = i * n_energies * 4

             temp_counts_arr= $                      ; Get longword count values
                long(data, byt_strt, n_energies)     ; for each energy
             BYTEORDER, temp_counts_arr, /LSWAP      ; Convert from Little-endian
             BYTEORDER, temp_counts_arr, /NTOHL      ; to HOST format
             counts_arr(nrec, i, 0:n_energies-1) = $ ; Transfer to output array
                temp_counts_arr

;             counts_arr(nrec, i, 0:n_energies-1) = $ ; Get longword count values
;                long(data, byt_strt, n_energies)     ; for each channel/energy
         endfor

      endif

      if( file_type eq 7) then begin
         ch_id_arr(nrec) = dat(7)                  ; IES (10h + ch num for data)
         data(nbyts3_indx) = dat(ib:ie)            ; Put the 3 byte series into
                                                   ; the target array. Every
                                                   ; fourth byte of target array
                                                   ; is zero
         temp_counts_arr= $                        ; Get longword count values
            long(data, 0, n_energies)              ; for each energy
         BYTEORDER, temp_counts_arr, /LSWAP        ; Convert from Little-endian
         BYTEORDER, temp_counts_arr, /NTOHL        ; to HOST format
         counts_arr(nrec, 0:n_energies-1) = $      ; Transfer to output array
            temp_counts_arr
;         counts_arr(nrec, 0:n_energies-1) = $      ; Get longword count values
;            long(data, 0, n_energies)              ; for each energy
      endif

      if( file_type eq 8) then begin
         ch_id_arr(nrec) = dat(9)                  ; HIST (10h + id of data)
         data(nbyts3_indx) = dat(ib:ie)            ; Put the 3 byte series into
                                                   ; the target array. Every
                                                   ; fourth byte of target array
                                                   ; is zero
         temp_counts_arr= $                        ; Get longword count values
            long(data, 0, n_energies)              ; for each energy
         BYTEORDER, temp_counts_arr, /LSWAP        ; Convert from Little-endian
         BYTEORDER, temp_counts_arr, /NTOHL        ; to HOST format
         counts_arr(nrec, 0:n_energies-1) = $      ; Transfer to output array
            temp_counts_arr
;         counts_arr(nrec, 0:n_energies-1) = $      ; Get longword count values
;            long(data, 0, n_energies)              ; for each energy

      endif
endfor

return
end

;##############################################################################
PRO RATES_TNEW_RNG, LEV1_CEP, TIME1, TIME2, BEG_TIME, END_TIME, IERROR

;   This procedure searches for the RATES record that has 'TIME1',
;    start time for which you want to extract data and  the record
;    that has 'TIME2', the time for which you want stop extracting data.
;      NOTE:  This procedure works only on times that are ascending 


;      PARAMETER                DESCRIPTION                       TYPE

;  LEV1_CEP            Data access file ( associated file )      File structure
;  TIME1               User input time to start extracting       Longword
;                      data
;  TIME2               User input time to stop extracting        Longword
;                      data
;  BEG_TIME            Beginning time used to start extracting   Longword
;                      data
;  END_TIME            Ending time used to stop extracting       Longword
;                      data
;  IERROR               0 --  Everything is okay                 Integer
;                      -1 --  Number of records is lt 2
;                      -2 --  None of Time range within begt and endt
;                      -3 --  Negative time in RATES file

;	revision
;	 4 May 1994	G. M. Boyd
;	14 Sep 1995	M. T. Redding	Added call to 
;					'Replac_NETWORK_LONGs_with_HOST' to
;					convert LONGWORD time value from 
;					NETWORK to HOST format following read
;					of Level-1 record
;	4 March 1996   G. M. Boyd	Made sure that at least one or both
;					of the requested times ( TIME1, TIME2 )
;					are in the data range( begt, endt ).
;______________________________________________________________________

 COMMON FILDEF,       num_recs_file, max_rec_size, file_type, iyear, iday, $
                      begt, endt
 COMMON RATES_ACC,    n_energies, nchans,  nrecs_tim_span
 COMMON REC_NUM,      first_rec_read, last_rec_read


;	Define constants for identifying the time parameter

  number_of_times = 1		; Just one spem per record
  time_loc        = [ 0 ]	; first 4 bytes in record contain time

;                    (TIME is located between position LOW and HIGH)
;        beginning time of data (msec)             begt
;        ending time of data    (msec)             endt

ierror = 0

if( num_recs_file lt 2L) then begin
   print, ' THE NUMBER OF RECORDS IN DATA FILE IS LESS THAN 2 '
   ierror = -1       
   return
endif

xb = begt                         ; First time in data file
xl = endt                         ; Last time in data file

in_range = 0
;;if( time1 ge xb ) and  ( time1 lt xl) then in_range = 1
;;if( time2 ge xb ) and  ( time2 le xl) then in_range = in_range + 1

; 4 March 1996
;	Below was revised to allow specified time range to completely
;	encompass existing times in file
;if ( time1 le xl) then in_range = 1
;if ( time2 ge xb) then in_range = in_range + 1
 
; 4 March 1996
if( time1 le xl ) and ( time2 ge xb ) then in_range = 1

if( in_range eq 0  ) then begin 
   print, ' '
   print, ' TIME RANGE CANNOT BE FOUND IN THIS FILE '
   print, ' day = ', iday, '  year = ', iyear
   print, ' FILE TIME RANGE IS ', begt, '   -   ', endt
   print, ' REQUESTED TIME SPAN IS ', time1, '   -   ', time2
   ierror  = -2
   return
endif 

dat = bytarr(max_rec_size)

for ntim = 0, 1 do begin                      ; Ntim = 0  find time1
   time = time1                               ; Ntim = 1  find time2
   if( ntim eq 1) then time = time2
   low_rec_num = 2L                           ; Data starts in 2nd record
   hi_rec_num = num_recs_file                 ; Num_recs_file = number of 
                                              ; records in file.  
   print, ' '
   print, ' time', time
   new_index = (hi_rec_num + low_rec_num)/2L  ; Initial record number 

   WHILE ( ( hi_rec_num - low_rec_num) gt 1L) DO BEGIN

       dat = lev1_cep( new_index )
       Replace_NETWORK_LONGs_with_HOST, $     ; Convert from Big-endian to
          number_of_times, time_loc, dat      ; HOST format
       time_in_rec = long(dat, 0)             ; Create longword starting with
                                              ; byte 0
       if ( time_in_rec lt 0 ) then begin  ; Should not have a negative time
           print, ' '
           print, ' THERE IS A NEGATIVE TIME IN RATES FILE '
           ierror = -3
           return
       endif

       if( time gt time_in_rec ) then begin
           low_rec_num = new_index             ; Set the new lowest rec number
       endif else begin
            hi_rec_num = new_index             ; Set the new highest rec number
       endelse
 
       new_index = (hi_rec_num+low_rec_num)/2L ; The new record number

   ENDWHILE

   if ( ntim eq 0 ) then begin
        first_rec_read = low_rec_num - 1L       ; Set first_rec_read to the 
        if( first_rec_read lt 1L ) then $          ; record number of the record 
        first_rec_read = 1L                     ; before the record in which
                                                ; TIME1 was found. If low_rec_
                                                ; num lt 1 ( data starts in 
                                                ; record 1 of file ),  set
                                                ; first_rec_read to 1L
        dat = lev1_cep( first_rec_read)      
        Replace_NETWORK_LONGs_with_HOST, $      ; Convert from Big-endian to
           number_of_times, time_loc, dat       ; HOST format
        beg_time = long(dat, 0)                 ; Get the time in that record

   endif else begin

        last_rec_read = low_rec_num + 1L        ; Set last_rec_read to the 
        if(last_rec_read ge num_recs_file) then $ ; record number of the record 
        last_rec_read = num_recs_file - 1       ; after the record in which
                                                ; TIME2 was found. If low_rec_
                                                ; num gt number of records in 
                                                ; set last_rec_read  to 
                                                ; num_recs_file - 1
        dat = lev1_cep( last_rec_read)      
        Replace_NETWORK_LONGs_with_HOST, $      ; Convert from Big-endian to
           number_of_times, time_loc, dat       ; HOST format
        end_time = long(dat, 0)                 ; Get the time in that record

    endelse

endfor

nrecs_tim_span = ( last_rec_read - first_rec_read ) + 1L

dat = 0                                 ; Release space

return
end
;##############################################################################
PRO RD_CEP_LEV1_HR, IUNIT, HEAD_REC, IERROR

;      Read the LEVEL-1 Header Record ( HR ) into HEAD_REC. HR is first
;      record on tape.  The first nine words ( long type words) are read.


;      PARAMETER                DESCRIPTION                       TYPE

;  IUNIT             LEVEL-1 file access unit                    Integer
;  HEAD_REC          Data read from LEVEL-1 file                 Long array
;                    ( first 9 long words of 1st rec )
;  IERROR               0 --  Successful read                    Integer
;                       1 --  Problem reading LEVEL-1
;                             Header Record (HR)

;	revision
;	12 Apr 1994	G. M. Boyd
;	14 Sep 1995	M. T. Redding	Added call to 'BYTEORDER' to convert
;					head_rec LONGWORDS from NETWORK to HOST
;					format
;_____________________________________________________________________


;              Read LEVEL-1 HR ( first seven long words )

ierror = 0           ; Error flag for reading, set to zero if everything okay 
 
head_rec =  lonarr(9)    

ON_IOERROR, READ_ERR
readu, iunit, head_rec

;	Convert head-rec from NETWORK-LONGWORDS to HOST-LONGWORDS format

BYTEORDER, head_rec, /NTOHL

return

READ_ERR:
 ierror = 1
 print, ' '
 print, format = '(1x,"ERROR READING LEVEL-1 HEADER RECORD" )
;    4 March 1996   note filename_level1 not passed in or in common
; print, format = '(1x, "READING FROM FILE ", A60)', filename_level1 
 print, !ERR_STRING

return
end

;##########################################################################
PRO READ_CEP_HK_DATA, iunit, time1, time2, beg_time, end_time

;       READ_CEP_HK_DATA		Reads the Housekeeping ( HK )  
;					parameters from the HK file and fills 
;					the HK_data STRUCTURE with all the 
;					data for the time span requested.
;	G. M. Boyd			27 March 1996

;____________________________________________________________________________

 COMMON FILDEF,      		num_recs_HK, nbyts_rec_HK, file_type, $
                                iyear_HK, day_HK, begt_HK, endt_HK
COMMON CEP_HK_EXTRACTION,       ten_mjrfrm_milsecs, num_recs_extracted, hk_data

;	LEV1_CEP		Associated file structure variable
;	TIME1			User requested start time extraction
;	TIME2			User requested end time extraction
;	BEG_TIME		The actual data extraction start time
;	END_TIME		The actual data extraction end time


;NOTE: The key parameters are part of the housekeeping file. It takes
;      10 frames to extract all the key parameters.  

Ten_mjrfrm_milsecs = 92000L	; Estimate of 10 major frames in milliseconds
if( ( time1 - Ten_mjrfrm_milsecs ) lt 0 ) then $
              tstart_10 = 0 ELSE tstart_10 = time1 - Ten_mjrfrm_milsecs
if( ( time2 + Ten_mjrfrm_milsecs ) gt endt_HK ) then $
              tstop_10 = endt_hk ELSE tstop_10 = time2 + Ten_mjrfrm_milsecs

print, ' Reading Housekeeping file:
result = SYSTIME( 1 )
LEV1_CEP = assoc( iunit, BYTARR( nbyts_rec_HK, num_recs_HK - 1L), nbyts_rec_HK )
hk_bytes = lev1_cep( 0 )

; 	Create a structure for the houskeeping data 
hsk = { hk,	tmsec: 0L,                  tmicro: 0L,			$
		sw_status: BYTARR( 2 ),     kp_hk_subcom: 0b,		$
		hw_status1: 0b,             mjf_seq_num: 0b,        	$
		hw_status2: 0b,             keyprms: BYTARR( 8 ),     	$ 
                sub_sens_stat: BYTARR( 2 ), sub_gen_stat: BYTARR( 2 ),	$ 
		sc_an0: 0b,           	    analog_1: 0b,		$
		sc_an1: 0b,           	    analog_2: 0b,		$
		sc_an2: 0b,           	    analog_3: 0b,		$
		sc_an3: 0b,           	    analog_4: 0b,		$
		sc_bstat: 0b,          	    CMDEcho_1: 0b,		$
		sc_th0: 0b,          	    CMDEcho_2: 0b,		$
		sc_th1: 0b,          	    CMDEcho_3: 0b,		$
		sc_th2: 0b,          	    CMDEcho_4: 0b,		$
		sc_th3: 0b,          	    CMDEcho_5: 0b,		$
                CheckSum: BYTARR( 2 ),      CEP_sub: BYTARR( 160 ),	$
                KeyParam_time: 0L,          HIST_s_e_time: LONARR( 8 )  }

   hk_data = REPLICATE( { HK }, num_recs_HK - 1L )

   ; 			Fill hk_data structure
     
      dummy = LONG( hk_bytes( 0 : 3, * ), 0, num_recs_HK - 1L )
      BYTEORDER, dummy, /NTOHL	 	  	; Convert to HOST format

      IF n_elements(dummy) eq 1 then dummy = dummy(0) ; mkc

      hk_data( * ).tmsec = dummy
      ;print, hk_data( 0 : 3 ).tmsec

      dummy = LONG( hk_bytes( 4 : 7, * ), 0, num_recs_HK - 1L )
      BYTEORDER, dummy, /NTOHL  		; Convert to HOST format
      IF n_elements(dummy) eq 1 then dummy = dummy(0) ; mkc
      hk_data( * ).tmicro       =    dummy

      dummy = LONG( hk_bytes( 206 : 209, * ), 0, num_recs_HK - 1L )
      BYTEORDER, dummy, /NTOHL	 	  	; Convert to HOST format
      IF n_elements(dummy) eq 1 then dummy = dummy(0) ; mkc
      hk_data( * ).KeyParam_time = dummy

      for itim = 0, 7 do begin
          ipos = 210 + ( itim * 4 )
          dummy = LONG( hk_bytes( ipos : ipos + 3, * ), 0, num_recs_HK - 1L )
          BYTEORDER, dummy, /NTOHL  		; Convert to HOST format
          hk_data( * ).HIST_s_e_time( itim ) = dummy
      endfor
      ;print, format = "(7i10)", hsk.HIST_s_e_time( 0 : 7 ) 

      hk_data( * ).sw_status( 0 : 1)      =   hk_bytes( 8 : 9,   * )
      hk_data( * ).keyprms( 0 : 7 )       =   hk_bytes( 14 : 21, * )  
      hk_data( * ).sub_sens_stat( 0 : 1 ) =   hk_bytes( 22 : 23, * )  
      hk_data( * ).sub_gen_stat( 0 : 1 )  =   hk_bytes( 24 : 25, * )  
      hk_data( * ).CheckSum( 0 : 1 )      =   hk_bytes( 44 : 45, * )
      hk_data( * ).CEP_sub( 0 : 159)      =   hk_bytes( 46 : 205,* )  

      hk_bytes = TRANSPOSE( hk_bytes )

      IF N_ELEMENTS(hk_bytes(*,10)) GT 1 THEN BEGIN ; mkc
      
        hk_data( * ).kp_hk_subcom           =   hk_bytes( *, 10 )
        hk_data( * ).hw_status1             =   hk_bytes( *, 11 )
        hk_data( * ).mjf_seq_num            =   hk_bytes( *, 12 )
        hk_data( * ).hw_status2             =   hk_bytes( *, 13 )
        hk_data( * ).sc_an0                 =   hk_bytes( *, 26 )
        hk_data( * ).analog_1               =   hk_bytes( *, 27 )
        hk_data( * ).sc_an1                 =   hk_bytes( *, 28 )
        hk_data( * ).analog_2               =   hk_bytes( *, 29 )
        hk_data( * ).sc_an2                 =   hk_bytes( *, 30 )
        hk_data( * ).analog_3               =   hk_bytes( *, 31 )
        hk_data( * ).sc_an3                 =   hk_bytes( *, 32 )
        hk_data( * ).analog_4               =   hk_bytes( *, 33 )
        hk_data( * ).sc_bstat               =   hk_bytes( *, 34 )
        hk_data( * ).CMDEcho_1              =   hk_bytes( *, 35 )
        hk_data( * ).sc_th0                 =   hk_bytes( *, 36 )
        hk_data( * ).CMDEcho_2              =   hk_bytes( *, 37 )
        hk_data( * ).sc_th1                 =   hk_bytes( *, 38 )
        hk_data( * ).CMDEcho_3              =   hk_bytes( *, 39 )
        hk_data( * ).sc_th2                 =   hk_bytes( *, 40 )
        hk_data( * ).CMDEcho_4              =   hk_bytes( *, 41 )
        hk_data( * ).sc_th3                 =   hk_bytes( *, 42 )
        hk_data( * ).CMDEcho_5              =   hk_bytes( *, 43 )

      ENDIF ELSE BEGIN
      
        hk_bytes = REFORM(hk_bytes)
      
        hk_data( * ).kp_hk_subcom           =   hk_bytes( 10) 
        hk_data( * ).hw_status1             =   hk_bytes( 11)
        hk_data( * ).mjf_seq_num            =   hk_bytes( 12)
        hk_data( * ).hw_status2             =   hk_bytes( 13)
        hk_data( * ).sc_an0                 =   hk_bytes( 26)
        hk_data( * ).analog_1               =   hk_bytes( 27)
        hk_data( * ).sc_an1                 =   hk_bytes( 28)
        hk_data( * ).analog_2               =   hk_bytes( 29)
        hk_data( * ).sc_an2                 =   hk_bytes( 30)
        hk_data( * ).analog_3               =   hk_bytes( 31)
        hk_data( * ).sc_an3                 =   hk_bytes( 32)
        hk_data( * ).analog_4               =   hk_bytes( 33)
        hk_data( * ).sc_bstat               =   hk_bytes( 34)
        hk_data( * ).CMDEcho_1              =   hk_bytes( 35)
        hk_data( * ).sc_th0                 =   hk_bytes( 36)
        hk_data( * ).CMDEcho_2              =   hk_bytes( 37)
        hk_data( * ).sc_th1                 =   hk_bytes( 38)
        hk_data( * ).CMDEcho_3              =   hk_bytes( 39)
        hk_data( * ).sc_th2                 =   hk_bytes( 40)
        hk_data( * ).CMDEcho_4              =   hk_bytes( 41)
        hk_data( * ).sc_th3                 =   hk_bytes( 42)
        hk_data( * ).CMDEcho_5              =   hk_bytes( 43)

      ENDELSE
      
      print, ' Time to read HK file', SYSTIME( 1 ) - result


      data_range_indx = where( ( hk_data.tmsec ge tstart_10 ) and $
                               ( hk_data.tmsec le tstop_10  ), count )

      num_recs_extracted = count	

   
if( num_recs_extracted eq 0 ) then begin
    print, ' NO DATA IN REQUESTED TIME SPAN', time1, time2
    print, ' STOPPED in procedure READ_CEP_HK_DATA'
    cep_kp = cep_kp( 0 )
    ;create crash so that calling
    ;routine's error checker gets it and processing continues!
    xxx = yyy1
    stop
endif else begin
    hk_data = hk_data( data_range_indx )
    beg_time = hk_data( 0 ).tmsec
    end_time = hk_data( count - 1 ).tmsec
    ;print, ' beg_time, end_time' , beg_time, end_time 
endelse

return
end

;#############################################################################
PRO Replace_NETWORK_LONGs_with_HOST, TotalToBEReplaced, LONG_loc, dat

;	Replace network formatted LONGWORD values with HOST formatted values

;	TotalToBeReplaced	Scalar containing the dimension size of 
;				'LONG_loc'
;	LONG_loc		Array containing pointers to locations 
;				within the dat array of the LONGWORDS to be
;				converted
;	dat			BYTE array containing the Level-1 record

;	revision:
;	13 Sep Aug 95	M.T.Redding

;--------------------------------------------------------------------

 
;	Define a local array to contain the LONGWORD values

   NetLONG = LONARR( TotalToBeReplaced )

; 	Extract and convert LONGWORDS from the BYTE buffer

   FOR i = 0, TotalToBEReplaced - 1 DO BEGIN
      NetLONG( i ) = LONG( dat, LONG_loc( i ) )
   ENDFOR

;	Convert the LONGWORDs from 'network' format to 'host'

   BYTEORDER, NetLONG, /NTOHL

;	Replace the converted LONGWORDS into the byte array

   FOR i = 0, TotalToBEReplaced - 1 DO BEGIN
      dat( LONG_loc( i ) ) = BYTE( NetLONG( i ), 0, 4 )
   ENDFOR

   RETURN
END



PRO Replace_NETWORK_SHORTs_with_HOST, TotalToBEReplaced, SHORT_loc, dat

;	Replace network formatted SHORTWORD values with HOST formatted values

;	TotalToBeReplaced	Scalar containing the dimension size of 
;				'SHORT_loc'
;	SHORT_loc		Array containing pointers to locations 
;				within the dat array of the SHORTWORDS to be
;				converted
;	dat			BYTE array containing the Level-1 record

;	revision:
;	13 Sep Aug 95	M.T.Redding
;       15 Aug 1996	G. M. Boyd	 Changed 'NetShort' from a long array
;					 to an integer array

;--------------------------------------------------------------------

 
;	Define a local array to contain the SHORTWORD values

   NetSHORT = INTARR( TotalToBeReplaced )

; 	Extract and convert SHORTWORDS from the BYTE buffer

   FOR i = 0, TotalToBEReplaced - 1 DO BEGIN
      NetSHORT( i ) = FIX( dat, SHORT_loc( i ) )
   ENDFOR

;	Convert the SHORTWORDs from 'network' format to 'host'

   BYTEORDER, NetSHORT, /NTOHS

;	Replace the converted SHORTWORDS into the byte array

   FOR i = 0, TotalToBEReplaced - 1 DO BEGIN
      dat( SHORT_loc( i ) ) = BYTE( NetSHORT( i ), 0, 2 )
   ENDFOR

   RETURN
END



;##############################################################################
PRO SETUP_COMMONS, NSPINS_TIM_SPAN, NBYTS_TIM_SPAN

;      Setup common variables that define the COMMON variables which are
;      addressed to process CEPPAD LEVEL-1 on the Remote Data Access Facility
;      (RDAF)


;      PARAMETER                DESCRIPTION                       TYPE

;  NSPINS_TIM_SPAN        Number of spins in time span           Longword
;  NBYTS_TIM_SPAN      Total number of bytes in time span        Longword


;	29 April 1994	G. M. Boyd
;	04 Apr 1996	M.T.Redding	Extended variable 'PLCPSpn_hed' in 
;					COMMON block POL_CEP_SH to 13 bytes -
;					additional byte contains SCOM - CEPPAD
;					spin number (0-255).  Corrected code
;					to include IPS MUX as 12th byte of
;	`				each record.  Likewise altered
;					variable 'head' in COMMON block 
;					SPN_HEAD.  Spin header file is now
;					expected to contain records for ALL
;					spins processed instead of just spins
;					at which any of the first 7 parameters
;					are noticed to have changed.

;------------------------------------------------------------------


;COMMON POL_SC,    plscmag_az, plscmag_el
;COMMON POL_EPH,
;COMMON POL_MAG,
;COMMON POL_ATT

;  POL_CEP_IPS_RTS - CEPPAD IPS Rates-histogram Common Block

;	Variable	                Description

;	PLCPIPS_Rts	  Longword array - Contains IPS Rates-histogram values
;			  accumulated through all IPS channels (0-9), 16
;			  energy ranges, and an integral channel for each
;		          spin time in the user specified time range.
;			  It is dimensioned (PLCPIPS_Rts_num,10,17)
;	PLCPIPS_Rts_num	  Longword - Contains the number of times (spins) of IPS
;			  Rates-histogram data within the user specified time 
;			  range.
;	PLCPIPS_Rts_tim  Longword array - Contains the time values in msec for
;			  each set of IPS Rates-histogram spin data within the 
;			  user specified time range.
;	PLCPIPS_Rts_mx    Byte array - Contains IPS bit flag indicating the
;                         the source IPS detectors used for corresponding sets 
;                         of data stored in the PLCPIPS_Rts array.

;  POL_CEP_IES_RTS - CEPPAD IPS Rates-histogram Common Block

;	Variable	                Description

;	PLCPIES_Rts	  Longword array - Contains IES Rates-histogram values
;			  accumulated through all IES channels (0-9), 16
;			  energy ranges, and an integral channel for each
;		          spin time in the user specified time range.
;			  It is dimensioned (PLCPIES_Rts_num,10,17)
;	PLCPIES_Rts_num	  Longword - Contains the number of times (spins) of IES
;			  Rates-histogram data within the user specified time 
;			  range.
;	PLCPIES_Rts_tim  Longword array - Contains the time values in msec for
;			  each set of IPS Rates-histogram spin data within the 
;			  user specified time range.
;	PLCPIES_Rts_id	  Longword array - Contains HIST identification values
;			  for corresponding sets of data in the
;			  PLCPHIST_Rts array.

;  POL_CEP_HIST_RTS - CEPPAD HIST Rates-histogram Common Block

;	Variable	                Description

;	PLCPHIST_Rts	  Longword array - Contains HIST Rates-histogram values
;			  accumulated through the one HIST channel and 256 energy
;			  ranges E0 - E255.  It is dimensioned 
;			  (256,PLCPHIST_Rts_num) to allow access of data by
;			  energy-range (0 to 255) and time (0 to 
;			  PLCPHIST_Rts_num-1).
;	PLCPIES_Rts_id	  Longword array - Contains HIST identification values
;			  for corresponding sets of data in the
;			  PLCPHIST_Rts array.

;	PLCPHIST_Rts_num  Longword - Contains the number of times (spins) of 
;		    	  HIST Rates-histogram data within the user specified
;			  time range.
;	PLCPHIST_Rts_tim Longword array - Contains the time values in msec 
; 			  corresponding to each set of HIST Rates-histogram 
;			  spin data within the user specified time range.

;  POL_CEP_SH - CEPPAD Spin Header Common

;	Variable	                Description

;	PLCPSpn_hed	Byte array containing all CEPPAD spin header ('SH') 
;			values stored in the Spin header Level-1 file for the
;			date requested by the user.  The array is dimensioned 
;			(13,PLCPSpn_hed_num).  Parameters indicated by the 
;			first index are the following:

;			Bytes	Description
;			0-3	Longword - Start of spin time in msec
;			4	Byte - TM table Number | TM Acquisition Mode
;			5	Byte - IPS Lookup Table Number
;			6	Byte - IES Lookup Table Number
;			7	Byte - IES Command byte
;			8	Byte - HIST Lookup Table Number
;			9-10	Integer - HIST Operating State
;			11	Byte - IPS Mux setting
;			12	Byte - SCOM - CEPPAD spin #
;	PLCPSpn_hed_num Number of data records in the Spin header Level-1 file
;			for the user's requested input 'day'

			
;  POL_CEP - POLAR CEPPAD 

;	Variable	                Description

;	PLCPNum_tables	Scalar Longword defining the number of T/M tables 
;			encountered in reading the Level-1 file corresponding 
;			to data stored in the PLCPIPS_bytes, PLCPIES_bytes 
;			and PLCPHST_bytes arrays
;	PLCPMax_tables	The maximum number of tables allowed to be accounted for
;			by the GET_DATA and PROCESS_DATA procedures


;  POL_CEP_IPS - CEPPAD IPS COMMON 

;	Variable	                Description

;	PLCPIPS_Time	Longword array containing the time of day in
;			milliseconds at the start of acquisition of each
;			spin containing IPS data most recently extracted 
;			from the level-1 file
;	PLCPIPS_bytes	Longword array containing the de-compressed count 
;			values accumulated through each 'sector' 'pixel'
;			(channel/energy-range) and spin as extracted from the
;			current Level-1 file portion.  The array allows for
;                       170 pixels -- 10 detector-channels x 17 energy-chans. 
;			Count values from a file section containing N spins of
;			data are packed into this single array in the following
;			order:
;
;			All existing count values from
;	==>		channel 1, energy 0, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;			    "         "      max sector for channel 1 energy 0
;			channel 2, energy 0, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;      ==>              ETC.
;
;	PLCPIPS_Org     PLCPIPS_Org(*, *, *, *)	 ( See description below )
;   		                    |  |  |  |
;                                   |  |  |  |__T/M table 
;                                   |  |  |_ Energy ( Depends on table)
;		                    |  |__ Detector channel (9 channels)
;       		            |__ Organization item  (4 parameters )
;	PLCPIPS_Start_Time Longword scalar containing the last requested
;			start time for IPS data - the time range of 
;			data actually stored in common resulting from 
;			an execution of 'GET_DATA' may be a subset of 
;			the requested time-range
;	PLCPIPS_Stop_Time  Longword scalar containing the last requested
;			stop time for IPS data

;  POL_CEP_IES - CEPPAD IES COMMON 

;	Variable	                Description

;	PLCPIES_Time	Longword array containing the time of day in
;			milliseconds at the start of acquisition of each
;			spin containing IES data most recently extracted 
;			from the level-1 file
;	PLCPIES_bytes	Longword array containging the de-compressed count 
;			values accumulated through each 'sector', 'pixel'
;			(channel/energy-range) and spin as extracted from the
;			current Level-1 file portion.  144 pixels -- 
;			9 detector-channels x 16 energy-channels are allowed
;			for
;			Count values from a file section containing N spins of
;			data are packed into this single array in the following
;			order:
;
;			All existing count values from
;	==>		channel 1, energy 0, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;			    "         "      max sector for channel 1 energy 0
;			channel 2, energy 0, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;      ==>              ETC.
;
;     	PLCPIES_Org     PLCPIES_Org(*, *, *, *)	 ( See description below )
;   		                    |  |  |  |
;                                   |  |  |  |__T/M table 
;                                   |  |  |_ Energy ( Depends on table)
;		                    |  |__ Detector channel (9 channels)
;       		            |__ Organization item  (4 parameters )
;	PLCPIES_Start_Time Longword scalar containing the last requested
;			start time for IES data - the time range of 
;			data actually stored in common resulting from 
;			an execution of 'GET_DATA' may be a subset of 
;			the requested time-range
;	PLCPIES_Stop_Time  Longword scalar containing the last requested
;			stop time for IES data

;  POL_CEP_HSTE - CEPPAD HISTE COMMON 

;       Variable	                Description

;	PLCPHSTE_Time	Longword array containing the time of day in
;			milliseconds at the start of acquisition of each
;			spin containing HIST-electron data most recently 
;			extracted from the level-1 file
;	PLCPHSTE_bytes	Longword array containging the de-compressed count 
;			values accumulated through each 'sector', 'pixel'
;			(channel/energy-range) and spin as extracted from the
;			current Level-1 file portion.  16 pixels -- 
;			1 detector-channels x 16 energy-channels are allowed
;			for
;			Count values from a file section containing N spins of
;			data are packed into this single array in the following
;			order:
;
;			All existing count values from
;			channel 1, energy 1, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;			    "         "      max sector for channel 1 energy 0
;			channel 1, energy 2, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;			    "         "      max sector for channel 1 energy 0
;                                               |
;      ==>              ETC.
;
;     	PLCPHSTE_Org     PLCPHSTE_Org(*, *, *, *)  ( See description below )
;   		                      |  |  |  |
;                                     |  |  |  |__T/M table 
;                                     |  |  |_ Energy ( Depends on table)
;		                      |  |__ Detector channel (9 channels)
;       		              |__ Organization item  (4 parameters )
;	PLCPHSTE_Start_Time Longword scalar containing the last requested
;			start time for HISTE data - the time range of 
;			data actually stored in common resulting from 
;			an execution of 'GET_DATA' may be a subset of 
;			the requested time-range
;	PLCPHSTE_Stop_Time  Longword scalar containing the last requested
;			stop time for HISTE data

;  POL_CEP_HSTP - CEPPAD HISTP COMMON 

;       Variable	                Description

;	PLCPHSTP_Time	Longword array containing the time of day in
;			milliseconds at the start of acquisition of each
;			spin containing HIST-proton data most recently 
;			extracted from the level-1 file
;	PLCPHSTP_bytes	Longword array containging the de-compressed count 
;			values accumulated through each 'sector', 'pixel'
;			(channel/energy-range) and spin as extracted from the
;			current Level-1 file portion.  16 pixels -- 
;			1 detector-channels x 16 energy-channels are allowed
;			for
;			Count values from a file section containing N spins of
;			data are packed into this single array in the following
;			order:
;
;			All existing count values from
;	==>		channel 1, energy 1, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;			    "         "      max sector for channel 1 energy 0
;			channel 1, energy 2, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;			    "         "      max sector for channel 1 energy 0
;						|
;						|
;      ==>              ETC.
;
;     	PLCPHSTP_Org     PLCPHSTP_Org(*, *, *, *)  ( See description below )
;   		                      |  |  |  |
;                                     |  |  |  |__T/M table 
;                                     |  |  |_ Energy ( Depends on table)
;		                      |  |__ Detector channel (9 channels)
;       		              |__ Organization item  (4 parameters )
;	PLCPHSTP_Start_Time Longword scalar containing the last requested
;			start time for HISTP data - the time range of 
;			data actually stored in common resulting from 
;			an execution of 'GET_DATA' may be a subset of 
;			the requested time-range
;	PLCPHSTP_Stop_Time  Longword scalar containing the last requested
;			stop time for HISTP data
;
; The Longword arrays for PLCPIPS_Org, PLCPIES_Org, PLCPHste_org, & PLCPHstp_Org
; description follows. The '...' can be substituted with IPS, IES, HSTE, or 
; HSTP.
; The plcp...org is a long word array whose dimension allocation follows:
; (4, num_channels, num_energies, plcpmax_tables ). This array defines the
; contents/organization of the 'plcp...bytes' array.  It is dimensioned to
; allow up to 'PLCPMAX_Tables' ( telemetry accumulation tables over the data
; range to be plotted ). The maximum number of tables allowed in this version
; of the program is one.

;  If 'plcp..._org' = plcpips_org = plcpips_org = lonarr(4, 10, 17, 2), then:
;    The first element ( 4 parameters)  are values that describe each of 
;    possible IPS pixels ( 10 channels/ 17 energy-channels) which together
;    describe the type of data and its location in the 'plcpips_bytes' array.
; The four parameters are :
;    Plcpips_org(0, *, *, * ) -- Number of sectors per spin as defined in 
;                                the telemetry accumulation table
;    Plcpips_org(1, *, *, * ) -- Number of spins per sample as defined in 
;                                the telemetry accumulation table
;    Plcpips_org(2, *, *, * ) -- Number of values stored  ( will be a function
;                                of the number of spins of data to which the 
;                                the current table is applied  AND the data
;                                accumulation rate ( the number of spins over
;                                which the data accumulated )
;    Plcpips_org(3, *, *, * ) -- Index location of first value in 
;                                'plcpips_bytes' array ( the value from the
;                                first sector and first spin of data to which
;                                the telemetry accumulation table applies
    


  COMMON POL_CEP,         plcpnum_tables, plcpmax_tables

  COMMON POL_CEP_IPS,     PLCPIPS_Time, PLCPIPS_bytes, PLCPIPS_Org, $
                          PLCPIPS_Start_Time, PLCPIPS_Stop_Time
  COMMON POL_CEP_IES,     PLCPIES_time, PLCPIES_bytes, PLCPIES_Org, $
                          PLCPIES_Start_Time, PLCPIES_Stop_Time
  COMMON POL_CEP_HSTE,    PLCPHSTE_time, PLCPHSTE_bytes, PLCPHSTE_Org, $
                          PLCPHSTE_Start_Time, PLCPHSTE_Stop_Time
  COMMON POL_CEP_HSTP,    PLCPHSTP_time, PLCPHSTP_bytes, PLCPHSTP_Org, $
                          PLCPHSTP_Start_Time, PLCPHSTP_Stop_Time
  COMMON POL_CEP_IPS_RTS, PLCPIPS_Rts, PLCPIPS_Rts_num, PLCPIPS_Rts_tim, $
                          PLCPIPS_Rts_mx, $
                          PLCPIPS_Rts_Start_Time, PLCPIPS_Rts_Stop_Time
  COMMON POL_CEP_IES_RTS, PLCPIES_Rts, PLCPIES_Rts_num, PLCPIES_Rts_tim, $
                          PLCPIES_Rts_ch, $
                          PLCPIES_Rts_Start_Time, PLCPIES_Rts_Stop_Time
  COMMON POL_CEP_HIST_RTS, PLCPHIST_Rts, PLCPHIST_Rts_num, PLCPHIST_Rts_tim, $
                          PLCPHIST_Rts_id, $
                          PLCPHIST_Rts_Start_Time, PLCPHIST_Rts_Stop_Time



 COMMON ACC_TABL,  num_energies, num_channels, spins_per_samp, sects_per_spin, $
                   total_sects_ene
 COMMON FILDEF,    num_recs_file, max_rec_size, file_type, iyear, iday, begt, $
                   endt
 COMMON REC_NUM,   first_rec_read, last_rec_read
 COMMON SENSOR,    inst
 COMMON RATES_ACC, n_energies, nchans,  nrecs_tim_span


plcpnum_tables = 0                          ; Counter for number of tables
                                            ; encountered in reading the 
                                            ; LEVEL-1 file corresponding to
                                            ; data stored in common block
                                        
plcpmax_tables = 1                          ; Maximun number of tables
                                            ; allowed 


if( file_type eq 2 ) then  begin                ; Define IPS NORMAL MODE common
     inst = 'IPS'
     plcpips_time = lonarr(nspins_tim_span)
     plcpips_bytes = bytarr(nbyts_tim_span)
endif

if( file_type eq 3 ) then begin                 ; Define IES NORMAL MODE common
     inst = 'IES'
     plcpies_time = lonarr(nspins_tim_span)
     plcpies_bytes = bytarr(nbyts_tim_span)
endif

if( file_type eq 4 ) then begin                 ; Define HISTE NORMAL MODE 
     inst = 'HISTE'                             ; common
     plcphste_time = lonarr(nspins_tim_span)
     plcphste_bytes = bytarr(nbyts_tim_span)
endif

if( file_type eq 5 ) then begin                  ; Define HISTP NORMAL MODE
     inst = 'HISTP'                              ; common
     plcphstp_time = lonarr(nspins_tim_span)
     plcphstp_bytes = bytarr(nbyts_tim_span)
endif

if( file_type eq 6 ) then begin                   ; Define IPS  RATES common
     inst = 'IPS'
     nchans = 10      
     n_energies = 17
     plcpips_rts_tim = lonarr(nrecs_tim_span)
     plcpips_rts = lonarr(nrecs_tim_span, nchans, n_energies)
     plcpips_rts_num = nrecs_tim_span
     plcpips_rts_mx = bytarr(nrecs_tim_span)
endif

if( file_type eq 7 ) then begin                    ; Define IES RATES  common
     inst = 'IES'
     n_energies = 256
     plcpies_rts_tim = lonarr(nrecs_tim_span)
     plcpies_rts = lonarr(nrecs_tim_span, n_energies)
     plcpies_rts_ch = bytarr(nrecs_tim_span)
     plcpies_rts_num = nrecs_tim_span
endif

if( file_type eq 8 ) then begin                     ; Define HIST RATES common
     inst = 'HIST'
     n_energies = 256
     plcphist_rts_tim = lonarr(nrecs_tim_span)
     plcphist_rts = lonarr(nrecs_tim_span, n_energies)
     plcphist_rts_id = bytarr(nrecs_tim_span)
     plcphist_rts_num = nrecs_tim_span
endif

return
end





;##############################################################################
PRO test_Get_CEP_DATA

;      To allow examination of common block variable filled by GET_CEP_DATA


;	29 April 1994	G. M. Boyd
;	04 Apr 1996	M.T.Redding	Extended variable 'PLCPSpn_hed' in 
;					COMMON block POL_CEP_SH to 13 bytes -
;					additional byte contains SCOM - CEPPAD
;					spin number (0-255).  Corrected code
;					to include IPS MUX as 12th byte of
;	`				each record.  Likewise altered
;					variable 'head' in COMMON block 
;					SPN_HEAD.  Spin header file is now
;					expected to contain records for ALL
;					spins processed instead of just spins
;					at which any of the first 7 parameters
;					are noticed to have changed.

;----------------------------------------------------------------------


;COMMON POL_SC,    plscmag_az, plscmag_el
;COMMON POL_EPH,
;COMMON POL_MAG,
;COMMON POL_ATT

;  POL_CEP_IPS_RTS - CEPPAD IPS Rates-histogram Common Block

;	Variable	                Description

;	PLCPIPS_Rts	  Longword array - Contains IPS Rates-histogram values
;			  accumulated through all IPS channels (0-9), 16
;			  energy ranges, and an integral channel for each
;		          spin time in the user specified time range.
;			  It is dimensioned (PLCPIPS_Rts_num,10,17)
;	PLCPIPS_Rts_num	  Longword - Contains the number of times (spins) of IPS
;			  Rates-histogram data within the user specified time 
;			  range.
;	PLCPIPS_Rts_tim  Longword array - Contains the time values in msec for
;			  each set of IPS Rates-histogram spin data within the 
;			  user specified time range.
;	PLCPIPS_Rts_mx    Byte array - Contains IPS bit flag indicating the
;                         the source IPS detectors used for corresponding sets 
;                         of data stored in the PLCPIPS_Rts array.

;  POL_CEP_IES_RTS - CEPPAD IPS Rates-histogram Common Block

;	Variable	                Description

;	PLCPIES_Rts	  Longword array - Contains IES Rates-histogram values
;			  accumulated through all IES channels (0-9), 16
;			  energy ranges, and an integral channel for each
;		          spin time in the user specified time range.
;			  It is dimensioned (PLCPIES_Rts_num,10,17)
;	PLCPIES_Rts_num	  Longword - Contains the number of times (spins) of IES
;			  Rates-histogram data within the user specified time 
;			  range.
;	PLCPIES_Rts_tim  Longword array - Contains the time values in msec for
;			  each set of IPS Rates-histogram spin data within the 
;			  user specified time range.
;	PLCPIES_Rts_id	  Longword array - Contains HIST identification values
;			  for corresponding sets of data in the
;			  PLCPHIST_Rts array.

;  POL_CEP_HIST_RTS - CEPPAD HIST Rates-histogram Common Block

;	Variable	                Description

;	PLCPHIST_Rts	  Longword array - Contains HIST Rates-histogram values
;			  accumulated through the one HIST channel and 256 energy
;			  ranges E0 - E255.  It is dimensioned 
;			  (256,PLCPHIST_Rts_num) to allow access of data by
;			  energy-range (0 to 255) and time (0 to 
;			  PLCPHIST_Rts_num-1).
;	PLCPIES_Rts_id	  Longword array - Contains HIST identification values
;			  for corresponding sets of data in the
;			  PLCPHIST_Rts array.

;	PLCPHIST_Rts_num  Longword - Contains the number of times (spins) of 
;		    	  HIST Rates-histogram data within the user specified
;			  time range.
;	PLCPHIST_Rts_tim Longword array - Contains the time values in msec 
; 			  corresponding to each set of HIST Rates-histogram 
;			  spin data within the user specified time range.

;  POL_CEP_SH - CEPPAD Spin Header Common

;	Variable	                Description

;	PLCPSpn_hed	Byte array containing all CEPPAD spin header ('SH') 
;			values stored in the Spin header Level-1 file for the
;			date requested by the user.  The array is dimensioned 
;			(13,PLCPSpn_hed_num).  Parameters indicated by the 
;			first index are the following:

;			Bytes	Description
;			0-3	Longword - Start of spin time in msec
;			4	Byte - TM table Number | TM Acquisition Mode
;			5	Byte - IPS Lookup Table Number
;			6	Byte - IES Lookup Table Number
;			7	Byte - IES Command byte
;			8	Byte - HIST Lookup Table Number
;			9-10	Integer - HIST Operating State
;			11	Byte - IPS Mux setting
;			12	Byte - SCOM - CEPPAD spion #
;	PLCPSpn_hed_num Number of data records in the Spin header Level-1 file
;			for the user's requested input 'day'

			
;  POL_CEP - POLAR CEPPAD 

;	Variable	                Description

;	PLCPNum_tables	Scalar Longword defining the number of T/M tables 
;			encountered in reading the Level-1 file corresponding 
;			to data stored in the PLCPIPS_bytes, PLCPIES_bytes 
;			and PLCPHST_bytes arrays
;	PLCPMax_tables	The maximum number of tables allowed to be accounted for
;			by the GET_DATA and PROCESS_DATA procedures


;  POL_CEP_IPS - CEPPAD IPS COMMON 

;	Variable	                Description

;	PLCPIPS_Time	Longword array containing the time of day in
;			milliseconds at the start of acquisition of each
;			spin containing IPS data most recently extracted 
;			from the level-1 file
;	PLCPIPS_bytes	Longword array containing the de-compressed count 
;			values accumulated through each 'sector' 'pixel'
;			(channel/energy-range) and spin as extracted from the
;			current Level-1 file portion.  The array allows for
;                       170 pixels -- 10 detector-channels x 17 energy-chans. 
;			Count values from a file section containing N spins of
;			data are packed into this single array in the following
;			order:
;
;			All existing count values from
;	==>		channel 1, energy 0, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;			    "         "      max sector for channel 1 energy 0
;			channel 2, energy 0, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;      ==>              ETC.
;
;	PLCPIPS_Org     PLCPIPS_Org(*, *, *, *)	 ( See description below )
;   		                    |  |  |  |
;                                   |  |  |  |__T/M table 
;                                   |  |  |_ Energy ( Depends on table)
;		                    |  |__ Detector channel (9 channels)
;       		            |__ Organization item  (4 parameters )
;	PLCPIPS_Start_Time Longword scalar containing the last requested
;			start time for IPS data - the time range of 
;			data actually stored in common resulting from 
;			an execution of 'GET_DATA' may be a subset of 
;			the requested time-range
;	PLCPIPS_Stop_Time  Longword scalar containing the last requested
;			stop time for IPS data

;  POL_CEP_IES - CEPPAD IES COMMON 

;	Variable	                Description

;	PLCPIES_Time	Longword array containing the time of day in
;			milliseconds at the start of acquisition of each
;			spin containing IES data most recently extracted 
;			from the level-1 file
;	PLCPIES_bytes	Longword array containging the de-compressed count 
;			values accumulated through each 'sector', 'pixel'
;			(channel/energy-range) and spin as extracted from the
;			current Level-1 file portion.  144 pixels -- 
;			9 detector-channels x 16 energy-channels are allowed
;			for
;			Count values from a file section containing N spins of
;			data are packed into this single array in the following
;			order:
;
;			All existing count values from
;	==>		channel 1, energy 0, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;			    "         "      max sector for channel 1 energy 0
;			channel 2, energy 0, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;      ==>              ETC.
;
;     	PLCPIES_Org     PLCPIES_Org(*, *, *, *)	 ( See description below )
;   		                    |  |  |  |
;                                   |  |  |  |__T/M table 
;                                   |  |  |_ Energy ( Depends on table)
;		                    |  |__ Detector channel (9 channels)
;       		            |__ Organization item  (4 parameters )
;	PLCPIES_Start_Time Longword scalar containing the last requested
;			start time for IES data - the time range of 
;			data actually stored in common resulting from 
;			an execution of 'GET_DATA' may be a subset of 
;			the requested time-range
;	PLCPIES_Stop_Time  Longword scalar containing the last requested
;			stop time for IES data

;  POL_CEP_HSTE - CEPPAD HISTE COMMON 

;       Variable	                Description

;	PLCPHSTE_Time	Longword array containing the time of day in
;			milliseconds at the start of acquisition of each
;			spin containing HIST-electron data most recently 
;			extracted from the level-1 file
;	PLCPHSTE_bytes	Longword array containging the de-compressed count 
;			values accumulated through each 'sector', 'pixel'
;			(channel/energy-range) and spin as extracted from the
;			current Level-1 file portion.  16 pixels -- 
;			1 detector-channels x 16 energy-channels are allowed
;			for
;			Count values from a file section containing N spins of
;			data are packed into this single array in the following
;			order:
;
;			All existing count values from
;			channel 1, energy 1, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;			    "         "      max sector for channel 1 energy 0
;			channel 1, energy 2, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;			    "         "      max sector for channel 1 energy 0
;                                               |
;      ==>              ETC.
;
;     	PLCPHSTE_Org     PLCPHSTE_Org(*, *, *, *)  ( See description below )
;   		                      |  |  |  |
;                                     |  |  |  |__T/M table 
;                                     |  |  |_ Energy ( Depends on table)
;		                      |  |__ Detector channel (9 channels)
;       		              |__ Organization item  (4 parameters )
;	PLCPHSTE_Start_Time Longword scalar containing the last requested
;			start time for HISTE data - the time range of 
;			data actually stored in common resulting from 
;			an execution of 'GET_DATA' may be a subset of 
;			the requested time-range
;	PLCPHSTE_Stop_Time  Longword scalar containing the last requested
;			stop time for HISTE data

;  POL_CEP_HSTP - CEPPAD HISTP COMMON 

;       Variable	                Description

;	PLCPHSTP_Time	Longword array containing the time of day in
;			milliseconds at the start of acquisition of each
;			spin containing HIST-proton data most recently 
;			extracted from the level-1 file
;	PLCPHSTP_bytes	Longword array containging the de-compressed count 
;			values accumulated through each 'sector', 'pixel'
;			(channel/energy-range) and spin as extracted from the
;			current Level-1 file portion.  16 pixels -- 
;			1 detector-channels x 16 energy-channels are allowed
;			for
;			Count values from a file section containing N spins of
;			data are packed into this single array in the following
;			order:
;
;			All existing count values from
;	==>		channel 1, energy 1, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;			    "         "      max sector for channel 1 energy 0
;			channel 1, energy 2, sector 0 from spins 0 to (N-1) 
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;			    "         "      max sector for channel 1 energy 0
;						|
;						|
;      ==>              ETC.
;
;     	PLCPHSTP_Org     PLCPHSTP_Org(*, *, *, *)  ( See description below )
;   		                      |  |  |  |
;                                     |  |  |  |__T/M table 
;                                     |  |  |_ Energy ( Depends on table)
;		                      |  |__ Detector channel (9 channels)
;       		              |__ Organization item  (4 parameters )
;	PLCPHSTP_Start_Time Longword scalar containing the last requested
;			start time for HISTP data - the time range of 
;			data actually stored in common resulting from 
;			an execution of 'GET_DATA' may be a subset of 
;			the requested time-range
;	PLCPHSTP_Stop_Time  Longword scalar containing the last requested
;			stop time for HISTP data
;
; The Longword arrays for PLCPIPS_Org, PLCPIES_Org, PLCPHste_org, & PLCPHstp_Org
; description follows. The '...' can be substituted with IPS, IES, HSTE, or 
; HSTP.
; The plcp...org is a long word array whose dimension allocation follows:
; (4, num_channels, num_energies, plcpmax_tables ). This array defines the
; contents/organization of the 'plcp...bytes' array.  It is dimensioned to
; allow up to 'PLCPMAX_Tables' ( telemetry accumulation tables over the data
; range to be plotted ). The maximum number of tables allowed in this version
; of the program is one.

;  If 'plcp..._org' = plcpips_org = plcpips_org = lonarr(4, 10, 17, 2), then:
;    The first element ( 4 parameters)  are values that describe each of 
;    possible IPS pixels ( 10 channels/ 17 energy-channels) which together
;    describe the type of data and its location in the 'plcpips_bytes' array.
; The four parameters are :
;    Plcpips_org(0, *, *, * ) -- Number of sectors per spin as defined in 
;                                the telemetry accumulation table
;    Plcpips_org(1, *, *, * ) -- Number of spins per sample as defined in 
;                                the telemetry accumulation table
;    Plcpips_org(2, *, *, * ) -- Number of values stored  ( will be a function
;                                of the number of spins of data to which the 
;                                the current table is applied  AND the data
;                                accumulation rate ( the number of spins over
;                                which the data accumulated )
;    Plcpips_org(3, *, *, * ) -- Index location of first value in 
;                                'plcpips_bytes' array ( the value from the
;                                first sector and first spin of data to which
;                                the telemetry accumulation table applies
    
;  COMMON MUXS                   Num_mux -- Longword number of mux values 
;                                           found in the requested time range
;                                           (IPS normal mode only)  
;                                Time_mux -- Longword array of times at 
;                                            which the mux values were found
;                                           (IPS normal mode only)  


;  COMMON SPN_HEAD               Num_head --     Longword number of spin header
;                                                records read from Spin Header
;                                                file ( same as variable, 
;                                                PLCPSPN_HED_NUM in 
;                                                common POL_CEP_SH )
;                                Head     --     Byte array of Spin Header (SH)
;                                                data ( same as variable
;                                                PLCPSPN_HED in common
;                                                POL_CEP_SH )




  COMMON POL_CEP,         plcpnum_tables, plcpmax_tables

  COMMON POL_CEP_IPS,     PLCPIPS_Time, PLCPIPS_bytes, PLCPIPS_Org, $
                          PLCPIPS_Start_Time, PLCPIPS_Stop_Time
  COMMON POL_CEP_IES,     PLCPIES_time, PLCPIES_bytes, PLCPIES_Org, $
                          PLCPIES_Start_Time, PLCPIES_Stop_Time
  COMMON POL_CEP_HSTE,    PLCPHSTE_time, PLCPHSTE_bytes, PLCPHSTE_Org, $
                          PLCPHSTE_Start_Time, PLCPHSTE_Stop_Time
  COMMON POL_CEP_HSTP,    PLCPHSTP_time, PLCPHSTP_bytes, PLCPHSTP_Org, $
                          PLCPHSTP_Start_Time, PLCPHSTP_Stop_Time
  COMMON POL_CEP_IPS_RTS, PLCPIPS_Rts, PLCPIPS_Rts_num, PLCPIPS_Rts_tim, $
                          PLCPIPS_Rts_mx, $
                          PLCPIPS_Rts_Start_Time, PLCPIPS_Rts_Stop_Time
  COMMON POL_CEP_IES_RTS, PLCPIES_Rts, PLCPIES_Rts_num, PLCPIES_Rts_tim, $
                          PLCPIES_Rts_ch, $
                          PLCPIES_Rts_Start_Time, PLCPIES_Rts_Stop_Time
  COMMON POL_CEP_HIST_RTS, PLCPHIST_Rts, PLCPHIST_Rts_num, PLCPHIST_Rts_tim, $
                          PLCPHIST_Rts_id, $
                          PLCPHIST_Rts_Start_Time, PLCPHIST_Rts_Stop_Time



 COMMON ACC_TABL,  num_energies, num_channels, spins_per_samp, sects_per_spin, $
                   total_sects_ene
 COMMON FILDEF,    num_recs_file, max_rec_size, file_type, iyear, iday, begt, $
                   endt
 COMMON REC_NUM,   first_rec_read, last_rec_read
 COMMON SENSOR,    inst
 COMMON RATES_ACC, n_energies, nchans,  nrecs_tim_span

  COMMON MUXS,            num_mux, mux_val, time_mux
  COMMON SPN_HEAD,        num_head, head

  COMMON POL_CEP_SH,      PLCPSpn_hed, PLCPSpn_hed_num


Get_Inputs:

;	Prompt for the Level-1 file to be read

   Level_1_filename = ''
   READ, Level_1_filename, PROMPT = 'Enter the Level-1 file name '
   READ, start_time, PROMPT = 'Enter the start-time '
   READ, stop_time, PROMPT = 'Enter the stop-time '

;	Execute the Level_1 file data extraction

   Get_CEP_DATA, Level_1_filename, start_time, stop_time

   STOP

   GOTO, Get_Inputs

   RETURN
END


;############################################################################
PRO EXTRACT_KEY_PARAMS

;		This procedure extracts the key parameters and 
;		fills the CEP_KP structure

 COMMON FILDEF,      		num_recs_HK, nbyts_rec_HK, file_type, $
                                iyear_HK, day_HK, begt_HK, endt_HK
COMMON CEP_KEY_PARAMETERs,	CEP_KP, num_kp
COMMON CEP_HK_EXTRACTION,       ten_mjrfrm_milsecs, num_recs_extracted, hk_data


;   Initialize key parameter structure

    cepkp  = { KP,	time: 0L,				$
				az:   0.0,			$
                                magel: 0.0,			$
                                magbx: 0.0,			$
                                magby: 0.0,			$
                                magbz: 0.0,			$
                                ips_err: 0.0,			$
                                ies_err: 0.0,			$
                                ips_coef: FLTARR( 4 ),		$
                                ies_coef: FLTARR( 4 ),		$
                                cubic_cntrls: 0,		$
                                ips_sper: FLTARR( 5 ),		$
                                ips_sper_drift: 0.0,		$  
                                ips_sper_angle: 0.0, 		$
                                ips_sper_rms:   0.0,		$
                                ies_sper: FLTARR( 5 ),		$
                                ies_sper_drift: 0.0,		$  
                                ies_sper_angle: 0.0, 		$
                                ies_sper_rms:   0.0,		$
                                ips_sun_offset: 0.0,		$
                                ies_sun_offset: 0.0                }

;   It takes 9 frames to collect all the key parameter data. 
     num_kp_sets = num_recs_extracted / 9
     if( num_kp_sets gt 0 ) then  begin
         cep_kp = REPLICATE( { KP }, num_kp_sets )
     endif else begin
         num_kp_sets = 1
         cep_kp = REPLICATE( { KP }, num_kp_sets )
         num_kp = 0
         return
     endelse
       
;	It takes 9 frames ( starting with frame 0 ) to collect all the 
;	key parameter data.  Find the positions when the top most 4bits 
;	of hk_data.kp_hk_subcom equal zero.
tmp = hk_data( 0 : num_recs_extracted - 1 ).kp_hk_subcom

kp_subcom =  tmp / 16 			; Top 4 most significant bits

;	Locate starting frames ( where 'kp_subcom' equal zero and
;	the key_parameter time is greater than zero )
kp_start_indx  = where( ( kp_subcom eq 0 ) and $
         ( hk_data( 0 : num_recs_extracted - 1).KeyParam_time gt 0 ), count )


;print, ' number of frame zeros found with positive times', count
if( count gt 0 ) then begin
    ;print, 'kp_start_indx', kp_start_indx
    if( ( kp_start_indx( count - 1 ) + 8 ) gt $
                     num_recs_extracted - 1 ) then begin 
         kp_start_indx = kp_start_indx( 0 : count - 2 )
         count = count - 1L
    endif
endif else begin
    num_kp = 0L
    cep_kp = cep_kp( 0 )
    RETURN
endelse


;print, format = "('mfrmt', 7i10 )", hk_data( kp_start_indx ).tmsec
;print, format = "('kptim', 7i10 )", hk_data( kp_start_indx ).KeyParam_time
; The key parameters are coefficients from a cubic fit and a 
; sperical fit (  16-bit values ), mag el, mag az, etc.
; Create the 16 bit array from the input byte arrays.
; There are four 16-bit words per frame.  It takes 9 frames to get
; a complete set of key parameters.


tmp = hk_data( 0 : num_recs_extracted - 1 ).keyprms 	; Byte array
n_integers = ( 8 * num_recs_extracted ) / 2		; Equivalent integers
tmp = FIX( REFORM( tmp,  2, n_integers ) )		; Reform the byte array
integer_values = ( tmp( 0,  * ) * 256 ) + tmp( 1, * )   ; Get integer value
tmp = FLOATING_PT( integer_values )			; Convert to floating pt

; 	Fill structure
num_kp = 0L
for k = 0L, count - 1L do begin

       ; Check for skip in output of data
       if( TOTAL( kp_subcom( kp_start_indx( k ) : $
                             kp_start_indx( k ) + 8 ) ) eq 36L ) and $
         ( ( hk_data( kp_start_indx( k ) + 8 ).tmsec - $
             hk_data( kp_start_indx( k )).tmsec ) lt 92000L ) then begin

           ;if(  ( kp_start_indx( k ) MOD 100 ) eq 0 ) then $
           ;print, format = "(' bytes', 8(1x, z2.2) )", $
           ;hk_data( kp_start_indx( k ) : kp_start_indx( k ) + 8 ).keyprms   
    
           ipos = kp_start_indx( k ) * 4
           ;if(  ( kp_start_indx( k ) MOD 100 ) eq 0 ) then $
           ;print, format = "(' integers',4(1x,z8.4) )",          $
           ;                    integer_values( ipos : ipos + 35 )
           

           ;if(  ( kp_start_indx( k ) MOD 100 ) eq 0 ) then $
           ;print, format = "(' floatpt', i14, 3e14.5/ (4e14.5 ) )", $ 
           ;hk_data( kp_start_indx( k ) ).KeyParam_time, $
           ;tmp( ipos + 1 : ipos + 35 )

	   ;          put data in structure

                cep_kp( num_kp ).time               = $
                                 hk_data( kp_start_indx( k ) ).KeyParam_time 
		cep_kp( num_kp ).az              =   tmp( ipos + 1 )
                cep_kp( num_kp ).magel           =   tmp( ipos + 2 )
                cep_kp( num_kp ).magbx           =   tmp( ipos + 3 )
		cep_kp( num_kp ).magby           =   tmp( ipos + 4 )
		cep_kp( num_kp ).magbz           =   tmp( ipos + 5 )
                cep_kp( num_kp ).cubic_cntrls    =   integer_values( ipos + 6 )
                cep_kp( num_kp ).ips_err         =   tmp( ipos + 7 )
                cep_kp( num_kp ).ies_err         =   tmp( ipos + 8 )
                cep_kp( num_kp ).ips_coef        =   tmp( ipos + 9 : ipos + 12 )
                cep_kp( num_kp ).ies_coef        =   tmp( ipos + 13 : ipos + 16)
                cep_kp( num_kp ).ips_sper        =   tmp( ipos + 17 : ipos + 21)
                cep_kp( num_kp ).ips_sper_drift  =   tmp( ipos + 22 )
                cep_kp( num_kp ).ips_sper_angle  =   tmp( ipos + 23 )
                cep_kp( num_kp ).ips_sper_rms    =   tmp( ipos + 24 )
                cep_kp( num_kp ).ies_sper        =   tmp( ipos + 25 : ipos + 29)
                cep_kp( num_kp ).ies_sper_drift  =   tmp( ipos + 30 )
                cep_kp( num_kp ).ies_sper_angle  =   tmp( ipos + 31 )
                cep_kp( num_kp ).ies_sper_rms    =   tmp( ipos + 32 )
                cep_kp( num_kp ).ips_sun_offset  =   tmp( ipos + 33 )
                cep_kp( num_kp ).ies_sun_offset  =   tmp( ipos + 34 )
                
          ;tpp = cep_kp( num_kp )          
          ;if(  ( kp_start_indx( k ) MOD 100 ) eq 0 ) then $
          ;print, format =$
          ;"(' floatpt', i14, 3e14.5/ 2e14.5, 6x,z8.4, e14.5 / (4e14.5 ) )", tpp

                num_kp = num_kp + 1L
       endif

endfor

if( num_kp gt 0 ) then  cep_kp = cep_kp( 0 : num_kp - 1 ) $
                  else  cep_kp = cep_kp( 0 ) 

tmp = 0
integer_values = 0

return
end
      
;############################################################################  
PRO UNPACK_SUBCOM_DATA

;	Unpack the CEPPAD generated SUBCOM values originally stored in POLAR 
;	telemetry word 15, minor frames 90 to 249.  They have been stored
;	in the Level-1 Housekeeping file in words( bytes ) 46 - 205. ( NOTE: 
;       read into HK_DATA structure tag name = CEP_sub.)

;	Several classes of information may be stored in these bytes.  Which 
;	class, is indicated by the value stored in what was originally minor 
;	frame 90 but here will be found in hk_data.CEP_sub( 0 ).
;       The Subcom flag value and a brief title of the class of data are 
;       given below.

;	hy_bytes.CEP_sub( 0 ) Subcom(90)	Title
;	(format)

;	  0		HIST Singles Output -
;			Compresses supersectors (* real sectors/ea.)
;			of HIST singles output
;	  1		HIST Events Output -
;			HIST Event data (1 event / sector)
;	  4		Table CRCs Dump -
;			CRCs from all DPU tables (automatically output
;			once per hour)
;	  5		Non-sectored T/M data -
;			IPS Calibration Report Format
;	253		Non-sectored T/M data -
;			Dump of Table ADdresses
;	254		Non-sectored Telemetry - 
;			Responds to DPUMEMDMP command by outputting
;			128 bytes of data

;--------------------------------------------------------------------

   COMMON FILDEF,            num_recs_HK, nbyts_rec_HK, file_type, $
                             iyear_HK, day_HK, begt_HK, endt_HK
   COMMON CEP_HK_EXTRACTION, ten_mjrfrm_milsecs, num_recs_extracted, hk_data

   COMMON CEP_HIST_Singles,      HIST_S, HIST_S_sets_extracted

   COMMON CEP_HIST_Events,       HIST_E, HIST_E_sets_extracted

   COMMON CEP_Table_CRCs,        Table_CRC, num_table_CRC

   COMMON CEP_IPS_calib,         IPSCal, num_IPS_cal

   COMMON CEP_Table_addresses,   Table_address, num_tab_add_recs 

   COMMON CEP_DPU_Mem_Dump,      DPU_MEMORY_DMP, num_mem_dmp_recs

;-----------------------------------------------------------------------


;	Unpack the Subcom values based on the format number

;///////////	HIST Singles Output	////////////////



;		Subcom_format = 0
zero_indx = where( $
     ( hk_data( 0 : num_recs_extracted - 1).CEP_sub( 0 ) eq  0 ), count )

   nset = 0L
   IF ( count gt 0 ) THEN BEGIN

      hists = {  hist , $
      time:    0L,     $
      sector:  0b,     $
      spin:    0b,     $
      data:    BYTARR( 16 )      }
      HIST_S = REPLICATE( { hist }, count * 8 )


      ;   There can be a max of 8 sets,  16 compressed bytes,  of data. 
      ;   If there is no data to output, the sector number is set to minus one.
      sector_position = [2L, 20L, 38L, 56L, 74L, 92L, 110L, 128L ]
      FOR NFRAM = 0L, COUNT - 1L DO BEGIN

          ;if (  ( zero_indx( nfram ) MOD 500L ) eq 0 ) then $
          ;print, format = "('mjfrm = ', i5, 1x, z2.2/(18(1x, z2.2 ) ) )", $
          ;zero_indx(nfram), hk_data(zero_indx(nfram) ).CEP_sub

          ; Find which sets can be extracted.
          set_indx = where( $
               ( hk_data( zero_indx(nfram) ).CEP_sub( sector_position ) ne $
                                               255 ), ngood_sects )
          if( set_indx( 0 ) ge 0 ) then begin  ; set_indx  between 0 and 7

              ; Extract the spin value for each set 
              wch_sect_indx = sector_position( set_indx )
              HIST_S( nset : nset + ngood_sects - 1L ).spin = $
                   hk_data( zero_indx(nfram) ).CEP_sub( wch_sect_indx - 1L )
              ;if (  ( zero_indx( nfram ) MOD 500L ) eq 0 ) then $
              ;print, format = "(' HIST_S.spin =', 8(1x,z2.2 ))", $
              ;HIST_S( nset : nset + ngood_sects - 1L ).spin

              ; Extract the sector values for each set
              HIST_S( nset : nset + ngood_sects - 1L ).sector = $
              hk_data( zero_indx(nfram) ).CEP_sub( wch_sect_indx ) 
              ;if (  ( zero_indx( nfram ) MOD 500L ) eq 0 ) then $
              ;print, format = "(' HIST_S.sector =', 8(1x,z2.2 ))", $
              ;HIST_S( nset : nset + ngood_sects - 1L ).sector
      
              ;if (  ( zero_indx( nfram ) MOD 500L ) eq 0 ) then $
              ;print, ' HIST_S.data follows'
              ; Extract the data sets
              for idat = 0, ngood_sects - 1 do begin
                  HIST_S(  nset ).data( 0 : 15 ) = $
                      hk_data( zero_indx(nfram) ).CEP_sub( $
                      LINDGEN( 16 ) +  wch_sect_indx( idat ) + 1L )
                  HIST_S( nset ).time = $
                  hk_data(zero_indx(nfram)).Hist_s_e_time(set_indx(idat) )

                  ;if (  ( zero_indx( nfram ) MOD 500L ) eq 0 ) then $
                  ;print, format = "( i11, 16(1x,z2.2 ) )", $
                  ;HIST_S( nset ).time, HIST_S( 0 : 15, nset ).data
 
                 nset = nset + 1
              endfor
           endif
      ENDFOR

      HIST_S = HIST_S( 0 : nset - 1 )

   ENDIF else begin

      HIST_S = { $
      time:    0L,     $
      sector:  0b,     $
      spin:    0b,     $
      data:    0b      }

   ENDELSE
   HIST_S_sets_extracted = nset


;///////////	HIST Events Output	////////////////

;		Subcom_format = 1
one_indx = where( $
           ( hk_data( 0 : num_recs_extracted - 1).CEP_sub( 0 ) eq 1 ), count )

   nset = 0L
   IF ( count gt 0 ) THEN BEGIN

       
      histe = {  hist , $
      time:    0L,     $
      sector:  0b,     $
      spin:    0b,     $
      data:    BYTARR( 16 )      }
      HIST_E = REPLICATE( { hist }, count * 8 )

      ;   There can be a max of 8 sets,  16 compressed bytes,  of data. 
      ;   If there is no data to output, the sector number is set to minus one.
      sector_position = [2L, 20L, 38L, 56L, 74L, 92L, 110L, 128L ]
      FOR NFRAM = 0L, COUNT - 1L DO BEGIN

          ;if (  ( one_indx( nfram ) MOD 500L ) eq 0 ) then $
          ;print, format = "('mjfrm = ', i5, 1x, z2.2/(18(1x, z2.2 ) ) )", $
          ;one_indx(nfram), hk_data(one_indx(nfram) ).CEP_sub

          ; Find which sets can be extracted.
          set_indx = where( $
               ( hk_data( one_indx(nfram) ).CEP_sub( sector_position ) ne $
                                               255 ), ngood_sects )
          if( set_indx( 0 ) ge 0 ) then begin  ; set_indx  between 0 and 7

              ; Extract the spin value for each set 
              wch_sect_indx = sector_position( set_indx )
              HIST_E( nset : nset + ngood_sects - 1L ).spin = $
                   hk_data( one_indx(nfram) ).CEP_sub( wch_sect_indx - 1L )
              ;if (  ( one_indx( nfram ) MOD 500L ) eq 0 ) then $
              ;print, format = "(' HIST_E.spin =', 8(1x,z2.2 ))", $
              ;HIST_E( nset : nset + ngood_sects - 1L ).spin

              ; Extract the sector values for each set
              HIST_E( nset : nset + ngood_sects - 1L ).sector = $
              hk_data( one_indx(nfram) ).CEP_sub( wch_sect_indx ) 
              ;if (  ( one_indx( nfram ) MOD 500L ) eq 0 ) then $
              ;print, format = "(' HIST_E.sector =', 8(1x,z2.2 ))", $
              ;HIST_E( nset : nset + ngood_sects - 1L ).sector
      
              ;if (  ( one_indx( nfram ) MOD 500L ) eq 0 ) then $
              ;print, ' HIST_E.data follows'
              ; Extract the data sets
              for idat = 0, ngood_sects - 1 do begin
                  HIST_E(  nset ).data( 0 : 15 ) = $
                      hk_data( one_indx(nfram) ).CEP_sub( $
                      LINDGEN( 16 ) +  wch_sect_indx( idat ) + 1L )
                  HIST_E( nset ).time = $
                  hk_data(one_indx(nfram)).HIST_s_e_time(set_indx(idat) )

                  ;if (  ( one_indx( nfram ) MOD 500L ) eq 0 ) then $
                  ;print, format = "( i11, 16(1x,z2.2 ) )", $
                  ;HIST_E( nset ).time, HIST_E( 0 : 15, nset ).data
 
                 nset = nset + 1
              endfor
           endif
      ENDFOR

      HIST_E = HIST_E( 0 : nset - 1 )

   ENDIF else begin

      HIST_E = { $
      time:    0L,     $
      sector:  0b,     $
      spin:    0b,     $
      data:    0b      }

   ENDELSE
   HIST_E_sets_extracted = nset


;///////////	Table CRCs dump 	////////////////

;		Subcom_format = 1
num_table_CRC = 0
four_indx = where( $
           ( hk_data( 0 : num_recs_extracted - 1).CEP_sub( 0 ) eq 4 ), count )


      ipos = ( INDGEN( 16 ) * 2 ) + 3
      if( count gt 0 ) then begin

 TableCRC = {  tc,  $
                Flt_SW:   0,	       $		; Flight software
                MST:      INTARR( 16), $		; Mode Switching Table
                SECT_A:   INTARR( 16), $   		; Sector & Accumulation
                IPS_CNF:  INTARR( 16), $		; IPS Configuration 
                IESLUT:   INTARR( 12), $		; IES Lookup Table
                HISTLUT:  INTARR(  4), $		; HIST Lookup Table
                TIME:     0L           }	        ; Main frame time 

  TABLE_CRC = replicate( { tc }, count ) 

          CRC_indx = INDGEN( 16 ) * 2 
          FOR nfram = 0L, count - 1L do begin

          ;if( nfram MOD 4 eq 0 ) then $ 
          ;print, format = "('mjfrm = ', i5, 1x, z2.2/(18(1x, z2.2 ) ) )", $
          ;four_indx(nfram), hk_data(four_indx(nfram) ).CEP_sub

              table_CRC( nfram ).Flt_SW = $
                    FIX( hk_data( four_indx(nfram )).CEP_sub( 1 ) ) * 256 + $
                    FIX( hk_data( four_indx(nfram )).CEP_sub( 2 ) ) 
          ;if( nfram MOD 4 eq 0 ) then $ 
          ;print, format = "('FLT_SW ', 8(1x, z8.4 ) )", $
          ;table_CRC(nfram).Flt_SW


              table_CRC( nfram ).MST( 0 : 15 ) = $
                    FIX( hk_data( four_indx(nfram )).CEP_sub( ipos ) ) * 256 + $
                    FIX( hk_data( four_indx(nfram )).CEP_sub( ipos + 1 ) )
          ;if( nfram MOD 4 eq 0 ) then $ 
          ;print, format = "('MST    ', 8(1x, z8.4 ) )", $
          ;table_CRC( nfram ).MST( 0 : 15 )


              table_CRC( nfram ).SECT_A( 0 : 15 ) = $
                    FIX(hk_data(four_indx(nfram)).CEP_sub( ipos + 32)) * 256 + $
                    FIX(hk_data(four_indx(nfram)).CEP_sub( ipos + 33))
          ;if( nfram MOD 4 eq 0 ) then $ 
          ;print, format = "('SECT_A ', 8(1x, z8.4 ) )", $
          ;table_CRC(nfram).SECT_A( 0 : 15 )


              table_CRC(nfram).IPS_CNF(0 : 15) = $
              FIX( hk_data( four_indx(nfram )).CEP_sub( ipos + 64 ) ) * 256 + $
              FIX( hk_data( four_indx(nfram )).CEP_sub( ipos + 65 ) )
          ;if( nfram MOD 4 eq 0 ) then $ 
          ;print, format = "('IPS_CNF', 8(1x, z8.4 ) )", $
          ;table_CRC( nfram ).IPS_CNF(0 : 15)


              indx = lonarr( 12 )
              indx = ipos( 0 : 11 ) 
              table_CRC(nfram).IESLUT(0 : 11) = $
              FIX( hk_data( four_indx(nfram )).CEP_sub(indx + 96)) * 256 + $
              FIX( hk_data( four_indx(nfram )).CEP_sub(indx + 97))
          ;if( nfram MOD 4 eq 0 ) then $ 
          ;print, format = "('IESLUT ', 8(1x, z8.4 ) )", $
          ;table_CRC(nfram).IESLUT(0 : 11)

              indx = lonarr( 4 )
              indx = ipos( 12 : 15 )
              table_CRC(nfram).HISTLUT(0 : 3) = $
              FIX( hk_data( four_indx(nfram)).CEP_sub(indx + 96)) * 256 + $
              FIX( hk_data( four_indx(nfram)).CEP_sub(indx + 97))
          ;if( nfram MOD 4 eq 0 ) then $ 
          ;print, format = "('HISTLUT', 4(1x, z8.4 ) )", $
          ;table_CRC(nfram).HISTLUT(0 : 3)

              table_CRC( nfram ).time = hk_data( four_indx( nfram ) ).tmsec

          ENDFOR

      ENDIF else begin

          Table_CRC = { $
                Flt_SW:   0,    $		; Flight software
                MST:      0,    $		; Mode Switching Table
                SECT_A:   0,    $	   	; Sector & Accumulation
                IPS_CNF:  0,    $		; IPS Configuration 
                IESLUT:   0,    $		; IES Lookup Table
                HISTLUT:  0,    $		; HIST Lookup Table
                TIME:     0L    }	        ; Main frame time 
    
      ENDELSE
      num_table_CRC = count


;///////////	IPS Calibration Report 	////////////////

;		Subcom_format = 5
five_indx = where( $
           ( hk_data( 0 : num_recs_extracted - 1).CEP_sub( 0 ) eq 5 ), count )
      num_IPS_cal = 0L

      IF( COUNT GT 0 ) then begin    
          ipcal = { cal,   $
                     time:       0L,         $	; Mfrm time in millisecs
                     Maskreq:    0,	     $
                     Maskrem:    0,          $
                     CurrentCal: 0b,         $
                     C1:	 BYTARR(9 ), $
                     B1:	 BYTARR(9 ), $
                     T1:	 BYTARR(9 ), $
                     C2:	 BYTARR(9 ), $
                     C5:	 BYTARR(9 ), $
                     C3:	 BYTARR(9 ), $
                     C0:	 BYTARR(9 ), $
                     B5:	 BYTARR(9 ), $
                     B3:	 BYTARR(9 ), $
                     B2:	 BYTARR(9 ), $
                     B0:	 BYTARR(9 ), $
                     T5:	 BYTARR(9 ), $
                     T3:	 BYTARR(9 ), $
                     T2:	 BYTARR(9 ), $
                     T0:	 BYTARR(9 )  }

  IPSCAL = replicate( { cal }, count ) 
;   NOTE:  Using Channel C1 as an example, the row value of 9 represents the 
;   following for each channel( see Dan Mabry for a more detailed description ) 
;           0 =  low  D/A for cal D/A = low res
;           1 =  high D/A for cal D/A = low res
;           2 =  cal  D/A for cal D/A = low res
;           3 =  low  D/A for cal D/A = high res
;           4 =  high D/A for cal D/A = high res
;           5 =  cal  D/A for cal D/A = high res
;           6 =  high D/A for cal = bin 14 top
;           7 =  cal  D/A for cal = bin 14 top
;           8 =  cal  D/A for cal = bin 4 top


;         IPSCAL.Maskreq = IPSCALMASK requested by command
          IPSCAL( 0 : count - 1 ).Maskreq = $
                           FIX( hk_data(five_indx).CEP_sub( 1 ) ) * 256  + $
                           FIX( hk_data(five_indx).CEP_sub( 2 ) ) 
          IPSCAL( 0 : count - 1 ).Maskreq = IPSCAL.Maskreq AND 32767

;         IPSCAL.Maskrem = IPSCALMASK remaining to execute
          IPSCAL( 0 : count - 1).Maskrem = $
                           FIX( hk_data(five_indx).CEP_sub( 3 ) ) * 256  + $
                           FIX( hk_data(five_indx).CEP_sub( 4 ) ) 
          IPSCAL( 0 : count - 1 ).Maskrem = IPSCAL.Maskrem AND 32767

;         IPSCAL.CurrentCal = IPSCAL state curently executing
          IPSCAL( 0 : count - 1).CurrentCal = hk_data( five_indx ).CEP_sub( 5 )
          for k = 0, count - 1 do begin
          IPSCAL( k ).C1( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub(  6 : 14 )
          IPSCAL( k ).B1( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub( 15 : 23 )
          IPSCAL( k ).T1( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub( 24 : 32 )
          IPSCAL( k ).C2( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub( 33 : 41 )
          IPSCAL( k ).C5( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub( 42 : 50 )
          IPSCAL( k ).C3( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub( 51 : 59 )
          IPSCAL( k ).C0( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub( 60 : 68 )
          IPSCAL( k ).B5( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub( 69 : 77 )
          IPSCAL( k ).B3( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub( 78 : 86 )
          IPSCAL( k ).B2( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub( 87 : 95 )
          IPSCAL( k ).B0( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub( 96 : 104)
          IPSCAL( k ).T5( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub(105 : 113)
          IPSCAL( k ).T3( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub(114 : 122)
          IPSCAL( k ).T2( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub(123 : 131)
          IPSCAL( k ).T0( 0 : 8 ) = hk_data( five_indx( k ) ).CEP_sub(132 : 140)
          endfor

          IPSCAL(0 : count - 1 ).time = $
                  hk_data( five_indx ).tmsec ; MFrm time in milliseconds

         ; print, format = "('mjfrm = ', i5, 1x, z2.2/(18(1x, z2.2 ) ) )", $
         ; five_indx(0), hk_data(five_indx(0) ).CEP_sub
         ;  print, format = $
         ;"(i10, 2(1x, z8.4 ), 1x, z2.2 / (9(1x, z2.2) ) )", IPSCAL( 0 ).time, $
         ; IPSCAL( 0 ).Maskreq, IPSCAL( 0 ).MaskRem, IPSCAL( 0 ).CurrentCal, $
         ; IPSCAL( 0 ).C1( 0 : 8 ), IPSCAL( 0 ).B1( 0 : 8 ), $
         ; IPSCAL( 0 ).T1( 0 : 8 ), IPSCAL( 0 ).C2( 0 : 8 ), $
         ; IPSCAL( 0 ).C5( 0 : 8 ), IPSCAL( 0 ).C3( 0 : 8 ), $
         ; IPSCAL( 0 ).C0( 0 : 8 ), IPSCAL( 0 ).B5( 0 : 8 )
           
 
     ENDIF ELSE BEGIN

          IPSCAL = { time:       0L,  $		; Mfrm time in millisecs
                     MASKREQ:    0,   $		; IPSCALMASK REQUESTED 
                     Maskrem:    0,   $		; IPScalmask to execute 
                     CurrentCal: 0b,  $ 	; IPS current cal state
                     C1:	 0b,  $
                     B1:	 0b,  $
                     T1:	 0b,  $
                     C2:	 0b,  $
                     C5:	 0b,  $
                     C3:	 0b,  $
                     C0:	 0b,  $
                     B5:	 0b,  $
                     B3:	 0b,  $
                     B2:	 0b,  $
                     B0:	 0b,  $
                     T5:	 0b,  $
                     T3:	 0b,  $
                     T2:	 0b,  $
                     T0:	 0b   } 

        ENDELSE
        num_IPS_cal = count



;///////////	IPS Calibration Report 	////////////////

;		Subcom_format = 253
INDX_253 = where( $
           ( hk_data( 0 : num_recs_extracted - 1).CEP_sub( 0 ) eq 253 ), count )

      IF( COUNT GT 0 ) then begin    
                    
          tab = { tadd,                                $
                      Time: 0L,                        $
                      SpinBuf0_page: 0,                $
                      SpinBuf0_offset: 0,              $
                      SpinBuf0_len: 0,                 $
                      ModeSwitching_page: 0,           $
                      ModeSwitching_offset: 0,         $
                      ModeSwitching_len: 0,            $
                      ModeSwitch_GenBase_page: 0,      $
                      ModeSwitch_GenBase_offset: 0,    $
                      SectAccumt0_page: 0,             $ 
                      SectAccumt0_offset: 0,           $
                      SectAccumt_len: 0,               $
                      IPS_configuration_tab0_page: 0,  $
                      IPS_configuration_tab0_offset: 0,$
                      IPS_configuration_tab_len: 0,    $
                      IES_lookup_tab0_page: 0,         $
                      IES_lookup_tab0_offset: 0,       $
                      IES_lookup_Tab_len: 0,           $
                      IES_lookup_GenBase_page: 0,      $
                      IES_lookup_GenBase_offset: 0,    $
                      HIST_lookup_tab0_page: 0,        $
                      HIST_lookup_tab0_offset: 0,      $
                      HIST_lookup_Tab_len: 0,          $
                      HIST_lookup_GenBase_page: 0,     $
                      HIST_lookup_GenBase_offset: 0,   $
                      Analogs_page:  0,                $
                      Analogs_offset:  0,              $
                      InputCmdStruct_page:  0,         $
                      InputCmdStruct_offset:  0,       $
                      CmdEchoStruct_page:  0,          $
                      CmdEchoStruct_offset: 0          } 
      TABLE_ADDRESS = REPLICATE( { tadd }, count )


          Table_address.time = hk_data( indx_253 ).tmsec

          Table_address.spinbuf0_page = $
                FIX( hk_data( indx_253 ).CEP_sub( 1  ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 2  ) )
          Table_address.spinbuf0_offset = $
                FIX( hk_data( indx_253 ).CEP_sub( 3  ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 4  ) )
          Table_address.spinbuf0_len = $
                FIX( hk_data( indx_253 ).CEP_sub( 5  ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 6  ) )

          Table_address.ModeSwitching_page = $
                FIX( hk_data( indx_253 ).CEP_sub( 7  ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 8  ) )
          Table_address.ModeSwitching_offset = $
                FIX( hk_data( indx_253 ).CEP_sub( 9  ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 10 ) )
          Table_address.ModeSwitching_len = $
                FIX( hk_data( indx_253 ).CEP_sub( 11 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 12 ) )
          Table_address.ModeSwitch_GenBase_page = $
                FIX( hk_data( indx_253 ).CEP_sub( 13 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 14 ) )
          Table_address.ModeSwitch_GenBase_offset = $
                FIX( hk_data( indx_253 ).CEP_sub( 15 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 16 ) )

          Table_address.SectAccumt0_page = $
                FIX( hk_data( indx_253 ).CEP_sub( 17 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 18 ) )
          Table_address.SectAccumt0_offset = $
                FIX( hk_data( indx_253 ).CEP_sub( 19 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 20 ) )
          Table_address.SectAccumt_len = $
                FIX( hk_data( indx_253 ).CEP_sub( 21 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 22 ) )

          Table_address.IPS_configuration_tab0_page = $
                FIX( hk_data( indx_253 ).CEP_sub( 23 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 24 ) )
          Table_address.IPS_configuration_tab0_offset = $
                FIX( hk_data( indx_253 ).CEP_sub( 25 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 26 ) )
          Table_address.IPS_configuration_tab_len = $
                FIX( hk_data( indx_253 ).CEP_sub( 27 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 28 ) )

          Table_address.IES_lookup_tab0_page = $
                FIX( hk_data( indx_253 ).CEP_sub( 29 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 30 ) )
          Table_address.IES_lookup_tab0_offset = $
                FIX( hk_data( indx_253 ).CEP_sub( 31 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 32 ) )
          Table_address.IES_lookup_Tab_len = $
                FIX( hk_data( indx_253 ).CEP_sub( 33 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 34 ) )
          Table_address.IES_lookup_GenBase_page = $
                FIX( hk_data( indx_253 ).CEP_sub( 35 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 36 ) )
          Table_address.IES_lookup_GenBase_offset = $
                FIX( hk_data( indx_253 ).CEP_sub( 37 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 38 ) )

          Table_address.HIST_lookup_tab0_page = $
                FIX( hk_data( indx_253 ).CEP_sub( 39 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 40 ) )
          Table_address.HIST_lookup_tab0_offset = $
                FIX( hk_data( indx_253 ).CEP_sub( 41 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 42 ) )
          Table_address.HIST_lookup_Tab_len = $
                FIX( hk_data( indx_253 ).CEP_sub( 43 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 44 ) )
          Table_address.HIST_lookup_GenBase_page = $
                FIX( hk_data( indx_253 ).CEP_sub( 45 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 46 ) )
          Table_address.HIST_lookup_GenBase_offset = $
                FIX( hk_data( indx_253 ).CEP_sub( 47 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 48 ) )

          Table_address.Analogs_page = $
                FIX( hk_data( indx_253 ).CEP_sub( 49 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 50 ) )
          Table_address.Analogs_offset = $
                FIX( hk_data( indx_253 ).CEP_sub( 51 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 52 ) )

          Table_address.InputCmdStruct_page = $
                FIX( hk_data( indx_253 ).CEP_sub( 53 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 54 ) )
          Table_address.InputCmdStruct_offset = $
                FIX( hk_data( indx_253 ).CEP_sub( 55 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 56 ) )

          Table_address.CmdEchoStruct_page = $
                FIX( hk_data( indx_253 ).CEP_sub( 57 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 58 ) )
          Table_address.CmdEchoStruct_offset = $
                FIX( hk_data( indx_253 ).CEP_sub( 59 ) ) * 256 +   $ 
                FIX( hk_data( indx_253 ).CEP_sub( 60 ) )

          ;print, format = "('mjfrm = ', i5, 1x, z2.2/(18(1x, z2.2 ) ) )", $
          ;indx_253( 0 ), hk_data( indx_253( 0 ) ).CEP_sub
          ;print, format = $
          ;"(i10/ (3(1x, z8.4) ) )", $
          ;Table_address.time( 0 ), Table_address.SpinBuf0_page( 0 ), $
          ;Table_address.SpinBuf0_offset( 0 ), table_address.SpinBuf0_len( 0 ), $
          ;Table_address.ModeSwitching_page( 0 ), $
          ;Table_address.ModeSwitching_offset( 0 ), $
          ;Table_address.ModeSwitching_len( 0 )


      ENDIF else begin
           
          Table_address = { Time: 0L,                        $
                            SpinBuf0_page: 0,                $
                            SpinBuf0_offset: 0,              $
                            SpinBuf0_len: 0,                 $
                            ModeSwitching_page: 0,           $
                            ModeSwitching_offset: 0,         $
                            ModeSwitching_len: 0,            $
                            ModeSwitch_GenBase_page: 0,      $
                            ModeSwitch_GenBase_offset: 0,    $
                            SectAccumt0_page: 0,             $ 
                            SectAccumt0_offset: 0,           $
                            SectAccumt_len: 0,               $
                            IPS_configuration_tab0_page: 0,  $
                            IPS_configuration_tab0_offset: 0,$
                            IPS_configuration_tab_len: 0,    $
                            IES_lookup_tab0_page: 0,         $
                            IES_lookup_tab0_offset: 0,       $
                            IES_lookup_Tab_len: 0,           $
                            IES_lookup_GenBase_page: 0,      $
                            IES_lookup_GenBase_offset: 0,    $
                            HIST_lookup_tab0_page: 0,        $
                            HIST_lookup_tab0_offset: 0,      $
                            HIST_lookup_Tab_len: 0,          $
                            HIST_lookup_GenBase_page: 0,     $
                            HIST_lookup_GenBase_offset: 0,   $
                            Analogs_page:  0,                $
                            Analogs_offset:  0,              $
                            InputCmdStruct_page:  0,         $
                            InputCmdStruct_offset:  0,       $
                            CmdEchoStruct_page:  0,          $
                            CmdEchoStruct_offset: 0          } 
      ENDELSE
      num_tab_add_recs = count


;///////////	IPS Calibration Report 	////////////////

;		Subcom_format = 254
INDX_254 = where( $
           ( hk_data( 0 : num_recs_extracted - 1).CEP_sub( 0 ) eq 254 ), count )


      IF( COUNT GT 0 ) then begin    
                    
          DPUDMP = { dump,                          $
                     Time: 0L,                      $ 
                     data_segment_address: 0,       $
                     data_offset_address:0,         $
                     data_dump: BYTARR( 128 )       }
          DPU_MEMORY_DMP = REPLICATE( { dump }, count )
 
          DPU_MEMORY_DMP.time = hk_data( indx_254 ).tmsec
          DPU_MEMORY_DMP.data_segment_address = $
                            FIX( hk_data( indx_254 ).CEP_sub( 1 )) * 256 + $
                            FIX( hk_data( indx_254 ).CEP_sub( 2 ))        
          DPU_MEMORY_DMP.data_offset_address = $
                            FIX( hk_data( indx_254 ).CEP_sub( 3 )) * 256 + $
                            FIX( hk_data( indx_254 ).CEP_sub( 4 ))        
     
          for nfram = 0L, count - 1L do begin
              DPU_MEMORY_DMP( nfram ).data_dump( 0 : 127 )  = $
                                hk_data( indx_254( nfram ) ).CEP_sub( 5 : 132)
          endfor

      ENDIF else begin

         DPU_MEMORY_DMP = { Time: 0L,                     $
                            data_segment_address: 0,       $
                            data_offset_address: 0,       $
                            data_dump: 0b                 }
      ENDELSE    
      num_mem_dmp_recs = count

RETURN
END



