;	Program MAKE_LEVEL_1

;	Version: 2.04	** 22 Nov 1996
;	Main features and bug fixes:

;  1.  Normal/Science mode Telemetry/Acquisition table number is compared with
;      the range of numbers found in the TM_format_tables.txt file.  Values
;      encountered that are not in range (as of Nov 22 1996 tables 0 - 4 are 
;      defined) are replaced with the value used in the preceeding spin.  
;      Besides the range check, if the table number changes from the preceeding
;      spin the spin number and spin time are used to determine if a valid 
;      change has occurred.  Valid changes are only supposed to occur on even 
;      16-spin boundaries (or, for data acquired before the July 3 DPU program 
;      patch  occassionally on the 2nd spin of 16). 

;  2.  The spin quality flag stored in the level-1 spin header (SH) file is now
;      filled with non-zero values when mode-switching occurs out-of-sync with 
;      the 16-spin blocks.  (Mode-switching is any change in the spin-header values e.g.
;      IPS LUT, IPS Mux, IES LUT, IES Command, HIST LUT HIST Command, TM/Acq 
;      format table number.)  The 4 least significant bits are set to indicate
;      the following conditions:
;
;      bit  condition
;      0    Mode ( TM/Acq format or spin header mode values below ) out of sync
;      1    IPS LUT or IPS Mux out of sync
;      2    IES LUT or IES Command out of sync
;      3    HIST LUT or HIST Command out of sync


;	Version: 2.03	** 02 July 1996
;	Main features and bug fixes:

;  1.  Corrected generation of 16-spin-to-32-time-block map for TM/Acq tables
;      indicating 0 bytes for some instruments (e.g. SAT3 and SAT4) and caused
;      Normal/science L1 file opening and mapping of data for instruments 
;      from which data are not accumulated to be skipped.


;	Version: 2.02	26 Jun 1996	
;	Main features and bug fixes:

;  1.  Corrected handling of 'commanded' mode changes - previous handling
;      sometimes created abnormally large normal/science mode L1 files which
;      while being read by the L1-reader program (GET_CEP_DATA), caused a 'lack 
;      of sufficient memory' condition.


;	Version: 2.01	14 Jun 1996	
;	Main features and bug fixes:

;  1.  Corrected writing of Spin header heading records (1-3) - added pad bytes
;      to obtain correct record lengths.  The problem exhibited itself while
;      GET_CEP_DATA attempted to read the spin header and extracted incorrect
;      year and day values from the file.  Spin header files created on a VMS
;      system (which is 'record-oriented'); it showed up on UNIX systems.


;	Version: 2.0	07 Jun 1996	
;	Main features:

;  1.  Stored spin times have been improved through application of 'sanity' and
;      consistency checks to eliminate out-of-order times due to bad data, data
;      drop-outs  and commanded instrument mode changes. 

;  2.  Fill values (-1) for missing times are no longer written to the
;      normal/science and rates-histogram level-1 files.  Instead, estimates are
;      computed for missing spin times and negatives of these 
;      (time = |estimate time|) are stored in the level-1 files.

;  3.  HouseKeeping (HK) and SpinHeader (SH) level-1 records have been expanded
;      to 242 and 14 bytes, respectively.  The additional HK bytes are used to 
;      store Key Parameter time and HIST-singles/HIST-events times which are 
;      now calculated in MAKE_LEVEL_1.  The additional SH bytes are for the 
;      CEPPAD spin number (SCOM) and reserved space for a 'spin-quality' flag.  
;      GET_CEP_DATA now contains routines to extract the HK level-1 data and 
;      store it in newly defined IDL common block variables. 

;  4.  GET_CEP_DATA now uses improved vectorized code adapted from Rob Sheldon's
;      L1-reader program to significantly speed up the reading of 
;      Normal/Science mode L1 files. 

;  5.  GET_CEP_DATA constructs and reads SH file with B.U. type filename if 
;      input L1 filename uses that convention.

;  6.  Various cleanup, speedup and bug fixes, including the elimination of
;      unnecessary routines and calling parameters have been performed on both 
;      codes (make_level_1.pro and get_cep_data.pro).

;  7.  Revisions to text files describing L1 files and common blocks containing
;      extracted L1 data have been made. 


;	Version 1.1	21 Mar 1996
;	Features:

;	Vectorized mapping of L0 to L1 data
;	Bug fixes to eliminate program crashes


;----------------------------------------------------------------------
;	Functions and procedures in this file


;	FUNCTION CHAR_WORD		Convert bytes to string
;	FUNCTION Julian_time		Calculate a rough estimate of the 
;					number of seconds since Dec. 31, 01 BC,
;					for ordering of events with known year,
;					month and day quantities
;	FUNCTION LONG_WORD		Convert 4 bytes to a longword 

;	PRO Add_new_spin		Transfer one spin to 16-spin buffer
;	PRO Append_HK			Write a CEPPAD HK data record
;	PRO Append_L1			Write Normal-mode TM data record to
;					IPS, IES, HISTe or HISTp Level-1 file(s)
;	PRO Append_SH			Write a CEPPAD Spin header data record
;	PRO Append_Rates		Write the current spin data to the
;					appropriate Rates-mode Level-1 file
;					(IPS-, IES- or HIST-Rates)
;	PRO CEP_REC_PARAMS		Set CEPPAD  data record and file 
;					description parameters
;					( record size, number of records, etc. )
;	PRO CEP_SETUP			Setup program constants
;	PRO CEP_setup2			Initialize various common block values
;	PRO Close_HK			Close CEPPAD Housekeeping Level-1 file
;	PRO Close_L1			Close all (IPS, IES, HISTe and HISTp)
;					Normal-mode Level-1 files
;	PRO Close_SH			Close L1 Spin Header file
;	PRO Construct_map		Create the CEPPAD IPS, IES, HISTe and 
;					HISTp mapping pointers from Level-0
;					16-spin organization to 32-time
;					(2 x Level-1) organization
;	PRO CORRECT_FORMAT		Check and revise Normal-Science mode
;					format table number
;	PRO Decode_tm_head		Parse the table heading record which
;					precedes the TM/Acquisition definition
;					table(s) in the CEPPAD_TM_tables.txt
;					file
;	PRO Def_Source_len		Determine the lengths of sections (by
;					spin, energy and channel) of data to be
;					loaded into the 16-spin Level-0 data
;					buffer
;	PRO Def_Source_loc		Determine the locations of the starts
;					of sections (by spin, energy and 
;					channel) of data to loaded into the 
;					16-spin data buffer
;	PRO Def_Target_loc		Determine the locations within the
;					'target' 32-time (2 x Level-1) data
;					buffer to which sections of data in the
;					16-spin buffer will be transferred
;	PRO Estimate_times		Interpolate/extrapolate to create an 
;					array of 16 non-negative times given a 
;					time array with some values 'missing' 
;					(-1L)
;	PRO Eval_time			Calculate a time value using a major
;					frame sequence number and a minor frame
;					number
;	PRO Extract_CEPPAD_spins	Control reading of CEPPAD Level-0 data 
;					records, processing of Housekeeping
;					data stored in each major frame and the
;					extraction from the CEPPAD science data
;					of individual spin data
;	PRO FILL_FLR			Fill File Label Record ( FLR ) structure

;	PRO FILL_MISSING_TIMES		Replace missing values ( -1L ) in the 
;					spin_blk_times array with estimates 
;					based on the existant times within the 
;					array using interpolation/extrapolation
;					where necessary.
;	PRO GET_MAJFRM_DATA		Control the filling of the arrays and/or
;					structures with a masterframe of CEPPAD
;					data ( Calls procedure 'UNPK_CEP_TM')
;	PRO Get_SW_status		Extract the individual status values 
;					from the CEPPAD Software status word(s)
;	PRO INIT_CEPDRH			Initialize Data Record Header ( DRH )
;					structure
;	PRO INIT_FLR			Initialize FLR structure
;	PRO Init_Subcom			Initialize the common block parameters
;					which will contain values extracted from
;					S/C telemetry word 15 - CEPPAD Subcom -
;					minor frames 90 - 249
;	PRO Init_16_blk			Initialize the 16-spin (Level-0) arrays,
;					counters and pointers
;	PRO Level_1_setup		Initialize various Level-1 file 
;					parameters
;	PRO Load_format			Read, based on the current spin's year,
;					day and time, the appropriate CEPPAD 
;					TM/Acquisition tables from the 
;					CEPPAD_TM_tables.txt file
;	PRO Make_level_1		Procedure to create Level-1 files from
;					a Level-0 file
;	PRO Map_to_time_blk		16-spin (Level-0) data to 32-time (2 x
;					Level-1 format) array section by section
;	PRO Open_HK			Obtain a unit number and define an
;					'associated' array for the Level-1
;					Housekeeping file for this day
;	PRO Open_L1			Obtain unit numbers and define
;					'associated' arrays for each (IPS, IES,
;					HISTe and/or HISTp) 'Normal-mode' 
;					Level-1 file to be used to store
;					reorganized Level-0 data
;	PRO Open_new_normal		Control preparation of normal-mode
;					files and mapping parameters based on
;					the TM format value included in the spin
;					data
;	PRO Open_new_Rates		Obtain unit number(s) and define
;					'associated' array for each Rates-mode
;					Level-1 file as it's type of data is
;					encountered 
;	PRO OPEN_PL_LEVEL0			Open POLAR CEPPAD LEVEL-0 file
;	PRO Open_SH			Obtain unit number, define the
;					'associated' array and initialize the
;					record pointer for the CEPPAD Spin
;					Header Level-1 file
;	PRO Process_HK			Transfer the Housekeeping and Subcom
;					values read in one record (major frame)
;					to the CEPPAD Housekeeping Level-1
;					file
;	PRO Process_MF			Control the handling of processing
;					needed for each major frame of data
;					read, including maintenance of the
;					Major frame time history and extraction
;					and storage of CEPPAD Housekeeping
;					parameters into the HK Level-1 file
;	PRO Process_new_spin		Control adding the new spin's data to 
;					the 16-spin buffer and mapping the 
;					current 16-spin buffer contents to the 
;					32-time buffer
;	PRO RD_PL_LEV0_FLR		Read FLR (first record in LEVEL-0 file)
;	PRO Read_CEP_TM_tables		Read CEPPAD Telemetry/Accumulation 
;					table file
;	PRO Read_data_rec		Read a record from the Level-0 file
;					and put science data and extract 
;					quality flag values 
;	PRO Read_next_table		Read TM/Accumulation table definition
;					numbers
;	PRO Shift_time_blk		Shift values from bottom half (last 16
;					spin times) of 32-time buffer(s) to
;					top half (first or previous 16 spin
;					times); reset bottom half values to 
;					'missing' (-1)
;	PRO Skip_CEP_TM_tables		Read past unwanted TM table in CEPPAD
;					TM table file
;	PRO Unpack_Subcom		Extract individual parameter values
;					stored in the CEPPAD SUBCOM byte -
;					word 15 minor frames 90-249
;	PRO Unpack_ths_spn		Unpack the data for one spin from the
;					major frame record into spin-header
;					parameters and a data array; also
;					change 'filled' zeroes to -1B
;	PRO UNPK_CEP_TM			Unpack CEPPAD telemetry and fills 
;					arrays and/or structures with a 
;					masterframe of data ( 250 minor frames )
;	PRO Verify_spin_end		Determine if the current location
;					(15-byte subrecord boundary) represents
;					the end of a set of spin data
;	PRO Verify_spin_start		Determine if the current location
;					(15-byte subrecord bouary) represents
;					the start of spin data
;----------------------------------------------------------------------





FUNCTION CHAR_WORD, BYT_STRT, NBYTS, BYT_ARRAY

;	Purpose:	Convert byte to string

;	BYT_ARRAY	Byte array containg values to be converted to character
;	BYT_STRT	starting postion in BYT_ARRAY for 1st character 
;	NBYTS		number of bytes to be converted ( number of characters 
;			in word )

;----------------------------------------------------------------------

byt_end = byt_strt + ( nbyts - 1)
cword = strtrim( string(byt_array(byt_strt:byt_end)), 1)
return,  cword
end





FUNCTION Julian_time, year, day, second

;	Calculate a crude estimate of the number of seconds since 
;	Dec 31 year 1 BC to be used for ORDERING events with known
;	year month and day quantities

;	Calculation assumes 366 days every year which makes the 
;	resulting value inaccurate for calculating absolute 
;	differences between Julian_times but does allow for correct 
;	ordering of Julian_times.

;	year		Integer containing the value of year - may be either 
;			4-digit (e.g. "1996") or 2-digit (e.g. "96")
;	day		Integer containing the Julian day - day of the year
;			this may be 0 (Dec 31 of the previous year) - 365 (366
;			if year is a 'Leap year'
;	second		Integer containing the seconds since 00:00 of the day -
;			may be 0 - 86399


;--------------------------------------------------------------------

;	Return 0.0D for zero date/time

   IF ( (( year LE 0 ) AND ( day LE 0 ))  AND  ( second LE 0 ) ) THEN BEGIN
      RETURN, 0.0D
   ENDIF

;	Expand abbreviated year values - assume dates are after 1992
   iyear = year
   IF ( iyear LT 93  ) THEN iyear = iyear + 2000
   IF ( iyear LT 100 ) THEN iyear = iyear + 1900

;	Calculate rough number of seconds for this date/time

   RETURN, ( iyear*366.d0 + day )*86400.d0 + second 

END






FUNCTION LONG_WORD, BYT_STRT, BYT_ARRAY

;     This procedure creates a long word from 4 bytes with BYT_STRT being
;     the most significant byte of the returned value LWORD

;	BYT_ARRAY	Byte array containg the 4 values to be converted to TYPE
;			LONG 
;	BYT_STRT	starting postion in BYT_ARRAY for the most significant
;			8 bits of the long_word 

;----------------------------------------------------------------------

common pwr2, p0, p1, p2, p3, p4, p5, p6, p7, p8, p12, p16, p20, p24

;----------------------------------------------------------------------

WORD =  long(byt_array,byt_strt,1)
LWORD = WORD(0)	; Make returned value a scalar
return, LWORD
end





pro add_new_spin, error_return

;	Transfer data for one spin from major frame buffer to 16-spin block

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block

;--------------------------------------------------------------------

COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                        sci_byt_per_spin, max_spins, $
                        nom_spin_dur, TRUE, FALSE, sync_byte, $
                        CEP_Sci_Man_byt_per_mnfrm, $
                        CEP_Conting_byt_per_mnfrm, $
                        CEP_Sci_Man_rec_len, $
                        CEP_Conting_rec_len

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

   COMMON CEP_lst_spn,     lst_spn_seq_no, lst_spn_min_frm, $
                           lst_spn_format, lst_spn_IPS_tab, $
                           lst_spn_IES_tab, lst_spn_IES_com, $
                           lst_spn_HIST_tab, lst_spn_HIST_com, $
                           lst_spn_SCOM, lst_spn_IPS_Mux, $
                           lst_spn_time, lst_spn_CRC, lst_spn_quality

   COMMON CEP_16_spin,     Spin_blk, Spin_blk_times, Spin_blk_SCOMs,	$
                           spin_blk_tot, spin_blk_seq_wrap,		$
                           est_16th_time, est_32nd_time, 		$
                           Bad_SCOM_MODE_sync, spin_quality

;-----------------------------------------------------------------------


;bugbug
;print, ' ADD_NEW_SPIN: 0'
;print, ths_spn_SCOM,		$
;       ths_spn_time,		$
;       ths_spn_format,		$
;       ths_spn_CRC,		$
;       ths_spn_seq_no,		$
;       ths_spn_min_frm,		$
;       ths_spn_IPS_tab,		$
;       ths_spn_IES_tab,		$
;       ths_spn_IES_com,		$
;       ths_spn_HIST_tab,	$
;       ths_spn_HIST_com,	$
;       ths_spn_IPS_Mux
;bugbug

;	Transfer the new spin data into the 16-spin block


   Spin_blk_times ( ths_spn_SCOM mod 16 )     = ths_spn_time
   Spin_blk_SCOMs ( ths_spn_SCOM mod 16 )     = ths_spn_SCOM
;   Spin_blk ( * , ( ths_spn_SCOM mod 16 ) )   = ths_spn_data(*)
   Spin_blk ( * , ( ths_spn_SCOM mod 16 ) )   = $
                        ths_spn_data( 0 : sci_byt_per_spin -1 )
   Spin_blk_tot                               = Spin_blk_tot + 1
   
;	Reset the last spin identifiers to the current spin's values for
;	comparison with future spins

   lst_spn_time     = ths_spn_time
   lst_spn_format   = ths_spn_format
   lst_spn_CRC      = ths_spn_CRC

   lst_spn_seq_no   = ths_spn_seq_no
   lst_spn_min_frm  = ths_spn_min_frm
   lst_spn_IPS_tab  = ths_spn_IPS_tab
   lst_spn_IES_tab  = ths_spn_IES_tab
   lst_spn_IES_com  = ths_spn_IES_com
   lst_spn_HIST_tab = ths_spn_HIST_tab
   lst_spn_HIST_com = ths_spn_HIST_com
   lst_spn_SCOM     = ths_spn_SCOM
   lst_spn_IPS_Mux  = ths_spn_IPS_Mux

      
;	Determine the estimate time of the 16 spin for the current block
;	(spins 0-15 are in the current spin block; spin 16 would be the first
;	spni in the next spin block - the variable name 'Est_16th_time' reflects
;	this counting i.e the 16th_time = 0-th_time in the next spin block)
;	based on an estimated spin duration time and the current spin's relative
;	location in the 16-spin block (SCOM mod 16).  Allow an extra quarter
;	spin on the final time to be on the safe side. 

;	Determine what the final time for spins in the next sequential
;	16-spin block should be based on the estimate for end of this 16-spin
;	block and the current nominal spin duration ... include an extra half
;	spin to be ensured of having a time that is beyond the next 16-spin
;	block

   IF spin_blk_tot LT 16 THEN BEGIN

      IF (ths_spn_time GT 0) THEN BEGIN 
         est_16th_time = ths_spn_time + $
                  (16-(ths_spn_scom MOD 16 )) * nom_spin_dur + $
                   LONG(.25 * nom_spin_dur)
         est_32nd_time = est_16th_time + $
                  (16 * nom_spin_dur) + $
                   LONG(0.5 * nom_spin_dur)
      ENDIF

   ENDIF 


; -----------------------------------------------------------------------
;	THE FOLLOWING IS COMMENTED UNTIL SUCH TIME AS IT IS DEEMED NECESSARY
;	TO INCLUDE DYNAMIC CALCULATION OF THE NOMINAL SPIN DURATION


;	Re-determine the nominal spin duration based on the times of the
;	most recently read spin data. Presuming that at least one spin of
;	data has already been written to the current 16-spin block, then
;	it is expected that the difference between SCOM values for the two
;	most recent spins will be positive and less than 15

;   IF spin_blk_tot gt 0 THEN BEGIN
;      nom_spin_dur = (ths_spn_time-lst_spn_time) / (ths_spn_SCOM-lst_spn_SCOM
;   ENDIF 

; -----------------------------------------------------------------------



   error_return = 0
   return
end



pro Append_HK, error_return, FLUSH_HK = flush_HK

;	Write a record of CEPPAD housekeeping parameters read from one
;	major frame to the CEPPAD Level-1 Housekeeping file.  Need to transfer
;	major frame values from HK and SUBCOM arrays.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure
;	flush_HK	Logical flag indicating whether all records in stored
;			in CEP_HK_temp_recs should be processed (times evaluated
;			or estimated) and appended to the HK file
;			Interpretation - 0 or unset (default) only the oldest
;			                            record in the buffer should
;			                            be processed
;			                 1          attempt to evaluated times
;			                            for all sets in each record
;			                            currently stored in
;			                            CEP_HK_temp_recs and append
;			                            results to the Housekeeping
;			                            file


;		The order of bytes per Level-1 record follows. 

;	byte #	minor	byte #	description
;		frame #	

;					Record # 1

;	0-3			Number of records in the CEPPAD HK file
;	4-7			Number of bytes per record in the CEPPAD HK file
;	8-11	DRH	20-23	Beginning Year
;	12-15	DRH	24-27	Beginning Day
;	16-19	DRH	28-31	Beginning Time Msec
;	20-23	DRH	32-35	Beginning Time Micro sec
;	24-27	DRH	20-23	Ending Year
;	28-31	DRH	24-27	Ending Day
;	32-35	DRH	28-31	Ending Time Msec
;	36-39	DRH	32-25	Ending Time Micro sec

;					Record # 2 - End of File
			
;	0-3	DRH	28-31	Beginning Time Msec
;	4-7	DRH	32-35	Beginning Time Micro sec
;	8	21	11	S/W Status byte 1
;	9	21	12	S/W Status byte 2
;	10	33	12	KP Subcom | HK Subcom
;	11	45	12	H/W Status # 1
;	12	55	11	Major Frame Sequence number
;	13	57	12	H/W Status # 2
;	14	69	11	KP #1 byte 1
;	15	69	12	KP # 1 byte 2
;	16	81	11	KP #2 byte 1
;	17	81	12	KP # 2 byte 2
;	18	93	11	KP #3 byte 1
;	19	93	12	KP # 3 byte 2
;	20	105	11	KP #4 byte 1
;	21	105	12	KP # 4 byte 2
;	22	117	11	Subcommed Sensor Status byte 1
;	23	117	12	Subcommed Sensor Status byte 2
;	24	129	11	Subcommed General Status byte 1
;	25	129	12	Subcommed General Status byte 2
;	26	141	11	SC-AN0
;	27	141	12	Analog # 1
;	28	153	11	SC-AN1
;	29	153	12	Analog # 2
;	30	165	11	SC-AN2
;	31	165	12	Analog # 3
;	32	177	11	SC-AN3
;	33	177	12	Analog # 4
;	34	189	11	SC-BSTAT
;	35	189	12	CMDEcho # 1
;	36	201	11	SC-TH0
;	37	201	12	CMDEcho # 2
;	38	213	11	SC-TH1
;	39	213	12	CMDEcho # 3
;	40	225	11	SC-TH2
;	41	225	12	CMDEcho # 4
;	42	237	11	SC-TH3
;	43	237	12	CMDEcho # 5
;	44	249	11	Checksum byte 1
;	45	249	12	Checksum byte 2
;	46-205	90-249	15	CEPPAD Subcom
;	206-209			Key Parameter time
;	210-213			HIST singles/event #1 spin time
;	214-217			HIST singles/event #2 spin time
;	218-221			HIST singles/event #3 spin time
;	222-225			HIST singles/event #4 spin time
;	226-229			HIST singles/event #5 spin time
;	230-233			HIST singles/event #6 spin time
;	234-237			HIST singles/event #7 spin time
;	238-241			HIST singles/event #8 spin time

;	revision
;	14 Sep 1995	M. T. Redding	Added call to 'BYTEORDER' to convert
;					LONGWORD integers to NETWORK format 
;					before writing to file. Also renamed
;					variables written to file to "net_..."
;					to make code more readable
;	20 Sep 1995	M. T. Redding	Expanded variables in common block
;					'CEP_HK_net' to include all parameters
;					to be written to the heading record.
;					Renamed some of these variables to
;					not conflict with variables in other
;					common blocks.
;					Replaced code updating the heading
;					record with execution of procedure
;					'Update_HK_Header'
;	25 Sep 1995	M. T. Redding	Commented out execution of 
;					'Update_HK_Header'  - revision of the 
;					header record is now done only prior
;					to closing the file instead of after
;					each new appended record.
;	01 Apr 1996	M.T.Redding	Corrected documented minor frame number
;					of KP Subcom | HK Subcom from '35' to 
;					'33' 
;       06 May 1996     M.T.Redding     Corrected index value of HK-subcom
;                                       corresponding to word 12 minor frame 33
;                                       (used to correspond to minor frame 35)
;                                       and SC-AN3 word 11 minor frame 177 (was
;                                       355 - corrected to 354)
;	07 May 1996	M.T.Redding	Added COMMON block CEP_HK_Times
;					containing variables CEP_KP_time, &
;					CEP_HIST_SE_times.
;					Also revised data HK record format to
;					include these LONGWORD variables after
;					the word-15 data - HK record length is
;					now extended to 242 bytes
;	24 May 1996	M.T.Redding	Altered to buffer current HK data in
;					common block array, get time values for
;					oldest record of data stored in buffer
;					transfer oldest data in buffer to HK
;					file.  
;					Also added FLUSH_HK keyword to cause
;					all records stored in buffer to be
;					processed and appended to the HK file

;--------------------------------------------------------------------

   COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                           sci_byt_per_spin, max_spins, $
                           nom_spin_dur, TRUE, FALSE, sync_byte, $
                           CEP_Sci_Man_byt_per_mnfrm, $
                           CEP_Conting_byt_per_mnfrm, $
                           CEP_Sci_Man_rec_len, $
                           CEP_Conting_rec_len

   COMMON CEP_HK,          CEP_HK_unit, CEP_HK_rec, CEP_HK_rec_len, CEP_HK_data

   COMMON Level_1_fil_typ, HK_filtyp, IPS_Nrm_filtyp, IES_Nrm_filtyp, $
                           HISTe_Nrm_filtyp, HISTp_Nrm_filtyp, $
                           IPS_Rts_filtyp, IES_Rts_filtyp, HIST_Rts_filtyp, $
                           Spin_Head_filtyp

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON CEP_HK_net,      net_CEP_HK_rec_len,     net_CEP_HK_filtyp, $
                           net_CEP_HK_rec,         net_CEP_HK_beg_year, $
                           net_CEP_HK_beg_day,     net_CEP_HK_beg_tim_msec, $
                           net_CEP_HK_end_tim_msec

   COMMON CEP_HK_times,    CEP_KP_time, $
                           CEP_HIST_SE_times
 
   COMMON CEP_HK_temp,     CEP_HK_max_temp_recs,	$
                           CEP_HK_temp_pointer,		$
                           CEP_HK_temp_recs
 
;-----------------------------------------------------------------------

   error_return = 0

;	Set default 'no flush' condition if not explicitly specified

   IF ( NOT ( KEYWORD_SET( flush_HK ) ) ) THEN flush_HK = 0


;	Transfer current HK data to temp_recs structure if current record has
;	a different time from the record most recently stored there

   transfer_current = FALSE
   IF ( CEP_HK_temp_pointer LT 0 ) THEN transfer_current = TRUE
   IF ( CEP_HK_temp_pointer GE 0 ) THEN BEGIN
      IF ( CEP_HK_temp_recs( CEP_HK_temp_pointer ).ATC_mils NE $
           DRH.ATC_mils ) THEN transfer_current = TRUE
   ENDIF

   IF ( transfer_current ) THEN BEGIN

      CEP_HK_temp_pointer = CEP_HK_temp_pointer + 1

      CEP_HK_temp_recs( CEP_HK_temp_pointer ).ATC_yr      = DRH.ATC_yr
      CEP_HK_temp_recs( CEP_HK_temp_pointer ).ATC_day     = DRH.ATC_day
      CEP_HK_temp_recs( CEP_HK_temp_pointer ).ATC_mils    = DRH.ATC_mils
      CEP_HK_temp_recs( CEP_HK_temp_pointer ).ATC_micro   = DRH.ATC_micro
      HKt = reform(transpose(HK),500)
      CEP_HK_temp_recs( CEP_HK_temp_pointer ).HKt         = $
         HKt( [  42,  43,  67,  91, 110, 115, 138, 139, 162, 163, $
                186, 187, 210, 211, 234, 235, 258, 259, 282, 283, $
                306, 307, 330, 331, 354, 355, 378, 379, 402, 403, $
                426, 427, 450, 451, 474, 475, 498, 499 ] )
      CEP_HK_temp_recs( CEP_HK_temp_pointer ).Subcom      = Subcom
      CEP_HK_temp_recs( CEP_HK_temp_pointer ).Min_frm_quality = Min_frm_quality
      CEP_HK_temp_recs( CEP_HK_temp_pointer ).CEP_KP_time = CEP_KP_time
      CEP_HK_temp_recs( CEP_HK_temp_pointer ).CEP_HIST_SE_times( * ) = -1L
   ENDIF


;	IF current pointer is at the last buffer location or if 'flush_HK'
;	has been specified, attempt to evaluate the HIST single/events set 
;	times for the oldest record in the buffer and append this to the HK
;	file

   WHILE ( ( CEP_HK_temp_pointer GE 0 ) AND $
           ( ( CEP_HK_temp_pointer EQ CEP_HK_max_temp_recs - 1 ) OR $ 
             ( flush_HK ) ) ) DO BEGIN


;	Evaluate HIST singles/events set times

      temp_times = CEP_HK_temp_recs( 0 ).CEP_HIST_SE_times( * )
      GET_HIST_SE_TIMES, CEP_HK_temp_recs( 0 ).Subcom,			$
                         CEP_HK_temp_recs( 0 ).Min_frm_quality, 	$
                         temp_times ,					$
                         error_return
      CEP_HK_temp_recs( 0 ).CEP_HIST_SE_times( * ) = temp_times 


;	Transfer Buffered HK and SUBCOM values for this record into a 
;	single byte array

      HK_data = BYTARR( CEP_HK_rec_len )

;	Items written to record 0 (HK-heading record) set value and type

      net_CEP_HK_End_year        = CEP_HK_temp_recs( 0 ).ATC_yr
      net_CEP_HK_End_day         = CEP_HK_temp_recs( 0 ).ATC_day
      net_CEP_HK_End_tim_msec    = CEP_HK_temp_recs( 0 ).ATC_mils
      net_CEP_HK_End_tim_micsec  = CEP_HK_temp_recs( 0 ).ATC_micro

;	Additional LONGWORD items written to records 1 - n

      net_CEP_KP_time       = CEP_HK_temp_recs( 0 ).CEP_KP_time
      net_CEP_HIST_SE_times = CEP_HK_temp_recs( 0 ).CEP_HIST_SE_times


;	Convert LONGWORD integers to NETWORK from HOST format

      BYTEORDER, net_CEP_HK_End_year,		$
                 net_CEP_HK_End_day,		$
                 net_CEP_HK_End_tim_msec,		$
                 net_CEP_HK_End_tim_micsec,	$ 
                 net_CEP_KP_time,			$
                 net_CEP_HIST_SE_times, / HTONL

;	Items that are written to records  1 - End of File (HK-data records)
			
      HK_data(  0:  3 ) = BYTE(net_CEP_HK_End_tim_msec,0,4)		; Beg time Msec
      HK_data(  4:  7 ) = BYTE(net_CEP_HK_End_tim_micsec,0,4)	; Beg time Micro sec

;      HKt = reform(transpose(HK),500)
;      index = [42,43,67,91,110,115,138,139,162,163,186,187,210,211,234,235, $
;   	258,259,282,283,306,307,330,331,354,355,378,379,402,403,426,427, $
;   	450,451,474,475,498,499]
      HK_data(  8 ) = CEP_HK_temp_recs( 0 ).HKt

      HK_data( 46 ) = CEP_HK_temp_recs( 0 ).Subcom(  90:249)	; CEPPAD Subcom

      HK_data( 206: 209 ) = BYTE( net_CEP_KP_time, 0, 4 )  
      HK_data( 210: 241 ) = BYTE( net_CEP_HIST_SE_times, 0, 32 ); HIST singles/event times


;	Append the new data record to the Level-1 file

      ON_IOERROR, Failed_Append_HK
      CEP_HK_rec = CEP_HK_rec + 1
      CEP_HK_data( CEP_HK_rec - 1) = HK_data

;	Convert LONGWORD integer from HOST to NETWORK format

      net_CEP_HK_rec     = CEP_HK_rec
      BYTEORDER, net_CEP_HK_rec, /HTONL

;	Update the heading record

      IF CEP_HK_rec eq 2 THEN BEGIN

;	The first time a record of data is written to the file make 
;	initial entries to the heading heading record after setting the
;	beginning year, day and time parameters

;	Convert LONGWORD integers from HOST to NETWORK format

         net_CEP_HK_rec_len   = CEP_HK_rec_len
         net_CEP_HK_filtyp = HK_filtyp
         BYTEORDER, net_CEP_HK_rec_len, $
                    net_CEP_HK_filtyp, /HTONL
   
         net_CEP_HK_Beg_year        = net_CEP_HK_End_year
         net_CEP_HK_beg_day         = net_CEP_HK_End_day
         net_CEP_HK_beg_tim_msec    = net_CEP_HK_End_tim_msec

         POINT_LUN, CEP_HK_unit, 0	; Reset file pointer to beginning
         WRITEU, CEP_HK_unit, $
              net_CEP_HK_rec, net_CEP_HK_rec_len, net_CEP_HK_filtyp, $
              net_CEP_HK_Beg_year, net_CEP_HK_beg_day, net_CEP_HK_beg_tim_msec, $
              net_CEP_HK_End_tim_msec

      ENDIF


;	Shove up all remaining records in temp_recs

      IF ( CEP_HK_temp_pointer GT 0 ) THEN BEGIN

         CEP_HK_temp_recs( 0: CEP_HK_temp_pointer - 1 ).ATC_yr = $
            CEP_HK_temp_recs( 1: CEP_HK_temp_pointer ).ATC_yr
         CEP_HK_temp_recs( 0: CEP_HK_temp_pointer - 1 ).ATC_day = $
            CEP_HK_temp_recs( 1: CEP_HK_temp_pointer ).ATC_day
         CEP_HK_temp_recs( 0: CEP_HK_temp_pointer - 1 ).ATC_mils = $
            CEP_HK_temp_recs( 1: CEP_HK_temp_pointer ).ATC_mils
         CEP_HK_temp_recs( 0: CEP_HK_temp_pointer - 1 ).ATC_micro = $
            CEP_HK_temp_recs( 1: CEP_HK_temp_pointer ).ATC_micro
         CEP_HK_temp_recs( 0: CEP_HK_temp_pointer - 1 ).HKt = $
            CEP_HK_temp_recs( 1: CEP_HK_temp_pointer ).HKt
         CEP_HK_temp_recs( 0: CEP_HK_temp_pointer - 1 ).Subcom = $
            CEP_HK_temp_recs( 1: CEP_HK_temp_pointer ).Subcom
         CEP_HK_temp_recs( 0: CEP_HK_temp_pointer - 1 ).Min_frm_quality = $
            CEP_HK_temp_recs( 1: CEP_HK_temp_pointer ).Min_frm_quality
         CEP_HK_temp_recs( 0: CEP_HK_temp_pointer - 1 ).CEP_KP_time = $
            CEP_HK_temp_recs( 1: CEP_HK_temp_pointer ).CEP_KP_time
         CEP_HK_temp_recs( 0: CEP_HK_temp_pointer - 1 ).CEP_HIST_SE_times( * )=$
            CEP_HK_temp_recs( 1: CEP_HK_temp_pointer ).CEP_HIST_SE_times( * )
      ENDIF

      CEP_HK_temp_pointer = CEP_HK_temp_pointer - 1

   ENDWHILE

   GOTO, Exit_Append_HK

Failed_Append_HK:
   print,'*** Append_HK ***"
   print,'Error #',!err,' encountered during CEPPAD HK Level-1 file update'
   print,!error
   print,!err_string
   error_return = 0


Exit_Append_HK:
   HK_data = 0B	; Deallocate the memory used by this array
   return
end





pro Append_L1, error_return

;	Write a record of 16-time data to each of the sensor Level-1 files
;	for each of the sensors for which files have been opened and rewrite
;	the heading record with an incremented record number.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1-4	Number of failed WRITEs to
;						Normal mode Level-1 files

;	revision
;	11-Aug-1995	M.T.Redding	Required 'bad_time_est' to be '0' (there
;					must be some defined times) for each 
;					Level-1 record to be stored
;	14 Sep 1995	M. T. Redding	Added common block 'CEP_Level_1_net';
;					lines to create NETWORK format LONGWORD
;					and SHORTWORD parameters. 
;					Changed code to write NETWORK format
;					parameter values to Level-1 files
;					instead of HOST format
;	20 Sep 1995	M. T. Redding	Expanded common block 'CEP_LEVEL_1_net'
;					to include separate variables for year, 
;					day, begging-time and ending-time for
;					each - IPS, IES, HISTe and HISTp level-1
;					files.
;					Renamed some variables in the code to
;					use the new instrument-specific 
;					variables.
;					Replaced code updating the header
;					records of each of the level-1 files
;					with execution of separate procedures.
;	25 Sep 1995	M. T. Redding	Commented out executions of 
;					'Update_IPS_L1_Header',
;					'Update_IES_L1_Header',
;					'Update_HISTe_L1_Header' and
;					'Update_HISTp_L1_Header'
;					  - revision of 
;					the header records are now done only 
;					prior to closing the file instead of 
;					after each new appended record.
;	01 May 1996	M. T. Redding	Corrected L1_DPU_TM_table to be the
;					4 LSB of ths_spn_format (was assumed to 
;					be 4 MSB )

;--------------------------------------------------------------------

   COMMON CEP_Level_1, $
		L1_number, L1_year, L1_day, L1_begt, L1_endt, $
		L1_DPU_TM_table, L1_TM_CRC, $
	Use_IPS_L1, IPS_L1_unit, IPS_L1_record, IPS_L1_assoc, $
		IPS_L1_rec_len, IPS_L1_phyrcl, $
	Use_IES_L1, IES_L1_unit, IES_L1_record, IES_L1_assoc, $
		IES_L1_rec_len, IES_L1_phyrcl, $
	Use_HISTe_L1, HISTe_L1_unit, HISTe_L1_record, HISTe_L1_assoc, $
		HISTe_L1_rec_len, HISTe_L1_phyrcl, $
	Use_HISTp_L1, HISTp_L1_unit, HISTp_L1_record, HISTp_L1_assoc, $
		HISTp_L1_rec_len, HISTp_L1_phyrcl

   COMMON CEP_lvl0_1_map, $
        Trgt_times, $
	num_source_spins,$
	num_IPS_energies, num_IPS_channels, IPS_spins_per_sample,$
		IPS_sectors_per_spin, IPS_Source_loc, IPS_Source_len,$
		IPS_Target_loc, IPS_trgt_tim_Loc, IPS_trgt_blk_len,$
                IPS_trgt_blk, IPS_source_byte_loc, IPS_trgt_byte_loc, $
	num_IES_energies, num_IES_channels, IES_spins_per_sample,$
		IES_sectors_per_spin, IES_Source_loc, IES_Source_len,$
		IES_Target_loc, IES_trgt_tim_loc, IES_trgt_blk_len,$
                IES_trgt_blk, IES_source_byte_loc, IES_trgt_byte_loc, $
	num_HISTe_energies, num_HISTe_channels, HISTe_spins_per_sample,$
		HISTe_sectors_per_spin, HISTe_Source_loc, HISTe_Source_len,$
		HISTe_Target_loc, HISTe_trgt_tim_loc, HISTe_trgt_blk_len,$
                HISTe_trgt_blk, HISTe_source_byte_loc, HISTe_trgt_byte_loc, $
	num_HISTp_energies, num_HISTp_channels, HISTp_spins_per_sample,$
		HISTp_sectors_per_spin, HISTp_Source_loc, HISTp_Source_len,$
		HISTp_Target_loc, HISTp_trgt_tim_loc, HISTp_trgt_blk_len,$
                HISTp_trgt_blk, HISTp_source_byte_loc, HISTp_trgt_byte_loc

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

   COMMON Level_1_fil_typ, HK_filtyp, IPS_Nrm_filtyp, IES_Nrm_filtyp, $
                           HISTe_Nrm_filtyp, HISTp_Nrm_filtyp, $
                           IPS_Rts_filtyp, IES_Rts_filtyp, HIST_Rts_filtyp, $
                           Spin_Head_filtyp

   COMMON CEP_LEVEL_1_net, $
           net_L1_DPU_TM_table, net_L1_TM_CRC, $

           net_IPS_L1_year, net_IPS_L1_day, $
           net_IPS_L1_begt, net_IPS_L1_endt, $
           net_IPS_L1_record, net_IPS_L1_rec_len, $
           net_IPS_NRM_filtyp, $
           net_num_IPS_energies, net_num_IPS_channels,$
           net_IPS_spins_per_sample, net_IPS_sectors_per_spin, $

           net_IES_L1_year, net_IES_L1_day, $
           net_IES_L1_begt, net_IES_L1_endt, $
           net_IES_L1_record, net_IES_L1_rec_len, $
           net_IES_NRM_filtyp, $
           net_num_IES_energies, net_num_IES_channels,$
           net_IES_spins_per_sample, net_IES_sectors_per_spin, $

           net_HISTe_L1_year, net_HISTe_L1_day, $
           net_HISTe_L1_begt, net_HISTe_L1_endt, $
           net_HISTe_L1_record, net_HISTe_L1_rec_len, $
           net_HISTe_NRM_filtyp, $
           net_num_HISTe_energies, net_num_HISTe_channels,$
           net_HISTe_spins_per_sample, net_HISTe_sectors_per_spin, $

           net_HISTp_L1_year, net_HISTp_L1_day, $
           net_HISTp_L1_begt, net_HISTp_L1_endt, $
           net_HISTp_L1_record, net_HISTp_L1_rec_len, $
           net_HISTp_NRM_filtyp, $
           net_num_HISTp_energies, net_num_HISTp_channels,$
           net_HISTp_spins_per_sample, net_HISTp_sectors_per_spin

         
;-----------------------------------------------------------------------


   idum = 0L	; Longword used in dummy read of heading records
   error_return = 0


;bugbug
;IF ( ( ths_spn_time GE 81751495 ) AND ( ths_spn_time LE 81929000 ) ) THEN BEGIN
;   print, ' APPEND_L1 0: '
;   print, 'Printing spin values between 81751495 and 81929000 '
;   print, ' Appending data to L1 file'
;   print, ' THS_SPN parameters
;   print, ths_spn_seq_no, ths_spn_min_frm, $
;          ths_spn_format, ths_spn_IPS_tab, $
;          ths_spn_IES_tab, ths_spn_IES_com, $
;          ths_spn_HIST_tab, ths_spn_HIST_com, $
;          ths_spn_SCOM, ths_spn_IPS_Mux, $
;          ths_spn_time, ths_spn_CRC
;ENDIF
;bugbug



;	Get estimates for any missing times in the data to be appended

      Estimate_times, Est_times, bad_time_est

;	IPS Level-1 file

   IF ( Use_IPS_L1 ) and ( IPS_L1_unit gt 0 ) and ( bad_time_est eq 0 ) $
                                                                      THEN BEGIN


;	Convert bytes in first half of time-block containing LONGWORD times 
;	from HOST to NETWORK format

      Replace_HOST_LONGs_with_NETWORK, num_source_spins, $
         ( IPS_trgt_tim_loc - IPS_trgt_tim_loc(0) ), IPS_trgt_blk

;	Append the new data from the first half of the 32-time array
;	to a new record in IPS the Level-1 file

      ON_IOERROR, Failed_IPS
      IPS_L1_record = IPS_L1_record + 1	; 'Complete' 16-time logical record no
      phs_rec_2nd_hlf = IPS_L1_record * 2L
      phs_rec_1st_hlf = phs_rec_2nd_hlf -1L
      IPS_L1_assoc(phs_rec_1st_hlf-1) = $ ; Write 1st half of 16-time block
            IPS_trgt_blk( 0 : (IPS_L1_rec_len / 2L) - 1L )
      IPS_L1_assoc(phs_rec_2nd_hlf-1) = $ ; Write remainder of 16-time block
            IPS_trgt_blk( (IPS_L1_rec_len / 2L) : IPS_L1_rec_len - 1L )


      L1_endt = LONG(Est_times(15))

;	Create NETWORK format version of LONGWORD endtime parameter and last
;	record number

      net_IPS_L1_record = IPS_L1_record
      net_IPS_L1_endt   = L1_endt 
      BYTEORDER, net_IPS_L1_record, $
                 net_IPS_L1_endt, /HTONL

;	Update the heading record

      POINT_LUN, IPS_L1_unit, 0	; Reset file pointer to beginning

      IF ( IPS_L1_record EQ 2 ) THEN BEGIN ; First data record just written

;	Get array filled with times (filled as necessary) for each of the
;	16 spins stored in Spin_blk at this time.
;	Define heading parameters 

         L1_year              = LONG(DRH.ATC_yr)
         L1_day               = LONG(DRH.ATC_day)
         IF NOT (bad_time_est) THEN BEGIN
            L1_begt           = LONG(Est_times(0))
         ENDIF ELSE BEGIN	; Current block somehow had NO defined times
            L1_begt           = 0L
         ENDELSE
         L1_DPU_TM_table      = LONG(ISHFT(ISHFT(ths_spn_format,4),-4))	; 4 LSB
         L1_TM_CRC            = LONG(ths_spn_CRC)

;	Create NETWORK format versions of LONG and SHORT integers parameters

         net_IPS_L1_year          = L1_year
         net_IPS_L1_day           = L1_day 
         net_IPS_L1_begt          = L1_begt 
         net_L1_DPU_TM_table      = L1_DPU_TM_table 
         net_L1_TM_CRC            = L1_TM_CRC 
         net_IPS_L1_rec_len       = IPS_L1_rec_len 
         net_IPS_Nrm_filtyp       = IPS_Nrm_filtyp 
         net_num_IPS_energies     = num_IPS_energies
         net_num_IPS_channels     = num_IPS_channels 
         net_IPS_spins_per_sample = IPS_spins_per_sample 
         net_IPS_sectors_per_spin = IPS_sectors_per_spin 
         BYTEORDER, net_IPS_L1_year, $
                    net_IPS_L1_day, $
                    net_IPS_L1_begt, /HTONL
         BYTEORDER, net_L1_DPU_TM_table, $
                    net_L1_TM_CRC, $
                    net_IPS_L1_rec_len, $
                    net_IPS_Nrm_filtyp, /HTONL
         BYTEORDER, net_num_IPS_energies, $
                    net_num_IPS_channels, $
                    net_IPS_spins_per_sample, $
                    net_IPS_sectors_per_spin, /HTONS

         WRITEU, IPS_L1_unit, $	; Write complete heading record
            net_IPS_L1_record, net_IPS_L1_rec_len, net_IPS_Nrm_filtyp, $
            net_IPS_L1_year, net_IPS_L1_day, $
            net_IPS_L1_begt, net_IPS_L1_endt, $
            net_L1_DPU_TM_table, net_L1_TM_CRC, $
            net_num_IPS_energies, net_num_IPS_channels, $
            net_IPS_spins_per_sample, net_IPS_sectors_per_spin 

      ENDIF

   ENDIF
   GOTO, Append_IES

Failed_IPS:
   print,'*** Append_L1 ***'
   print,'Error #',!err,' encountered during IPS Level-1 file update'
   print,!error
   error_return = error_return + 1


;	IES Level-1 file

Append_IES:
   IF ( Use_IES_L1 ) and ( IES_L1_unit gt 0 ) and ( bad_time_est eq 0 ) $
                                                                      THEN BEGIN


;	Convert bytes in first half of time-block containing LONGWORD times 
;	from HOST to NETWORK format

      Replace_HOST_LONGs_with_NETWORK, num_source_spins, $
         ( IES_trgt_tim_loc - IES_trgt_tim_loc(0) ), IES_trgt_blk

;	Append the new data from the first half of the 32-time array
;	to a new record in IES the Level-1 file

      ON_IOERROR, Failed_IES
      IES_L1_record = IES_L1_record + 1	; 'Complete' 16-time logical record no
      phs_rec_2nd_hlf = IES_L1_record * 2L
      phs_rec_1st_hlf = phs_rec_2nd_hlf -1L
      IES_L1_assoc(phs_rec_1st_hlf-1) = $ ; Write 1st half of 16-time block
            IES_trgt_blk( 0 : (IES_L1_rec_len / 2L) - 1L )
      IES_L1_assoc(phs_rec_2nd_hlf-1) = $ ; Write remainder of 16-time block
            IES_trgt_blk( (IES_L1_rec_len / 2L) : IES_L1_rec_len - 1L )


         L1_endt = LONG(Est_times(15))

;	Create NETWORK format version of LONGWORD endtime parameter and last
;	record number

         net_IES_L1_record = IES_L1_record
         net_IES_L1_endt   = L1_endt 
         BYTEORDER, net_IES_L1_record, $
                    net_IES_L1_endt, /HTONL

;	Update the heading record

      POINT_LUN, IES_L1_unit, 0	; Reset file pointer to beginning

      IF ( IES_L1_record EQ 2 ) THEN BEGIN ; First data record just written

;	Get array filled with times (filled as necessary) for each of the
;	16 spins stored in Spin_blk at this time.
;	Define heading parameters 

         L1_year              = LONG(DRH.ATC_yr)
         L1_day               = LONG(DRH.ATC_day)
         IF NOT (bad_time_est) THEN BEGIN
            L1_begt           = LONG(Est_times(0))
         ENDIF ELSE BEGIN	; Current block somehow had NO defined times
            L1_begt           = 0L
         ENDELSE
         L1_DPU_TM_table      = LONG(ISHFT(ISHFT(ths_spn_format,4),-4))	; 4 LSB
         L1_TM_CRC            = LONG(ths_spn_CRC)

;	Create NETWORK format versions of LONG and SHORT integers parameters

         net_IES_L1_year          = L1_year
         net_IES_L1_day           = L1_day 
         net_IES_L1_begt          = L1_begt 
         net_L1_DPU_TM_table      = L1_DPU_TM_table 
         net_L1_TM_CRC            = L1_TM_CRC 
         net_IES_L1_rec_len       = IES_L1_rec_len 
         net_IES_Nrm_filtyp       = IES_Nrm_filtyp 
         net_num_IES_energies     = num_IES_energies
         net_num_IES_channels     = num_IES_channels 
         net_IES_spins_per_sample = IES_spins_per_sample 
         net_IES_sectors_per_spin = IES_sectors_per_spin 
         BYTEORDER, net_IES_L1_year, $
                    net_IES_L1_day, $
                    net_IES_L1_begt, /HTONL
         BYTEORDER, net_L1_DPU_TM_table, $
                    net_L1_TM_CRC, $
                    net_IES_L1_rec_len, $
                    net_IES_Nrm_filtyp, /HTONL
         BYTEORDER, net_num_IES_energies, $
                    net_num_IES_channels, $
                    net_IES_spins_per_sample, $
                    net_IES_sectors_per_spin, /HTONS

         WRITEU, IES_L1_unit, $	; Write complete heading record
            net_IES_L1_record, net_IES_L1_rec_len, net_IES_Nrm_filtyp, $
            net_IES_L1_year, net_IES_L1_day, $
            net_IES_L1_begt, net_IES_L1_endt, $
            net_L1_DPU_TM_table, net_L1_TM_CRC, $
            net_num_IES_energies, net_num_IES_channels, $
            net_IES_spins_per_sample, net_IES_sectors_per_spin 

      ENDIF

   ENDIF
   GOTO, Append_HISTe

Failed_IES:
   print,'*** Append_L1 ***'
   print,'Error #',!err,' encountered during IES Level-1 file update'
   print,!error
   error_return = error_return + 1


;	HISTe Level-1 file

Append_HISTe:
   IF ( Use_HISTe_L1 ) and ( HISTe_L1_unit gt 0 ) and ( bad_time_est eq 0 ) $
                                                                      THEN BEGIN


;	Convert bytes in first half of time-block containing LONGWORD times 
;	from HOST to NETWORK format

      Replace_HOST_LONGs_with_NETWORK, num_source_spins, $
         ( HISTe_trgt_tim_loc - HISTe_trgt_tim_loc(0) ), HISTe_trgt_blk

;	Append the new data from the first half of the 32-time array
;	to a new record in HISTe the Level-1 file

      ON_IOERROR, Failed_HISTe
      HISTe_L1_record = HISTe_L1_record + 1	; 'Complete' 16-time logical record no
      phs_rec_2nd_hlf = HISTe_L1_record * 2L
      phs_rec_1st_hlf = phs_rec_2nd_hlf -1L
      HISTe_L1_assoc(phs_rec_1st_hlf-1) = $ ; Write 1st half of 16-time block
            HISTe_trgt_blk( 0 : (HISTe_L1_rec_len / 2L) - 1L )
      HISTe_L1_assoc(phs_rec_2nd_hlf-1) = $ ; Write remainder of 16-time block
            HISTe_trgt_blk( (HISTe_L1_rec_len / 2L) : HISTe_L1_rec_len - 1L )


         L1_endt = LONG(Est_times(15))

;	Create NETWORK format version of LONGWORD endtime parameter and last
;	record number

         net_HISTe_L1_record = HISTe_L1_record
         net_HISTe_L1_endt   = L1_endt 
         BYTEORDER, net_HISTe_L1_record, $
                    net_HISTe_L1_endt, /HTONL

;	Update the heading record

      POINT_LUN, HISTe_L1_unit, 0	; Reset file pointer to beginning

      IF ( HISTe_L1_record EQ 2 ) THEN BEGIN ; First data record just written

;	Get array filled with times (filled as necessary) for each of the
;	16 spins stored in Spin_blk at this time.
;	Define heading parameters 

         L1_year              = LONG(DRH.ATC_yr)
         L1_day               = LONG(DRH.ATC_day)
         IF NOT (bad_time_est) THEN BEGIN
            L1_begt           = LONG(Est_times(0))
         ENDIF ELSE BEGIN	; Current block somehow had NO defined times
            L1_begt           = 0L
         ENDELSE
         L1_DPU_TM_table      = LONG(ISHFT(ISHFT(ths_spn_format,4),-4))	; 4 LSB
         L1_TM_CRC            = LONG(ths_spn_CRC)

;	Create NETWORK format versions of LONG and SHORT integers parameters

         net_HISTe_L1_year          = L1_year
         net_HISTe_L1_day           = L1_day 
         net_HISTe_L1_begt          = L1_begt 
         net_L1_DPU_TM_table        = L1_DPU_TM_table 
         net_L1_TM_CRC              = L1_TM_CRC 
         net_HISTe_L1_rec_len       = HISTe_L1_rec_len 
         net_HISTe_Nrm_filtyp       = HISTe_Nrm_filtyp 
         net_num_HISTe_energies     = num_HISTe_energies
         net_num_HISTe_channels     = num_HISTe_channels 
         net_HISTe_spins_per_sample = HISTe_spins_per_sample 
         net_HISTe_sectors_per_spin = HISTe_sectors_per_spin 
         BYTEORDER, net_HISTe_L1_year, $
                    net_HISTe_L1_day, $
                    net_HISTe_L1_begt, /HTONL
         BYTEORDER, net_L1_DPU_TM_table, $
                    net_L1_TM_CRC, $
                    net_HISTe_L1_rec_len, $
                    net_HISTe_Nrm_filtyp, /HTONL
         BYTEORDER, net_num_HISTe_energies, $
                    net_num_HISTe_channels, $
                    net_HISTe_spins_per_sample, $
                    net_HISTe_sectors_per_spin, /HTONS

         WRITEU, HISTe_L1_unit, $	; Write complete heading record
            net_HISTe_L1_record, net_HISTe_L1_rec_len, net_HISTe_Nrm_filtyp, $
            net_HISTe_L1_year, net_HISTe_L1_day, $
            net_HISTe_L1_begt, net_HISTe_L1_endt, $
            net_L1_DPU_TM_table, net_L1_TM_CRC, $
            net_num_HISTe_energies, net_num_HISTe_channels, $
            net_HISTe_spins_per_sample, net_HISTe_sectors_per_spin 

      ENDIF

   ENDIF
   GOTO, Append_HISTp

Failed_HISTe:
   print,'*** Append_L1 ***'
   print,'Error #',!err,' encountered during HISTe Level-1 file update'
   print,!error
   error_return = error_return + 1


;	HISTp Level-1 file

Append_HISTp:
   IF ( Use_HISTp_L1 ) and ( HISTp_L1_unit gt 0 ) and ( bad_time_est eq 0 ) $
                                                                      THEN BEGIN


;	Convert bytes in first half of time-block containing LONGWORD times 
;	from HOST to NETWORK format

      Replace_HOST_LONGs_with_NETWORK, num_source_spins, $
         ( HISTp_trgt_tim_loc - HISTp_trgt_tim_loc(0) ), HISTp_trgt_blk

;	Append the new data from the first half of the 32-time array
;	to a new record in HISTp the Level-1 file

      ON_IOERROR, Failed_HISTp

      HISTp_L1_record = HISTp_L1_record + 1	; 'Complete' 16-time logical record no
      phs_rec_2nd_hlf = HISTp_L1_record * 2L
      phs_rec_1st_hlf = phs_rec_2nd_hlf -1L
      HISTp_L1_assoc(phs_rec_1st_hlf-1) = $ ; Write 1st half of 16-time block
            HISTp_trgt_blk( 0 : (HISTp_L1_rec_len / 2L) - 1L )
      HISTp_L1_assoc(phs_rec_2nd_hlf-1) = $ ; Write remainder of 16-time block
            HISTp_trgt_blk( (HISTp_L1_rec_len / 2L) : HISTp_L1_rec_len - 1L )

         L1_endt = LONG(Est_times(15))

;	Create NETWORK format version of LONGWORD endtime parameter and last
;	record number

         net_HISTp_L1_record = HISTp_L1_record
         net_HISTp_L1_endt   = L1_endt 
         BYTEORDER, net_HISTp_L1_record, $
                    net_HISTp_L1_endt, /HTONL


;	Update the heading record

      POINT_LUN, HISTp_L1_unit, 0	; Reset file pointer to beginning

      IF ( HISTp_L1_record EQ 2 ) THEN BEGIN ; First data record just written

;	Get array filled with times (filled as necessary) for each of the
;	16 spins stored in Spin_blk at this time.
;	Define heading parameters 

         L1_year              = LONG(DRH.ATC_yr)
         L1_day               = LONG(DRH.ATC_day)
         IF NOT (bad_time_est) THEN BEGIN
            L1_begt           = LONG(Est_times(0))
         ENDIF ELSE BEGIN	; Current block somehow had NO defined times
            L1_begt           = 0L
         ENDELSE
         L1_DPU_TM_table      = LONG(ISHFT(ISHFT(ths_spn_format,4),-4))	; 4 LSB
         L1_TM_CRC            = LONG(ths_spn_CRC)

;	Create NETWORK format versions of LONG and SHORT integers parameters

         net_HISTp_L1_year          = L1_year
         net_HISTp_L1_day           = L1_day 
         net_HISTp_L1_begt          = L1_begt 
         net_L1_DPU_TM_table        = L1_DPU_TM_table 
         net_L1_TM_CRC              = L1_TM_CRC 
         net_HISTp_L1_rec_len       = HISTp_L1_rec_len 
         net_HISTp_Nrm_filtyp       = HISTp_Nrm_filtyp 
         net_num_HISTp_energies     = num_HISTp_energies
         net_num_HISTp_channels     = num_HISTp_channels 
         net_HISTp_spins_per_sample = HISTp_spins_per_sample 
         net_HISTp_sectors_per_spin = HISTp_sectors_per_spin 
         BYTEORDER, net_HISTp_L1_year, $
                    net_HISTp_L1_day, $
                    net_HISTp_L1_begt, /HTONL
         BYTEORDER, net_L1_DPU_TM_table, $
                    net_L1_TM_CRC, $
                    net_HISTp_L1_rec_len, $
                    net_HISTp_Nrm_filtyp, /HTONL
         BYTEORDER, net_num_HISTp_energies, $
                    net_num_HISTp_channels, $
                    net_HISTp_spins_per_sample, $
                    net_HISTp_sectors_per_spin, /HTONS

         WRITEU, HISTp_L1_unit, $	; Write complete heading record
            net_HISTp_L1_record, net_HISTp_L1_rec_len, net_HISTp_Nrm_filtyp, $
            net_HISTp_L1_year, net_HISTp_L1_day, $
            net_HISTp_L1_begt, net_HISTp_L1_endt, $
            net_L1_DPU_TM_table, net_L1_TM_CRC, $
            net_num_HISTp_energies, net_num_HISTp_channels, $
            net_HISTp_spins_per_sample, net_HISTp_sectors_per_spin 

      ENDIF

   ENDIF
   GOTO, Exit_Append_L1

Failed_HISTp:
   print,'*** Append_L1 ***'
   print,'Error #',!err,' encountered during HISTp Level-1 file update'
   print,!error
   error_return = error_return + 1


Exit_Append_L1:
   return
end





pro Append_SH, error_return

;	Write a record containing the evaluated time and the spin header
;	information for each spin containing heading heading information
;	which differs from its predecessor.  The file should also contain
;	records for the first spin and the last spin with valid time 
;	evaluations from the Level-0 data.  Only write a record if the
;	time value to be written is GE 0.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1-	File not open or Write error
;						encountered

;	revision
;	15 Sep 1995	M. T. Redding	Added code to convert HOST format
;					LONGWORDs to NETWORK format before
;					writing to file. 
;					Added new variables ('net_...') or
;					renamed others so to maintain 
;					independence of existing common block 
;					variables
;	20 Sep 1995	M. T. Redding	Added common block 'CEP_Spin_Head_net'
;					to contain all the Spin header file
;					heading record variables.
;					Renamed some variables to include the
;					characters 'SH_' to differentiate them
;					from similar heading variables stored
;					in other common blocks.
;					Replaced heading record update code
;					with execution of procedure
;					'Update_SH_header'
;	25 Sep 1995	M. T. Redding	Commented out executions of 
;					'Update_SH_Header',
;					  - revision of 
;					the header record is now done only 
;					prior to closing the file instead of 
;					after each new appended record.
;	01 Apr 1996	M.T.Redding	Commented out section which check for 
;					changes in spin header values - the
;					Spin header file should now contain
;					records corresponding to EACH spin for
;					which a spin time has been successfully
;					evaluated - this is necessary so that
;					times may be evaluated for values 
;					stored in the Housekeeping file which 
;					are referenced to the CEPPAD spin #.
;					Expanded variable 'temp' to 13 bytes
;					and added line to write SCOM value
;					into the extra byte - NOTE: THE ORIGINAL
;					ORDERING OF THE CEPPAD SPIN HEADER INFO
;					HASS NOT BEEN RETAINED IN THE SPIN 
;					HEADER FILE. SCOM IS BYTE 11 IN THE 
;					TELEMETRY BUT BYTE 12 IN THE SPIN HEADER
;					FILE - THIS WAS DONE TO ALLOW BACKWARD 
;					COMPATIBILTY FOR SOFTWARE USING THE 
;					SPIN HEADER FILE BEFORE THIS REVISION.
;	05 Jun 1996	M.T.Redding	Parameterized sizing of 'temp' array
;					to use 'SH_rec_len' to transparently
;					reflect the increase in SH record size
;					done May 30 - also added dummy fill of
;					14th temp array value as a hook for
;					filling in a 'spin-quality' flag

;--------------------------------------------------------------------

   COMMON CEP_Spin_head,   SH_unit, SH_record, SH_assoc, SH_rec_len

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

   COMMON CEP_lst_spn,     lst_spn_seq_no, lst_spn_min_frm, $
                           lst_spn_format, lst_spn_IPS_tab, $
                           lst_spn_IES_tab, lst_spn_IES_com, $
                           lst_spn_HIST_tab, lst_spn_HIST_com, $
                           lst_spn_SCOM, lst_spn_IPS_Mux, $
                           lst_spn_time, lst_spn_CRC, lst_spn_quality

   COMMON Level_1_fil_typ, HK_filtyp, IPS_Nrm_filtyp, IES_Nrm_filtyp, $
                           HISTe_Nrm_filtyp, HISTp_Nrm_filtyp, $
                           IPS_Rts_filtyp, IES_Rts_filtyp, HIST_Rts_filtyp, $
                           Spin_Head_filtyp

   COMMON CEP_Spin_head_net, net_SH_L1_record, net_SH_L1_rec_len, $
                           net_Spin_Head_filtyp, net_SH_L1_year, $
                           net_SH_L1_day, net_SH_L1_begt, $
                           net_SH_L1_endt

;-----------------------------------------------------------------------


   idum = 0L	; Longword used in dummy read of heading records
   error_return = 0


;	Open up the Spin header file if this is the first time it is 
;	has been accessed

   IF ( SH_unit LE -3 ) THEN $
        Open_SH, error_return


;	Append a record with revised spin header values

   IF ( SH_unit gt 0 ) THEN BEGIN


;	Check for a valid time

      IF ( ths_spn_time GE 0L ) THEN BEGIN


;	Create NETWORK format version of time and HIST command state parameters

      net_ths_spn_time     = ths_spn_time
      net_ths_spn_HIST_com = ths_spn_HIST_com
      BYTEORDER, net_ths_spn_time,     /HTONL
      BYTEORDER, net_ths_spn_HIST_com, /HTONS

;	Transfer heading values to temporary array for writing to file using
;	array "associated' with the file

            temp = BYTARR( SH_rec_len )
            temp(  0 )  = BYTE(net_ths_spn_time,0,4)
            temp(  4 )  = ths_spn_format
            temp(  5 )  = ths_spn_IPS_tab
            temp(  6 )  = ths_spn_IES_tab
            temp(  7 )  = ths_spn_IES_com
            temp(  8 )  = ths_spn_HIST_tab
            temp(  9 )  = BYTE(net_ths_spn_HIST_com,0,2)
            temp( 11 )  = ths_spn_IPS_Mux
            temp( 12 )  = ths_spn_SCOM
            temp( 13 )  = ths_spn_quality
;;            temp( 13 )  = 0B
            net_SH_L1_endt = LONG(ths_spn_time)


;	Append the new data from the first half of the 32-time array
;	to a new record in the Spin Header Level-1 file

            ON_IOERROR, Failed_SH
            SH_record = SH_record + 1
            SH_assoc(SH_record-1) = Temp(*)

;	Create NETWORK format versions of LONG and SHORT word parameters

               net_SH_L1_record = SH_record
               BYTEORDER, net_SH_L1_record, $
                          net_SH_L1_endt, /HTONL

;	Update the heading record

            POINT_LUN, SH_unit, 0	; Reset file pointer to beginning

            IF ( SH_record EQ 4 ) THEN BEGIN ; First data record just written

;	Get array filled with times (filled as necessary) for each of the
;	16 spins stored in Spin_blk at this time.
;	Define heading parameters 

               net_SH_L1_year = LONG(DRH.ATC_yr)
               net_SH_L1_day  = LONG(DRH.ATC_day)
               net_SH_L1_begt = net_SH_L1_endt

;	Create NETWORK format versions of LONG and SHORT word parameters

               net_SH_L1_rec_len    = SH_rec_len
               net_Spin_Head_filtyp = Spin_Head_filtyp

               BYTEORDER, net_SH_L1_rec_len, $
                          net_Spin_Head_filtyp, $
                          net_SH_L1_year, $
                          net_SH_L1_day, /HTONL

;		Write complete heading records
               WRITEU, SH_unit, $
                          net_SH_L1_record, net_SH_L1_rec_len, $
                          net_Spin_Head_filtyp, 0B, 0B
               WRITEU, SH_unit, $
                          net_SH_L1_year, net_SH_L1_day, net_SH_L1_begt, 0B, 0B
               WRITEU, SH_unit, $
                          net_SH_L1_endt, REPLICATE( 0B, 10 )
      
            ENDIF

      ENDIF	; Valid time value ( GE 0L )
      error_return = 0
      RETURN

   ENDIF $	; Spin header file open and flag says to use it
   ELSE BEGIN
      print, ' Attempt to update Spin header file before file is opened'
      error_return = 1
      RETURN
   ENDELSE

Failed_SH:
   print, '*** Append_SH ***'
   print, 'Error #', !err, ' encountered during Spin header Level-1 file update'
   print, !error
   error_return = error_return + 1
   RETURN
END





pro Append_Rates, error_return

;	Write one more record of Rates data to the appropriate Level-1
;	Rates data file.  Create (if this is the first record of data
;	written) or update the heading record with the revised number of 
;	records and end time for the file.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed Write to either IPS,
;						IES or HIST Rates Level-1 file

;	revision
;	15 Sep 1995	M. T. Redding	Added common block CEP_Rates_Lv1_net
;					and code to convert HOST format LONG
;					and SHORT word parameters before writing
;					to the Level-1 file
;	20 Sep 1995	M. T. Redding	Expanded CEP_Rates_Lvl_net common block
;					parameters to include all values to be
;					written to the heading record.  Renamed
;					some of the variables to include the
;					characters 'IPS_Rts_', 'IES_Rts_' or
;					'HIST_Rts_'
;					Replaced code updating the heading
;					records of each of the Rates_L1 files
;					with execution of procedures to do this.
;	25 Sep 1995	M. T. Redding	Commented out executions of 
;					'Update_IPS_Rates_Header',
;					'Update_IES_Rates_Header' and
;					'Update_HIST_Rates_Header'
;					  - revision of 
;					the header records are now done only 
;					prior to closing the file instead of 
;					after each new appended record.
;--------------------------------------------------------------------

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

   COMMON Level_1_fil_typ, HK_filtyp, IPS_Nrm_filtyp, IES_Nrm_filtyp, $
                           HISTe_Nrm_filtyp, HISTp_Nrm_filtyp, $
                           IPS_Rts_filtyp, IES_Rts_filtyp, HIST_Rts_filtyp, $
                           Spin_Head_filtyp

   COMMON CEP_Rates_Lvl_1, IPS_Rates_unit, IPS_Rates_rec, $
                           IPS_Rates_assoc, IPS_Rates_recl, $
                           IPS_Rates_format, $
                           IES_Rates_unit, IES_Rates_rec, $
                           IES_Rates_assoc, IES_Rates_recl, $
                           IES_Rates_format, $
                           HIST_Rates_unit, HIST_Rates_rec, $
                           HIST_Rates_assoc, HIST_Rates_recl, $
                           HIST_Rates_format

   COMMON CEP_Rates_Lvl_net, $
                           net_IPS_Rates_rec, $
                           net_IPS_Rates_recl, $
                           net_IPS_Rates_format, $
                           net_IPS_Rts_filtyp, $
                           net_IPS_Rts_L1_year, $
                           net_IPS_Rts_L1_day, $
                           net_IPS_Rts_L1_begt, $
                           net_IPS_Rts_L1_endt, $

                           net_IES_Rates_rec, $
                           net_IES_Rates_recl, $
                           net_IES_Rates_format, $
                           net_IES_Rts_filtyp,$
                           net_IES_Rts_L1_year, $
                           net_IES_Rts_L1_day, $
                           net_IES_Rts_L1_begt, $
                           net_IES_Rts_L1_endt, $

                           net_HIST_Rates_rec, $
                           net_HIST_Rates_recl, $
                           net_HIST_Rates_format, $
                           net_HIST_Rts_filtyp, $
                           net_HIST_Rts_L1_year, $
                           net_HIST_Rts_L1_day, $
                           net_HIST_Rts_L1_begt, $
                           net_HIST_Rts_L1_endt

;-----------------------------------------------------------------------


   idum = 0L	; Longword used in dummy read of heading records
   error_return = 0


;			IPS RATES-MODE SPIN DATA
;	Write to IPS Rates-mode Level-1 file if it is open and the spin data
;	is IPS Rates-mode data and if the spin data has a non-zero time

   IF ( (( ths_spn_format EQ IPS_Rates_format ) AND $
         ( IPS_Rates_unit GT 0 )) AND $
         ( ths_spn_time GE 0L ) ) $
                                                                    THEN BEGIN


;	Create NETWORK format versions of time and HIST command state parameters

      net_ths_spn_time     = ths_spn_time
      net_ths_spn_HIST_com = ths_spn_HIST_com
      BYTEORDER, net_ths_spn_time,     /HTONL
      BYTEORDER, net_ths_spn_HIST_com, /HTONS

;	Reorganize and transfer data to a single byte array

      Temp_array = BYTARR(IPS_Rates_recl)
      Temp_array(  0:  3 ) = BYTE(net_ths_spn_time,0,4)	; Evaluated time
      Temp_array(  4:  4 ) = ths_spn_seq_no		; Major frm sequence no
      Temp_array(  5:  5 ) = ths_spn_min_frm		; Minor frame number
      Temp_array(  6:  6 ) = ths_spn_IPS_tab		; IPS Config table no
      Temp_array(  7:  7 ) = ths_spn_IES_tab		; IES Lookup table no
      Temp_array(  8:  8 ) = ths_spn_IES_com		; IES Command state
      Temp_array(  9:  9 ) = ths_spn_HIST_tab		; HIST Lookup table no
      Temp_array( 10: 11 ) = BYTE(net_ths_spn_HIST_com,0,2) ; HIST Command state
      Temp_array( 12: 12 ) = ths_spn_SCOM		; CEPPAD spin counter
      Temp_array( 13: 13 ) = ths_spn_IPS_Mux		; IPS Mux state
      Temp_array( 14:523 ) = ths_spn_data( 0: 509 )	; Rates-Histogram values

;	Create NETWORK format version of end time for heading records

      net_IPS_Rts_L1_endt = LONG(ths_spn_time)
      BYTEORDER, net_IPS_Rts_L1_endt, /HTONL

;	Append the new data from to a new record in IPS Rates-mode Level-1 file

      ON_IOERROR, Failed_IPS_Rates
      IPS_Rates_rec = IPS_Rates_rec + 1
      IPS_Rates_assoc(IPS_Rates_rec-1) = $
                       Temp_array( 0 : IPS_Rates_recl -1 )

;	Update the heading record

      POINT_LUN, IPS_Rates_unit, 0	; Reset file pointer to beginning

      net_IPS_Rates_rec = IPS_Rates_rec
      BYTEORDER, net_IPS_Rates_rec, /HTONL

      IF ( IPS_Rates_rec EQ 2 ) THEN BEGIN ; First data record just written

;	Define more NETWORK format versions of heading record parameters 

         net_IPS_Rates_recl   = IPS_Rates_recl
         net_IPS_Rts_filtyp   = IPS_Rts_filtyp
         net_IPS_Rts_L1_year  = LONG(DRH.ATC_yr)
         net_IPS_Rts_L1_day   = LONG(DRH.ATC_day)
         net_IPS_Rts_L1_begt  = net_IPS_Rts_L1_endt	; Set for first data rec

         BYTEORDER, net_IPS_Rates_recl, $
                    net_IPS_Rts_filtyp, $
                    net_IPS_Rts_L1_year, $
                    net_IPS_Rts_L1_day, /HTONL

         WRITEU, IPS_Rates_unit, $	; Write complete heading record
                    net_IPS_Rates_rec, net_IPS_Rates_recl, net_IPS_Rts_filtyp, $
                    net_IPS_Rts_L1_year, net_IPS_Rts_L1_day, $
                    net_IPS_Rts_L1_begt, net_IPS_Rts_L1_endt

      ENDIF

      RETURN

Failed_IPS_rates:
      print,'*** Append_Rates ***"
      print,'Error #',!err, $
               ' encountered during IPS Rates-mode Level-1 file update'
      print,!error
      error_return = 1
      RETURN

   ENDIF ; ( ( ths_spn_format EQ IPS_Rates_format ) AND ( IPS_Rates_unit gt 0 ) ) 


;			IES RATES-MODE SPIN DATA
;	Write to IES Rates-mode Level-1 file if it is open and the spin data
;	is IES Rates-mode data and if the spin data has a non-zero time

   IF ( (( ths_spn_format EQ IES_Rates_format ) AND $
         ( IES_Rates_unit GT 0 )) AND $
         ( ths_spn_time GE 0L ) ) $
                                                                    THEN BEGIN


;	Create NETWORK format versions of time and HIST command state parameters

      net_ths_spn_time     = ths_spn_time
      net_ths_spn_HIST_com = ths_spn_HIST_com
      BYTEORDER, net_ths_spn_time,     /HTONL
      BYTEORDER, net_ths_spn_HIST_com, /HTONS

;	Reorganize and transfer data to a single byte array

      Temp_array = BYTARR(IES_Rates_recl)
      Temp_array(  0:  3 ) = BYTE(net_ths_spn_time,0,4)	; Evaluated time
      Temp_array(  4:  4 ) = ths_spn_seq_no		; Major frm sequence no
      Temp_array(  5:  5 ) = ths_spn_min_frm		; Minor frame number
      Temp_array(  6:  6 ) = ths_spn_IPS_tab		; IPS Config table no
      Temp_array(  7:  7 ) = ths_spn_IES_tab		; IES Lookup table no
							; = 10h + channel number
							; for data
      Temp_array(  8:  8 ) = ths_spn_IES_com		; IES Command state
      Temp_array(  9:  9 ) = ths_spn_HIST_tab		; HIST Lookup table no
      Temp_array( 10: 11 ) = BYTE(net_ths_spn_HIST_com,0,2) ; HIST Command state
      Temp_array( 12: 12 ) = ths_spn_SCOM		; CEPPAD spin counter
      Temp_array( 13: 13 ) = ths_spn_IPS_Mux		; IPS Mux state
      Temp_array( 14:781 ) = ths_spn_data( 0: 767 )	; Rates-Histogram values

;	Create NETWORK format version of end time for heading records

      net_IES_Rts_L1_endt = LONG(ths_spn_time)
      BYTEORDER, net_IES_Rts_L1_endt, /HTONL

;	Append the new data to a new record in IES Rates-mode Level-1 file

      ON_IOERROR, Failed_IES_Rates
      IES_Rates_rec = IES_Rates_rec + 1
      IES_Rates_assoc(IES_Rates_rec-1) = $
                       Temp_array( 0 : IES_Rates_recl -1 )

;	Update the heading record

      POINT_LUN, IES_Rates_unit, 0	; Reset file pointer to beginning

      net_IES_Rates_rec = IES_Rates_rec
      BYTEORDER, net_IES_Rates_rec, /HTONL

      IF ( IES_Rates_rec EQ 2 ) THEN BEGIN ; First data record just written

;	Define more NETWORK format versions of heading record parameters 

         net_IES_Rates_recl   = IES_Rates_recl
         net_IES_Rts_filtyp   = IES_Rts_filtyp
         net_IES_Rts_L1_year  = LONG(DRH.ATC_yr)
         net_IES_Rts_L1_day   = LONG(DRH.ATC_day)
         net_IES_Rts_L1_begt  = net_IES_Rts_L1_endt	; Set for first data rec

         BYTEORDER, net_IES_Rates_recl, $
                    net_IES_Rts_filtyp, $
                    net_IES_Rts_L1_year, $
                    net_IES_Rts_L1_day, /HTONL

         WRITEU, IES_Rates_unit, $	; Write complete heading record
                    net_IES_Rates_rec, net_IES_Rates_recl, net_IES_Rts_filtyp, $
                    net_IES_Rts_L1_year, net_IES_Rts_L1_day, $
                    net_IES_Rts_L1_begt, net_IES_Rts_L1_endt

      ENDIF

      RETURN

Failed_IES_rates:
      print,'*** Append_Rates ***"
      print,'Error #',!err, $
               ' encountered during IES Rates-mode Level-1 file update'
      print,!error
      error_return = 1
      RETURN

   ENDIF ; ( ( ths_spn_format EQ IES_Rates_format ) AND ( IES_Rates_unit gt 0 ) ) 


;			HIST RATES-MODE SPIN DATA
;	Write to HIST Rates-mode Level-1 file if it is open and the spin data
;	is HIST Rates-mode data and if the spin data has a non-zero time

   IF ( (( ths_spn_format EQ HIST_Rates_format ) AND $
         ( HIST_Rates_unit GT 0 )) AND $
         ( ths_spn_time GE 0L ) ) $
                                                                    THEN BEGIN


;	Create NETWORK format versions of time and HIST command state parameters

      net_ths_spn_time     = ths_spn_time
      net_ths_spn_HIST_com = ths_spn_HIST_com
      BYTEORDER, net_ths_spn_time,     /HTONL
      BYTEORDER, net_ths_spn_HIST_com, /HTONS

;	Reorganize and transfer data to a single byte array

      Temp_array = BYTARR(HIST_Rates_recl)
      Temp_array(  0:  3 ) = BYTE(net_ths_spn_time,0,4)	; Evaluated time
      Temp_array(  4:  4 ) = ths_spn_seq_no		; Major frm sequence no
      Temp_array(  5:  5 ) = ths_spn_min_frm		; Minor frame number
      Temp_array(  6:  6 ) = ths_spn_IPS_tab		; IPS Config table no
      Temp_array(  7:  7 ) = ths_spn_IES_tab		; IES Lookup table no
      Temp_array(  8:  8 ) = ths_spn_IES_com		; IES Command state
      Temp_array(  9:  9 ) = ths_spn_HIST_tab		; HIST Lookup table no
							; = 10h + channel number
							; for data
      Temp_array( 10: 11 ) = BYTE(net_ths_spn_HIST_com,0,2) ; HIST Command state
      Temp_array( 12: 12 ) = ths_spn_SCOM		; CEPPAD spin counter
      Temp_array( 13: 13 ) = ths_spn_IPS_Mux		; IPS Mux state
      Temp_array( 14:781 ) = ths_spn_data(  0: 767 )	; Rates-Histogram values

;	Create NETWORK format version of end time for heading records

      net_HIST_Rts_L1_endt = LONG(ths_spn_time)
      BYTEORDER, net_HIST_Rts_L1_endt, /HTONL

;	Append the new data to a new record in HIST Rates-mode Level-1 file

      ON_IOERROR, Failed_HIST_Rates
      HIST_Rates_rec = HIST_Rates_rec + 1
      HIST_Rates_assoc(HIST_Rates_rec-1) = $
                       Temp_array( 0 : HIST_Rates_recl -1 )

;	Update the heading record

      POINT_LUN, HIST_Rates_unit, 0	; Reset file pointer to beginning

      net_HIST_Rates_rec = HIST_Rates_rec
      BYTEORDER, net_HIST_Rates_rec, /HTONL

      IF ( HIST_Rates_rec EQ 2 ) THEN BEGIN ; First data record just written

;	Define more NETWORK format versions of heading record parameters 

         net_HIST_Rates_recl   = HIST_Rates_recl
         net_HIST_Rts_filtyp   = HIST_Rts_filtyp
         net_HIST_Rts_L1_year  = LONG(DRH.ATC_yr)
         net_HIST_Rts_L1_day   = LONG(DRH.ATC_day)
         net_HIST_Rts_L1_begt  = net_HIST_Rts_L1_endt		; Set for first data rec

         BYTEORDER, net_HIST_Rates_recl, $
                    net_HIST_Rts_filtyp, $
                    net_HIST_Rts_L1_year, $
                    net_HIST_Rts_L1_day, /HTONL

         WRITEU, HIST_Rates_unit, $	; Write complete heading record
                 net_HIST_Rates_rec, net_HIST_Rates_recl, net_HIST_Rts_filtyp, $
                 net_HIST_Rts_L1_year, net_HIST_Rts_L1_day, $
                 net_HIST_Rts_L1_begt, net_HIST_Rts_L1_endt

      ENDIF

      RETURN

Failed_HIST_rates:
      print,'*** Append_Rates ***"
      print,'Error #',!err, $
               ' encountered during HIST Rates-mode Level-1 file update'
      print,!error
      error_return = 1
      RETURN

   ENDIF ; ( ( ths_spn_format EQ HIST_Rates_format ) AND ( HIST_Rates_unit gt 0 )) 



END




PRO CEP_REC_PARAMS, FLR, IERROR

;	This procedure extracts (sets via hard-code) from the File Label 
;	Record (FLR) structure, the physical logical record characteristics 
;	which define the major frame attributes needed for processing a
;	CEPPAD Level-0 data file.

;	Values t be determined:
;	cep_byts_rec	Bytes per physical record
;	cep_byts_mnfr	Bytes per minor frame
;	max_phys_rec	Maximum physical records

;		Passed Parameters

;	FLR
;	IERROR		Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Error

;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block
;					Commented out the statements which
;					'hard-coded' values for 'cep_byts-rec',
;					'max_phys_rec' and 'cep_byts_mnfrm' -
;					added diagnostic statements to see what
;					values are read from the file.
; ----------------------------------------------------------------

 common cep_phys_rec, cep_byts_rec, cep_byts_mnfr, max_phys_rec

COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                        sci_byt_per_spin, max_spins, $
                        nom_spin_dur, TRUE, FALSE, sync_byte, $
                        CEP_Sci_Man_byt_per_mnfrm, $
                        CEP_Conting_byt_per_mnfrm, $
                        CEP_Sci_Man_rec_len, $
                        CEP_Conting_rec_len

; ----------------------------------------------------------------

ierror = 0

;        Bytes per physical record
cep_byts_rec = flr.phys_rec_len
;cep_byts_rec = 4800		; ** Hard-coded to allow use of incorrectly
				; ** structured Level-0 demo file
;print, ' ** cep_byts_rec Hard-coded to allow use of incorrectly structured Level-0 demo file'

;        Maximum physical records written in this file
max_phys_rec = flr.phys_recs_file 
;max_phys_rec = 106		; ** Hard-coded to allow use of incorrectly
				; ** structured Level-0 demo file
;print, ' ** max_phys_rec Hard-coded to allow use of incorrectly structured Level-0 demo file'

;        Bytes per minor frame
cep_byts_mnfr = -1
if( flr.phys_rec_len eq CEP_Sci_Man_rec_len) then $
    cep_byts_mnfr = CEP_Sci_Man_byt_per_mnfrm
if( flr.phys_rec_len eq CEP_Conting_rec_len) then $
    cep_byts_mnfr = CEP_Conting_byt_per_mnfrm
;cep_byts_mnfr = 18		; ** Hard-coded to allow use of incorrectly
				; ** structured Level-0 demo file
;print, ' ** cep_byts_mnfr Hard-coded to allow use of incorrectly structured Level-0 demo file'

if ( cep_byts_mnfr lt 0) then begin
      print, ' INCORRECT VALUE FOR BYTES IN A PHYSICAL RECORD', cep_byts_rec
      ierror = 1
endif 

return
end




PRO CEP_SETUP

;	This procedure defines up constants.  Parameters defined include
;	CEPPAD sensor ID character strings; LONG constants containing powers
;	of 2; and the definition of the number of minor frames per record
;	(major frame).

; ----------------------------------------------------------------

 common plcp_sensor_id, plcp_ips_id, plcp_ies_id, plcp_histp_id, plcp_histe_id
 common pwr2, p0, p1, p2, p3, p4, p5, p6, p7, p8, p12, p16, p20, p24
 common pl_phy_rec, mnfrm_rec  

; ----------------------------------------------------------------

 plcp_ips_id = 'IPS'
 plcp_ies_id = 'IES'
 plcp_histe_id = 'HISTE'
 plcp_histp_id = 'HISTP'

 p0 = long(1)
 p1 = long(2)
 p2 = long(4)
 p3 = long(8)
 p4 = long(16)
 p5 = long(32)
 p6 = long(64)
 p7 = long(128)
 p8 = long(256)
 p12 = p8 * p4
 p16 = p8 * p8
 p20 = p12 * p8
 p24 = p12 * p12

 mnfrm_rec = fix(250)

return 
end




pro CEP_setup2, error_return

;	initialize various CEPPAD common block values

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block and defined them
;	01 Apr 1996	M.T.Redding	Increase SH_rec_len in COMMON block
;					CEP_Spin_head to 13 byte to allow for
;					addition of SCOM value in byte 12
;       10 Apr 1996	M. T. Redding   Added 'ignore_32_times' to 'CEP_32_time'
;                                       common block and set initial value to
;                                       'FALSE'
;	01 May 1996	M. T. Redding	Added COMMON block File_Control with
;					structure variable file_info

;	06 May 1996	M. T. Redding	Added COMON block CEP_spin_times with 
;					array variable 'previous_spin_times'.
;	07 May 1996	M. T. Redding	Changed initial value of CEP_HK_rec_len
;					to 242 bytes
;					Added variable accept_KP_mjf_lag to 
;					COMMON block CEP_time_eval
;	24 May 1996	M.T.Redding	Added initialization of variables in
;					COMMON block CEP_HK_temp
;	30 May 1996	M.T.Redding	Increase SH_rec_len in COMMON block
;					CEP_Spin_head to 14 byte to allow for
;					addition of spin quality flag (byte 14)
;	07 Jun 1996	M.T.Redding	Removed COMMON blocks 'CEP_SW_Stat',
;					'CEP_HW_Stat' and 'CEP_HK_Subcoms'
;	08 Nov 1996	M.T.Redding	Added 'max_DPU_table_no to file_info
;					structure in COMMON block File_Control

;--------------------------------------------------------------------

COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                        sci_byt_per_spin, max_spins, $
                        nom_spin_dur, TRUE, FALSE, sync_byte, $
                        CEP_Sci_Man_byt_per_mnfrm, $
                        CEP_Conting_byt_per_mnfrm, $
                        CEP_Sci_Man_rec_len, $
                        CEP_Conting_rec_len

COMMON CEP_time_eval,   accept_mjf_lag, mjf_dtime, mnf_dtime, half_mnf_dtime, $
                        accept_SE_mjf_lag, accept_KP_mjf_lag 

COMMON CEP_mf_history,  previous_times

COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                        Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

COMMON CEP_HK,          CEP_HK_unit, CEP_HK_rec, CEP_HK_rec_len, CEP_HK_data

COMMON CEP_32_time,     ignore_16_times, ignore_32_times


COMMON CEP_two_frm,     Buffer_index, Two_frm_sci_dat, Two_frm_qual

   COMMON CEP_lst_spn,     lst_spn_seq_no, lst_spn_min_frm, $
                           lst_spn_format, lst_spn_IPS_tab, $
                           lst_spn_IES_tab, lst_spn_IES_com, $
                           lst_spn_HIST_tab, lst_spn_HIST_com, $
                           lst_spn_SCOM, lst_spn_IPS_Mux, $
                           lst_spn_time, lst_spn_CRC, lst_spn_quality

   COMMON CEP_16_spin,     Spin_blk, Spin_blk_times, Spin_blk_SCOMs,	$
                           spin_blk_tot, spin_blk_seq_wrap,		$
                           est_16th_time, est_32nd_time, 		$
                           Bad_SCOM_MODE_sync, spin_quality

COMMON Level_1_fil_typ, HK_filtyp, IPS_Nrm_filtyp, IES_Nrm_filtyp, $
                        HISTe_Nrm_filtyp, HISTp_Nrm_filtyp, $
                        IPS_Rts_filtyp, IES_Rts_filtyp, HIST_Rts_filtyp, $
                        Spin_Head_filtyp

COMMON CEP_Rates_Lvl_1, IPS_Rates_unit, IPS_Rates_rec, $
                        IPS_Rates_assoc, IPS_Rates_recl, $
                        IPS_Rates_format, $
                        IES_Rates_unit, IES_Rates_rec, $
                        IES_Rates_assoc, IES_Rates_recl, $
                        IES_Rates_format, $
                        HIST_Rates_unit, HIST_Rates_rec, $
                        HIST_Rates_assoc, HIST_Rates_recl, $
                        HIST_Rates_format

COMMON CEP_Spin_head,   SH_unit, SH_record, SH_assoc, SH_rec_len

COMMON File_Control,    File_info

COMMON CEP_spin_history, previous_spin_times, spin_durations

COMMON CEP_HK_temp,     CEP_HK_max_temp_recs,	$
                        CEP_HK_temp_pointer,		$
                        CEP_HK_temp_recs
 
COMMON ML1_SW_Constants, make_level_1_version, make_level_1_date
;-----------------------------------------------------------------------

;		COMMON ML1_SW_Constants

   make_level_1_version = '2.04'
   make_level_1_date    = '** 22 November 1996'
;;   make_level_1_version = '2.03'
;;   make_level_1_date    = '** 02 July 1996'


;		COMMON CEP_constants

   sci_byt_per_mnfrm = 15	; CEPPAD science bytes each minor frame
   frames_per_rec = 250		; Level-0 minor frame per major frame
   frames_per_spin = 160	; Level-0 minor frames per spin

   sci_byt_per_spin = 2388	; Default value - Total number of bytes CEPPAD 
				; outputs each spin while in spin-sync mode
   max_spins = 16		; maximum number of consecutive spins
				; to be re-organized
   nom_spin_dur = 6000L		; nominal spin duration in milliseconds
   TRUE  = 1B			; Logical 'TRUE'
   FALSE = 0B			; Logical 'FALSE'
   sync_byte = 'A5'xb		; CEPPAD spin sync value
   CEP_Sci_Man_byt_per_mnfrm = 18	; CEPPAD Science/Manauver mode number
				; of bytes per minor frame
   CEP_Conting_byt_per_mnfrm = 33	; CEPPAD Contingency mode number of
				; bytes per minor frame
   CEP_Sci_Man_rec_len = 4800	; CEPPAD Science/Maneuver mode record length
   CEP_Conting_rec_len = 8552	; CEPPAD Contingency mode record length


;		COMMON CEP_time_eval

   accept_mjf_lag = 5L		; Number of major frames of lag acceptable
				; between acquistion and output of CEPPAD 
				; telemetry

   mjf_dtime = 9200L		; major frame delta time (msecs)
   mnf_dtime = 36.8		; minor frame delta time (msecs)
   half_mnf_dtime = 18.4	; one-half minor frame delta time
   accept_SE_mjf_lag = 20L	; Number of major frames lag for HIST singles
				; and HIST events times
   accept_KP_mjf_lag = 22L	; Number of major frames lag for HIST singles
				; and HIST events times - It take 16 spins to
				; accumulate the data, the KP fit calculations
				; must be completed within the next 16 spins
				; so the oldest time that should be accepted is 
				; 32 spins before the current time (using 6.0s
				; for the nominal spin duration and 9.2s for
				; the major frame delta time we get 
				; 32*6/9.2 = 20.9 frames to which we have added
				; an extra frame to allow for spin duration
				; slightly larger than 6.0s/spin


;		COMMON CEP_32_time

   ignore_16_times = TRUE
   ignore_32_times = FALSE


;		COMMON CEP_16_spin

   Bad_SCOM_MODE_sync = FALSE
   spin_quality       = 0B


;		COMMON CEP_two_frm

;	Allow for 2 major frame's of CEPPAD science data further divided
;	by logical minor frame

   Two_frm_sci_dat = BYTARR(sci_byt_per_mnfrm,frames_per_rec,2)

;	Allow for 2 major frame's of CEPPAD minor frame quality flag values
;	divided by logical minor frame

   Two_frm_qual = BYTARR(frames_per_rec,2)

;	Allow for maintaining  a history of up to 256 ATC msec times
;	Initialize with -1L to indicate 'missing' values

   previous_times = LONARR(256)
   previous_times(*) = -1L


;		COMMON CEP_HK

;	Define the HK Level-1 file record length. Initialize the file 
;	unit number and the record pointer.

   CEP_HK_rec_len = 242L
   CEP_HK_unit     = -3
   CEP_HK_rec      =  0L


;		COMMON Level_1_fil_typ

;	Define Level-1 file type values

   HK_filtyp         = 1L
   IPS_Nrm_filtyp    = 2L
   IES_Nrm_filtyp    = 3L
   HISTe_Nrm_filtyp  = 4L
   HISTp_Nrm_filtyp  = 5L
   IPS_Rts_filtyp    = 6L
   IES_Rts_filtyp    = 7L
   HIST_Rts_filtyp   = 8L
   Spin_Head_filtyp  = 9L 



;		COMMON CEP_Rates_Lvl_1

;	Define Rates-histogram mode Level-1 file type values

   IPS_Rates_unit    =  -3L	; IPS Rates-mode Level-1 file unit number
   IPS_Rates_rec     =   0L	; Total number of records written
   IPS_Rates_recl    = 524L	; IPS Rates_mode Level-1 record length (bytes)
   IPS_Rates_format  = 191B	; CEPPAD format value indicating IPS Rates-mode
   IES_Rates_unit    =  -3L	; IES Rates-mode Level-1 file unit number
   IES_Rates_rec     =   0L	; Total number of records written
   IES_Rates_recl    = 784L	; IES Rates_mode Level-1 record length (bytes)
   IES_Rates_format  = 223B	; CEPPAD format value indicating IES Rates-mode
   HIST_Rates_unit   =  -3L	; HIST Rates-mode Level-1 file unit number
   HIST_Rates_rec    =   0L	; Total number of records written
   HIST_Rates_recl   = 784L	; HIST Rates_mode Level-1 record length (bytes)
   HIST_Rates_format = 255B	; CEPPAD format value indicating HIST Rates-mode


;		COMMON CEP_lst_spn

;	Initialize the Last-spin parameter values

;   lst_spn_seq_no   =  0B
;   lst_spn_min_frm  =  0B
   lst_spn_format   = 'A5'XB	; A5H is not currently used as a format value
				; It is used as the sync value so if new format
				; values are defined it is expected this value
				; would not be selected
   lst_spn_IPS_tab  = -1B
   lst_spn_IES_tab  = -1B
   lst_spn_IES_com  = -1B
   lst_spn_HIST_tab = -1B
   lst_spn_HIST_com = -1
;   lst_spn_SCOM     =  0B
   lst_spn_IPS_Mux  =  0B
   lst_spn_time     = -1L
   lst_spn_CRC      =  0L
   lst_spn_quality  =  0B


;		COMMON CEP_Spin_head

;	Initialize the Spin header file parameters to pre-opened state

   SH_unit          = -3	; Out of IDL LUN range
   SH_record        =  0L	; No record written yet
   SH_rec_len       = 14L	; Record length 14 bytes ; rev 30 May 1996


;		COMMON File_Control

   File_info    =    { last_normal_TM_table :  -1B,	$
                       max_DPU_table_no     :  16B    }	; valid values are 0<=15


;		COMMON CEP_spin_history


   previous_spin_times = LONARR(256)
   previous_spin_times(*) = -1L
   spin_durations = LONARR(256)
   spin_durations(*) = -1L

;		COMMON CEP_HK_temp
   CEP_HK_max_temp_recs = 13
   CEP_HK_temp_pointer  = -1L
   temp_recs  = { ATC_yr :		-1L,				$
                  ATC_day:		-1L,				$
                  ATC_mils:		-1L,				$
                  ATC_micro:		-1L,				$
                  HKt:			BYTARR( 38 ),			$
                  Subcom:		BYTARR( frames_per_rec ),	$
                  Min_frm_quality:	BYTARR( frames_per_rec ),	$
                  CEP_KP_time:		-1L,				$
                  CEP_HIST_SE_times:	LONARR( 8 )			}
   CEP_HK_temp_recs = REPLICATE( temp_recs, CEP_HK_max_temp_recs )

   error_return = 0
 return
end



PRO CHECK_ACQ_MODE, New_Acq_mode, error_return

;	Check for a change in the data acquisition mode and return the results
;	in a logical variable.

;	A new acquisition mode is indicated by a change from the last spin
;	spin-header values of IPS-LUT (Look up table), IES-LUT, HIST-LUT,
;	IES-command or HIST-operation-state

;		Passed Parameters

;	New_Acq_mode	Logical flag indicating whether a change in the 
;			data acquisition mode for the three CEPPAD instruments
;			- IPS, IES and HIST - has occurred between the current 
;			spin and the previous spin
;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not implemented in this routine

;	revision
;	10 Apr 96 	M. T. Redding
;	11 Jun 96	M. T. Redding	Added Bad_SCOM_MODE_sync to COMMON
;					block CEP_16_spin and added code to set
;					it if new mode is encountered other than
;					on the first spin of the 16-spin block

;--------------------------------------------------------------------

   COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                           sci_byt_per_spin, max_spins, $
                           nom_spin_dur, TRUE, FALSE, sync_byte, $
                           CEP_Sci_Man_byt_per_mnfrm, $
                           CEP_Conting_byt_per_mnfrm, $
                           CEP_Sci_Man_rec_len, $
                           CEP_Conting_rec_len

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

   COMMON CEP_lst_spn,     lst_spn_seq_no, lst_spn_min_frm, $
                           lst_spn_format, lst_spn_IPS_tab, $
                           lst_spn_IES_tab, lst_spn_IES_com, $
                           lst_spn_HIST_tab, lst_spn_HIST_com, $
                           lst_spn_SCOM, lst_spn_IPS_Mux, $
                           lst_spn_time, lst_spn_CRC, lst_spn_quality

   COMMON CEP_16_spin,     Spin_blk, Spin_blk_times, Spin_blk_SCOMs,	$
                           spin_blk_tot, spin_blk_seq_wrap,		$
                           est_16th_time, est_32nd_time, 		$
                           Bad_SCOM_MODE_sync, spin_quality

;-----------------------------------------------------------------------

   error_return = 0

;	Initialize the flag value

   New_Acq_mode = FALSE

;	Don't bother checking if this spin's time is already after the estimated
;	16th time - the code already handles this case.


;bugbug	Start Diagnostic check and print
;;   mode_change_found = FALSE
;;   Change_in = '             '
;;   IF ( lst_spn_IPS_tab  NE ths_spn_IPS_tab  ) THEN BEGIN
;;      mode_change_found = TRUE
;;      change_in = change_in + 'IPStab '
;;   ENDIF
;;   IF ( lst_spn_IES_tab  NE ths_spn_IES_tab  ) THEN BEGIN
;;      mode_change_found = TRUE
;;      change_in = change_in + 'IEStab '
;;   ENDIF
;;   IF ( lst_spn_IES_com  NE ths_spn_IES_com  ) THEN BEGIN
;;      mode_change_found = TRUE
;;      change_in = change_in + 'IEScom '
;;   ENDIF
;;   IF ( lst_spn_HIST_tab NE ths_spn_HIST_tab ) THEN BEGIN
;;      mode_change_found = TRUE
;;      change_in = change_in + 'HISTtab '
;;   ENDIF
;;   IF ( lst_spn_HIST_com NE ths_spn_HIST_com ) THEN BEGIN
;;      mode_change_found = TRUE
;;      change_in = change_in + 'HISTcom '
;;   ENDIF
;;   IF ( lst_spn_IPS_Mux NE ths_spn_IPS_Mux ) THEN BEGIN
;;      mode_change_found = TRUE
;;      change_in = change_in + 'IPSMux '
;;   ENDIF
;;
;;   IF ( Mode_change_found ) THEN BEGIN
;;
;;      IF ( ( ths_spn_time LT est_16th_time ) AND $
;;        ( ths_spn_SCOM/16 EQ lst_spn_SCOM/16 ) ) THEN BEGIN
;;
;;;;         IF ( ( ths_spn_SCOM mod 16 NE 0 ) AND $
;;;;              ( ths_spn_SCOM/16 EQ lst_spn_SCOM/16 ) ) THEN $
;;         IF ( ths_spn_SCOM mod 16 NE 0 ) THEN $
;;            STRPUT, change_in, '*unexpected* ', 0
;;         print, FORMAT = "(f12.3,7z3, z5, 2z3, z3, 2x, A)", $
;;                ths_spn_time/1000.0d0, $
;;                ths_spn_seq_no, $
;;                ths_spn_min_frm, $
;;                ths_spn_format, $
;;                ths_spn_IPS_tab, $
;;                ths_spn_IES_tab, $
;;                ths_spn_IES_com, $
;;                ths_spn_HIST_tab, $
;;                ths_spn_HIST_com, $
;;                ths_spn_SCOM, $
;;                ths_spn_IPS_Mux, $
;;                lst_spn_SCOM, $
;;                change_in
;;      ENDIF
;;   ENDIF
;bugbug	Stop Diagnostic check and print


;	Do real check here

   New_Acq_mode = ( lst_spn_IPS_tab  NE ths_spn_IPS_tab  ) OR $
                  ( lst_spn_IPS_MUX  NE ths_spn_IPS_MUX  ) OR $
                  ( lst_spn_IES_tab  NE ths_spn_IES_tab  ) OR $
                  ( lst_spn_IES_com  NE ths_spn_IES_com  ) OR $
                  ( lst_spn_HIST_tab NE ths_spn_HIST_tab ) OR $
                  ( lst_spn_HIST_com NE ths_spn_HIST_com )


;   Spin_Quality = 0B
;   Bad_SCOM_MODE_sync = FALSE

   IF ( New_Acq_mode ) THEN BEGIN

      IF ( ( ths_spn_time LT est_16th_time ) AND $
           ( ths_spn_SCOM/16 EQ lst_spn_SCOM/16 ) ) THEN BEGIN


;	Revise the bad-MODE-sync flag for the spin block if appropriate -
;	it will be reset only when a recognized valid mode change occurs
;	between spins 0 and 1

         Bad_SCOM_MODE_sync = TRUE

;	Set bits 0-2 depending on whether the IPS, IES or HIST instrument
;	tables or MUX/command changed thereby indicating which instrument's
;	multi-spin data were partially accumulated using mixed modes

         Spin_Quality = Spin_quality OR 1B OR $
            ISHFT( ( ths_spn_IPS_tab NE lst_spn_IPS_tab ), 1 ) OR $
            ISHFT( ( ths_spn_IPS_MUX NE lst_spn_IPS_MUX ), 1 ) OR $
            ISHFT( ( ths_spn_IES_tab NE lst_spn_IES_tab ), 2 ) OR $
            ISHFT( ( ths_spn_IES_com NE lst_spn_IES_com ), 2 ) OR $
            ISHFT( ( ths_spn_HIST_tab NE lst_spn_HIST_tab ), 3 ) OR $
            ISHFT( ( ths_spn_HIST_com NE lst_spn_HIST_com ), 3 ) 

      ENDIF ELSE BEGIN


;	A valid mode change has occurred - reset the bad-mode flag and quality
;	value

         IF ( ( ths_spn_SCOM mod 16 ) EQ 0 ) THEN BEGIN
            Bad_SCOM_MODE_sync = FALSE
            Spin_Quality = 0B
         ENDIF

      ENDELSE

   ENDIF


   RETURN
END



PRO CHECK_SPIN_HEADER, Spin_header_OK, Commanded_Change, error_return

;	Check the combination of evaluated time (ths_spn_time) and spin 
;	number (ths_spn_SCOM) compared with the current major frame ATC-time
;	and the previously estimated 16th time for consistancy and set the 
;	logical flag to indicate the results.
;	These checks are required to trap instances of corrupted spin header
;	major frame sequence number and Scom values so that spins with 
;	questionable spin times or incorrect spin numbers may be identified
;	and excluded from further processing.

;	For the spin time to be consistant it may be less than the current
;	major frame ATC-time by no more than a predefined number of major 
;	frame.  It also may not exceed an estimate of the next major frame's
;	ATC-time. 
;	For spins having times within the current 16-spin block, the spin time
;	must be within 1/2 nom_spin_dur of a time estimated for the spin based
;	on the spin's SCOM value.

;	An attempt is made here to identify the case(s) where 
;	( ths_spn_SCOM EQ 30B ) (or 31B).  A value of 30B may be occasionally 
;	encountered on the 15th spin of a spin block out sequence with the 
;	other spin numbers in the block.  When this occurs it signals that a 
;	'commanded' mode change has occurred.  The data for such 15th and 16th 
;	spins should be discarded because the data output on these spins
;	represents an aborted accumulation.

;		Passed Parameters

;	Spin_header_OK	Logical flag indicating whether the
;			ths_spn_time/ths_spn_SCOM values are consistant with
;			the current major frame ATC-time and the estimated 16th
;			time for the current 16-spin block
;	Commanded_Change	Logical flag indicating a 'Commanded mode 
;			change has been encountered on this spin
;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not implemented in the routine

;	revision
;	08 Apr 1996	M. T. Redding
;	22 May 1996	M. T. Redding	Added positional argument 
;					'Commanded_change' to indicate whether
;					a premature SCOM value of '1E' or '1F'
;					has been encountered - Such a spin will
;					still be considered to have a 
;					consistent  Spin header even though its
;					SCOM value will out of sync with the
;					preceding spins.
;	18 Jun 1996	M.T.Redding	Changed from .5 to .75 spin duration
;					offset applied to the Est_16th_time to
;					obtain the Min_SCOM_time which should be
;					1/2 spin duration before the current
;					spin time
;	20 Jun 1996	M.T.Redding	Changed setting of initial value of 
;					'Commanded_change' so that it remains
;					set to TRUE for the spin with SCOM 31 
;					if the previous spin SCOM was 30 i.e.
;					the previous spin started the 
;					commanded change condition
;--------------------------------------------------------------------

   COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                           sci_byt_per_spin, max_spins, $
                           nom_spin_dur, TRUE, FALSE, sync_byte, $
                           CEP_Sci_Man_byt_per_mnfrm, $
                           CEP_Conting_byt_per_mnfrm, $
                           CEP_Sci_Man_rec_len, $
                           CEP_Conting_rec_len

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

   COMMON CEP_lst_spn,     lst_spn_seq_no, lst_spn_min_frm, $
                           lst_spn_format, lst_spn_IPS_tab, $
                           lst_spn_IES_tab, lst_spn_IES_com, $
                           lst_spn_HIST_tab, lst_spn_HIST_com, $
                           lst_spn_SCOM, lst_spn_IPS_Mux, $
                           lst_spn_time, lst_spn_CRC, lst_spn_quality

   COMMON CEP_16_spin,     Spin_blk, Spin_blk_times, Spin_blk_SCOMs,	$
                           spin_blk_tot, spin_blk_seq_wrap,		$
                           est_16th_time, est_32nd_time, 		$
                           Bad_SCOM_MODE_sync, spin_quality

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

COMMON CEP_time_eval,   accept_mjf_lag, mjf_dtime, mnf_dtime, half_mnf_dtime, $
                        accept_SE_mjf_lag, accept_KP_mjf_lag 

;-----------------------------------------------------------------------

   error_return = 0


;	Initialize the flag values

   Spin_header_OK   = TRUE

;   IF ( N_ELEMENTS( Commanded_change ) NE 0 ) THEN BEGIN
;      IF ( Commanded_change AND  $
;           ( lst_spn_SCOM EQ 30 ) AND $
;           ( ths_spn_SCOM EQ 31 ) AND $
;           ( ( ths_spn_time - lst_spn_time ) LE  $
;             ( LONG( FLOAT( nom_spin_dur ) * 1.5 ) ) ) ) THEN BEGIN
;         Commanded_change = TRUE
;      ENDIF ELSE BEGIN
;         Commanded_change = FALSE
;      ENDELSE
;   ENDIF ELSE BEGIN
      Commanded_change = FALSE
;   ENDELSE


;	Don't bother checking if this spin's time is already after the estimated
;	16th time - the code already handles this case.

   IF ( ths_spn_time LT est_16th_time ) THEN BEGIN

;	Check for minimum time value

      IF ( ths_spn_time LT ( MF_ATC_time - accept_mjf_lag * mjf_dtime ) ) $
                                                                THEN BEGIN
         Spin_header_OK = FALSE
         RETURN
      ENDIF

;	Check for maximum time value ( estimate of next major frame time )

      IF ( ths_spn_time GT ( MF_ATC_time + mjf_dtime ) ) THEN BEGIN
         Spin_header_OK = FALSE
         RETURN
      ENDIF

;	Check for consistant SCOM value using the estimated 16th spin time.
;	The spin time must be within 1/2 spin duration of the estimated time
;	for this SCOM - Only apply this test for spin with SCOM values other
;	than 30B and 31B - these two SCOM values may show up as a result of
;	a commanded mode change which forces a reset of the SCOM counters to
;	30B on the 15th spin and 31B on the 16th spin.

;	The Est_16th_time used with a spin with SCOM value = 0 mod 16 will be 
;	will be the one calculated for the previous spin block since a 0 SCOM
;	spin encountered here has not been used yet to determine an
;	Est_16th_time - this means the 'remaining_SCOM' value used below is
;	adjusted from '16' to '0': Note that Est_16th_time is calculated as
;	an upper limit for the spins in the current spin block - it has a value
;	1/4 spin after the time for the 0-th spin (16 mod 16 (this is where the
;	'16th' in Est_16th_time comes from)) in the next spin block
;	

      remaining_SCOM = 16L - LONG( ths_spn_SCOM MOD 16 )
      remaining_SCOM = remaining_SCOM - ( 16L * ( remaining_SCOM EQ 16L ) )
      min_SCOM_time = est_16th_time - ( remaining_SCOM + 0.75 ) * nom_spin_dur
      max_SCOM_time = min_SCOM_time + nom_spin_dur

      IF ( ( ths_spn_SCOM NE 30B ) AND ( ths_spn_SCOM NE 31B ) ) THEN BEGIN
         IF ( ths_spn_time LT min_SCOM_time ) THEN Spin_header_OK = FALSE
         IF ( ths_spn_time GT max_SCOM_time ) THEN Spin_header_OK = FALSE
      ENDIF ELSE BEGIN
         IF ( ths_spn_time LT min_SCOM_time ) THEN Commanded_change = TRUE
         IF ( ths_spn_time GT max_SCOM_time ) THEN Commanded_change = TRUE
      ENDELSE

      IF ( NOT Spin_header_OK ) THEN BEGIN
         print, '** CHECK_SPIN_HEADER **  - inconsistant Time/SCOM found -'
         print, ' MF_ATC_time  = ', MF_ATC_time
         print, ' ths_spn_time = ', ths_spn_time, $ 
                ' ths_spn_SCOM = ', ths_spn_SCOM, $
                ' est_16th_time = ', est_16th_time 
      ENDIF

      IF ( Commanded_change ) THEN BEGIN
         print, '** CHECK_SPIN_HEADER **  - Commanded change encountered -'
         print, ' MF_ATC_time  = ', MF_ATC_time
         print, ' ths_spn_time = ', ths_spn_time, $ 
                ' ths_spn_SCOM = ', ths_spn_SCOM, $
                ' lst_spn_time = ', lst_spn_time, $ 
                ' est_16th_time = ', est_16th_time 
      ENDIF

   ENDIF


   RETURN
END




pro Close_HK, error_return

;	Close the CEPPAD Level-1 Housekeeping file

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed close of Level_1 HK file

;	Mar 06 1996	M.T.Redding	Corrected printed variable 'err' to be
;					system variable '!ERR
;					Added check for valid LUN 
;					( CEP_HK_unit GT 0 ) before performing
;					'FREE_LUN'

;--------------------------------------------------------------------

   COMMON CEP_HK,          CEP_HK_unit, CEP_HK_rec, CEP_HK_rec_len, CEP_HK_data


;-----------------------------------------------------------------------

   error_return = 0

   ON_IOERROR, Failed_Close

   IF ( CEP_HK_unit GT 0 ) THEN FREE_LUN, CEP_HK_unit
   GOTO, Exit_HK_Close

Failed_Close:
   print, '*** Close_HK ***'
   print, '  Error # ', !ERR, ' encountered closing CEPPAD Level-1 HK file'
   print, !ERR_STRING
   error_return = 1

Exit_HK_Close:
   return
end



pro Close_L1, error_return

;	Close each currently opened IPS, IES HISTe and HISTp Level_1 data file

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1-4	Number of failed closings of
;						IPS, IES, HISTe and HISTp
;						Normal mode Level-1 files

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block

;--------------------------------------------------------------------

   COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                           sci_byt_per_spin, max_spins, $
                           nom_spin_dur, TRUE, FALSE, sync_byte, $
                           CEP_Sci_Man_byt_per_mnfrm, $
                           CEP_Conting_byt_per_mnfrm, $
                           CEP_Sci_Man_rec_len, $
                           CEP_Conting_rec_len

   COMMON CEP_Level_1, $
		L1_number, L1_year, L1_day, L1_begt, L1_endt, $
		L1_DPU_TM_table, L1_TM_CRC, $
	Use_IPS_L1, IPS_L1_unit, IPS_L1_record, IPS_L1_assoc, $
		IPS_L1_rec_len, IPS_L1_phyrcl, $
	Use_IES_L1, IES_L1_unit, IES_L1_record, IES_L1_assoc, $
		IES_L1_rec_len, IES_L1_phyrcl, $
	Use_HISTe_L1, HISTe_L1_unit, HISTe_L1_record, HISTe_L1_assoc, $
		HISTe_L1_rec_len, HISTe_L1_phyrcl, $
	Use_HISTp_L1, HISTp_L1_unit, HISTp_L1_record, HISTp_L1_assoc, $
		HISTp_L1_rec_len, HISTp_L1_phyrcl

;-----------------------------------------------------------------------


   ON_IOERROR, IPS_error_check
   error_return = 0
   IF ( Use_IPS_L1 ) and (IPS_L1_unit gt 0) THEN BEGIN
      CLOSE, IPS_L1_unit
      IPS_L1_unit = -3
      Use_IPS_L1  = FALSE
   ENDIF
   GOTO, IES_close

IPS_error_check: IF !err ne 0 THEN $
      print,'*** Close_L1 *** Error #',!err,' closing IPS Level-1 file'
      print,!error
      error_return = error_return + 1


IES_close:
   ON_IOERROR, IES_error_check
   IF ( Use_IES_L1 ) and ( IES_L1_unit ge 0 ) THEN BEGIN
      CLOSE, IES_L1_unit
      IES_L1_unit = -3
      Use_IES_L1  = FALSE
   ENDIF
   GOTO, HISTe_close

IES_error_check: IF !err ne 0 THEN $
      print,'*** Close_L1 *** Error #',!err,' closing IES Level-1 file'
      print,!error
      error_return = error_return + 1


HISTe_close:
   ON_IOERROR, HISTe_error_check
   IF ( Use_HISTe_L1 ) and ( HISTe_L1_unit gt 0 ) THEN BEGIN
      CLOSE, HISTe_L1_unit
      HISTe_L1_unit = -3
      Use_HISTe_L1  = FALSE
   ENDIF
   GOTO, HISTp_close

HISTe_error_check: IF !err ne 0 THEN $
      print,'*** Close_L1 *** Error #',!err,' closing HISTe Level-1 file'
      print,!error
      error_return = error_return + 1


HISTp_close:
   ON_IOERROR, HISTp_error_check
   IF ( Use_HISTp_L1 ) and ( HISTp_L1_unit gt 0 ) THEN BEGIN
      CLOSE, HISTp_L1_unit
      HISTp_L1_unit = -3
      Use_HISTp_L1  = FALSE
   ENDIF
   GOTO, Exit_Close_L1

HISTp_error_check: IF !err ne 0 THEN $
      print,'*** Close_L1 *** Error #',!err,' closing HISTp Level-1 file'
      print,!error
      error_return = error_return + 1


Exit_Close_L1:
   return
end





PRO Close_Rates_L1, error_return

;	Close each currently opened IPS, IES HIST Rates mode Level_1 data file

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1-4	Number of failed closings of
;						IPS, IES, HISTe and HISTp
;						Normal mode Level-1 files

;	revision:
;	22 Sep 1995 	M.T.Redding

;--------------------------------------------------------------------

   COMMON CEP_Rates_Lvl_1, IPS_Rates_unit, IPS_Rates_rec, $
                           IPS_Rates_assoc, IPS_Rates_recl, $
                           IPS_Rates_format, $
                           IES_Rates_unit, IES_Rates_rec, $
                           IES_Rates_assoc, IES_Rates_recl, $
                           IES_Rates_format, $
                           HIST_Rates_unit, HIST_Rates_rec, $
                           HIST_Rates_assoc, HIST_Rates_recl, $
                           HIST_Rates_format

;-----------------------------------------------------------------------


IPS_close:
   ON_IOERROR, IPS_error_check
   IF (IPS_Rates_unit gt 0) THEN BEGIN
      CLOSE, IPS_Rates_unit
      IPS_Rates_unit = -3
   ENDIF
   GOTO, IES_close

IPS_error_check: IF !err ne 0 THEN $
      print, '*** Close_Rates_L1 *** Error #', $
             !err, ' closing IPS Rates Level-1 file'
      print,!error
      error_return = error_return + 1


IES_close:
   ON_IOERROR, IES_error_check
   IF (IES_Rates_unit gt 0) THEN BEGIN
      CLOSE, IES_Rates_unit
      IES_Rates_unit = -3
   ENDIF
   GOTO, HIST_close

IES_error_check: IF !err ne 0 THEN $
      print, '*** Close_Rates_L1 *** Error #', $
             !err, ' closing IES Rates Level-1 file'
      print,!error
      error_return = error_return + 1


HIST_close:
   ON_IOERROR, HIST_error_check
   IF (HIST_Rates_unit gt 0) THEN BEGIN
      CLOSE, HIST_Rates_unit
      HIST_Rates_unit = -3
   ENDIF
   GOTO, Exit_Close_Rates_L1

HIST_error_check: IF !err ne 0 THEN $
      print, '*** Close_Rates_L1 *** Error #', $
             !err, ' closing HIST Rates Level-1 file'
      print,!error
      error_return = error_return + 1


Exit_Close_Rates_L1:


   RETURN
END





PRO Close_SH, error_return

;	Close the CEPPAD Level-1 Spin Header file

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed close of Level_1 HK file
;	revision
;	Mar 06 1996	M.T.Redding	Corrected variable 'err' to be '!err'
;					in Failed Close print statement
;					Added check for valid LUN 
;					( SH_unit GT 0 ) before performing
;					'FREE_LUN'


;--------------------------------------------------------------------

   COMMON CEP_Spin_head,   SH_unit, SH_record, SH_assoc, SH_rec_len

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no



;-----------------------------------------------------------------------

   error_return = 0

   ON_IOERROR, Failed_Close

   IF ( SH_unit GT 0 ) THEN FREE_LUN, SH_unit
   GOTO, Exit_SH_Close

Failed_Close:
   print, '*** Close_SH ***'
   print, '  Error # ', !err,' encountered closing CEPPAD Level-1 SH file'
   print, !ERR_STRING
   error_return = 1

Exit_SH_Close:
   return
end



PRO Construct_map, error_return

;	Create the CEPPAD IPS,IES, HIST-electron and HIST-proton Level-0 to 
;	Level-1 file map arrays.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

; ----------------------------------------------------------------

   COMMON CEP_lvl0_1_map, $
        Trgt_times, $
	num_source_spins,$
	num_IPS_energies, num_IPS_channels, IPS_spins_per_sample,$
		IPS_sectors_per_spin, IPS_Source_loc, IPS_Source_len,$
		IPS_Target_loc, IPS_trgt_tim_Loc, IPS_trgt_blk_len,$
                IPS_trgt_blk, IPS_source_byte_loc, IPS_trgt_byte_loc, $
	num_IES_energies, num_IES_channels, IES_spins_per_sample,$
		IES_sectors_per_spin, IES_Source_loc, IES_Source_len,$
		IES_Target_loc, IES_trgt_tim_loc, IES_trgt_blk_len,$
                IES_trgt_blk, IES_source_byte_loc, IES_trgt_byte_loc, $
	num_HISTe_energies, num_HISTe_channels, HISTe_spins_per_sample,$
		HISTe_sectors_per_spin, HISTe_Source_loc, HISTe_Source_len,$
		HISTe_Target_loc, HISTe_trgt_tim_loc, HISTe_trgt_blk_len,$
                HISTe_trgt_blk, HISTe_source_byte_loc, HISTe_trgt_byte_loc, $
	num_HISTp_energies, num_HISTp_channels, HISTp_spins_per_sample,$
		HISTp_sectors_per_spin, HISTp_Source_loc, HISTp_Source_len,$
		HISTp_Target_loc, HISTp_trgt_tim_loc, HISTp_trgt_blk_len,$
                HISTp_trgt_blk, HISTp_source_byte_loc, HISTp_trgt_byte_loc



; ----------------------------------------------------------------

; num_source_spins		number of spins of data (16)
; num_IPS_energies		number of IPS energy channels (e.g. 17 P0-P16)
; num_IPS_channels		number of IPS detectors or pixels (e.g. 10)
; IPS_spins_per_sample		array - defined for each IPS energy
; IPS_sectors_per_spin		array - defined for each IPS energy and detector
; IPS_Source_loc		array - start locations within the 16-spin 
;				source array of data to be remapped into the
;				32-time target array -defined for each spin 
;				and energy
; IPS_Source_len		number of bytes corresponding to IPS_Source_loc
;				to be transfered to the 32-time array
; IPS_trgt_blk			Array containing time ordered 32-spins of IPS 
;				data
; IPS_Target_loc		index location within the 32-time target array
;				to which the values defined by IPS_Source_loc
;				and IPS_Source_len should be transfered
; IPS_trgt_tim_Loc		array - index location within the 32-time 
;				target array to which the 4 byte (Long) time 
;				values associated with the start of each spin
;				should be transfered - dimensioned to 
;				num_source_spins (16) the locations point to 
;				the 2nd set of 16-times in the 32-time array ...
;				the locations of the time values stored in the 
;				1st 16-times may be had by subtracting the 
;				first location (IPS_trgt_tim_loc) from each of 
;				the values defined in IPS_trgt_tim_loc
; IPS_trgt_blk_len		total number of bytes in the 32-time target 
;				array - the target array may be thought of as
;				2 16-time arrays
; IPS_trgt_byte_loc

; num_IES_energies		See analogous description for IPS variables
; num_IES_channels	
; IES_spins_per_sample	
; IES_sectors_per_spin	
; IES_Source_loc	
; IES_Source_len	
; IES_trgt_blk
; IES_Target_loc	
; IES_trgt_tim_loc	
; IES_trgt_blk_len	
; num_HISTe_energies		See analogous description for IPS variables
; num_HISTe_channels	
; HISTe_spins_per_sample
; HISTe_sectors_per_spin
; HISTe_Source_loc	
; HISTe_Source_len	
; HISTe_trgt_blk
; HISTe_Target_loc	
; HISTe_trgt_tim_loc	
; HISTe_trgt_blk_len	
; num_HISTp_energies		See analogous description for IPS variables
; num_HISTp_channels	
; HISTp_spins_per_sample
; HISTp_sectors_per_spin
; HISTp_Source_loc	
; HISTp_Source_len	
; HISTp_trgt_blk
; HISTp_Target_loc	
; HISTp_trgt_tim_loc	
; HISTp_trgt_blk_len	


;----------------------------------------------------------------------

   num_source_spins = 16


;	Determine the locations in the 'source' array ... containing 16-spins'
;	worth of starting positions of science data (spin header and 
;	housekeeping values have been removed.  Locations of the start of data 
;	from each spin, and energy are determined.

   Def_source_loc, error_return, num_source_spins
;	num_IPS_energies, num_IES_energies, num_HISTe_energies, $
;		num_HISTp_energies, $
;	num_IPS_channels, num_IES_channels, num_HISTe_channels, $
;		num_HISTp_channels, $
;	IPS_Spins_per_sample, IES_Spins_per_sample, HISTe_Spins_per_sample, $
;		HISTp_Spins_per_sample, $
;	IPS_Sectors_per_spin, IES_Sectors_per_spin, HISTe_Sectors_per_spin, $
;		HISTp_Sectors_per_spin, $
;	IPS_Source_loc, IES_Source_loc, HISTe_Source_loc, $
;		HISTp_Source_loc


;	Determine the lengths of data (number of values) stored starting at the
;	locations in each of the 'Source-loc' arrays.

   Def_Source_len, error_return, $
        num_source_spins, $
	num_IPS_energies, num_IPS_channels, $
	IPS_Spins_per_sample, IPS_Sectors_per_spin, IPS_Source_len

   Def_Source_len, error_return, $
        num_source_spins, $
	num_IES_energies, num_IES_channels, $
	IES_Spins_per_sample, IES_Sectors_per_spin, IES_Source_len

   Def_Source_len, error_return, $
        num_source_spins, $
	num_HISTe_energies, num_HISTe_channels, $
	 HISTe_Spins_per_sample, HISTe_Sectors_per_spin, HISTe_Source_len

   Def_Source_len, error_return, $
        num_source_spins, $
	num_HISTp_energies, num_HISTp_channels, $
	HISTp_Spins_per_sample, HISTp_Sectors_per_spin, HISTp_Source_len


;	Determine the locations in the source (spin-block) array of each byte
;	to be mapped to the time-block array
   Def_source_byte_loc, error_return


;	Determine the locations to which data should be transfered in the
;	'Target' arrays ... containing 32 spins of data organized into 32
;	subrecords of varying size

   Def_Target_loc, error_return, $
		num_source_spins, num_IPS_energies, num_IPS_channels, $
		IPS_Spins_per_sample, IPS_Sectors_per_spin, $
		IPS_Source_len, IPS_Target_loc, IPS_trgt_tim_loc, $
		IPS_trgt_blk_len, IPS_trgt_byte_loc

   Def_Target_loc, error_return, $
		num_source_spins, num_IES_energies, num_IES_channels, $
		IES_Spins_per_sample, IES_Sectors_per_spin, $
		IES_Source_len, IES_Target_loc, IES_trgt_tim_loc, $
		IES_trgt_blk_len, IES_trgt_byte_loc

   Def_Target_loc, error_return, $
		num_source_spins, num_HISTe_energies, num_HISTe_channels, $
		HISTe_Spins_per_sample, HISTe_Sectors_per_spin, $
		HISTe_Source_len, HISTe_Target_loc, HISTe_trgt_tim_loc, $
		HISTe_trgt_blk_len, HISTe_trgt_byte_loc

   Def_Target_loc, error_return, $
		num_source_spins, num_HISTp_energies, num_HISTp_channels, $
		HISTp_Spins_per_sample, HISTp_Sectors_per_spin, $
		HISTp_Source_len, HISTp_Target_loc, HISTp_trgt_tim_loc, $
		HISTp_trgt_blk_len, HISTp_trgt_byte_loc


;	Define/dimension the arrays to contain the 32 time-organized spin data

   IPS_trgt_blk       = bytarr(IPS_trgt_blk_len)
   IPS_trgt_blk(*)    = -1L
   IES_trgt_blk       = bytarr(IES_trgt_blk_len)
   IES_trgt_blk(*)    = -1L
   HISTe_trgt_blk     = bytarr(HISTe_trgt_blk_len)
   HISTe_trgt_blk(*)  = -1L
   HISTp_trgt_blk     = bytarr(HISTp_trgt_blk_len)
   HISTp_trgt_blk(*)  = -1L

   Trgt_times         = LONARR( num_source_spins * 2 )
   Trgt_times(*)      = -1L

   RETURN

END




PRO CORRECT_FORMAT, Format_number_OK, error_return

;	Check the value of ths_spn_format for legitimate values (Normal Science
;	mode only) and for changes in sync with ths_spn_SCOM and change the 
;	value back to the previous value if it is found to be questionable.

;	If the format is a normal Science mode format the the table number
;	contained in the bits 0-3 must be between 0 and the highest DPU table
;	number contained in the TM_table.txt file.  The highest table number
;	is set each time the TM_table.txt file is read.  It is found in variable
;	file_info.max_DPU_table_no.

;	Changes to a Normal Science mode format are expected to occur only on
;	spins with SCOM = 0 mod 16 ( and occassionally also 1 mod 16 in data
;	acquired before the DPU patch was uploaded Jul 3 1996).  It is also 
;	possible for a new Normal Science mode format to be found in the 
;	telemetry on a spin with SCOM NE 0 if there has been a data skip in
;	which the first spin (first and second before Jul 3 1996) has been 
;	skipped.  

;	IF the format value is found to be inconsistant with the ths_spn_SCOM
;	and lst_spn_SCOM then it is "corrected" to the previous format value.

;		Passed Parameters

;	Format_number_OK	Logical flag indicating whether the originally
;				read ths_spn_format value was consistant with 
;				ths_spn_SCOM and lst_spn_SCOM
;	error_return		Integer - error condition indicator passed back 
;				to calling procedure 
;				Interpretation - 0	No error
;						 1	Not implemented in the routine

;	revision
;	08 Nov 1996	M. T. Redding

;--------------------------------------------------------------------

   COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                           sci_byt_per_spin, max_spins, $
                           nom_spin_dur, TRUE, FALSE, sync_byte, $
                           CEP_Sci_Man_byt_per_mnfrm, $
                           CEP_Conting_byt_per_mnfrm, $
                           CEP_Sci_Man_rec_len, $
                           CEP_Conting_rec_len

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

   COMMON CEP_lst_spn,     lst_spn_seq_no, lst_spn_min_frm, $
                           lst_spn_format, lst_spn_IPS_tab, $
                           lst_spn_IES_tab, lst_spn_IES_com, $
                           lst_spn_HIST_tab, lst_spn_HIST_com, $
                           lst_spn_SCOM, lst_spn_IPS_Mux, $
                           lst_spn_time, lst_spn_CRC, lst_spn_quality

   COMMON CEP_16_spin,     Spin_blk, Spin_blk_times, Spin_blk_SCOMs,	$
                           spin_blk_tot, spin_blk_seq_wrap,		$
                           est_16th_time, est_32nd_time, 		$
                           Bad_SCOM_MODE_sync, spin_quality

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

COMMON CEP_time_eval,   accept_mjf_lag, mjf_dtime, mnf_dtime, half_mnf_dtime, $
                        accept_SE_mjf_lag, accept_KP_mjf_lag 

   COMMON File_Control,    File_info

;-----------------------------------------------------------------------

   error_return = 0


;	Initialize the flag value

   Format_no_header_OK   = TRUE


;	Only check the format if it is for a Normal Science mode format

   IF ( ( ths_spn_format / 16 ) EQ 0 ) THEN BEGIN


;	Check for valid table number and change it if necessary

      IF ( ths_spn_format GT file_info.max_DPU_table_no ) THEN BEGIN

         PRINT, ' '
         PRINT, ' Invalid DPU table number encountered  - table = ', $
                ths_spn_format mod 16
         PRINT, ' Changing to previous format value     - format = ', $
                lst_spn_format
         ths_spn_format = lst_spn_format
         Format_number_OK = FALSE
         RETURN

      ENDIF


;	Check for change that is inconsistant with SCOM values


;			Allow the very first format encountered, whatever it is,
;			to pass

      IF ( file_info.last_normal_TM_table GE 0 ) AND			$
         ( file_info.max_DPU_table_no LT 16B ) AND			$
         ( file_info.last_normal_TM_table LE file_info.max_DPU_table_no ) $
                                                                   THEN BEGIN

;			Check for change in format from last spin

         IF ( ths_spn_format NE lst_spn_format ) THEN BEGIN

;			Check spins in the same 16-spin block

            IF ( ths_spn_SCOM EQ lst_spn_SCOM ) THEN BEGIN
               IF ( ( lst_spn_time NE -1L ) AND $
                    ( ths_spn_time - lst_spn_time ) LT 102000 ) THEN BEGIN

;			Check for out of normal sync changes

                  IF ( ( ( ths_spn_SCOM mod 16 ) NE 0 ) AND $
                       ( ( ths_spn_SCOM mod 16 ) NE 1 ) ) THEN BEGIN

                     Format_number_OK = FALSE

                        PRINT, ' '
                        PRINT, ' Out of sync Format changed - ths_spn_format = ', $
                               ths_spn_format
                        PRINT, ' Changing to previous format value - format = ', $
                               lst_spn_format
                        ths_spn_format = lst_spn_format
                        Format_number_OK = FALSE
                        RETURN

                  ENDIF
               ENDIF
            ENDIF
         ENDIF
      ENDIF

   ENDIF

   RETURN

END




pro Decode_tm_head, head_string, $
                     start_year, start_day, start_second, $
                     end_year, end_day, end_second, $
                     DPU_table_no, DPU_revision_no, Table_CRC_value

;		Passed Parameters

;	head_string
;	start_year
;	start_day
;	start_second
;	end_year
;	end_day
;	end_second
;	DPU_table_no
;	DPU_revision_no
;	Table_CRC_value

; ----------------------------------------------------------------

;	Initialize values to indicate no table information given

   start_year        =  0L
   start_day         =  0L
   start_second      =  0L
   end_year          =  0L
   end_day           =  0L
   end_second        =  0L
   DPU_table_no      = -1L
   DPU_revision_no   =  0L
   Table_CRC_value   =  0L

;	Convert the characters in the string to upper case

   heading_string = STRUPCASE(head_string)
   len = STRLEN(heading_string)
   IF len gt 0 THEN BEGIN

;	Decode the table heading string - strip values off the end of the string

;	Table CRC value

      pos = STRPOS( heading_string, 'CRC' ) 
      IF pos ne -1 THEN BEGIN
         eq_pos = STRPOS(STRMID(heading_string,pos,len-pos),'=')+pos
         Table_CRC_value   = LONG(STRMID(heading_string,eq_pos+1,len))
         heading_string = STRMID(heading_string,0,pos)
      ENDIF

;	Revision number for the DPU table

      len = STRLEN(heading_string)
      pos = STRPOS( heading_string, 'REV' ) 
      IF pos ne -1 THEN BEGIN
         eq_pos = STRPOS(STRMID(heading_string,pos,len-pos),'=')+pos
         DPU_revision_no   = LONG(STRMID(heading_string,eq_pos+1,len))
         heading_string = STRMID(heading_string,0,pos)
      ENDIF


;	DPU table number

      len = STRLEN(heading_string)
      pos = STRPOS( heading_string, 'DPU_TABLE' ) 
      IF pos ne -1 THEN BEGIN
         eq_pos = STRPOS(STRMID(heading_string,pos,len-pos),'=')+pos
         DPU_table_no      = LONG(STRMID(heading_string,eq_pos+1,len))
         heading_string = STRMID(heading_string,0,pos)
      ENDIF

;	Date (year,day,second) table superceded in DPU

      len = STRLEN(heading_string)
      pos = STRPOS( heading_string, 'SUPERCEDED' ) 
      IF pos ne -1 THEN BEGIN
         eq_pos = STRPOS(STRMID(heading_string,pos,len-pos),'=')+pos
         temp_string    = STRMID(heading_string,eq_pos+1,len)
         heading_string = STRMID(heading_string,0,pos)

;	Parse the substring containing just the end date and time

         len = STRLEN(temp_string)
         dash_pos = STRPOS( temp_string, '-' ) 
         IF dash_pos ne -1 THEN BEGIN
            end_year       = LONG(STRMID(temp_string,0,dash_pos))
            temp_string    = STRMID(temp_string,dash_pos+1,len)
            len = STRLEN(temp_string)
            dash_pos = STRPOS( temp_string, '-' ) 
            IF dash_pos ne -1 THEN BEGIN
               end_day        = LONG(STRMID(temp_string,0,dash_pos))
               temp_string    = STRMID(temp_string,dash_pos+1,len)
               len = STRLEN(temp_string)
               end_second     = LONG(STRMID(temp_string,0,len))
            ENDIF
         ENDIF
      ENDIF

;	Date (year,day,second) table loaded in DPU

      len = STRLEN(heading_string)
      pos = STRPOS( heading_string, 'LOADED' ) 
      IF pos ne -1 THEN BEGIN
         eq_pos = STRPOS(STRMID(heading_string,pos,len-pos),'=')+pos
         temp_string    = STRMID(heading_string,eq_pos+1,len)
         heading_string = STRMID(heading_string,0,pos)

;	Parse the substring containing just the start date and time

         len = STRLEN(temp_string)
         dash_pos = STRPOS( temp_string, '-' ) 
         IF dash_pos ne -1 THEN BEGIN
            start_year       = LONG(STRMID(temp_string,0,dash_pos))
            temp_string    = STRMID(temp_string,dash_pos+1,len)
            len = STRLEN(temp_string)
            dash_pos = STRPOS( temp_string, '-' ) 
            IF dash_pos ne -1 THEN BEGIN
               start_day        = LONG(STRMID(temp_string,0,dash_pos))
               temp_string    = STRMID(temp_string,dash_pos+1,len)
               len = STRLEN(temp_string)
               start_second     = LONG(STRMID(temp_string,0,len))
            ENDIF
         ENDIF
      ENDIF



   ENDIF

   RETURN
END





PRO Def_source_byte_loc, error_return


;	Determine the locations, byte-by-byte in the source (spin-block)
;	array from which bytes will be mapped to the time-block array



;	COMMON block variables used as inputs to this procedure

;	num_spins
;	num_IPS_energies
;	num_IES_energies
;	num_HISTe_energies
;	num_HISTp_energies
;	IPS_source_len(num_spins,num_IPS_energies)
;	IES_source_len(num_spins,num_IES_energies)
;	HISTe_source_len(num_spins,num_HISTe_energies)
;	HISTp_source_len(num_spins,num_HISTp_energies)
;	IPS_Source_loc(num_spins,num_IPS_energies)
;	IES_Source_loc(num_spins,num_IES_energies)
;	HISTe_Source_loc(num_spins,num_HISTe_energies)
;	HISTp_Source_loc(num_spins,num_HISTp_energies)


;	COMMON block variables changed by this procedure

;	IPS_source_byte_loc(IPS_source_len)
;	IES_source_byte_loc(IES_source_len)
;	HISTe_source_byte_loc(HISTe_source_len)
;	HISTp_source_byte_loc(HISTp_source_len)


;	Variables passed back to the calling procedure

;	error_return


;	Variables used locally by this procedure

;	ispin
;	ienergy
;	jchannel
;	total_sectors
;	start_source_byte

;	revision
;	01-Jul 1996	M.T.Redding	Added lines requiring that the total
;					IPS, IES, HISTe and HISTp bytes be GT 0
;					for respective output parameters to be
;					determined.
;					Revised comments describing variables
;					used by this routine.
; ----------------------------------------------------------------

   COMMON CEP_lvl0_1_map, $
        Trgt_times, $
	num_source_spins,$
	num_IPS_energies, num_IPS_channels, IPS_spins_per_sample,$
		IPS_sectors_per_spin, IPS_Source_loc, IPS_Source_len,$
		IPS_Target_loc, IPS_trgt_tim_Loc, IPS_trgt_blk_len,$
                IPS_trgt_blk, IPS_source_byte_loc, IPS_trgt_byte_loc, $
	num_IES_energies, num_IES_channels, IES_spins_per_sample,$
		IES_sectors_per_spin, IES_Source_loc, IES_Source_len,$
		IES_Target_loc, IES_trgt_tim_loc, IES_trgt_blk_len,$
                IES_trgt_blk, IES_source_byte_loc, IES_trgt_byte_loc, $
	num_HISTe_energies, num_HISTe_channels, HISTe_spins_per_sample,$
		HISTe_sectors_per_spin, HISTe_Source_loc, HISTe_Source_len,$
		HISTe_Target_loc, HISTe_trgt_tim_loc, HISTe_trgt_blk_len,$
                HISTe_trgt_blk, HISTe_source_byte_loc, HISTe_trgt_byte_loc, $
	num_HISTp_energies, num_HISTp_channels, HISTp_spins_per_sample,$
		HISTp_sectors_per_spin, HISTp_Source_loc, HISTp_Source_len,$
		HISTp_Target_loc, HISTp_trgt_tim_loc, HISTp_trgt_blk_len,$
                HISTp_trgt_blk, HISTp_source_byte_loc, HISTp_trgt_byte_loc



;----------------------------------------------------------------------


;		IPS
 
   start_source_byte_loc = 0L
   IF ( TOTAL( IPS_source_len ) GT 0 ) THEN BEGIN
      IPS_source_byte_loc = LONARR( TOTAL( IPS_source_len ) )
      FOR ispin = 0, num_source_spins-1 DO BEGIN
         FOR jenergy = 0, num_IPS_energies-1 DO BEGIN
            IF ( IPS_source_len( ispin, jenergy ) GT 0 ) THEN $
               IPS_source_byte_loc( start_source_byte_loc ) = $
                               LINDGEN( IPS_source_len( ispin, jenergy ) ) + $
                               IPS_source_loc( ispin, jenergy )
            start_source_byte_loc = start_source_byte_loc + $
                                    IPS_source_len( ispin, jenergy ) 
         ENDFOR
      ENDFOR
   ENDIF


;		IES

   start_source_byte_loc = 0L
   IF ( TOTAL( IES_source_len ) GT 0 ) THEN BEGIN
      IES_source_byte_loc = LONARR( TOTAL( IES_source_len ) )
      FOR ispin = 0, num_source_spins-1 DO BEGIN
         FOR jenergy = 0, num_IES_energies-1 DO BEGIN
            IF ( IES_source_len( ispin, jenergy ) GT 0 ) THEN $
               IES_source_byte_loc( start_source_byte_loc ) = $
                               LINDGEN( IES_source_len( ispin, jenergy ) ) + $
                               IES_source_loc( ispin, jenergy )
            start_source_byte_loc = start_source_byte_loc + $
                                    IES_source_len( ispin, jenergy ) 
         ENDFOR
      ENDFOR
   ENDIF


;		HIST electrons

   start_source_byte_loc = 0L
   IF ( TOTAL( HISTe_source_len ) GT 0 ) THEN BEGIN
      HISTe_source_byte_loc = LONARR( TOTAL( HISTe_source_len ) )
      FOR ispin = 0, num_source_spins-1 DO BEGIN
         FOR jenergy = 0, num_HISTe_energies-1 DO BEGIN
            IF ( HISTe_source_len( ispin, jenergy ) GT 0 ) THEN $
               HISTe_source_byte_loc( start_source_byte_loc ) = $
                               LINDGEN( HISTe_source_len( ispin, jenergy ) ) + $
                               HISTe_source_loc( ispin, jenergy )
            start_source_byte_loc = start_source_byte_loc + $
                                    HISTe_source_len( ispin, jenergy ) 
         ENDFOR
      ENDFOR
   ENDIF


;		HIST protons

   start_source_byte_loc = 0L
   IF ( TOTAL( HISTp_source_len ) GT 0 ) THEN BEGIN
      HISTp_source_byte_loc = LONARR( TOTAL( HISTp_source_len ) )
      FOR ispin = 0, num_source_spins-1 DO BEGIN
         FOR jenergy = 0, num_HISTp_energies-1 DO BEGIN
            IF ( HISTp_source_len( ispin, jenergy ) GT 0 ) THEN $
               HISTp_source_byte_loc( start_source_byte_loc ) = $
                               LINDGEN( HISTp_source_len( ispin, jenergy ) ) + $
                               HISTp_source_loc( ispin, jenergy )
            start_source_byte_loc = start_source_byte_loc + $
                                    HISTp_source_len( ispin, jenergy ) 
         ENDFOR
      ENDFOR
   ENDIF

   RETURN

END








PRO Def_Source_len, error_return, num_spins, $
		num_energies, num_channels, Spins_per_sample, $
		Sectors_per_spin, Source_len


;	Determine the lengths of sections of data stored in the level-0 file
;	by spin and energy.


;	Variables passed to this procedure

;	num_spins
;	num_energies
;	num_channels
;	Spins_per_sample(num_energies)
;	Sectors_per_spin(num_energies,num_channels)


;	Variables passed to calling procedure

;	error_return
;	Source_len(num_spins,num_energies)



;	Variables used locally in this procedure

;	num_spins
;	nom_part
;	part_accounted_for
;	Total_len(num_energies)
;	Len_per_spin(num_energies)

; ----------------------------------------------------------------

;	Determine the lengths (in number of values) of data to be transfered
;	from the 16-spin source array by spin and energy value (column)

;	Calculate the actual number of values accumulated in each energy
;	over the energy's defined 'spins_per_sample'.  Note that this number
;	may be different from the number of values written into the telemetry
;	because the overall total of values written into the telemtry may
;	include some padding so that the total of values may be split up
;	evenly and written to 'spins_per_sample' consecutive spins.

;	For each energy ,sum up the individual sectors_per_spin values for
;	all detector channels



   Total_len = intarr(num_energies)
   FOR ienergy = 0, num_energies-1 DO BEGIN
      Total_len(ienergy) = 0
      FOR jchannel = 0, num_channels-1 DO BEGIN
         Total_len(ienergy) = $
               Total_len(ienergy) + Sectors_per_spin(ienergy,jchannel)
      ENDFOR
   ENDFOR


;	Determine the nominal length of data to be read each spin for each
;	energy - when the 'spins_per_sample' value is unequal to '1' this
;	nominal value will be equal to:
;		Total_len/spins_per_sample
;	or the next higher integer if this division has a remainder.
;	For each spin determine the length of data to be read from each energy
;	and put this into the Source_len array

   Source_len = intarr(num_spins,num_energies)
   FOR ienergy = 0, num_energies-1 DO BEGIN
      Nom_part = $
                FIX( ( Total_len(ienergy) + (Spins_per_sample(ienergy) -1) ) / $
                              Spins_per_sample(ienergy) )
      FOR jspin = 0, num_spins-1, Spins_per_sample(ienergy) DO BEGIN
         part_accounted_for = 0
         FOR k = 0, Spins_per_sample(ienergy)-1 DO BEGIN
            IF part_accounted_for lt Total_len(ienergy) THEN BEGIN 
               IF Total_len(ienergy)-part_accounted_for le Nom_part THEN BEGIN
                  len_next_piece =Total_len(ienergy)-part_accounted_for
               ENDIF ELSE BEGIN
                  len_next_piece = Nom_part
               ENDELSE
               Source_len(jspin+k,ienergy) = len_next_piece
               part_accounted_for = part_accounted_for + len_next_piece
            ENDIF ELSE BEGIN 
               Source_len(jspin+k,ienergy) = 0
            ENDELSE
         ENDFOR
      ENDFOR
   ENDFOR


END





PRO Def_source_loc, error_return, num_spins


;	Determine the array locations by energy and spin of IPS, IES, HISTe 
;	and HISTp data in the 16-spin source array



;	Variables passed to this procedure

;	num_spins
;	num_IPS_energies
;	num_IES_energies
;	num_HISTe_energies
;	num_HISTp_energies
;	IPS_Spins_per_sample(num_IPS_energies)
;	IES_Spins_per_sample(num_IES_energies)
;	HISTe_Spins_per_sample(num_HISTe_energies)
;	HISTp_Spins_per_sample(num_HISTp_energies)
;	IPS_Sectors_per_spin(num_IPS_energies,num_IPS_channels)
;	IES_Sectors_per_spin(num_IES_energies,num_IES_channels)
;	HISTe_Sectors_per_spin(num_HISTe_energies,num_HISTe_channels)
;	HISTp_Sectors_per_spin(num_HISTp_energies,num_HISTp_channels)


;	Variables passed to the calling procedure

;	error_return
;	IPS_Source_loc(num_spins,num_IPS_energies)
;	IES_Source_loc(num_spins,num_IES_energies)
;	HISTe_Source_loc(num_spins,num_HISTe_energies)
;	HISTp_Source_loc(num_spins,num_HISTp_energies)


;	Variables used locally by this procedure

;	ispin
;	ienergy
;	jchannel
;	total_sectors
;	IPS_total_len
;	IES_total_len
;	HISTe_total_len
;	HISTp_total_len
;	IPS_len_per_spin(num_IES_energies)
;	IES_len_per_spin(num_IES_energies)
;	HISTe_len_per_spin(num_HISTe_energies)
;	HISTp_len_per_spin(num_HISTp_energies)
;	IPS_source_start(num_spins)
;	IES_source_start(num_spins)
;	HISTe_source_start(num_spins)
;	HISTp_source_start(num_spins)

; ----------------------------------------------------------------

   COMMON CEP_lvl0_1_map, $
        Trgt_times, $
	num_source_spins,$
	num_IPS_energies, num_IPS_channels, IPS_spins_per_sample,$
		IPS_sectors_per_spin, IPS_Source_loc, IPS_Source_len,$
		IPS_Target_loc, IPS_trgt_tim_Loc, IPS_trgt_blk_len,$
                IPS_trgt_blk, IPS_source_byte_loc, IPS_trgt_byte_loc, $
	num_IES_energies, num_IES_channels, IES_spins_per_sample,$
		IES_sectors_per_spin, IES_Source_loc, IES_Source_len,$
		IES_Target_loc, IES_trgt_tim_loc, IES_trgt_blk_len,$
                IES_trgt_blk, IES_source_byte_loc, IES_trgt_byte_loc, $
	num_HISTe_energies, num_HISTe_channels, HISTe_spins_per_sample,$
		HISTe_sectors_per_spin, HISTe_Source_loc, HISTe_Source_len,$
		HISTe_Target_loc, HISTe_trgt_tim_loc, HISTe_trgt_blk_len,$
                HISTe_trgt_blk, HISTe_source_byte_loc, HISTe_trgt_byte_loc, $
	num_HISTp_energies, num_HISTp_channels, HISTp_spins_per_sample,$
		HISTp_sectors_per_spin, HISTp_Source_loc, HISTp_Source_len,$
		HISTp_Target_loc, HISTp_trgt_tim_loc, HISTp_trgt_blk_len,$
                HISTp_trgt_blk, HISTp_source_byte_loc, HISTp_trgt_byte_loc



;----------------------------------------------------------------------


;	Determine total number of values output in each energy each spin
;	Sum up the sectors-per-spin values defined for all the detector channels
;	in an energy and divided by the spins_per_sample value defined for that
;	energy.  This value should be an integer equal to the calculated 
;	fraction if the fraction is an integer or, otherwise,  the next higher 
;	integer.
;	Determine total lengths (over all energies) of IPS, IES, HIST-e and 
;	HIST-p data output each spin.

;	IPS 'individual' energy and 'total' number of values output 
;	each spin.

   IPS_total_len = 0
   IPS_len_per_spin = intarr(num_IPS_energies)
   FOR ienergy = 0, num_IPS_energies-1 DO BEGIN
      total_sectors = 0
      FOR jchannel = 0, num_IPS_channels-1 DO BEGIN
         total_sectors = $
               total_sectors + IPS_Sectors_per_spin(ienergy,jchannel)
      ENDFOR
      IPS_len_per_spin(ienergy) = $
               FIX( (total_sectors + (IPS_Spins_per_sample(ienergy) - 1) ) / $
                             IPS_Spins_per_sample(ienergy) )
      IPS_total_len = IPS_total_len + IPS_len_per_spin(ienergy)
   ENDFOR


;	IES 'individual' energy and 'total' number of values output 
;	each spin.

   IES_total_len = 0
   IES_len_per_spin = intarr(num_IES_energies)
   FOR ienergy = 0, num_IES_energies-1 DO BEGIN
      total_sectors = 0
      FOR jchannel = 0, num_IES_channels-1 DO BEGIN
         total_sectors = $
               total_sectors + IES_Sectors_per_spin(ienergy,jchannel)
      ENDFOR
      IES_len_per_spin(ienergy) = $
               FIX( (total_sectors + (IES_Spins_per_sample(ienergy) - 1) ) / $
                             IES_Spins_per_sample(ienergy) )
      IES_total_len = IES_total_len + IES_len_per_spin(ienergy)
   ENDFOR


;	HIST electron 'individual' energy and 'total' number of values output 
;	each spin.

   HISTe_total_len = 0
   HISTe_len_per_spin = intarr(num_HISTe_energies)
   FOR ienergy = 0, num_HISTe_energies-1 DO BEGIN
      total_sectors = 0
      FOR jchannel = 0, num_HISTe_channels-1 DO BEGIN
         total_sectors = $
               total_sectors + HISTe_Sectors_per_spin(ienergy,jchannel)
      ENDFOR
      HISTe_len_per_spin(ienergy) = $
               FIX( (total_sectors + (HISTe_Spins_per_sample(ienergy) - 1) ) / $
                             HISTe_Spins_per_sample(ienergy) )
      HISTe_total_len = HISTe_total_len + HISTe_len_per_spin(ienergy)
   ENDFOR


;	HIST proton 'individual' energy and 'total' number of values output 
;	each spin.

   HISTp_total_len = 0
   HISTp_len_per_spin = intarr(num_HISTp_energies)
   FOR ienergy = 0, num_HISTp_energies-1 DO BEGIN
      total_sectors = 0
      FOR jchannel = 0, num_HISTp_channels-1 DO BEGIN
         total_sectors = $
               total_sectors + HISTp_Sectors_per_spin(ienergy,jchannel)
      ENDFOR
      HISTp_len_per_spin(ienergy) = $
               FIX( (total_sectors + (HISTp_Spins_per_sample(ienergy) - 1) )/ $
                             HISTp_Spins_per_sample(ienergy) )
      HISTp_total_len = HISTp_total_len + HISTp_len_per_spin(ienergy)
   ENDFOR


;	Calculate locations in 16-spin block of start of IPS, IES, HIST-e and 
;	HIST-p data (assume spin header data has been extracted from data stream
;	and only spin data is present in source array).

   next_loc = 0
   IPS_source_start = lonarr(num_spins)
   IES_source_start = lonarr(num_spins)
   HISTe_source_start = lonarr(num_spins)
   HISTp_source_start = lonarr(num_spins)
   FOR ispin = 0, num_spins-1 DO BEGIN
      IPS_source_start(ispin) = next_loc
      IES_source_start(ispin) = IPS_source_start(ispin) + IPS_total_len
      HISTe_source_start(ispin) = IES_source_start(ispin) + IES_total_len
      HISTp_source_start(ispin) = HISTe_source_start(ispin) + HISTe_total_len
      next_loc = HISTp_source_start(ispin) + HISTp_total_len
   ENDFOR


;	Determine values for the Source_loc array by starting with the initial
;	position in the array and with each spin/energy combination encountered
;	adding the length of the data output from the previous energy ... to
;	get the location of the start of data for the present spin/energy
;	combination.

;		IPS
 
   IPS_Source_loc = lonarr(num_spins,num_IPS_energies)
   FOR ispin = 0, num_spins-1 DO BEGIN

;	The location of energy 0 data is the start of IPS data within the 
;	current spin.
      IPS_Source_loc(ispin,0) = IPS_source_start(ispin)

;	Loop through the remaining energies - determine the location by adding
;	the previous energy's data length to the previous energy's source
;	location.
      FOR jenergy = 1, num_IPS_energies-1 DO BEGIN
         IPS_source_loc(ispin,jenergy) = $
                                IPS_source_loc(ispin,jenergy-1) + $
                                IPS_len_per_spin(jenergy-1)	
      ENDFOR

   ENDFOR


;		IES

   IES_Source_loc = lonarr(num_spins,num_IES_energies)
   FOR ispin = 0, num_spins-1 DO BEGIN

;	The location of energy 0 data is the start of IES data within the 
;	current spin.
      IES_Source_loc(ispin,0) = IES_source_start(ispin)

;	Loop through the remaining energies - determine the location by adding
;	the previous energy's data length to the previous energy's source
;	location.
      FOR jenergy = 1, num_IES_energies-1 DO BEGIN
         IES_source_loc(ispin,jenergy) = $
                                IES_source_loc(ispin,jenergy-1) + $
                                IES_len_per_spin(jenergy-1)	
      ENDFOR

   ENDFOR


;		HIST elctrons

   HISTe_Source_loc = lonarr(num_spins,num_HISTe_energies)
   FOR ispin = 0, num_spins-1 DO BEGIN

;	The location of energy 0 data is the start of HISTe data within the 
;	current spin.
      HISTe_Source_loc(ispin,0) = HISTe_source_start(ispin)

;	Loop through the remaining energies - determine the location by adding
;	the previous energy's data length to the previous energy's source
;	location.
      FOR jenergy = 1, num_HISTe_energies-1 DO BEGIN
         HISTe_source_loc(ispin,jenergy) = $
                                HISTe_source_loc(ispin,jenergy-1) + $
                                HISTe_len_per_spin(jenergy-1)	
      ENDFOR

   ENDFOR


;		HIST protons

   HISTp_Source_loc = lonarr(num_spins,num_HISTp_energies)
   FOR ispin = 0, num_spins-1 DO BEGIN

;	The location of energy 0 data is the start of HISTe data within the 
;	current spin.
      HISTp_Source_loc(ispin,0) = HISTp_source_start(ispin)

;	Loop through the remaining energies - determine the location by adding
;	the previous energy's data length to the previous energy's source
;	location.
      FOR jenergy = 1, num_HISTp_energies-1 DO BEGIN
         HISTp_source_loc(ispin,jenergy) = $
                                HISTp_source_loc(ispin,jenergy-1) + $
                                HISTp_len_per_spin(jenergy-1)	
      ENDFOR

   ENDFOR


END








PRO Def_Target_loc, error_return, $
		num_spins, num_energies, num_channels, $
		Spins_per_sample, Sectors_per_spin, $
		Source_len, Target_loc,	Target_time_loc, Target_block_length, $
                Target_byte_loc


;	Determine the locations in the 'target' 32-time level-1 array
;	into which the level-0 data will be transfered

;	Scalars and arrays passed to this procedure

;	num_spins
;	num_energies
;	num_channels
;	Spins_per_sample(num_energies)
;	Sectors_per_spin(num_energies,num_channels)
;	Source_len(num_spins,num_energies)


;	Scalars and arrays passed to the calling procedure

;	error_return
;	Target_loc(num_spins,num_energies)
;	Target_time_loc(num_spins)
;	Target_block_length
;	Target_byte_loc

;	Scalars and arrays used locally by this procedure

;	ienergy
;	ispin
;	jchannel
;	k
;	isubrecord
;	next_loc
;	num_subrecords
;	Energy_len(num_energies)
;	Subrecord_start_loc(num_subrecords)
;	Target_subrecord(num_spins,num_energies)
;	Energy_start_loc(isubrecord,jenergy)
;	start_byte_loc

;	revision
;	01-Jul 1996	M.T.Redding	Added line to require the TOTAL of
;					Source_len GT 0 before determining
;					output Target_byte_loc values - all
;					other output parameters are still
;					determined

; ----------------------------------------------------------------

   num_subrecords = 2*num_spins
   num_midsubrecord = num_spins

;	Calculate the total number of data values (indexes) for each energy
;	associated with a a single time.  In other words, the data for a given
;	energy may be accumulated and output over several spins but all of
;	the data values for this energy will be associated with a single
;	time; how many values from this enery will be associated with this 
;	single time? ... Do this for all energies.

   Energy_len = intarr(num_energies)
   FOR ienergy = 0, num_energies-1 DO BEGIN
      Energy_len(ienergy) = 0
      FOR jchannel = 0, num_channels-1 DO BEGIN
         Energy_len(ienergy) = Energy_len(ienergy) $
                             + Sectors_per_spin(ienergy,jchannel)
      ENDFOR
   ENDFOR


;	Calculate the lengths of the time-subrecords of the target array and 
;	determine the starting locations of the data portions (to be written
;	after the time value) of each.


   Subrecord_start_loc = lonarr(num_subrecords)
   Target_time_loc = lonarr(num_spins)
   next_loc = 0L
   FOR isubrecord = 0, num_subrecords-1 DO BEGIN

;	Assign the beginning of the subrecord to the 4-byte time value - there
;	are twice as many subrecords as spins - Time values will only be written
;	from the source array to the second half of the target array. The 
;	following code assigns location values twice to each target time 
;	location array element so that the locations of the beginings of 
;	subrecords in the 'last half' of the target array are assigned as time 
;	locations.

   Target_time_loc( isubrecord mod (num_spins) ) = next_loc

;	Allow the first 4 positions for the time value and assign the starting 
;	location

      next_loc = next_loc + 4
      Subrecord_start_loc(isubrecord) = next_loc

;	Add up lengths for all energies written to this subrecord to get the 
;	next subrecord start location

      FOR jenergy = 0, num_energies-1 DO BEGIN
         IF ( isubrecord MOD Spins_per_sample(jenergy) ) eq 0 THEN $
                  next_loc = next_loc + Energy_len(jenergy)
      ENDFOR

   ENDFOR
   Target_block_length = next_loc


;	Determine the time-subrecords to which data from each energy and spin 
;	are to be written.

   Target_subrecord = lonarr(num_spins,num_energies)
   FOR ispin = 0, num_spins-1 DO BEGIN
      FOR jenergy = 0, num_energies-1 DO BEGIN
         IF Spins_per_sample(jenergy) eq 1 THEN BEGIN
            Target_subrecord(ispin,jenergy) = num_midsubrecord + ispin 
         ENDIF ELSE BEGIN
            Target_subrecord(ispin,jenergy) = num_midsubrecord + $
                          ispin - ( ispin mod Spins_per_sample(jenergy) ) $
                                    - Spins_per_sample(jenergy)
         ENDELSE
      ENDFOR
   ENDFOR


;	Determine the start locations of each set of energy data within the
;	time-subrecords

   Energy_start_loc = lonarr(num_subrecords,num_energies)
   FOR isubrecord = 0, num_subrecords-1 DO BEGIN
      next_loc = Subrecord_start_loc(isubrecord)
      FOR jenergy = 0, num_energies-1 DO BEGIN
         Energy_start_loc(isubrecord,jenergy) = next_loc
         IF ( isubrecord MOD Spins_per_sample(jenergy) ) eq 0 THEN $
                  next_loc = next_loc + Energy_len(jenergy)
;         next_loc = next_loc + Energy_len(jenergy)
      ENDFOR
   ENDFOR


;	Make assignments to Target_loc array

   Target_loc = lonarr(num_spins,num_energies)
   FOR jenergy = 0, num_energies-1 DO BEGIN
      FOR ispin = 0, num_spins-1, Spins_per_sample(jenergy) DO BEGIN
         next_loc = Energy_start_loc(Target_subrecord(ispin,jenergy),jenergy)
         FOR k = 0, Spins_per_sample(jenergy)-1 DO BEGIN
            Target_loc(ispin+k,jenergy) = next_loc
            next_loc = next_loc + Source_len(ispin+k,jenergy)
         ENDFOR
      ENDFOR
   ENDFOR

;	Make assignments to Target_byte_loc array

   start_byte_loc = 0L
   IF ( TOTAL( Source_len) GT 0 ) THEN BEGIN
      Target_byte_loc = lonarr( TOTAL( Source_len) )
      FOR ispin = 0, num_spins-1 DO BEGIN
         FOR jenergy = 0, num_energies-1 DO BEGIN
            IF ( Source_len( ispin, jenergy ) GT 0 ) THEN $
               Target_byte_loc( start_byte_loc ) = $
                            LINDGEN( Source_len( ispin, jenergy ) ) + $
                               Target_loc( ispin, jenergy )
            start_byte_loc = start_byte_loc + Source_len( ispin, jenergy )
         ENDFOR
      ENDFOR
   ENDIF

END






pro Estimate_times, est_16_times, error_return

;	Make an array of 16 times based on the existant times within the
;	the first 16 times of the Trgt_times array (corresponding to the
;	32_time block) and filling in the missing times (those having values
;	of '-1') using interpolation/extrapolation.

;		Passed Parameters

;	est_16_times	array of 16 'estimated' times
;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	No defined times in 16-time 
;						array

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block
;	14 Mar 1996	M. T. Redding	Corrected handling of time blocks - now
;					fills in ALL -1L times if at least one
;					non-minus one value exists in Trgt_times
;					In particular, undefined (-1L) final
;					times will be estimated (previous
;					version left these undefined.
;					Changed the way estimates for undefined 
;					times are determined - now estimates
;					are interpolated between defined times
;	06 May 1996	M.T.Redding	Changed from single- to double-precision
;	30 May 1996	M.T.Redding	Changed all references to Trgt_times to
;					use Absolute value since the values
;					stored there now may be negative to
;					indicate they were 'estimated' rather
;					'evaluated' in the normal way
;--------------------------------------------------------------------

   COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                           sci_byt_per_spin, max_spins, $
                           nom_spin_dur, TRUE, FALSE, sync_byte, $
                           CEP_Sci_Man_byt_per_mnfrm, $
                           CEP_Conting_byt_per_mnfrm, $
                           CEP_Sci_Man_rec_len, $
                           CEP_Conting_rec_len

   COMMON CEP_lvl0_1_map, $
        Trgt_times, $
	num_source_spins,$
	num_IPS_energies, num_IPS_channels, IPS_spins_per_sample,$
		IPS_sectors_per_spin, IPS_Source_loc, IPS_Source_len,$
		IPS_Target_loc, IPS_trgt_tim_Loc, IPS_trgt_blk_len,$
                IPS_trgt_blk, IPS_source_byte_loc, IPS_trgt_byte_loc, $
	num_IES_energies, num_IES_channels, IES_spins_per_sample,$
		IES_sectors_per_spin, IES_Source_loc, IES_Source_len,$
		IES_Target_loc, IES_trgt_tim_loc, IES_trgt_blk_len,$
                IES_trgt_blk, IES_source_byte_loc, IES_trgt_byte_loc, $
	num_HISTe_energies, num_HISTe_channels, HISTe_spins_per_sample,$
		HISTe_sectors_per_spin, HISTe_Source_loc, HISTe_Source_len,$
		HISTe_Target_loc, HISTe_trgt_tim_loc, HISTe_trgt_blk_len,$
                HISTe_trgt_blk, HISTe_source_byte_loc, HISTe_trgt_byte_loc, $
	num_HISTp_energies, num_HISTp_channels, HISTp_spins_per_sample,$
		HISTp_sectors_per_spin, HISTp_Source_loc, HISTp_Source_len,$
		HISTp_Target_loc, HISTp_trgt_tim_loc, HISTp_trgt_blk_len,$
                HISTp_trgt_blk, HISTp_source_byte_loc, HISTp_trgt_byte_loc



;-----------------------------------------------------------------------

;	Initialize the time array to be filled.

   est_16_times    = lonarr(16)
   est_16_times(*) = -1L

   non_missing = where( Trgt_times(0:15) NE -1L )

;	Fill in any missing times

   size_non_missing = SIZE( non_missing )
   dim_non_missing  = size_non_missing(0)

   IF ( dim_non_missing EQ 0 ) THEN BEGIN	; No defined times in this block
       print, ' '
       print, ' 16-spin block encountered with NO defined times '
       error_return = 1
       RETURN
   ENDIF

;	Transfer what times do exist from the Trgt_times array 

   est_16_times(non_missing) = ABS( Trgt_times(non_missing) )
   len_non_missing  = size_non_missing(1)
   IF ( len_non_missing EQ 16 ) THEN BEGIN	; All 16 times already defined 
      error_return = 0
      RETURN
   ENDIF

;	Extrapolate where necessary to get 1st and last time in array

   first_non_missing = non_missing(0)
   last_non_missing  = non_missing(len_non_missing-1)


;	Determine an appropriate nominal time increment - base it on existing
;	time values if possible, otherwise use the defined nominal spin 
;	duration

   IF ( len_non_missing GE 2 ) THEN BEGIN
      ave_inc = DOUBLE( ABS( Trgt_times(non_missing(len_non_missing-1) )) - $
                        ABS( Trgt_times(non_missing(0)) )) / $
                DOUBLE( non_missing(len_non_missing-1) - non_missing(0) )
   ENDIF ELSE BEGIN
      ave_inc = DOUBLE( nom_spin_dur )
   ENDELSE


;	Estimate value for index 0 if necessary

   If ( first_non_missing NE 0 ) THEN BEGIN
      est_16_times(0) = LONG( DOUBLE( ABS( Trgt_times(first_non_missing) ) ) - $
                              DOUBLE( ABS( first_non_missing ) * ave_inc ) )
   ENDIF

;	Estimate value for index 15 if necessary

   If ( last_non_missing NE 15 ) THEN BEGIN
      est_16_times(15) = LONG( DOUBLE( ABS( Trgt_times(last_non_missing) ) ) + $
                               DOUBLE( 15 - last_non_missing ) * ave_inc )
   ENDIF

;	Now get the status of the Estimated time array

   non_missing = where( Est_16_times NE -1L )
   size_non_missing = SIZE( non_missing )
   dim_non_missing  = size_non_missing(0)

   IF ( dim_non_missing EQ 0 ) THEN BEGIN	; No defined times in this block
       print, ' 16-spin block encountered with NO defined times '
       error_return = 1
       RETURN
   ENDIF
   len_non_missing  = size_non_missing(1)

   IF ( len_non_missing EQ 16 ) THEN BEGIN	; All 16 times already defined 
      error_return = 0
      RETURN
   ENDIF

;	Calculate each missing time by adding to its predecessor an average
;	increment based on the non-missing time values.

   IF ( len_non_missing EQ 1 ) THEN BEGIN

;	Determine all but the first time using the nominal spin duration
;	and the first time

      Est_16_times = LONG( DINDGEN( 16 ) * ave_inc + Est_16_times( 0 ) )

   ENDIF ELSE BEGIN

;	Estimate values for undefined times by interpolating between existing
;	defined times

      FOR index = 0, len_non_missing-2 DO BEGIN

         ave_inc = DOUBLE( Est_16_times(non_missing(index+1)) - $
                          Est_16_times(non_missing(index)) ) / $
                   DOUBLE( non_missing(index+1) - non_missing(index) )  

         IF ( non_missing(index+1) - non_missing(index) GT 1 ) THEN BEGIN
            Est_16_times( non_missing(index)+1: non_missing(index+1)-1 ) = $
                    LONG( DOUBLE( Est_16_times(non_missing(index)) ) + $
                          ( DINDGEN( ( non_missing(index+1) - $
                                       non_missing(index) - 1 ) ) + 1.D0 ) * $
                          ave_inc )
         ENDIF

      ENDFOR

   ENDELSE

   error_return = 0

   RETURN

END





pro Eval_time, mjf_sequence_no, mnf_no, mnf_midpnt_time, error_return

;	Evaluate the start spin time using the major frame sequence
;	number and the minor frame number.

;		Passed Parameters

;	mjf_sequence_no	Major frame sequnce number (0-255)
;	mnf_no		Minor frame number
;	mnf_midpnt_time	Minor frame midpoint time
;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block
;	27 Oct 95	M.T.Redding	Made 'DOUBLE(mnf_no)' during
;					evaluation of mnf_midpnt_time

;--------------------------------------------------------------------

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                           sci_byt_per_spin, max_spins, $
                           nom_spin_dur, TRUE, FALSE, sync_byte, $
                           CEP_Sci_Man_byt_per_mnfrm, $
                           CEP_Conting_byt_per_mnfrm, $
                           CEP_Sci_Man_rec_len, $
                           CEP_Conting_rec_len

   COMMON CEP_mf_history,  previous_times

COMMON CEP_time_eval,   accept_mjf_lag, mjf_dtime, mnf_dtime, half_mnf_dtime, $
                        accept_SE_mjf_lag, accept_KP_mjf_lag 

;-----------------------------------------------------------------------


;	Find the major frame base time using the CEPPAD major frame 
;	sequence number.  Do a sanity check on the time value stored
;	there by comparing with the current major frame's time.  
;	Return a value in milliseconds if found otherwise return -1L.

;	Select the major time pointed to by the specified sequence number


   base_time = previous_times(mjf_sequence_no)

   IF (base_time lt 0 ) or $
         (MF_ATC_time gt base_time + accept_mjf_lag * mjf_dtime) THEN BEGIN
      mnf_midpnt_time = -1L
   ENDIF ELSE BEGIN
      mnf_midpnt_time = base_time + LONG(mnf_no*mnf_dtime + half_mnf_dtime)
   ENDELSE
   error_return = 0
   
   return
end




pro Eval_KP_time, mjf_sequence_no, mnf_no, mnf_midpnt_time, error_return

;	Evaluate the start spin time of Key Parameter data using the major 
;	frame sequence number and the minor frame number.
;	This routine differs from EVAL_TIME in that it uses a different
;	major frame lag constant for allowing reasonable times

;		Passed Parameters

;	mjf_sequence_no	Major frame sequnce number (0-255)
;	mnf_no		Minor frame number
;	mnf_midpnt_time	Minor frame midpoint time
;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block
;	27 Oct 95	M.T.Redding	Made 'DOUBLE(mnf_no)' during
;					evaluation of mnf_midpnt_time

;--------------------------------------------------------------------

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                           sci_byt_per_spin, max_spins, $
                           nom_spin_dur, TRUE, FALSE, sync_byte, $
                           CEP_Sci_Man_byt_per_mnfrm, $
                           CEP_Conting_byt_per_mnfrm, $
                           CEP_Sci_Man_rec_len, $
                           CEP_Conting_rec_len

   COMMON CEP_mf_history,  previous_times

COMMON CEP_time_eval,   accept_mjf_lag, mjf_dtime, mnf_dtime, half_mnf_dtime, $
                        accept_SE_mjf_lag, accept_KP_mjf_lag 

;-----------------------------------------------------------------------

;	Find the major frame base time using the CEPPAD major frame 
;	sequence number.  Do a sanity check on the time value stored
;	there by comparing with the current major frame's time.  
;	Return a value in milliseconds if found otherwise return -1L.

;	Select the major time pointed to by the specified sequence number


   base_time = previous_times( mjf_sequence_no )

   IF (base_time LT 0 ) OR $
         ( MF_ATC_time GT ( base_time + accept_KP_mjf_lag * mjf_dtime ) ) $
                                                                   THEN BEGIN
      mnf_midpnt_time = -1L
   ENDIF ELSE BEGIN
      mnf_midpnt_time = base_time + LONG( mnf_no*mnf_dtime + half_mnf_dtime )
   ENDELSE

   error_return = 0
   
   return
end




pro Extract_CEPPAD_spins, error_return

;	Control the reading of the CEPPAD Level-0 data records, processing
;	of Housekeeping data save in each major record and extraction of
;	data stored there from individual spins

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not used

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block
;					Commented out execution of CEP_Setup2
;					here - instead it is executed earlier
;					from Make_Level_1
;	22 Sep 1995	M. T. Redding	Added calls to update the headers and
;					close SH, Rates and HK L1 files as well
;					as a final update of the Normal mode
;					L1 file headers before closing
;	25 Sep 1995	M. T. Redding	Removed update of heading record
;					an close of HK file - these are both
;					performed in Process_HK.
;       10 Apr 1996	M. T. Redding   Added 'ignore_32_times' to 'CEP_32_time'
;                                       common block
;	06 May 1996	M. T. Redding	Added call to FILL_MISSING_TIMES before 
;					each call to MAP_TO_TIME_BLK to replace
;					any 'missing' (-1L) time values with
;					estimate values based on spin durations
;					determined in the current 16-spin block

;-----------------------------------------------------------------------

COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                        sci_byt_per_spin, max_spins, $
                        nom_spin_dur, TRUE, FALSE, sync_byte, $
                        CEP_Sci_Man_byt_per_mnfrm, $
                        CEP_Conting_byt_per_mnfrm, $
                        CEP_Sci_Man_rec_len, $
                        CEP_Conting_rec_len

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

COMMON CEP_32_time,     ignore_16_times, ignore_32_times

   COMMON CEP_lst_spn,     lst_spn_seq_no, lst_spn_min_frm, $
                           lst_spn_format, lst_spn_IPS_tab, $
                           lst_spn_IES_tab, lst_spn_IES_com, $
                           lst_spn_HIST_tab, lst_spn_HIST_com, $
                           lst_spn_SCOM, lst_spn_IPS_Mux, $
                           lst_spn_time, lst_spn_CRC, lst_spn_quality

   COMMON CEP_16_spin,     Spin_blk, Spin_blk_times, Spin_blk_SCOMs,	$
                           spin_blk_tot, spin_blk_seq_wrap,		$
                           est_16th_time, est_32nd_time, 		$
                           Bad_SCOM_MODE_sync, spin_quality

COMMON Dat_struct,      CEP_level_0_unit, data, CEP_Level_0_FLR

COMMON CEP_two_frm,     Buffer_index, Two_frm_sci_dat, Two_frm_qual

;--------------------------------------------------------------------

;	Perform CEPPAD specific setup, initialize the 16-spin buffer and 
;	pointers, initialize the Level-1 files parameters and CEPPAD
;	Subcom (Spacecraft Telemetry word 15) parameters

;   CEP_setup2, error_return
   Init_16_blk, error_return
   Level_1_setup, error_return
   Init_Subcom, error_return

;	Read first two data records (major frames) to initialized the buffers

;	First data record
   buffer_index = 0
   data_rec_no  = 1
   Read_data_rec, data_rec_no, buffer_index, error_return

;	Second data record
   buffer_index = 1
   data_rec_no  = 2
   Read_data_rec, data_rec_no, buffer_index, error_return

;	Initialize the spin pointers within the major frame buffers

   ths_spn_loc    = 0
   ths_spn_buf_no = 0
   nxt_spn_loc    = 0
   nxt_spn_buf_no = 0

;	Initialize run control parameters

   need_start_spin  = TRUE
   need_end_spin    = TRUE
   need_new_spin    = TRUE
   spin_start_found = FALSE
   spin_end_found   = FALSE

;	Read through the entire data file, processing POLAR Major frame data
;	and spacecraft and CEPPAD housekeeping data, all the while searching
;	for CEPPAD spin data ... extracting individual spins of data until 
;	all records are exhausted

   WHILE need_new_spin DO BEGIN		; Looking for both the start and end
					; of spin data

;	Look for beginning of spin data

      WHILE need_start_spin DO BEGIN		; Need to identify the sync
						; byte which will indicate the
						; start of spin data

         IF ( NOT EOF( CEP_Level_0_unit ) OR $
            ( frames_per_rec - ths_spn_loc GT frames_per_spin ) ) THEN BEGIN

            IF ths_spn_loc ge frames_per_rec THEN BEGIN

;		Prospective start of spin data is in the next major frame
;		Reference the spin start location pointer to the beginning of 
;		the new (alternate) record, then read another major frame 
;		(record) into the buffer previously used by the record just
;		completed  i.e. switch the use of the two frame science_data 
;		array (2-record buffer).

;		Reset prospective start of spin location to be in alternate 
;		buffer

               ths_spn_loc    = ths_spn_loc - frames_per_rec
               ths_spn_buf_no = buffer_index

;		Get another major frame's worth of data and put it into what 
;		is now the alternate buffer.

               IF NOT EOF(CEP_Level_0_unit) THEN BEGIN
                  buffer_index = ABS(buffer_index - 1)
                  data_rec_no  = data_rec_no + 1
                  Read_data_rec, data_rec_no, buffer_index, error_return

               ENDIF ELSE BEGIN
                  print, ' -- End-of-file encountered in Level-0 data file'

;		See if there enough minor frames left in current buffer to 
;		contain another spin of data

                  IF frames_per_rec - ths_spn_loc lt frames_per_spin $
                                                                THEN BEGIN

;		File records exhausted. No more spins left in Level-0 file. 
;		Transfer the remaining contents of the 16 spin buffer to the
;		32-time buffer and post the first 16 times and then the second
;		16 times to the Level-1 files.  Set flags to exit this loop; 
;		cancel the need for new_spin, start_spin and end_spin.

                     IF spin_blk_tot gt 0 THEN BEGIN

                        FILL_MISSING_TIMES, bad_filled_times
                        IF ( bad_filled_times EQ 0 ) THEN $
                           Map_to_time_blk, error_return
                        IF NOT Ignore_16_times THEN BEGIN
                           Append_L1, error_return
                        ENDIF

                        Shift_time_blk, error_return
                        IF ( bad_filled_times EQ 1 ) THEN $
                           ignore_16_times = TRUE ELSE $
                           ignore_16_times = FALSE

                        IF NOT ignore_16_times THEN BEGIN
                           Map_to_time_blk, error_return
                           Append_L1, error_return
                        ENDIF

                     ENDIF

                     need_new_spin   = FALSE
                     need_start_spin = FALSE
                     need_end_spin   = FALSE
                  ENDIF

                  print, ' '
                  print, ' -- Level-0 data file records exhausted --'
               ENDELSE

            ENDIF

;	See if prospective location has the beginning of spin data

            IF need_start_spin THEN BEGIN
               Verify_spin_start, spin_start_found, error_return

               IF NOT spin_start_found THEN BEGIN
                  ths_spn_loc     = ths_spn_loc + 10
               ENDIF ELSE BEGIN	; start of spin found
                  need_start_spin = FALSE
               ENDELSE

            ENDIF

         ENDIF	ELSE BEGIN	; ( NOT EOF(CEP_Level_0_unit ) OR $
				; ( frames_per_rec - ths_spn_loc GT $
				; frames_per_spin )

;		File records exhausted. No more spins left in Level-0 file. 
;		Transfer the remaining contents of the 16 spin buffer to the
;		32-time buffer and post the first 16 times and then the second
;		16 times to the Level-1 files.  Set flags to exit this loop; 
;		cancel the need for new_spin, start_spin and end_spin.

            IF spin_blk_tot gt 0 THEN BEGIN

               FILL_MISSING_TIMES, bad_filled_times
               IF ( bad_filled_times EQ 0 ) THEN $
                  Map_to_time_blk, error_return
               IF NOT Ignore_16_times THEN $
                  Append_L1, error_return

               Shift_time_blk, error_return
               IF ( bad_filled_times EQ 1 ) THEN $
                  ignore_16_times = TRUE ELSE $
                  ignore_16_times = FALSE
               IF NOT Ignore_16_times THEN $
                  Append_L1, error_return

            ENDIF

            need_new_spin   = FALSE
            need_start_spin = FALSE
            need_end_spin   = FALSE

            print, ' -- Level-0 data file records exhausted --'
         ENDELSE


      ENDWHILE	; need_start_spin
		

;	Verify this is actually spin data by checking the values at what
;	could be the start of the next spin of data

      IF need_end_spin THEN BEGIN
         nxt_spn_loc    = ths_spn_loc + frames_per_spin
         nxt_spn_buf_no = ths_spn_buf_no
         IF nxt_spn_loc ge frames_per_rec THEN BEGIN

;		Prospective start of next data is in the next major frame
;		Determine the end spin location with reference to the start 
;		of the new record and reset the buffer number

            nxt_spn_loc    = nxt_spn_loc - frames_per_rec
            nxt_spn_buf_no = ABS(ths_spn_buf_no - 1)
         ENDIF

         Verify_spin_end, spin_end_found, error_return

         IF NOT spin_end_found THEN BEGIN
            need_start_spin = TRUE
         ENDIF ELSE BEGIN	; end of spin found
            need_new_spin   = TRUE
            need_end_spin   = FALSE
         ENDELSE
      ENDIF 	; need_end_spin eq TRUE

      IF spin_end_found THEN BEGIN
          Unpack_ths_spn, error_return		; All data from 160 minor frames
          Process_new_spin, error_return	; Control copy to 16-spin buffer
;						; and mapping to 32-time buffer
;						; and writing to Level-1 files

      ENDIF

;		Set up to extract the next spin of data

         IF need_new_spin THEN BEGIN

            IF spin_end_found THEN BEGIN ;Look for next spin at end of this spin
               ths_spn_loc      = ths_spn_loc + frames_per_spin
               ths_spn_buf_no   = nxt_spn_buf_no
            ENDIF ELSE BEGIN ; Look for next spin at next possible sync location
               ths_spn_loc      = ths_spn_loc + 10
            ENDELSE

;		Reset flags to find both ends of new spin data
            need_start_spin  = TRUE
            need_end_spin    = TRUE
            spin_end_found   = FALSE
            spin_start_found = FALSE
         ENDIF

   ENDWHILE	; need_new_spin = TRUE

;	Write file-closing message to log

         print, ' Closing level 1 files at ending spin time = ', $
                lst_spn_time/1000.

         Update_L1_header, error_return
         Close_L1, error_return

         Update_Rates_headers, error_return
         Close_Rates_L1, error_return

         Update_SH_Header, error_return
         CLose_SH, error_return

   print, ' '
   print, ' -- Normal Termination of spin extraction --'
   error_return = 0
   RETURN
END



PRO FILL_FLR, FILE_LABEL_REC, FLR

; This procedure fills the file label record structure (FLR) by converting
; and combining bytes to their appropriate data types

;	revision:
;	02 Aug 95	M.T.Redding	Added extraction of edit file name and
;					edit file message key (reflecting the
;					changes specified in Nov 93 DATA
;					FORMAT CONTROL DOCUMENT - this required
;					changing the byte locations of the
;					other 'edit-file' variables

;	revision
;	15 Sep 1995	M. T. Redding	Replaced call to LONG_WORD with explicit
;					code to add products of bytes and powers
;					of 256 to obtain 'HOST' format LONGWORD
;					values from Level-0 file values
;	21 Mar 1995	M.T.Redding	Corrected extraction indexing of FLR 
;					Edit-file parameters - now uses 128
;					bytes per edit-file instead of 56
; ----------------------------------------------------------------

flr.spacecraft_id = $
        file_label_rec(  3)*16777216L + file_label_rec(  2)*65536L + $
        file_label_rec(  1)*256L +      file_label_rec(  0)
flr.instrument_num = $
        file_label_rec(  7)*16777216L + file_label_rec(  6)*65536L + $
        file_label_rec(  5)*256L +      file_label_rec(  4)
flr.instrument_nam = char_word(8, 4, file_label_rec)

flr.phys_rec_posit = $
        file_label_rec( 15)*16777216L + file_label_rec( 14)*65536L + $
        file_label_rec( 13)*256L +      file_label_rec( 12)
flr.phys_rec_mjfrm = $
        file_label_rec( 19)*16777216L + file_label_rec( 18)*65536L + $
        file_label_rec( 17)*256L +      file_label_rec( 16)
flr.phys_recs_file = $
        file_label_rec( 23)*16777216L + file_label_rec( 22)*65536L + $
        file_label_rec( 21)*256L +      file_label_rec( 20)

flr.beg_mjfrm_cnt = $
        file_label_rec( 27)*16777216L + file_label_rec( 26)*65536L + $
        file_label_rec( 25)*256L +      file_label_rec( 24)
flr.end_mjfrm_cnt = $
        file_label_rec( 31)*16777216L + file_label_rec( 30)*65536L + $
        file_label_rec( 29)*256L +      file_label_rec( 28)

flr.beg_spc_clock(0:7) = file_label_rec(32:39)
flr.end_spc_clock(0:7) = file_label_rec(40:47)

flr.year_beg = $
        file_label_rec( 51)*16777216L + file_label_rec( 50)*65536L + $
        file_label_rec( 49)*256L +      file_label_rec( 48)
flr.day_beg = $
        file_label_rec( 55)*16777216L + file_label_rec( 54)*65536L + $
        file_label_rec( 53)*256L +      file_label_rec( 52)
flr.milsec_beg = $
        file_label_rec( 59)*16777216L + file_label_rec( 58)*65536L + $
        file_label_rec( 57)*256L +      file_label_rec( 56)
flr.micro_sec_beg = $
        file_label_rec( 63)*16777216L + file_label_rec( 62)*65536L + $
        file_label_rec( 61)*256L +      file_label_rec( 60)

flr.year_end = $
        file_label_rec( 67)*16777216L + file_label_rec( 66)*65536L + $
        file_label_rec( 65)*256L +      file_label_rec( 64)
flr.day_end =  $
        file_label_rec( 71)*16777216L + file_label_rec( 70)*65536L + $
        file_label_rec( 69)*256L +      file_label_rec( 68)
flr.milsec_end = $
        file_label_rec( 75)*16777216L + file_label_rec( 74)*65536L + $
        file_label_rec( 73)*256L +      file_label_rec( 72)
flr.micro_sec_end = $
        file_label_rec( 79)*16777216L + file_label_rec( 78)*65536L + $
        file_label_rec( 77)*256L +      file_label_rec( 76)

flr.mjfrm_expct = $
        file_label_rec( 83)*16777216L + file_label_rec( 82)*65536L + $
        file_label_rec( 81)*256L +      file_label_rec( 80)
flr.mjfrm_file = $
        file_label_rec( 87)*16777216L + file_label_rec( 86)*65536L + $
        file_label_rec( 85)*256L +      file_label_rec( 84)
flr.mjfrm_lev_gap = $
        file_label_rec( 91)*16777216L + file_label_rec( 90)*65536L + $
        file_label_rec( 89)*256L +      file_label_rec( 88)

flr.data_covage_typ = char_word( 92, 4, file_label_rec)
flr.decom_rerun_num = $
        file_label_rec( 99)*16777216L + file_label_rec( 98)*65536L + $
        file_label_rec( 97)*256L +      file_label_rec( 96)
flr.decom_prog_vers = char_word( 100, 8, file_label_rec)
flr.decom_datbse_no = char_word( 108, 8, file_label_rec)
flr.decom_ydhms_mil = char_word( 116, 16, file_label_rec)

flr.inst_file_name  = char_word( 132, 44, file_label_rec)
flr.phys_rec_len = $
        file_label_rec(179)*16777216L + file_label_rec(178)*65536L + $
        file_label_rec(177)*256L +      file_label_rec(176)

flr.merge_rerun_no = $
        file_label_rec(203)*16777216L + file_label_rec(202)*65536L + $
        file_label_rec(201)*256L +      file_label_rec(200)
flr.merge_vers_no = char_word( 204, 8, file_label_rec)
flr.merge_ydhms_mil = char_word( 212, 16, file_label_rec)
flr.n_edit_files = $
        file_label_rec(231)*16777216L + file_label_rec(230)*65536L + $
        file_label_rec(229)*256L +      file_label_rec(228)
edit0 = strarr(20)	; Edit file names (44 characters)
edit1 = strarr(20)	; Edit file keys (24 characters)
edit2 = intarr(20)	; Edit file rerun numbers (longword integer)
edit3 = strarr(20)	; Edit file program version numbers (8 characters)
edit4 = strarr(20)	; Edit file run date/time (16 characters)
edit5 = strarr(20)	; Edit filedata type (4 characters)
edit6 = strarr(20)	; Edit file message key (28 characters)

for i = 0, flr.n_edit_files - 1 do begin
     ii =  i * 128
     edit0(i) = char_word( ii+232, 44, file_label_rec)
     edit1(i) = char_word( ii+276, 24, file_label_rec)
     edit2(i) = $
        file_label_rec(ii+303)*16777216L + file_label_rec(ii+302)*65536L + $
        file_label_rec(ii+301)*256L +      file_label_rec(ii+300)
     edit3(i) = char_word( ii+304,  8, file_label_rec)
     edit4(i) = char_word( ii+312, 16, file_label_rec)
     edit5(i) = char_word( ii+328,  4, file_label_rec)
     edit6(i) = char_word( ii+332, 28, file_label_rec)
endfor    

flr.edit_fil_name  = edit0
flr.edit_fil_key   = edit1
flr.edit_rerun_num = edit2
flr.edit_prog_vers = edit3
flr.edit_ydhms_run = edit4
flr.edit_data_type = edit5
flr.edit_msg_key   = edit6

return
end




pro Fill_missing_times, error_return

;	Replace missing values ( -1L ) in the spin_blk_times array with 
;	estimates based on the existant times within the array using 
;	interpolation/extrapolation where necessary.  Actual replacement values 
;	are -1L * estimate value - this is done so that identification of
;	estimate times can still be done.
;	This routine replaces Spin-block times as compared with procedure
;	'Estimate_times' which determines temporary values for the first 16 
;	times found in the Time-block.
;	The routine also recalculates the average spin-duration value found
;	in the variable 'nom-spin-dur' as well as revising the 256-element
;	spin duration history correspong with 'previous_spin_times' for
;	all estimated elements in the spin block except the last 
;	(SCOM mod 16 = 15 )

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	No defined times in 16-time 
;						array

;	revision:
;	06 May 1996	M.T.Redding	Derived from procedure 'Estimate_times' 
;	07 May 1996	M.T.Redding	Added code to calculate the average
;					spin-duration and redefine 
;					'nom_spin_dur' with that value.
;	23 May 1996	M.T.Redding	Added array 'filled_tim_indx' and lines
;					to convert estimated time values to
;					negatives
;					Only passes back sets of times 
;					after 12:00 midnight (0.0) and only
;					sets which are monotonically increasing
;	29 May 1996	M.T.Redding	Added COMMON block CEP_spin_history so
;					that estimates of spins times may be
;					used to update the previous_spin_times
;					array
;	04 Jun 1996	M.T.Redding	Added code to revise the newly added
;					spin duration array 'spin_durations'
;					for spins (other than spin 16) for 
;					which spin times are estimated
;	13 Jun 1996	M.T.Redding	Added code to make '-1Ls' all  times is
;					either the 1st or 16th times are
;					estimated andtuen out to be negative

;--------------------------------------------------------------------

   COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                           sci_byt_per_spin, max_spins, $
                           nom_spin_dur, TRUE, FALSE, sync_byte, $
                           CEP_Sci_Man_byt_per_mnfrm, $
                           CEP_Conting_byt_per_mnfrm, $
                           CEP_Sci_Man_rec_len, $
                           CEP_Conting_rec_len

   COMMON CEP_16_spin,     Spin_blk, Spin_blk_times, Spin_blk_SCOMs,	$
                           spin_blk_tot, spin_blk_seq_wrap,		$
                           est_16th_time, est_32nd_time, 		$
                           Bad_SCOM_MODE_sync, spin_quality

   COMMON CEP_spin_history, previous_spin_times, spin_durations

;bugbug
   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

   COMMON CEP_lst_spn,     lst_spn_seq_no, lst_spn_min_frm, $
                           lst_spn_format, lst_spn_IPS_tab, $
                           lst_spn_IES_tab, lst_spn_IES_com, $
                           lst_spn_HIST_tab, lst_spn_HIST_com, $
                           lst_spn_SCOM, lst_spn_IPS_Mux, $
                           lst_spn_time, lst_spn_CRC, lst_spn_quality
;bugbug

;-----------------------------------------------------------------------

;bugbug
;IF ( ( ths_spn_time GE 0 ) AND ( ths_spn_time LE -1 ) ) THEN BEGIN
;   print, ' Fill_missing_times 0: Inputs'
;   print, ' nom_spin_dur = ', nom_spin_dur
;   print, ' Spin_blk_times = '
;   print, Spin_blk_times,  FORMAT = "(8i9)"
;
;   print, ' THS_SPN parameters
;   print, ths_spn_seq_no, ths_spn_min_frm, $
;          ths_spn_format, ths_spn_IPS_tab, $
;          ths_spn_IES_tab, ths_spn_IES_com, $
;          ths_spn_HIST_tab, ths_spn_HIST_com, $
;          ths_spn_SCOM, ths_spn_IPS_Mux, $
;          ths_spn_time, ths_spn_CRC
;   print, ' LST_SPN parameters
;   print, lst_spn_seq_no, lst_spn_min_frm, $
;          lst_spn_format, lst_spn_IPS_tab, $
;          lst_spn_IES_tab, lst_spn_IES_com, $
;          lst_spn_HIST_tab, lst_spn_HIST_com, $
;          lst_spn_SCOM, lst_spn_IPS_Mux, $
;          lst_spn_time, lst_spn_CRC
;   print, ' est_16th_time = ', est_16th_time
;   print, ' '
;ENDIF
;bugbug


   error_return = 0

;	Initialize the time array to be filled.

   est_16_times      = lonarr( 16 )
   est_16_times( * ) = -1L

   good_tim_indx   = WHERE( Spin_blk_times( 0: 15 ) NE -1L )
   filled_tim_indx = WHERE( Spin_blk_times( 0: 15 ) EQ -1L, filled_tim_count )

;	Fill in any missing times

   size_good_tim_indx = SIZE( good_tim_indx )
   dim_good_tim_indx  = size_good_tim_indx( 0 )

   IF ( dim_good_tim_indx EQ 0 ) THEN BEGIN	; No defined times in this block
       error_return = 1
       RETURN
   ENDIF

;	Transfer what times do exist from the Spin_blk_times array 

   est_16_times( good_tim_indx ) = Spin_blk_times( good_tim_indx )
   len_good_tim_indx  = size_good_tim_indx( 1 )
   IF ( len_good_tim_indx EQ 16 ) THEN BEGIN	; All 16 times already defined 
      times_lt_zero = WHERE( est_16_times LT 0, num_times_lt_zero )
      IF ( num_times_lt_zero EQ 0 ) THEN BEGIN
         error_return = 0
      ENDIF ELSE BEGIN
         Spin_blk_times( * ) = -1L
         error_return = 1
      ENDELSE
      RETURN
   ENDIF


;	Construct an SCOM array for this block

   ths_blk_SCOMs = BINDGEN( 16 ) + $
                   ( ( Spin_blk_SCOMs( good_tim_indx(0) ) )/ 16B) * 16B


;	Extrapolate where necessary to get 1st and last time in array

   first_good_tim_indx = good_tim_indx( 0 )
   last_good_tim_indx  = good_tim_indx( len_good_tim_indx - 1 )


;	Determine an appropriate nominal time increment - base it on existing
;	time values if possible, otherwise use the defined nominal spin 
;	duration

   IF ( len_good_tim_indx GE 2 ) THEN BEGIN
      ave_inc = $
         DOUBLE( Spin_blk_times( good_tim_indx( len_good_tim_indx - 1 )) - $
                 Spin_blk_times( good_tim_indx( 0 ) ) ) / $
         DOUBLE( good_tim_indx( len_good_tim_indx - 1 ) - good_tim_indx( 0 ) )
   ENDIF ELSE BEGIN
      ave_inc = DOUBLE( nom_spin_dur )
   ENDELSE


;	Estimate value for index 0 if necessary

   If ( first_good_tim_indx NE 0 ) THEN BEGIN
      est_16_times( 0 ) = $
         LONG( DOUBLE( Spin_blk_times( first_good_tim_indx) ) - $
               DOUBLE( first_good_tim_indx ) * ave_inc )
   ENDIF

;	Estimate value for index 15 if necessary

   If ( last_good_tim_indx NE 15 ) THEN BEGIN
      est_16_times( 15 ) = $
         LONG( DOUBLE( Spin_blk_times( last_good_tim_indx) ) + $
               DOUBLE( 15 - last_good_tim_indx ) * ave_inc )
   ENDIF

;	Now get the status of the Estimated time array

   good_tim_indx      = WHERE( Est_16_times NE -1L )
   size_good_tim_indx = SIZE( good_tim_indx )
   dim_good_tim_indx  = size_good_tim_indx( 0 )
   len_good_tim_indx  = size_good_tim_indx( 1 )

   IF ( len_good_tim_indx EQ 16 ) THEN BEGIN	; All 16 times already defined 
      error_return   = 0

;	Update spin-time history array

      previous_spin_times( ths_blk_SCOMs ) = est_16_times( * )


;	Transfer working values and convert estimates to negative

      Spin_blk_times = Est_16_times
      Spin_blk_times( filled_tim_indx ) = -1L * Spin_blk_times( filled_tim_indx)


;	Determine the average spin-duration

         nom_spin_dur = ( Est_16_times(15) - Est_16_times(0) ) / 15L

;bugbug
;IF ( nom_spin_dur LT 5000 ) THEN BEGIN
;   print, ' Fill_missing_times 1: Unusually low spin duration calculated'
;   print, ' nom_spin_dur = ', nom_spin_dur
;   print, ' Est_16_times = '
;   print, Est_16_times, FORMAT = "(8i9)"
;   print, ' Spin_blk_times = '
;   print, Spin_blk_times,  FORMAT = "(8i9)"
;
;   print, ' THS_SPN parameters
;   print, ths_spn_seq_no, ths_spn_min_frm, $
;          ths_spn_format, ths_spn_IPS_tab, $
;          ths_spn_IES_tab, ths_spn_IES_com, $
;          ths_spn_HIST_tab, ths_spn_HIST_com, $
;          ths_spn_SCOM, ths_spn_IPS_Mux, $
;          ths_spn_time, ths_spn_CRC
;   print, ' LST_SPN parameters
;   print, lst_spn_seq_no, lst_spn_min_frm, $
;          lst_spn_format, lst_spn_IPS_tab, $
;          lst_spn_IES_tab, lst_spn_IES_com, $
;          lst_spn_HIST_tab, lst_spn_HIST_com, $
;          lst_spn_SCOM, lst_spn_IPS_Mux, $
;          lst_spn_time, lst_spn_CRC
;   print, ' est_16th_time = ', est_16th_time
;ENDIF
;bugbug


;	Update spin-time duration array

         n_corrected_dur = N_ELEMENTS( filled_tim_indx )
         IF ( n_corrected_dur GT 0 ) THEN BEGIN
            IF ( filled_tim_indx( n_corrected_dur - 1 ) EQ 15 ) THEN $
               n_corrected_dur = n_corrected_dur - 1
            IF ( n_corrected_dur GT 0 ) THEN $
               spin_durations( ths_blk_SCOMs( 0 ) + $
                               filled_tim_indx( 0: n_corrected_dur - 1 ) ) = $
                  Est_16_times( $
                     filled_tim_indx( 0: n_corrected_dur - 1 ) + 1 ) - $
                  Est_16_times( $
                     filled_tim_indx( 0: n_corrected_dur - 1 ) )
         ENDIF

;	Return all -1s if any estimate times are negative

         times_lt_zero = WHERE( ( est_16_times LT 0 ), num_times_lt_zero )
         IF ( num_times_lt_zero GT 0 ) THEN BEGIN
            Spin_blk_times( * ) = -1L
            error_return = 1
            RETURN
         ENDIF


      RETURN
   ENDIF

;	Calculate each missing time by adding to its predecessor an average
;	increment based on the non-missing time values.

   IF ( len_good_tim_indx EQ 1 ) THEN BEGIN

;	Determine all but the first time using the nominal spin duration
;	and the first time

      Est_16_times = LONG( DINDGEN( 16 ) * ave_inc + Est_16_times( 0 ) )

   ENDIF ELSE BEGIN

;	Estimate values for undefined times by interpolating between existing
;	defined times

      FOR index = 0, len_good_tim_indx - 2 DO BEGIN

         IF ( good_tim_indx( index + 1 ) - good_tim_indx( index ) GT 1 ) $
                                                                     THEN BEGIN

            ave_inc = DOUBLE( Est_16_times( good_tim_indx( index + 1 ) ) - $
                             Est_16_times( good_tim_indx( index ) ) ) / $
                      DOUBLE( good_tim_indx( index + 1 ) - $
                             good_tim_indx( index ) )

            Est_16_times( good_tim_indx( index ) + 1: $
                          good_tim_indx( index + 1 ) - 1 ) = $
               LONG( DOUBLE( Est_16_times( good_tim_indx( index ) ) ) + $
                     ( DINDGEN( ( good_tim_indx( index + 1 ) - $
                                  good_tim_indx( index ) - 1 ) ) + 1.D0 ) * $
                     ave_inc )
         ENDIF

      ENDFOR

   ENDELSE

   error_return = 0

;	Update spin-time history array

   previous_spin_times( ths_blk_SCOMs ) = est_16_times( * )


;	Return all -1s if any estimate times are negative

   times_lt_zero = WHERE( ( est_16_times LT 0 ), num_times_lt_zero )
   IF ( num_times_lt_zero GT 0 ) THEN BEGIN
      Spin_blk_times( * ) = -1L
      error_return = 1

      RETURN
   ENDIF


;	Transfer working values and convert estimates to negative

   Spin_blk_times = Est_16_times
   Spin_blk_times( filled_tim_indx ) = -1L * Spin_blk_times( filled_tim_indx)


;	Determine the average spin-duration

   nom_spin_dur = ( Est_16_times(15) - Est_16_times(0) ) / 15L

;bugbug
;IF ( nom_spin_dur LT 5000 ) THEN BEGIN
;   print, ' Fill_missing_times 2: Unusually low spin duration calculated'
;   print, ' nom_spin_dur = ', nom_spin_dur
;   print, ' Est_16_times = '
;   print, Est_16_times, FORMAT = "(8i9)"
;   print, ' Spin_blk_times = '
;   print, Spin_blk_times,  FORMAT = "(8i9)"
;
;   print, ' THS_SPN parameters
;   print, ths_spn_seq_no, ths_spn_min_frm, $
;          ths_spn_format, ths_spn_IPS_tab, $
;          ths_spn_IES_tab, ths_spn_IES_com, $
;          ths_spn_HIST_tab, ths_spn_HIST_com, $
;          ths_spn_SCOM, ths_spn_IPS_Mux, $
;          ths_spn_time, ths_spn_CRC
;   print, ' LST_SPN parameters
;   print, lst_spn_seq_no, lst_spn_min_frm, $
;          lst_spn_format, lst_spn_IPS_tab, $
;          lst_spn_IES_tab, lst_spn_IES_com, $
;          lst_spn_HIST_tab, lst_spn_HIST_com, $
;          lst_spn_SCOM, lst_spn_IPS_Mux, $
;          lst_spn_time, lst_spn_CRC
;   print, ' est_16th_time = ', est_16th_time
;ENDIF
;bugbug


;	Update spin-time duration array

      n_corrected_dur = N_ELEMENTS( filled_tim_indx )
      IF ( n_corrected_dur GT 0 ) THEN BEGIN
         IF ( filled_tim_indx( n_corrected_dur - 1 ) EQ 15 ) THEN $
            n_corrected_dur = n_corrected_dur - 1
         IF ( n_corrected_dur GT 0 ) THEN $
            spin_durations( ths_blk_SCOMs( 0 ) + $
                            filled_tim_indx( 0: n_corrected_dur - 1 ) ) = $
               Est_16_times( $
                  filled_tim_indx( 0: n_corrected_dur - 1 ) + 1 ) - $
               Est_16_times( $
                  filled_tim_indx( 0: n_corrected_dur - 1 ) )
      ENDIF

   RETURN
END






pro Get_checksum, error_return

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;--------------------------------------------------------------------

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON Table_CRCs,      CRC_Flt_SW, CRC_MST, CRC_SAT, CRC_ICT, $
                           CRC_IESLUT, CRC_HISTLUT

;-----------------------------------------------------------------------

   error_return = 0

   Subcom_format = Subcom(90)

;	Unpack the Subcom values based on the format number

;	Table CRCs Dump

;		Subcom_format = 4
   IF ( Subcom_format EQ 4 ) THEN BEGIN
      FOR i = 0, 15 DO CRC_MST(i)     = FIX(Subcom( 94+i)*256) + Subcom( 93+i)
   ENDIF

RETURN
end



PRO GET_HIST_SE_TIMES, Sbcom,			$
                       Mn_frm_quality, 		$
                       CEP_HIST_SE_times,	$
                       error_return

;	Calculate the mid-sector time associated with each set of HIST-singles
;	or HIST-events data.

;	Determine if minor frame quality allows it and type of data stored
;	in word-15 minor frames 90-249 contains either HIST singles data or
;	HIST event data - if so extract the spin times associated to the data
;	by SCOM values from the history of spin times.  Times extracted must
;	be within a certain number of major frame before the current major
;	frame.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revisions
;	07 May 1996	M.T.Redding
;	24 May 1996	M.T.Redding	Uses positional arguments for
;					Sbcom, Mn_frm_quality and 
;					CEP_HIST_SE_times instead of COMMON
;					block entries
;	03 Jun 1996	M.T.Redding	Calculates mid_sector time for each
;					set

;--------------------------------------------------------------------


   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON CEP_mf_history,  previous_times

   COMMON CEP_spin_history, previous_spin_times, spin_durations

   COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                           sci_byt_per_spin, max_spins, $
                           nom_spin_dur, TRUE, FALSE, sync_byte, $
                           CEP_Sci_Man_byt_per_mnfrm, $
                           CEP_Conting_byt_per_mnfrm, $
                           CEP_Sci_Man_rec_len, $
                           CEP_Conting_rec_len

;-----------------------------------------------------------------------

;	Initialize HIST singles/events times to -1L

   CEP_HIST_SE_times = LONARR( 8 )
   CEP_HIST_SE_times( * )= -1L

;	Check for HIST-singles or HIST-events mode

   IF ( ( Mn_frm_quality( 90 ) EQ 0 ) AND $
        ( ( Sbcom( 90 ) EQ 0B ) OR ( Sbcom( 90 ) EQ 1B ) ) ) THEN BEGIN

      FOR iset = 0, 7 DO BEGIN

         HIST_S_E_SCOM = Sbcom( 91 + iset * 18 )
;;         IF ( ( Mn_frm_quality( 91 + iset * 18 ) EQ 0 ) AND	$
;;              ( Mn_frm_quality( 92 + iset * 18 ) EQ 0 ) AND	$
;;              ( HIST_S_E_SCOM NE 255B ) ) THEN BEGIN
         IF ( ( Mn_frm_quality( 91 + iset * 18 ) EQ 0 ) AND	$
              ( Mn_frm_quality( 92 + iset * 18 ) EQ 0 ) ) THEN BEGIN

            GET_SPIN_TIME, HIST_S_E_SCOM,	$
                           spin_time,		$
                           spin_duration,	$
                           error_return

;	Using predefined values base on expected sector numbers ('0', '8',
;	'16', and '24'), the number of sectors per spin (4) and the current
;	nominal spin duration, determine the time at the midpoint of the 
;	sampling interval (mid-sector) i.e. calculate the spin_time + 1/8,
;	3/8, 5/8 and 7/8 spin duration

            IF ( spin_time NE -1L ) THEN BEGIN
               IF ( spin_duration EQ -1L ) THEN spin_duration = nom_spin_dur
               HIST_S_E_sector = Sbcom( 92 + iset * 18 )
               CEP_HIST_SE_times( iset ) = $
                  LONG( DOUBLE( spin_time ) + $
                     ( DOUBLE( spin_duration * ( 1 + HIST_S_E_sector / 4) ) / $
                       8.0d0 ) )
            ENDIF

         ENDIF 

      ENDFOR

   ENDIF

   RETURN
end





pro GET_KP_TIME, error_return

;	Determine if minor frame quality allows it and if the Key Parameter
;	subcom indicates that this major frame should contain a valid time
;	tag (sequence no/minor frame no) and evaluate the time

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revisions
;	07 May 1996	M.T.Redding

;--------------------------------------------------------------------


   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON CEP_mf_history,  previous_times

   COMMON CEP_HK_times,    CEP_KP_time, $
                           CEP_HIST_SE_times

;-----------------------------------------------------------------------


;	Initialize Key Parameter to -1L

   CEP_KP_time = -1L

;	Evaluate Key Parameter time where necessary

   IF ( ( Min_frm_quality( 33 ) EQ 0 ) AND $
        ( Min_frm_quality( 69 ) EQ 0 ) )THEN BEGIN
      KP_subcom = ISHFT( HK( 33, 1 ), -4 )	; 4 MSB
      IF ( KP_subcom EQ 0 ) THEN BEGIN

;	Check for non-zero values occuring in at least one of the KP parameters
;	stored in this major frame.
;	We assume that if all are zero, then Key parameters were not being
;	calculated by the DPU and so the time value should not be evaluated 
;	here (using the '0' seq number and '0' minor frame value).

         IF ( ( HK(  69, 0 ) NE 0 )                OR	$
              ( HK(  69, 1 ) NE 0 )                OR	$
              ( ( Min_frm_quality( 81 ) EQ 0 ) AND	$
                  ( ( HK(  81, 1 ) NE 0 ) OR		$
                    ( HK(  81, 1 ) NE 0 ) ) )      OR	$
              ( ( Min_frm_quality( 93 ) EQ 0 ) AND	$
                  ( ( HK(  93, 0 ) NE 0 ) OR		$
                    ( HK(  93, 1 ) NE 0 ) ) )      OR	$
              ( ( Min_frm_quality( 105 ) EQ 0 ) AND	$
                  ( ( HK( 105, 1 ) NE 0 ) OR		$
                    ( HK( 105, 1 ) NE 0 ) ) ) )      THEN BEGIN
              EVAL_KP_TIME, HK(69,0), HK(69,1), CEP_KP_time, error_return
         ENDIF
      ENDIF
   ENDIF

   RETURN
end





PRO GET_MAJFRM_DATA, WCH_REC, CEPDRH, CEP_SCIENCE, CEPHK, CEPSC, IERROR

;    This procedure takes a masterframe of data, 'CEP_TELEMETRY', and fills
;    the structure, and 'CEPDRH', 'CEP_SCIENCE', 'CEP_HK',and  'CEPSC' arrays.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

; ----------------------------------------------------------------

 common cep_phys_rec, cep_byts_rec, cep_byts_mnfr, max_phys_rec
 common dat_struct, iunit, data, FLR
 common pl_phys_rec, mnfrm_rec


; ----------------------------------------------------------------

cep_telemetry = data(wch_rec)      ; Cep_telemetry is a masterframe of data.
                                   ; Wch_rec indicates which masterframe of data

if( wch_rec gt max_phys_rec - 1) or (wch_rec lt 1 ) then begin
     print, ' '
     print, ' TRYING TO ACCESS A NON-EXISTENT RECORD,',wch_rec
     ierror = 1
     return
endif


; In each level zero file at the beginning of each data record ( CEP_TELEMETRY)
; are three-hundred Data Record Header (DRH) bytes described by the data 
; structure, CEPDRH.

;  The CEPPAD housekeeping arrays are CEPHK(250, 2) representing words 11 and
;     12 from each minor frame of the spacecraft instruments telemetry stream.
;     In CEPPADs Level0 files they are words 0 and 1 of each minor frame.

;  The CEPPAD subcom array is CEPSC, word 2 in the level0 file.

;  The CEPPAD science array is CEP_SCIENCE wrds 3 - 17 from each minor frame
;  in the level0 file 

; UNPK_CEP_TM procedure unpacks the CEPPAD telemetry and fills up the appropri-
; ate arrays and/or structures as described above with a masterframe 
; (250 minor frames) of data


UNPK_CEP_TM, cep_telemetry, cepdrh, cep_science, cephk, cepsc

return
end




PRO GET_SPIN_TIME, SCOM, spin_time, spin_duration, error_return

;	Pass back the time referenced by the SCOM value if it is within 
;	the a certain range of the current major frame time.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revisions
;	07 May 1996	M.T.Redding
;	04 Jun 1996	M.T.Redding	Added spin_duration as a return variable

;--------------------------------------------------------------------


   COMMON CEP_spin_history, previous_spin_times, spin_durations

   COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                           sci_byt_per_spin, max_spins, $
                           nom_spin_dur, TRUE, FALSE, sync_byte, $
                           CEP_Sci_Man_byt_per_mnfrm, $
                           CEP_Conting_byt_per_mnfrm, $
                           CEP_Sci_Man_rec_len, $
                           CEP_Conting_rec_len

COMMON CEP_time_eval,   accept_mjf_lag, mjf_dtime, mnf_dtime, half_mnf_dtime, $
                        accept_SE_mjf_lag, accept_KP_mjf_lag 

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

;-----------------------------------------------------------------------




   temp_time = previous_spin_times( SCOM )

   IF ( previous_spin_times( SCOM ) GT MF_ATC_time - $
                                    accept_SE_mjf_lag * mjf_dtime ) AND $
      ( previous_spin_times( SCOM ) LT MF_ATC_time + mjf_dtime ) THEN BEGIN
      spin_time     = previous_spin_times( SCOM )
      spin_duration = spin_durations( SCOM )
   ENDIF ELSE BEGIN
      spin_time     = -1L
      spin_duration = nom_spin_dur
   ENDELSE

   error_return = 0
   


   RETURN
end



PRO INIT_CEPDRH, CEPDRH
                   
; Initialize  data DATA RECORD HEADER (DRH) structure

; ----------------------------------------------------------------

CEPDRH =  { HEADER, inst_num:  long(0),  $
rec_num:  long(0), $
mfrm_cnt: long(0), $ 
spc_clk:  lonarr(6), $
atc_yr:   long(0), $ 
atc_day :  long(0), $ 
atc_mils:  long(0), $ 
atc_micro:  long(0), $ 
mnfr_fil:  long(0), $ 
mnfr_syn:  long(0), $ 
tm_mode :  long(0), $ 
mnfr_qua:  bytarr(250) }

return
end




PRO INIT_FLR, FLR

; Initialize  data FILE LABEL RECORD (FLR) structure

;	02 Aug 95	M.T.Redding	Added FLR.edit_fil_name and
;					FLR.edit_msg_key to reflext changes
;					described in Nov 93 DATA FORMAT CONTROL
;					DOCUMENT
; ----------------------------------------------------------------

  FLR =   { LABEL_REC,              $
  spacecraft_id:   long(0),         $
  instrument_num:  long(0),         $
  instrument_nam:  strarr(1),       $
  phys_rec_posit:  long(0),         $
  phys_rec_mjfrm:  long(0),         $
  phys_recs_file:  long(0),         $
  beg_mjfrm_cnt:   long(0),         $
  end_mjfrm_cnt:   long(0),         $
  beg_spc_clock:   bytarr(8),       $
  end_spc_clock:   bytarr(8),       $
  year_beg:        long(0),         $
  day_beg:         long(0),         $
  milsec_beg:      long(0),         $
  micro_sec_beg:   long(0),         $
  year_end:        long(0),         $
  day_end:         long(0),         $
  milsec_end:      long(0),         $
  micro_sec_end:   long(0),         $
  mjfrm_expct:     long(0),         $
  mjfrm_file:      long(0),         $
  mjfrm_lev_gap:   long(0),         $    
  data_covage_typ: strarr(1),       $
  decom_rerun_num: long(0),         $
  decom_prog_vers: strarr(1),       $
  decom_datbse_no: strarr(1),       $
  decom_ydhms_mil: strarr(1),       $
  inst_file_name : strarr(1),       $
  phys_rec_len:    long(0),         $
  merge_rerun_no:  long(0),         $
  merge_vers_no:   strarr(1),       $
  merge_ydhms_mil: strarr(1),       $ 
  n_edit_files:    long(0),         $
  edit_fil_name:   strarr(20),      $
  edit_fil_key:    strarr(20),      $
  edit_rerun_num:  lonarr(20),      $
  edit_prog_vers:  strarr(20),      $  
  edit_ydhms_run:  strarr(20),      $
  edit_data_type:  strarr(20),      $
  edit_msg_key:    strarr(20)       }
        
  RETURN
  END




pro Init_Subcom, error_return 

;	Initialize the Common block parameters which will contain values
;	extracted from S/C telemetry word 15 - CEPPAD Subcom from minor
;	frame 90 to 249.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revision
;	15 Sep 1995 M. T. Redding	Added 12 items to common block
;					'HW_tests'
;--------------------------------------------------------------------

   COMMON HIST_Singles,    HIST_S_spin, HIST_S_sector, HIST_S_data

   COMMON HIST_Events,     HIST_E_spin, HIST_E_sector, HIST_E_data

   COMMON Table_CRCs,      CRC_Flt_SW, CRC_MST, CRC_SAT, CRC_ICT, $
                           CRC_IESLUT, CRC_HISTLUT

   COMMON IPS_calib,       IPSCalMaskreq, IPSCalMaskrem, IPSCalState, IPS_cal

   COMMON Table_addresses, SpinBuf0_pag, SpinBuf0_off, SpinBuf0_len, $
                           MST0_pag, MST0_off, MST0_len, $
                           MSTGen_pag, MSTGen_off, $
                           SAT0_pag, SAT0_off, SAT0_len, $
                           ICT0_pag, ICT0_off, ICT0_len, $
                           IESLUT_pag, IESLUT_off, IESLUT_len, $
                           IESLUTGen_pag, IESLUTGen_off, $
                           HISTLUT0_pag, HISTLUT0_off, HISTLUT0_len, $
                           HISTLUTGen_pag, HISTLUTGen_off, $
                           Analogs_pag, Analogs_off, $
                           InputCmdStr_pag, InputCmdStr_off, $
                           CmdEchoStr_pag, CmdEchoStr_off

   COMMON DPU_Mem_Dump,    DPU_Dump_pag, DPU_Dump_off, DPU_Dump_data

   COMMON HW_tests,        Test_Board,               Test_RAM_24_31, $
                           Test_RAM_16_23,           Test_RAM_08_15, $
                           Test_RAM_00_07,           Test_CPU, $
                           Test_SC,                  Test_EPP_Direct, $
                           Test_EPP_Bank,            Test_HPP_Direct, $
                           Test_HPP_Bank,            Test_IPSKM5_00H_Add, $
                           Test_IPSKM5_00H_Dat,      Test_IPSKM5_02H_Add, $
                           Test_IPSKM5_02H_Dat,      Test_IPSScal_Add, $
                           Test_IPSScal_Dat,         Test_IPSScal_Msk, $
                           Test_Relay_Cnt,           Test_LV_Success_Cnt, $
                           Test_HV_Success_Cnt,      Test_IPS_7_4_Success_Cnt, $
                           Test_IPS_5_Success_Cnt,   Test_PROM_Success_Cnt, $
                           Test_Error_Cnt

;-----------------------------------------------------------------------

   error_return = 0

;	COMMON HIST_Singles

   HIST_S_spin      = BYTARR(8)
   HIST_S_sector    = BYTARR(8)
   HIST_S_data      = BYTARR(16,8)


;	COMMON HIST_Events

   HIST_E_spin      = BYTARR(8)
   HIST_E_sector    = BYTARR(8)
   HIST_E_data      = BYTARR(16,8)


;	COMMON Table_CRCs

   CRC_Flt_SW       = FIX(0)
   CRC_MST          = INTARR(16)
   CRC_SAT          = INTARR(16)
   CRC_ICT          = INTARR(16)
   CRC_IESLUT       = INTARR(12)
   CRC_HISTLUT      = INTARR(4)


;	COMMON IPS_calib

;		Structure to be defined
   IPS_cal = $
      { IPSCalMaskreq : FIX(0) , $
        IPSCalMaskrem : FIX(0) , $
        IPSCalState   : BYTE(0) , $
        C1 : BYTARR(9) , $
				; C1(0) : Ch. C1-low D/A for cal D/A=low res
				; C1(1) : Ch. C1-high D/A for cal D/A=low res
				; C1(2) : Ch. C1-cal D/A for cal D/A=low res
				; C1(3) : Ch. C1-low D/A for cal D/A=high res
				; C1(4) : Ch. C1-high D/A for cal D/A=high res
				; C1(5) : Ch. C1-cal D/A for cal D/A=high res
				; C1(6) : Ch. C1-high D/A for cal=bin 14 top
				; C1(7) : Ch. C1-cal D/A for cal=bin 14 top
				; C1(8) : Ch. C1-cal D/A for cal=bin 14 top
        B1 : BYTARR(9) , $	; same format as Chan C1
        T1 : BYTARR(9) , $	; same format as Chan C1
        C2 : BYTARR(9) , $	; same format as Chan C1
        C5 : BYTARR(9) , $	; same format as Chan C1
        C3 : BYTARR(9) , $	; same format as Chan C1
        C0 : BYTARR(9) , $	; same format as Chan C1
        B5 : BYTARR(9) , $	; same format as Chan C1
        B3 : BYTARR(9) , $	; same format as Chan C1
        B2 : BYTARR(9) , $	; same format as Chan C1
        B0 : BYTARR(9) , $	; same format as Chan C1
        T5 : BYTARR(9) , $	; same format as Chan C1
        T3 : BYTARR(9) , $	; same format as Chan C1
        T2 : BYTARR(9) , $	; same format as Chan C1
        T0 : BYTARR(9) }	; same format as Chan C1


;	COMMON Table_addresses

   SpinBuf0_pag    = FIX(0)
   SpinBuf0_off    = FIX(0)
   SpinBuf0_len    = FIX(0)
   MST0_pag        = FIX(0)
   MST0_off        = FIX(0)
   MST0_len        = FIX(0)
   MSTGen_pag      = FIX(0)
   MSTGen_off      = FIX(0)
   SAT0_pag        = FIX(0)
   SAT0_off        = FIX(0)
   SAT0_len        = FIX(0)
   ICT0_pag        = FIX(0)
   ICT0_off        = FIX(0)
   ICT0_len        = FIX(0)
   IESLUT_pag      = FIX(0)
   IESLUT_off      = FIX(0)
   IESLUT_len      = FIX(0)
   IESLUTGen_pag   = FIX(0)
   IESLUTGen_off   = FIX(0)
   HISTLUT0_pag    = FIX(0)
   HISTLUT0_off    = FIX(0)
   HISTLUT0_len    = FIX(0)
   HISTLUTGen_pag  = FIX(0)
   HISTLUTGen_off  = FIX(0)
   Analogs_pag     = FIX(0)
   Analogs_off     = FIX(0)
   InputCmdStr_pag = FIX(0)
   InputCmdStr_off = FIX(0)
   CmdEchoStr_pag  = FIX(0)
   CmdEchoStr_off  = FIX(0)


;	COMMON DPU_Mem_Dump

   DPU_Dump_pag    = FIX(0)
   DPU_Dump_off    = FIX(0)
   DPU_Dump_data   = BYTARR(128)


;	COMMON HW_tests

Test_Board               = BYTE(0)
Test_RAM_24_31           = BYTE(0)
Test_RAM_16_23           = BYTE(0)
Test_RAM_08_15           = BYTE(0)
Test_RAM_00_07           = BYTE(0)
Test_CPU                 = BYTE(0)
Test_SC                  = BYTE(0)
Test_EPP_Direct          = FIX(0)
Test_EPP_Bank            = BYTE(0)
Test_HPP_Direct          = FIX(0)
Test_HPP_Bank            = BYTE(0)
Test_IPSKM5_00H_Add      = FIX(0)
Test_IPSKM5_00H_Dat      = FIX(0)
Test_IPSKM5_02H_Add      = FIX(0)
Test_IPSKM5_02H_Dat      = FIX(0)
Test_IPSScal_Add         = FIX(0)
Test_IPSScal_Dat         = FIX(0)
Test_IPSScal_Msk         = FIX(0)
Test_Relay_Cnt           = FIX(0)
Test_LV_Success_Cnt      = FIX(0)
Test_HV_Success_Cnt      = FIX(0)
Test_IPS_7_4_Success_Cnt = FIX(0)
Test_IPS_5_Success_Cnt   = FIX(0)
Test_PROM_Success_Cnt    = FIX(0)
Test_Error_Cnt           = FIX(0)


RETURN
end



pro init_16_blk, error_return

;	initialize all 16-spin block arrays, counters and pointers

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block

;--------------------------------------------------------------------

COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                        sci_byt_per_spin, max_spins, $
                        nom_spin_dur, TRUE, FALSE, sync_byte, $
                        CEP_Sci_Man_byt_per_mnfrm, $
                        CEP_Conting_byt_per_mnfrm, $
                        CEP_Sci_Man_rec_len, $
                        CEP_Conting_rec_len

   COMMON CEP_lst_spn,     lst_spn_seq_no, lst_spn_min_frm, $
                           lst_spn_format, lst_spn_IPS_tab, $
                           lst_spn_IES_tab, lst_spn_IES_com, $
                           lst_spn_HIST_tab, lst_spn_HIST_com, $
                           lst_spn_SCOM, lst_spn_IPS_Mux, $
                           lst_spn_time, lst_spn_CRC, lst_spn_quality

   COMMON CEP_16_spin,     Spin_blk, Spin_blk_times, Spin_blk_SCOMs,	$
                           spin_blk_tot, spin_blk_seq_wrap,		$
                           est_16th_time, est_32nd_time, 		$
                           Bad_SCOM_MODE_sync, spin_quality

;bugbug
   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality
;bugbug

;-----------------------------------------------------------------------

;bugbug
;IF ( N_ELEMENTS( ths_spn_time ) GT 0 ) THEN BEGIN
;   IF ( ( ths_spn_time GE 0 ) AND ( ths_spn_time LE -1 ) ) THEN BEGIN
;      print, ' INIT_16_BLK 0:'
;   ENDIF
;ENDIF
;bugbug


;	Initialize 16-spin block with -1B 'missing' data value.

   Spin_blk = bytarr(sci_byt_per_spin,max_spins)
   Spin_blk(*,*) = -1B

;	Reset the counter registering the number of spins currently
;	copied into the 16-spin block.

   spin_blk_tot = 0

;	initialize 16-spin times to the 'missing' value -1.

   spin_blk_times = lonarr(max_spins)
   spin_blk_times(*) = -1L

;	initialize 16-spin sequence numbers - use longwords (original
;	sequence numbers are byte) so that a 'missing' value of -1
;	may be specified

   Spin_blk_SCOMs    = lonarr(max_spins)
   Spin_blk_SCOMs(*) = -1

;	Set sequence number wrap flag to indicate that the current set
;	of sequence numbers does not wrap from just less than 255 to 
;	just more than 0

   spin_blk_seq_wrap = FALSE

;	Un-set the last processed spin SCOM indicator

   lst_spn_SCOM = -1L

;	Estimated  16th spin time = -1 implies no estimate has been made yet

   est_16th_time = -1L

;	Estimated 16th SCOM value = -1 implies no estimate has been made yet

   est_32nd_time = -1L

;	Set the flag indicating the most recent acuisition mode change occurred
;	in sync with the spin counter (SCOM)

;   Bad_SCOM_MODE_sync = FALSE	; commented out - state will be changed only whe
				; valid change has occurred - otherwise it is
				; carried through for all succeeding spins

;	Carry over bad-sync state from the preceeding spin block - bits 2-4
;	will be set as IPS, IES or HIST LUTs commands or Mux values are changed
;	out of sync with SCOM

   IF ( spin_quality NE 0 ) THEN spin_quality = 1B

   error_return = 0
 return
end



pro Level_1_setup, error_return

;	initialize Level-1 file parameters

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block

;--------------------------------------------------------------------
 
   COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                           sci_byt_per_spin, max_spins, $
                           nom_spin_dur, TRUE, FALSE, sync_byte, $
                           CEP_Sci_Man_byt_per_mnfrm, $
                           CEP_Conting_byt_per_mnfrm, $
                           CEP_Sci_Man_rec_len, $
                           CEP_Conting_rec_len

   COMMON Dat_struct,      CEP_level_0_unit, data, CEP_Level_0_FLR

   COMMON CEP_Level_1, $
		L1_number, L1_year, L1_day, L1_begt, L1_endt, $
		L1_DPU_TM_table, L1_TM_CRC, $
	Use_IPS_L1, IPS_L1_unit, IPS_L1_record, IPS_L1_assoc, $
		IPS_L1_rec_len, IPS_L1_phyrcl, $
	Use_IES_L1, IES_L1_unit, IES_L1_record, IES_L1_assoc, $
		IES_L1_rec_len, IES_L1_phyrcl, $
	Use_HISTe_L1, HISTe_L1_unit, HISTe_L1_record, HISTe_L1_assoc, $
		HISTe_L1_rec_len, HISTe_L1_phyrcl, $
	Use_HISTp_L1, HISTp_L1_unit, HISTp_L1_record, HISTp_L1_assoc, $
		HISTp_L1_rec_len, HISTp_L1_phyrcl

;--------------------------------------------------------------------

;		COMMON CEP_Level_1


;	Initialize the Level-1 version/volume number - a new version will be
;	created for each change in CEPPAD TM format table

   L1_number = 0

;	Define the Level-1 data file year and day based on the Level-0 File 
;	Label Record values for year and day

   L1_year   = LONG(CEP_Level_0_FLR.year_beg)
   L1_day    = LONG(CEP_Level_0_FLR.day_beg)

;	Define the Level-1 file paramters to show no file has been defined yet

   Use_IPS_L1       = FALSE	; Flag indicating the Level-1 file is open
   IPS_L1_unit      = -4	; Out of range unit number
   IPS_L1_record    = -1L	; File pointer / record number

   Use_IES_L1       = FALSE
   IES_L1_unit      = -4
   IES_L1_record    = -1L

   Use_HISTe_L1     = FALSE
   HISTe_L1_unit    = -4
   HISTe_L1_record  = -1L

   Use_HISTp_L1     = FALSE
   HISTp_L1_unit    = -4
   HISTp_L1_record  = -1L

   RETURN
end



pro Load_format, error_return

;	Use current year, day and time to extract form the DPU T/M table
;	history file the appropriate T/M format for this spin.
;	All tables in the CEPPAD TM table file are looked at, in particular,
;	the DPU table number and the table start time (table superceded
;	time and table revision number are not used) to determine the
;	table defined for the CEPPAD DPU TM table closest in time to the 
;	spin time in question.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed TM table file open
;					 2	Failed TM table file read

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block
;	25 Sep 95	M. T. Redding	Corrected the interpretation of 
;					ths_spn_format to use the 4 LSBs rather
;					than the 4 MSBs as Normal-Science mode 
;					TM acquisition table number. The 4 MSB
;					contain the 'mode' as used in this
;					procedure.
;	11 Nov 96	M.T.Redding	Added COMMON block File_Control and
;					lines to initialize and revise
;					file_info.max_DPU_table_no each time
;					the telemetry table is read

;--------------------------------------------------------------------

COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                        sci_byt_per_spin, max_spins, $
                        nom_spin_dur, TRUE, FALSE, sync_byte, $
                        CEP_Sci_Man_byt_per_mnfrm, $
                        CEP_Conting_byt_per_mnfrm, $
                        CEP_Sci_Man_rec_len, $
                        CEP_Conting_rec_len

COMMON Dat_struct,      CEP_level_0_unit, data, CEP_Level_0_FLR

COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                        Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON CEP_lst_spn,     lst_spn_seq_no, lst_spn_min_frm, $
                           lst_spn_format, lst_spn_IPS_tab, $
                           lst_spn_IES_tab, lst_spn_IES_com, $
                           lst_spn_HIST_tab, lst_spn_HIST_com, $
                           lst_spn_SCOM, lst_spn_IPS_Mux, $
                           lst_spn_time, lst_spn_CRC, lst_spn_quality

   COMMON CEP_16_spin,     Spin_blk, Spin_blk_times, Spin_blk_SCOMs,	$
                           spin_blk_tot, spin_blk_seq_wrap,		$
                           est_16th_time, est_32nd_time, 		$
                           Bad_SCOM_MODE_sync, spin_quality

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

   COMMON CEP_lvl0_1_map, $
        Trgt_times, $
	num_source_spins,$
	num_IPS_energies, num_IPS_channels, IPS_spins_per_sample,$
		IPS_sectors_per_spin, IPS_Source_loc, IPS_Source_len,$
		IPS_Target_loc, IPS_trgt_tim_Loc, IPS_trgt_blk_len,$
                IPS_trgt_blk, IPS_source_byte_loc, IPS_trgt_byte_loc, $
	num_IES_energies, num_IES_channels, IES_spins_per_sample,$
		IES_sectors_per_spin, IES_Source_loc, IES_Source_len,$
		IES_Target_loc, IES_trgt_tim_loc, IES_trgt_blk_len,$
                IES_trgt_blk, IES_source_byte_loc, IES_trgt_byte_loc, $
	num_HISTe_energies, num_HISTe_channels, HISTe_spins_per_sample,$
		HISTe_sectors_per_spin, HISTe_Source_loc, HISTe_Source_len,$
		HISTe_Target_loc, HISTe_trgt_tim_loc, HISTe_trgt_blk_len,$
                HISTe_trgt_blk, HISTe_source_byte_loc, HISTe_trgt_byte_loc, $
	num_HISTp_energies, num_HISTp_channels, HISTp_spins_per_sample,$
		HISTp_sectors_per_spin, HISTp_Source_loc, HISTp_Source_len,$
		HISTp_Target_loc, HISTp_trgt_tim_loc, HISTp_trgt_blk_len,$
                HISTp_trgt_blk, HISTp_source_byte_loc, HISTp_trgt_byte_loc

   COMMON File_Control,    File_info



;-----------------------------------------------------------------------


   found_one = FALSE
   found_jtime = 0.0	; Julian time for non-existant TM table

;	Break down the format number

   mode          = BYTE(ISHFT(ths_spn_format,-4))		; 4 MSB

   IF mode eq 0 THEN BEGIN		; CEPPAD Normal TM mode

      norm_table_no = BYTE(ISHFT(ISHFT(ths_spn_format,4),-4))	; 4 LSB
      print, ' '
      print, FORMAT='( "  Normal mode T/M table = ",z1)', norm_table_no

;	Get just the 'seconds' portion of the spins time from 'milliseconds'
;	Combine the date/time values into a single value for comparing
;	with TM table entry date/times

      iseconds = LONG(DRH.ATC_mils)/1000
      spin_jtime = Julian_time( DRH.ATC_yr, DRH.ATC_day, iseconds )

;	Open the CEPPAD Normal telemetry tables file and find the table
;	matching the table number and date and time

      ON_IOERROR, Failed_open
      GET_Lun, TM_table_unit
      OPENR, TM_table_unit, concat_dir ( GETENV('PO_CEP_TM_TABLE_DIR'), 'ceppad_tm_tables.txt')

;	Read the file and extract the matching table values - table loaded
;	in the matching CEPPAD DPU table, closest in time and before the time
;	of the spin data.

      ON_IOERROR, Failed_read

      max_DPU_table_no = -1
      WHILE NOT EOF(TM_table_unit) DO BEGIN

         head_string = ''
         READF, TM_table_unit, head_string

;		Decode the table heading string

         Decode_tm_head, head_string, $
                         start_year, start_day, start_second, $
                         end_year, end_day, end_second, $
                         DPU_table_no, DPU_revision_no, Table_CRC_value


;		Revise maximum DPU table number

         File_info.max_DPU_table_no = BYTE( MAX( [ max_DPU_table_no, $
                                                   FIX( DPU_table_no ) ] ) )


;		Check for matching DPU TM table number

         IF  ( DPU_table_no  EQ  norm_table_no ) THEN BEGIN
         
;		See if this spin date/time is within the table's time range
;		(if the table has a supercede date/time) and see that this 
;		table is the closest one available to the spin's date/time


;		Combine the date/time values into a single value

            start_jtime = Julian_time( start_year, start_day, start_second )


;		See if spin time corresponds to table time range and table 
;		start time is newest one read so far to the spin time

;			Spin time is after beginning of table range
            IF (( start_jtime LE  spin_jtime )            AND $

;			Spin time is after previously found table
                (( NOT found_one ) OR $
                     ( spin_jtime - start_jtime  LT $
                              spin_jtime - found_jtime ))) THEN BEGIN

;	see if this table is more recent than any already found to match the 
;	spin date/time

               Read_CEP_TM_tables, error_return, TM_table_unit
               IF ( error_return EQ 0 ) THEN BEGIN
                  found_one = TRUE
                  found_jtime = start_jtime
               ENDIF ELSE BEGIN
                  printf, ' ** Load_format ** '
                  printf, '      Aborting run '
                  stop
               ENDELSE
            ENDIF ELSE BEGIN	; Spin time not in table's range
               Skip_CEP_TM_tables, error_return, TM_table_unit               
            ENDELSE
         ENDIF ELSE BEGIN	; Not a matching DPU table number
            Skip_CEP_TM_tables, error_return, TM_table_unit               
         ENDELSE

      ENDWHILE	; ( NOT EOF(TM_table_unit) )

      FREE_Lun, TM_table_unit
   ENDIF	; mode EQ 0

   IF found_one THEN BEGIN
      error_return = 0
      RETURN
   ENDIF ELSE BEGIN
      print, '  -LOAD_FORMAT-  Unable to find a table defined for the'
      print, '                 spin-time/DPU-table specified'
      print, '                 ths_spn_format   = ', ths_spn_format
      print, '                 DPU table number = ', norm_table_no
      print, '                 DRH.ATC_yr       = ',DRH.ATC_yr
      print, '                 DRH.ATC_day      = ',DRH.ATC_day
      print, '                 DRH.ATC_mils     = ',DRH.ATC_mils
      print, '   Terminating process'
      stop
   ENDELSE

;	Not CEPPAD normal mode (=0)

   error_return = 0
   RETURN

Failed_open:
   ON_IOERROR, NULL
   print, 'Error ',!err,' encountered during CEPPAD TM Table file open'
   error_return = 1
   FREE_Lun, TM_table_unit
   RETURN

Failed_read:
   ON_IOERROR, NULL
   print, 'Error ',!err,' encountered during CEPPAD TM Table file read'
   error_return = 2
   FREE_Lun, TM_table_unit
   RETURN

end







PRO Make_level_1, filename_level0

;	Procedure to create Level-1 files from Level-0 file
;	Derived from testina.pro

;		Passed Parameters

;	filename_level0

;________________________________________________________________________
       
;       PROCEDURES CALLED:                  PURPOSE

;       CEP_REC_PARAMS               Set CEPPAD  data record and file 
;                                    description parameters
;                                    ( record size, number of records, etc. )

;       CEP_SETUP                    Setup program constants

;       CHAR_WORD                    Convert bytes to string (a function)

;       EXTRACT_CEPPAD_SPINS         Control reading of CEPPAD Level-0 data 
;				     records, processing of Housekeeping
;				     data stored in each major frame, and the
;				     extraction from the CEPPAD science data
;				     of individual spin data


;       FILL_FLR                     Fill File Label Record ( FLR )
;                                    structure

;       GET_MAJFRM_DATA              Control the filling of the arrays and/or
;                                    structures with a masterframe of CEPPAD
;                                    data ( Calls procedure 'UNPK_CEP_TM')

;       INIT_CEPDRH                  Initialize Data Record Header ( DRH )
;                                    structure

;       INIT_FLR                     Initialize FLR structure

;       LONG_WORD                    Convert 4 bytes to a longword (a function)

;       OPEN_PL_LEVEL0               Open POLAR CEPPAD LEVEL-0 file

;       RD_PL_LEV0_FLR               Read FLR (first record in LEVEL-0 file)
 
;       UNPK_CEP_TM                  Unpack CEPPAD telemetry and fills 
;                                    arrays and/or structures with a 
;                                    masterframe of data ( 250 minor frames )

;	revision:
;	02 Aug 95	M.T.Redding	Changed minimum level-0 record length
;					from 1352 to 2792
;					CEP_Setup2 is performed here after
;					CEP_Setup instead of from
;					Extract_CEPPAD_spins to allow 
;					initialization of common block
;					parameters before they are needed
;	06 Mar 95	M.T.Redding	Eliminated use of 'FORMAT' in error
;					print statement following unsuccessful
;					file open
;	02 Jun 1996	M.T.Redding	Removed dbug_lev as positional argument
;					and all other references to 'severity',
;					'diag_lev' and 'local_diag' (COMMON
;					block DIAGNOS variables) from all
;					procedures
;	07 Jun 1996	M.T.Redding	Removed 'rec_beg' and 'rec_end' from
;					
; ----------------------------------------------------------------

 common cep_phys_rec, cep_byts_rec, cep_byts_mnfr, max_phys_rec
 common dat_struct, iunit, data, FLR
 COMMON CEP_major_frame, DRH, HK, Subcom, Sci_data, $
                        Min_frm_quality, MF_ATC_time, MF_CEP_seq_no
 COMMON ML1_SW_Constants, make_level_1_version, make_level_1_date
; ----------------------------------------------------------------

;	Initialize COMMON block constants and variables

CEP_SETUP
CEP_setup2, error_return


;	Display startup message

PRINT, ' MAKE_LEVEL_1  version ', make_level_1_version, ' ', make_level_1_date
PRINT, ' The Aerospace Corporation'
PRINT, ' Processing level-0 file:  ', filename_level0


; Each instrument level zero file has at the beginning of each file a record 
; called FILE LABEL RECORD (FLR) followed by a variable number of data records.
; The level zero file's physical record length is fixed but varies according
; to the instrument. The minimum physical record size is 2792 bytes which is
; the minimum size of a FLR. If the data record size is greater than 2792 bytes,
; the FLR record is made to be the size of the instrument data record with 
; filled bytes.

; Initialize  data FILE LABEL RECORD (FLR) structure

  INIT_FLR, flr

; Each level zero file has at the beginning of each data record ( CEP_TELEMETRY)
; three-hundred DATA RECORD HEADER (DRH) bytes described by the CEPDRH data 
; structure 

; Initialize  DATA RECORD HEADER (DRH) structure, CEPDRH

; INIT_CEPDRH, cepdrh
 INIT_CEPDRH, DRH


;        OPEN  LEVEL0 FILE TO READ THE FILE LABEL RECORD (1st physical record)

GET_LUN, iunit                 ; Get available unit number (iunit) which
                               ;    specifies the file unit from which the 
                               ;    input is taken

OPEN_PL_LEVEL0, filename_level0, iunit, ierror_0

if( ierror_0 ne 0) then begin
     print, ' '
     print, ' ERROR OPENING FILE ', filename_level0
     print, ' STOPPED IN PROCEDURE MAKE_LEVEL_1'
     FREE_LUN, iunit
     stop
endif

;               READ LEVEL0 FILE LABEL RECORD 

RD_PL_LEV0_FLR, iunit, filename_level0, file_label_rec, ierror_read

;                  FILL FILE LABEL RECORD STRUCTURE

if( ierror_read eq 0) then begin 
       FILL_FLR, file_label_rec, flr
endif else begin
           print,' '
           print,' ERROR READING FILE_LABEL RECORD'
           print, ' STOPPED IN PROCEDURE MAKE_LEVEL_1'
           FREE_LUN, iunit
           stop          
endelse

;       RETRIEVE CEPPAD FLR PARAMETERS THAT DESCRIBE DATA RECORD 

CEP_REC_PARAMS, flr, ierror_p
 if( ierror_p ne 0) then begin
      print, ' '
      print, ' ERRORS WITH FLR STRUCTURE PARAMETERS'
      print, ' STOPPED IN PROCEDURE MAKE_LEVEL_1'
      FREE_LUN, iunit
      stop
endif

;          Data is an array structure with level zero file 
data = assoc(iunit, bytarr(cep_byts_rec) )

Extract_CEPPAD_spins, error_return

;	Close the Level-0 file

FREE_LUN, iunit

end





pro Map_to_time_blk, error_return

;	Transfer science data from 16-spin block into 32-time block

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revision
;	01 Jul 1996	M.T.Redding	Transfer of IPS, IES HISTe or HISTp
;					data from spin block to target block is
;					now done only if the trgt_blk_len value
;					( e.g. IPS_trgt_blk_len, etc.)
;					is greater the 128 bytes (128 bytes is
;					the default size required for just the
;					32 LONGWORD time values) - For data
;					accumulated using SAT3 (IPS only) or 
;					SAT4 (IES only) no mapping of the other
;					instruments' data is performed 

;--------------------------------------------------------------------

   COMMON CEP_lst_spn,     lst_spn_seq_no, lst_spn_min_frm, $
                           lst_spn_format, lst_spn_IPS_tab, $
                           lst_spn_IES_tab, lst_spn_IES_com, $
                           lst_spn_HIST_tab, lst_spn_HIST_com, $
                           lst_spn_SCOM, lst_spn_IPS_Mux, $
                           lst_spn_time, lst_spn_CRC, lst_spn_quality

   COMMON CEP_16_spin,     Spin_blk, Spin_blk_times, Spin_blk_SCOMs,	$
                           spin_blk_tot, spin_blk_seq_wrap,		$
                           est_16th_time, est_32nd_time, 		$
                           Bad_SCOM_MODE_sync, spin_quality

   COMMON CEP_lvl0_1_map, $
        Trgt_times, $
	num_source_spins,$
	num_IPS_energies, num_IPS_channels, IPS_spins_per_sample,$
		IPS_sectors_per_spin, IPS_Source_loc, IPS_Source_len,$
		IPS_Target_loc, IPS_trgt_tim_Loc, IPS_trgt_blk_len,$
                IPS_trgt_blk, IPS_source_byte_loc, IPS_trgt_byte_loc, $
	num_IES_energies, num_IES_channels, IES_spins_per_sample,$
		IES_sectors_per_spin, IES_Source_loc, IES_Source_len,$
		IES_Target_loc, IES_trgt_tim_loc, IES_trgt_blk_len,$
                IES_trgt_blk, IES_source_byte_loc, IES_trgt_byte_loc, $
	num_HISTe_energies, num_HISTe_channels, HISTe_spins_per_sample,$
		HISTe_sectors_per_spin, HISTe_Source_loc, HISTe_Source_len,$
		HISTe_Target_loc, HISTe_trgt_tim_loc, HISTe_trgt_blk_len,$
                HISTe_trgt_blk, HISTe_source_byte_loc, HISTe_trgt_byte_loc, $
	num_HISTp_energies, num_HISTp_channels, HISTp_spins_per_sample,$
		HISTp_sectors_per_spin, HISTp_Source_loc, HISTp_Source_len,$
		HISTp_Target_loc, HISTp_trgt_tim_loc, HISTp_trgt_blk_len,$
                HISTp_trgt_blk, HISTp_source_byte_loc, HISTp_trgt_byte_loc



;-----------------------------------------------------------------------

;	Redimension the Spin_blk array from 2 to 1 dimension for access by
;	the mapping pointers

   Spin_blk_size = SIZE( Spin_blk)
   Spin_blk_dims = Spin_blk_size(0)
   IF ( Spin_blk_dims NE 2 ) THEN BEGIN
      print, ' Spin_blk array not properly dimensioned'
      print, ' Number of dimensions = ', Spin_blk_size(0)
      print, '     - STOPPING - '
      STOP
   ENDIF

;	Save the dimensions so that the original dimensions may be restored
;	later

   Spin_blk_x_dim = Spin_blk_size(1)
   Spin_blk_y_dim = Spin_blk_size(2)
   Spin_blk = REFORM( Spin_blk, (Spin_blk_x_dim * Spin_blk_y_dim) )


;	Transfer all IPS data

   FOR i = 0, num_source_spins -1 DO BEGIN
      IPS_trgt_blk(IPS_trgt_tim_loc(i)) = BYTE(Spin_blk_times,i*4,4)
   ENDFOR

   IF ( IPS_trgt_blk_len GT 128 ) THEN $
      IPS_trgt_blk( IPS_trgt_byte_loc ) = Spin_blk( IPS_source_byte_loc )

;	Transfer all IES data

   FOR i = 0, num_source_spins -1 DO BEGIN
      IES_trgt_blk(IES_trgt_tim_loc(i)) = BYTE(Spin_blk_times,i*4,4)

   ENDFOR

   IF ( IES_trgt_blk_len GT 128 ) THEN $
      IES_trgt_blk( IES_trgt_byte_loc ) = Spin_blk( IES_source_byte_loc )

;	Transfer all HISTe data

   FOR i = 0, num_source_spins -1 DO BEGIN
      HISTe_trgt_blk(HISTe_trgt_tim_loc(i)) = BYTE(Spin_blk_times,i*4,4)
   ENDFOR

   IF ( HISTe_trgt_blk_len GT 128 ) THEN $
      HISTe_trgt_blk( HISTe_trgt_byte_loc ) = Spin_blk( HISTe_source_byte_loc )

;	Transfer all HISTp data

   FOR i = 0, num_source_spins -1 DO BEGIN
      HISTp_trgt_blk(HISTp_trgt_tim_loc(i)) = BYTE(Spin_blk_times,i*4,4)
   ENDFOR

   IF ( HISTp_trgt_blk_len GT 128 ) THEN $
      HISTp_trgt_blk( HISTp_trgt_byte_loc ) = Spin_blk( HISTp_source_byte_loc )


;	Also map the 16-spin times into the last 16 positions of the Trgt_times
;	array

   Trgt_times(16:31) = Spin_blk_times(0:15)


;	Restore the original dimensions to the Spin_blk array

   Spin_blk = REFORM( Spin_blk, Spin_blk_x_dim, Spin_blk_y_dim )

   error_return = 0
   return
end




pro Open_HK, error_return

;	Obtain unit number, construct filename, define associated array, 
;	initialize the record pointer and open the Level-1 Housekeeping
;	file with fixed record size  for write access.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed HK file open

;	revision
;	Mar 06 1996	M.T.Redding	Corrected printed variable 'err' to be
;					system variable '!ERR

;--------------------------------------------------------------------

   COMMON CEP_HK,          CEP_HK_unit, CEP_HK_rec, CEP_HK_rec_len, CEP_HK_data

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no


;-----------------------------------------------------------------------

   error_return = 0

   IF ( CEP_HK_unit LE -2 ) THEN BEGIN	; Only open if no already open

;	Make up string pieces of Level-1 file names using the data year, day and
;	the next available Level-1 volume number.
;	Example Level-1 file name: PO_L1_CEP_HK_1994123.DAT

;	Year and Day

      Year_string = STRING(format='(I4.4)',DRH.ATC_yr)
      Day_string  = STRING(format='(I3.3)',DRH.ATC_day)

      ON_IOERROR, Failed_Open

;	Obtain an LU for the CEPPAD Housekeeping Level-1 file
      CEP_HK_unit = -3	; Initialize to an out of IDL LUN range value 
			; to indicate file not open
      GET_LUN, L1_unit

;	Construct the Level-1 file name
      L1_file = 'PO_CEP_L1_HK_' + $
                    Year_string + Day_string + '.DAT'


;	Open the file with fixed record length for writing

      OPENW, L1_unit, L1_file, ERROR=err, CEP_HK_rec_len
      CEP_HK_unit = L1_unit
      CEP_HK_rec = 1L
      CEP_HK_data = ASSOC(CEP_HK_unit,BYTARR(CEP_HK_rec_len))
      GOTO, Exit_HK_Open

Failed_Open:
      print, '*** Open_HK ***'
      print, '  Error # ', !ERR,' encountered opening file ',L1_file
      print, !ERR_STRING
      error_return = 1

   ENDIF	; ( CEP_HK_unit LE -2 )


Exit_HK_Open:
   return
end



pro Open_L1, error_return

;	Obtain unit numbers for and then open individual Level-1 files
;	for IPS, IES HISTe and HISTp CEPPAD data.  After opening the files
;	define associated variable with each as well as the record length
;	and initialize the record number

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1-4	Number of failed Normal mode
;						file opens

;	revision
;	Mar 06 1996	M.T.Redding	Corrected printed variable 'err' to be
;					system variable '!ERR
;	May 01 1996	M.T.Redding	Added COMMON block File_Control
;					containing structure variable File_info.
;					Added code to use and update 
;					File_info.last_normal_TM_table so that
;					if the previously active normal-science
;					mode L1 files had been opened using the
;					same TM/Acquisition table then those
;					files are RE-OPENED for update rather
;					than new files created (as for a
;					different TM/Acq table)
;	May 23 1996	M.T.Redding	'Open new L1 file' message is now only
;					printed for new files - reopening
;					existing files is done 'silently'

;--------------------------------------------------------------------

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON CEP_Level_1, $
		L1_number, L1_year, L1_day, L1_begt, L1_endt, $
		L1_DPU_TM_table, L1_TM_CRC, $
	Use_IPS_L1, IPS_L1_unit, IPS_L1_record, IPS_L1_assoc, $
		IPS_L1_rec_len, IPS_L1_phyrcl, $
	Use_IES_L1, IES_L1_unit, IES_L1_record, IES_L1_assoc, $
		IES_L1_rec_len, IES_L1_phyrcl, $
	Use_HISTe_L1, HISTe_L1_unit, HISTe_L1_record, HISTe_L1_assoc, $
		HISTe_L1_rec_len, HISTe_L1_phyrcl, $
	Use_HISTp_L1, HISTp_L1_unit, HISTp_L1_record, HISTp_L1_assoc, $
		HISTp_L1_rec_len, HISTp_L1_phyrcl

   COMMON CEP_lvl0_1_map, $
        Trgt_times, $
	num_source_spins,$
	num_IPS_energies, num_IPS_channels, IPS_spins_per_sample,$
		IPS_sectors_per_spin, IPS_Source_loc, IPS_Source_len,$
		IPS_Target_loc, IPS_trgt_tim_Loc, IPS_trgt_blk_len,$
                IPS_trgt_blk, IPS_source_byte_loc, IPS_trgt_byte_loc, $
	num_IES_energies, num_IES_channels, IES_spins_per_sample,$
		IES_sectors_per_spin, IES_Source_loc, IES_Source_len,$
		IES_Target_loc, IES_trgt_tim_loc, IES_trgt_blk_len,$
                IES_trgt_blk, IES_source_byte_loc, IES_trgt_byte_loc, $
	num_HISTe_energies, num_HISTe_channels, HISTe_spins_per_sample,$
		HISTe_sectors_per_spin, HISTe_Source_loc, HISTe_Source_len,$
		HISTe_Target_loc, HISTe_trgt_tim_loc, HISTe_trgt_blk_len,$
                HISTe_trgt_blk, HISTe_source_byte_loc, HISTe_trgt_byte_loc, $
	num_HISTp_energies, num_HISTp_channels, HISTp_spins_per_sample,$
		HISTp_sectors_per_spin, HISTp_Source_loc, HISTp_Source_len,$
		HISTp_Target_loc, HISTp_trgt_tim_loc, HISTp_trgt_blk_len,$
                HISTp_trgt_blk, HISTp_source_byte_loc, HISTp_trgt_byte_loc


   COMMON File_Control, File_info

;-----------------------------------------------------------------------

;	Determine if new L1 files must be created or if previously used ones
;	should be re-opened for update

   ths_spn_TM_table = BYTE( ISHFT( ISHFT( ths_spn_format, 4 ), -4 ))	; 4 LSB
   new_normal_file = 0B
   IF ( ths_spn_TM_table NE file_info.last_normal_TM_table ) THEN BEGIN
      file_info.last_normal_TM_table = ths_spn_TM_table 
      new_normal_file = 1B
   ENDIF

;	Make up string pieces of Level-1 file names using the data year, day and
;	the next available Level-1 volume number.
;	Example Level-1 file name: PO_L1_CEP_IPS_1994123_01.DAT

   IF ( new_normal_file ) THEN $
      L1_number = L1_number + 1	;Increment the Level-1 number (volume/version)
   L1_string = STRING(format='(I2.2)',L1_number)

;	Year and Day

   Year_string = STRING(format='(I4.4)',L1_year)
   Day_string  = STRING(format='(I3.3)',L1_day)

;	Open each type of sensor Level-1 file required

;	IPS Level-1 file
   ON_IOERROR, Failed_IPS
   IPS_L1_unit = -3	; Out of IDL LUN range - implies file not currently open
   IF Use_IPS_L1 THEN BEGIN

;	Obtain an LU for the IPS Level-1 file

      GET_LUN, L1_unit

;	Construct the Level-1 file name

      L1_file = 'PO_CEP_L1_IPS_' + $
                    Year_string + Day_string + '_' + L1_string + '.DAT'

;	Determine the Level-1 record length based on the 32-time array size

      IPS_L1_rec_len = LONG( IPS_trgt_blk_len / 2 )	; Length of array 
							; containing data 
							; over 16 times
      IPS_L1_phyrcl = ( IPS_L1_rec_len + 1L ) / 2L	; Physical record will
							; contain just 1/2 od
							; 16-time data

;	Open the file with fixed record length for writing

      IF ( new_normal_file ) THEN BEGIN
         IPS_L1_record = 1L
         OPENW, L1_unit, L1_file, ERROR=err, IPS_L1_phyrcl
      ENDIF ELSE BEGIN
         OPENU, L1_unit, L1_file, ERROR=err, IPS_L1_phyrcl
      ENDELSE
      IPS_L1_unit = L1_unit
      IPS_L1_assoc = ASSOC(IPS_L1_unit,BYTARR(IPS_L1_phyrcl))

;	Write new file open message to log

      IF ( new_normal_file ) THEN BEGIN
         print, ' '
         print, 'Opened new CEPPAD Normal Mode IPS Level 1 file '
         print, '    file = ', L1_file, $
                '    starting at time  = ', ths_spn_time/1000.
      ENDIF ELSE BEGIN
         print, ' '
         print, 'Re-opened CEPPAD Normal Mode IPS Level 1 file '
         print, '    file = ', L1_file, $
                '    starting at time  = ', ths_spn_time/1000.
      ENDELSE

   ENDIF
   GOTO, IES_Open

Failed_IPS:
   print, '*** Open_L1 ***'
   print, '  Error # ', !ERR,' encountered opening file ',L1_file
   print, !ERR_STRING
   error_return = error_return + 1


;	IES Level-1 file
IES_Open:
   ON_IOERROR, Failed_IES
   IES_L1_unit = -3	; Out of IDL LUN range - implies file not currently open
   IF Use_IES_L1 THEN BEGIN

;	Obtain an LU for the IES Level-1 file

      GET_LUN, L1_unit

;	Construct the Level-1 file name

      L1_file = 'PO_CEP_L1_IES_' + $
                    Year_string + Day_string + '_' + L1_string + '.DAT'

;	Determine the Level-1 record length based on the 32-time array size

      IES_L1_rec_len = LONG( IES_trgt_blk_len / 2 )	; Length of array 
							; containing data 
							; over 16 times
      IES_L1_phyrcl = ( IES_L1_rec_len + 1L ) / 2L	; Physical record will
							; contain just 1/2 od
							; 16-time data

;	Open the file with fixed record length for writing

      IF ( new_normal_file ) THEN BEGIN
         IES_L1_record = 1L
         OPENW, L1_unit, L1_file, ERROR=err, IES_L1_phyrcl
      ENDIF ELSE BEGIN
         OPENU, L1_unit, L1_file, ERROR=err, IES_L1_phyrcl
      ENDELSE
      IES_L1_unit = L1_unit
      IES_L1_assoc = ASSOC(IES_L1_unit,BYTARR(IES_L1_phyrcl))

;	Write new file open message to log

      IF ( new_normal_file ) THEN BEGIN
         print, ' '
         print, 'Opened new CEPPAD Normal Mode IES Level 1 file '
         print, '    file = ', L1_file, $
                '    starting at time  = ', ths_spn_time/1000.
      ENDIF ELSE BEGIN
         print, ' '
         print, 'Re-opened CEPPAD Normal Mode IES Level 1 file '
         print, '    file = ', L1_file, $
                '    starting at time  = ', ths_spn_time/1000.
      ENDELSE


   ENDIF
   GOTO, HISTe_Open

Failed_IES:
   print, '*** Open_L1 ***'
   print, '  Error # ', !ERR,' encountered opening file ',L1_file
   print, !ERR_STRING
   error_return = error_return + 1


;	HISTe Level-1 file
HISTe_Open:
   ON_IOERROR, Failed_HISTe
   HISTe_L1_unit = -3	; Out of IDL LUN range - implies file not currently open
   IF Use_HISTe_L1 THEN BEGIN

;	Obtain an LU for the HISTe Level-1 file

      GET_LUN, L1_unit

;	Construct the Level-1 file name

      L1_file = 'PO_CEP_L1_HISTe_' + $
                    Year_string + Day_string + '_' + L1_string + '.DAT'

;	Determine the Level-1 record length based on the 32-time array size

      HISTe_L1_rec_len = LONG( HISTe_trgt_blk_len / 2 )	; Length of array 
							; containing data 
							; over 16 times
      HISTe_L1_phyrcl = ( HISTe_L1_rec_len + 1L ) / 2L	; Physical record will
							; contain just 1/2 od
							; 16-time data

;	Open the file with fixed record length for writing

      IF ( new_normal_file ) THEN BEGIN
         HISTe_L1_record = 1L
         OPENW, L1_unit, L1_file, ERROR=err, HISTe_L1_phyrcl
      ENDIF ELSE BEGIN
         OPENU, L1_unit, L1_file, ERROR=err, HISTe_L1_phyrcl
      ENDELSE
      HISTe_L1_unit = L1_unit
      HISTe_L1_assoc = ASSOC(HISTe_L1_unit,BYTARR(HISTe_L1_phyrcl))

;	Write new file open message to log

      IF ( new_normal_file ) THEN BEGIN
         print, ' '
         print, 'Opened new CEPPAD Normal Mode HISTe Level 1 file '
         print, '    file = ', L1_file, $
                '    starting at time  = ', ths_spn_time/1000.
      ENDIF ELSE BEGIN
         print, ' '
         print, 'Re-opened CEPPAD Normal Mode HISTe Level 1 file '
         print, '    file = ', L1_file, $
                '    starting at time  = ', ths_spn_time/1000.
      ENDELSE


   ENDIF
   GOTO, HISTp_Open

Failed_HISTe:
   print, '*** Open_L1 ***'
   print, '  Error # ', !ERR, ' encountered opening file ',L1_file
   print, !ERR_STRING
   error_return = error_return + 1


;	HISTp Level-1 file
HISTp_Open:
   ON_IOERROR, Failed_HISTp
   HISTp_L1_unit = -3	; Out of IDL LUN range - implies file not currently open
   IF Use_HISTp_L1 THEN BEGIN

;	Obtain an LU for the HISTp Level-1 file

      GET_LUN, L1_unit

;	Construct the Level-1 file name

      L1_file = 'PO_CEP_L1_HISTp_' + $
                    Year_string + Day_string + '_' + L1_string + '.DAT'

;	Determine the Level-1 record length based on the 32-time array size

      HISTp_L1_rec_len = LONG( HISTp_trgt_blk_len / 2 )	; Length of array 
							; containing data 
							; over 16 times
      HISTp_L1_phyrcl = ( HISTp_L1_rec_len + 1L ) / 2L	; Physical record will
							; contain just 1/2 od
							; 16-time data
;	Open the file with fixed record length for writing

      IF ( new_normal_file ) THEN BEGIN
         HISTp_L1_record = 1L
         OPENW, L1_unit, L1_file, ERROR=err, HISTp_L1_phyrcl
      ENDIF ELSE BEGIN
         OPENU, L1_unit, L1_file, ERROR=err, HISTp_L1_phyrcl
      ENDELSE
      HISTp_L1_unit = L1_unit
      HISTp_L1_assoc = ASSOC(HISTp_L1_unit,BYTARR(HISTp_L1_phyrcl))

;	Write new file open message to log

      IF ( new_normal_file ) THEN BEGIN
         print, ' '
         print, 'Opened new CEPPAD Normal Mode HISTp Level 1 file '
         print, '    file = ', L1_file, $
                '    starting at time  = ', ths_spn_time/1000.
      ENDIF ELSE BEGIN
         print, ' '
         print, 'Re-opened CEPPAD Normal Mode HISTp Level 1 file '
         print, '    file = ', L1_file, $
                '    starting at time  = ', ths_spn_time/1000.
      ENDELSE


   ENDIF
   GOTO, Exit_Open_L1

Failed_HISTp:
   print, '*** Open_L1 ***'
   print, '  Error # ', !ERR, ' encountered opening file ',L1_file
   print, !ERR_STRING
   error_return = error_return + 1


Exit_Open_L1:
   return
end



pro Open_new_normal, error_return

;	Set up the files and flags for a CEPPAD 'Normal'/'Science' mode 
;	telemetry - 'Normal'/'Science' <==> ths_spn_format / 16 = 0 (the 
;	4 most significant bits of the format byte are '0')

;	Controls the opening of new Level-1 data file and Level-1 HK
;	files necesitated by a change in the CEPPAD TM format

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block
;	25 Sep 95	M. T. Redding	Revised procedure description defining
;					'Normal'/'Science' mode
;       10 Apr 1996	M. T. Redding   Added 'ignore_32_times' to 'CEP_32_time'
;                                       common block

;--------------------------------------------------------------------

COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                        sci_byt_per_spin, max_spins, $
                        nom_spin_dur, TRUE, FALSE, sync_byte, $
                        CEP_Sci_Man_byt_per_mnfrm, $
                        CEP_Conting_byt_per_mnfrm, $
                        CEP_Sci_Man_rec_len, $
                        CEP_Conting_rec_len

COMMON CEP_32_time,     ignore_16_times, ignore_32_times

   COMMON CEP_lst_spn,     lst_spn_seq_no, lst_spn_min_frm, $
                           lst_spn_format, lst_spn_IPS_tab, $
                           lst_spn_IES_tab, lst_spn_IES_com, $
                           lst_spn_HIST_tab, lst_spn_HIST_com, $
                           lst_spn_SCOM, lst_spn_IPS_Mux, $
                           lst_spn_time, lst_spn_CRC, lst_spn_quality

   COMMON CEP_16_spin,     Spin_blk, Spin_blk_times, Spin_blk_SCOMs,	$
                           spin_blk_tot, spin_blk_seq_wrap,		$
                           est_16th_time, est_32nd_time, 		$
                           Bad_SCOM_MODE_sync, spin_quality

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

   COMMON CEP_lvl0_1_map, $
        Trgt_times, $
	num_source_spins,$
	num_IPS_energies, num_IPS_channels, IPS_spins_per_sample,$
		IPS_sectors_per_spin, IPS_Source_loc, IPS_Source_len,$
		IPS_Target_loc, IPS_trgt_tim_Loc, IPS_trgt_blk_len,$
                IPS_trgt_blk, IPS_source_byte_loc, IPS_trgt_byte_loc, $
	num_IES_energies, num_IES_channels, IES_spins_per_sample,$
		IES_sectors_per_spin, IES_Source_loc, IES_Source_len,$
		IES_Target_loc, IES_trgt_tim_loc, IES_trgt_blk_len,$
                IES_trgt_blk, IES_source_byte_loc, IES_trgt_byte_loc, $
	num_HISTe_energies, num_HISTe_channels, HISTe_spins_per_sample,$
		HISTe_sectors_per_spin, HISTe_Source_loc, HISTe_Source_len,$
		HISTe_Target_loc, HISTe_trgt_tim_loc, HISTe_trgt_blk_len,$
                HISTe_trgt_blk, HISTe_source_byte_loc, HISTe_trgt_byte_loc, $
	num_HISTp_energies, num_HISTp_channels, HISTp_spins_per_sample,$
		HISTp_sectors_per_spin, HISTp_Source_loc, HISTp_Source_len,$
		HISTp_Target_loc, HISTp_trgt_tim_loc, HISTp_trgt_blk_len,$
                HISTp_trgt_blk, HISTp_source_byte_loc, HISTp_trgt_byte_loc



   COMMON CEP_Level_1, $
		L1_number, L1_year, L1_day, L1_begt, L1_endt, $
		L1_DPU_TM_table, L1_TM_CRC, $
	Use_IPS_L1, IPS_L1_unit, IPS_L1_record, IPS_L1_assoc, $
		IPS_L1_rec_len, IPS_L1_phyrcl, $
	Use_IES_L1, IES_L1_unit, IES_L1_record, IES_L1_assoc, $
		IES_L1_rec_len, IES_L1_phyrcl, $
	Use_HISTe_L1, HISTe_L1_unit, HISTe_L1_record, HISTe_L1_assoc, $
		HISTe_L1_rec_len, HISTe_L1_phyrcl, $
	Use_HISTp_L1, HISTp_L1_unit, HISTp_L1_record, HISTp_L1_assoc, $
		HISTp_L1_rec_len, HISTp_L1_phyrcl

;-----------------------------------------------------------------------

;	Get the table number.
;	Get the corresponding TM format and generate the 
;	16-spin-to-32-time mapping parameters.
;	Open the new level-1 files.
;	Set the flag 'ignore_16_times' to TRUE so that the next 16
;	spins of data will be ignored when it comes time to append
;	them to the Level-1 file ... Presumably the data contained
;	in them will be of mixed format - some values acquired using
;	the previous TM table, some acquired using the current table.

   Load_format, error_return
   Construct_map, error_return

;	Redefine the number of science bytes per spin.  This number is also
;	the number of bytes between locations of start of science data of
;	consecutive spins.  This is (IPS_Source_loc(1,x) - IPS_Source_loc(0,x)),
;	(IES_Source_loc(1,x) - IES_Source_loc(0,x)), 
;	(HISTe_Source_loc(1,x) - HISTe_Source_loc(0,x)) or
;	(HISTp_Source_loc(1,x) - HISTp_Source_loc(0,x)), or more simply, since
;	IPS_Source_loc(0,0) = 0, just 'IPS_Source_loc(1,0)'.
;	CEPPAD outputs 2388 bytes of science data each spin but the mapping 
;	used here only accounts for the nominal total of bytes defined by the 
;	TM/Accumulation table.  Depending on the contents of such a table the 
;	nominal total number of bytes output on a spin could be less than 2388. 
;	For example, a particular combination of 'spins-per-sample' and 
;	'sectors-per-spin' used on a given table might require only 2376 bytes 
;	per spin.  In this case only 2376 bytes of science data would be 
;	transferred from the level-0 file to the 16-spin buffer 'Spin_blk' 
;	each spin.  The mapping used to transfer bytes from the 16-spin buffer 
;	to the 32-time buffer will then assume 2376 is the length of the 
;	individual spin's science data.

   sci_byt_per_spin = IPS_Source_loc(1,0)

;	Resize the 16-spin buffer to hold exactly the number of non-filled
;	science data bytes actually included in each spin's data and 
;	initialize 16-spin block with -1B 'missing' data value.

   Spin_blk = bytarr(sci_byt_per_spin,max_spins)
   Spin_blk(*,*) = -1B


;	Set the flags indicating that individual sensor's data is included
;	with the current telemetry and should be written to a Level-1 file.
;	If there is no data the block length will be '128'  - which accounts 
;	for 2 blocks, each containing just the 16 Longword time values. So if 
;	the block length is 128 or less don't bother trying to create a 
;	Level-1 file for this sensor - there's no data in the telemetry from it.

;		IPS
   IF IPS_trgt_blk_len GT 128 THEN BEGIN
      USE_IPS_L1 = TRUE
   ENDIF ELSE BEGIN
      USE_IPS_L1 = FALSE
   ENDELSE

;		IES
   IF IES_trgt_blk_len GT 128 THEN BEGIN
      USE_IES_L1 = TRUE
   ENDIF ELSE BEGIN
      USE_IES_L1 = FALSE
   ENDELSE

;		HISTe
   IF HISTe_trgt_blk_len GT 128 THEN BEGIN
      USE_HISTe_L1 = TRUE
   ENDIF ELSE BEGIN
      USE_HISTe_L1 = FALSE
   ENDELSE

;		HISTp
   IF HISTp_trgt_blk_len GT 128 THEN BEGIN
      USE_HISTp_L1 = TRUE
   ENDIF ELSE BEGIN
      USE_HISTp_L1 = FALSE
   ENDELSE

;	Open all Level-1 data files for the sensors for which data is stored 
;	in the Level-0 file using the current 'normal' CEPPAD TM table

   Open_L1, error_return

;	Set last process spin T/M format to an unrecognized value

   lst_spn_format = 'A5'XB	; A5H is not currently used as a format value
				; It is used as the sync value so if new format
				; values are defined it is expected this value
				; would not be selected

;	Open Level-1 Housekeeping file if necessary

   Open_HK, error_return



   RETURN
end





pro Open_new_Rates, error_return

;	Set up the files and flags for a CEPPAD 'Rates-histogram' mode 
;	Level-1 data file.
;	- 'Rates-mode' <==> ths_spn_format / 16 NE 0 (at least one of the 
;	4 Most significant bits of the format byte is not 0). Possible
;	Rates modes and their bit values are the following:
;	001sssss		IPS rates histogram, sector ssss only
;	010sssss		IES rates histogram, sector ssss only
;	011sssss		HIST rates histogram, sector ssss only
;	10111111	"191"	IPS histogram, full spin accumulation
;	11011111	"223"	IES histogram, full spin accumulation
;	11111111	"225"	HIST histogram, full spin accumulation

;	Controls the opening of new Level-1 data file and Level-1 HK
;	files necesitated by a change in the CEPPAD TM format

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed open of IPS, IES or HIST
;						Rates mode Level-1 file

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block

;	Mar 06 1996	M.T.Redding	Corrected printed variable 'err' to be
;					system variable '!ERR

;--------------------------------------------------------------------

COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                        sci_byt_per_spin, max_spins, $
                        nom_spin_dur, TRUE, FALSE, sync_byte, $
                        CEP_Sci_Man_byt_per_mnfrm, $
                        CEP_Conting_byt_per_mnfrm, $
                        CEP_Sci_Man_rec_len, $
                        CEP_Conting_rec_len

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

   COMMON CEP_Level_1, $
		L1_number, L1_year, L1_day, L1_begt, L1_endt, $
		L1_DPU_TM_table, L1_TM_CRC, $
	Use_IPS_L1, IPS_L1_unit, IPS_L1_record, IPS_L1_assoc, $
		IPS_L1_rec_len, IPS_L1_phyrcl, $
	Use_IES_L1, IES_L1_unit, IES_L1_record, IES_L1_assoc, $
		IES_L1_rec_len, IES_L1_phyrcl, $
	Use_HISTe_L1, HISTe_L1_unit, HISTe_L1_record, HISTe_L1_assoc, $
		HISTe_L1_rec_len, HISTe_L1_phyrcl, $
	Use_HISTp_L1, HISTp_L1_unit, HISTp_L1_record, HISTp_L1_assoc, $
		HISTp_L1_rec_len, HISTp_L1_phyrcl

   COMMON CEP_Rates_Lvl_1, IPS_Rates_unit, IPS_Rates_rec, $
                           IPS_Rates_assoc, IPS_Rates_recl, $
                           IPS_Rates_format, $
                           IES_Rates_unit, IES_Rates_rec, $
                           IES_Rates_assoc, IES_Rates_recl, $
                           IES_Rates_format, $
                           HIST_Rates_unit, HIST_Rates_rec, $
                           HIST_Rates_assoc, HIST_Rates_recl, $
                           HIST_Rates_format

;-----------------------------------------------------------------------

;	Open the indicated Rates-Level-1 file if not already open.

;		IPS Rates-mode Level-1 file
   IF ( (ths_spn_format EQ IPS_Rates_format ) AND ( IPS_Rates_unit LE -3 ) ) $
                                                                      THEN BEGIN

;	Construct the Level-1 file name

      Year_string = STRING(format='(I4.4)',L1_year)
      Day_string  = STRING(format='(I3.3)',L1_day)
      L1_file = 'PO_CEP_L1_IPS_RATES_' + Year_string + Day_string + '.DAT'

;	Obtain an LU for the IPS Rates-mode Level-1 file

      ON_IOERROR, Failed_IPS_Rates
      GET_LUN, L1_unit

;	Open the file with fixed record length for writing
;	Write new file open message to log

      print, 'Opening new CEPPAD Rates Mode IPS Level 1 file '
      print, '    file = ', L1_file, $
             '    starting at time  = ', ths_spn_time/1000.

      OPENW, L1_unit, L1_file, ERROR=err, IPS_Rates_recl 
      IPS_Rates_unit = L1_unit
      IPS_Rates_assoc = ASSOC(IPS_Rates_unit,BYTARR(IPS_Rates_recl))
      IPS_Rates_rec   = 1L	; Initialize the total number of records 
				; stored in the file
      RETURN			; Return after successful IPS Rates file open

Failed_IPS_Rates:
      print, '*** Open_new_Rates ***'
      print, '  Error # ', !ERR, ' encountered opening file ',L1_file
      print, !ERR_STRING
      error_return = 1
      RETURN

   ENDIF	;   ( (ths_spn_format EQ IPS_Rates_format ) AND 
		;      ( IPS_Rates_unit LE -3 ) ) 



;		IES Rates-mode Level-1 file
   IF ( (ths_spn_format EQ IES_Rates_format ) AND ( IES_Rates_unit LE -3 ) ) $
                                                                     THEN BEGIN

;	Construct the Level-1 file name

      Year_string = STRING(format='(I4.4)',L1_year)
      Day_string  = STRING(format='(I3.3)',L1_day)
      L1_file = 'PO_CEP_L1_IES_RATES_' + Year_string + Day_string + '.DAT'

;	Obtain an LU for the IES Rates-mode Level-1 file

      ON_IOERROR, Failed_IES_Rates
      GET_LUN, L1_unit

;	Open the file with fixed record length for writing
;	Write new file open message to log

      print, 'Opening new CEPPAD Rates Mode IES Level 1 file '
      print, '    file = ', L1_file, $
             '    starting at time  = ', ths_spn_time/1000.

      OPENW, L1_unit, L1_file, ERROR=err, IES_Rates_recl 
      IES_Rates_unit = L1_unit
      IES_Rates_assoc = ASSOC(IES_Rates_unit,BYTARR(IES_Rates_recl))
      IES_Rates_rec   = 1L	; Initialize the total number of records 
				; stored in the file
      RETURN			; Return after successful IES Rates file open

Failed_IES_Rates:
      print, '*** Open_new_Rates ***'
      print, '  Error # ', !ERR, ' encountered opening file ',L1_file
      print, !ERR_STRING
      error_return = 1
      RETURN

   ENDIF	;   ( (ths_spn_format EQ IES_Rates_format ) AND 
		;       ( IES_Rates_unit LE -3 ) ) 



;		HIST Rates-mode Level-1 file
   IF ( (ths_spn_format EQ HIST_Rates_format ) AND ( HIST_Rates_unit LE -3 ) ) $
                                                                  THEN BEGIN

;	Construct the Level-1 file name

      Year_string = STRING(format='(I4.4)',L1_year)
      Day_string  = STRING(format='(I3.3)',L1_day)
      L1_file = 'PO_CEP_L1_HIST_RATES_' + Year_string + Day_string + '.DAT'

;	Obtain an LU for the HIST Rates-mode Level-1 file

      ON_IOERROR, Failed_HIST_Rates
      GET_LUN, L1_unit

;	Open the file with fixed record length for writing
;	Write new file open message to log

      print, 'Opening new CEPPAD Rates Mode HIST Level 1 file '
      print, '    file = ', L1_file, $
             '    starting at time  = ', ths_spn_time/1000.

      OPENW, L1_unit, L1_file, ERROR=err, HIST_Rates_recl 
      HIST_Rates_unit = L1_unit
      HIST_Rates_assoc = ASSOC(HIST_Rates_unit,BYTARR(HIST_Rates_recl))
      HIST_Rates_rec   = 1L	; Initialize the total number of records 
				; stored in the file
      RETURN			; Return after successful HIST Rates file open

Failed_HIST_Rates:
      print, '*** Open_new_Rates ***'
      print, '  Error # ', !ERR, ' encountered opening file ',L1_file
      print, !ERR_STRING
      error_return = 1
      RETURN

   ENDIF	;   ( (ths_spn_format EQ HIST_Rates_format ) AND 
		;     ( HIST_Rates_unit LE -3 ) ) 

END





PRO OPEN_PL_LEVEL0, FILENAME_LEVEL0, IUNIT, IERROR

;                Open POLAR LEVEL0 file
;_________________________________________________G. M. Boyd 21 January 1994


;		Passed Parameters

;	FILENAME_LEVEL0
;	IUNIT
;	IERROR

; ----------------------------------------------------------------

ierror = 0                     ; If no error occurs in opening level0 file,
                               ;    ierror is equal to zero.

ON_IOERROR,  OPEN_ERR          ;  Error opening file  go to OPEN_ERR
OPENR, iunit, filename_level0, $
       /stream    ; Open file, FILENAME_LEVEL0, to be read


return

OPEN_ERR:
print, format = '(1x, "AN ERROR HAS OCCURED OPENING ",A80)', FILENAME_LEVEL0
print, !ERR_STRING
ierror = 1

return
end





pro Open_SH, error_return

;	Obtain unit number, construct filename, define associated array, 
;	initialize the record pointer and open the Level-1 Spin Header
;	file with fixed record size  for write access.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed SH Level-1 file open

;	Mar 06 1996	M.T.Redding	Corrected printed variable 'err' to be
;					system variable '!ERR

;--------------------------------------------------------------------

   COMMON CEP_Spin_head,   SH_unit, SH_record, SH_assoc, SH_rec_len

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no


;-----------------------------------------------------------------------

   error_return = 0

   IF ( SH_unit LE -2 ) THEN BEGIN	; Only open if not already open

;	Make up string pieces of Level-1 file name using the data year, day and
;	the next available Level-1 volume number.
;	Example Level-1 file name: PO_L1_CEP_SH_1994123.DAT

;	Year and Day

      Year_string = STRING(format='(I4.4)',DRH.ATC_yr)
      Day_string  = STRING(format='(I3.3)',DRH.ATC_day)

      ON_IOERROR, Failed_Open

;	Obtain an LU for the CEPPAD Spin header Level-1 file
      SH_unit = -3	; Initialize to an out of IDL LUN range value 
			; to indicate file not open
      GET_LUN, L1_unit

;	Construct the Level-1 file name
      L1_file = 'PO_CEP_L1_SH_' + $
                    Year_string + Day_string + '.DAT'


;	Open the file with fixed record length for writing

      OPENW, L1_unit, L1_file, ERROR=err, SH_rec_len
      SH_unit      = L1_unit
      SH_record    = 3L
      SH_assoc     = ASSOC(SH_unit,BYTARR(SH_rec_len))
      error_return = 0
      GOTO, Exit_SH_Open

Failed_Open:
      print, '*** Open_SH ***'
      print, '  Error # ', !ERR, ' encountered opening file ',L1_file
      print, !ERR_STRING
      error_return = 1

   ENDIF	; ( SH_unit LE -2 )


Exit_SH_Open:
   return
end





pro Process_HK, error_return

;	Transfer the Housekeeping and Subcom values read in one record (one
;	major frame) to the Housekeeping Level-1 file

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revisions
;	07 May 1996	M.T.Redding	Added COMMON block CEP_HK_Times
;					containing variables CEP_KP_time, &
;					CEP_HIST_SE_times (HIST singles/events
;					spin times.
;					Also added code fill these variables
;					with either fill values (-1L) or
;					evaluations using previous major frame
;					times and previous spin times, resp.

;--------------------------------------------------------------------

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON CEP_mf_history,  previous_times

   COMMON Dat_struct,      CEP_level_0_unit, data, CEP_Level_0_FLR

   COMMON CEP_HK,          CEP_HK_unit, CEP_HK_rec, CEP_HK_rec_len, CEP_HK_data

   COMMON CEP_HK_times,    CEP_KP_time, $
                           CEP_HIST_SE_times

;-----------------------------------------------------------------------

;	Create Level-1 Housekeeping filename and open it up
   IF CEP_HK_unit le 0 THEN BEGIN
      Open_HK,error_return
      IF error_return ne 0 THEN RETURN
   ENDIF

   GET_KP_TIME, error_return

;	Write new HK values to the next record and update the heading record

   Append_HK, error_return
   IF error_return ne 0 THEN RETURN

;	 Close up the Level-1 housekeeping file if there are no more
;	records in the Level-0 file

   IF EOF(CEP_level_0_unit) THEN BEGIN
      Append_HK, /FLUSH_HK, error_return
      Update_HK_Header, error_return
      CLose_HK, error_return
   ENDIF

   RETURN
end





pro Process_MF, error_return

;	Control the handling of processing needed for each major frame
;	of data read.  This includes revising the Major frame time history 
;	and the extraction and stprage of CEPPAD Housekeeping parameters into
;	a CEPPAD Level-1 Housekeeping file.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;--------------------------------------------------------------------

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON CEP_mf_history,  previous_times

   COMMON Dat_struct,      CEP_level_0_unit, data, CEP_Level_0_FLR

   COMMON CEP_HK,          CEP_HK_unit, CEP_HK_rec, CEP_HK_rec_len, CEP_HK_data

   COMMON CEP_two_frm,     Buffer_index, Two_frm_sci_dat, Two_frm_qual

;-----------------------------------------------------------------------

;	Unpack the minor frame quality flags, major frame ATC time and 
;	the CEPPAD major frame sequence number


   MF_ATC_time = DRH.atc_mils
   MF_CEP_seq_no = HK(55,0)
   Min_frm_quality = DRH.mnfr_qua


;	Unpack the CEPPAD Subcom values stored in Spacecraft telemetry word 15
;	minor frame 90 to 249

;   Unpack_Subcom, error_return
    Get_Checksum, error_return

;	Transfer the major frame science bytes to the 2-frame science
;	byte buffer

  Two_frm_sci_dat(0,0,buffer_index) = Science_data
  Two_frm_qual(0,buffer_index)      = Min_frm_quality


;	Update the Major frame time history array

   Previous_times(MF_CEP_seq_no) = MF_ATC_time


;	Extract and write into the CEPPAD Level-1 Hk file the HK and Subcom
;	words read in major frame

   Process_HK, error_return

   error_return = 0
   return
end





pro Process_new_spin, error_return

;	Control adding the new spin's data to the 16-spin block array;
;	mapping the current 16-spin block to the 32-time block array;

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block
;	20 Sep 95	M. T. Redding	Corrected reset of 'lst_spn' common
;					block parameters to include all
;					heading parameters
;	25 Sep 95	M. T. Redding	Corrected the interpretation of 
;					ths_spn_format to use the 4 LSBs rather
;					than the 4 MSBs as Normal-Science mode 
;					TM acquisition table number 
;	10 Apr 96	M. T. Redding	Added 'ignore_32_times' to 'CEP_32_time'
;					common block
;					Added call to CHECK_ACQ_MODE to check 
;					for change of mode (IPS-LUT, IES-LUT, 
;					HIST-LUT, IPS-Mux) from previous spin.  
;					If such a change occurs any pending 
;					'Normal'/'Science' mode data is appended
;					to the Level-1 file and the time-blk and
;					spin-blk buffers are re-initialized and
;					flags set to skip 32 spins of data 
;					(mixed data will occur from mapping data
;					from different modes)
;					Added 'new_format', 
;					'Spin_Header_OK' and 
;					'New_Acq_Mode' as flag values 
;					indicating these states.
;					Revised the blocking of logic used to
;					write/append normal/rates/fixed mode
;					data.
;	Apr 11 96	M.T.Redding	Added COMMON block CEP_major_frame
;	06 May 1996	M. T. Redding	Added call to FILL_MISSING_TIMES before 
;					each call to MAP_TO_TIME_BLK to replace
;					and 'missing' (-1L) time values with
;					estimate values based on spin durations
;					determined in the current 16-spin block
;					Added COMON block CEP_spin_times with 
;					array variable 'previous_spin_times' 
;					and code to revise ths_spn_SCOM entry
;					with ths_spn_time.
;	May 22 96	M.T.Redding	Added 'Commanded_change' to call to
;					'Check_Spin_Header'
;	Jun 20 1996	M.T.Redding	Put variable 'Commanded_change' into 
;					COMMON block 'PRC_NEW_SPN_WORK'

;--------------------------------------------------------------------

   COMMON PRC_NEW_SPN_WORK, Commanded_change

   COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                           sci_byt_per_spin, max_spins, $
                           nom_spin_dur, TRUE, FALSE, sync_byte, $
                           CEP_Sci_Man_byt_per_mnfrm, $
                           CEP_Conting_byt_per_mnfrm, $
                           CEP_Sci_Man_rec_len, $
                           CEP_Conting_rec_len

   COMMON CEP_32_time,     ignore_16_times, ignore_32_times

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

   COMMON CEP_lst_spn,     lst_spn_seq_no, lst_spn_min_frm, $
                           lst_spn_format, lst_spn_IPS_tab, $
                           lst_spn_IES_tab, lst_spn_IES_com, $
                           lst_spn_HIST_tab, lst_spn_HIST_com, $
                           lst_spn_SCOM, lst_spn_IPS_Mux, $
                           lst_spn_time, lst_spn_CRC, lst_spn_quality

   COMMON CEP_16_spin,     Spin_blk, Spin_blk_times, Spin_blk_SCOMs,	$
                           spin_blk_tot, spin_blk_seq_wrap,		$
                           est_16th_time, est_32nd_time, 		$
                           Bad_SCOM_MODE_sync, spin_quality

   COMMON Table_CRCs,      CRC_Flt_SW, CRC_MST, CRC_SAT, CRC_ICT, $
                           CRC_IESLUT, CRC_HISTLUT

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON CEP_spin_history, previous_spin_times, spin_durations

;-----------------------------------------------------------------------

;	Check for inconsistancies in the current spin's time and SCOM as 
;	compared with the estimated 16th time (based on previously processed 
;	spins' times), the current major frame time and the estimate for
;	the next major frame time

   CHECK_SPIN_HEADER, Spin_header_OK, Commanded_Change, error_return


   IF ( Spin_Header_OK ) THEN BEGIN


;	Check and if necessary revise Normal-Science mode format table number

      CORRECT_FORMAT, Format_number_OK, error_return


;	Update the spin history array and spin duration array for ths spin

      previous_spin_times( ths_spn_SCOM ) = ths_spn_time
      spin_durations( ths_spn_SCOM )      = nom_spin_dur

;	If possible, update the spin duration array for the immediately 
;	preceding spin duration using the current and previous spin time
;	Also redefine the spin duration for this spin using the previous
;	spin's duration value

      preceeding_SCOM = BYTE( ths_spn_SCOM - 1B )
      IF ( previous_spin_times( preceeding_SCOM ) NE -1B ) THEN BEGIN
         temp_dur = ths_spn_time - previous_spin_times( preceeding_SCOM )
         IF ( ( temp_dur LE ( 1.5 * FLOAT( nom_spin_dur ) ) ) AND $
              ( temp_dur GT ( 0.5 * FLOAT( nom_spin_dur ) ) ) ) THEN BEGIN
            spin_durations( preceeding_SCOM ) = temp_dur
            spin_durations( ths_spn_SCOM )    = temp_dur
         ENDIF
      ENDIF


;	Check for changes in the telemetry format.
;	Revise previous spin header quality flags for this spin block if a
;	change in mode has just occurred and it is not in sync with the SCOM.
;	Update the Spin header file with any new spin header info.

      new_format = ( lst_spn_format NE ths_spn_format )
      CHECK_ACQ_MODE, New_Acq_mode, error_return
      IF ( Bad_SCOM_MODE_sync ) THEN BEGIN
         ths_spn_quality = spin_quality	; Mode change out of sync with SCOM
         IF ( ( lst_spn_quality EQ 0 ) AND				$
              ( ( lst_spn_SCOM / 16 ) EQ ( ths_spn_SCOM / 16 ) ) AND	$
              ( ( ths_spn_time - lst_spn_time ) LT 96000 ) ) OR		$
            ( ( ( lst_spn_SCOM mod 16 ) EQ 0 ) AND			$
              ( ( lst_spn_SCOM / 16 ) EQ ( ths_spn_SCOM / 16 ) ) AND	$
              ( ( ths_spn_time - lst_spn_time ) LT 96000 ) ) THEN	$
            REVISE_STORED_SPIN_HEADER, QUALITY = spin_quality, error_return
      ENDIF ELSE BEGIN
         ths_spn_quality = spin_quality	; Mode change in sync with SCOM
         IF ( ( lst_spn_quality NE 0 ) AND				$
              ( ( lst_spn_SCOM / 16 ) EQ ( ths_spn_SCOM / 16 ) ) AND	$
              ( ( ths_spn_time - lst_spn_time ) LT 96000 ) ) THEN	$
            REVISE_STORED_SPIN_HEADER, QUALITY = spin_quality, error_return
      ENDELSE
      Append_SH, error_return


;	Always expect to trash the first 16 spins/times of data TM'd using a 
;	new TM 	format
;	- the flag 'ignore_16_times' will be set to a TRUE logical value when 
;	a new format is encountered
;	- it will be unset to a FALSE value after a new 16-spin block is 
;	encountered and the previous 16-time block is processed (which may be
;	either appending to the level-1 files, pushing up the last 16 times
;	of the 32-time block, and re-initializing the last 16 times of the
;	32-time block; or, if 'ignore_16_times' is TRUE then just going
;	straight to the pushing up the bottom and re-initializing the last 
;	16-times in the 32-time block



;	NEW CEPPAD TM FORMAT, MODE OR TIME_SCOM INCONSISTANCY

;	Handle newly encountered CEPPAD TM format
;   	IF a new TM format was used for this spin 
;       ( ths_spn_format NE lst_spn_format )

      IF new_format THEN BEGIN

         IF ( ( lst_spn_format / 16) EQ 0 ) THEN BEGIN

;	Finish handling any remaining 'Normal-mode' data in the 32-time buffer

;	Handle case in which the first 16 times in the 32-time block
;	array have been flagged to be included in the Level-1 data

            IF NOT ( ignore_16_times ) THEN BEGIN

;	Map the current 16-spin buffer to the time block.
;	Re-initialize the 16-spin buffer.
;	Append the first 16-times of data in the 32-time block to 
;	the level-1 files.

               FILL_MISSING_TIMES, bad_filled_times 
               IF ( bad_filled_times EQ 0 ) THEN $
                  Map_to_time_blk, error_return
               Init_16_blk, error_return
               Append_L1, error_return

            ENDIF	; ( NOT ignore_16_times )

;		Close any Level-1 data files that have been opened so far

;	Write file closing message to log

            PRINT, ' Closing level 1 files at ending spin time = ', $
                                                            lst_spn_time/1000.
            Update_L1_header, error_return
            Close_L1, error_return

         ENDIF	; ( ( lst_spn_format / 16) EQ 0 )


;	OPEN ANY NEW FILES REQUIRED BY THIS NEW TM FORMAT

         IF ( (ths_spn_format / 16) EQ 0 ) THEN BEGIN

;	Open new 'Normal-mode' Level-1 file(s)

;	Set up the files and flags for a CEPPAD 'Normal'/'Science' mode 
;	telemetry - 'Normal'/'Science' <==> ths_spn_format / 16 = 0 (the 
;	4 most significant bits of the format byte are '0')
;	Initialize the 32-time buffer and set flag to not copy the
;	first 16 spins of data ( Which TM table was in effect prior to
;	this spin is not known so the first 16 spins of data output
;	using a TM table must be skipped.)

            Open_new_normal, error_return
            Shift_time_blk, error_return	; init last 16 time areas
            Shift_time_blk, error_return	; init first 16 time areas
            ignore_16_times = TRUE		; 
            IF ( (ths_spn_SCOM MOD 16) EQ 0 ) THEN ignore_32_times = FALSE $
                                              ELSE ignore_32_times = TRUE

         ENDIF ELSE BEGIN

;	Handle change to a Rates-histogram mode

            IF ( ISHFT( ths_spn_format, -4) GT 1 ) THEN BEGIN

;	Rates histogram mode may include either full-spin or single sectored
;	telemetry for IPS, IES or HIST instruments.  Rates-histogram modes
;	are indicated by non-zero values in bits 5/6 of ths_spn_format byte.
;	When bits 5/6 are evaluated, values of 1, 2 and 3 indicate the 
;	IPS, IES or HIST instruments, respecively.  Bit 7 indicates either
;	single-sectored telemetry ('0') or full-spin telemetry ('1').

;	The mode is a Rates Histogram mode ( ths_spn_format = '191' (IPS-);
;	'223' (IES-); '255 (HIST-Rates) )

;	Open up a new Rates-mode Level-1 file if not already open

               Open_new_Rates, error_return

            ENDIF ELSE BEGIN

;	Handle fixed telemetry mode 

;		** NO processing defined yet for fixed telemetry mode **

            ENDELSE

         ENDELSE

      ENDIF ELSE BEGIN
      
;	NOT A NEW TM FORMAT

;	Handle data acquisition mode change

         IF  ( Commanded_change ) THEN BEGIN

;	Write warning message if a change in mode occurs other than in the
;	first spin encountered in a 16-spin block

            IF ( ths_spn_time LE est_16th_time ) THEN BEGIN
               PRINT, ' PROCESS_NEW_SPIN: New Acquisition Mode encountered', $
                      ' middle of 16-spin block '
               print, ' lst_spn_format   = ', lst_spn_format, $
                         ' ths_spn_format   = ', ths_spn_format
               print, ' lst_spn_IPS_tab  = ', lst_spn_IPS_tab, $
                         ' ths_spn_IPS_tab  = ', ths_spn_IPS_tab
               print, ' lst_spn_IES_tab  = ', lst_spn_IES_tab, $
                         ' ths_spn_IES_tab  = ', ths_spn_IES_tab
               print, ' lst_spn_IES_com  = ', lst_spn_IES_com, $
                         ' ths_spn_IES_com  = ', ths_spn_IES_com
               print, ' lst_spn_HIST_tab = ', lst_spn_HIST_tab, $
                         ' ths_spn_HIST_tab = ', ths_spn_HIST_tab
               print, ' lst_spn_HIST_com = ', lst_spn_HIST_com, $
                         ' ths_spn_HIST_com = ', ths_spn_HIST_com
               print, ' lst_spn_SCOM     = ', lst_spn_SCOM, $
                         ' ths_spn_SCOM     = ', ths_spn_SCOM
               print, ' lst_spn_IPS_Mux  = ', lst_spn_IPS_Mux, $
                         ' ths_spn_IPS_Mux  = ', ths_spn_IPS_Mux
               print, ' lst_spn_time     = ', lst_spn_time, $
                         ' ths_spn_time     = ', ths_spn_time
               PRINT, ' est_16th_time    = ', est_16th_time/1000.d0
            ENDIF


;	PROCESSING FOR NEW ACQUISITION MODE ENCOUNTERED IN THE
;	MIDDLE OF A 16-SPIN BLOCK - "COMMANDED CHANGE" - 
;	SCOM 30 occurring out-of-sequence

;	NEW MODE IS NORMAL/SCIENCE

            IF ( ( ths_spn_format / 16 ) EQ 0 ) THEN BEGIN

;	Save the data from the older 16 spins if possible

               IF NOT ( ignore_16_times ) THEN BEGIN
                  Append_L1, error_return
               ENDIF


;	Initialize the 32-time buffer and set flag to not copy the
;	first 32 spins of data ( Neither this spin nor any following
;	in this spin block (SCOM 0-15 mod 16) should be stored because
;	the data contained in them is mixed up (due to the SCOM skip
;	or mode change).  Spin data is first placed in a buffer
;	containing the 16 spins of a 'spin-block'.  The values here
;	are mapped out to a time-block buffer containing data for
;	32 spins (the previous 16-spins and the current 16-spins).
;	When a change of mode occurs or a time/SCOM inconsistancy
;	occurs (as might occur when a mode change occurs), the
;	spin-block and the time block buffers are cleared out and
;	the current spin (and any subsequent spins in the new 
;	spin-block will eventually be mapped to the 32-time buffer.
;	Since the data in these spins is suspect all 32-times will
;	need to be skipped.

               Shift_time_blk, error_return	; init first 16 time areas
               Shift_time_blk, error_return	; init first 16 time areas
               Init_16_blk, error_return	; init 16-spin block
               ignore_16_times = TRUE		; flag to not save 1st 16 times
               ignore_32_times = TRUE		; flag to not save 2nd 16 times

            ENDIF

         ENDIF

      ENDELSE



;	TRANSFER AND/OR WRITE 1 NEW SPIN DATA SET 



;	NORMAL TM MODE

      IF ( ths_spn_format / 16 EQ 0 ) THEN BEGIN

;	Assign the current CRC number if CEPPAD Normal TM

         norm_table_no = ISHFT(ISHFT(ths_spn_format,4),-4)	; use the 4 LSB
         ths_spn_CRC   = CRC_MST(norm_table_no)


;	NEW 16-SPIN BLOCK

;	Handle the case of the spin belonging to a new 16-spin block
;	( an estimated 16th time has been calculated and 
;	ths_spn_time GT est_16th_time )

         IF ( ( ( ths_spn_time GT est_16th_time ) AND $
                ( est_16th_time GT 0 ) ) OR $
              ( ( lst_spn_time GT 0 ) AND $
                ( ( lst_spn_SCOM MOD 16 ) NE 15 ) AND $
                ( ths_spn_SCOM / 16 NE lst_spn_SCOM / 16 ) ) ) THEN BEGIN


;	Generate estimates for missing times in 16-spin block.
;	Map the current 16-spin buffer to the time block.
;	Store what was the 'incomplete' data associated with the second
;	set of 16 times in the time block if the (now) first 16 times in the 
;	time block are not 'bad' or questionable quality.
;	Re-initialize the last 16 times of the 32-time block.
;	Push up last 16 times in the 32-time block into the first 16 
;	time positions.

            FILL_MISSING_TIMES, bad_filled_times 
            IF ( bad_filled_times EQ 0 ) THEN BEGIN
               MAP_TO_TIME_BLK, error_return
            ENDIF
            IF ( NOT ignore_16_times ) THEN BEGIN
               APPEND_l1, error_return
            ENDIF
            INIT_16_BLK, error_return
            SHIFT_TIME_BLK, error_return

;	Reset the flags for storing data in the 32-time buffer.
;	Leave ignore_16_times set to TRUE if just skipped the first 16
;	of 32 spins so that the next set of 16 spins will also be skipped.

            IF ( Commanded_change ) THEN BEGIN
               ignore_32_times = TRUE
               ignore_16_times = TRUE
            ENDIF ELSE BEGIN
               IF ignore_32_times THEN BEGIN
                  ignore_32_times = FALSE
                  ignore_16_times = TRUE
               ENDIF ELSE BEGIN
                  ignore_16_times = FALSE
               ENDELSE
               IF ( bad_filled_times EQ 1 ) THEN Ignore_16_times = TRUE
            ENDELSE


;	Additional handling if the currents spin is beyond even the next 
;	16-spin block

            IF ( (ths_spn_time GT est_32nd_time) AND ( est_32nd_time GT 0 ) ) $
                                                                    THEN BEGIN

;	If the first 16 times in the time block are not 'bad' or otherwise 
;	of questionable quality, append the first 16-times of data in the 
;	32-time block to the level-1 files.
;	Push up last 16 times in the 32-time block into the first 16 time 
;	positions - do this twice to completely reinitialize the 32-time block.
;	Re-initialize the last 16 times of the 32-time block.

               IF ( NOT ignore_16_times ) THEN BEGIN
                  Append_L1, error_return
               ENDIF
               Shift_time_blk, error_return
               Shift_time_blk, error_return
               ignore_16_times = TRUE

            ENDIF

         ENDIF




;	SPIN DATA CLEAR TO BE ADDED TO 16-SPIN BLOCK

;	Add this spin data to the 16-spin block.
;	Increment spin_blk_tot.
;	Reset last spin identifiers for comparisons with the next spin
;	values to be read.

         ADD_NEW_SPIN, error_return


;	CHECK STATUS OF 16-SPIN BLOCK 

;	Handle just filled 16-spin block

         IF ( ths_spn_SCOM mod 16 EQ 15 ) THEN BEGIN


;	Generate estimates for missing times in 16-spin block.
;	Map the current 16-spin buffer to the time block .
;	Initialize the 16-spin buffer and reset spin_blk_tot to 0.
;	Append the first 16-times of data in the 32-time block to 
;	the level-1 files.

            FILL_MISSING_TIMES, bad_filled_times 
            Map_to_time_blk, error_return
            Init_16_blk, error_return
            IF ( NOT (ignore_16_times) ) THEN BEGIN
               Append_L1, error_return
            ENDIF


;	Push up the last 16 times in the 32-time block into the.
;	first 16 time positions and initialize the last 16 times of the 
;	32-time block.

            Shift_time_blk, error_return

;	Re-set the flags handling storage of the 32-time buffer data.
;	Leave ignore_16_times set to TRUE if just skipped the first 16
;	of 32 spins so that the next set of 16 spins will alos be skipped

            IF ignore_32_times THEN BEGIN
               ignore_32_times = FALSE
               ignore_16_times = TRUE
            ENDIF ELSE BEGIN
               ignore_16_times = FALSE
            ENDELSE

         ENDIF	; ( ths_spn_SCOM mod 16 EQ 15 )



      ENDIF ELSE BEGIN


;			RATES-Mode TM

         IF ( ISHFT( ths_spn_format, -4 ) GT 1 ) THEN BEGIN

            ths_spn_CRC   = -1
            Append_Rates, error_return

         ENDIF ELSE BEGIN

;	Fixed telemetry mode - ( ISHFT( ths_spn_format, -4 ) EQ 1 )

;	** No fixed telemetry mode processing defined yet

         ENDELSE



      ENDELSE

;	Reset the last spin identifiers to the current spin's values for
;	comparison with future spins

      lst_spn_SCOM     = ths_spn_SCOM
      lst_spn_time     = ths_spn_time
      lst_spn_format   = ths_spn_format
      lst_spn_CRC      = ths_spn_CRC
      lst_spn_seq_no   = ths_spn_seq_no
      lst_spn_min_frm  = ths_spn_min_frm
      lst_spn_IPS_tab  = ths_spn_IPS_tab
      lst_spn_IES_tab  = ths_spn_IES_tab
      lst_spn_IES_com  = ths_spn_IES_com
      lst_spn_HIST_tab = ths_spn_HIST_tab
      lst_spn_HIST_com = ths_spn_HIST_com
      lst_spn_SCOM     = ths_spn_SCOM
      lst_spn_IPS_Mux  = ths_spn_IPS_Mux
      lst_spn_quality  = ths_spn_quality

   ENDIF ELSE BEGIN	; ( Spin_Header_OK )

;	Print message indicating bad spin header encountered

;;      PRINT, ' Inconsistant Spin header encountered at time = ', $
;;                                  MF_ATC_time / 1000.0d0
;;      PRINT, ' Previous good spin was at time = ', lst_spn_time / 1000.0d0
   ENDELSE


   error_return = 0

   RETURN
END





PRO RD_PL_LEV0_FLR, IUNIT, FILENAME_LEVEL0, FILE_LABEL_REC, IERROR

;  Read the Level-Zero Data File Label Record (RFL) Definition which has at
;                       least 2792 bytes
; 

;		Passed Parameters

;	IUNIT
;	FILENAME_LEVEL0
;	FILE_LABEL_REC
;	IERROR		Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not used


;	02 Aug 95	M.T.Redding	Changed minimum level-0 record length
;					from 1352 to 2792

; ----------------------------------------------------------------

;              Read Level0 data FLR Definition 

ierror = 0           ; Error flag for reading, set to zero if everything okay 
 
file_label_rec =  bytarr(2792)

ON_IOERROR, READ_ERR
readu, iunit, file_label_rec

return

READ_ERR:
 ierror = 1
 print, ' '
 print, format = '(1x,"ERROR READING POLAR CEPPAD DATA FILE LABEL RECORD" )
 print, format = '(1x, "READING FROM FILE ", A60)', filename_level0 
 print, !ERR_STRING

return
end





PRO Read_CEP_TM_tables, error_return, TM_table_unit


;	PURPOSE:	Read CEPPAD Telemetry/Accumulation table values

;		Passed Parameters

;	TM_table_unit
;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure


; ----------------------------------------------------------------

   COMMON CEP_lvl0_1_map, $
        Trgt_times, $
	num_source_spins,$
	num_IPS_energies, num_IPS_channels, IPS_spins_per_sample,$
		IPS_sectors_per_spin, IPS_Source_loc, IPS_Source_len,$
		IPS_Target_loc, IPS_trgt_tim_Loc, IPS_trgt_blk_len,$
                IPS_trgt_blk, IPS_source_byte_loc, IPS_trgt_byte_loc, $
	num_IES_energies, num_IES_channels, IES_spins_per_sample,$
		IES_sectors_per_spin, IES_Source_loc, IES_Source_len,$
		IES_Target_loc, IES_trgt_tim_loc, IES_trgt_blk_len,$
                IES_trgt_blk, IES_source_byte_loc, IES_trgt_byte_loc, $
	num_HISTe_energies, num_HISTe_channels, HISTe_spins_per_sample,$
		HISTe_sectors_per_spin, HISTe_Source_loc, HISTe_Source_len,$
		HISTe_Target_loc, HISTe_trgt_tim_loc, HISTe_trgt_blk_len,$
                HISTe_trgt_blk, HISTe_source_byte_loc, HISTe_trgt_byte_loc, $
	num_HISTp_energies, num_HISTp_channels, HISTp_spins_per_sample,$
		HISTp_sectors_per_spin, HISTp_Source_loc, HISTp_Source_len,$
		HISTp_Target_loc, HISTp_trgt_tim_loc, HISTp_trgt_blk_len,$
                HISTp_trgt_blk, HISTp_source_byte_loc, HISTp_trgt_byte_loc



;----------------------------------------------------------------------


;	Read IPS TM table entries

   Read_next_table, error_return, $
		TM_table_unit, num_IPS_energies, num_IPS_channels, $
		IPS_Spins_per_sample, IPS_Sectors_per_spin


;	Read IES TM table entries

   Read_next_table, error_return, $
		TM_table_unit, num_IES_energies, num_IES_channels, $
		IES_Spins_per_sample, IES_Sectors_per_spin


;	Read HISTe TM table entries

   Read_next_table, error_return, $
		TM_table_unit, num_HISTe_energies, num_HISTe_channels, $
		HISTe_Spins_per_sample, HISTe_Sectors_per_spin


;	Read HISTp TM table entries

   Read_next_table, error_return, $
		TM_table_unit, num_HISTp_energies, num_HISTp_channels, $
		HISTp_Spins_per_sample, HISTp_Sectors_per_spin

   RETURN
END




pro Read_data_rec, data_rec_no, buffer_index, error_return

;	Read a record from the level_0 file and put science data and quality
;	flag values into the buffer designated

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 99	Error reading Major frame data

;--------------------------------------------------------------------

COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                        Min_frm_quality, MF_ATC_time, MF_CEP_seq_no
COMMON CEP_mf_history, previous_times


;-----------------------------------------------------------------------

   error_return = 0

   Get_majfrm_data, data_rec_no, DRH, Science_data, $
                                                HK, Subcom, error_return


;	Perform processing needed on all major frame of data

   Process_MF, error_return

   If error_return ne 0 THEN BEGIN
      print, ' *** Read_data_rec ***'
      print, ' Error #', error_return,' returned from procedure GET_MAJFRM_DATA'
      print, ' Initiating termination'
      error_return = 99
      RETURN
   ENDIF

   RETURN
end




PRO Read_next_table, error_return, $
		TM_table_unit, num_energies, num_channels, $
		Spins_per_sample, Sectors_per_spin


;	PURPOSE:	Read one table from the CEPPAD Telemetry/Accumulation 
;			table file


;		Passed Parameters

;	TM_table_unit
;	num_energies
;	num_channels
;	Spins_per_sample
;	Sectors_per_spin
;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure


; ----------------------------------------------------------------

   num_energies = 0	; define as integer
   num_channels = 0	; define as integer

;	Read TM table array dimensions

   READF, TM_table_unit, num_energies, num_channels

;	Dimension arrays to this table's size

   Spins_per_sample = intarr(num_energies)
   Sectors_per_spin = intarr(num_energies, num_channels)

;	Read the array values defining the Acquisition/Accumulation scheme

   READF, TM_table_unit, Spins_per_sample
   READF, TM_table_unit, Sectors_per_spin


   RETURN
END


PRO Replace_host_LONGs_with_network, NumberToBeReplaced, LocationOfLONGs, $
                                     TargetBuffer

;	Replace host format LONG words with 'network' format LONGs words
;	in specified byte array

;	NumberToBeReplaced	Number of LONGwords (4 bytes) to be replaced
;	LocationOfLONGs		Array containing pointers to locations 
;				within the byte array of the LONGWORDS to be
;				converted
;	TargetBuffer		BYTE array containing the LONGWORDS to be
;				replaced

;	revision:
;	11 Sep Aug 95	M.T.Redding

;--------------------------------------------------------------------

;	Define a local array to contain the LONGWORD values

   NetLONG = LONARR( NumberToBeReplaced )

; 	Extract and convert LONGWORDS from the BYTE buffer

   FOR i = 0, NumberToBeReplaced - 1 DO BEGIN
      NetLONG( i ) = LONG( TargetBuffer, LocationOfLONGs( i ) )
   ENDFOR

;	Convert the LONGWORDs from 'host' format to 'network'

   BYTEORDER, NetLONG, /HTONL

;	Replace the converted LONGWORDS into the byte array

   FOR i = 0, NumberToBeReplaced - 1 DO BEGIN
      TargetBuffer( LocationOfLONGs( i ) ) = BYTE( NetLONG( i ), 0, 4 )
   ENDFOR

   RETURN
END



PRO REVISE_STORED_SPIN_HEADER, QUALITY = quality, error_return

;	Revise spin header information for spin previously stored in the
;	Spin Header file with the keyword value specifed.  

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1-	File not open or Write error
;						encountered

;		Keyword paramters

;	quality		Spin quality value to replace previous spin's value

;	revision
;	11 Jun 1996	M.T.Redding

;--------------------------------------------------------------------

   COMMON CEP_Spin_head,   SH_unit, SH_record, SH_assoc, SH_rec_len

   COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                           Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

   COMMON CEP_lst_spn,     lst_spn_seq_no, lst_spn_min_frm, $
                           lst_spn_format, lst_spn_IPS_tab, $
                           lst_spn_IES_tab, lst_spn_IES_com, $
                           lst_spn_HIST_tab, lst_spn_HIST_com, $
                           lst_spn_SCOM, lst_spn_IPS_Mux, $
                           lst_spn_time, lst_spn_CRC, lst_spn_quality

   COMMON Level_1_fil_typ, HK_filtyp, IPS_Nrm_filtyp, IES_Nrm_filtyp, $
                           HISTe_Nrm_filtyp, HISTp_Nrm_filtyp, $
                           IPS_Rts_filtyp, IES_Rts_filtyp, HIST_Rts_filtyp, $
                           Spin_Head_filtyp

   COMMON CEP_Spin_head_net, net_SH_L1_record, net_SH_L1_rec_len, $
                           net_Spin_Head_filtyp, net_SH_L1_year, $
                           net_SH_L1_day, net_SH_L1_begt, $
                           net_SH_L1_endt

   COMMON File_Control, File_info

;-----------------------------------------------------------------------


   error_return = 0

;	Set flags indicating which parameters are to be revised

   revise_quality = 0
   IF ( N_ELEMENTS( quality ) NE 0 ) THEN revise_quality = 1


;	Only try to revise a spin if the Spin Header file is already open
;	and the current Spin header record number is greater than or equal
;	to 4 ( the second record containing spin header data)

   IF ( ( SH_unit LE 0 ) OR ( SH_record LT 4 ) ) THEN BEGIN
      PRINT, ' REVISE_STORED_SPIN_HEADER: attempt to revise a non-existant spin'
      IF ( SH_unit LE 0 ) THEN BEGIN
         PRINT, ' Spin header file is not open yet -  SH_unit = ', SH_unit
      ENDIF ELSE BEGIN
         PRINT, ' Current Spin header record number = ', SH_record 
      ENDELSE
      RETURN
   END


;	Read the most recently written Spin header data record

   temp = BYTARR( SH_rec_len )
   ON_IOERROR, Failed_SH_read
   temp( * ) = SH_assoc( SH_record - 1 )


;	Revise the Header parameter(s) specified

;		Quality flag

   IF ( revise_quality ) THEN BEGIN
      temp( 13 ) = quality
   ENDIF


;	Rewrite the revised record entries

   ON_IOERROR, Failed_SH_write
   SH_assoc( SH_record - 1 ) = temp( * )


;   temp(  0 )  = BYTE(net_ths_spn_time,0,4)
;   temp(  4 )  = ths_spn_format
;   temp(  5 )  = ths_spn_IPS_tab
;   temp(  6 )  = ths_spn_IES_tab
;   temp(  7 )  = ths_spn_IES_com
;   temp(  8 )  = ths_spn_HIST_tab
;   temp(  9 )  = BYTE(net_ths_spn_HIST_com,0,2)
;   temp( 11 )  = ths_spn_IPS_Mux
;   temp( 12 )  = ths_spn_SCOM
;   temp( 13 )  = ths_spn_quality
;   net_SH_L1_endt = LONG(ths_spn_time)

   error_return = 0
   RETURN


Failed_SH_write:
   PRINT, '*** REVISE_STORED_SPIN_HEADER ***'
   PRINT, 'Error #', !err, ' encountered during Spin header Level-1 file update'
   PRINT, !ERROR
   error_return = error_return + 1
   RETURN

Failed_SH_read:
   PRINT, '*** REVISE_STORED_SPIN_HEADER ***'
   PRINT, 'Error #', !err, ' encountered during Spin header Level-1 file read'
   PRINT, !ERROR
   error_return = error_return + 1
   RETURN
END





pro Shift_time_blk, error_return

;	Shift values from bottom half (last 16 spin times) of target 
;	block arrays to top half (first 16 spin times) initialize - 
;	put in 'missing' values into - bottom half of target block
;	arrays

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;	 				 1	Not set in this procedure

;--------------------------------------------------------------------

   COMMON CEP_lvl0_1_map, $
        Trgt_times, $
	num_source_spins,$
	num_IPS_energies, num_IPS_channels, IPS_spins_per_sample,$
		IPS_sectors_per_spin, IPS_Source_loc, IPS_Source_len,$
		IPS_Target_loc, IPS_trgt_tim_Loc, IPS_trgt_blk_len,$
                IPS_trgt_blk, IPS_source_byte_loc, IPS_trgt_byte_loc, $
	num_IES_energies, num_IES_channels, IES_spins_per_sample,$
		IES_sectors_per_spin, IES_Source_loc, IES_Source_len,$
		IES_Target_loc, IES_trgt_tim_loc, IES_trgt_blk_len,$
                IES_trgt_blk, IES_source_byte_loc, IES_trgt_byte_loc, $
	num_HISTe_energies, num_HISTe_channels, HISTe_spins_per_sample,$
		HISTe_sectors_per_spin, HISTe_Source_loc, HISTe_Source_len,$
		HISTe_Target_loc, HISTe_trgt_tim_loc, HISTe_trgt_blk_len,$
                HISTe_trgt_blk, HISTe_source_byte_loc, HISTe_trgt_byte_loc, $
	num_HISTp_energies, num_HISTp_channels, HISTp_spins_per_sample,$
		HISTp_sectors_per_spin, HISTp_Source_loc, HISTp_Source_len,$
		HISTp_Target_loc, HISTp_trgt_tim_loc, HISTp_trgt_blk_len,$
                HISTp_trgt_blk, HISTp_source_byte_loc, HISTp_trgt_byte_loc



;-----------------------------------------------------------------------


;	Shift last half of the target times array to first half and
;	re-initialize the last half to 'missing'

   Trgt_times( 0:15) = Trgt_times(16:31)
   Trgt_times(16:31) = -1L


;	Shift last half of array to first half and initialize the IPS 
;	target array
   
   IPS_trgt_blk( 0 : IPS_trgt_blk_len/2 -1 ) = $
            IPS_trgt_blk( IPS_trgt_blk_len/2 : IPS_trgt_blk_len -1 ) 
   IPS_trgt_blk( IPS_trgt_blk_len/2 : IPS_trgt_blk_len -1 ) = -1B


;	Shift last half of array to first half and initialize the IES
;	target array
   
   IES_trgt_blk( 0 : IES_trgt_blk_len/2 -1 ) = $
            IES_trgt_blk( IES_trgt_blk_len/2 : IES_trgt_blk_len -1 ) 
   IES_trgt_blk( IES_trgt_blk_len/2 : IES_trgt_blk_len -1 ) = -1B


;	Shift last half of array to first half and initialize the HISTe
;	target array
   
   HISTe_trgt_blk( 0 : HISTe_trgt_blk_len/2 -1 ) = $
            HISTe_trgt_blk( HISTe_trgt_blk_len/2 : HISTe_trgt_blk_len -1 ) 
   HISTe_trgt_blk( HISTe_trgt_blk_len/2 : HISTe_trgt_blk_len -1 ) = -1B


;	Shift last half of array to first half and initialize the HISTp
;	target array
   
   HISTp_trgt_blk( 0 : HISTp_trgt_blk_len/2 -1 ) = $
            HISTp_trgt_blk( HISTp_trgt_blk_len/2 : HISTp_trgt_blk_len -1 ) 
   HISTp_trgt_blk( HISTp_trgt_blk_len/2 : HISTp_trgt_blk_len -1 ) = -1B



   error_return = 0
   return
end




PRO Skip_CEP_TM_tables, error_return, TM_table_unit


;	PURPOSE:	Skip over individual sensor (IPS, IES, HISTe & HISTp)
;			table entries following table header record in order
;			to position Telemetry/Accumulation file pointer at the
;			beginning of the next DPU table

;		Passed Parameters

;	TM_table_unit
;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure


; ----------------------------------------------------------------

;	Make dummy read of the IPS TM table entries

   Read_next_table, error_return, $
		TM_table_unit, num_dum_energies, num_dum_channels, $
		dum_Spins_per_sample, dum_Sectors_per_spin


;	Make dummy read of the IES TM table entries

   Read_next_table, error_return, $
		TM_table_unit, num_dum_energies, num_dum_channels, $
		dum_Spins_per_sample, dum_Sectors_per_spin


;	Make dummy read of the HISTe TM table entries

   Read_next_table, error_return, $
		TM_table_unit, num_dum_energies, num_dum_channels, $
		dum_Spins_per_sample, dum_Sectors_per_spin


;	Make dummy read of the HISTp TM table entries

   Read_next_table, error_return, $
		TM_table_unit, num_dum_energies, num_dum_channels, $
		dum_Spins_per_sample, dum_Sectors_per_spin


   RETURN
END




pro Unpack_ths_spn, error_return

;	Unpack the data for one spin from the major frame record into 
;	spin-header parameters and a data array.  Change filled zeroes 
;	to 255B using minor frame quality flags.

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block
;	04 Apr 1996	M.T.Redding	Added code to utilize the minor frame
;					quality indicator and to put fill values
;					( 255B ) into the spin data array for
;					data comining from questionable quality
;					minor frames

;-----------------------------------------------------------------------

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                        Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

COMMON CEP_two_frm,     Buffer_index, Two_frm_sci_dat, Two_frm_qual

COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                        sci_byt_per_spin, max_spins, $
                        nom_spin_dur, TRUE, FALSE, sync_byte, $
                        CEP_Sci_Man_byt_per_mnfrm, $
                        CEP_Conting_byt_per_mnfrm, $
                        CEP_Sci_Man_rec_len, $
                        CEP_Conting_rec_len

;-----------------------------------------------------------------------

;	Initialize the ths_spn_block to 'missing' value

   Spin_output = bytarr(15,frames_per_spin)	;

   last_minor_frame = ths_spn_loc + frames_per_spin - 1

   mf1 = ths_spn_loc
   mf2 = last_minor_frame < (frames_per_rec-1)
   num_frames_1 = mf2 - mf1 + 1

   qual = TRANSPOSE( REBIN( two_frm_qual( mf1: mf2, ths_spn_buf_no ), $
                            num_frames_1, 15 ) )
   fill = TRANSPOSE( REBIN( REPLICATE( 255B, 15 ), 15, num_frames_1 ) )

   spin_output(0, 0) = ( two_frm_sci_dat( *, mf1: mf2, ths_spn_buf_no ) * $
                       ( qual( *, * ) EQ 0B ) ) + $
                       ( fill( *, * ) * $
                       ( qual( *, * ) NE 0B ) )

   if last_minor_frame ge frames_per_rec then begin

      mf3 = 0
      mf4 = last_minor_frame - frames_per_rec
      num_frames_2 = mf4 + 1


      qual = TRANSPOSE( REBIN( two_frm_qual( mf3: mf4, nxt_spn_buf_no ), $
                               num_frames_2, 15 ) )
      fill = TRANSPOSE( REBIN( REPLICATE( 255B, 15 ), 15, num_frames_2 ) )

      spin_output(0, num_frames_1) = $
                       ( two_frm_sci_dat( *, mf3: mf4, nxt_spn_buf_no ) * $
                       ( qual( *, * ) EQ 0B ) ) + $
                       ( fill( *, * ) * $
                       ( qual( *, * ) NE 0B ) )

   endif

;	Unpack the spin-header values

   Spin_output = REFORM(Spin_output,2400)

   ths_spn_seq_no   = Spin_output(1)
   ths_spn_min_frm  = Spin_output(2)
   ths_spn_format   = Spin_output(3)
   ths_spn_IPS_tab  = Spin_output(4)
   ths_spn_IES_tab  = Spin_output(5)
   ths_spn_IES_com  = Spin_output(6)
   ths_spn_HIST_tab = Spin_output(7)
   ths_spn_HIST_com = Spin_output(8)*256 + Spin_output(9)
   ths_spn_SCOM     = Spin_output(10)
   ths_spn_IPS_Mux  = Spin_output(11)


;	Transfer the spin data

   ths_spn_data = Spin_output(12:2399)

;	Evaluate the start spin time using the major frame sequence
;	number and the minor frame number.

   Eval_time, ths_spn_seq_no, ths_spn_min_frm, ths_spn_time, error_return

   error_return = 0
   Spin_output = 0B	; Deallocate the space used by this array
   qual        = 0B
   fill        = 0B
   return
END




PRO UNPK_CEP_TM,  CEP_TELEMETRY, CEPDRH, CEP_SCIENCE, CEPHK, CEPSC

; Each level zero file has at the beginning of each data record ( CEP_TELEMETRY)
; three-hundred Data Record Header (DRH) bytes described by data structure, 
; CEPDRH.  CEPDRH is filled below.
                   
;	CEP_TELEMETRY
;	CEPDRH
;	CEP_SCIENCE
;	CEPHK
;	CEPSC


;	revision
;	15 Sep 1995	M. T. Redding	Replaced call to funciton LONG_WORD with
;					explicit code to sum products of bytes
;					and powers of 256 to obtain HOST format
;					LONGWORD values from Level-0 
;					'Little-endian' LONGWORDs
; ----------------------------------------------------------------

 common pwr2, p0, p1, p2, p3, p4, p5, p6, p7, p8, p12, p16, p20, p24
 common cep_phys_rec, cep_byts_rec, cep_byts_mnfr
 common pl_phy_rec, mnfrm_rec  

; ----------------------------------------------------------------

cepdrh.inst_num = $	; Bytes 0 - 3
        cep_telemetry(  3)*16777216L + cep_telemetry(  2)*65536L + $
        cep_telemetry(  1)*256L +      cep_telemetry(  0)
cepdrh.rec_num = $	; Bytes 4 - 7
        cep_telemetry(  7)*16777216L + cep_telemetry(  6)*65536L + $
        cep_telemetry(  5)*256L +      cep_telemetry(  4)

cepdrh.mfrm_cnt = $	; Bytes 8 - 11 
        cep_telemetry( 11)*16777216L + cep_telemetry( 10)*65536L + $
        cep_telemetry(  9)*256L +      cep_telemetry(  8)

;     Spacecraft clock bytes 12, 13, 14, 15, 16, 17, 18, 19  (integer*8)
;                    map to  19, 18, 17, 16, 15, 14, 13, 12
;  All of byte 12 and the 6 most significant bits (msb) of byte 13
cepdrh.spc_clk(0) = (long (cep_telemetry(19)) * p6)  + $        ; Bytes 12 - 13
                         (long (cep_telemetry(18)) /p2 )        ;   (14 bits)

;The 2 least significant bits (lsb) of byte 13, all of byte 14, 7msb of byte 15 
lsb_2 = long(cep_telemetry(18)) -  (long(cep_telemetry(18))/p2) * p2
cepdrh.spc_clk(1) = ( lsb_2 * p12 * p3)              + $        ; Bytes 13 - 15
                    (long (cep_telemetry(17)) * p7)  + $        ;   (17bits)
                    (long (cep_telemetry(16)) / p1) 

; The lsb of byte 15, all of byte 16 and msb of byte 17
lsb_1 = long(cep_telemetry(16)) - (long(cep_telemetry(16))/p1) * p1
msb_1 = long(cep_telemetry(14))/p7 
cepdrh.spc_clk(2)= (lsb_1 * p8 * p1)                + $         ; Bytes 15 - 17
                   (long (cep_telemetry(15)) * p1)  + $         ;   (10bits)
                    msb_1

; The 7 lsb of byte 17 and 3 msb of byte 18
lsb_7 = long(cep_telemetry(14)) - (long(cep_telemetry(14))/p7) * p7 
msb_3 = long(cep_telemetry(13))/p5
cepdrh.spc_clk(3)= (lsb_7 * p3) + msb_3                    ; Bytes 17 - 18
                                                                ;   (10bits)

;   The 5 lsb of byte 18
cepdrh.spc_clk(4)= long(cep_telemetry(13)) - $                  ; Byte 18
                   (long(cep_telemetry(13))/p5) * p5            ;   (5 bits)

cepdrh.spc_clk(5)= long(cep_telemetry(12))                      ; Byte 19

cepdrh.atc_yr = $	; Bytes 20 - 23
        cep_telemetry( 23)*16777216L + cep_telemetry( 22)*65536L + $
        cep_telemetry( 21)*256L +      cep_telemetry( 20)

cepdrh.atc_day  = $	; Bytes 24 - 27
        cep_telemetry( 27)*16777216L + cep_telemetry( 26)*65536L + $
        cep_telemetry( 25)*256L +      cep_telemetry( 24)

cepdrh.atc_mils = $	; Bytes 28 - 31
        cep_telemetry( 31)*16777216L + cep_telemetry( 30)*65536L + $
        cep_telemetry( 29)*256L +      cep_telemetry( 28)

cepdrh.atc_micro = $	; Bytes 32 - 35
        cep_telemetry( 35)*16777216L + cep_telemetry( 34)*65536L + $
        cep_telemetry( 33)*256L +      cep_telemetry( 32)

cepdrh.mnfr_fil = $	; Bytes 36 - 39
        cep_telemetry( 39)*16777216L + cep_telemetry( 38)*65536L + $
        cep_telemetry( 37)*256L +      cep_telemetry( 36)

cepdrh.mnfr_syn = $	; Bytes 40 - 43
        cep_telemetry( 43)*16777216L + cep_telemetry( 42)*65536L + $
        cep_telemetry( 41)*256L +      cep_telemetry( 40)

cepdrh.tm_mode = $	; Bytes 44 - 47
        cep_telemetry( 47)*16777216L + cep_telemetry( 46)*65536L + $
        cep_telemetry( 45)*256L +      cep_telemetry( 44)

cepdrh.mnfr_qua = cep_telemetry(48:297)                         ; Bytes 48 - 297

;     Fill the CEPPAD science array, cep_science,  wrds 3 - 17 from each 
;     minor frame in the level0 file 

ct = byte( cep_telemetry(300:*),0, cep_byts_mnfr, mnfrm_rec)
ct = transpose(ct)
cep_science = transpose( ct(*, 3:17))

;	Flag bad quality data with 255B
qual = cepdrh.mnfr_qua 
q = byte( qual( lindgen(mnfrm_rec*15)/15), 0, 15, mnfrm_rec)
cep_science = cep_science * (q eq 0B) + 255B * (q ne 0B)

;     Fill the CEPPAD housekeeping arrays cephk(250,2) representing words 11 and
;     12 from each minor frame of the spacecraft instruments telemetry stream.
;     In CEPPADs Level0 files they are words 0 and 1 of each minor frame.

cephk = ct(*, 0:1)

;     Fill the CEPPAD subcom array, cepsc, with wrd 2 from each minor frame 
;     in the level0 file 

cepsc = ct(*,2)

return
end




PRO Update_HISTe_L1_header, error_return

;	Revise the HISTe Level-1 file header record with the current values for
;	record-count and last-time.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed update of L1 file

;	revision
;	20 Sep 1995	M. T. Redding	Created from code previously located 
;					in procedure 'Appand_L1'
;	05 Mar 1996	M. T. Redding	Added check of HISTe_L1_record with 
;					message and RETURN if it is LE 1 - in
;					this case no update of header record
;					is performed
;--------------------------------------------------------------------

   COMMON CEP_Level_1, $
		L1_number, L1_year, L1_day, L1_begt, L1_endt, $
		L1_DPU_TM_table, L1_TM_CRC, $
	Use_IPS_L1, IPS_L1_unit, IPS_L1_record, IPS_L1_assoc, $
		IPS_L1_rec_len, IPS_L1_phyrcl, $
	Use_IES_L1, IES_L1_unit, IES_L1_record, IES_L1_assoc, $
		IES_L1_rec_len, IES_L1_phyrcl, $
	Use_HISTe_L1, HISTe_L1_unit, HISTe_L1_record, HISTe_L1_assoc, $
		HISTe_L1_rec_len, HISTe_L1_phyrcl, $
	Use_HISTp_L1, HISTp_L1_unit, HISTp_L1_record, HISTp_L1_assoc, $
		HISTp_L1_rec_len, HISTp_L1_phyrcl

   COMMON CEP_LEVEL_1_net, $
           net_L1_DPU_TM_table, net_L1_TM_CRC, $

           net_IPS_L1_year, net_IPS_L1_day, $
           net_IPS_L1_begt, net_IPS_L1_endt, $
           net_IPS_L1_record, net_IPS_L1_rec_len, $
           net_IPS_NRM_filtyp, $
           net_num_IPS_energies, net_num_IPS_channels,$
           net_IPS_spins_per_sample, net_IPS_sectors_per_spin, $

           net_IES_L1_year, net_IES_L1_day, $
           net_IES_L1_begt, net_IES_L1_endt, $
           net_IES_L1_record, net_IES_L1_rec_len, $
           net_IES_NRM_filtyp, $
           net_num_IES_energies, net_num_IES_channels,$
           net_IES_spins_per_sample, net_IES_sectors_per_spin, $

           net_HISTe_L1_year, net_HISTe_L1_day, $
           net_HISTe_L1_begt, net_HISTe_L1_endt, $
           net_HISTe_L1_record, net_HISTe_L1_rec_len, $
           net_HISTe_NRM_filtyp, $
           net_num_HISTe_energies, net_num_HISTe_channels,$
           net_HISTe_spins_per_sample, net_HISTe_sectors_per_spin, $

           net_HISTp_L1_year, net_HISTp_L1_day, $
           net_HISTp_L1_begt, net_HISTp_L1_endt, $
           net_HISTp_L1_record, net_HISTp_L1_rec_len, $
           net_HISTp_NRM_filtyp, $
           net_num_HISTp_energies, net_num_HISTp_channels,$
           net_HISTp_spins_per_sample, net_HISTp_sectors_per_spin

         
;-----------------------------------------------------------------------


   idum = 0L	; Longword used in dummy read of heading records
   error_return = 0

   IF ( HISTe_L1_record LE 1 ) THEN BEGIN
      PRINT, ' NO RECORDS HAVE BEEN WRITTEN TO HISTe LEVEL 1 FILE '
      RETURN
   ENDIF

   ON_IOERROR, Failed_HISTe_L1_update

   POINT_LUN, HISTe_L1_unit, 0	; Reset file pointer to beginning
   WRITEU, HISTe_L1_unit, $		; Rewrite with updated record pointer
           net_HISTe_L1_record, net_HISTe_L1_rec_len, net_HISTe_Nrm_filtyp, $
           net_HISTe_L1_year, net_HISTe_L1_day, $
           net_HISTe_L1_begt, net_HISTe_L1_endt, $
           net_L1_DPU_TM_table, net_L1_TM_CRC, $
           net_num_HISTe_energies, net_num_HISTe_channels, $
           net_HISTe_spins_per_sample, net_HISTe_sectors_per_spin
   RETURN

Failed_HISTe_L1_update:
   PRINT, '*** Append_L1 ***'
   PRINT, 'Error #',!err,' encountered during HISTe Level-1 file update header'
   PRINT, !error
   error_return = error_return + 1
   RETURN

END





PRO Update_HISTp_L1_header, error_return

;	Revise the HISTp Level-1 file header record with the current values for
;	record-count and last-time.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed update of L1 file

;	revision
;	20 Sep 1995	M. T. Redding	Created from code previously located 
;					in procedure 'Appand_L1'
;	05 Mar 1996	M. T. Redding	Added check of HISTp_L1_record with 
;					message and RETURN if it is LE 1 - in
;					this case no update of header record
;					is performed
;--------------------------------------------------------------------

   COMMON CEP_Level_1, $
		L1_number, L1_year, L1_day, L1_begt, L1_endt, $
		L1_DPU_TM_table, L1_TM_CRC, $
	Use_IPS_L1, IPS_L1_unit, IPS_L1_record, IPS_L1_assoc, $
		IPS_L1_rec_len, IPS_L1_phyrcl, $
	Use_IES_L1, IES_L1_unit, IES_L1_record, IES_L1_assoc, $
		IES_L1_rec_len, IES_L1_phyrcl, $
	Use_HISTe_L1, HISTe_L1_unit, HISTe_L1_record, HISTe_L1_assoc, $
		HISTe_L1_rec_len, HISTe_L1_phyrcl, $
	Use_HISTp_L1, HISTp_L1_unit, HISTp_L1_record, HISTp_L1_assoc, $
		HISTp_L1_rec_len, HISTp_L1_phyrcl

   COMMON CEP_LEVEL_1_net, $
           net_L1_DPU_TM_table, net_L1_TM_CRC, $

           net_IPS_L1_year, net_IPS_L1_day, $
           net_IPS_L1_begt, net_IPS_L1_endt, $
           net_IPS_L1_record, net_IPS_L1_rec_len, $
           net_IPS_NRM_filtyp, $
           net_num_IPS_energies, net_num_IPS_channels,$
           net_IPS_spins_per_sample, net_IPS_sectors_per_spin, $

           net_IES_L1_year, net_IES_L1_day, $
           net_IES_L1_begt, net_IES_L1_endt, $
           net_IES_L1_record, net_IES_L1_rec_len, $
           net_IES_NRM_filtyp, $
           net_num_IES_energies, net_num_IES_channels,$
           net_IES_spins_per_sample, net_IES_sectors_per_spin, $

           net_HISTe_L1_year, net_HISTe_L1_day, $
           net_HISTe_L1_begt, net_HISTe_L1_endt, $
           net_HISTe_L1_record, net_HISTe_L1_rec_len, $
           net_HISTe_NRM_filtyp, $
           net_num_HISTe_energies, net_num_HISTe_channels,$
           net_HISTe_spins_per_sample, net_HISTe_sectors_per_spin, $

           net_HISTp_L1_year, net_HISTp_L1_day, $
           net_HISTp_L1_begt, net_HISTp_L1_endt, $
           net_HISTp_L1_record, net_HISTp_L1_rec_len, $
           net_HISTp_NRM_filtyp, $
           net_num_HISTp_energies, net_num_HISTp_channels,$
           net_HISTp_spins_per_sample, net_HISTp_sectors_per_spin

         
;-----------------------------------------------------------------------


   idum = 0L	; Longword used in dummy read of heading records
   error_return = 0

   IF ( HISTp_L1_record LE 1 ) THEN BEGIN
      PRINT, ' NO RECORDS HAVE BEEN WRITTEN TO HISTp LEVEL 1 FILE '
      RETURN
   ENDIF

   ON_IOERROR, Failed_HISTp_L1_update
   POINT_LUN, HISTp_L1_unit, 0	; Reset file pointer to beginning
   WRITEU, HISTp_L1_unit, $		; Rewrite with updated record pointer
           net_HISTp_L1_record, net_HISTp_L1_rec_len, net_HISTp_Nrm_filtyp, $
           net_HISTp_L1_year, net_HISTp_L1_day, $
           net_HISTp_L1_begt, net_HISTp_L1_endt, $
           net_L1_DPU_TM_table, net_L1_TM_CRC, $
           net_num_HISTp_energies, net_num_HISTp_channels, $
           net_HISTp_spins_per_sample, net_HISTp_sectors_per_spin
   RETURN

Failed_HISTp_L1_update:
   PRINT, '*** Append_L1 ***'
   PRINT, 'Error #',!err,' encountered during HISTe Level-1 file header update'
   PRINT, !error
   error_return = error_return + 1
   RETURN

END





PRO Update_HIST_Rates_header, error_return

;	Revised the heading record with the current number of records and
;	last spin time values.
;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed update of header record

;	revision
;	20 Sep 1995	M. T. Redding	Created with code previously located in
;					Append_Rates.
;	05 Mar 1996	M. T. Redding	Added check of HIST_Rates_rec with 
;					message and RETURN if it is LE 1 - in
;					this case no update of header record
;					is performed
;--------------------------------------------------------------------

   COMMON CEP_Rates_Lvl_1, IPS_Rates_unit, IPS_Rates_rec, $
                           IPS_Rates_assoc, IPS_Rates_recl, $
                           IPS_Rates_format, $
                           IES_Rates_unit, IES_Rates_rec, $
                           IES_Rates_assoc, IES_Rates_recl, $
                           IES_Rates_format, $
                           HIST_Rates_unit, HIST_Rates_rec, $
                           HIST_Rates_assoc, HIST_Rates_recl, $
                           HIST_Rates_format

   COMMON CEP_Rates_Lvl_net, $
                           net_IPS_Rates_rec, $
                           net_IPS_Rates_recl, $
                           net_IPS_Rates_format, $
                           net_IPS_Rts_filtyp, $
                           net_IPS_Rts_L1_year, $
                           net_IPS_Rts_L1_day, $
                           net_IPS_Rts_L1_begt, $
                           net_IPS_Rts_L1_endt, $

                           net_IES_Rates_rec, $
                           net_IES_Rates_recl, $
                           net_IES_Rates_format, $
                           net_IES_Rts_filtyp,$
                           net_IES_Rts_L1_year, $
                           net_IES_Rts_L1_day, $
                           net_IES_Rts_L1_begt, $
                           net_IES_Rts_L1_endt, $

                           net_HIST_Rates_rec, $
                           net_HIST_Rates_recl, $
                           net_HIST_Rates_format, $
                           net_HIST_Rts_filtyp, $
                           net_HIST_Rts_L1_year, $
                           net_HIST_Rts_L1_day, $
                           net_HIST_Rts_L1_begt, $
                           net_HIST_Rts_L1_endt

;-----------------------------------------------------------------------


   idum = 0L	; Longword used in dummy read of heading records
   error_return = 0

   IF ( HIST_Rates_rec LE 1 ) THEN BEGIN
      PRINT, ' NO RECORDS HAVE BEEN WRITTEN TO HIST RATES FILE '
      RETURN
   ENDIF

   ON_IOERROR, Failed_HIST_Rates_Update

   POINT_LUN, HIST_Rates_unit, 0	; Reset file pointer to beginning
   WRITEU, HIST_Rates_unit, $	; Rewrite with updated record pointer and 
				; last-time value
           net_HIST_Rates_rec, net_HIST_Rates_recl, net_HIST_Rts_filtyp, $
           net_HIST_Rts_L1_year, net_HIST_Rts_L1_day, net_HIST_Rts_L1_begt, $
           net_HIST_Rts_L1_endt
   RETURN

Failed_HIST_Rates_Update:
      print,'*** Update_HIST_Rate_Header ***"
      print,'Error #',!err, $
               ' encountered during HIST Rates-mode Level-1 file header update'
      print,!error
      error_return = 1
      RETURN

END




PRO Update_HK_Header, error_return

;	Revise the Housekeeping file header record with the current record
;	count and last-time values

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed update

;	revision
;	20 Sep 1995	M. T. Redding	Created from code extracted from 
;					procedure 'Append_HK' 
;	05 Mar 1996	M. T. Redding	Added check of CEP_HK_rec with 
;					message and RETURN if it is LE 1 - in
;					this case no update of header record
;					is performed
;--------------------------------------------------------------------

   COMMON CEP_HK,          CEP_HK_unit, CEP_HK_rec, CEP_HK_rec_len, CEP_HK_data

   COMMON CEP_HK_net,      net_CEP_HK_rec_len,     net_CEP_HK_filtyp, $
                           net_CEP_HK_rec,         net_CEP_HK_beg_year, $
                           net_CEP_HK_beg_day,     net_CEP_HK_beg_tim_msec, $
                           net_CEP_HK_end_tim_msec
 
;-----------------------------------------------------------------------

   error_return = 0
   dummy = 0L	; Define Dummy as a Longword integer for masked read

   IF ( CEP_HK_rec LE 1 ) THEN BEGIN
      PRINT, ' NO RECORDS HAVE BEEN WRITTEN TO Housekeeping FILE '
      RETURN
   ENDIF

   ON_IOERROR, Failed_Update_HK_header

;	Write the heading record with the current values for end year, day, 
;	and time

   POINT_LUN, CEP_HK_unit, 0	; Reset file pointer to beginning

   WRITEU, CEP_HK_unit, $
           net_CEP_HK_rec, net_CEP_HK_rec_len, net_CEP_HK_filtyp, $
           net_CEP_HK_Beg_year, net_CEP_HK_beg_day, net_CEP_HK_beg_tim_msec, $
           net_CEP_HK_End_tim_msec
   RETURN

Failed_Update_HK_header:
   print,'*** Append_HK ***"
   print,'Error #',!err,' encountered during CEPPAD HK Level-1 file header update'
   print,!error
   error_return = 0
   RETURN

END





PRO Update_IES_L1_header, error_return

;	Revise the IES Level-1 file header record with the current values for
;	record-count and last-time.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed update of L1 file

;	revision
;	20 Sep 1995	M. T. Redding	Created from code previously located 
;					in procedure 'Appand_L1'
;	05 Mar 1996	M. T. Redding	Added check of IES_L1_record with 
;					message and RETURN if it is LE 1 - in
;					this case no update of header record
;					is performed
;--------------------------------------------------------------------

   COMMON CEP_Level_1, $
		L1_number, L1_year, L1_day, L1_begt, L1_endt, $
		L1_DPU_TM_table, L1_TM_CRC, $
	Use_IPS_L1, IPS_L1_unit, IPS_L1_record, IPS_L1_assoc, $
		IPS_L1_rec_len, IPS_L1_phyrcl, $
	Use_IES_L1, IES_L1_unit, IES_L1_record, IES_L1_assoc, $
		IES_L1_rec_len, IES_L1_phyrcl, $
	Use_HISTe_L1, HISTe_L1_unit, HISTe_L1_record, HISTe_L1_assoc, $
		HISTe_L1_rec_len, HISTe_L1_phyrcl, $
	Use_HISTp_L1, HISTp_L1_unit, HISTp_L1_record, HISTp_L1_assoc, $
		HISTp_L1_rec_len, HISTp_L1_phyrcl

   COMMON CEP_LEVEL_1_net, $
           net_L1_DPU_TM_table, net_L1_TM_CRC, $

           net_IPS_L1_year, net_IPS_L1_day, $
           net_IPS_L1_begt, net_IPS_L1_endt, $
           net_IPS_L1_record, net_IPS_L1_rec_len, $
           net_IPS_NRM_filtyp, $
           net_num_IPS_energies, net_num_IPS_channels,$
           net_IPS_spins_per_sample, net_IPS_sectors_per_spin, $

           net_IES_L1_year, net_IES_L1_day, $
           net_IES_L1_begt, net_IES_L1_endt, $
           net_IES_L1_record, net_IES_L1_rec_len, $
           net_IES_NRM_filtyp, $
           net_num_IES_energies, net_num_IES_channels,$
           net_IES_spins_per_sample, net_IES_sectors_per_spin, $

           net_HISTe_L1_year, net_HISTe_L1_day, $
           net_HISTe_L1_begt, net_HISTe_L1_endt, $
           net_HISTe_L1_record, net_HISTe_L1_rec_len, $
           net_HISTe_NRM_filtyp, $
           net_num_HISTe_energies, net_num_HISTe_channels,$
           net_HISTe_spins_per_sample, net_HISTe_sectors_per_spin, $

           net_HISTp_L1_year, net_HISTp_L1_day, $
           net_HISTp_L1_begt, net_HISTp_L1_endt, $
           net_HISTp_L1_record, net_HISTp_L1_rec_len, $
           net_HISTp_NRM_filtyp, $
           net_num_HISTp_energies, net_num_HISTp_channels,$
           net_HISTp_spins_per_sample, net_HISTp_sectors_per_spin

         
;-----------------------------------------------------------------------


   idum = 0L	; Longword used in dummy read of heading records
   error_return = 0

   IF ( IES_L1_record LE 1 ) THEN BEGIN
      PRINT, ' NO RECORDS HAVE BEEN WRITTEN TO IES LEVEL 1 FILE '
      RETURN
   ENDIF

   ON_IOERROR, Failed_IES_L1_update
   POINT_LUN, IES_L1_unit, 0	; Reset file pointer to beginning
   WRITEU, IES_L1_unit, $		; Rewrite with updated record pointer
           net_IES_L1_record, net_IES_L1_rec_len, net_IES_Nrm_filtyp, $
           net_IES_L1_year, net_IES_L1_day, $
           net_IES_L1_begt, net_IES_L1_endt, $
           net_L1_DPU_TM_table, net_L1_TM_CRC, $
           net_num_IES_energies, net_num_IES_channels, $
           net_IES_spins_per_sample, net_IES_sectors_per_spin
   RETURN

Failed_IES_L1_update:
   PRINT, '*** Append_L1 ***'
   PRINT, 'Error #',!err,' encountered during IES Level-1 file header update'
   PRINT, !error
   error_return = error_return + 1
   RETURN

END





PRO Update_IES_Rates_header, error_return

;	Revised the heading record with the current number of records and
;	last spin time values.
;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed update of header record

;	revision
;	20 Sep 1995	M. T. Redding	Created with code previously located in
;					Append_Rates.
;	05 Mar 1996	M. T. Redding	Added check of IES_Rates_rec with 
;					message and RETURN if it is LE 1 - in
;					this case no update of header record
;					is performed
;--------------------------------------------------------------------

   COMMON CEP_Rates_Lvl_1, IPS_Rates_unit, IPS_Rates_rec, $
                           IPS_Rates_assoc, IPS_Rates_recl, $
                           IPS_Rates_format, $
                           IES_Rates_unit, IES_Rates_rec, $
                           IES_Rates_assoc, IES_Rates_recl, $
                           IES_Rates_format, $
                           HIST_Rates_unit, HIST_Rates_rec, $
                           HIST_Rates_assoc, HIST_Rates_recl, $
                           HIST_Rates_format

   COMMON CEP_Rates_Lvl_net, $
                           net_IPS_Rates_rec, $
                           net_IPS_Rates_recl, $
                           net_IPS_Rates_format, $
                           net_IPS_Rts_filtyp, $
                           net_IPS_Rts_L1_year, $
                           net_IPS_Rts_L1_day, $
                           net_IPS_Rts_L1_begt, $
                           net_IPS_Rts_L1_endt, $

                           net_IES_Rates_rec, $
                           net_IES_Rates_recl, $
                           net_IES_Rates_format, $
                           net_IES_Rts_filtyp,$
                           net_IES_Rts_L1_year, $
                           net_IES_Rts_L1_day, $
                           net_IES_Rts_L1_begt, $
                           net_IES_Rts_L1_endt, $

                           net_HIST_Rates_rec, $
                           net_HIST_Rates_recl, $
                           net_HIST_Rates_format, $
                           net_HIST_Rts_filtyp, $
                           net_HIST_Rts_L1_year, $
                           net_HIST_Rts_L1_day, $
                           net_HIST_Rts_L1_begt, $
                           net_HIST_Rts_L1_endt

;-----------------------------------------------------------------------


   idum = 0L	; Longword used in dummy read of heading records
   error_return = 0

   IF ( IES_Rates_rec LE 1 ) THEN BEGIN
      PRINT, ' NO RECORDS HAVE BEEN WRITTEN TO IES RATES FILE '
      RETURN
   ENDIF

   ON_IOERROR, Failed_IES_Rates_Update

   POINT_LUN, IES_Rates_unit, 0	; Reset file pointer to beginning
   WRITEU, IES_Rates_unit, $	; Rewrite with updated record pointer and 
				; last-time value
           net_IES_Rates_rec, net_IES_Rates_recl, net_IES_Rts_filtyp, $
           net_IES_Rts_L1_year, net_IES_Rts_L1_day, net_IES_Rts_L1_begt, $
           net_IES_Rts_L1_endt
   RETURN

Failed_IES_Rates_Update:
      print,'*** Update_IES_Rate_Header ***"
      print,'Error #',!err, $
               ' encountered during IES Rates-mode Level-1 file header update'
      print,!error
      error_return = 1
      RETURN

END




PRO Update_IPS_L1_header, error_return

;	Revise the IPS Level-1 file header record with the current values for
;	record-count and last-time.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed update of L1 file

;	revision
;	20 Sep 1995	M. T. Redding	Created from code previously located 
;					in procedure 'Appand_L1'
;	05 Mar 1996	M. T. Redding	Added check of IPS_L1_record with 
;					message and RETURN if it is LE 1 - in
;					this case no update of header record
;					is performed
;--------------------------------------------------------------------

   COMMON CEP_Level_1, $
		L1_number, L1_year, L1_day, L1_begt, L1_endt, $
		L1_DPU_TM_table, L1_TM_CRC, $
	Use_IPS_L1, IPS_L1_unit, IPS_L1_record, IPS_L1_assoc, $
		IPS_L1_rec_len, IPS_L1_phyrcl, $
	Use_IES_L1, IES_L1_unit, IES_L1_record, IES_L1_assoc, $
		IES_L1_rec_len, IES_L1_phyrcl, $
	Use_HISTe_L1, HISTe_L1_unit, HISTe_L1_record, HISTe_L1_assoc, $
		HISTe_L1_rec_len, HISTe_L1_phyrcl, $
	Use_HISTp_L1, HISTp_L1_unit, HISTp_L1_record, HISTp_L1_assoc, $
		HISTp_L1_rec_len, HISTp_L1_phyrcl

   COMMON CEP_LEVEL_1_net, $
           net_L1_DPU_TM_table, net_L1_TM_CRC, $

           net_IPS_L1_year, net_IPS_L1_day, $
           net_IPS_L1_begt, net_IPS_L1_endt, $
           net_IPS_L1_record, net_IPS_L1_rec_len, $
           net_IPS_NRM_filtyp, $
           net_num_IPS_energies, net_num_IPS_channels,$
           net_IPS_spins_per_sample, net_IPS_sectors_per_spin, $

           net_IES_L1_year, net_IES_L1_day, $
           net_IES_L1_begt, net_IES_L1_endt, $
           net_IES_L1_record, net_IES_L1_rec_len, $
           net_IES_NRM_filtyp, $
           net_num_IES_energies, net_num_IES_channels,$
           net_IES_spins_per_sample, net_IES_sectors_per_spin, $

           net_HISTe_L1_year, net_HISTe_L1_day, $
           net_HISTe_L1_begt, net_HISTe_L1_endt, $
           net_HISTe_L1_record, net_HISTe_L1_rec_len, $
           net_HISTe_NRM_filtyp, $
           net_num_HISTe_energies, net_num_HISTe_channels,$
           net_HISTe_spins_per_sample, net_HISTe_sectors_per_spin, $

           net_HISTp_L1_year, net_HISTp_L1_day, $
           net_HISTp_L1_begt, net_HISTp_L1_endt, $
           net_HISTp_L1_record, net_HISTp_L1_rec_len, $
           net_HISTp_NRM_filtyp, $
           net_num_HISTp_energies, net_num_HISTp_channels,$
           net_HISTp_spins_per_sample, net_HISTp_sectors_per_spin

         
;-----------------------------------------------------------------------


   idum = 0L	; Longword used in dummy read of heading records
   error_return = 0

   IF ( IPS_L1_record LE 1 ) THEN BEGIN
      PRINT, ' NO RECORDS HAVE BEEN WRITTEN TO IPS LEVEL 1 FILE '
      RETURN
   ENDIF

   ON_IOERROR, Failed_IPS_L1_update
   POINT_LUN, IPS_L1_unit, 0	; Reset file pointer to beginning
   WRITEU, IPS_L1_unit, $		; Rewrite with updated record pointer
           net_IPS_L1_record, net_IPS_L1_rec_len, net_IPS_Nrm_filtyp, $
           net_IPS_L1_year, net_IPS_L1_day, $
           net_IPS_L1_begt, net_IPS_L1_endt, $
           net_L1_DPU_TM_table, net_L1_TM_CRC, $
           net_num_IPS_energies, net_num_IPS_channels, $
           net_IPS_spins_per_sample, net_IPS_sectors_per_spin
   RETURN

Failed_IPS_L1_update:
   PRINT, '*** Append_L1 ***'
   PRINT, 'Error #',!err,' encountered during IPS Level-1 file header update'
   PRINT, !error
   error_return = error_return + 1
   RETURN

END





PRO Update_IPS_Rates_header, error_return

;	Revised the heading record with the current number of records and
;	last spin time values.
;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed update of header record

;	revision
;	20 Sep 1995	M. T. Redding	Created with code previously located in
;					Append_Rates.
;	05 Mar 1996	M. T. Redding	Added check of IPS_Rates_rec with 
;					message and RETURN if it is LE 1 - in
;					this case no update of header record
;					is performed
;--------------------------------------------------------------------

   COMMON CEP_Rates_Lvl_1, IPS_Rates_unit, IPS_Rates_rec, $
                           IPS_Rates_assoc, IPS_Rates_recl, $
                           IPS_Rates_format, $
                           IES_Rates_unit, IES_Rates_rec, $
                           IES_Rates_assoc, IES_Rates_recl, $
                           IES_Rates_format, $
                           HIST_Rates_unit, HIST_Rates_rec, $
                           HIST_Rates_assoc, HIST_Rates_recl, $
                           HIST_Rates_format

   COMMON CEP_Rates_Lvl_net, $
                           net_IPS_Rates_rec, $
                           net_IPS_Rates_recl, $
                           net_IPS_Rates_format, $
                           net_IPS_Rts_filtyp, $
                           net_IPS_Rts_L1_year, $
                           net_IPS_Rts_L1_day, $
                           net_IPS_Rts_L1_begt, $
                           net_IPS_Rts_L1_endt, $

                           net_IES_Rates_rec, $
                           net_IES_Rates_recl, $
                           net_IES_Rates_format, $
                           net_IES_Rts_filtyp,$
                           net_IES_Rts_L1_year, $
                           net_IES_Rts_L1_day, $
                           net_IES_Rts_L1_begt, $
                           net_IES_Rts_L1_endt, $

                           net_HIST_Rates_rec, $
                           net_HIST_Rates_recl, $
                           net_HIST_Rates_format, $
                           net_HIST_Rts_filtyp, $
                           net_HIST_Rts_L1_year, $
                           net_HIST_Rts_L1_day, $
                           net_HIST_Rts_L1_begt, $
                           net_HIST_Rts_L1_endt

;-----------------------------------------------------------------------


   idum = 0L	; Longword used in dummy read of heading records
   error_return = 0

   IF ( IPS_Rates_rec LE 1 ) THEN BEGIN
      PRINT, ' NO RECORDS HAVE BEEN WRITTEN TO IPS RATES FILE '
      RETURN
   ENDIF

   ON_IOERROR, Failed_IPS_Rates_Update

   POINT_LUN, IPS_Rates_unit, 0	; Reset file pointer to beginning
   WRITEU, IPS_Rates_unit, $	; Rewrite with updated record pointer and 
				; last-time value
           net_IPS_Rates_rec, net_IPS_Rates_recl, net_IPS_Rts_filtyp, $
           net_IPS_Rts_L1_year, net_IPS_Rts_L1_day, net_IPS_Rts_L1_begt, $
           net_IPS_Rts_L1_endt
   RETURN

Failed_IPS_Rates_Update:
      print,'*** Update_IPS_Rate_Header ***"
      print,'Error #',!err, $
               ' encountered during IPS Rates-mode Level-1 file header update'
      print,!error
      error_return = 1
      RETURN

END




PRO Update_L1_header, error_return

;	Revise each currently opened IPS, IES HISTe and HISTp Level_1 data file
;	heading record with the current record count and last-time values 

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1-4	Number of failed closings of
;						IPS, IES, HISTe and HISTp
;						Normal mode Level-1 files

;	revision:
;	22 Sep 1995	M.T.Redding

;--------------------------------------------------------------------

   COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                           sci_byt_per_spin, max_spins, $
                           nom_spin_dur, TRUE, FALSE, sync_byte, $
                           CEP_Sci_Man_byt_per_mnfrm, $
                           CEP_Conting_byt_per_mnfrm, $
                           CEP_Sci_Man_rec_len, $
                           CEP_Conting_rec_len

   COMMON CEP_Level_1, $
		L1_number, L1_year, L1_day, L1_begt, L1_endt, $
		L1_DPU_TM_table, L1_TM_CRC, $
	Use_IPS_L1, IPS_L1_unit, IPS_L1_record, IPS_L1_assoc, $
		IPS_L1_rec_len, IPS_L1_phyrcl, $
	Use_IES_L1, IES_L1_unit, IES_L1_record, IES_L1_assoc, $
		IES_L1_rec_len, IES_L1_phyrcl, $
	Use_HISTe_L1, HISTe_L1_unit, HISTe_L1_record, HISTe_L1_assoc, $
		HISTe_L1_rec_len, HISTe_L1_phyrcl, $
	Use_HISTp_L1, HISTp_L1_unit, HISTp_L1_record, HISTp_L1_assoc, $
		HISTp_L1_rec_len, HISTp_L1_phyrcl

;-----------------------------------------------------------------------

   error_return = 0
   IF ( Use_IPS_L1 ) and (IPS_L1_unit gt 0) THEN BEGIN
      Update_IPS_L1_header, error_return
   ENDIF


   IF ( Use_IES_L1 ) and ( IES_L1_unit ge 0 ) THEN BEGIN
      Update_IES_L1_header, error_return
   ENDIF


   IF ( Use_HISTe_L1 ) and ( HISTe_L1_unit gt 0 ) THEN BEGIN
      Update_HISTe_L1_header, error_return
   ENDIF


   IF ( Use_HISTp_L1 ) and ( HISTp_L1_unit gt 0 ) THEN BEGIN
      Update_HISTp_L1_header, error_return
   ENDIF



   RETURN
END





PRO Update_Rates_headers, error_return

;	Cause all currently open Rates L1 files header records to be updated
;	with the current record count and last-time values

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Failed update of header record

;	revision
;	22 Sep 1995	M. T. Redding
;--------------------------------------------------------------------

   COMMON CEP_Rates_Lvl_1, IPS_Rates_unit, IPS_Rates_rec, $
                           IPS_Rates_assoc, IPS_Rates_recl, $
                           IPS_Rates_format, $
                           IES_Rates_unit, IES_Rates_rec, $
                           IES_Rates_assoc, IES_Rates_recl, $
                           IES_Rates_format, $
                           HIST_Rates_unit, HIST_Rates_rec, $
                           HIST_Rates_assoc, HIST_Rates_recl, $
                           HIST_Rates_format

;-----------------------------------------------------------------------

   IF (IPS_Rates_unit gt 0) THEN BEGIN
      Update_IPS_Rates_header, error_return
   ENDIF

   IF (IES_Rates_unit gt 0) THEN BEGIN
      Update_IES_Rates_header, error_return
   ENDIF

   IF (HIST_Rates_unit gt 0) THEN BEGIN
      Update_HIST_Rates_header, error_return
   ENDIF


   RETURN

END




PRO Update_SH_header, error_return

;	Revise the Spin header Level-1 file heading record with the current
;	values for record-count and last-time.

;		Passed Parameters

;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1 	Failed update

;	revision
;	20 Sep 1995	M. T. Redding	Created from code previousy located in
;					procedure 'Append_SH'
;	05 Mar 1996	M. T. Redding	Added check of SH_record with 
;					message and RETURN if it is LE 3 - in
;					this case no update of header record
;					is performed
;	13 Jun 1996	M. T. Redding	Corrected record positioning call 
;					(POINT_LUN) to use SH_rec_len*2 instead
;					of 24 (which was only appropriate for
;					record length of 12 bytes anyway)

;--------------------------------------------------------------------

   COMMON CEP_Spin_head,   SH_unit, SH_record, SH_assoc, SH_rec_len

   COMMON CEP_Spin_head_net, net_SH_L1_record, net_SH_L1_rec_len, $
                           net_Spin_Head_filtyp, net_SH_L1_year, $
                           net_SH_L1_day, net_SH_L1_begt, $
                           net_SH_L1_endt

;-----------------------------------------------------------------------

   error_return = 0

   IF ( SH_record LE 3 ) THEN BEGIN
      PRINT, ' NO RECORDS HAVE BEEN WRITTEN TO Spin Header (SH) FILE '
      RETURN
   ENDIF

   ON_IOERROR, Failed_SH_header_update

   POINT_LUN, SH_unit, 0	; Reset file pointer to 1st record
   WRITEU, SH_unit, $
           net_SH_L1_record, net_SH_L1_rec_len, $
           net_Spin_Head_filtyp, 0B, 0B
   POINT_LUN, SH_unit, SH_rec_len*2	; Reset file pointer to 3rd record
   WRITEU, SH_unit, $
           net_SH_L1_endt, REPLICATE( 0B, 10 )
   error_return = 0
   RETURN

Failed_SH_header_update:
   print, '*** Update_SH_header ***'
   print, 'Error #', !err, $
          ' encountered during Spin-header Level-1 file header update'
   print, !error
   error_return = error_return + 1
   RETURN
END





PRO Verify_spin_end, end_found, error_return

;	Determine if the current location represents the end of spin data

;	end_found
;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block

;--------------------------------------------------------------------

COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                        sci_byt_per_spin, max_spins, $
                        nom_spin_dur, TRUE, FALSE, sync_byte, $
                        CEP_Sci_Man_byt_per_mnfrm, $
                        CEP_Conting_byt_per_mnfrm, $
                        CEP_Sci_Man_rec_len, $
                        CEP_Conting_rec_len

COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                        Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

COMMON CEP_two_frm,     Buffer_index, Two_frm_sci_dat, Two_frm_qual

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

;-----------------------------------------------------------------------

;	Look for the sync value A5h in the first byte of or ten zero 
;	filled minor frames following the location in question.

   IF (Two_frm_sci_dat(0,nxt_spn_loc, nxt_spn_buf_no) EQ sync_byte)  THEN BEGIN
      end_found = TRUE
   ENDIF ELSE BEGIN	; Look for 10 'zero-filled' minor frames following the
			; end of spin data.  If they are there then interpret
			; that as an end-of-spin indicator


;     Find any non-zero values in the next 10 minor frames

      Result = WHERE(Two_frm_sci_dat(*,nxt_spn_loc:nxt_spn_loc+9, $
                            nxt_spn_buf_no) NE 0, Count)
      IF (Count EQ 0) THEN BEGIN
         end_found = TRUE
      ENDIF ELSE BEGIN
         end_found = FALSE
      ENDELSE
   ENDELSE

   error_return = 0
   return
end




pro Verify_spin_start, start_found, error_return

;	Determine if the current location represents the start of spin data

;	start_found
;	error_return	Integer - error condition indicator passed back to 
;			calling procedure 
;			Interpretation - 0	No error
;					 1	Not set in this procedure

;	revision:
;	02 Aug 95	M.T.Redding	Added 'CEP_Sci_Man_byt_per_mnfrm',
;					'CEP_Conting_byt_per_mnfrm',
;					'CEP_Sci_Man_rec_len' and
;					'CEP_Conting_rec_len' to CEP_constants
;					common block

;--------------------------------------------------------------------

COMMON CEP_constants,   frames_per_rec, frames_per_spin, $
                        sci_byt_per_spin, max_spins, $
                        nom_spin_dur, TRUE, FALSE, sync_byte, $
                        CEP_Sci_Man_byt_per_mnfrm, $
                        CEP_Conting_byt_per_mnfrm, $
                        CEP_Sci_Man_rec_len, $
                        CEP_Conting_rec_len

COMMON CEP_major_frame, DRH, HK, Subcom, Science_data, $
                        Min_frm_quality, MF_ATC_time, MF_CEP_seq_no

COMMON CEP_two_frm,     Buffer_index, Two_frm_sci_dat, Two_frm_qual

   COMMON CEP_ths_spn,     ths_spn_loc, ths_spn_buf_no, $
                           nxt_spn_loc, nxt_spn_buf_no, $
                           ths_spn_seq_no, ths_spn_min_frm, $
                           ths_spn_format, ths_spn_IPS_tab, $
                           ths_spn_IES_tab, ths_spn_IES_com, $
                           ths_spn_HIST_tab, ths_spn_HIST_com, $
                           ths_spn_SCOM, ths_spn_IPS_Mux, $
                           ths_spn_data, ths_spn_time, $
                           ths_spn_CRC, ths_spn_quality

;-----------------------------------------------------------------------

;	Look for the sync value A5h in the first byte of the proposed
;	spin data

   IF Two_frm_sci_dat(0,ths_spn_loc,ths_spn_buf_no) EQ sync_byte THEN BEGIN
      start_found = TRUE
   ENDIF ELSE BEGIN
      start_found = FALSE
   ENDELSE

   error_return = 0
   return
end



