;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_calibrate
;
; Purpose     :	Processes required data.
;
; Explanation : Process data.
;
; Use         : < res = cep_calibrate( full_detector, SECTOR=sector )>
;
; Inputs      : full_detector : INT detector for display.
;
; Opt. Inputs : None.
;
; Outputs     : Updates structures in COMMON block.
;               res : INT flag indicating error.
;
; Opt. Outputs:	None.
;
; Keywords    : RANGE  : overrides data min and max.
;               FORMAT : overrides default format.
;
; Calls       :	ies_calibrate.
;
; Common      :	ies_draw_widget_block.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	From cep survey.
;
; Written     :	Version 0.0, Martin Carter, RAL, 28/11/97
;
; Modified    :    Version 0.1, MKC, 18/2/98
;                    Added HISTe, HISTp processing.
;                    Added RANGE keyword.
;                  Version 0.2, 7/10/98
;                    Added Magnetic field and attitude arrays to common block and to ies extdata
;                    Allowed detector sum.
;                  Version 0.3, 30/10/98
;                    Separated from cep_extract.
;                  Version 0.4, 12/01/99
;                    Changed back to including bad data.
;                  Version 0.5, 22/2/99
;                    Added output_info tags.
;                  Version 0.6, 27/02/02
;                    Changed status around.
;                  Version 0.7, 31/05/02
;                    Added full_band tag to outoput_info.
;                  Version 0.8, 29/09/03
;                    Added npitchs to output_info.
;
; Version     :	Version 0.8, 29/09/03
;-
;**********************************************************

FUNCTION cep_calibrate, full_detector, RANGE=range, DETECTOR_AVERAGE=detector_average, FORMAT=format

  ; define common blocks used internally for draw widgets
  ; full_header   = structure containing original data header information
  ; full_data     = structure array containing original data
  ; header        = structure containing extracted data header information  ** used
  ; data          = structure array containing extracted data               ** used
  ; output_info   = structure containing plotting and processing info       ** used
  ; output_chans  = array containing processed channel positions
  ; output_array  = array containing processed data for plotting
  ; ids           = structure giving widget IDs
  ; dnfull        = co-ordinate conversion array for full window
  ; dnscroll      = co-ordinate conversion array for magnified windows
  ; files         = STRARR giving data file names
  ; ephem         = structure containing ephemeris data
  ; mf            = structure containing magnetic field data
  ; at            = structure containing attitude data

  COMMON ies_draw_widget_block, input_header, input_data, header, data, $
           output_info, output_chans, output_array, ids, dnfull, dnscroll, files, ephem, mf, at

  IF NOT KEYWORD_SET(format) THEN format = 'ENERGY vs DETECTOR'

  ; set up processing info

  output_info = {log:1, cal:1, div:1, rat:1, shift:0, subtract:0, track:1, $
         rng:0.0, npitchs:0, button_labels:STRARR(12), button_values:[1,0,0,0,0,0,0,0,0,0,0,0], procname:'SPECTRAL', $
         full_only:0, full_detector:full_detector, full_sector:0, full_band:0, axes:1, band_labels:STRARR(20), noutputs:0, $
         format:format, ptitle:STRARR(20), time:0.0D0, ysamples:0, $
         minimum:1.e2, maximum:1.e8, elow:0.0, ehigh:0.0, min:1, max:1, elo:0, ehi:0 }

  ; ies_calibrate sets .ptitle :
  ;    'ENERGY vs DETECTOR' : 'Dt' + header.ptitle
  ;    'ENERGY vs SECTOR'   : 'Sc' + STRTRIM(sector_list,2)
  ;    'SECTOR vs DETECTOR' : 'Dt' + header.ptitle
  ;    'DETECTOR vs SECTOR' : 'Sc' + STRTRIM(sector_list,2)
  ;    'PITCH ANGLE vs ENERGY' : 'Energy band '+STRTRIM(INDGEN(header.nbands),2)
  ;
  ; ies_window_plot, if pg_pos set, uses (via ies_do_plot, /FULL)
  ;   output_info.ptitle(element) to label color bar
  ;   where
  ;    'ENERGY vs DETECTOR' : element = output_info.full_detector
  ;    'ENERGY vs SECTOR'   : element  = output_info.full_sector
  ;    'SECTOR vs DETECTOR' : element  = output_info.full_detector
  ;    'DETECTOR vs SECTOR' : element  = 0
  ;    'PITCH ANGLE vs ENERGY' : element  = header.nbands-1

  IF format EQ 'ENERGY vs DETECTOR' THEN BEGIN

    ; do instrument specific stuff

    ; set up processing info

    IF STRPOS(input_header.datatype,'IES') GE 0 THEN BEGIN

      ; set data range

      output_info.minimum = 1.e0
      output_info.maximum = 1.e6

    ENDIF ELSE IF STRPOS(input_header.datatype,'IPS') GE 0 THEN BEGIN

      ; set data range

      output_info.minimum = 1.e-1
      output_info.maximum = 1.e5

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTe') GE 0 THEN BEGIN

      ; set data range

      output_info.minimum = 1.e-3
      output_info.maximum = 1.e2

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTp') GE 0 THEN BEGIN

      ; set data range

      output_info.minimum = 1.e-3
      output_info.maximum = 1.e2

    ENDIF

    ; check range

    IF KEYWORD_SET(range) THEN BEGIN
      output_info.minimum = range(0)
      output_info.maximum = range(1)
    ENDIF

    output_info.button_values[3] = 1

    ; process data

    IF ies_calibrate(/QUIET) THEN RETURN, 0

    IF STRPOS(input_header.datatype,'IES') GE 0 THEN BEGIN

      ; change detector labels to fit

      output_info.ptitle = 'IES ' + header.ptitle
      IF full_detector EQ header.ndetectors THEN output_info.ptitle[full_detector] = 'IES av'

      ; set lowest energy displayed

      output_info.elo = 30.0

    ENDIF ELSE IF STRPOS(input_header.datatype,'IPS') GE 0 THEN BEGIN

      ; change detector labels to fit

      output_info.ptitle = 'IPS ' + header.ptitle
      IF full_detector EQ header.ndetectors THEN output_info.ptitle[full_detector] = 'IPS av'

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTe') GE 0 THEN BEGIN

      ; change detector labels to fit

      output_info.ptitle = 'HISTe (ABC)'

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTp') GE 0 THEN BEGIN

      ; change detector labels to fit

      output_info.ptitle = 'HISTp (ABC)'

    ENDIF

  ENDIF ELSE IF format EQ 'PITCH ANGLE vs ENERGY' THEN BEGIN

    ; check range

    IF KEYWORD_SET(range) THEN BEGIN
      output_info.minimum = range(0)
      output_info.maximum = range(1)
    ENDIF

    ; energy summed so dont want /keV

    output_info.div = 0

    ; select no. of pitch angle bins

    output_info.npitchs = 9

    ; process data

    IF ies_calibrate(/QUIET) THEN RETURN, 0

    ; set color bar title

    IF STRPOS(input_header.datatype,'IES') GE 0 THEN BEGIN

      output_info.ptitle[header.nbands-1] = 'IES'

    ENDIF ELSE IF STRPOS(input_header.datatype,'IPS') GE 0 THEN BEGIN

      output_info.ptitle[header.nbands-1] = 'IPS'

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTe') GE 0 THEN BEGIN

      output_info.ptitle[header.nbands-1] = 'HISTe'

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTp') GE 0 THEN BEGIN

      output_info.ptitle[header.nbands-1] = 'HISTp'

    ENDIF

  ENDIF ELSE IF format EQ 'SECTOR vs DETECTOR' THEN BEGIN

    ; set up processing info for sector plot

    ; check range

    IF KEYWORD_SET(range) THEN BEGIN
      output_info.minimum = range(0)
      output_info.maximum = range(1)
    ENDIF

    ; energy summed so dont want /keV

    output_info.div = 0

    ; process data

    IF ies_calibrate(/QUIET) THEN RETURN, 0

    ; change detector labels to fit

    IF STRPOS(input_header.datatype,'IES') GE 0 THEN BEGIN

      output_info.ptitle[header.nbands-1] = 'IES'

    ENDIF ELSE IF STRPOS(input_header.datatype,'IPS') GE 0 THEN BEGIN

      output_info.ptitle[header.nbands-1] = 'IPS'

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTe') GE 0 THEN BEGIN

      output_info.ptitle[header.nbands-1] = 'HISTe'

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTp') GE 0 THEN BEGIN

      output_info.ptitle[header.nbands-1] = 'HISTp'

    ENDIF

  ENDIF

  RETURN, 1

END
