;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_decomp
;
; Purpose     :	Decompresses BYTE array of CEPPAD level 0 or level 1 data 
;
; Explanation :
;		Data Compression Scheme (19 bits to 8 bits)
;				29-August-1990
;
;
;	    Input bit stream		Exp.	Mant.	  Ex. Time (cycles)
;
;	000 0000 0000 0000 0000		0000	0000		380
;	000 0000 0000 0000 0001		0000	0001		380
;	000 0000 0000 0000 001a		0000	001a		380
;	000 0000 0000 0000 01ab		0000	01ab		380
;	000 0000 0000 0000 1abc		0000	1abc		380
;	000 0000 0000 0001 abcd		0001	abcd		358
;	000 0000 0000 001a bcd-		0010	abcd		339
;	000 0000 0000 01ab cd--		0011	abcd		320
;	000 0000 0000 1abc d---		0100	abcd		301
;	000 0000 0001 abcd ----		0101	abcd		282
;	000 0000 001a bcd- ----		0110	abcd		263
;	000 0000 01ab cd-- ----		0111	abcd		244
;	000 0000 1abc d--- ----		1000	abcd		225
;	000 0001 abcd ---- ----		1001	abcd		206
;	000 001a bcd- ---- ----		1010	abcd		187
;	000 01ab cd-- ---- ----		1011	abcd		168
;	000 1abc d--- ---- ----		1100	abcd		149
;	001 abcd ---- ---- ----		1101	abcd		130
;	01a bcd- ---- ---- ----		1110	abcd		111
;	1ab cd-- ---- ---- ----		1111	abcd		 92
;	
;	Algorithm for decompression:
;
;		if (E < 2) then
;		  N = (E * 16) + M
;		else
;		  N = Integer [ (16 + M + 0.5) * 2 ** (E - 1) ]
;
;		M = Mantissa
;		E = Exponent
;
;		N max = 516,096 (after decompression)
;		Error max = +/- 1.6%
;
;		CEPPAD 8086 cycle time = 200 nsecs
;		Worst case execution time = 76 usecs
;		Maximum impact on CPU if 3910 bytes use compression
;			= 0.297 secs / 9.2 secs (3.23%)
;
; Use         : <output = cep_decomp( array, table )>
;
; Inputs      : array : BYTE array of compressed data.
;               table : LONARR look up table applied to input array. 
;                       If 0 will be calculated. 
;
; Opt. Inputs : None.
;
; Outputs     : output : LONG array of decompressed data;
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       :	datatype.
;                
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 12/1/96
;
; Modified    :	Version 0.1, MKC, 10/9/97.
;                 Added error return to top level.
;                  Version 0.2, MKC, 10/2/98.
;                 Removed error return to top level.
;
; Version     :	Version 0.1, 10/9/97
;-
;**********************************************************

FUNCTION cep_decomp, input, table

  ; check input byte array

  IF datatype(input) NE 'BYT' THEN MESSAGE,'Array must be BYTE array', /TRACEBACK

  ; if table not set up then calculate table

  IF N_ELEMENTS(table) NE 256 THEN BEGIN

    ; set up table

    table = LONARR(256)

    ; loop through possible inputs

    FOR k = 0, 255 DO BEGIN

      ; get exponent and mantissa of each input

      exponent = ISHFT(k,-4)

      mantissa = k AND 'F'x

      ; calculate output

      IF exponent LT 2 THEN table(k) = 16*exponent + mantissa $
                       ELSE table(k) = LONG( (16 + mantissa + 0.5) * ISHFT(1,exponent-1) )

    ENDFOR

  ENDIF

  ; return results

  RETURN, table(input)

END
