;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_extract
;
; Purpose     :	Extracts and processes required data.
;
; Explanation : Extract and process data.
;
; Use         : < res = cep_extract(time_range, detectors, FORMAT=format) >
;
; Inputs      : time_range : DOUBLE[2] TAI time range.
;               detectors  : INTARR(input_header.ndetectors) flags indicating whether to use detectors or not.
;
; Opt. Inputs : None.
;
; Outputs     : Updates header and data structures in COMMON block.
;               res  : INT FLAG indicating error status
;
; Opt. Outputs:	None.
;
; Keywords    : FORMAT : string indicating plot required.
;               MODES  : STRARR specifying data modes allowed ie 'ABC'
;
; Prev. Hist. :	From cep survey.
;
; Written     :	Version 0.0, Martin Carter, RAL, 28/11/97
;
; Modified    :    Version 0.1, MKC, 18/2/98
;                    Added HISTe, HISTp processing.
;                    Added RANGE keyword.
;                  Version 0.2, 7/10/98
;                    Added Magnetic field and attitude arrays to common block and to ies extdata
;                    Allowed detector sum.
;                  Version 0.3, 30/10/98
;                    Separated calibrate stuff.
;                    Remove bottom IES channel.
;                  Version 0.4, 27/02/02
;                    Changed to function to return error code.
;                  Version 0.5, 01/03/02
;                    Changed so omits bottom 4 IPS energy channels (was 2).
;
; Version     :	Version 0.5, 01/03/02
;-
;**********************************************************

FUNCTION cep_extract, time_range, detectors, FORMAT=format, MODES=modes

  ; define common blocks used internally for draw widgets
  ; full_header   = structure containing original data header information
  ; full_data     = structure array containing original data
  ; header        = structure containing extracted data header information  ** used
  ; data          = structure array containing extracted data               ** used
  ; output_info   = structure containing plotting and processing info       ** used
  ; output_chans  = array containing processed channel positions
  ; output_array  = array containing processed data for plotting
  ; ids           = structure giving widget IDs
  ; dnfull        = co-ordinate conversion array for full window
  ; dnscroll      = co-ordinate conversion array for magnified windows
  ; files         = STRARR giving data file names
  ; ephem         = structure containing ephemeris data
  ; mf            = structure containing magnetic field data
  ; at            = structure containing attitude data

  COMMON ies_draw_widget_block, input_header, input_data, header, data, $
           output_info, output_chans, output_array, ids, dnfull, dnscroll, files, ephem, mf, at

  IF NOT KEYWORD_SET(format) THEN format = 'ENERGY vs DETECTOR'

  IF format EQ 'ENERGY vs DETECTOR' THEN BEGIN

    ; set up top and bottom channels for display

    botchan = 0
    topchan = input_header.nbands - 1

    ; do instrument specific stuff

    IF STRPOS(input_header.datatype,'IES') GE 0 THEN BEGIN

      ; kill integral channel if present

      IF input_header.nbands EQ 16 THEN topchan = 14

    ENDIF ELSE IF STRPOS(input_header.datatype,'IPS') GE 0 THEN BEGIN

      ; kill integral channel if present

      IF input_header.nbands EQ 17 THEN topchan = 15

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTe') GE 0 THEN BEGIN

      ; kill first 2 channels if all present

      IF input_header.nbands EQ 16 THEN botchan = 2

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTp') GE 0 THEN BEGIN

    ENDIF

   ; extract data for detectors

    ies_extdata, { sttime:time_range(0),  entime:time_range(1), use_dets:detectors }, $
                 { use_bands:REPLICATE(1,input_header.nbands), nbands:0, botchan:botchan, topchan:topchan, $
                   lb:INDGEN(input_header.nbands), ub:INDGEN(input_header.nbands) }, $
                   input_header, input_data, 0, header, data, errmsg, EPHEM=ephem, MF=mf, AT=at, MODES=modes

    IF errmsg NE '' THEN GOTO, error

  ENDIF ELSE IF format EQ 'PITCH ANGLE vs ENERGY' THEN BEGIN

    ; do instrument specific stuff

    IF STRPOS(input_header.datatype,'IES') GE 0 THEN BEGIN

      IF input_header.nbands EQ 16 THEN BEGIN

        ; remove integral channel and pedestal

        lb = 5
        ub = 14

      ENDIF ELSE BEGIN

        ; omit bottom energy channel

        ub = input_header.nbands-1
        lb = 1 < ub

      ENDELSE

    ENDIF ELSE IF STRPOS(input_header.datatype,'IPS') GE 0 THEN BEGIN

      ; remove integral channel

      ub = (input_header.nbands-2) > 0

      ;  remove bottom 4 channels

      lb = 4 < ub

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTe') GE 0 THEN BEGIN

      ub = (input_header.nbands-1) > 0

      ;  remove bottom 2 channels

      lb = 2 < ub

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTp') GE 0 THEN BEGIN

      ub = (input_header.nbands-1) > 0

      ;  remove bottom 2 channels

      lb = 2 < ub

    ENDIF

    ; extract data for detectors

    ies_extdata, { sttime:time_range(0),  entime:time_range(1), use_dets:detectors }, $
                 { use_bands:[1], nbands:1, botchan:0, topchan:0, lb:[lb], ub:[ub] }, $
                   input_header, input_data, 0, header, data, errmsg, EPHEM=ephem, MF=mf, AT=at, MODES=modes

    IF errmsg NE '' THEN GOTO, error

  ENDIF ELSE IF format EQ 'SECTOR vs DETECTOR' THEN BEGIN

    ; do sector plot

    ; do instrument specific stuff

    IF STRPOS(input_header.datatype,'IES') GE 0 THEN BEGIN

      IF input_header.nbands EQ 16 THEN BEGIN

        ; remove integral channel and pedestal

        lb = 5
        ub = 14

      ENDIF ELSE BEGIN

        ; use bottom energy channel for compatability with previous

        ub = input_header.nbands-1
        lb = 0

      ENDELSE

    ENDIF ELSE IF STRPOS(input_header.datatype,'IPS') GE 0 THEN BEGIN

      ; remove integral channel

      ub = (input_header.nbands-2) > 0

      ;  remove bottom 2 channels

      lb = 2 < ub

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTe') GE 0 THEN BEGIN

      ub = (input_header.nbands-1) > 0

      ;  remove bottom 2 channels

      lb = 2 < ub

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTp') GE 0 THEN BEGIN

      ub = (input_header.nbands-1) > 0

      ;  remove bottom 2 channels

      lb = 2 < ub

    ENDIF

    ; extract data for detectors

    ies_extdata, { sttime:time_range(0),  entime:time_range(1), use_dets:detectors }, $
                 { use_bands:[1], nbands:1, botchan:0, topchan:0, lb:[lb], ub:[ub] }, $
                   input_header, input_data, 0, header, data, errmsg, EPHEM=ephem, MF=mf, AT=at, MODES=modes

    IF errmsg NE '' THEN GOTO, error

  ENDIF

  RETURN, 1

error :

  MESSAGE, errmsg, /INFORMATIONAL

  RETURN, 0

END
