;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_extract
;
; Purpose     :	Extracts and processes required data.
;
; Explanation : Extract and process data.
;
; Use         : < error = cep_extract (time_range, detectors, SECTOR=sector) >
;
; Inputs      : time_range : DOUBLE[2] TAI time range.
;               detectors  : INTARR(input_header.ndetectors) flags indicating whether to use detectors or not.
;
; Opt. Inputs : None.
;
; Outputs     : Updates header and data structures in COMMON block.
;               error : INT flag indicating error.
;
; Opt. Outputs:	None.
;
; Keywords    : SECTOR : flag indicating sector plot required.
;               RANGE  : overrides data min and max.
;
; Calls       :	ies_extdata, ies_calibrate.
;
; Common      :	ies_draw_widget_block.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	From cep survey.
;
; Written     :	Version 0.0, Martin Carter, RAL, 28/11/97
;
; Modified    :    Version 0.1, MKC, 18/2/98
;                    Added HISTe, HISTp processing.
;                    Added RANGE keyword.
;               Version 0.2, 7/10/98
;                 Added Magnetic field and attitude arrays to common block.
;                 Allowed detector sum.
;
; Version     :	Version 0.2, 7/10/98
;-
;**********************************************************

FUNCTION cep_extract_temp, time_range, detectors, SECTOR=sector, RANGE=range

  ; define common blocks used internally for draw widgets
  ; full_header   = structure containing original data header information
  ; full_data     = structure array containing original data
  ; header        = structure containing extracted data header information  ** used
  ; data          = structure array containing extracted data               ** used
  ; output_info   = structure containing plotting and processing info       ** used
  ; output_chans  = array containing processed channel positions
  ; output_array  = array containing processed data for plotting
  ; ids           = structure giving widget IDs
  ; dnfull        = co-ordinate conversion array for full window
  ; dnscroll      = co-ordinate conversion array for magnified windows
  ; files         = STRARR giving data file names
  ; ephem         = structure containing ephemeris data
  ; mf            = structure containing magnetic field data
  ; at            = structure containing attitude data

  COMMON ies_draw_widget_block, input_header, input_data, header, data, $
           output_info, output_chans, output_array, ids, dnfull, dnscroll, files, ephem, mf, at

  IF NOT KEYWORD_SET(sector) THEN BEGIN

    ; set up top and bottom channels for display

    botchan = 0
    topchan = input_header.nbands - 1

    ; do instrument specific stuff

    IF STRPOS(input_header.datatype,'IES') GE 0 THEN BEGIN

      ; kill integral channel if present

      IF input_header.nbands EQ 16 THEN topchan = 14

      ; set data range

      data_min = 1.e0
      data_max = 1.e6

    ENDIF ELSE IF STRPOS(input_header.datatype,'IPS') GE 0 THEN BEGIN

      ; kill integral channel if present

      IF input_header.nbands EQ 17 THEN topchan = 15

      ; set data range

      data_min = 1.e-1
      data_max = 1.e5

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTe') GE 0 THEN BEGIN

      ; kill first 2 channels if all present

      IF input_header.nbands EQ 16 THEN botchan = 2

      ; set data range

      data_min = 1.e-3
      data_max = 1.e2

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTp') GE 0 THEN BEGIN

      ; set data range

      data_min = 1.e-3
      data_max = 1.e2

    ENDIF

    ; extract data for detectors

    ies_extdata, { sttime:time_range(0),  entime:time_range(1), use_dets:detectors }, $
                 { use_bands:REPLICATE(1,input_header.nbands), nbands:0, botchan:botchan, topchan:topchan, $
                   lb:INDGEN(input_header.nbands), ub:INDGEN(input_header.nbands) }, $
                   input_header, input_data, 0, header, data, errmsg

    IF errmsg NE '' THEN MESSAGE, errmsg

    ; set up processing info

    output_info = { minimum:data_min, maximum:data_max, log:1, cal:1, div:1, rat:1, shift:0, subtract:0, track:1, $
                    rng:0.0, button_labels:['','','','','',''], button_values:[1,0,1,1,0,0], procname:'SPECTRAL', $
                    full_only:0, full_detector:0, full_sector:0, axes:1, band_labels:STRARR(20), noutputs:0, $
                    format:'ENERGY vs DETECTOR', ptitle:STRARR(20), time:0.0D0, ysamples:0, elo:0.0, ehi:0.0  }

    ; process data

    IF ies_calibrate(/QUIET) THEN RETURN, 1

    ; do instrument specific stuff

    IF STRPOS(input_header.datatype,'IES') GE 0 THEN BEGIN

      ; change detector labels to fit

      output_info.ptitle = 'IES ' + header.ptitle

      ; set lowest energy displayed

      output_info.elo = 30.0

    ENDIF ELSE IF STRPOS(input_header.datatype,'IPS') GE 0 THEN BEGIN

      ; change detector labels to fit

      output_info.ptitle = 'IPS ' + header.ptitle

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTe') GE 0 THEN BEGIN

      ; change detector labels to fit

      output_info.ptitle = 'HISTe ' + header.ptitle

    ENDIF ELSE IF STRPOS(input_header.datatype,'HISTp') GE 0 THEN BEGIN

      ; change detector labels to fit

      output_info.ptitle = 'HISTp ' + header.ptitle

    ENDIF

  ENDIF ELSE BEGIN

    ; do sector plot

    ; remove integral channel and pedestal if present

    IF input_header.nbands EQ 16 THEN BEGIN
      lb = 5
      ub = 14
    ENDIF ELSE BEGIN
      lb = 0
      ub = input_header.nbands-1
    ENDELSE

    ; extract data for detectors

    ies_extdata, { sttime:time_range(0),  entime:time_range(1), use_dets:detectors }, $
                 { use_bands:[1], nbands:1, botchan:0, topchan:0, lb:[lb], ub:[ub] }, $
                   input_header, input_data, 0, header, data, errmsg

    IF errmsg NE '' THEN MESSAGE, errmsg

    ; set up processing info for sector plot

    output_info = { minimum:1.e2, maximum:1.e8, log:1, cal:1, div:0, rat:1, shift:0, subtract:0, track:1, rng:0.0, $
                    button_labels:['','','','','',''], button_values:[1,0,1,0,0,0], procname:'SPECTRAL', $
                    full_only:0, full_detector:0, full_sector:0, axes:1, band_labels:STRARR(20), noutputs:0, $
                    format:'SECTOR vs DETECTOR', ptitle:STRARR(20), time:0.0D0, ysamples:0, elo:0.0, ehi:0.0  }

    ; process data

    IF ies_calibrate(/QUIET) THEN RETURN, 1

    ; change detector labels to fit

    output_info.ptitle = 'IES ' + header.ptitle

  ENDELSE

  RETURN,0

END
