;+
; Project     :	CLUSTER - POLAR - PULSAUR
;
; Name        :	CEP_EXTTIME
;
; Purpose     :	Sets up start and end times from requested time range for IES data array.
;
; Explanation : Sets up start and end times from requested time range for IES data array.
;               Allows requested time range to specify time in form hh:mm:ss.
;               Uses the day from the header record to give the period.
;
; Use         : < cep_exttime, header, start_time, end_time, T0=input_start_time, T1=input_end_time >
;
; Inputs      : header : structure containing data set info.
;               start_time : DOUBLE TAI time of start of data extraction
;               end_time   : DOUBLE TAI time of end of data extraction
;
; Opt. Inputs : None.
;
; Outputs     : Changes common block values.
;
; Opt. Outputs:	None.
;
; Keywords    : T0          = DOUBLE TAI time default start
;               T1          = DOUBLE TAI time default start
;
; Calls       :	None.
;                
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Command preparation.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 13/11/9
;
; Modified    : Version 0.1, 15/4/97.
;                 Modified to just set start and end times.
;               Version 0.2, 28/8/97
;                 Changed so that input times not changed.
;               Version 0.3, 11/07/02
;                 Modified to allow time range to be selected across day boundary.
;
; Version     :	Version 0.3, 11/07/02
;-
;**********************************************************

PRO cep_exttime, header, start_time, end_time, T0=input_start_time, T1=input_end_time

  ; deal with start times

  IF KEYWORD_SET ( input_start_time ) THEN BEGIN

    ; if input_start_time specified in form hh:mm:ss then adds in day from start_time
    ; check if year specified

    IF STRPOS(input_start_time,'/') GE 0 OR STRPOS(input_start_time,'-') GE 0 THEN BEGIN

      start_time = anytim2tai(input_start_time)

    ENDIF ELSE BEGIN

      day_start = anytim2utc(header.tstart)
      start_time = anytim2utc(input_start_time)
      start_time.mjd = day_start.mjd
      start_time = utc2tai ( start_time )

    ENDELSE

  ENDIF ELSE start_time = header.tstart 

  ; deal with end times

  IF KEYWORD_SET ( input_end_time ) THEN BEGIN

    ; if input_end_time specified in form hh:mm:ss then adds in day from start_time
    ; check if year specified

    IF STRPOS(input_start_time,'/') GE 0 OR STRPOS(input_start_time,'-') GE 0 THEN BEGIN

      end_time = anytim2tai(input_end_time)

    ENDIF ELSE BEGIN

      end_time = anytim2utc(header.tstart)
      end_time.time = 0
      end_time = utc2tai ( end_time )

      end_time = end_time + anytim2tai(input_end_time)-anytim2tai('00:00:00')
 
    ENDELSE

  ENDIF ELSE end_time = header.tstop

END