;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_get_l1_files
;
; Purpose     :	Gets the CEPPAD level one file names.
;
; Explanation : Retrieves the level one file names in directory pointed to by L1DIR
;               and the specifier. The specifier may contain wildcard characters.
;               If full file name specified will also pick up any files for same instrument and
;               same day i.e there may be multiple normal mode files with the same day number.
;               If error or no file selected then returns ''.
;               Also picks up any files with .gz or .z extensions.
;
; Use         : <filenames = cep_get_l1_files(specifier, L1DIR=l1dir)>
;
; Inputs      : specifier : STRING containing experiment and data type and date in year day of year format.
;                                  Form is as used in level 1 file ie. 'IES_RATES_1995088' or 'IES_'.
;
; Opt. Inputs : None.
;
; Outputs     : filename : String containing level one file names.
;               Also strip directory name from specifier.
;
; Opt. Outputs:	None.
;
; Keywords    : L1DIR : STRING giving L1 directory.
;
; Calls       :	chk_dir, concat_dir, strip_dirname.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Level 0 software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 12/2/96
;
; Modified    :	Version 0.1, 21/5/97
;                    Allowed l1 file names with .DAT* extension to allow compressed files.
;                  Version 0.2, 29/9/97
;                    Allowed L1 directory to be specified by KEYWORD.
;                    Updated comment produced.
;                  Version 0.3, 8/1/98
;                    Changed TRACEBACK option.
;                  Version 0.4, 24/11/98
;                    Stripped directory name from specifier.
;                  Version 0.5, 17/3/99
;                    Made Y2K compliant.
;                  Version 0.6, 10/12/99
;                    Checked for Y2K compatability.
;                  Version 0.7, 09/10/03
;                 Changed to ies get dir.
;
; Version     :	Version 0.7, 09/10/03
;-
;**********************************************************

FUNCTION cep_get_l1_files, specifier, L1DIR=l1dir

  ; set up L1 directory

  IF NOT KEYWORD_SET(l1dir) THEN l1dir = ies_get_dir('L1',/POLAR)

  ; check L1 directory OK

  IF NOT chk_dir ( l1dir ) THEN BEGIN

    MESSAGE, 'Invalid level 1 directory', /CONT

    RETURN, ''

  ENDIF

  ; strip off directory name
  ; NB filename may have wild cards in it

  specifier = strip_dirname(specifier)

  ; check for existence of IF root and remove if present

  pos = STRPOS(specifier,'PO_CEP_L1_')

  IF pos GE 0 THEN BEGIN

    len = STRLEN(specifier) - pos - 10

    IF len GT 0 THEN specifier = STRMID ( specifier, pos+10, len)

  ENDIF

  ; chop off .DAT end if present

  pos = STRPOS(specifier, '.DAT' )

  IF pos GE 0 THEN specifier = STRMID(specifier,0,pos)

  ; chop off 0n v no. if present
  ; l1 files have either forms :
  ; PO_CEP_L1_IES_1996357_01.DAT.gz or PO_CEP_L1_HK_1996221.DAT.gz
  ; decide between forms ie specifier = IES_19* or HK_1996221 or IES_1996357_01

  pos = STRPOS(specifier, '_0' ) ; assumes less than 10 L1 files

  IF pos GE 0 THEN specifier = STRMID(specifier,0,pos)

  ; list level one files into string array

  count = 0

  files = FINDFILE ( concat_dir ( l1dir, 'PO_CEP_L1_' + specifier + '*.DAT*' ), COUNT=count)

  ; action determined by no. of files

  IF count EQ 0 THEN BEGIN

    ; no files

    MESSAGE, 'No level 1 data files found, specifier=' + concat_dir ( l1dir, 'PO_CEP_L1_' + specifier + '*.DAT*'), /CONT

    RETURN, ''

  ENDIF

  ; return l1 files

  RETURN, files

END
