;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_get_l1day
;
; Purpose     :	From calendar date used in level zero filename calculates Julian year and day number.
;               Optionally returns the level zero file version number.
;
; Explanation : Level 0 files have date format 'yymmddnn.*' or '*yyyymmdd_vnn.*' (for Y2K yy = 00)
;               Level 1 files have date format 'yyyyddd'
;               This routine converts between the two.
;               However, the level 1 filename is actually constructed from
;               information within the level 0 file rather than from its file name.
;               This complicates Julian days etc. and so this routine
;               may not always be sufficient.
;
; Use         : <l1dat = cep_get_l1day ( lzfile, VERSION=lz_version ) >
;
; Inputs      : lzfile : STRING containing level 0 file name.
;
; Opt. Inputs : None.
;
; Outputs     : l1dat : STRING containing level 1 date.
;
; Opt. Outputs:	None.
;
; Keywords    : VERSION : LONG lz file version number.
;
; Calls       :	strip_dirname.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 12/1/96
;
; Modified    :	Version 0.1, 12/11/96
;                 Fixed bug when lz file names of form 'yymmdd' encountered with dd=19.
;               Version 0.2, 10/12/99
;                 Updated for Y2K compatability.
;                 Added stripping of directory from lzfile.
;                 Found bug : did not return LZ version no. correctly from cdrom data i.e always 0 ??
;
; Version     :	Version 0.2, 10/12/99
;-
;**********************************************************

FUNCTION cep_get_l1day, lzfile, VERSION=lz_version

  ; take off directory

  tlzfile = strip_dirname(lzfile)

  ; take off file mode

  IF STRPOS(tlzfile,'.') LT 0 THEN GOTO, file_name_error

  tlzfile = STRMID(tlzfile,0,STRPOS(tlzfile,'.'))

  ; construct day, month, year from level 0 file name

  ; check if filename only 8 characters long

  IF STRLEN(tlzfile) EQ 8 THEN BEGIN

    ; assume file has date form yymmddnn

    IF STRMID (tlzfile, 0, 1) EQ '9' THEN cent = '19' $
    ELSE IF STRMID (tlzfile, 0, 1) EQ '0' THEN cent = '20' $
    ELSE GOTO, file_name_error

    ; get yyyymmdd

    l1_year    = cent + STRMID ( tlzfile, 0, 2)

    l1_month   = LONG ( STRMID ( tlzfile, 2, 2) )

    l1_day     = LONG ( STRMID ( tlzfile, 4, 2) )

    lz_version = LONG ( STRMID ( tlzfile, 6, 2 ) )

  ENDIF ELSE IF STRLEN(tlzfile) GE 12 THEN BEGIN

    ; assume file has date form yyyymmdd_vnn

    ; strip out format

    tlzfile = STRMID(tlzfile,STRLEN(tlzfile)-12,12)

    ; get yyyymmdd

    l1_year  = STRMID ( tlzfile, 0, 4)

    l1_month = LONG ( STRMID ( tlzfile, 4, 2) )

    l1_day   = LONG ( STRMID ( tlzfile, 6, 2) )

    lz_version = LONG ( STRMID ( tlzfile, 10, 2 ) )

  ENDIF ELSE GOTO, file_name_error

  ; work out day of year

  dayofyear = JULDAY ( l1_month, l1_day, l1_year ) - JULDAY ( 1, 1, l1_year ) + 1

  RETURN, l1_year + STRING(FORMAT='(I3.3)',dayofyear)

  file_name_error : MESSAGE, 'INVALID FORM FOR LZ FILE : ' + lzfile

END

