;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_get_luts
;
; Purpose     :	Get channel positions for each detector for the different lookup tables for a given experiment
;               and a calibration factor for each LUT.
;
; Explanation : The bin positions and sizes of the energy channels differ from detector to detector.
;               The bin positions of the energy channels for each detector are stored in loadable lookup tables.
;               Separate lookup tables exist for different integration times and the large and small pixel sizes.
;               This routine returns an array of lookup tables.
;               The lookup tables will occasionally be changed.
;               The new lookup tables are appended to the existing ones.
;               Associated with the lookup tables will be a calibration table giving
;               a calibration factor for each table. This normalizes the data taken with each table
;               to counts per cm2 per sr (detector deadtime removed for IES).
;
; Use         : <luts = cep_get_luts (experiment, count_calibration, energy_calibration, lut_changes, $
;                                     energy_efficiency, lut_map)>
;
; Inputs      : experiment      : String giving experiment.
;
; Opt. Inputs : None.
;
; Outputs     : luts : BYTARR (2, no_of_energies, no_of_detectors, no_of_luts, no_of_changes)
;                      top and bottom bin positions of energy channels for each detector,
;               count_calibration  : FLTARR(no_of_luts,no_of_changes) calibration factor for each look up table.
;               energy_calibration : FLOAT calibration factor for bins to energy.
;                                          may have energy offset also.
;               energy_efficiency  : FLTARR(no_of_energies,no_of_luts,no_of_changes)
;                                          efficiency factor for each energy and look up table.
;
;               lut_changes        : STRARR(no_of_changes-1) time of each LUT change.
;
; Opt. Outputs:	lut_map : INTARR(no_of_luts,no_of_changes) integration mode for each LUT
;                                          Only present for IES calibration file.
;
; Keywords    : None.
;
; Calls       :	concat_dir.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 20//3/96
;
; Modified    :	Version 0.1, MKC, 23/9/96.
;                 Added IPS calibration factors.
;                 Removed no_of_energies, no_of_detectors, no_of_luts from argument list.
;               Version 0.2, 12/11/96
;                 Changed pedestal to pedestals.
;                 Changed IES_look_up_tables to IES_CALIBRATION.
;                 Added widths to pedestals.
;                 Renamed IPS variables to conform to IES stuff.
;                 Added similar HSTe and HSTp variables.
;               Version 0.3, 16/1/97
;                 Changed definition of pedestals to be raw pedestal data.
;                 Added lut_map to index into pedestals array.
;                 Made lut_map and pedestals optional.
;               Version 0.4, 4/7/97
;                 Added energy efficiency for HIST to argument list.
;                 Removed pedestals and lut_map from argument list.
;               Version 0.5, 10/10/97
;                 Put back lut_map into argument list.
;               Version 0.6, 5/2/99
;                 Added lut_changes to argument list.
;               Version 0.7, 09/10/03
;                 Changed to ies get dir.
;
; Version     :	Version 0.7, 09/10/03
;-
;**********************************************************

FUNCTION cep_get_luts, experiment, count_calibration, energy_calibration, lut_changes, energy_efficiency, lut_map

  ; luts               : BYTARR (2, no_of_energies, no_of_detectors, no_of_luts, no_of_changes)
  ;                         IES  [2,16,9,2,3]
  ;                         IPS  [2,17,10,1,2]
  ;                         HSTE [2,16,1,4,6]
  ;                         HSTP [2,16,1,4,6]
  ; count_calibration  : FLTARR(no_of_luts,no_of_changes)
  ; energy_calibration : FLOAT (FLTARR(2) for IES)
  ; energy_efficiency  : FLTARR(no_of_energies,no_of_luts,no_of_changes)
  ; lut_changes        : STRARR(no_of_changes-1)
  ; lut_map            : INTARR(no_of_luts,no_of_changes) (only for IES)

  luts = 0
  pedestals = 0
  lut_map = 0
  lut_changes = 0
  energy_efficiency = 0
  count_calibration = 0
  energy_calibration = 0
  temperatures = 0

  RESTORE, concat_dir(ies_get_dir('TM_TABLE',/POLAR), experiment + '_CALIBRATION')

  RETURN, luts

END
