;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_get_lz_files
;
; Purpose     :	Gets the CEPPAD level zero file names.
;
; Explanation : Uses PICKFILE to retrieve the level zero file name in directory pointed to by PO_CEP_LZ_DIR.
;               Uses FINDFILE to get the file name. If the name returned by PICKFILE contains wildcards then
;               may generate a number of level 0 files.
;               If error or no file selected then returns ''.
;
; Use         : <lzfiles = cep_get_lz_files()>
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : filename : String containing level zero file names;
;
; Opt. Outputs:	None.
;
; Keywords    : ALL : FLAG indicating that all LZ files in specified directory should be processed.
;                          Avoids use of pickfile.
;
; Calls       :	chk_dir, concat_dir.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Level 0 software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 12/2/96
;
; Modified    :	Version 0.1, 4/3/96.
;                 Removed file filter to allow any file name.
;               Version 0.2, 3/11/98
;                 Checked different cases for lz directory.
;               Version 0.3, 10/12/99
;                 Checked for Y2K compatability.
;                 Replaced PICKFILE with ies_pickfile.
;       	    Version 0.4, 30/09/03
;                 Removed FIX_FILTER keyword to ies_pickfile.
;                 Removed insistence that LZ directory files exists.
;                 Added /ALL keyword.
;               Version 0.5, 09/10/03
;                 Changed to ies get dir.
;
; Version     :	Version 0.5, 09/10/03
;-
;**********************************************************

FUNCTION cep_get_lz_files, ALL=all

  ; set up LZ directory

  lzdir = ies_get_dir('LZ',/POLAR)

  ; check LZ directory OK

  IF NOT chk_dir ( lzdir ) THEN BEGIN

    lzdir = STRUPCASE(lzdir)

    IF NOT chk_dir ( lzdir ) THEN BEGIN

      lzdir = STRLOWCASE(lzdir)

      IF NOT chk_dir ( lzdir ) THEN BEGIN

        MESSAGE, 'Invalid level 0 directory', /CONT, /TRACEBACK

      ENDIF

    ENDIF

  ENDIF

  ; get LZ files

  IF KEYWORD_SET(all) THEN BEGIN

    ; select all LZ files in directory

    filter = concat_dir ( lzdir, 'PO_LZ_CEP_*.DAT')

    ; get multiple files

    files = FINDFILE ( filter, COUNT=count)

  ENDIF ELSE BEGIN

    ; pick level 0 data file, may contain wild character so file may not exist
    ; allow any level 0 file name format

    files = ies_pickfile ( PATH=lzdir, /READ, TITLE='Select level 0 files', /MULTIPLE_FILES, COUNT = count)

  ENDELSE

  ; check some files found

  IF count EQ 0 THEN BEGIN

    ; no files

    MESSAGE, 'No level 0 data files found', /CONT, /TRACEBACK

    RETURN, ''

  ENDIF

  ; return lz files

  RETURN, files

END
