;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_get_lzday
;
; Purpose     :	From Julian year and day number calculates calendar date used in level zero filename.
;               
; Explanation : This routine is used to get a date string in the form 'yyyymmdd'.
;               This is used to get the level 0 file name which has format 'yymmdd*' or '*yyyymmdd*'
;               Level 1 files have date format 'yyyyddd_vvv.DAT' or 'yyyyddd.DAT'.
;               Intermediate files have date format 'yyyyddd.DAT' or 'yyyymmdd.DAT'
;
; Use         : <lzdat = cep_get_lzday ( input_string ) >
;
; Inputs      : input_string : STRING containing file name with date.
;
; Opt. Inputs : None.
;
; Outputs     : lzdat : STRING containing level 0 date.
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       :	strip_dirname.
;                
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 16/1/97
;
; Modified    :	Version 0.1, MKC, 22/9/97.
;                 Added new intermediate format file type.
;               Version 0.2, MKC, 2/11/98
;                 Removed directory if presnet from input_string to avoid possible conflicts.  
;               Version 0.3, 22/12/99
;                 Updated for Y2K compatability.
;
; Version     :	Version 0.3, 22/12/99
;-
;**********************************************************

FUNCTION cep_get_lzday, input_string 

  ; if input string has only 7 or 8 letters then assume is already in right form

  IF STRLEN(input_string) EQ 7 OR STRLEN(input_string) EQ 8 THEN new_string = input_string ELSE BEGIN

    ; remove directory

    new_string = strip_dirname(input_string)

    ; construct 'yyyyddd' from level 1 file name

    start_pos = STRPOS ( new_string, '19' )
    start_pos_20 = STRPOS ( new_string, '20' )

    IF (start_pos_20 GE 0 AND start_pos LT 0) OR $
       (start_pos_20 GE 0 AND start_pos_20 LT start_pos) THEN start_pos = start_pos_20

    IF start_pos LT 0 THEN MESSAGE, 'INVALID DATE FORMAT : ' + input_string, /TRACEBACK

    end_pos = STRPOS ( new_string, '.DAT' )

    IF end_pos LT 0 OR end_pos LE start_pos THEN MESSAGE, 'INVALID DATE FORMAT : ' + input_string, /TRACEBACK

    new_string = STRMID  (new_string, start_pos, end_pos-start_pos)

    ; cope with 'yyyyddd_vvv.DAT' format

    IF STRPOS ( new_string, '_' ) GT 0 THEN new_string = STRMID ( new_string, 0, STRPOS ( new_string, '_' ) )

  ENDELSE

  ; check form 

  IF STRLEN(new_string) EQ 7 THEN BEGIN  

    ; input string has l1 date form 19yyddd
    l1_year  = LONG ( STRMID ( new_string, 0, 4) )

    l1_doy   = LONG ( STRMID ( new_string, 4, 3) )

    ; work out calendar date

    CALDAT, l1_doy + JULDAY ( 1, 1, l1_year ) - 1, lz_month, lz_day, lz_year

    ; set output string

    RETURN, STRING(FORMAT='(I4.4)',lz_year) + STRING(FORMAT='(I2.2)',lz_month) + STRING(FORMAT='(I2.2)',lz_day)

  ENDIF ELSE IF STRLEN(new_string) EQ 8 THEN BEGIN

    ; new string already has lz form

    RETURN, new_string

  ENDIF ELSE MESSAGE, 'INVALID DATE FORMAT : ' + input_string, /TRACEBACK

END

