;+
; Project     :	CLUSTER - PEACE
;
; Name        :	CEP_GET_TEMPERATURE
;
; Purpose     : Returns average POLAR IES temperature over specified range.
;
; Explanation :	Uses the HK intermediate data files to calculate the average IES temperature.
;               Returns -1 if no data available (NB SC shouldnt get this cold).
;
; Use         : < temp = cep_get_temperature ( times ) >
;
; Inputs      : times         : DBLARR(npoints) TAI times of samples.
;
; Opt. Inputs : None.
;
; Outputs     : temp          : FLOAT temperature in degrees C.
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       : ies_get_dir.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 8/9/97
;
; Modified    :	Version 0.1.
;                    Uses ies_get_dir
;               Version 0.2.
;                    Added -f option to rm.
;           	Version 0.3, 11/07/02
;                 Used ies_spawn.
;           	Version 0.4, 02/10/03
;                 Renamed from ies get temperature.
;                 Changed ies get dir.
;                 Changed printout.
;
; Version     :	Version 0.4, 02/10/03
;-
;**********************************************************

FUNCTION cep_get_temperature, times

  COMMON temperature_block, hk_file, input_data, failed_mjd

  start_time = anytim2utc(times(0))

  end_time   = anytim2utc(times(N_ELEMENTS(times)-1))

  ; check if same day

  IF start_time.mjd NE end_time.mjd THEN BEGIN
    end_time.mjd  = start_time.mjd
    end_time.time = 86400.0*1000.0
  ENDIF

  IF KEYWORD_SET(failed_mjd) THEN IF start_time.mjd EQ failed_mjd THEN RETURN,-1

  failed_mjd = start_time.mjd

  ; get HK file

  date  = anytim2utc(start_time,/EXTERNAL)

  ; get hk file

  tmp_files = FINDFILE ( concat_dir ( ies_get_dir('HK',/POLAR), 'PO_CEP_RAL_HK_'+ $
    STRING(date.year,FORMAT='(I4.4)')+STRING(date.month,FORMAT='(I2.2)')+STRING(date.day,FORMAT='(I2.2)')+'.DAT*' ), COUNT=count)

  IF count EQ 0 THEN BEGIN
    MESSAGE, 'WARNING, No HK intermediate format file found', /CONT
    RETURN, -1
  ENDIF

  IF N_ELEMENTS(hk_file) EQ 0 THEN restore_file = 1 ELSE $
  IF hk_file NE tmp_files(0) THEN restore_file = 1 ELSE restore_file = 0

  IF restore_file THEN BEGIN

    hk_file = tmp_files(0)

    ; decompress if necessary
    ; NB tmp_file may be changed

    tmp_file = hk_file

    IF ies_test_gzip(tmp_file) THEN BEGIN
      ies_gunzip, tmp_file
      gzip = 1
    ENDIF ELSE gzip = 0

    ; set restore variables

    input_header = 0
    input_data   = 0

    ; restore data

    PRINT, 'Restoring ', tmp_file

    RESTORE, tmp_file

    ; remove tmp file

    IF gzip THEN IF ies_spawn('rm -f ' + tmp_file) THEN MESSAGE, 'REMOVE FAILED'

  ENDIF

  ; extract temperatures

  ; get valid data

  list = WHERE ( input_data.data NE 0 AND input_data.data NE 255, count)

  IF count EQ 0 THEN BEGIN
    MESSAGE, 'WARNING, No valid HK data found', /CONT
    RETURN, -1
  ENDIF

  input_data = input_data(list)

  ; get data in time range
  ; round to nearest minute

  list = WHERE ( input_data.time GE utc2tai(start_time)-60. AND input_data.time LE utc2tai(end_time)+60., count )

  IF count EQ 0 THEN BEGIN
    MESSAGE, 'WARNING, No valid HK data in time range', /CONT
    RETURN, -1
  ENDIF

  failed_mjd = 0

  ; return average temperature

  RETURN, TOTAL(input_data(list).data)/count

END