;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_hk
;
; Purpose     :	Extracts IES temperature or magnetic field information from Level zero HK data.
;
; Explanation : Extracts IES temperature or magnetic field information from Level zero HK data and stores as intermediate format file.
;
; Use         : < cep_hk, mjulday, output_header, output_data, BF=bf>
;
; Inputs      : datatype        : STRING giving data type.
;               mjulday         : LONG giving modified Julian day.
;               experiment      : String indicating experiment i.e 'MF' or 'HK'.
;
; Opt. Inputs : None.
;
; Outputs     : output_header : header structure
;               output_data   : data structure
;
; Opt. Outputs:	None.
;
; Keywords    : BF         : Flag indicating magnetic field output required.
;
; Calls       :	ies_data_structure, ies_header_structure, cep_decomp, cep_times.
;
; Common      :	CEP_HEALTH_NOSUB, CEP_KEY_PARAMETERS.
;
;       NOSUB_CEP_HEALTH	        STRUCTURE
;   NOSUB_CEP_HEALTH TAG NAMES	TYPE			DESCRIPTION
;       TIME			LONG	Time in milliseconds
;	SC_AN0			BYTE	See 'Polar Command & Telemetry Handbook'
;	SC_AN1			BYTE    See 'Polar Command & Telemetry Handbook'
;	SC_AN2			BYTE	See 'Polar Command & Telemetry Handbook'
;	SC_AN3			BYTE	See 'Polar Command & Telemetry Handbook'See 'Polar Command & Telemetry Handbook'
;	SC_bstat		BYTE	See 'Polar Command & Telemetry Handbook'
;	SC_th0			BYTE	See 'Polar Command & Telemetry Handbook'
;	SC_th1			BYTE	See 'Polar Command & Telemetry Handbook'
;	SC_th2			BYTE	See 'Polar Command & Telemetry Handbook'
;	SC_th3			BYTE	See 'Polar Command & Telemetry Handbook'
;	CMDEcho_1		BYTE	See 'Polar Command & Telemetry Handbook'
;	CMDEcho_2		BYTE	See 'Polar Command & Telemetry Handbook'
;	CMDEcho_3		BYTE	See 'Polar Command & Telemetry Handbook'
;	CMDEcho_4		BYTE	See 'Polar Command & Telemetry Handbook'
;       CMDEcho_5		BYTE	See 'Polar Command & Telemetry Handbook'
;	hw_status		BYTE    See 'CEPPAD Telemetry Format Handbook'
;       hw_status		BYTE	See 'CEPPAD Telemetry Format Handbook'
;       sw_status		BYTE	See 'CEPPAD Telemetry Format Handbook'
;       CheckSum		BYTE	See 'Polar Command & Telemetry Handbook'
;
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 12/1/96
;
; Modified    :	Version 0.1, 2/2/98
;                    Fixed bug when only one sample.
;               Version 0.2, 22/10/98
;                    Added BF keyword.
;               Version 0.3, 02/10/03
;                    Fixed bug cause by changes in IDL version 5.5.
;               Version 0.4, 15/10/03
;                    Added datatype and experiment to argument list in line with cep reorg.
;                    Expcitly named HK and MF data sets.
;
; Version     :	Version 0.4, 15/10/03
;-
;**********************************************************

PRO cep_hk, datatype, mjulday, output_header, output_data, experiment

  COMMON CEP_HEALTH_NOSUB, nosub_cep_health, recs_extracted
  COMMON CEP_KEY_PARAMETERS, cep_kp, num_kp

  IF experiment EQ 'MF' THEN BEGIN

    ; produce BF intermediate data

    ; set up times

    ; get CDS_INT_TIME structure

    get_utc, dummy

    ; get tai of start of Julian day

    start_of_day = utc2tai ( {CDS_INT_TIME, mjd:mjulday, time:0L} )

    ; convert spin_times to tai

    tai_times = start_of_day + ABS(cep_kp.time)/1000.0D0 ; not sure why ABS anymore

    ; create output data

    no_of_samples   = N_ELEMENTS(cep_kp.time)

    output_header = ies_header_structure ( datatype, no_of_samples, 2, 1, 1, 1, $
                                  EXPERIMENT=experiment, $
                                  TIME_START = tai_times(0), $
                                  TIME_STOP  = tai_times(no_of_samples-1), $
                                  TITLE  = datatype, $
                                  XTITLE = 'UT:', $
                                  YTITLE = '[deg]', $
                                  ZTITLE = 'Magnetic field', $
                                  PTITLE = ['AZ','MAG_EL'] )

    output_header.ch_positions(1,*,*,*) = 1

    output_data   = ies_data_structure ( no_of_samples, 2, 1, 1, EXPERIMENT=experiment)

    output_data.time = tai_times
    output_data.data[0,0,0] = cep_kp.az
    output_data.data[0,1,0] = cep_kp.magel

  ENDIF ELSE BEGIN

    ; produce HK intermediate data

    ; set up times

    ; get CDS_INT_TIME structure

    get_utc, dummy

    ; get tai of start of Julian day

    start_of_day = utc2tai ( {CDS_INT_TIME, mjd:mjulday, time:0L} )

    ; convert spin_times to tai

    tai_times = start_of_day + ABS(nosub_cep_health.time)/1000.0D0

    ; get IES temperatures

    p = NOSUB_CEP_HEALTH.sc_th3

    list = WHERE(p NE 0, count)

    IF count GT 0 THEN p = p(list) & tai_times = tai_times(list)

    temps = (8.28167e1-2.03057e-1*p-6.59092e-3*p*p+5.08593e-5*p*p*p-1.19468e-7*p*p*p*p)

    ; plot,(tai_times-tai_times(0))/3600.,temps,yrange=[0,30]

    ; create output data

    no_of_samples   = N_ELEMENTS(p)

    output_header = ies_header_structure ( datatype, no_of_samples, 1, 1, 1, 1, $
                                  TIME_START = tai_times(0), $
                                  TIME_STOP  = tai_times(no_of_samples-1), $
                                  TITLE  = datatype, $
                                  XTITLE = 'UT:', $
                                  YTITLE = 'IES', $
                                  ZTITLE = 'Temperature', $
                                  PTITLE = 'IES temp' )

    output_header.ch_positions(1,*,*,*) = 1

    output_data   = ies_data_structure ( no_of_samples, 1, 1, 1)

    IF N_ELEMENTS(tai_times) GT 1 THEN BEGIN

      output_data.time = tai_times
      output_data.data[*] = transpose(temps)

    ENDIF ELSE BEGIN

      output_data.time = tai_times[0]
      output_data.data = temps[0]

    ENDELSE

  ENDELSE

END
