;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_l1toif
;
; Purpose     :	Extracts RAL intermediate format data from level 1 files.
;
; Explanation : Processes L1 files for single instrument according to file spec.
;               Where the level 1 data for a given day spans multiple files then all
;               the files for the day will be picked up.
;               For each UT day produces a RAL intermediate format file.
;               The CEPPAD instrument used is determined from the level 1 filename.
;               The RAL output produced is determined by the mode.
;               The level 1 files may be compressed in unix systems with .gz file type.
;
;               The default is to prompt for the file specification and to use the
;               'SPIN_1' data mode.
;               If SURVEY file already exists then will not bother reprocessing since
;               guaranteed to be 24 hours of data.
;               If needs to process and /GZIP set then leaves file in compressed format and
;               file name returned contains .gz extension. If no need to process then returns file
;               name as was.
;
; Use         : <cep_l1toif, filespec, MODE=mode, $
;                            T0=start_time, T1=end_time, $
;                            LEAVE=leave, EXISTS=exists, FILE=file, PEDESTAL=pedestal, BF=bf, /GZIP>
;
; Inputs      : filespec : String containing L1 filename including instrument/mode/date file specification.
;                          If no 'RATES' string present in the filename then defaults to
;                          normal mode CEPPAD level 1 files.
;                          Note however that will still accept a file specification as previoudly.
;                          This may contain the instrument and date ie. 'IES_1994215' or
;                          just part of the file name specification ie. 'IES_'.
;                          Point is want to allow file spec but dont want to use pickfile to choose it.
;                          Can still allow pickfile to choose file name but does not allow file space to be entered on PC.
;
; Opt. Inputs : None.
;
; Outputs     : RAL intermediate format file.
;
; Opt. Outputs:	None.
;
; Keywords    : MODE       : String giving RAL intermediate data output required.
;               T0         : STRING giving start time of period of interest in level 1
;                            data. Example format = '10:00:00', ignores day.
;               T1         : STRING giving end   time of period of interest in level 1
;                            data. Example format = '11:00:00', ignores day.
;               LEAVE      : LOGICAL requesting that the level 1 data be left in the
;                            COMMON block defined by the AEROSPACE routine
;                            get_cep_data. Processing is stopped.
;               FILE       : STRING to receive intermediate format file name.
;               PEDESTAL   : Flag indicating pedestal output required.
;               BF         : Flag indicating magnetic field output required.
;               GZIP       : indicates compression of output files (Unix only) required.
;               L1DIR      : STRING giving L1 directory.
;               FORCE      : Forces creation of intermediate format file even if already exists.
;               DEBUG      : STOPs in routine.
;               FILTER     : STRING for PICKFILE filter ie '*HISTe*'. This is useful
;                            when the number of files in the directory is too large to
;                            be displayed by PICKFILE.
;
; Calls       :	datatype, cep_parse_l1day, cep_get_l1_files, cep_reorg, get_cep_data, concat_dir, cep_chk_monotonic, ies_gunzip,
;               ies_test_gzip, cep_get_lzday, ies_get_dir. ies_pickfile.
;
; Common      : POL_CEP_SH, POL_CEP_IPS, POL_CEP_IES, POL_CEP_HSTE, POL_CEP_HSTP, POL_CEP_IPS_RTS, $
;               POL_CEP_IES_RTS, POL_CEP_HIST_RTS.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 12/1/96
;
; Modified    :	Version 0.1, MKC, 26/3/96.
;                            Added exists and leave keywords.
;               Version 0.2, MKC, 13/06/96
;                            Modified call to get_cep_data to reflect SW change.
;                            Removed spin_time from cep_reorg argument list.
;               Version 0.3, MKC, 17/9/96
;                            Added spin header record time to cep_reorg call.
;                            Removed shift keyword.
;                            Use start time and end time in level 1 data extraction.
;                            Added CLEAN_n modes. Removed mode check.
;                            Used L1 directory path.
;                            Added FILE keyword.
;               Version 0.4, 8/11/96
;                            Changed default mode to SPIN_1
;               Version 0.5, 12/11/96
;                            Forced mode=*SURVEY* to be full 24 hours.
;                            Does not recreate SURVEY files if present already.
;               Version 0.6, 19/12/96
;                            If more than one level 1 file then processes all files regardless of
;                            whether a survey file or not.
;               Version 0.7, 5/2/97
;                            Removed /FIX_FILTER from PICKFILE.
;                            Added pedestal_shift to cep_reorg and PEDESTAL keyword.
;               Version 0.8, 11/3/97.
;                            Changed pedestal_shift to pedestal_info.
;               Version 0.9, 15/4/97.
;                            Allow wild card l1 file to be specifed in PICKFILE.
;                            Added pedestal environment variable.
;               Version 1.0, 2/5/97.
;                            Save only pedestal shift in Pedestal file.
;                            Added /XDR keyword to SAVE.
;               Version 1.1, 21/5/97.
;                            Added decompression of .gz or .z files using ies_gunzip.
;                            Added GZIP keyword.
;                            Removed restriction that if more than one level 1 file then processes all files regardless of
;                            whether a survey file or not.
;               Version 1.2, 19/6/97.
;                            Added HK data extraction. Added FILTER keyword.
;               Version 1.3, 8/9/97.
;                            Added new instrument data directories environment variables.
;               Version 1.4, MKC, 10/9/97.
;                            Added error return to top level.
;                            Changed intermediate format file name.
;               Version 1.5, 29/9/97, MKC
;                            Added changeable L1 directory.
;               Version 1.6, 5/11/97, MKC
;                            Added FORCE keyword.
;               Version 1.7, 10/2/98, MKC
;                            Added DEBUG keyword.
;                            Added ies_get_dir
;               Version 1.8, 31/7/98, MKC
;                            Removed start_time and end_time keywords to cep_reorg.
;               Version 1.9, 3/9/98, MKC
;                            Changed comments for SH file.
;               Version 2.0, 22/10/98, MKC
;                            Added extraction of magnetic field data from HK file.
;                            Fix some Y2K problems.
;               Version 2.1, 13/12/99, MKC
;                            Updated for Y2K.
;                            Changed behaviour of filespec.
;                            Removed KEYWORD EXISTS.
;                            Changed so that LEAVE does not process to intermediate format.
;                            Replaced PICKFILE with ies_pickfile.
;                            Removed FILTER keyword.
;            	Version 2.2 11/07/02
;                            Used ies_spawn.
;            	Version 2.3 24/07/02
;                            Changed to use /COMPRESS option instead of GZIP.
;            	Version 2.4 02/10/03
;                            Changed ies get dir.
;                            Changed cep_hk argument list more like cep reorg.
;                            Changed printout.
;            	Version 2.5, 20/01/04
;                            Fixed so that pedestal data results go in ifdata directory as previously.
;
; Version     :	Version 2.5, 20/01/04
;;-
;**********************************************************

PRO cep_l1toif, filespec, MODE=mode, T0=start_time, T1=end_time, LEAVE=leave, FILE=file, PEDESTAL=pedestal, BF=bf, GZIP=gzip, $
                L1DIR=l1dir, FORCE=force, DEBUG=debug, FILTER=filter

  ; set up common blocks to access level 1 data
  ; taken from get_cep_data

  COMMON POL_CEP_SH,      PLCPSpn_hed, PLCPSpn_hed_num
  COMMON POL_CEP_IPS,     PLCPIPS_Time, PLCPIPS_bytes, PLCPIPS_Org, $
                          PLCPIPS_Start_Time, PLCPIPS_Stop_Time
  COMMON POL_CEP_IES,     PLCPIES_time, PLCPIES_bytes, PLCPIES_Org, $
                          PLCPIES_Start_Time, PLCPIES_Stop_Time
  COMMON POL_CEP_HSTE,    PLCPHSTE_time, PLCPHSTE_bytes, PLCPHSTE_Org, $
                          PLCPHSTE_Start_Time, PLCPHSTE_Stop_Time
  COMMON POL_CEP_HSTP,    PLCPHSTP_time, PLCPHSTP_bytes, PLCPHSTP_Org, $
                          PLCPHSTP_Start_Time, PLCPHSTP_Stop_Time
  COMMON POL_CEP_IPS_RTS, PLCPIPS_Rts, PLCPIPS_Rts_num, PLCPIPS_Rts_tim, $
                          PLCPIPS_Rts_mx, $
                          PLCPIPS_Rts_Start_Time, PLCPIPS_Rts_Stop_Time
  COMMON POL_CEP_IES_RTS, PLCPIES_Rts, PLCPIES_Rts_num, PLCPIES_Rts_tim, $
                          PLCPIES_Rts_ch, $
                          PLCPIES_Rts_Start_Time, PLCPIES_Rts_Stop_Time
  COMMON POL_CEP_HIST_RTS, PLCPHIST_Rts, PLCPHIST_Rts_num, PLCPHIST_Rts_tim, $
                           PLCPHIST_Rts_id, $
                           PLCPHIST_Rts_Start_Time, PLCPHIST_Rts_Stop_Time

  ; come back out to top level unless debug set

  IF KEYWORD_SET(debug) THEN ON_ERROR, 0 ELSE ON_ERROR, 1

  ; set up number of pedestal values retained

;  npedinfo = 1
  npedinfo = 3

  ; check input

  ; check if filename defined
  ; if filespec not defined then allows multiple L1 files to be picked
  ; else uses previous logic for identifying all files associated with filespec

  IF datatype(filespec) NE 'STR' THEN BEGIN

    ; set filter

    IF NOT KEYWORD_SET(filter) THEN filter = 'PO_CEP_L1*.DAT*'

    ; prompt for level 1 file from single Julian day and particular instrument and mode
    ; allows wild card to be specified under Unix

    filespec = ies_pickfile ( PATH=ies_get_dir('L1',/POLAR), FILTER=filter, /READ, $
                              TITLE='Select level 1 file spec', GET_PATH=l1dir )

    IF filespec EQ '' THEN BEGIN

      PRINT, 'Level 1 file not selected'

      RETURN

    ENDIF

  ENDIF

  ; get level 1 files

  l1files = cep_get_l1_files(filespec, L1DIR=l1dir)

  ; check file selected

  IF l1files[0] EQ '' THEN BEGIN

    PRINT, 'Level 1 file not selected'

    RETURN

  ENDIF

  ; ensure files sorted in time order

  l1files = l1files ( SORT ( l1files ) )

  PRINT, 'Level 1 files : '

  FOR f = 0, N_ELEMENTS(l1files)-1 DO PRINT, l1files(f)

  ; check mode

  IF datatype(mode) NE 'STR' THEN mode = 'SPIN_1'

  ; print information

  PRINT, 'Beginning level 1 to RAL intermediate format conversion'

  ; set up old file label

  old_file_label = ''

  ; read level 1 files

  FOR f = 0, N_ELEMENTS ( l1files ) - 1 DO BEGIN

    ; get level 1 date in modified Julian day format

    l1_mjulday = cep_parse_l1day ( l1files(f) )

    ; split name into l1_label i.e 'IES_1994088*' and l1_root ie directory/PO_CEP_L1_

    pos = STRPOS(l1files(f),'PO_CEP_L1_')

    IF pos LT 0 THEN MESSAGE, 'Invalid file name : ' + l1files(f), /TRACEBACK

    l1_root = STRMID(l1files(f), 0, pos + 10)

    len = STRLEN(l1files(f)) - pos - 10

    IF len LE 0 THEN MESSAGE, 'Invalid file name : ' + l1files(f), /TRACEBACK

    l1_label = STRMID ( l1files(f), pos+10, len)

    ; get root and date ie. 'IES' and '1996100'

    ;IF pos LT 1 THEN pos = STRPOS(l1_label, '200' )

    ;RF fix - the above doesn't work for day 199!

    FOR year = 1996, 2020 DO BEGIN
        yyyy = string(year, format = "(i4.4)")
        pos = STRPOS(l1_label, yyyy )
        IF pos NE -1 THEN BREAK 
    ENDFOR
   
    IF pos LT 1 THEN MESSAGE, 'Invalid file root : ' + l1_label, /TRACEBACK ELSE $
      file_root = STRMID (l1_label, 0, pos-1)

    IF pos + 7 GT STRLEN(l1_label) THEN MESSAGE, 'Invalid file date : ' + l1_label, /TRACEBACK ELSE $
      file_date = STRMID (l1_label, pos, 7)

    ; get file label ie 'IES_19961231'

    file_label = file_root + '_' + cep_get_lzday(file_date)

    ; test if intermediate format SURVEY file exists
    ; removed restriction that if more than one l1 file then always process each file
    ; want to be able to specify whole bunch of l1 files and only process some of them
    ; restriction was brought in to cope with instance where more than one l1 file per
    ; day, in that case get temporary survey plot file created which then prevents next
    ; part of l1 file being processed

    IF file_root EQ 'HK'  AND KEYWORD_SET(bf) THEN $
      if_name = concat_dir ( ies_get_dir('MF',/POLAR), 'PO_CEP_RAL_MF_'+cep_get_lzday(file_date)) $
    ELSE IF file_root EQ 'HK' THEN $
      if_name = concat_dir ( ies_get_dir('HK',/POLAR), 'PO_CEP_RAL_'+file_label) $
    ELSE IF file_root EQ 'IES' THEN $
      if_name = concat_dir ( ies_get_dir('IES',/POLAR), 'PO_CEP_RAL_'+mode+'_'+file_label) $
    ELSE IF file_root EQ 'IPS' THEN $
      if_name = concat_dir ( ies_get_dir('IPS',/POLAR), 'PO_CEP_RAL_'+mode+'_'+file_label)  $
    ELSE IF file_root EQ 'HISTe' THEN $
      if_name = concat_dir ( ies_get_dir('HISTe',/POLAR), 'PO_CEP_RAL_'+mode+'_'+file_label)  $
    ELSE IF file_root EQ 'HISTp' THEN $
      if_name = concat_dir ( ies_get_dir('HISTp',/POLAR), 'PO_CEP_RAL_'+mode+'_'+file_label)  $
    ELSE $
      if_name = concat_dir ( ies_get_dir('IES',/POLAR), 'PO_CEP_RAL_'+mode+'_'+file_label)

    if_files = FINDFILE ( if_name+'.DAT*', COUNT=count )

    ; if IES then check if pedestal file also exists if needed

    IF file_root EQ 'IES' AND count GT 0 AND KEYWORD_SET(pedestal) THEN $
      dummy = FINDFILE ( concat_dir ( ies_get_dir('PD',/POLAR), 'PO_CEP_RAL_'+mode+'_'+file_label)+'.PEDESTALS*', COUNT=count )

    ; create if file

    IF KEYWORD_SET(force) OR count EQ 0 OR (STRPOS(mode,'SURVEY') LT 0 AND file_root EQ 'HK') $
       OR file_label EQ old_file_label THEN BEGIN

      ; read level one data
      ; arguments : filename of level 1 data
      ;             start record
      ;             end record
      ;             output level

      ; decompress file if necessary
      ; decompressed SH file must be in same directory as decompressed instrument file for get_cep_data
      ; so SH file must be in same compressed state as instrumnet file

      IF ies_test_gzip(l1files(f)) THEN BEGIN

        ; get the corresponding sh file name
        ; data file has name i.e PO_CEP_L1_IES_1996366_01.DAT
        ; sh file has name i.e PO_CEP_L1_SH_1996366.DAT

        shfile = l1_root + 'SH_' + file_date + '.DAT.gz'

        l1file = l1files(f)

        ; decompress data file

        ies_gunzip, l1file
reta
        ; decompress sh file into if directory

        IF shfile NE '' THEN ies_gunzip, shfile

      ENDIF ELSE l1file = l1files(f)

      PRINT, 'Reading level 1 file : ', l1file

      ; set start and stop records so that they always lie outside of file

      IF KEYWORD_SET ( start_time ) AND STRPOS(mode,'SURVEY') LT 0 THEN BEGIN
        st_time = anytim2utc(start_time)
        st_time = st_time.time/1000
      ENDIF ELSE st_time = 0

      IF KEYWORD_SET ( end_time ) AND STRPOS(mode,'SURVEY') LT 0 THEN BEGIN
        en_time = anytim2utc(end_time)
        en_time = en_time.time/1000
      ENDIF ELSE en_time = 100000

      ; get l1 data

      PRINT, '********************BEGINNING GET_CEP_DATA*************************'

      get_cep_data, l1file, st_time, en_time

      PRINT, '********************FINISHED GET_CEP_DATA**************************'

      ; delete decompressed file if necessary

      IF ies_test_gzip(l1files(f)) THEN BEGIN

        ; delete decompressed data file

        IF ies_spawn('rm ' + l1file) THEN MESSAGE, 'REMOVE FAILED'

        ; delete sh file into if directory

        IF shfile NE '' THEN IF ies_spawn('rm ' + shfile) THEN MESSAGE, 'REMOVE FAILED'

      ENDIF

      ; check if leave

      IF KEYWORD_SET(leave) THEN RETURN

      ; do current file

      IF STRPOS ( file_root, 'RATES' ) GE 0 THEN BEGIN

      ENDIF ELSE BEGIN

        ; reorganize level 1 data into RAL format

        PRINT, 'Reorganizing into RAL level 1 format'

        IF file_root EQ 'IES' THEN BEGIN

          cep_reorg, 'POLAR/CEPPAD/IES normal mode ', l1_mjulday, $
                      PLCPIES_time, PLCPIES_bytes, PLCPIES_Org, $
                      LONG(PLCPSpn_hed(0:3,*),0,(SIZE(PLCPSpn_hed))(2)), PLCPSpn_hed(6,*), $
                      input_header, input_data, $
                     'IES', mode, pedestal_info_array

          ; zero POL_CEP_IES COMMON block parameters

          PLCPIES_time       = 0
          PLCPIES_bytes      = 0
          PLCPIES_Org        = 0
          PLCPIES_Start_Time = 0
          PLCPIES_Stop_Time  = 0

        ENDIF ELSE IF file_root EQ 'IPS' THEN BEGIN

          cep_reorg, 'POLAR/CEPPAD/IPS normal mode ', l1_mjulday, $
                      PLCPIPS_time, PLCPIPS_bytes, PLCPIPS_Org,  $
                      LONG(PLCPSpn_hed(0:3,*),0,(SIZE(PLCPSpn_hed))(2)), PLCPSpn_hed(5,*), $
                      input_header, input_data,  'IPS', mode, pedestal_info_array

          ; zero POL_CEP_IES COMMON block parameters

          PLCPIPS_time       = 0
          PLCPIPS_bytes      = 0
          PLCPIPS_Org        = 0
          PLCPIPS_Start_Time = 0
          PLCPIPS_Stop_Time  = 0

        ENDIF ELSE IF file_root EQ 'HISTp' THEN BEGIN

          cep_reorg, 'POLAR/CEPPAD/HISTp normal mode ', l1_mjulday, $
                      PLCPHSTP_time, PLCPHSTP_bytes, PLCPHSTP_Org, $
                      LONG(PLCPSpn_hed(0:3,*),0,(SIZE(PLCPSpn_hed))(2)), PLCPSpn_hed(8,*), $
                      input_header, input_data,  $
                     'HSTP', mode, pedestal_info_array

          ; zero POL_CEP_HSTP COMMON block parameters

          PLCPHSTP_time       = 0
          PLCPHSTP_bytes      = 0
          PLCPHSTP_Org        = 0
          PLCPHSTP_Start_Time = 0
          PLCPHSTP_Stop_Time  = 0

        ENDIF ELSE IF file_root EQ 'HISTe' THEN BEGIN

          cep_reorg, 'POLAR/CEPPAD/HISTe normal mode ', l1_mjulday, $
                      PLCPHSTE_time, PLCPHSTE_bytes, PLCPHSTE_Org, $
                      LONG(PLCPSpn_hed(0:3,*),0,(SIZE(PLCPSpn_hed))(2)), PLCPSpn_hed(8,*), $
                      input_header, input_data, $
                     'HSTE', mode, pedestal_info_array

          ; zero POL_CEP_HSTE COMMON block parameters

          PLCPHSTE_time       = 0
          PLCPHSTE_bytes      = 0
          PLCPHSTE_Org        = 0
          PLCPHSTE_Start_Time = 0
          PLCPHSTE_Stop_Time  = 0

        ENDIF ELSE IF file_root EQ 'HK' THEN BEGIN

          IF KEYWORD_SET(bf) THEN $
            cep_hk, 'POLAR/CEPPAD/MF normal mode ', l1_mjulday, input_header, input_data, 'MF' $
          ELSE $
            cep_hk, 'POLAR/CEPPAD/HK normal mode ', l1_mjulday, input_header, input_data, 'HK'

        ENDIF

        ; check data monotonic
        list = cep_chk_monotonic(input_data.time)

        IF list(0) GE 0 THEN PRINT, 'WARNING, Spin times not monotonic : ', list

        ; get new file name

        file = if_name+'.DAT'

        ; check if first file or new file label

        IF file_label NE old_file_label THEN BEGIN

          ; save data in RAL format

          SAVE, file=file, input_header, input_data, /XDR, /COMPRESS

          PRINT, 'Created file : ' + file

          ; check if PEDESTAL output required

          IF KEYWORD_SET(pedestal) THEN BEGIN

            ; check if CLEAN data

            IF STRPOS ( mode, 'CLEAN') GE 0 THEN BEGIN

              ; get file name

              ped_file = concat_dir ( ies_get_dir('PD',/POLAR), 'PO_CEP_RAL_'+mode+'_'+file_label+'.PEDESTALS' )

              ; save data

              time = input_data.time
              luts = input_data.lut
              pedestal_info = pedestal_info_array(0:npedinfo-1,*,*,*)
              SAVE, file=ped_file, time, luts, pedestal_info, /XDR, /COMPRESS

              PRINT, 'Created file : ' + ped_file

              time = 0
              luts = 0
              pedestal_info = 0
              pedestal_info_array = 0

              ; gzip files if required

              ;IF KEYWORD_SET(gzip) THEN ies_gzip, ped_file

            ENDIF ELSE PRINT,'WARNING, NO PEDESTAL DATA OUTPUT SINCE NOT CLEAN DATA'

          ENDIF

          ; release memory asap

          input_data = 0

        ENDIF ELSE BEGIN

          ; restore previous data and append current

          ; copy current data

          header = input_header
          data   = input_data

          ; ungzip files if required
          ; NB file name is invented so will not have the .gz extension
          ; tmp_file will be created in tmp directory

          IF KEYWORD_SET(gzip) THEN BEGIN
            tmp_file = file + '.gz'
            ies_gunzip, tmp_file
          ENDIF ELSE tmp_file = file

          ; restore previous data

          PRINT, 'Restoring file : ' + tmp_file

          RESTORE, tmp_file

          ; remove tmp file if necessary

          IF KEYWORD_SET(gzip) THEN IF ies_spawn('rm ' + tmp_file) THEN MESSAGE, 'REMOVE FAILED'

          ; check compatability

          IF input_header.tstop GE header.tstart THEN $
            PRINT, 'WARNING data sets have non-monotonic times : ',input_header.tstop,header.tstart

          ; append current to previous
          ; arrays only extended for multiple files on the same day
          ; release memory asap

          input_header.npoints = input_header.npoints + header.npoints
          input_header.tstop   = header.tstop
          input_data           = [input_data, data]
          data = 0

          ; save data in RAL format into original directory

          SAVE, file=file, input_header, input_data, /XDR, /COMPRESS

          PRINT, 'Created file : ' + file

          ; check if PEDESTAL output required

          IF KEYWORD_SET(pedestal) THEN BEGIN

            ; check if CLEAN data

            IF STRPOS ( mode, 'CLEAN') GE 0 THEN BEGIN

              ; get file name

              ped_file = concat_dir ( ies_get_dir('PD',/POLAR), 'PO_CEP_RAL_'+mode+'_'+file_label+'.PEDESTALS' )

              ; ungzip files if required

              IF KEYWORD_SET(gzip) THEN BEGIN
                tmp_file = ped_file + '.gz'
                ies_gunzip, tmp_file
              ENDIF ELSE tmp_file = ped_file

              ; restore previous data

              time = 0
              luts = 0
              pedestal_info = 0

              PRINT, 'Restoring file : ' + tmp_file

              RESTORE, tmp_file

              ; remove tmp file if necessary

              IF KEYWORD_SET(gzip) THEN IF ies_spawn('rm ' + tmp_file) THEN MESSAGE, 'REMOVE FAILED'

              ; append current to previous

              pedestal_info = [[pedestal_info],[pedestal_info_array(0:npedinfo-1,*,*,*)]]

              ; save data

              time = input_data.time
              luts = input_data.lut
              SAVE, file=ped_file, time, luts, pedestal_info, /XDR, /COMPRESS
              time = 0
              luts = 0
              pedestal_info = 0
              pedestal_info_array = 0

              ; gzip files if required

              ;IF KEYWORD_SET(gzip) THEN ies_gzip, ped_file

            ENDIF ELSE PRINT,'WARNING, NO PEDESTAL DATA OUTPUT SINCE NOT CLEAN DATA'

          ENDIF

          ; release memory asap

          input_data = 0

        ENDELSE

        ; gzip files if required

        ;IF KEYWORD_SET(gzip) THEN ies_gzip, file

     ENDELSE ; end of non RATES case

     ; set up old file_label

     old_file_label = file_label

   ENDIF ELSE BEGIN

      file = if_files(0)

      PRINT, 'SURVEY PLOT INTERMEDIATE FILE ALREADY EXISTS : '+file

    ENDELSE

  ENDFOR ; end read level 1 files

END


