;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_luts
;
; Purpose     :	Sets energy channel to bin look up tables for each sample in period of interest.
;
; Explanation : The spin header records contain a time and lut number for each valid spin.
;               There may be more spin header records than spin records or vice versa.
;               This routine matches the spin times of the two record sets and produces a
;               lut number for each sample.
;
; Use         : < sample_luts = cep_luts( input_times, lut_times, sample_times, luts, bad_sample_luts, $
;                                         experiment, no_of_samples, lut_changes, no_of_luts) >
;
; Inputs      : input_times     : LONARR giving times of spin records in milliseconds from
;                                 start of day in period of interest.
;               lut_times       : LONARR giving times of spin header records in milliseconds from start of day.
;               sample_times    : DBLARR giving TAI times of samples in period of interest.
;               luts            : BYTARR giving LUT table used for each spin from spin header record.
;               experiment      : STRING indicating experiment i.e 'IES', 'IPS', 'HSTP' or 'HSTE'.
;               no_of_samples   : LONG   giving no. of samples required in period of interest.
;               lut_changes     : STRARR(no_of_changes-1) time of each LUT change.
;               no_of_luts      : INT giving no. of LUTs.
;
; Opt. Inputs : None.
;
; Outputs     : sample_luts     : INTARR giving LUT table for each sample in period of interest.
;               bad_sample_luts : LONARR indices of samples with bad luts for which no spin header record is present.
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       :	None.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 18/9/96
;
; Modified    :	Version 0.1, MKC, 21/10/96.
;                 Replace mode with no_of_samples.
;               Version 0.2, MKC, 18/8/97.
;                 Added HIST LUT changes.
;                 Corrected bug setting IPS luts to zero.
;               Version 0.3, MKC, 5/2/99.
;                 Use lut_changes.
;                 NB. Previously did not check HISTp luts. Still do not check IPS luts.
;                 Corrected bug not checking first lut.
;               Version 0.4, 20/07/01
;                 Used ies runs. NB Previously could miss last run.
;
; Version     :	Version 0.4, 20/07/01
;-
;**********************************************************

FUNCTION cep_luts, input_times, lut_times, sample_times, luts, bad_sample_luts, experiment, no_of_samples, $
                   lut_changes, no_of_luts

  ; get no. of spins

  no_of_spins = N_ELEMENTS(input_times)

  ; adjust IPS luts to zero

  xluts = luts
  IF (experiment EQ 'IPS') THEN xluts[*] = 0

  ; initialize

  spin_luts = LONARR(no_of_spins)
  l = 0
  lastlut = xluts[0]
  bad_spin_luts = -1
  bad_sample_luts = -1

  ; loop through spin times locating each lut time if present

  FOR  k = 0L, no_of_spins-1 DO BEGIN

    input_time = ABS(input_times(k))

    going = 1

    WHILE going DO BEGIN

      IF l LT N_ELEMENTS(lut_times)-1 THEN BEGIN
        IF input_time GT lut_times(l) THEN l = l + 1 ELSE going = 0
      ENDIF ELSE going = 0

    ENDWHILE

    IF input_time EQ lut_times(l) THEN BEGIN

      ; check luts
      ; have pass for IPS

      IF xluts(l) GE no_of_luts THEN BEGIN

        MESSAGE,'Invalid LUT encountered : '+STRTRIM(xluts(l),2), /CONT

        spin_luts(k) = lastlut
        IF bad_spin_luts(0) GT 0 THEN bad_spin_luts = [bad_spin_luts,k] $
        ELSE bad_spin_luts = [k]

      ENDIF ELSE BEGIN

        spin_luts(k) = xluts(l)
        lastlut = xluts(l)

      ENDELSE

    ENDIF ELSE BEGIN

      spin_luts(k) = lastlut
      IF bad_spin_luts(0) GT 0 THEN bad_spin_luts = [bad_spin_luts,k] $
      ELSE bad_spin_luts = [k]

    ENDELSE

  ENDFOR

  IF no_of_samples LT no_of_spins THEN BEGIN

    ; sum over spins required
    ; select first lut of each spin block

    sample_luts = REBIN( spin_luts, no_of_samples, /SAMPLE)

    IF bad_spin_luts(0) GT 0 THEN BEGIN

      ; get list of bad samples

      bad_sample_luts = bad_spin_luts*no_of_spins/no_of_samples

      ; remove duplicate elements
      ; NB remembering to keep as array if only one element

      bad_sample_luts = bad_sample_luts([ies_runs(bad_sample_luts)])

    ENDIF

  ENDIF ELSE IF no_of_spins LT no_of_samples THEN BEGIN

    ; want sectors time aligned
    ; duplicate luts -- extend array using /SAMPLE

    sample_luts = REBIN( spin_luts, no_of_samples, /SAMPLE)

    ; check if any bad times

    IF bad_spin_luts(0) GT 0 THEN BEGIN

      ; set indices corresponding to sectors

      ns = no_of_samples/no_of_spins

      bad_sample_luts = REFORM ( REPLICATE(1,ns) # (ns*bad_spin_luts) + $
                         LINDGEN(ns) # REPLICATE(1,N_ELEMENTS(bad_spin_luts)), ns*N_ELEMENTS(bad_spin_luts) )

    ENDIF

  ENDIF ELSE BEGIN

    sample_luts = spin_luts
    bad_sample_luts = bad_spin_luts

  ENDELSE

  ; printout warning message

  IF bad_spin_luts(0) GT 0 THEN PRINT, 'Bad spin luts  = ', STRTRIM(N_ELEMENTS(bad_spin_luts),1) , '/', STRTRIM(no_of_spins,1)

  ; loop through changes adjusting LUTs

  FOR k = 0, N_ELEMENTS(lut_changes)-1 DO BEGIN

    list = WHERE ( sample_times GT utc2tai(lut_changes[k]), count)

    IF count GT 0 THEN sample_luts(list) = sample_luts(list) + no_of_luts

  ENDFOR

  RETURN, sample_luts

END
