;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_lztoif
;
; Purpose     :	Converts CEPPAD level 0 data to RAL intermediate format.
;               Acts as a wrapper for cep_lztol1 and cep_l1toif.
;
; Explanation : Prompts for CEPPAD level 0 file names.
;               Converts each level 0 file to level 1 files.
;               Converts the level 1 files indicated by the SPECIFIER to the RAL
;               intermediate format.
;
; Use         : < cep_lztoif, SPECIFIER=specifier, MODE=mode, T0=start_time, T1=end_time,
;                             PEDESTAL=pedestal, /GZIP >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : CEPPAD level 1 and RAL intermediate format files.
;
; Opt. Outputs:	None.
;
; Keywords    : SPECIFIER  : String which determines the CEPPAD level 1 file name used to
;                            produce the RAL intermediate format file.
;                            It gives the CEPPAD instrument and type of level 1 data and
;                              'IES_', 'IPS_', 'HISTe', 'HISTp'.
;                            Default is 'IES_' .
;                            At present only the normal mode data type is supported.
;                            The date used for the level 1 file is determined from the level 0
;                            file name.
;               MODE       : String giving RAL intermediate data output required.
;               T0         : STRING giving start time of period of interest in level 1
;                            data. Example format = '10:00:00', ignores day.
;               T1         : STRING giving end   time of period of interest in level 1
;                            data. Example format = '11:00:00', ignores day.
;               FILE       : STRING to receive intermediate format file name.
;               PEDESTAL   : Flag indicating pedestal output required.
;               GZIP       : indicates compression of output files (Unix only) required.
;               ALL        : FLAG indicating that all LZ files in specified directory should be processed.
;                                 Avoids use of pickfile.
;               COPY       : Indicates should copy OR and AT files from CDROM.
;
; Calls       :	cep_lztol1, cep_l1toif, cep_get_l1day, strip_dirname.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 12/1/96
;
; Modified    :	Version 0.1, MKC, 24/2/97
;                 Added PEDESTAL keyword.
;                 Changed to use _EXTRA form to avoid problem with keyword values being passed back out of
;                 cep_l1toif when more than one file.
;               Version 0.2, 21/5/97.
;                            Added decompression of .gz or .z files.
;                            Added GZIP keyword.
;               Version 0.3, 16/12/99
;                            Removed LEAVE and EXISTS keywords.
;               Version 0.4, 03/10/03
;                            Added /ALL keyword.
;                            Added /COPY keyword.
;
; Version     :	0.4, 03/10/03
;-
;**********************************************************

PRO cep_lztoif, SPECIFIER=specifier, GZIP=gzip, ALL=all, COPY=copy, _EXTRA=extra

  ; do level 0 to level 1 conversion

  lzfiles = cep_lztol1(/GZIP, ALL=all, COPY=copy)

  IF lzfiles(0) EQ '' THEN RETURN

  ; set default to normal mode IES data

  IF NOT KEYWORD_SET(specifier) THEN specifier = 'IES_'

  ; loop through level 0 files

  FOR f = 0, N_ELEMENTS(lzfiles)-1 DO BEGIN

    ; do level 1 to RAL intermediate format conversion

    cep_l1toif, specifier + cep_get_l1day ( strip_dirname(lzfiles(f)) ), GZIP=gzip, _EXTRA=extra

  ENDFOR

END
