;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_lztol1
;
; Purpose     :	Provides wrapper for CEPPAD level 0 to level 1 extraction routine.
;
; Explanation : Uses environment variables for level 0 and level 1 directories.
;               Allows selection of level 0 file. If level 0 file contains wildcard
;               characters then may process multiple level 0 files.
;               Also creates file containing ephemeris data structure from cdf file.
;               Allows level zero file to be compressed.
;
; Use         : <lzfiles = cep_lztol1(/GZIP, LZ_VERSION=lz_version)>
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : Creates CEPPAD level 1 data files.
;               lzfiles : CEPPAD level 0 file names processed into level 1 files.
;
; Opt. Outputs:	None.
;
; Keywords    : GZIP       : indicates compression of output files (Unix only) required.
;               LZ_VERSION : INT level zero file version number
;               ALL        : FLAG indicating that all LZ files in specified directory should be processed.
;                                 Avoids use of pickfile.
;               COPY       : Indicates should copy OR and AT files from CDROM.
;
; Calls       :	cep_get_lz_files, make_level_1, cep_get_or, cep_get_at, cep_get_l1_files, ies_gzip,
;               ies_test_gzip, ies_gunzip.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 12/2/96
;
; Modified    :	Version 0.1, MKC, 13/06/96
;                 Modified call to make_level_1 according to new SW version.
;               Version 0.2, MKC, 18/11/96
;                 Added read ephemeris CDF file.
;               Version 0.3, MKC, 19/12/96
;                 Changed so that finds ephemeris file even if version number different from lz data file.
;               Version 0.4, MKC, 19/2/97
;                 Modified so that returns ephemeris file type.
;                 Closes all logical units to cope with bug in make_level_1.
;               Version 0.5, MKC, 9/5/97
;                 Added /XDR keyword to SAVE.
;                 Added search for ephemeris files in pre directory if def directory fails.
;                 Added GZIP keyword and used ies_gzip.
;                 Added cep_get_ephem.
;                 Allowed lz file to be compressed.
;               Version 0.6, MKC, 21/8/97
;                 Changed so that processes all ephemeris cdf files found.
;                 Changed so that always processes level zero files above version 1.
;               Version 0.7, MKC, 1/9/98
;                 Changed so that does not decompress level zero if not l1 already exists.
;               Version 0.8, MKC, 6/10/98
;                 Replaced cep_get_ephem with cep_get_or and added cep_get_at.
;               Version 0.9, MKC, 8/2/99
;                 Passed out level zero file version number for each file.
;               Version 1.0, 13/12/99
;                 Checked for Y2K compatability.
;                 Removed strip_dirname.
;               Version 1.1, 11/07/02
;                 Used ies_spawn.
;               Version 1.2, 03/10/03
;                 Added /ALL keyword.
;                 Added /COPY keyword.
;               Version 1.3, 09/10/03
;                 Changed to ies get dir.
;                 Changed printout.
;               Version 1.4, 08/01/04
;                 Stopped gzipping of existing gzipped L1 files.
;                 Added printout if not version 1 LZ file.
;
; Version     :	Version 1.4, 08/01/04
;-
;**********************************************************

FUNCTION cep_lztol1, GZIP=gzip, LZ_VERSION=lz_version, ALL=all, COPY=copy

  ; process ephemeris files

  PRINT, 'Processing cdf files'

  IF KEYWORD_SET(copy) THEN BEGIN

    cep_get_or

    cep_get_at

  ENDIF

  ; get level 0 files

  lzfiles = cep_get_lz_files(ALL=all)

  IF lzfiles(0) EQ '' THEN RETURN, ''

  ; print info

  PRINT, 'Converting level 0 files to level 1 : '

  FOR f = 0,  N_ELEMENTS(lzfiles)-1 DO PRINT, lzfiles(f)

  ; set up array to hold lz file version numbers

  lz_version = INTARR(N_ELEMENTS(lzfiles))

  ; change to level 1 directory

  CD, ies_get_dir('L1',/POLAR), CURRENT=currentdir

  ; loop through level 0 files

  FOR f = 0,  N_ELEMENTS(lzfiles)-1 DO BEGIN

    PRINT, 'Converting level 0 file : ', lzfiles[f]

    lzfile = lzfiles[f]

    ; check if l1 files already exist and level zero version number
    ; if all required files exist do not recreate
    ; NB possible that only the first one of multiple l1 files for this day exist

    ies_l1files = cep_get_l1_files('IES_' + cep_get_l1day ( lzfile, VERSION=lz_vs ))
    ips_l1files = cep_get_l1_files('IPS_' + cep_get_l1day ( lzfile ))
    histp_l1files = cep_get_l1_files('HISTp_' + cep_get_l1day ( lzfile ))
    histe_l1files = cep_get_l1_files('HISTe_' + cep_get_l1day ( lzfile ))
    sh_l1files = cep_get_l1_files('SH_' + cep_get_l1day ( lzfile ))
    hk_l1files = cep_get_l1_files('HK_' + cep_get_l1day ( lzfile ))

    lz_version[f] = lz_vs

    ; get explanation

    explanation = ''

    IF ies_l1files[0] EQ '' THEN explanation = explanation + ', IES'

    IF ips_l1files[0] EQ '' THEN explanation = explanation + ', IPS'

    IF histp_l1files[0] EQ '' THEN explanation = explanation + ', HISTp'

    IF histe_l1files[0] EQ '' THEN explanation = explanation + ', HISTe'

    IF sh_l1files[0] EQ '' THEN explanation = explanation + ', SH'

    IF explanation NE '' THEN explanation = 'NO' + explanation + ' L1 FILES'

    IF lz_version[f] GT 1 THEN BEGIN

      IF explanation NE '' THEN explanation = explanation + ', '

      explanation = explanation + 'NEW VERSION OF LZ FILE'

    ENDIF

    ; create L1 fif necessary

    IF explanation NE '' THEN BEGIN

      ; print explanation

      PRINT, explanation

      ; decompress file if necessary

      IF ies_test_gzip(lzfile) THEN ies_gunzip, lzfile

      ; create level 1 files
      ; arguments : filename of level 0 data
      ;             dummy
      ;             dummy
      ;             output level

      PRINT, '********************BEGINNING MAKE_LEVEL_1*************************'

      make_level_1, lzfile

      PRINT, '********************FINISHED MAKE_LEVEL_1**************************'

      ; close all open files and free logical units
      ; believe make_level_1 leaves some logical units assigned

      CLOSE, /ALL

      ; gzip files if required

      IF KEYWORD_SET(gzip) THEN BEGIN

        l1files = cep_get_l1_files('*' + cep_get_l1day ( lzfile ))

        ; NB if forced to create L1 file then gzipped file may already be present

        list = WHERE(STRPOS(STRUPCASE(l1files),'.GZ') LT 0, count)

        FOR k=0, count-1 DO ies_gzip, l1files[list[k]]

      ENDIF

      ; delete decompressed file if necessary
      ; ie if lzfiles[f] is compressed a temporary uncompressed version was created :
      ; delete this.

      IF ies_test_gzip(lzfiles(f)) THEN BEGIN

        ; delete decompressed data file

        IF ies_spawn('rm ' + lzfile) THEN MESSAGE, 'REMOVE FAILED'

      ENDIF

    ENDIF ELSE PRINT, 'LEVEL 1 FILES EXISTS, WILL NOT RECREATE'

  ENDFOR

  ; change back to old directory

  CD, currentdir

  ; return level 0 file name

  RETURN, lzfiles

END
