;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_movie
;
; Purpose     :	Produces png/gif files for movie.
;
; Explanation : Using survey plot data extracts data for active part of each orbit and creates individual
;               instrument png/gif files. SURVEY plots cover UT days during which there may be one to two
;               orbits. These are extracted and where possible abutted to previous data sets.
;
; Use         : < if_files = cep_movie ( /L1, /RF ) >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : if_files : STRARR containing names of IES intermediate format files used.
;               IES, IPS png/gif files i.e 'mov_ips_19970322_0.png'
;
;
; Opt. Outputs:	CEPPAD level 1 and RAL intermediate format files.
;
; Keywords    : /L1         = Specifies that the l1 files already exist.
;               /RF         = Specifies that the if files already exist.
;                                if STRING or STRARR then assumes file names
;               /NOCT       = does not load default color table
;               /IES        = specifies IES data used
;               /IPS        = specifies IPS data used
;               /HISTe      = specifies HISTe data used
;               /HISTp      = specifies HISTp data used
;               /DEBUG      = STOPs in routine.
;               /WINDOWS    = plots output in x window.
;               /PS         = plots output as postscript file
;                                if string then assumes it is a file name
;               /PNG        = plots output as PNG file
;                                if string then assumes it is a file name
;               /GIF        = plots output as GIF file
;                                if string then assumes it is a file name
;               /BWPS       = plots output as black and white postscript file
;                                if string then assumes it is a file name
;               /CHECK      = checks first if decompressed file already exists.
;               /REMOVE     = Remove tmp file created.
;               /ALL        = FLAG indicating that all LZ files in specified directory should be processed.
;                                  Avoids use of pickfile.
;               /COPY       = Indicates should copy OR and AT files from CDROM.
;
;
; Common      :	ies_draw_widget_block.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 9/5/97
;                 Adapted from cep survey.
;
; Modified    :	Version 0.1, 29/8/97.
;                 Added elo, ehi to output_info
;               Version 0.1, MKC, 10/9/97.
;                 Added error return to top level.
;                 Changed format of if file name.
;               Version 0.2, MKC, 27/11/97.
;                 Use ies_restore_gzip.
;               Version 0.3, MKC, 18/2/98.
;                 Use IES/IPS/HISTE/HISTP keywords.
;                 Added DEBUG and output keywords.
;               Version 0.4, MKC, 7/4/98.
;                 Added CHECK keyword.
;               Version 0.5, MKC, 19/5/98.
;                 Fixed bug with output file.
;               Version 0.6, 7/10/98
;                 Added Magnetic field and attitude arrays to common block.
;                 Changed color table.
;                 Added COLTAB keyword to ies_window_plot.
;                 Added REMOVE keyword.
;               Version 0.7, 13/12/99
;                 Checked for Y2K compatability.
;               Version 0.8, 10/12/00
;                 Added PNG output
;               Version 0.9, 10/06/02
;                 Moved out cep_do_plot.
;               Version 1.0, 10/07/02
;                 Added set ncolors.
;                 Added JAVA keyword NB only applicable at present to daily plots.
;               Version 1.1, 23/07/02
;                 Removed code resetting color table..
;               Version 1.2, 02/10/03
;                 Removed JAVA keyword.
;                 Added /ALL keyword.
;                 Added /COPY keyword.
;                 Changed printout.
;
; Version     :	Version 1.2, 02/10/03
;-
;**********************************************************

FUNCTION cep_movie, L1=l1, RF=rf, WINDOWS=windows, PS=postscript, PNG=png, GIF=gif, BWPS=bwps, NOCT=noct, $
                    IES=ies, IPS=ips, HISTE=histe, HISTP=histp, DEBUG=debug, NOAXIS=noaxis, $
                    CHECK=check, REMOVE=remove, ALL=all, COPY=copy

   ; define common blocks used internally for draw widgets
   ; full_header   = structure containing original data header information
   ; full_data     = structure array containing original data
   ; header        = structure containing extracted data header information  ** used
   ; data          = structure array containing extracted data               ** used
   ; output_info   = structure containing plotting and processing info       ** used
   ; output_chans  = array containing processed channel positions
   ; output_array  = array containing processed data for plotting
   ; ids           = structure giving widget IDs
   ; dnfull        = co-ordinate conversion array for full window
   ; dnscroll      = co-ordinate conversion array for magnified windows
   ; files         = STRARR giving data file names
   ; ephem         = structure containing ephemeris data
   ; mf            = structure containing magnetic field data
   ; at            = structure containing attitude data

  COMMON ies_draw_widget_block, input_header, input_data, header, data, output_info, output_chans, output_array, $
                                ids, dnfull, dnscroll, files, ephem, mf, at

  ; come back out to top level unless debug set

  IF KEYWORD_SET(debug) THEN ON_ERROR, 0 ELSE ON_ERROR, 1

  ; set colour table and no of colors

  IF NOT KEYWORD_SET(noct) THEN coltab = 39 ; RAINBOW white

  dummy = ies_ncolors(/SET,PRIVATE=coltab)

  ; set default plot type

  IF NOT KEYWORD_SET(windows) AND NOT KEYWORD_SET(postscript) AND $
     NOT KEYWORD_SET(png) AND NOT KEYWORD_SET(gif) AND NOT KEYWORD_SET(bwps) THEN windows=1

  ; set file ending

  IF KEYWORD_SET(postscript) OR KEYWORD_SET(bwps) THEN file_end = '.ps' ELSE $
    IF KEYWORD_SET(png) THEN file_end = '.png' ELSE $
    IF KEYWORD_SET(gif) THEN file_end = '.gif' ELSE $
    file_end = ''

  ; set up default keyword

  IF NOT KEYWORD_SET(ies) AND NOT KEYWORD_SET(ips) AND $
     NOT KEYWORD_SET(histe) AND NOT KEYWORD_SET(histp) THEN ies = 1

  ; set up data names

  names = ['HISTe','IES','HISTp','IPS']

  ; get list of data used

  list  = WHERE([KEYWORD_SET(histe),KEYWORD_SET(ies),KEYWORD_SET(histp),KEYWORD_SET(ips)])

  ; get intermediate format files

  survey_files = cep_survey_files( L1=l1, RF=rf, DEBUG=debug, ALL=all, COPY=copy)

  ; loop through level 0 files

  FOR f = 0, N_ELEMENTS(survey_files(0,*))-1 DO BEGIN

    ; set up file name variables

    data_files  = survey_files([2,0,3,1],f)

    ; print message

    FOR k=0, N_ELEMENTS(list)-1 DO PRINT, names(list(k))+' file : ', data_files(list(k))

    ; set up output file root

    file_root ='mov_'
    FOR k = 0, N_ELEMENTS(list)-1 DO file_root = file_root + names(list(k)) +'_'
    file_root = file_root + STRMID(data_files(0),STRPOS(data_files(0),'.DAT')-8,8)

    ; restore all data for current day

    FOR data_no = 0, N_ELEMENTS(list)-1 DO BEGIN

      ies_restore_gzip, data_files(list(data_no)), input_header, input_data, CHECK=CHECK, REMOVE=remove

      ; store current data

      CASE data_no OF
        0 : BEGIN
              current_header_0 = input_header
              current_data_0   = input_data
            END
        1 : BEGIN
              current_header_1 = input_header
              current_data_1   = input_data
            END
        2 : BEGIN
              current_header_2 = input_header
              current_data_2   = input_data
            END
        3 : BEGIN
              current_header_3 = input_header
              current_data_3   = input_data
            END
      ENDCASE

    ENDFOR

    ; get ephemeris data for midpoint of range

    ephem = get_ephem ( (input_header.tstart+input_header.tstop)/2.0 )

    ; zero attitude and magnetic field data

    at = 0
    mf = 0

    ; check ephemeris data present

    IF N_TAGS(ephem) GT 0 THEN BEGIN

      ; set up flist

      IF N_ELEMENTS(flist) EQ 0 THEN flist = f ELSE flist = [flist,f]

      ; get time ranges for data

      time_ranges = eph_alt_range ( ephem )

      PRINT, 'TIME RANGES = ', anytim2cal(time_ranges)

      ; do ephemeris
      ; check if overlaps previous data set

      IF N_ELEMENTS(previous_ephem) GT 0 THEN BEGIN

        IF previous_time_range(1) GE input_header.tstart OR $
           time_ranges(0,0) LE previous_header_0.tstop THEN BEGIN

          ; concatenate ephemeris data

          ephem = [previous_ephem,ephem]

        ENDIF

      ENDIF

      ; check if overlaps previous data set

      IF N_ELEMENTS(previous_ephem) GT 0 THEN BEGIN

        ; loop over data sets

        FOR data_no = 0, N_ELEMENTS(list)-1 DO BEGIN

          ; get detectors

          CASE list(data_no) OF
            0 : detector = [0]                   ; HISTE data
            1 : detector = [0,0,0,1,0,0,0,0,0]   ; IES data for detector 4
            2 : detector = [0]                   ; HISTP data
            3 : detector = [0,0,0,0,1,0,0,0,0,0] ; IPS data for detector C2
          ENDCASE

          ; get current and previous data

          CASE data_no OF
            0 : BEGIN
                  input_header = current_header_0
                  input_data   = current_data_0
                  previous_header = previous_header_0
                  previous_data   = previous_data_0
                END
            1 : BEGIN
                  input_header = current_header_1
                  input_data   = current_data_1
                  previous_header = previous_header_1
                  previous_data   = previous_data_1
                END
            2 : BEGIN
                  input_header = current_header_2
                  input_data   = current_data_2
                  previous_header = previous_header_2
                  previous_data   = previous_data_2
                END
            3 : BEGIN
                  input_header = current_header_3
                  input_data   = current_data_3
                  previous_header = previous_header_3
                  previous_data   = previous_data_3
                END
          ENDCASE

          IF previous_time_range(1) GE input_header.tstart THEN BEGIN

            ; concatenate data sets

            input_data = [previous_data(0:N_ELEMENTS(previous_data)-2),input_data]
            input_header.tstart = previous_header.tstart
            input_header.npoints = input_header.npoints + previous_header.npoints - 1

            ; do plot

            cep_do_plot, previous_time_range, file_root+'_1'+file_end, output_file, detector, data_no, data_no LT (N_ELEMENTS(list)-1), NOAXIS=noaxis

          ENDIF ELSE IF time_ranges(0,0) LE previous_header.tstop THEN BEGIN

            ; concatenate data sets

            input_data = [previous_data(0:N_ELEMENTS(previous_data)-2),input_data]
            input_header.tstart = previous_header.tstart
            input_header.npoints = input_header.npoints + previous_header.npoints - 1

            ; do plot

            cep_do_plot, time_ranges(*,0), file_root+'_0'+file_end, output_file, detector, data_no, data_no LT (N_ELEMENTS(list)-1), NOAXIS=noaxis

          ENDIF ELSE IF time_ranges(1,0) LE input_header.tstop THEN BEGIN

            ; do plot

            cep_do_plot, time_ranges(*,0), file_root+'_0'+file_end, output_file, detector, data_no, data_no LT (N_ELEMENTS(list)-1), NOAXIS=noaxis

          ENDIF

        ENDFOR

      ENDIF ELSE BEGIN

        ; loop over data sets

        FOR data_no = 0, N_ELEMENTS(list)-1 DO BEGIN

          ; get detectors

          CASE list(data_no) OF
            0 : detector = [0]                   ; HISTE data
            1 : detector = [0,0,0,1,0,0,0,0,0]   ; IES data for detector 4
            2 : detector = [0]                   ; HISTP data
            3 : detector = [0,0,0,0,1,0,0,0,0,0] ; IPS data for detector C2
          ENDCASE

          ; get current and previous data

          CASE data_no OF
            0 : BEGIN
                  input_header = current_header_0
                  input_data   = current_data_0
                END
            1 : BEGIN
                  input_header = current_header_1
                  input_data   = current_data_1
                END
            2 : BEGIN
                  input_header = current_header_2
                  input_data   = current_data_2
                END
            3 : BEGIN
                  input_header = current_header_3
                  input_data   = current_data_3
                END
          ENDCASE

          IF time_ranges(1,0) LE input_header.tstop THEN BEGIN

            ; do plot

            cep_do_plot, time_ranges(*,0), file_root+'_0'+file_end, output_file, detector, data_no, data_no LT (N_ELEMENTS(list)-1), NOAXIS=noaxis

          ENDIF

        ENDFOR

      ENDELSE

      IF N_ELEMENTS(time_ranges) GT 2 THEN BEGIN

        ; loop over data sets

        FOR data_no = 0, N_ELEMENTS(list)-1 DO BEGIN

          ; get detectors

          CASE list(data_no) OF
            0 : detector = [0]                   ; HISTE data
            1 : detector = [0,0,0,1,0,0,0,0,0]   ; IES data for detector 4
            2 : detector = [0]                   ; HISTP data
            3 : detector = [0,0,0,0,1,0,0,0,0,0] ; IPS data for detector C2
          ENDCASE

          ; get current and previous data

          CASE data_no OF
            0 : BEGIN
                  input_header = current_header_0
                  input_data   = current_data_0
                END
            1 : BEGIN
                  input_header = current_header_1
                  input_data   = current_data_1
                END
            2 : BEGIN
                  input_header = current_header_2
                  input_data   = current_data_2
                END
            3 : BEGIN
                  input_header = current_header_3
                  input_data   = current_data_3
                END
          ENDCASE

          IF time_ranges(1,1) LE input_header.tstop THEN BEGIN

            ; do plot

            cep_do_plot, time_ranges(*,0), file_root+'_1'+file_end, output_file, detector, data_no, data_no LT (N_ELEMENTS(list)-1), NOAXIS=noaxis

          ENDIF

        ENDFOR

      ENDIF

      ; loop over data sets

      FOR data_no = 0, N_ELEMENTS(list)-1 DO BEGIN

        ; store current data

        CASE data_no OF
          0 : BEGIN
                previous_header_0 = current_header_0
                previous_data_0   = current_data_0
              END
          1 : BEGIN
                previous_header_1 = current_header_1
                previous_data_1   = current_data_1
              END
          2 : BEGIN
                previous_header_2 = current_header_2
                previous_data_2   = current_data_2
              END
          3 : BEGIN
                previous_header_3 = current_header_3
                previous_data_3   = current_data_3
              END
        ENDCASE

      ENDFOR

      ; store previous ephemeris and time ranges

      previous_ephem = ephem
      IF N_ELEMENTS(time_ranges) GT 2 THEN previous_time_range = time_ranges(*,1) ELSE previous_time_range = time_ranges(*,0)

    ENDIF

  ENDFOR

  ; leave with right file name

  files = [ '', data_files(N_ELEMENTS(list)-1)]

  ; set up processing info

  IF N_ELEMENTS(output_info) GT 0 THEN BEGIN
    output_info.minimum = -1
    output_info.maximum = -1
  ENDIF

  ; return file names

  IF N_ELEMENTS(flist) GT 0 THEN RETURN, survey_files(*,flist) ELSE BEGIN
    MESSAGE, 'NO VALID DATA FOUND', /CONT
    RETURN, -1
  ENDELSE

END

