;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_newtimes
;
; Purpose     :	Reorganizes times into samples required for the RAL intermediate format.
;
; Explanation : Input times are spin times in TAI format. Gets times for samples required for intermediate format.
;               NB If spins expanded into sectors then will interpolate for sector times.
;
; Use         : < cep_newtimes, spin_times, bad_spin_times, no_of_samples, sample_times, bad_sample_times >
;
; Inputs      : spin_times     : DBLARR giving TAI times of spins.
;               bad_spin_times : LONARR giving spin indeces for bad spin spin_times.
;               no_of_samples  : INT no. of samples required for this mode.
;
; Opt. Inputs : None.
;
; Outputs     : sample_times     : DBLARR giving TAI times of samples.
;               bad_sample_times : LONARR giving indeces for bad sample times.
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       :	None.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 12/1/96
;
; Modified    :	Version 0.1, MKC, 13/06/96.
;                 The level 0-1 SW has been modified to correct bad spin times.
;                 The part of cep_times which does this has been removed.
;                 The spin time has been removed from the argument list.
;                 The end of data flag has been hardcoded 100 ms from last spin.
;               Version 0.2, MKC, 18/9/96
;                 Corrected bad spin times list.
;                 Fixed expanding REBIN bug.
;                 Removed extra dummy record.
;                 Added no_of_sectors to argument list.
;               Version 0.3, MKC, 1/10/96
;                 Split into cep_times and cep_newtimes.
;               Version 0.4, 14/11/96
;                 Added sample_times and bad_sample_times.
;               Version 0.5, 20/07/01
;                 Used ies runs. NB Previously could miss last run.
;
; Version     :	Version 0.5, 20/07/01
;-
;**********************************************************

PRO cep_newtimes, spin_times, bad_spin_times, no_of_samples, sample_times, bad_sample_times

  ; get no of spins in period of interest

  no_of_spins = N_ELEMENTS(spin_times)

  ; initialize bad samples

  bad_sample_times = -1

  ; check if want to sample spins

  IF no_of_samples LT no_of_spins THEN BEGIN

    ; select first time of each spin block

    sample_times = REBIN( spin_times, no_of_samples, /SAMPLE)

    ; check if any bad times

    IF bad_spin_times(0) GE 0 THEN BEGIN

      ; get list of bad spin blocks

      bad_sample_times = bad_spin_times/(no_of_spins/no_of_samples)

      ; remove duplicate elements
      ; NB remembering to keep as array if only one element

      bad_sample_times = bad_sample_times([ies_runs(bad_sample_times)])

    ENDIF

  ENDIF ELSE IF no_of_samples GT no_of_spins THEN BEGIN

    ; work out dummy times for sectors
    ; NB last no_of_sectors times will be constant

    sample_times = REBIN( spin_times, no_of_samples)

    ; check if any bad times

    IF bad_spin_times(0) GE 0 THEN BEGIN

      bad_count = N_ELEMENTS(bad_spin_times)

      ; set indices corresponding to sectors

      no_of_sectors = no_of_samples/no_of_spins

      bad_sample_times = REFORM ( REPLICATE(1,no_of_sectors) # (no_of_sectors*bad_spin_times) + $
                                  LINDGEN(no_of_sectors) # REPLICATE(1,bad_count), no_of_sectors*bad_count )

    ENDIF

  ENDIF ELSE BEGIN

    sample_times = spin_times
    bad_sample_times = bad_spin_times

  ENDELSE

END
