;+
; Project     :	CLUSTER - RAPID
;
; Name        :	CEP_PEDESTAL_INFO
;
; Purpose     : Derives pedestal information and optionally subtracts pedestal from each sample.
;
; Explanation :	Derives pedestal information and optionally subtracts pedestal from each sample.
;
; Use         : < pedestals = cep_pedestal_info (ped, pp, array, ch_positions, pedestal_posn, max_pos) >
;
; Inputs      : ped              : FLTARR[npeds] estimate of pedestal contribution over pedestal channels.
;               pp               : INT pedestal shift index
;               array            : FLTARR(nbands) containing input data for each energy channel
;                                  May be changed on output if SUB flag set.
;               ch_positions     : INTARR(2,nbands) containing energy channel bin positions for given detector and LUT
;               pedestal_posn    : FLOAT result of fit to pedestal position.
;               max_pos          : INT peak pedestal channel
;
; Opt. Inputs : None.
;
; Outputs     : pedestals : FLTARR[6]
;                           0 : pedestal shift in bins.
;                           1 : distance of the centroid of all channels (excluding integral channel)
;                               from the fitted pedestal position
;                           2 : occupancy defined as the ratio of signal starting 2 channel above peak
;                               to total counts
;
; Opt. Outputs:	None.
;
; Keywords    : SUB    : FLAG if set subtracts pedestal from array.
;
; Written     :	Version 0.0, 26/11/03
;                 Separate from cep pedestals.
;
; Modified    :
;
; Version     :	Version 0.0, 26/11/03
;-
;**********************************************************

FUNCTION cep_pedestal_info, ped, pp, array, ch_positions, pedestal_posn, max_pos, SUB=sub

  ; get no. of energy bands

  nbands = N_ELEMENTS(array)

  ; set up no. of pedestal channels

  npeds = N_ELEMENTS(ped)

  ; set up pedestal info

  pedestals = FLTARR(6)

  ; pedestal shift

  pedestals[0] = [pp-50]/10.0

;            This wasnt much good as pedestal width tracked pedestal position
;            IF max_rarray GT 0 THEN pedestal_info(1,k,s,d) = tot_sarray/FLOAT(max_rarray)   ; pedestal width

            ; the values pedestal_info(1:5) are used in cep_ SW to produce the pedestal data file
            ; the data in that case is raw

  ; 1 = centroid

  denom = TOTAL(array[0:14<(nbands-1)])

  IF denom GT 0 THEN pedestals[1] = TOTAL( array[0:14<(nbands-1)]*TOTAL(ch_positions[*,0:14<(nbands-1)],1)/2.0 ) $
                         / denom - pedestal_posn

  ; subtract pedestals

  IF KEYWORD_SET(sub) THEN array[0:npeds-1] = (array[0:npeds-1] - ped) > 0

  ; 2 = occupancy
  ; measure used is ratio of signal counts to pedestal counts (omit integral channel)

  denom = TOTAL(ped)

  IF denom GT 0 AND max_pos+2 LT 14<(nbands-1) THEN pedestals[2] = TOTAL(array[max_pos+2:14<(nbands-1)])/denom


            ; 1 = sum of signal counts

;              pedestal_info(1,k,s,d) = TOTAL(output_array(s,d,max_pos+2:14,k))

              ; if data subtracted pedestal contribution will be missing
              ; pedestal_info(1,k,s,d) = TOTAL(output_array(s,d,max_pos:14,k))

              ; 2 = sum of pedestal counts
              ; this is what is subtracted as the pedestal contribution anyway

;              pedestal_info(2,k,s,d) = TOTAL(ped)

              ; 3 = centroid pedestal shift

;              denom = pedestal_info(2,k,s,d)
;              IF denom GT 0 THEN pedestal_info(3,k,s,d) = $
;                TOTAL( output_array(s,d,max_pos:14,k)*TOTAL(ch_positions(*,max_pos:14,d,lut),1)/2.0 )/denom

              ; 4 = total centroid

;              denom = TOTAL(output_array(s,d,0:14,k))
;              IF denom GT 0 THEN pedestal_info(4,k,s,d) = $
;                TOTAL( output_array(s,d,0:14,k)*TOTAL(ch_positions(*,0:14,d,lut),1)/2.0 ) / denom  - pedestal_posn

              ; 5 = signal centroid

;             denom = TOTAL(output_array(s,d,6:14,k))
;              IF denom GT 0 THEN pedestal_info(5,k,s,d) = TOTAL( ;output_array(s,d,6:14,k)*TOTAL(ch_positions(*,6:14,d,lut),1)/2.0 ) / $
;                                                          denom


  RETURN, pedestals

END