;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_pedestal_plot
;
; Purpose     :	Produces IES pedestal plot.
;
; Explanation : Produces IES pedestal plot.
;
; Use         : < if_files = cep_pedestal_plot(/L1, /RF, /WINDOWS, /PS, /PNG, /GIF, /BWPS, $
;                                    T0=input_start_time, T1=input_end_time) >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : if_files : STRARR containing names of IES intermediate format files used.
;               CEPPAD level 1 and RAL intermediate format files.
;
; Opt. Outputs:	None.
;
; Keywords    : /L1         = Specifies that the l1 files already exist.
;               /RF         = Specifies that the if files already exist.
;                                if STRING or STRARR then assumes file names
;               /WINDOWS    = plots output in x window.
;               /PS         = plots output as postscript file
;                                if string then assumes it is a file name
;               /PNG        = plots output as PNG file
;                                if string then assumes it is a file name
;               /GIF        = plots output as GIF file
;                                if string then assumes it is a file name
;               /BWPS       = plots output as black and white postscript file
;                                if string then assumes it is a file name
;               /DEBUG      = STOPs in routine.
;               /CHECK      = Check if decompressed file already exists before decompressing.
;               /WEB        = Indicates want WEB type file name : yymmdd.ext.
;               /REMOVE     = Indicates should remove temporary files
;               /ALL        = FLAG indicating that all LZ files in specified directory should be processed.
;                                  Avoids use of pickfile.
;               /COPY       = Indicates should copy OR and AT files from CDROM.
;
; Prev. Hist. :	Created from cep survey.
;
; Written     :	Version 0.0, Martin Carter, RAL, 28/11/97
;
; Modified    :	Version 0.1., 8/4/98
;                    Added ies_get_dir.
;               Version 0.2, 22/12/99
;                 Added DEBUG keyword.
;               Version 0.3, 10/12/00
;                 Added PNG output
;               Version 0.4, 10/07/02
;                 Added CHECK keyword.
;                 Added REMOVE keyword.
;                 Added JAVA keyword.
;                 Changed to a function.
;               Version 0.5, 02/10/03
;                 Changed JAVA keyword to WEB
;                 Used WEB output environment variable.
;                 Changed use of ies get dir.
;                 Added /ALL keyword.
;                 Added /COPY keyword.
;                 Changed ies get dir.
;                 Changed printout.
;
; Version     :	Version 0.5, 02/10/03
;-
;**********************************************************

FUNCTION cep_pedestal_plot, L1=l1, RF=rf, WINDOWS=windows, PS=postscript, PNG=png, $
  GIF=gif, BWPS=bwps, DEBUG=debug, CHECK=check, WEB=web, REMOVE=remove, ALL=all, COPY=copy

  ; come back out to top level

  ON_ERROR, 1

  ; get intermediate format files

  survey_files = cep_survey_files( L1=l1, RF=rf, DEBUG=debug, ALL=all, COPY=copy )

  IF survey_files(0,0) EQ '' THEN BEGIN
    MESSAGE, 'No files found for processing', /CONT
    RETURN, ''
  ENDIF

  PRINT, 'CREATING IES PEDESTAL PLOTS'

  FOR f = 0, N_ELEMENTS(survey_files(0,*))-1 DO BEGIN

    ; set up file name variables

    ies_file = survey_files(0,f)

    ; do pedestal plot

    ; get pedestal data file name

    str_pos = STRPOS(ies_file,'.DAT')

    IF str_pos GT 0 THEN BEGIN

      pedestal_file = concat_dir ( ies_get_dir('PD',/POLAR), strip_dirname(STRMID(ies_file,0,str_pos) + '.PEDESTALS*'))

      ; test exists

      pedestal_file = FINDFILE(pedestal_file,COUNT=count)

      IF count EQ 1 THEN BEGIN

        IF NOT KEYWORD_SET(web) THEN BEGIN

          ; get pedestal output directory

          dir = ies_get_dir('PED',/POLAR)

          IF KEYWORD_SET(postscript) THEN ped_postscript = concat_dir ( dir, 'ped_' + cep_get_lzday(ies_file) + '.ps') ELSE $
          IF KEYWORD_SET(bwps) THEN ped_bwps = concat_dir ( dir, 'ped_' + cep_get_lzday(ies_file) + '.ps') ELSE $
          IF KEYWORD_SET(png) THEN ped_png = concat_dir ( dir, 'ped_' + cep_get_lzday(ies_file) + '.png') ELSE $
          IF KEYWORD_SET(gif) THEN ped_gif = concat_dir ( dir, 'ped_' + cep_get_lzday(ies_file) + '.gif')

        ENDIF ELSE BEGIN

          ; get pedestal output directory

          dir = ies_get_dir('WEB',/POLAR)

          IF KEYWORD_SET(postscript) THEN ped_postscript = concat_dir ( dir, STRMID(cep_get_lzday(ies_file),2,6) + '.ps') ELSE $
          IF KEYWORD_SET(bwps) THEN ped_bwps = concat_dir ( dir, STRMID(cep_get_lzday(ies_file),2,6) + '.ps') ELSE $
          IF KEYWORD_SET(png) THEN ped_png = concat_dir ( dir, STRMID(cep_get_lzday(ies_file),2,6) + '.png') ELSE $
          IF KEYWORD_SET(gif) THEN ped_gif = concat_dir ( dir, STRMID(cep_get_lzday(ies_file),2,6) + '.gif')

        ENDELSE

        ; do pedestal plot

        ies_pedestal_plot, YRANGE=[-5,10], PS=ped_postscript, PNG=ped_png, $
          GIF=ped_gif, BWPS=ped_bwps, FILE=pedestal_file(0), CHECK=check, REMOVE=remove

      ENDIF ELSE MESSAGE, 'WARNING, NUMBER OF PEDESTAL FILES = '+STRTRIM(count,2), /CONT

    ENDIF

  ENDFOR

  RETURN, survey_files

END
