;+
; Project     :	POLAR - CEPPAD
;
; Name        :	CEP_PEDESTAL_SAMPLES
;
; Purpose     : Gets pedestal sample info for given detector and LUT.
;
; Explanation :	Gets pedestal sample info for given detector and LUT.
;
; Use         : < cep_pedestal_samples, npeds, ch_positions, detectors, lut_map, d, lut, temperature_index, $
;                          samples, max_sample_lists, pedestal_posn, pedestal_width, PRECALCULATE=precalculate>
;
; Inputs      : npeds             : INT no. of pedestal channels
;               ch_positions      : FLTARR(2, no_of_energies, ndetectors, no_of_luts) LUT definitions
;               detectors         : INTARR[ndetectors] detector no.s corresponding to chosen detectors 1:9
;               lut_map           : INTARR[no_of_luts] map of LUT into integration mode
;               d                 : INT detector no. 0-ndetectors-1
;               lut               : INT LUT no
;               temperature_index : INT index into temperature
;
; Opt. Inputs : None.
;
; Outputs     : samples          : FLTARR[npeds,nshifts] is an array containing shifted pedestal calibration
;                                  data sampled as for the real data. Only the first npeds energy channels are kept.
;               max_sample_list  : FLTARR(1+nshifts, npeds) indices of shifts with peak in channel 1+channel.
;                                  Element 0 contains no of indices.max_samples contains the index of the peak
;                                  for each shift. This is used to restrict the search for max cross-correlation
;                                  to samples with the same peak position as the data
;               pedestal_posn    : FLOAT result of fit to pedestal.
;               pedestal_width   : FLOAT result of fit to pedestal.
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, 21/11/03
;                 Separated from rcs pedestals.
;
; Modified    : Version 0.1, 12/12/03
;                 Performed JIT calculation.
;
; Version     :	Version 0.1, 12/12/03
;-
;**********************************************************

PRO cep_pedestal_samples, npeds, ch_positions, detectors, lut_map, d, lut, temperature_index, $
                          samples, max_sample_lists, pedestal_posn, pedestal_width

  COMMON pedestal_sample_block, tinfo, tsamples, tmax_sample_lists, tpedestal_posns, tpedestal_widths

  ; get calibration info samples for cross_correlation

  ; check if already calculated
  ; NB find get slightly different results if use the COMMON block values rather than
  ;    the direct values -- ie pedestal_posn is fitted value with different seeds each time
  ;    so will differ slightly and is used in ies shift

  IF temperature_index EQ tinfo.temperature_index AND tinfo.done[d,lut] THEN BEGIN

    ; get from COMMON block

    samples = tsamples[*,*,d,lut]

    max_sample_lists = tmax_sample_lists[*,*,d,lut]

    pedestal_posn = tpedestal_posns[d,lut]

    pedestal_width = tpedestal_widths[d,lut]

  ENDIF ELSE BEGIN

    ; calculate sampling

    ies_pedestal_samples, npeds, tinfo.nshifts, tinfo.offset, detectors[d], lut_map[lut], temperature_index, ch_positions[*,*,d,lut], $
                              samples, max_sample_lists, pedestal_posn, pedestal_width

    ; check if temperature changed

    IF temperature_index NE tinfo.temperature_index THEN BEGIN

      tinfo.temperature_index = temperature_index

      tinfo.done = 0

    ENDIF

    ; set up COMMON block

    tsamples[*,*,d,lut] = samples

    tmax_sample_lists[*,*,d,lut] = max_sample_lists

    tpedestal_posns[d,lut] = pedestal_posn

    tpedestal_widths[d,lut] = pedestal_width

    tinfo.done[d,lut] = 1

  ENDELSE

END