;+
; Project     :	CLUSTER - PEACE
;
; Name        :	CEP_PEDESTALS
;
; Purpose     : Locates pedestal bin position and optionally subtracts pedestal from each sample for POLAR.
;
; Explanation :	Associated with each column of the input image is a LUT number which defines the
;               bin positions of the y samples. This routine calculates pedestal shift required to a
;               fraction of a channel by cross-correlating the known pedestal profile with the sampled data.
;               Optionally a centroid algorithm may be used instead.
;               The absolute pedestal position is calculated by combining the pedestal shift with the pedestal position
;               measured in the calibration data. This is calculated putting the pedestal data on bin centres.
;               The pedestal contribution may then be subtracted if required. It is assumed that the input data
;               is undivided by energy.
;               The initial LUTs had no pedestal bins ie pedestal were off the bottom of the spectra.
;               In this case it is not possible to work out the changing pedestal position nor to subtract the pedestal.
;               A fixed pedestal position from the calibration data is used.
;
;               For POLAR, normalized the shifted pedestal samples to the data in the pedestal channels using a factor
;               calculated to give the least squares error over channels either side of the peak channel.
;
; Use         : < pedestals = cep_pedestals ( output_array, times, lut_list, detectors, nsectors, ndetectors, $
;                                             nbands, ch_positions, lut_map, pedestal_info, $
;                                             SUB=sub, CHISQ=chisq, CENTROID=centroid) >
;
; Inputs      : output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing input data
;               times         : DBLARR(nsamples) TAI times of samples
;               lut_list      : STRUCTURE ARRAY containing LUT run info.
;               detectors     : INTARR(ndetectors) detector numbers used 1-9
;               nsectors      : INT no. of sectors.
;               ndetectors    : INT no. of detectors.
;               nbands        : INT no. of energy channels.
;               ch_positions  : INTARR(2,nbands,ndetectors,nluts) containing energy channel bin positions for each detector and LUT
;               lut_map       : INTARR(nluts) maps lut nos into integration modes.
;
; Opt. Inputs : None.
;
; Outputs     : pedestals : FLTARR ( nsamples, nsectors, ndetectors) pedestal bin position
;               pedestal_info : FLTARR(nvalues, nsamples, nsectors, ndetectors)
;
; Opt. Outputs:	None.
;
; Keywords    : SUB      : indicates whether pedestal should be subtracted or not.
;               CHISQ    : flag indicating should use errors for weighting residuals.
;               CENTROID : flag indicating should use centroid calculation of pedestal position.
;
; Written     :	Version 0.0, Martin Carter, RAL, 13/11/96
;                 Adapted from ies rebin y.
;
; Modified    :	Version 0.1, 16/1/97.
;                 Combined with ies_subtract and forced data to be undivided.
;               Version 0.2, 16/1/97.
;                 Use new pedestal position and subtraction algorithm.
;                 Added pedestal shift output.
;               Version 0.3, 11/3/97.
;                 Added more info to pedestal shift array.
;               Version 0.4, 7/4/97.
;                 Fixed where peds not calculated.
;               Version 0.5, 19/5/97.
;                 Stopped pedestal calculation if LUT contains no pedestal bins.
;               Version 0.6, 22/8/97.
;                 Removed pedestals and lut_map tags and used routine ies get samples.
;               Version 0.7, 19/8/00
;                 Explicitly allowed for LUT=-1
;               Version 0.8, 20/9/00
;                 Added RAPID keyword.
;                 Corrected algorithm with CENTROID option selected.
;                 Changed argument list of ies get samples and used array of pedestal channels with
;                 keyword option.
;                 Found bug in normalization of subtracted pedestal for case where ERRORS=0.
;                 NB This does not affect CLEAN data since created with /ERRORS set.
;               Version 0.9, 11/10/00
;                 Added more comments.
;                 NB changed ies get samples so uses bin centres.
;                 NB processing with ERRORS set probably has a bug due to normalization to 1.
;                 Passed lut_map in ies_pedestals argument list instead of picking up from calibration file.
;               Version 1.0, 20/07/01
;                 Used ies runs.
;               Version 1.1, 01/10/03
;                 Removed RAPID keyword -- this routine only applies to POLAR data.
;                 Renamed routine ies get pedestal.
;                 Added routine cep temperature index.
;                 Changed argument list of ies get pedestal.
;                 Renamed from ies pedestals.
;               Version 1.2, 24/10/03
;                 Changed argument to lut_list.
;               Version 1.3, 21/11/03
;                 Used ies_get_sample_lists.
;                 Removed PED_CHANS keyword.
;                 Updated ies get samples argument list.
;                 Used ies do cross correlation and ies do centroid.
;                 Changed ERRORS keyword to CHISQ.
;                 Made qarray nbands wide.
;                 Used cep pedestal info.
;                 Added ies get pedestal calibration.
;               Version 1.4, 04/12/03
;                 Used cep setup pedestal calibration.
;                 Changed so that get a temperature index for the entire time range rather than for each LUT run.
;                 Move temperature index to cep setup pedestal calibration.
;                 Removed check that lut > 0 since performed in lut_list.
;               Version 1.5, 20/01/04
;                 Rearranged ies do cross correlation and ies do centroid.
;
; Version     :	Version 1.5, 20/01/04
;-
;**********************************************************

FUNCTION cep_pedestals, output_array, times, lut_list, detectors, nsectors, ndetectors, nbands, ch_positions, lut_map, $
                        pedestal_info, SUB=sub, CHISQ=chisq, CENTROID=centroid

  ; get no of samples

  npoints = N_ELEMENTS(times)

  ; the values pedestal_info(1:5) are used in cep_ SW to produce the pedestal data file
  ; the data in that case is raw
  ; do not perform calculations if data not raw

  pedestals = FLTARR(npoints, nsectors, ndetectors)

  pedestal_info = FLTARR(6,npoints, nsectors, ndetectors)

  ; set up no. of pedestal samples

  npeds = 6

  ; check if sensible no of channels for pedestal correction
  ; NB do not at present stop user from deleting energu channels and misapplying correction

  IF npeds-1 GE (SIZE(ch_positions))(2) THEN BEGIN
    MESSAGE, 'WARNING, INVALID NO. OF ENERGY CHANNELS FOR PEDESTAL CORRECTION', /CONT, /TRACEBACK
    RETURN, pedestals
  END

  ; check if pedestal calibration data defined yet

  cep_setup_pedestal_calibration, npeds, ndetectors, (SIZE(ch_positions))[4]

  ; calculate temperature index (need pedestal histogram data set up)
  ; NB previously got index for each LUT run rather than entire period

  temperature_index = cep_temperature_index(times)

  ; get pedestal bin position for each sample
  ; cross-correlate with known pedestal shape

  ; loop through runs

  FOR l = 0, N_ELEMENTS(lut_list)-1 DO BEGIN

    ; get lut and no. of points in run
    ; NB lut_list has .lut > 0

    lut = lut_list[l].lut

    ; previously calculated temperature index for each run
    ; temperature_index = cep_temperature_index(times[lut_list[l].st:lut_list[l].en])

    ; loop over each detector

    FOR d = 0, ndetectors-1 DO BEGIN

      ; do cross_correlation or centroid algorithm

      IF KEYWORD_SET(centroid) OR (lut LT 12) THEN BEGIN

        ; get calibration data pedestal position and width

        ies_fit_pedestal, ies_get_pedestal_histogram( lut_map[lut], detectors[d], temperature_index), pedestal_posn, pedestal_width

      ENDIF ELSE BEGIN

        cep_pedestal_samples, npeds, ch_positions, detectors, lut_map, d, lut, temperature_index, $
          samples, max_sample_lists, pedestal_posn, pedestal_width

      ENDELSE

      ; loop over each sample

      FOR k = lut_list[l].st, lut_list[l].en DO BEGIN

        ; loop over each sector

        FOR s = 0, nsectors-1 DO BEGIN

          ; find peak for each sample

          ; tarray is output_array for given sector, detector and sample [nbands]

          tarray = output_array[s,d,*,k]

          IF KEYWORD_SET(centroid) THEN BEGIN

            pp  = ies_do_centroid( npeds, tarray, ch_positions[*,*,d,lut], $
                    pedestal_posn, pedestal_width, PED=ped, MAX_POS=max_pos)

            ; get pedestal info

            pedestal_info[*,k,s,d] = cep_pedestal_info ( ped, pp, tarray, ch_positions[*,*,d,lut], pedestal_posn, max_pos, SUB=sub )

          ENDIF ELSE IF (lut LT 12) THEN BEGIN

            ; pedestal recovery not possible : pedestals out of sight
            ; use fixed pedestal position derived from calibration data
            ; do not allow pedestal subtraction

            dummy = MAX(tarray[0:14<(nbands-1)],max_pos)

            ; NB cant do subtraction

            pedestal_info[*,k,s,d] = cep_pedestal_info( tarray[0:npeds-1], 50, tarray, ch_positions[*,*,d,lut], $
                                        pedestal_posn, max_pos)

          ENDIF ELSE BEGIN

            pp = ies_do_cross_correlation(npeds, tarray, ch_positions[*,*,d,lut], $
                   max_sample_lists, samples, CHISQ=chisq, /POLAR, PED=ped, MAX_POS=max_pos)

            ; get pedestal info

            pedestal_info[*,k,s,d] = cep_pedestal_info ( ped, pp, tarray, ch_positions[*,*,d,lut], pedestal_posn, max_pos, SUB=sub )

          ENDELSE

          IF KEYWORD_SET(sub) THEN output_array[s,d,*,k] = tarray

          ; set up bin pedestal position and shift

          pedestals[k,s,d] = pedestal_posn - pedestal_info[0,k,s,d]

        ENDFOR ; loop over s

      ENDFOR ; loop over k

    ENDFOR ; loop over d

  ENDFOR ; loop over l

  RETURN, pedestals

END