;+
; Project     :	CLUSTER - PEACE
;
; Name        :	CEP_PEDESTALS
;
; Purpose     : Locates pedestal bin position and optionally subtracts pedestal from each sample for POLAR.
;
; Explanation :	Associated with each column of the input image is a LUT number which defines the
;               bin positions of the y samples. This routine calculates pedestal shift required to a
;               fraction of a channel by cross-correlating the known pedestal profile with the sampled data.
;               Optionally a centroid algorithm may be used instead.
;               The absolute pedestal position is calculated by combining the pedestal shift with the pedestal position
;               measured in the calibration data. This is calculated putting the pedestal data on bin centres.
;               The pedestal contribution may then be subtracted if required. It is assumed that the input data
;               is undivided by energy.
;               The initial LUTs had no pedestal bins ie pedestal were off the bottom of the spectra.
;               In this case it is not possible to work out the changing pedestal position not to subtract the pedestal.
;               A fixed pedestal position from the calibration data is used.
;
;               For POLAR, normalized the shifted pedestal samples to the data in the pedestal channels using a factor
;               calculated to give the least squares error over channels either side of the peak channel.
;
; Use         : < pedestals = cep_pedestals ( output_array, times, lut_list, detectors, nsectors, ndetectors, $
;                                             nbands, ch_positions, lut_map, pedestal_info, $
;                                             SUB=sub, ERRORS=errors, CENTROID=centroid, PED_CHANS=ped_chans) >
;
; Inputs      : output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing input data
;               times         : DBLARR(nsamples) TAI times of samples
;               lut_list      : STRUCTURE ARRAY containing LUT run info.
;               detectors     : INTARR(ndetectors) detector numbers used 1-9
;               nsectors      : INT no. of sectors.
;               ndetectors    : INT no. of detectors.
;               nbands        : INT no. of energy channels.
;               ch_positions  : INTARR(2,nbands,ndetectors,nluts) containing energy channel bin positions for each detector and LUT
;               lut_map       : INTARR(nluts) maps lut nos into integration modes.
;
; Opt. Inputs : None.
;
; Outputs     : pedestals : FLTARR ( nsamples, nsectors, ndetectors) pedestal bin position
;               pedestal_info : FLTARR(nvalues, nsamples, nsectors, ndetectors)
;                               0 : pedestal shift in bins.
;                               1 : distance of the centroid of all channels (excluding integral channel)
;                                   from the fitted pedestal position
;                               2 : occupancy defined as the ratio of signal starting 2 channel above peak
;                                   to total counts
;
; Opt. Outputs:	None.
;
; Keywords    : SUB      : indicates whether pedestal should be subtracted or not.
;               ERRORS   : flag indicating should use errors for weighting residuals.
;               CENTROID : flag indicating should use centroid calculation of pedestal position.
;
; Calls       :	ies_get_samples.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 13/11/96
;                 Adapted from ies rebin y.
;
; Modified    :	Version 0.1, 16/1/97.
;                 Combined with ies_subtract and forced data to be undivided.
;               Version 0.2, 16/1/97.
;                 Use new pedestal position and subtraction algorithm.
;                 Added pedestal shift output.
;               Version 0.3, 11/3/97.
;                 Added more info to pedestal shift array.
;               Version 0.4, 7/4/97.
;                 Fixed where peds not calculated.
;               Version 0.5, 19/5/97.
;                 Stopped pedestal calculation if LUT contains no pedestal bins.
;               Version 0.6, 22/8/97.
;                 Removed pedestals and lut_map tags and used routine ies_get_samples.
;               Version 0.7, 19/8/00
;                 Explicitly allowed for LUT=-1
;               Version 0.8, 20/9/00
;                 Added RAPID keyword.
;                 Corrected algorithm with CENTROID option selected.
;                 Changed argument list of ies_get_samples and used array of pedestal channels with
;                 keyword option.
;                 Found bug in normalization of subtracted pedestal for case where ERRORS=0.
;                 NB This does not affect CLEAN data since created with /ERRORS set.
;               Version 0.9, 11/10/00
;                 Added more comments.
;                 NB changed ies_get_samples so uses bin centres.
;                 NB processing with ERRORS set probably has a bug due to normalization to 1.
;                 Passed lut_map in ies_pedestals argument list instead of picking up from calibration file.
;               Version 1.0, 20/07/01
;                 Used ies runs.
;               Version 1.1, 01/10/03
;                 Removed RAPID keyword -- this routine only applies to POLAR data.
;                 Renamed routine ies get pedestal.
;                 Added routine cep temperature index.
;                 Changed argument list of ies get pedestal.
;                 Renamed from ies pedestals.
;               Version 1.2, 24/10/03
;                 Changed argument to lut_list.
;
; Version     :	Version 1.2, 24/10/03
;-
;**********************************************************

FUNCTION cep_pedestals_old, output_array, times, lut_list, detectors, nsectors, ndetectors, nbands, ch_positions, lut_map, $
                        pedestal_info, SUB=sub, ERRORS=errors, CENTROID=centroid, PED_CHANS=ped_chans

  ; get no of samples

  npoints = N_ELEMENTS(times)

  ; set up no of samples around peak to check residuals

  n=1
  ns = 2*n+1

  pedestals = FLTARR(npoints, nsectors, ndetectors)
  pedestal_info = FLTARR(6,npoints, nsectors, ndetectors)

  ; set up no. of pedestal samples

  IF NOT KEYWORD_SET(ped_chans) THEN BEGIN

    npeds = 6

    ped_chans = INDGEN(npeds)

  ENDIF ELSE BEGIN

    npeds = N_ELEMENTS(ped_chans)

  ENDELSE

  ; check if sensible no of channels for pedestal correction
  ; NB do not at present stop user from deleting energu channels and misapplying correction

  IF MAX(ped_chans) GE (SIZE(ch_positions))(2) THEN BEGIN
    MESSAGE, 'WARNING, INVALID NO. OF ENERGY CHANNELS FOR PEDESTAL CORRECTION', /CONT, /TRACEBACK
    RETURN, pedestals
  END

  ; get pedestal bin position for each sample
  ; cross-correlate with known pedestal shape

  ; loop through runs

  FOR l = 0, N_ELEMENTS(lut_list)-1 DO BEGIN

    ; get lut and no. of points in run

    lut = lut_list[l].lut

    ; set time range

    time_range = times[lut_list[l].st:lut_list[l].en]

    ; loop over each detector

    FOR d = 0, ndetectors-1 DO BEGIN

      ; check if pedestal recovery not possible

      IF (lut LT 12) THEN BEGIN

        ; pedestals out of sight
        ; use fixed pedestal position derived from calibration data
        ; do not allow pedestal subtraction

        ; get calibration data pedestal position and width

        ies_fit_pedestal, ies_get_pedestal_histogram( lut, detectors[d], lut_map, cep_temperature_index(time_range)), pedestal_posn, pedestal_width

        ; loop over each sample

        FOR k = lut_list[l].st, lut_list[l].en DO BEGIN

          ; loop over each sector

          FOR s = 0, nsectors-1 DO BEGIN

            ; set up bin pedestal position and shift

            pedestals(k,s,d) = pedestal_posn
            pedestal_info(0,k,s,d) = 0.0                        ; pedestal shift

            ; the values pedestal_info(1:5) are used in cep_ SW to produce the pedestal data file
            ; the data in that case is raw

            ; 1 = centroid

            denom = TOTAL(output_array(s,d,0:14<(nbands-1),k))
            IF denom GT 0 THEN pedestal_info(1,k,s,d) = $
              TOTAL( output_array(s,d,0:14<(nbands-1),k)*TOTAL(ch_positions(*,0:14<(nbands-1),d,lut),1)/2.0 ) $
                / denom - pedestal_posn

            ; 2 = occupancy

            dummy = MAX(output_array(s,d,0:14<(nbands-1),k),max_pos)
            denom = TOTAL(output_array(s,d,0:14<(nbands-1),k))
            IF denom GT 0 AND max_pos+2 LT 14<(nbands-1) THEN pedestal_info(2,k,s,d) = $
              TOTAL(output_array(s,d,max_pos+2:14<(nbands-1),k))/denom

          ENDFOR ; loop over s

        ENDFOR ; loop over k

      ENDIF ELSE BEGIN

        ; do cross_correlation or centroid algorithm

        ; get faster access copies of data
        ; qarray is output_array for  [nsectors,npeds,npoints]

        qarray = REFORM( output_array(*,d,ped_chans,*), nsectors, npeds, npoints)

        IF NOT KEYWORD_SET(centroid) THEN BEGIN

          ; get pedestal calibration data

          pdata = ies_get_pedestal_histogram( lut, detectors[d], lut_map, cep_temperature_index(time_range))

          ; qsamples is an array containing shifted pedestal calibration data sampled as for the real data
          ; only the first npeds samples are kept

          qsamples = ies_get_samples ( pdata, npeds, ch_positions[*,*,d,lut], $
                          pedestal_posn, pedestal_width )
max_sample_lists = ies_get_sample_lists(qsamples)
          ; max_qsamples contains the index of the peak for each shift
          ; this is used to restrict the search for max cross-correlation
          ; to samples with the same peak position as the data

          max_qsamples = FLTARR(N_ELEMENTS(qsamples[0,*]))

          ; loop over qsample bins

          FOR b = 0, N_ELEMENTS(qsamples[0,*])-1 DO BEGIN

             dummy = MAX ( qsamples[*,b], max_pos)

             max_qsamples[b] = max_pos

          ENDFOR

          ; get lists of qsamples indices at each maxima position
          ; if no counts then set list to [0]
          ; these are used as look up tables to speed up search for shifted
          ; samples with the same peak as the actual data

          list0 = WHERE( max_qsamples EQ 0, count) > 0
          IF npeds GT 1 THEN list1 = WHERE( max_qsamples EQ 1, count) > 0
          IF npeds GT 2 THEN list2 = WHERE( max_qsamples EQ 2, count) > 0
          IF npeds GT 3 THEN list3 = WHERE( max_qsamples EQ 3, count) > 0
          IF npeds GT 4 THEN list4 = WHERE( max_qsamples EQ 4, count) > 0
          IF npeds GT 5 THEN list5 = WHERE( max_qsamples EQ 5, count) > 0

        ENDIF ELSE BEGIN

          ; get calibration data pedestal position and width

         ies_fit_pedestal, ies_get_pedestal_histogram( lut, detectors[d], lut_map, cep_temperature_index(time_range)), pedestal_posn, pedestal_width

        ENDELSE

        ; loop over each sample

        FOR k = lut_list[l].st, lut_list[l].en DO BEGIN

          ; loop over each sector

          FOR s = 0, nsectors-1 DO BEGIN

            ; the values pedestal_info(1:5) are used in cep_ SW to produce the pedestal data file
            ; the data in that case is raw
            ; do not perform calculations if data not raw

            ; 1 = centroid

            denom = TOTAL(output_array(s,d,0:14<(nbands-1),k))
            IF denom GT 0 THEN pedestal_info(1,k,s,d) = $
              TOTAL( output_array(s,d,0:14<(nbands-1),k)*TOTAL(ch_positions(*,0:14<(nbands-1),d,lut),1)/2.0 ) $
                     / denom - pedestal_posn

            ; find peak for each sample

            ; get copy of low energies for sample
            ; qarray is output_array for  [nsectors,npeds,npoints]
            ; rarray is output_array for given sector and given sample [npeds]

            rarray = REFORM(qarray[s,*,k])

            ; find array maximum position

            max_rarray = MAX(rarray,max_pos)

            ; get indices for 2n+1 samples

            ; get n samples either side of peak

            istart = (max_pos-n) > 0
            iend   = (max_pos+n)<(npeds-1)
            inds   = istart+INDGEN(iend-istart+1)

;            ; get peak and max shoulder samples
;            IF max_pos EQ 0         THEN inds = [max_pos,max_pos+1] ELSE $
;            IF max_pos EQ (npeds-1) THEN inds = [max_pos-1,max_pos] ELSE $
;            IF rarray(max_pos-1) GT rarray(max_pos+1) THEN inds = [max_pos-1,max_pos] ELSE $
;               inds = [max_pos,max_pos+1]

;            ; get peak and left shoulder samples
;            IF max_pos EQ 0 THEN inds = [0,1] ELSE inds = [max_pos-1,max_pos]

            IF NOT KEYWORD_SET(centroid) THEN BEGIN

              ; find qsamples with peak at max_pos

              CASE max_pos OF
                0 : max_list = list0
                1 : max_list = list1
                2 : max_list = list2
                3 : max_list = list3
                4 : max_list = list4
                5 : max_list = list5
                ELSE : max_list = WHERE( max_qsamples EQ max_pos, count) > 0
              ENDCASE

              ; sarray is actual data for given sector and point for fixed no of channels either side of the peak

              sarray   = rarray[inds]

              ; tot_sarray is the sum of counts in sarray

              tot_sarray = TOTAL(sarray)

              ; form a 2-D array replicating sarray for each shifted sample with same peak

              sarray   = sarray#REPLICATE(1,N_ELEMENTS(max_list))

              ; ssamples are the shifted samples for channels either side of the peak with same peak as the data

              ssamples = qsamples[*,max_list]

              ; extract inds samples and normalize psamples to sarray

;              ; use peak

;              norm_sarray = max_rarray
;              norm_ssamples = REFORM(ssamples(max_pos,*))

              ; extract array

              ssamples = ssamples[inds,*]

;              ; use total

;              norm_sarray = tot_sarray
;              norm_ssamples = TOTAL(ssamples,1)

              ; find least squares fit to normalization factor for each pedestal position

              IF KEYWORD_SET(errors) THEN BEGIN
                norm_sarray   = TOTAL(ssamples,1)
                norm_ssamples = TOTAL(ssamples*ssamples/(sarray>1),1)
              ENDIF ELSE BEGIN
                norm_sarray   = TOTAL(sarray*sarray,1)
                norm_ssamples = TOTAL(ssamples*ssamples,1)
              ENDELSE

              ; combine normalization factors

              list = WHERE(norm_ssamples LE 0, count)

              IF count GT 0 THEN norm_ssamples(list) = -1

              norm_ssamples = norm_sarray / norm_ssamples

              ; apply normalization factors

    sssamples = ssamples * ( REPLICATE(1,N_ELEMENTS(inds)) # ies_get_normalization(sarray,ssamples,chisq=errors) )

              ssamples = ssamples * ( REPLICATE(1,N_ELEMENTS(inds)) # REFORM([norm_ssamples],N_ELEMENTS(max_list)) )
if total(sssamples-ssamples) NE 0 then stop
              ; take residuals with sampled pedestal array

              residuals =  sarray - ssamples

              ; get sum of squares

              IF KEYWORD_SET(errors) THEN BEGIN
                smsqs  = TOTAL((residuals*residuals)/(sarray>1),1)  ; takes into account Poisson errors
              ENDIF ELSE BEGIN
                smsqs  = TOTAL(residuals*residuals,1)
              ENDELSE

              ; find min

              dummy = MIN ( smsqs, p )

              ; index pp into qsamples

              pp = max_list[p]

              ; subtract pedestals

              ; calculate pedestal contribution

              ped = qsamples[*,pp]*norm_ssamples[p]

              IF KEYWORD_SET(sub) THEN BEGIN

                ; subtract and set output

;                output_array(s,d,ped_chans,k) = ABS(rarray - ped)
                output_array(s,d,ped_chans,k) = (rarray - ped) > 0

              ENDIF ; endif sub

            ENDIF ELSE BEGIN ; endif not centroid

              ; use CENTROID calculation to work out shift relative to pedestal calibration data

              ; sarray is actual data for given sector and point for fixed no of channels either side of the peak

              sarray = rarray[inds]

              ; calculate centroid of data samples

              posns  = TOTAL(ch_positions[*,inds,d,lut],1)/2.0

              tot_sarray = TOTAL(sarray)

              IF tot_sarray GT 0 THEN pos = TOTAL(sarray*posns)/tot_sarray ELSE pos = pedestal_posn

              ; fix pp as above

              pp = 10*(pedestal_posn+5-pos)

              ; subtract pedestals

              IF KEYWORD_SET(sub) THEN BEGIN

                ; get distances of band boundaries from centroid

                dists_0 = REFORM(ch_positions(0,ped_chans,d,lut))-pos
                dists_1 = REFORM(ch_positions(1,ped_chans,d,lut))-pos

                ; normalize distance to pedestal Gaussian widths

                dists_0 = dists_0/pedestal_width
                dists_1 = dists_1/pedestal_width

                ; get areas under Gaussian from -inf

                int_0 = GAUSSINT(dists_0)
                int_1 = GAUSSINT(dists_1)

                ; get Gaussian contribution to band

                gs = int_1 - int_0

                ; get required data samples

                ssamples = gs[inds]

                ; normalize required data samples

                IF KEYWORD_SET(errors) THEN BEGIN
                  norm_sarray   = TOTAL(ssamples)
                  norm_ssamples = TOTAL(ssamples*ssamples/(sarray>1))
                ENDIF ELSE BEGIN
                  norm_sarray   = TOTAL(sarray*sarray)
                  norm_ssamples = TOTAL(ssamples*ssamples)
                ENDELSE

                ; combine normalization factors

                IF norm_ssamples GT 0 THEN norm_ssamples = norm_sarray / norm_ssamples ELSE norm_ssamples = - norm_sarray

                ; calculate pedestal contribution

                ped = gs*norm_ssamples

                ; subtract and set output

;                output_array(s,d,ped_chans,k) = ABS(rarray - ped)
                output_array(s,d,ped_chans,k) = (rarray - ped) > 0

              ENDIF ; endif sub

            ENDELSE ; endif centroid

            ; set up bin pedestal position and shift

            pedestals[k,s,d] = pedestal_posn - (pp-50)/10.0

            pedestal_info[0,k,s,d] = [pp-50]/10.0                                           ; pedestal shift
             ps = ies_do_cross_correlation(npeds, rarray, ch_positions[*,*,d,lut], $
                max_sample_lists, qsamples, dummy, dummy, dummy, pedestal_posn, CHISQ=errors, /POLAR)
if ps[0] ne pedestal_info[0,k,s,d]  then stop

;            This wasnt much good as pedestal width tracked pedestal position
;            IF max_rarray GT 0 THEN pedestal_info(1,k,s,d) = tot_sarray/FLOAT(max_rarray)   ; pedestal width

            ; the values pedestal_info(1:5) are used in cep_ SW to produce the pedestal data file
            ; the data in that case is raw

            ; 2 = occupancy

            denom = TOTAL(ped)
            IF denom GT 0 AND max_pos+2 LT 14<(nbands-1) THEN pedestal_info(2,k,s,d) = $
              TOTAL(output_array(s,d,max_pos+2:14<(nbands-1),k))/denom

            ; 1 = sum of signal counts

;              pedestal_info(1,k,s,d) = TOTAL(output_array(s,d,max_pos+2:14,k))

              ; if data subtracted pedestal contribution will be missing
              ; pedestal_info(1,k,s,d) = TOTAL(output_array(s,d,max_pos:14,k))

              ; 2 = sum of pedestal counts
              ; this is what is subtracted as the pedestal contribution anyway

;              pedestal_info(2,k,s,d) = TOTAL(ped)

              ; 3 = centroid pedestal shift

;              denom = pedestal_info(2,k,s,d)
;              IF denom GT 0 THEN pedestal_info(3,k,s,d) = $
;                TOTAL( output_array(s,d,max_pos:14,k)*TOTAL(ch_positions(*,max_pos:14,d,lut),1)/2.0 )/denom

              ; 4 = total centroid

;              denom = TOTAL(output_array(s,d,0:14,k))
;              IF denom GT 0 THEN pedestal_info(4,k,s,d) = $
;                TOTAL( output_array(s,d,0:14,k)*TOTAL(ch_positions(*,0:14,d,lut),1)/2.0 ) / denom  - pedestal_posn

              ; 5 = signal centroid

;             denom = TOTAL(output_array(s,d,6:14,k))
;              IF denom GT 0 THEN pedestal_info(5,k,s,d) = TOTAL( ;output_array(s,d,6:14,k)*TOTAL(ch_positions(*,6:14,d,lut),1)/2.0 ) / $
;                                                          denom

          ENDFOR ; loop over s

        ENDFOR ; loop over k

      ENDELSE ; LUT >=12

;window,0
;plot,pedestal_info(0,*,0,d)

    ENDFOR ; loop over d

  ENDFOR ; loop over l

  RETURN, pedestals

END