
;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_reorg
;
; Purpose     :	Reorganizes CEPPAD level 1 data into RAL level 1 format
;               with fixed azimuthal and time resolution for each detector/energy combination.
;
; Explanation : The level 1 data may have different numbers of sectors per spin i.e
;               different azimuthal angular resolution, and different numbers of spins
;               per sample ie. different time resolution, for each detector/energy channel
;               combination.
;               This routine fixes on a number of particular azimuthal and time resolutions
;               and creates an intermediate data set at these resolutions by averaging
;               over or interpolating between sectors and spins. The interpolation performed
;               is simply to evenly divide counts in a sample between a number of sample points.
;               Times are converted to seconds during a Julian day.
;
;               If a spin record has an invalid time (<0) or invalid count (=255) then the
;               data is flagged as such.
;
;               The last record is flagged to indicate the end of this particular
;               block of data (flag=3).
;
;               If the mode begins with the string 'CLEAN' then the data will be corrected
;               for pedestal shift and the pedestal subtracted and calibration correction
;               applied and converted to count rates per keV.
;               Note that the correction is applied to each sector before summing over sectors
;               as required but after summing over samples.
;
; Use         : < cep_reorg, datatype, mjulday, spin_times, input_bytes, org, lut_times, luts, $
;                            output_header, output_data, experiment, mode, pedestal_info, $
;                            START_TIME=start_time, END_IIME=end_time, DETECTORS=detectors >
;
; Inputs      : datatype        : STRING giving data type.
;               mjulday         : LONG giving modified Julian day.
;               input_times     : LONARR giving times of samples in milliseconds from start of day.
;                       Longword array containing the time of
;                       day in milliseconds at the start of
;                       acquisition of each spin containing
;                       LEVEL-1 IPS data for the user requested
;                       time range, 'TIME1 - TIME2'
;               input_bytes     : BYTARR giving compressed data samples.
;	             	Longword array containing the de-compressed count
;			values accumulated through each 'sector', 'pixel'
;			(channel/energy-range) and spin as extracted from the
;			current Level-1 file portion.  16 pixels --
;			1 detector-channels x 16 energy-channels are allowed
;			for
;			Count values from a file section containing N spins of
;			data are packed into this single array in the following
;			order:
;
;			All existing count values from
;	==>		channel 1, energy 0, sector 0 from spins 0 to (N-1)
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;			    "         "      max sector for channel 1 energy 0
;			channel 2, energy 0, sector 0 from spins 0 to (N-1)
;			    "         "      sector 1         "
;			    "         "      sector 2         "
;						|
;			    "         "      max sector for channel 2 energy 0
;                       ETC.
;               org             : LONARR(*,*,*,*) organizational array.
;                       The plcp...org is a long word array whose dimension allocation follows:
;                       (4, num_channels, num_energies, plcpmax_tables ). This array defines the
;                       contents/organization of the 'plcp...bytes' array.  It is dimensioned to
;                       allow up to 'plcpmax_tables'
;
;                       If 'plcp..._org' = plcpips_org = lonarr(4, 10, 17, 2), then:
;                       The first element ( 4 parameters)  are values that describe each of
;                       possible IPS pixels ( 10 channels/ 17 energy-channels) which together
;                       describe the type of data and its location in the 'plcpips_bytes' array.
;                       The four parameters are :
;                          Plcpips_org(0, *, *, * ) -- Number of sectors per spin as defined in
;                                                      the telemetry accumulation table
;                          Plcpips_org(1, *, *, * ) -- Number of spins per sample as defined in
;                                                      the telemetry accumulation table
;    Plcpips_org(2, *, *, * ) -- Number of values stored  ( will be a function
;                                of the number of spins of data to which the
;                                the current table is applied  AND the data
;                                accumulation rate ( the number of spins over
;                                which the data accumulated )
;    Plcpips_org(3, *, *, * ) -- Index location of first value in
;                                'plcpips_bytes' array ( the value from the
;                                first sector and first spin of data to which
;                                the telemetry accumulation table applies
;               lut_times       : LONARR giving times for spin header records.
;               luts            : BYTARR giving LUT table used for each spin from spin header record.
;               experiment      : String indicating experiment i.e 'IES', 'IPS', 'HSTP' or 'HSTE'.
;               mode            : String indicating data output required i.e
;                                'SPIN_AVERAGED'       : Spin samples summed over sectors.
;                                'MAJOR_SPIN_AVERAGED' : Major spin samples for each sector.
;                                'INTEGRAL_ENERGY'     : Spin samples for each sector for integral energy.
;                                'FULL_RESOLUTION'     : Sector samples for all energies.
;                                'FULL_ANGLE'          : Sector samples for all energies spin synced.
;                                'LOW_RESOLUTION'      : Major spin samples summed over sectors.
;                                'SPIN_N'              : N sectors.
;                                'CLEAN_N'             : N sectors with pedestal adjustement.
;                                'CLEAN_SURVEY'        : CLEANed survey plot data.
;                                'SURVEY'              : Survey plot data.
;
; Opt. Inputs : None.
;
; Outputs     : output_header : header structure
;               output_data   : data structure
;
; Opt. Outputs:	pedestal_info : FLTARR(nvalues,npoints,nsectors,ndetectors) 0 : pedestal shift in bins.
;
; Keywords    : START_TIME : STRING giving start time of period of interest in level 1
;                            data. Example format = '10:00:00', ignores day.
;               END_TIME   : STRING giving end   time of period of interest in level 1
;                            data. Example format = '11:00:00', ignores day.
;               DETECTORS  : INTARR detector number associated with each data entry.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 12/1/96
;
; Modified    :	Version 0.1, 2/4/96, MKC
;                            Switched to SHIFT=2 if luts wrong.
;                       0.2, 13/6/96, MKC
;                            Removed spin_time from cep_times and cep_reorg argument list.
;                       0.3, 16/9/96, MKC
;                            Note that shift didnt work properly. Removed SHIFT keyword.
;                            Added 3rd set of LUTs.
;                            Added spin header record times to argument list and matched with spin times.
;                            Corrected treament of bad spin times.
;                            Added routine cep_luts.
;                            Added major frame tags.
;                            Fixed expanding REBIN bug.
;                            Remove extra dummy record. Make last record dummy record.
;                            Added no_of_sectors to cep_times.
;                            Corrected bad spin counts for FULL_RESOLUTION mode.
;                            Changed processing structure to allow CLEAN modes.
;                        0.4 13/11/96
;                            New ies_clean. Changed so that only corrects, calibrates
;                            converts to rates and per kev.
;                            Added sample_times and bad_sample_times.
;                            Converted from counts/spin to counts so rate calculations come out right.
;                        0.5 16/1/97
;                            Renamed ies_clean to ies shift and converted to procedure.
;                            Explicitly set up clean version number.
;                            Subtracted pedestal before bin division.
;                        0.6 16/1/97
;                            Used new pedestal position and subtraction routines.
;                            Added lut_map to cep_get_luts.
;                            Zeroed bad data to avoid it affecting pedestal shift.
;                            Added pedestal shift argument.
;                        0.7, 11/3/97.
;                            Changed pedestal_shift to pedestal_info.
;                        0.8 4/7/97
;                            Added energy_efficiency to cep_get_luts.
;                            Changed so that SURVEY format files contain calibrated data.
;                            removed .pedestals and .lut_map header tags.
;                            Changed ies_pedestals argument list.
;                        0.9 10/10/97
;                            Added energy_calibration offset.
;                            Set up energy_calibration and count_calibration in ies_header_structure.
;                            Put back .lut_map header tag.
;                        1.0 22/10/97
;                            Fixed if only one sample in L1 file.
;                        1.1, 4/3/98
;                            Changed so that ies shift uses undivided data.
;                            Changed so that ies shift works in bins rather than keV.
;                            Changed so that ies shift does not update nbands.
;                            Set clean algorithm to 4.
;                        1.2 31/7/98
;                            Corrected so that deals with selected time intervals properly.
;                            NB. Time interval selection dealt with in calling routine.
;                                Now copes with summation over multiple units of 16 spins.
;                        1.3 7/1/99
;                            Changed loop variable to LONG to avoid 32768 limit.
;                        1.4 5/2/99
;                            Added lut_changes.
;                        1.5 20/09/00
;                            Moved set up of pedestal calibration data from ies get samples.
;                        1.6 13/10/00
;                            Passed lut_map in ies_pedestals argument list instead of picking up from calibration file.
;                        1.7 20/07/01
;                            Used ies runs. NB Previously could miss last run.
;                        1.8 29/01/02
;                            Corrected processing for INTEGRAL_ENERGY mode.
;                        1.9, 14/04/03
;                            Changed argument list of ies_rat.
;                            Changed where output_data.time set.
;                        2.0, 02/10/03
;                            Changed argument list of ies get pedestal calibration.
;                            Renamed routine ies pedestals.
;                            Used ies get dir.
;                        2.1, 24/10/03
;                            Use ies lut list.
;                        2.2, 14/11/03
;                            Renamed output to output_array.
;                            Distinguished between output_chans [keV] and ch_positions[bins].
;                            Used routine ies bins to kev.
;                        2.3, 24/11/03
;                            Changed ERRORS keyword to CHISQ in cep_pedestals.
;                            Moved ies get pedestal calibration to cep pedestals.
;                        2.4, 17/12/03
;                            Used ies set ch positions..
;
; Version     :	Version  2.4, 17/12/03
;-
;**********************************************************

PRO cep_reorg, datatype, mjulday, input_times, input_bytes, org, lut_times, luts, $
               output_header, output_data, experiment, mode, pedestal_info, $
               START_TIME=start_time, END_TIME=end_time

  ; initialize LUT table for decompression

  lut_table = 0

  ; each energy/detector combination will have the same no. spins
  ; calculate no of spins etc. for first detector/energy
  ; for each energy/detector combination have :
  ;    samples per sector -->
  ;    sectors per spin !

  sectors_per_spin   = org(0,0,0)
  spins_per_sample   = org(1,0,0)
  samples_per_sector = org(2,0,0)
  first_value        = org(3,0,0)

  ; get no of spins in data

  no_of_spins = samples_per_sector * spins_per_sample

  ; check no of spins

  IF no_of_spins NE N_ELEMENTS(input_times) THEN MESSAGE, 'Invalid no. of spins = ' + STRTRIM(no_of_spins,1), /TRACEBACK

  ; check no of major spins

  IF (no_of_spins MOD 16) NE 0 THEN MESSAGE, 'Invalid no. of major spins = ' + STRTRIM(no_of_spins, 1), /TRACEBACK

  ; copy spin times

  spin_times = input_times

  ; select period of interest and get TAI times and fix bad times

  cep_times, mjulday, spin_times, bad_spin_times, first, last, START_TIME=start_time, END_TIME=end_time

  ; get no. of spins

  no_of_spins =  (last-first)*16

 ; initialize list of bad spins

  bad_spin_counts = BYTARR(no_of_spins)

  ; get bin positions and calibration factors

  ch_positions = cep_get_luts ( experiment, count_calibration, energy_calibration, lut_changes, $
                                energy_efficiency, lut_map)

  ; get no of original energies for experiment
  ; NB may be changed according to intermediate data format

  nenergies = (SIZE(ch_positions))[2]

  ; get no. of detectors

  no_of_detectors = (SIZE(ch_positions))[3]

  ; get no. of LUTs

  no_of_luts = (SIZE(ch_positions))[4]

  ; get no. of LUT changes

  no_of_changes = (SIZE(ch_positions))[5]

  ; reform ch_positions, energy_efficiency and count_calibration

  ch_positions = REFORM(ch_positions, 2, nenergies, no_of_detectors, no_of_luts*no_of_changes)

  IF KEYWORD_SET(energy_efficiency) THEN $
    energy_efficiency = REFORM(energy_efficiency, nenergies, no_of_luts*no_of_changes)

  IF KEYWORD_SET(count_calibration) THEN $
    count_calibration = REFORM(count_calibration, no_of_luts*no_of_changes)

  ; set up IPS

  IF experiment EQ 'IPS' THEN BEGIN

    ; reorder IPS detectors
    ; gives orientation of detectors relative to anti-spin direction (SAT #0 table)
    ; telemetry order is T1/C1/B1,  C2,  C5,  C0,  B5,  B2,  B0,  T5,  T2,  T0
    ; detector order from spin direction is B0  B2  B5  C0  C2  C5  T5  T2  T0
    ; header.detectors gives the position of each detector relative to anti-spin direction
    ; i.e T0 T2 T5 C5 C2 C0 B5 B2 B0
    ;     0  1  2  3  4  5  6  7  8

    detectors = [9,4,3,5,6,7,8,2,1,0]
    ptitle = [ 'T1/C1/B1', 'C2', 'C5', 'C0', 'B5', 'B2', 'B0', 'T5', 'T2', 'T0']

  ENDIF ELSE BEGIN

    ; if detectors not set then will get 1-ndetectors
    ; for IES detectors from anti-spin direction are therefore labelled 1-9
    ; believe this is correct for small pixels and opposite way around for large pixels

    detectors = 0
    ptitle = 0

  ENDELSE

  ; ies_header_structure resets the no_of_samples, no. of energies and no. of sectors according to the mode.

  no_of_energies  = nenergies
  no_of_sectors   = 32
  no_of_samples   = no_of_spins

  output_header = ies_header_structure ( datatype+mode, no_of_samples, no_of_detectors, no_of_energies, no_of_sectors, no_of_luts*no_of_changes, $
                                  TITLE  = datatype + mode, $
                                  XTITLE = 'UT:', $
                                  YTITLE = 'Energy', $
                                  ZTITLE = 'Cts', $
                                  PTITLE = ptitle, $
                                  DETECTORS = detectors, $
                                  EXPERIMENT=experiment, $
                                  MODE=mode, $
                                  ENERGY_EFFICIENCY=energy_efficiency, $
                                  ENERGY_CALIBRATION=energy_calibration, $
                                  COUNT_CALIBRATION=count_calibration, $
                                  LUT_MAP=lut_map, $
                                  SPINS_PER_SAMPLE=nspins )

  ; cope with averaging over multiples of 16 spins
  ; NB no_of_spins and nspins are multiples of 16

  IF nspins GT 16 THEN BEGIN

    no_of_spins = (no_of_spins/nspins)*nspins

    spin_times = spin_times(0:no_of_spins-1)

    last = first + no_of_spins/16

  ENDIF

  output_data   = ies_data_structure ( no_of_samples, no_of_detectors, no_of_energies, no_of_sectors, EXPERIMENT=experiment, MODE=mode )

  ; get times for each sample

  cep_newtimes, spin_times, bad_spin_times, no_of_samples, sample_times, bad_sample_times

  ; get lut for each spin time

  sample_luts = cep_luts ( input_times(first*16:last*16-1), lut_times, sample_times, luts, bad_sample_luts, $
                           experiment, no_of_samples, lut_changes, no_of_luts)

  ; set up times
  ; NB last time is the start time of the last spin
  ;    output_data array has size no_of_samples

  IF N_ELEMENTS(sample_times) GT 1 THEN output_data.time = sample_times ELSE output_data.time = sample_times[0]

  ; set up data
  ; data transformed into counts per spin
  ; for each sector or summed for each spin

  ; check if integral energy mode

  IF mode EQ 'INTEGRAL_ENERGY' THEN BEGIN

    start_energy = nenergies-1

    ch_positions = ch_positions[*,start_energy,*,*]

  ENDIF ELSE BEGIN

    start_energy = 0

  ENDELSE


  ; if CLEAN mode then create pedestal info array

  IF STRMID(mode,0,5) EQ 'CLEAN' THEN BEGIN

    pedestal_info = FLTARR(6,no_of_samples,16,no_of_detectors)

  ENDIF ELSE BEGIN

    pedestal_info = 0

  ENDELSE

  ; loop over detectors

  FOR detector = 0, no_of_detectors-1 DO BEGIN

    ; print some info

    PRINT, 'Processing detector ', STRTRIM(detector,1)

    ; get sys_time

    sys_time = SYSTIME(1)

    ; if CLEAN mode then create temporary output array for single detector

    IF STRMID(mode,0,5) EQ 'CLEAN' THEN BEGIN

       ; correct sectors at 16 sectors per spin before averaging sectors

       nsectors = 16
       output_array = FLTARR ( nsectors, 1, (nenergies-1)>1, no_of_samples )

    ENDIF ELSE nsectors = no_of_sectors

    ; loop over energies

    FOR energy = start_energy, nenergies-1 DO BEGIN

      ; get abbreviation for energy/detector organization items

      sectors_per_spin   = org(0,detector,energy)
      spins_per_sample   = org(1,detector,energy)
      samples_per_sector = org(2,detector,energy)
      first_value        = org(3,detector,energy)

      ; get no. of values

      no_of_values = samples_per_sector * sectors_per_spin

      ; extract data for energy/detector combination for first to last

      data = input_bytes(first_value:first_value+no_of_values-1)

      ; reform data into spin samples * sector samples array

      data = REFORM ( data, samples_per_sector, sectors_per_spin )

      ; extract data for requested time interval
      ; adjust no of data samples

      data = data(first*16/spins_per_sample:last*16/spins_per_sample-1,*)

      samples_per_sector = (last-first)*16/spins_per_sample

      ; transpose array into sector_per_spin samples * samples_per_sector samples array

      data = TRANSPOSE ( data )

      ; get list of spin frames with bad samples
      ; NB must be done before any data scaling takes place
      ; a given sample may cover a number of spins
      ; this sets a flag for all these spins

      bad_counts_list = WHERE ( data EQ 255, no_of_bad_counts )

      ; zero bad data to avoid it affecting pedestal shift

      IF no_of_bad_counts GT 0 THEN data(bad_counts_list) = 0

      ; get index into bad_spin_counts

      bad_counts = ( bad_counts_list / sectors_per_spin ) * spins_per_sample

      ; loop through bad counts setting associated spins to one

      FOR k = 1L, no_of_bad_counts DO bad_spin_counts(bad_counts(k-1):bad_counts(k-1)+spins_per_sample-1) = REPLICATE(1,spins_per_sample)

      ; extract and decompress data for energy/detector combination

      data = cep_decomp ( data, lut_table )

      ; convert to float

      data = FLOAT(data)

      ; diagnostic information

      IF 0 THEN BEGIN

        PRINT, 'Detector = ', detector
        PRINT, 'Energy   = ', energy
        PRINT, 'Sectors per spin = ', sectors_per_spin
        PRINT, 'Spins per sample = ', spins_per_sample
        PRINT, 'Samples per sector = ', samples_per_sector
        PRINT, 'First value    = ', first_value
        HELP, data, first, last, input_times, spin_times
      ;  ERASE
      ; TVSCL, data

      ENDIF

      ; average or duplicate sectors
      ; NB maintain same total counts

      IF no_of_samples GT no_of_spins THEN BEGIN

        ; duplicate sector samples and renormalize to give same total counts per spin
        ; extend array using /SAMPLE

        ns = no_of_samples/no_of_spins

        data = REBIN ( data, ns, samples_per_sector, /SAMPLE ) / (ns/sectors_per_spin)

      ENDIF ELSE IF nsectors GT sectors_per_spin THEN BEGIN

        ; duplicate sector samples and renormalize to give same total counts per spin
        ; (nsectors/sectors_per_spin) is the no of times duplicated each sample
        ; extend array using /SAMPLE

        data = REBIN ( data, nsectors, samples_per_sector, /SAMPLE ) / (nsectors/sectors_per_spin)

      ENDIF ELSE BEGIN

        ; average over sectors and renormalize to give same total counts per spin ie sum over sectors
        ; (sectors_per_spin / nsectors) is the no. of samples averaged together
        ; REBIN gives an average

        data = REBIN ( data, nsectors, samples_per_sector ) * (sectors_per_spin / nsectors)

      ENDELSE

      ; sum or duplicate samples
      ; NB maintain same total counts (then rate remains the same)

      IF no_of_samples GT no_of_spins THEN BEGIN

        ; duplicate samples using /SAMPLE and renormalize to give same total counts

        ns = no_of_samples/no_of_spins

        data = REBIN ( data, ns, no_of_spins, /SAMPLE ) / (no_of_spins/samples_per_sector)

      ENDIF ELSE IF no_of_samples GT samples_per_sector THEN BEGIN

        ; duplicate samples using /SAMPLE and renormalize to give same total counts
        ; (no_of_samples/samples_per_sector) is the no of times duplicated each sample

        data = REBIN ( data, nsectors, no_of_samples, /SAMPLE ) / (no_of_samples/samples_per_sector)

      ENDIF ELSE IF no_of_samples LT samples_per_sector THEN BEGIN

        ; average over samples and renormalize to give same total counts ie sum over samples
        ; (samples_per_sector/no_of_samples) is the no. of samples averaged together
        ; REBIN gives an average

        data = REBIN ( data, nsectors, no_of_samples ) * (samples_per_sector/no_of_samples)

      ENDIF

      ; reformat data

      IF STRMID(mode,0,5) EQ 'CLEAN'  THEN BEGIN

        IF energy LT nenergies-1 THEN output_array[*,0,energy,*] =  data

      ENDIF ELSE BEGIN

        ; check mode

        IF mode EQ 'INTEGRAL_ENERGY' THEN BEGIN

          ; integral energy channel

          output_data.data ( *, detector, 0 ) = data

        ENDIF ELSE IF no_of_samples GT no_of_spins THEN BEGIN

          ; sector counts arranged along time axis

          output_data.data ( 0, detector, energy ) = REFORM ( data, no_of_samples )

        ENDIF ELSE BEGIN

          ; all other modes
          ; NB if no_of_sectors=1 need to reform data

          output_data.data ( *, detector, energy ) = REFORM ( data )

        ENDELSE

      ENDELSE

    ENDFOR ; end loop over energies

    IF STRMID(mode,0,5) EQ 'CLEAN' THEN BEGIN

      ; NB Omit integral channel
      ; NB nbands, output_chans updated

      old_ch_positions = ch_positions[*,0:(nenergies-2)>0,detector,*]
      nbands = nenergies-1

      ; get list of lut runs

      lut_list = ies_lut_list(sample_luts, output_header.nluts )

      ; calculate pedestal positions and subtract pedestals
      ; NB working in bins rather than keV

      pedestal_posns = cep_pedestals ( output_array, output_data.time, lut_list, $
                                       output_header.detectors[detector], $
                                       nsectors, 1, nbands, old_ch_positions, $
                                       output_header.lut_map, ps, /SUB, /CHISQ)

      pedestal_info(*,*,*,detector) = ps

      ; apply rate calibration i.e convert from counts to counts/sec

      ies_rat, output_data, nsectors, nbands, 1, output_array

      ; apply count calibration i.e convert from counts to counts/sr/cm2

      ies_cal, lut_list, count_calibration, output_array

      ; get new energy channels

      new_ch_positions = ies_set_ch_positions(no_of_energies, 1, N_ELEMENTS(old_ch_positions(0,0,0,*)) )

      ; correct undivided data
      ; working in bins rather than keV

      ies_shift, lut_list, nsectors, no_of_energies, 1, nbands, old_ch_positions, new_ch_positions, pedestal_posns, output_array

      nbands = no_of_energies

      ; take into account channel sizes
      ; convert from counts per channel to count per keV

      ies_div, lut_list, nbands, 1, ies_bins_to_kev(energy_calibration, new_ch_positions), output_array

      ; data is corrected at 16 sectors per spin before being finally output
      ; this is to adjust so that still have flux when sum over sectors

      IF no_of_sectors GT nsectors THEN BEGIN

        ; duplicate sector samples and renormalize to give same total counts per spin
        ; extend array using /SAMPLE

        output_data.data (*, detector, *) = REBIN ( output_array, no_of_sectors, 1, no_of_energies, no_of_samples, /SAMPLE ) / (no_of_sectors/nsectors)

      ENDIF ELSE BEGIN

        ; sum over sectors
        ; REBIN averages samples

        output_data.data (*, detector, *) = REBIN ( output_array, no_of_sectors, 1, no_of_energies, no_of_samples ) * (nsectors / no_of_sectors)

      ENDELSE

    ENDIF ELSE IF STRPOS(mode,'SURVEY') GE 0 THEN BEGIN

      output_array = output_data.data (*, detector, *)
      nbands = nenergies

      ; get list of lut runs

      lut_list = ies_lut_list(sample_luts, output_header.nluts )

      ; take into account channel sizes
      ; convert from counts per channel to count per keV

      ies_div, lut_list, nbands, 1, ies_bins_to_kev(energy_calibration, ch_positions[*,*,detector,*]), output_array

      ; apply rate calibration i.e convert from counts to counts/sec

      ies_rat, output_data, nsectors, nbands, 1, output_array

      ; apply count calibration i.e convert from counts to counts/sr/cm2

      ies_cal, lut_list, count_calibration, output_array

      ; apply energy efficiency

      IF KEYWORD_SET(energy_efficiency) THEN ies_efficiency, lut_list, energy_efficiency, output_array

      ; set up data

      output_data.data(*, detector, *) = output_array

    ENDIF

  ENDFOR ; end loop over detectors

  ; set up output structures

  ; set up bin positions and tags

  IF STRMID(mode,0,5) EQ 'CLEAN' THEN BEGIN

    ; NB energy channels now same for each detector

    FOR detector = 0, no_of_detectors-1 DO output_header.ch_positions[*,*,detector,*] = new_ch_positions[*,*,0,*]

    output_header.cal   = 1
    output_header.div   = 1
    output_header.rat   = 1

    output_header.subtract = 1
    output_header.clean = 4

  ENDIF ELSE IF STRPOS(mode,'SURVEY') GE 0 THEN BEGIN

    output_header.ch_positions = ch_positions

    output_header.cal   = 1
    output_header.div   = 1
    output_header.rat   = 1

  ENDIF ELSE BEGIN

    output_header.ch_positions = ch_positions

  ENDELSE

  ; set up luts
  ; NB if cleaned then all luts will be the same

  IF N_ELEMENTS(sample_luts) GT 1 THEN output_data.lut = sample_luts $
                                  ELSE output_data.lut = sample_luts(0)

  ; set up times

  ; flag bad spin times with bit 6 for all energies

  IF bad_sample_times(0) GE 0 THEN output_data(bad_sample_times).flag = output_data(bad_sample_times).flag OR '40'XB

  ; flag bad spin luts with bit 4 for all energies

  IF bad_sample_luts(0) GE 0 THEN output_data(bad_sample_luts).flag = output_data(bad_sample_luts).flag OR '10'XB

  output_header.tstart = output_data(0).time
  output_header.tstop  = output_data(no_of_samples-1).time

  ; deal with any bad counts

  ; get list of bad spins

  bad_spin_list = WHERE ( bad_spin_counts EQ 1, bad_spin_count )

  ; printout warning message

  PRINT, 'Bad spin counts = ', STRTRIM(bad_spin_count,1) , '/', STRTRIM(no_of_spins,1)

  IF bad_spin_count GT 0 THEN BEGIN

    ; check if major spins required

    IF no_of_samples LT no_of_spins THEN BEGIN

      ns = no_of_spins/no_of_samples

      ; get list of bad major spins

      bad_spin_list = bad_spin_list/ns

      ; remove duplicate elements
      ; NB remembering to keep as array if only one element

      bad_spin_list = bad_spin_list([ies_runs(bad_spin_list)])

      ; set no of elements

      bad_spin_count = N_ELEMENTS(bad_spin_list)

;      ; get spins corresponding to bad major spins
;
;      bad_spin_list = REFORM ( REPLICATE(1,ns) # (ns*bad_major_spin_list) + $
;                               LINDGEN(ns) # REPLICATE(1,bad_major_spin_count), ns*bad_major_spin_count )

    ENDIF ELSE IF no_of_samples GT no_of_spins THEN BEGIN

      ns = no_of_samples/no_of_spins

      ; set indeces corresponding to sectors

      bad_spin_list = REFORM ( REPLICATE(1,ns) # (ns*bad_spin_list) + $
                         LINDGEN(ns) # REPLICATE(1,bad_spin_count), ns*bad_spin_count )

    ENDIF

    ; flag bad spins with bit 5 for all energies

    output_data(bad_spin_list).flag = output_data(bad_spin_list).flag OR '20'XB

  ENDIF

  ; set major frame tags

  IF no_of_samples GE no_of_spins THEN BEGIN

    ns = no_of_samples/no_of_spins

    output_data(16*ns*LINDGEN(no_of_spins/16)).flag = $
              output_data(16*ns*LINDGEN(no_of_spins/16)).flag OR '01'XB

  ENDIF ELSE IF no_of_samples LT no_of_spins THEN BEGIN

    ns = no_of_spins/no_of_samples

    output_data(16/ns*LINDGEN(no_of_spins/16)).flag = output_data(16/ns*LINDGEN(no_of_spins/16)).flag OR '01'XB

  ENDIF

  ; check if full resolution mode

  IF no_of_samples GT no_of_spins THEN BEGIN

    ns = no_of_samples/no_of_spins

    ; remove sectors from last spin

    no_of_samples = no_of_samples - ns + 1

    output_data = output_data(0:no_of_samples-1)

    output_header.npoints = no_of_samples

  ENDIF

  ; flag point at end of array

  output_data(no_of_samples-1).flag = '80'XB

END
