;+
; Project     :	CLUSTER - PEACE
;
; Name        :	CEP_SETUP_PEDESTAL_CALIBRATION
;
; Purpose     : Retrieves pedestal calibration data and sets up pedestal samples.
;
; Explanation :	Pedestal calibration data is stored in IES_CALIBRATION files.
;               This routine restores the relevant pedestal calibration data provided it has not already
;               been stored and stores in the pedestal calibration data COMMON block.
;               It then calculates for each LUT the counts in the pedestal channels for different positions of the pedestal
;               data.
;
; Use         : < cep_setup_pedestal_calibration, npeds, ndetectors, no_of_luts >
;
; Inputs      : npeds             : INT no. of pedestal channels
;               ndetectors        : INT no. of selected detectors.
;               no_of_luts        : INT no. of luts
;
; Opt. Inputs : None.
;
; Outputs     : None.

; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 04/12/03
;                 Adapted from rcs setup pedestal calibration.
;
; Modified    :	Version 0.1, 24/01/04
;                 Changed max_sample_lists to an INT array
;
; Version     :	Version 0.1, 24/01/04
;-
;**********************************************************

PRO cep_setup_pedestal_calibration, npeds, ndetectors, no_of_luts

  COMMON pedestal_sample_block, tinfo, tsamples, tmax_sample_lists, tpedestal_posns, tpedestal_widths

  ; set up COMMON block and restore calibration data

  ies_get_pedestal_calibration, concat_dir(ies_get_dir('TM_TABLE',/POLAR), 'IES_CALIBRATION')

  ; set up tinfo structure which defines what calculations are stored in COMMON block

  tinfo = {temperature_index:-1L, done:INTARR(ndetectors, no_of_luts), nshifts:251, offset:50}

  ; set up samples array

  tsamples = FLTARR( npeds, tinfo.nshifts, ndetectors, no_of_luts)

  ; set up max_sample_lists array
  ; list of shifts for each pedestal channel maxima position
  ; 1st element contains no of samples
  ; these are used as look up tables to speed up search for shifted
  ; samples with the same peak as the actual data

  tmax_sample_lists = INTARR(1+tinfo.nshifts, npeds, ndetectors, no_of_luts)

  ; set up pedestal positions

  tpedestal_posns = FLTARR( ndetectors, no_of_luts)

  ; set up pedestal widths

  tpedestal_widths = FLTARR( ndetectors, no_of_luts)

END