;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_survey_files
;
; Purpose     :	Converts CEPPAD level 0 data to RAL intermediate format for survey plots.
;
; Explanation : Produces CEPPAD intermediate format survey data.
;
; Use         : < survey_files = cep_survey_files ( /L1, /RF, PEDESTAL=pedestal, BF=bf, /GZIP ) >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : survey_files : STRARR(2,n) containing names of IES, IPS intermediate format files used.
;               CEPPAD level 1 and RAL intermediate format files.
;
; Opt. Outputs:	None.
;
; Keywords    : /L1         = Specifies that the l1 files already exist.
;               /RF         = Specifies that the if files already exist.
;                             Either Keyword set or STRARR(5,n) containing if files.
;               /PEDESTAL   = Specifies that the pedetsal file should be created.
;               /BF         = Specifies that the magnetic field file should be created.
;               /GZIP       = gzips files on uniz system.
;               /FORCE      = forces creation of intermediate files.
;               /DEBUG      = STOPs in routine.
;               /FILTER     = STRING for PICKFILE filter ie '*HISTe*'. This is useful
;                             when the number of files in the directory is too large to
;                             be displayed by PICKFILE.
;               /ALL        = FLAG indicating that all LZ files in specified directory should be processed.
;                                 Avoids use of pickfile.
;               /COPY       = Indicates should copy OR and AT files from CDROM.
;
; Calls       :	cep_lztol1, cep_l1toif, cep_get_l1day, strip_dirname, concat_dir, cep_get_l1_files,
;               datatype, ies_get_dir.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	Adapted from cep survey.
;
; Written     :	Version 0.0, Martin Carter, RAL, 0.0, 2/5/97
;
; Modified    :	0.1, 23/5/97, MKC
;                 Added GZIP keyword.
;               0.2, 8/9/97, MKC
;                 Added IPS and IES intermediate format directories.
;               0.3, 11/9/97, MKC
;                 Added HK and HIST intermediate format files.
;               0.4, 29/9/97, MKC
;                 Added changeable L1 directory.
;                 Allowed compressed of uncompressed pre-existing if files.
;               0.5, 7/1/98, MKC
;                 Removed tracebcak to avoid unnecessary error warnings.
;               0.6, 10/2/98, MKC
;                 Added DEBUG keyword.
;                 Changed so that allows any instrument to be used
;                 for filespec selection of l1 files.
;               0.7, 19/5/98, MKC
;                 Fixed problem with rf files and made l1 stuff Y2K compliant.
;               0.8, 22/10/98, MKC
;                 Added magnetic field.
;               0.9, 8/2/99
;                 Added level zero version number and forced creation of intremediate file
;                 if processing version 2 level zero file. 
;               1.0, 17/3/99
;                 Brought into line with cep_parse_l1day so can use l1 HK as specifier.
;               1.1, 11/5/99
;                 Added FORCE keyword to argument list.
;               1.2, 13/12/99
;                 Checked for Y2K compatability.
;                 Replaced PICKFILE with ies_pickfile.
;                 This routine now replaces cep_lztoif.
;                 Added FILTER keyword.
;               1.3, 11/07/02
;                 Changed error message to be more informative.
;                 Changed so that does not STOP if not all IF files exist for particular day.
;                 Changed to soft errors.
;               1.4, 03/10/03
;                 Added /ALL keyword.
;                 Added /COPY keyword.
;               1.5, 09/10/03
;                 Used ies get dir.
;
; Version     :	Version 1.5, 09/10/03
;-
;**********************************************************

FUNCTION cep_survey_files, L1=l1, RF=rf, PEDESTAL=pedestal, BF=bf, GZIP=gzip, FORCE=force, DEBUG=debug, $
                           FILTER=filter, ALL=all, COPY=copy

  ; come back out to top level unless debug set

  IF KEYWORD_SET(debug) THEN ON_ERROR, 0 ELSE ON_ERROR, 1

  ; get l1 directory

  l1_dir = ies_get_dir('L1',/POLAR)

  ; get lz files

  IF NOT KEYWORD_SET(l1) AND NOT KEYWORD_SET(rf) THEN BEGIN

    ; do level 0 to level 1 conversion

    lzfiles = cep_lztol1(GZIP=gzip, LZ_VERSION=lz_version, ALL=all, COPY=copy)

    IF lzfiles(0) EQ '' THEN RETURN, ''

    nfiles = N_ELEMENTS(lzfiles)

  ENDIF ELSE IF KEYWORD_SET(rf) THEN BEGIN

    IF datatype(rf) EQ 'STR' THEN rffiles = rf ELSE BEGIN

      ; get specifier

      IF NOT KEYWORD_SET(filter) THEN filter = '*CLEAN_SURVEY_IES*.DAT*'

      files = ies_pickfile ( PATH=ies_get_dir('IES',/POLAR), FILTER=filter, $
                             /READ, TITLE='Select IES intermediate format data files', /MULTIPLE_FILES )

      IF files[0] EQ '' THEN BEGIN

        MESSAGE, 'No intermediate format data files found', /INFORMATIONAL

        RETURN, ''

      ENDIF

      ; list if files into string array

      rffiles = STRARR(5,N_ELEMENTS(files))

      rffiles(0,*) = files

    ENDELSE

    nfiles = N_ELEMENTS(rffiles(0,*))

  ENDIF ELSE IF KEYWORD_SET(l1) THEN BEGIN

    ; prompt for level 1 file spec
    ; NB multiple files cannot be defined only filespec with wild card

    IF NOT KEYWORD_SET(filter) THEN filter = 'PO_CEP_L1_IES_*.DAT*'

    filespec = ies_pickfile ( PATH=l1_dir, FILTER=filter, $
                              /READ, TITLE='Select level 1 format data filespec', GET_PATH=l1_dir )

    ; check file selected

    IF filespec EQ '' THEN MESSAGE, 'Level 1 file not selected'

    ; get l1 files

    l1files = cep_get_l1_files(filespec, L1DIR=l1_dir)

    IF l1files[0] EQ '' THEN MESSAGE, 'Level 1 file not selected'

    nfiles = N_ELEMENTS(l1files)

  ENDIF ELSE BEGIN

    nfiles = 1

  ENDELSE

  ; set flag indicating whether standard survey plot

  standard = 1

  ; set up output STRARR, ensure at least one file if no lz file

  survey_files = STRARR(5,nfiles)

  survey_count = 0

  ; loop through level 0 files

  FOR f = 0, nfiles-1 DO BEGIN

    IF KEYWORD_SET(rf) THEN BEGIN

      ; claim that RAL intermediate format files already exist, get file names

      ies_file   = rffiles(0,f)

      ips_file   = rffiles(1,f)

      histe_file = rffiles(2,f)

      histp_file = rffiles(3,f)

      hk_file    = rffiles(4,f)

      IF ips_file EQ '' OR histe_file EQ '' OR histp_file EQ '' OR hk_file EQ '' THEN BEGIN

        IF STRMID(strip_dirname(ies_file),0,28) EQ 'PO_CEP_RAL_CLEAN_SURVEY_IES_' THEN BEGIN

          endfile = strip_dirname(ies_file)

          IF STRPOS(endfile,'.DAT') LE 28 THEN MESSAGE, 'Invalid file name ' + ies_file

          endfile = STRMID(endfile,28,STRPOS(endfile,'.DAT')-28) + '.DAT*'

          ips_file = concat_dir ( ies_get_dir('IPS',/POLAR),'PO_CEP_RAL_SURVEY_IPS_' + endfile )

          histe_file = concat_dir ( ies_get_dir('HISTe',/POLAR),'PO_CEP_RAL_SURVEY_HISTe_' + endfile )

          histp_file = concat_dir ( ies_get_dir('HISTp',/POLAR),'PO_CEP_RAL_SURVEY_HISTp_' + endfile )

          hk_file = concat_dir ( ies_get_dir('HK',/POLAR),'PO_CEP_RAL_HK_' + endfile )

        ENDIF ELSE BEGIN

          ips_file = ies_pickfile ( PATH=ies_get_dir('IPS',/POLAR), FILTER='*_IPS_*.DAT*', $
                                /READ, /MUST_EXIST, TITLE='Select IPS intermediate format data file' )

          histe_file = ies_pickfile ( PATH=ies_get_dir('HISTe',/POLAR), FILTER='*_HISTe_*.DAT*', $
                                /READ, /MUST_EXIST, TITLE='Select HISTe intermediate format data file' )

          histp_file = ies_pickfile ( PATH=ies_get_dir('HISTp',/POLAR), FILTER='*_HISTp_*.DAT*', $
                                /READ, /MUST_EXIST, TITLE='Select HISTp intermediate format data file' )

          hk_file = ies_pickfile ( PATH=ies_get_dir('HK',/POLAR), FILTER='*_HK_*.DAT*', $
                                /READ, /MUST_EXIST, TITLE='Select HK intermediate format data file' )

        ENDELSE

      ENDIF

    ENDIF ELSE IF KEYWORD_SET(l1) THEN BEGIN

      ; claim that l1 format files already exist, get file names
      ; Just need date
      ; l1 files have either forms :
      ; PO_CEP_L1_IES_1996357_01.DAT.gz or PO_CEP_L1_HK_1996221.DAT.gz
      ; this is used in cep_parse_l1day

      ; decide between forms

      pos = STRPOS ( l1files[f], '.DAT' )

      IF pos LT 10 THEN MESSAGE, 'Invalid level 1 file name : ' + l1files[f], /TRACEBACK

      IF STRMID( l1files[f], pos-3, 1) EQ '_' THEN pos = pos - 10 ELSE pos = pos - 7

      l1_end = STRMID(l1files(f),pos,7)

      ies_file_spec = concat_dir ( l1_dir,'PO_CEP_L1_IES_' + l1_end )

      ips_file_spec = concat_dir ( l1_dir,'PO_CEP_L1_IPS_' + l1_end )

      histe_file_spec = concat_dir ( l1_dir,'PO_CEP_L1_HISTe_' + l1_end )

      histp_file_spec = concat_dir ( l1_dir,'PO_CEP_L1_HISTp_' + l1_end )

      hk_file_spec = concat_dir ( l1_dir,'PO_CEP_L1_HK_' + l1_end)

      IF ies_file_spec EQ '' OR ips_file_spec EQ '' OR histe_file_spec EQ '' OR histp_file_spec EQ '' $
        OR hk_file_spec EQ '' THEN MESSAGE, 'INVALID LEVEL 1 FORMAT FILE NAME', /TRACEBACK

      ; do level 1 to RAL intermediate format conversion on existing l1 files
      ; prompt for file names
      ; NB HK file needed for IES processing
      ; if FORCE used then will recreate for each L1 file with same name

      cep_l1toif, hk_file_spec, MODE='SURVEY', FILE=hk_file, GZIP=gzip, L1DIR=l1_dir, DEBUG=debug, FORCE=force

      cep_l1toif, hk_file_spec, MODE='SURVEY', FILE=bf_file, GZIP=gzip, L1DIR=l1_dir, /BF, DEBUG=debug, FORCE=force

      cep_l1toif, ies_file_spec, MODE='CLEAN_SURVEY', FILE=ies_file, PEDESTAL=pedestal, GZIP=gzip, L1DIR=l1_dir, DEBUG=debug, FORCE=force

      cep_l1toif, ips_file_spec, MODE='SURVEY', FILE=ips_file, GZIP=gzip, L1DIR=l1_dir, DEBUG=debug, FORCE=force

      cep_l1toif, histe_file_spec, MODE='SURVEY', FILE=histe_file, GZIP=gzip, L1DIR=l1_dir, DEBUG=debug, FORCE=force

      cep_l1toif, histp_file_spec, MODE='SURVEY', FILE=histp_file, GZIP=gzip, L1DIR=l1_dir, DEBUG=debug, FORCE=force

    ENDIF ELSE BEGIN

      ; do level 1 to RAL intermediate format conversion
      ; NB HK file needed for IES processing
      ; if level zero file greater than version 1 then force creation of intermediate format file

      cep_l1toif, 'HK_' + cep_get_l1day ( lzfiles(f) ), MODE='SURVEY', FILE=hk_file, GZIP=gzip, L1DIR=l1_dir, FORCE=KEYWORD_SET(force) OR (lz_version[f] GT 1), DEBUG=debug

      cep_l1toif, 'HK_' + cep_get_l1day ( lzfiles(f) ), MODE='SURVEY', FILE=bf_file, /BF, GZIP=gzip, L1DIR=l1_dir, FORCE=KEYWORD_SET(force) OR (lz_version[f] GT 1), DEBUG=debug

      cep_l1toif, 'IES_' + cep_get_l1day ( lzfiles(f) ), MODE='CLEAN_SURVEY', FILE=ies_file, PEDESTAL=pedestal, GZIP=gzip, L1DIR=l1_dir, FORCE=KEYWORD_SET(force) OR (lz_version[f] GT 1), DEBUG=debug

      cep_l1toif, 'IPS_' + cep_get_l1day ( lzfiles(f) ), MODE='SURVEY', FILE=ips_file, GZIP=gzip, L1DIR=l1_dir, FORCE=KEYWORD_SET(force) OR (lz_version[f] GT 1), DEBUG=debug

      cep_l1toif, 'HISTe_' + cep_get_l1day ( lzfiles(f) ), MODE='SURVEY', FILE=histe_file, GZIP=gzip, L1DIR=l1_dir, FORCE=KEYWORD_SET(force) OR (lz_version[f] GT 1), DEBUG=debug

      cep_l1toif, 'HISTp_' + cep_get_l1day ( lzfiles(f) ), MODE='SURVEY', FILE=histp_file, GZIP=gzip, L1DIR=l1_dir, FORCE=KEYWORD_SET(force) OR (lz_version[f] GT 1), DEBUG=debug

    ENDELSE

    IF ies_file EQ '' OR ips_file EQ '' OR histe_file EQ '' OR histp_file EQ '' OR hk_file EQ '' THEN $
      MESSAGE, 'INVALID INTERMEDIATE FORMAT FILE NAME', /TRACEBACK

    ; if all IF files exist then add to output
    ; else generate warning

    ok = 1

    files = STRARR(5)

    files[0] = FINDFILE ( ies_file, COUNT=count )

    IF count NE 1 THEN BEGIN
      ok = 0
      IF count EQ 0 THEN MESSAGE, 'IES file does not exist : ' + ies_file, /INFORMATIONAL $
                    ELSE MESSAGE, 'Multiple IES files : ' + ies_file
    ENDIF

    files[1] = FINDFILE ( ips_file, COUNT=count )

    IF count NE 1 THEN BEGIN
      ok = 0
      IF count EQ 0 THEN MESSAGE, 'IPS file does not exist : ' + ips_file, /INFORMATIONAL $
                    ELSE MESSAGE, 'Multiple IPS files : ' + ips_file
    ENDIF

    files[2] = FINDFILE ( histe_file, COUNT=count )

    IF count NE 1 THEN BEGIN
      ok = 0
      IF count EQ 0 THEN MESSAGE, 'HISTe file does not exist : ' + histe_file, /INFORMATIONAL $
                    ELSE MESSAGE, 'Multiple HISTe files : ' + histe_file
    ENDIF

    files[3] = FINDFILE ( histp_file, COUNT=count )

    IF count NE 1 THEN BEGIN
      ok = 0
      IF count EQ 0 THEN MESSAGE, 'HISTp file does not exist : ' + histp_file, /INFORMATIONAL $
                    ELSE MESSAGE, 'Multiple HISTp files : ' + histp_file
    ENDIF

    files[4] = FINDFILE ( hk_file, COUNT=count )

    IF count NE 1 THEN BEGIN
      ok = 0
      IF count EQ 0 THEN MESSAGE, 'HK file does not exist : ' + hk_file, /INFORMATIONAL $
                    ELSE MESSAGE, 'Multiple HK files : ' + hk_file
    ENDIF

    IF ok THEN BEGIN

      ; add to output

      survey_files[*,survey_count] = files

      survey_count = survey_count + 1

    ENDIF

  ENDFOR

  IF survey_count EQ 0 THEN BEGIN

    MESSAGE, 'No complete sets of IF files found', /INFORMATIONAL

    RETURN, ''

  ENDIF

  RETURN, survey_files[*,0:survey_count-1]

END
