;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_survey_nplots
;
; Purpose     :	Produces postscript and png/gif survey plots.
;
; Explanation : Provides wrapper for calls to cep survey plot.
;
; Use         : < cep_survey_nplots, T0=start_time, T1=end_time >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : CEPPAD level 1 and RAL intermediate format files and postscript and png/gif plots.
;
; Opt. Outputs:	None.
;
; Keywords    : /L1         = Specifies that the l1 files already exist.
;               /RF         = Specifies that the if files already exist.
;                                if STRING or STRARR then assumes file names
;               T0         : STRING giving start time of period of interest in level 1
;                            data. Example format = '10:00:00', ignores day.
;               T1         : STRING giving end   time of period of interest in level 1
;                            data. Example format = '11:00:00', ignores day.
;               TYPES        = INT[] giving types of survey plot required.
;                                0 : IES/IPS standard plot
;                                1 : HISTE/IES/IPS pitch angle plot
;               /DEBUG      = STOPs in routine.
;               /CHECK      = Checks if decompressed file already exists.
;               /PEDESTAL   = Generates pedestal plots.
;               /ALL        = FLAG indicating that all LZ files in specified directory should be processed.
;                                  Avoids use of pickfile.
;               /COPY       = Indicates should copy OR and AT files from CDROM.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 26/02/02
;
; Modified    :	Version 0.1, 11/06/02
;                 If routine is interrupted then gets worng PNG directory.
;                 Fixed so that copes OK.
;               Version 0.2, 11/07/02
;                 Changed so that does not remove decompressed files.
;                 Added keyword CHECK.
;                 Even if check not set then does not check on next plot.
;                 If no files then returns.
;                 Modified cep survey keywords.
;                 Added PEDESTALS keyword.
;                 Changed TYPE keyword to TYPES.
;                 Allowed types to be an array.
;                 Allowed nplots to be zero.
;               Version 0.3, 02/10/03
;                 Changed JAVA keyword to WEB
;                 Renamed cep survey.
;                 Changed environment variable used for pedestal plots.
;                 Added /ALL keyword.
;                 Added /COPY keyword.
;                 Changed printout.
;
; Version     :	Version 0.3, 02/10/03
;-
;**********************************************************

PRO cep_modenv, name, type, n

  ; if INITIAL environment variable not set up then sets up

  IF GETENV(name+'INITIAL') EQ '' THEN BEGIN

    SETENV, name+'INITIAL='+GETENV(name)

  ENDIF

  ; then sets environment variable to INITIAL environment variable + subdirectories

  dir = GETENV(name+'INITIAL')

  dir = concat_dir(dir,type)

  dir = concat_dir(dir,STRTRIM(STRING(n),2))

  SETENV, name+'='+dir

END

PRO cep_survey_nplots, RF=rf, TYPES=types, DEBUG=debug, NPLOTS=nplots, LIST=list, CHECK=check, $
                       PEDESTALS=pedestals, ALL=all, COPY=copy

  IF N_ELEMENTS(nplots) EQ 0 THEN nplots = 4

  PRINT, 'CREATING PNG SURVEY PLOTS'

  IF NOT KEYWORD_SET(list) THEN list = INDGEN(1+nplots)

  FOR k = 0, N_ELEMENTS(list)-1 DO BEGIN

    kk = list[k]

    IF kk GT 0 THEN BEGIN

      ; just set up hh:mm:ss

      dummy = anytim2tai('1/jan/2000')

      start_time = STRMID(anytim2cal(dummy+(86400L/nplots)*(kk-1)),10,8)

      end_time = STRMID(anytim2cal(dummy+(86400L/nplots)*kk),10,8)

      IF kk EQ nplots THEN end_time = '24:00:00'

    ENDIF ELSE BEGIN

      start_time = '00:00:00'

      end_time = '24:00:00'

    ENDELSE

    ; do ies and pa survey plots

    FOR t = 0, N_ELEMENTS(types)-1 DO BEGIN

      ; NB so far only created ies and pa sub-directories for types

      CASE types[t] OF
        0 : type = 'ies'
        3 : type = 'pa'
        ELSE : MESSAGE, 'UNKNOWN TYPE'
      ENDCASE

      cep_modenv, 'PO_CEP_WEB_DIR', type, kk

      rf = cep_survey_plot ( RF=rf, /PNG, T0=start_time, T1=end_time, $
                              TYPE=types[t], DEBUG=debug, /WEB, $
                              CHECK=KEYWORD_SET(check) OR (k NE 0) OR (t NE 0), ALL=all, COPY=copy )

      IF rf[0] EQ '' THEN RETURN

    ENDFOR

    IF KEYWORD_SET(pedestals) THEN BEGIN

      ; do ped plot

      cep_modenv, 'PO_CEP_WEB_DIR', 'ped', kk

      rf = cep_pedestal_plot( RF=rf, /PNG, DEBUG=debug, CHECK=check, /WEB, ALL=all, COPY=copy)

      IF rf[0] EQ '' THEN RETURN

    ENDIF

  ENDFOR

END
